/*
   Copyright : Copyright (c) MOSEK ApS, Denmark. All rights reserved.

   File :      cqo1.java

   Purpose :   Demonstrates how to solve a small conic qaudratic
               optimization problem using the MOSEK API.
*/
package com.mosek.example;

import mosek.*;

public class cqo1 {
  static final int numcon = 1;
  static final int numvar = 6;

  public static void main (String[] args) throws java.lang.Exception {
    // Since the value infinity is never used, we define
    // 'infinity' symbolic purposes only
    double infinity = 0;

    mosek.boundkey[] bkc    = { mosek.boundkey.fx };
    double[] blc = { 1.0 };
    double[] buc = { 1.0 };

    mosek.boundkey[] bkx
    = {mosek.boundkey.lo,
       mosek.boundkey.lo,
       mosek.boundkey.lo,
       mosek.boundkey.fr,
       mosek.boundkey.fr,
       mosek.boundkey.fr
      };
    double[] blx = { 0.0,
                     0.0,
                     0.0,
                     -infinity,
                     -infinity,
                     -infinity
                   };
    double[] bux = { +infinity,
                     +infinity,
                     +infinity,
                     +infinity,
                     +infinity,
                     +infinity
                   };

    double[] c   = { 0.0,
                     0.0,
                     0.0,
                     1.0,
                     1.0,
                     1.0
                   };

    double[][] aval   = {
      {1.0},
      {1.0},
      {2.0}
    };
    int[][]    asub   = {
      {0},
      {0},
      {0}
    };

    int[] csub = new int[3];
    double[] xx  = new double[numvar];
    // create a new environment object
    try (Env  env  = new Env();
         Task task = new Task(env, 0, 0)) {
      // Directs the log task stream to the user specified
      // method task_msg_obj.stream
      task.set_Stream(
        mosek.streamtype.log,
        new mosek.Stream()
      { public void stream(String msg) { System.out.print(msg); }});

      /* Give MOSEK an estimate of the size of the input data.
      This is done to increase the speed of inputting data.
      However, it is optional. */
      /*TAG:begin-append*/
      /* Append 'numcon' empty constraints.
         The constraints will initially have no bounds. */
      task.appendcons(numcon);

      /* Append 'numvar' variables.
         The variables will initially be fixed at zero (x=0). */
      task.appendvars(numvar);
      /*TAG:end-append*/

      /*TAG:begin-cfix*/
      /* Optionally add a constant term to the objective. */
      task.putcfix(0.0);
      /*TAG:end-cfix*/
      for (int j = 0; j < numvar; ++j) {
        /*TAG:begin-putcj*/
        /* Set the linear term c_j in the objective.*/
        task.putcj(j, c[j]);
        /*TAG:end-putcj*/
        /*TAG:begin-putbound-var*/
        /* Set the bounds on variable j.
           blx[j] <= x_j <= bux[j] */
        task.putbound(mosek.accmode.var, j, bkx[j], blx[j], bux[j]);
        /*TAG:end-putbound-var*/
      }

      for (int j = 0; j < aval.length; ++j)
        /*TAG:begin-putavec*/
        /* Input column j of A */
        task.putacol(j,                     /* Variable (column) index.*/
                     asub[j],               /* Row index of non-zeros in column j.*/
                     aval[j]);              /* Non-zero Values of column j. */
      /*TAG:end-putavec*/

      /*TAG:begin-putbound-con*/
      /* Set the bounds on constraints.
      for i=1, ...,numcon : blc[i] <= constraint i <= buc[i] */
      for (int i = 0; i < numcon; ++i)
        task.putbound(mosek.accmode.con, i, bkc[i], blc[i], buc[i]);
      /*TAG:end-putbound-con*/

      csub[0] = 3;
      csub[1] = 0;
      csub[2] = 1;
      //TAG:begin-appendcone
      task.appendcone(mosek.conetype.quad,
                      0.0, /* For future use only, can be set to 0.0 */
                      csub);
      //TAG:end-appendcone
      csub[0] = 4;
      csub[1] = 5;
      csub[2] = 2;

      task.appendcone(mosek.conetype.rquad, 0.0, csub);

      task.putobjsense(mosek.objsense.minimize);

      System.out.println ("optimize");
      /* Solve the problem */
      /*TAG:begin-optimize*/
      mosek.rescode r = task.optimize();
      /*TAG:end-optimize*/
      System.out.println (" Mosek warning:" + r.toString());
      // Print a summary containing information
      // about the solution for debugging purposes
      task.solutionsummary(mosek.streamtype.msg);

      mosek.solsta solsta[] = new mosek.solsta[1];

      /* Get status information about the solution */
      task.getsolsta(mosek.soltype.itr, solsta);

      /*TAG:begin-getsolution*/
      task.getxx(mosek.soltype.itr, // Interior solution.
                 xx);
      /*TAG:end-getsolution*/

      switch (solsta[0]) {
        case optimal:
        case near_optimal:
          System.out.println("Optimal primal solution\n");
          for (int j = 0; j < numvar; ++j)
            System.out.println ("x[" + j + "]:" + xx[j]);
          break;
        case dual_infeas_cer:
        case prim_infeas_cer:
        case near_dual_infeas_cer:
        case near_prim_infeas_cer:
          System.out.println("Primal or dual infeasibility.\n");
          break;
        case unknown:
          System.out.println("Unknown solution status.\n");
          break;
        default:
          System.out.println("Other solution status");
          break;
      }
    } catch (mosek.Exception e) {
      System.out.println ("An error/warning was encountered");
      System.out.println (e.toString());
      throw e;
    }
  }
}
