#include "monty.h"
#include "fusion_p.h"

template<typename T> struct _checked_ptr_1
{
  T * ptr;
  size_t sz;
  _checked_ptr_1<T>(T * ptr, size_t sz) : ptr(ptr), sz(sz) {}
  T & operator[](ptrdiff_t i) { assert(i >= 0 && i < sz); return ptr[i]; }
  const T & operator[](ptrdiff_t i) const { assert(i >= 0 && i < sz); return ptr[i]; }
  void update(T * ptr_, size_t sz_) { ptr = ptr_; sz = sz_; }
};

template<typename T> struct _checked_ptr_2
{
  T * ptr;
  size_t sz0, sz1;
  _checked_ptr_2<T>(T * ptr, size_t sz0, size_t sz1) : ptr(ptr), sz0(sz0),sz1(sz1) {}
  T & operator()(ptrdiff_t i0, ptrdiff_t i1) { assert(i0 >= 0 && i0 < sz0 && i1 >= 0 && i1 < sz1); return ptr[i0 * sz1 + i1]; }
  const T & operator()(ptrdiff_t i0, ptrdiff_t i1) const { assert(i0 >= 0 && i0 < sz0 && i1 >= 0 && i1 < sz1); return ptr[i0 * sz1 + i1]; }
  void update(T * ptr_, size_t sz0_, size_t sz1_) { ptr = ptr_; sz0 = sz0_; sz1 = sz1_; }
};

// See http://www.cplusplus.com/articles/DE18T05o/
//                 A      && ( (       B      && B1 && B2      ) || (      C      &&   C1 &&   C2  ) )
// A a and b a have different signs
// B, B1, B2 a and b are both non-negative, r is less than both a and b
// C, C1, C2 a and b are both negative, r is greater than both a and b

struct InternalNumericOverflow {};

template<typename T>
static inline T checked_add(T a, T b, bool & overflow) {
  int r = a+b;
  overflow = ((a ^ b) >= 0) && ( (( (a & b) >= 0) && r < a && r < b) || ( ((a & b) < 0) && a < r && b < r) );
  return r;
}

template<typename T>
static inline T safe_add_(T a, T b) {
  bool overflow;
  T r = checked_add(a,b,overflow);
  if (overflow)
    throw InternalNumericOverflow();
  else
    return r;
}

static inline int      safe_add(int32_t a, int32_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int64_t a, int64_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int32_t a, int64_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int64_t a, int32_t b) { return safe_add_<int64_t>(a,b); }

template<typename T>
static inline T checked_mul(T a, T b, bool & overflow) {
  T r = a * b;
  overflow = b != 0 && (((a ^ b) >= 0 && a > std::numeric_limits<T>::max() / b) || ((a ^ b) < 0 && a < std::numeric_limits<T>::min() / b));
  return r;
}

template<typename T>
static inline T safe_mul_(T a, T b) {
  bool overflow;
  T r = checked_mul(a,b,overflow);
  if (overflow)
    throw InternalNumericOverflow();
  else
    return r;
}

static inline           int safe_mul(int a, int b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int64_t a, int64_t b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int a, int64_t b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int64_t a, int b) { return safe_mul_<int64_t>(a,b); }

std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::DJCDomainType val)
{
  switch (val)
  {
    case DJCDomainType::EqualTo: os << "DJCDomainType::EqualTo"; break;
    case DJCDomainType::LessThan: os << "DJCDomainType::LessThan"; break;
    case DJCDomainType::GreaterThan: os << "DJCDomainType::GreaterThan"; break;
    case DJCDomainType::IsFree: os << "DJCDomainType::IsFree"; break;
    case DJCDomainType::InRange: os << "DJCDomainType::InRange"; break;
    case DJCDomainType::InQCone: os << "DJCDomainType::InQCone"; break;
    case DJCDomainType::InRotatedQCone: os << "DJCDomainType::InRotatedQCone"; break;
    case DJCDomainType::InPExpCone: os << "DJCDomainType::InPExpCone"; break;
    case DJCDomainType::InPPowCone: os << "DJCDomainType::InPPowCone"; break;
    case DJCDomainType::InDExpCone: os << "DJCDomainType::InDExpCone"; break;
    case DJCDomainType::InDPowCone: os << "DJCDomainType::InDPowCone"; break;
    case DJCDomainType::InOneNormCone: os << "DJCDomainType::InOneNormCone"; break;
    case DJCDomainType::InInfNormCone: os << "DJCDomainType::InInfNormCone"; break;
    case DJCDomainType::InPGeoMeanCone: os << "DJCDomainType::InPGeoMeanCone"; break;
    case DJCDomainType::InDGeoMeanCone: os << "DJCDomainType::InDGeoMeanCone"; break;
    case DJCDomainType::InPSDCone: os << "DJCDomainType::InPSDCone"; break;
    default: os << "DJCDomainType.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::StatusKey val)
{
  switch (val)
  {
    case StatusKey::Unknown: os << "StatusKey::Unknown"; break;
    case StatusKey::Basic: os << "StatusKey::Basic"; break;
    case StatusKey::SuperBasic: os << "StatusKey::SuperBasic"; break;
    case StatusKey::OnBound: os << "StatusKey::OnBound"; break;
    case StatusKey::Infinity: os << "StatusKey::Infinity"; break;
    default: os << "StatusKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolutionType val)
{
  switch (val)
  {
    case SolutionType::Default: os << "SolutionType::Default"; break;
    case SolutionType::Basic: os << "SolutionType::Basic"; break;
    case SolutionType::Interior: os << "SolutionType::Interior"; break;
    case SolutionType::Integer: os << "SolutionType::Integer"; break;
    default: os << "SolutionType.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolverStatus val)
{
  switch (val)
  {
    case SolverStatus::OK: os << "SolverStatus::OK"; break;
    case SolverStatus::Error: os << "SolverStatus::Error"; break;
    case SolverStatus::LostRace: os << "SolverStatus::LostRace"; break;
    default: os << "SolverStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::ProblemStatus val)
{
  switch (val)
  {
    case ProblemStatus::Unknown: os << "ProblemStatus::Unknown"; break;
    case ProblemStatus::PrimalAndDualFeasible: os << "ProblemStatus::PrimalAndDualFeasible"; break;
    case ProblemStatus::PrimalFeasible: os << "ProblemStatus::PrimalFeasible"; break;
    case ProblemStatus::DualFeasible: os << "ProblemStatus::DualFeasible"; break;
    case ProblemStatus::PrimalInfeasible: os << "ProblemStatus::PrimalInfeasible"; break;
    case ProblemStatus::DualInfeasible: os << "ProblemStatus::DualInfeasible"; break;
    case ProblemStatus::PrimalAndDualInfeasible: os << "ProblemStatus::PrimalAndDualInfeasible"; break;
    case ProblemStatus::IllPosed: os << "ProblemStatus::IllPosed"; break;
    case ProblemStatus::PrimalInfeasibleOrUnbounded: os << "ProblemStatus::PrimalInfeasibleOrUnbounded"; break;
    default: os << "ProblemStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::AccSolutionStatus val)
{
  switch (val)
  {
    case AccSolutionStatus::Anything: os << "AccSolutionStatus::Anything"; break;
    case AccSolutionStatus::Optimal: os << "AccSolutionStatus::Optimal"; break;
    case AccSolutionStatus::Feasible: os << "AccSolutionStatus::Feasible"; break;
    case AccSolutionStatus::Certificate: os << "AccSolutionStatus::Certificate"; break;
    default: os << "AccSolutionStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolutionStatus val)
{
  switch (val)
  {
    case SolutionStatus::Undefined: os << "SolutionStatus::Undefined"; break;
    case SolutionStatus::Unknown: os << "SolutionStatus::Unknown"; break;
    case SolutionStatus::Optimal: os << "SolutionStatus::Optimal"; break;
    case SolutionStatus::Feasible: os << "SolutionStatus::Feasible"; break;
    case SolutionStatus::Certificate: os << "SolutionStatus::Certificate"; break;
    case SolutionStatus::IllposedCert: os << "SolutionStatus::IllposedCert"; break;
    default: os << "SolutionStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::ObjectiveSense val)
{
  switch (val)
  {
    case ObjectiveSense::Undefined: os << "ObjectiveSense::Undefined"; break;
    case ObjectiveSense::Minimize: os << "ObjectiveSense::Minimize"; break;
    case ObjectiveSense::Maximize: os << "ObjectiveSense::Maximize"; break;
    default: os << "ObjectiveSense.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::QConeKey val)
{
  switch (val)
  {
    case QConeKey::InQCone: os << "QConeKey::InQCone"; break;
    case QConeKey::InRotatedQCone: os << "QConeKey::InRotatedQCone"; break;
    case QConeKey::InPExpCone: os << "QConeKey::InPExpCone"; break;
    case QConeKey::InPPowCone: os << "QConeKey::InPPowCone"; break;
    case QConeKey::InPPowConeSeq: os << "QConeKey::InPPowConeSeq"; break;
    case QConeKey::InDExpCone: os << "QConeKey::InDExpCone"; break;
    case QConeKey::InDPowCone: os << "QConeKey::InDPowCone"; break;
    case QConeKey::InDPowConeSeq: os << "QConeKey::InDPowConeSeq"; break;
    case QConeKey::InPGeoMeanCone: os << "QConeKey::InPGeoMeanCone"; break;
    case QConeKey::InDGeoMeanCone: os << "QConeKey::InDGeoMeanCone"; break;
    case QConeKey::Positive: os << "QConeKey::Positive"; break;
    case QConeKey::Negative: os << "QConeKey::Negative"; break;
    case QConeKey::Unbounded: os << "QConeKey::Unbounded"; break;
    case QConeKey::Zero: os << "QConeKey::Zero"; break;
    case QConeKey::InPSDCone: os << "QConeKey::InPSDCone"; break;
    case QConeKey::InSVecPSDCone: os << "QConeKey::InSVecPSDCone"; break;
    default: os << "QConeKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::PSDKey val)
{
  switch (val)
  {
    case PSDKey::IsSymPSD: os << "PSDKey::IsSymPSD"; break;
    case PSDKey::IsTrilPSD: os << "PSDKey::IsTrilPSD"; break;
    default: os << "PSDKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::RelationKey val)
{
  switch (val)
  {
    case RelationKey::EqualsTo: os << "RelationKey::EqualsTo"; break;
    case RelationKey::LessThan: os << "RelationKey::LessThan"; break;
    case RelationKey::GreaterThan: os << "RelationKey::GreaterThan"; break;
    case RelationKey::IsFree: os << "RelationKey::IsFree"; break;
    case RelationKey::InRange: os << "RelationKey::InRange"; break;
    default: os << "RelationKey.<invalid>"; break;
  }
  return os;
}
// class Disjunction { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:350:4-355:5
mosek::fusion::Disjunction::Disjunction(mosek::fusion::p_Disjunction *_impl) : _impl(_impl) { /*std::cout << "Disjunction() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Disjunction::~Disjunction() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Disjunction::p_Disjunction
  (::mosek::fusion::Disjunction * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.Disjunction.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:352:6-354:7
mosek::fusion::Disjunction::t mosek::fusion::p_Disjunction::_new_Disjunction
  (int64_t _7_id)
{ return new mosek::fusion::Disjunction(_7_id); }
mosek::fusion::Disjunction::Disjunction
  (int64_t _7_id) : 
   Disjunction(new mosek::fusion::p_Disjunction(this))
{ mosek::fusion::p_Disjunction::_get_impl(this)->_initialize(_7_id); }
void mosek::fusion::p_Disjunction::_initialize  (int64_t _7_id)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Disjunction::_get_impl(::mosek::fusion::Disjunction::t(_pubthis))->id = _7_id;
  }
}
// mosek.fusion.Disjunction.ctor

void mosek::fusion::p_Disjunction::destroy()
{
}
void mosek::fusion::Disjunction::destroy() { mosek::fusion::p_Disjunction::_get_impl(this)->destroy(); }
// } class Disjunction
// class DisjunctionTerms { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:321:4-345:5
mosek::fusion::DisjunctionTerms::DisjunctionTerms(mosek::fusion::p_DisjunctionTerms *_impl) : _impl(_impl) { /*std::cout << "DisjunctionTerms() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DisjunctionTerms::~DisjunctionTerms() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DisjunctionTerms::p_DisjunctionTerms
  (::mosek::fusion::DisjunctionTerms * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:344:6-97
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term)
{ return new mosek::fusion::DisjunctionTerms(_8_terms1,_9_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_8_terms1,_9_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_8_terms1,::mosek::fusion::p_DJC::term(_9_term));
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:343:6-98
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term)
{ return new mosek::fusion::DisjunctionTerms(_10_terms1,_11_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_10_terms1,_11_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_10_terms1,::mosek::fusion::p_DJC::term(_11_term));
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:340:6-342:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2)
{ return new mosek::fusion::DisjunctionTerms(_12_terms1,_13_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_12_terms1,_13_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_12_terms1,mosek::fusion::p_DisjunctionTerms::_get_impl(_13_term2)->terms);
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:334:6-339:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2)
{ return new mosek::fusion::DisjunctionTerms(_14_term1,_15_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_14_term1,_15_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _16_n1 = (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0));
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(safe_add( (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0)),(int)((_15_term2)->size(0)) ))));
    {
      int32_t _17_i = (int)0;
      for(;(_17_i < (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0)));++ _17_i)
      {
        {
          ((*terms)(_17_i)) = ((*mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)(_17_i));
        }
      }
    }
    {
      int32_t _18_i = (int)0;
      for(;(_18_i < (int)((_15_term2)->size(0)));++ _18_i)
      {
        {
          ((*terms)(safe_add( _18_i,_16_n1 ))) = ((*_15_term2)(_18_i));
        }
      }
    }
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:329:6-333:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2)
{ return new mosek::fusion::DisjunctionTerms(_19_term1,_20_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_19_term1,_20_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(safe_add( (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)),(int)1 ))));
    {
      int32_t _21_i = (int)0;
      for(;(_21_i < (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)));++ _21_i)
      {
        {
          ((*terms)(_21_i)) = ((*mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)(_21_i));
        }
      }
    }
    ((*terms)((int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)))) = _20_term2;
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:326:6-328:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms)
{ return new mosek::fusion::DisjunctionTerms(_22_terms); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_22_terms); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9282_;
    for (int32_t _23_i = (int)0, _9283_ = (int)((_22_terms)->size(0)); _23_i < _9283_ ; ++_23_i)
    {
      _9282_.push_back(((*_22_terms)(_23_i)));
    }
    auto _9284_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9282_.size()),_9282_.begin(),_9282_.end()));
    mosek::fusion::p_DisjunctionTerms::_get_impl(::mosek::fusion::DisjunctionTerms::t(_pubthis))->terms = _9284_;
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:325:6-79
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term)
{ return new mosek::fusion::DisjunctionTerms(_24_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_24_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(::mosek::fusion::p_DJC::term(_24_term).get())});
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:323:6-80
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term)
{ return new mosek::fusion::DisjunctionTerms(_25_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_25_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(::mosek::fusion::p_DJC::term(_25_term).get())});
  }
}
// mosek.fusion.DisjunctionTerms.ctor

void mosek::fusion::p_DisjunctionTerms::destroy()
{
  terms.reset();
}
void mosek::fusion::DisjunctionTerms::destroy() { mosek::fusion::p_DisjunctionTerms::_get_impl(this)->destroy(); }
// } class DisjunctionTerms
// class Term { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:243:4-317:5
mosek::fusion::Term::Term(mosek::fusion::p_Term *_impl) : _impl(_impl) { /*std::cout << "Term() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Term::~Term() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Term::p_Term
  (::mosek::fusion::Term * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.Term.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:261:6-269:7
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist)
{ return new mosek::fusion::Term(_26_elist,_27_dlist); }
mosek::fusion::Term::Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_26_elist,_27_dlist); }
void mosek::fusion::p_Term::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _28_n = (int)((_26_elist)->size(0));
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_28_n)));
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_28_n)));
    int32_t _9285_ = (int)0;
    int32_t _9286_ = _28_n;
    for (int32_t _29_i = _9285_; _29_i < _9286_; ++_29_i)
    {
      {
        ((*exprs)(_29_i)) = ((*_26_elist)(_29_i));
        ((*domains)(_29_i)) = ((*_27_dlist)(_29_i));
      }
    }
  }
}
// mosek.fusion.Term.ctor

// mosek.fusion.Term.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:259:6-90
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d)
{ return new mosek::fusion::Term(_30_e,_31_d); }
mosek::fusion::Term::Term
  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_30_e,_31_d); }
void mosek::fusion::p_Term::_initialize  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >{::monty::rc_ptr<::mosek::fusion::Expression>(_30_e.get())});
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >{::monty::rc_ptr<::mosek::fusion::DJCDomain>(_31_d.get())});
  }
}
// mosek.fusion.Term.ctor

// mosek.fusion.Term.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:247:6-255:7
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t)
{ return new mosek::fusion::Term(_32_t); }
mosek::fusion::Term::Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_32_t); }
void mosek::fusion::p_Term::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _33_n = (int)((_32_t)->size(0));
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_33_n)));
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_33_n)));
    int32_t _9287_ = (int)0;
    int32_t _9288_ = _33_n;
    for (int32_t _34_i = _9287_; _34_i < _9288_; ++_34_i)
    {
      {
        ((*exprs)(_34_i)) = ((*mosek::fusion::p_Term::_get_impl(((*_32_t)(_34_i)))->exprs)((int)0));
        ((*domains)(_34_i)) = ((*mosek::fusion::p_Term::_get_impl(((*_32_t)(_34_i)))->domains)((int)0));
      }
    }
  }
}
// mosek.fusion.Term.ctor

// Begin mosek.fusion.Term.numaccterms
// Method mosek.fusion.Term.numaccterms @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:316:6-128
int32_t mosek::fusion::p_Term::numaccterms()
{
  int32_t _35_n = (int)0;
  int32_t _9289_ = (int)0;
  int32_t _9290_ = (int)((domains)->size(0));
  for (int32_t _36_i = _9289_; _36_i < _9290_; ++_36_i)
  {
    {
      _35_n += mosek::fusion::p_DJCDomain::_get_impl(((*domains)(_36_i)).get())->numaccterms();
    }
  }
  return _35_n;
}
// End mosek.fusion.Term.numaccterms

// Begin mosek.fusion.Term.numaccrows
// Method mosek.fusion.Term.numaccrows @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:315:6-127
int32_t mosek::fusion::p_Term::numaccrows()
{
  int32_t _37_n = (int)0;
  int32_t _9291_ = (int)0;
  int32_t _9292_ = (int)((domains)->size(0));
  for (int32_t _38_i = _9291_; _38_i < _9292_; ++_38_i)
  {
    {
      _37_n += mosek::fusion::p_DJCDomain::_get_impl(((*domains)(_38_i)).get())->numaccrows();
    }
  }
  return _37_n;
}
// End mosek.fusion.Term.numaccrows

// Begin mosek.fusion.Term.num
// Method mosek.fusion.Term.num @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:314:6-67
int32_t mosek::fusion::p_Term::num()
{
  return (int)((domains)->size(0));
}
// End mosek.fusion.Term.num

// Begin mosek.fusion.Term.size
// Method mosek.fusion.Term.size @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:313:6-121
int32_t mosek::fusion::Term :: size() { return mosek::fusion::p_Term::_get_impl(this)->size(); }
int32_t mosek::fusion::p_Term::size()
{
  int32_t _39_s = (int)0;
  int32_t _9293_ = (int)0;
  int32_t _9294_ = (int)((domains)->size(0));
  for (int32_t _40_i = _9293_; _40_i < _9294_; ++_40_i)
  {
    {
      _39_s += ((*domains)(_40_i))->size();
    }
  }
  return _39_s;
}
// End mosek.fusion.Term.size

// Begin mosek.fusion.Term.toDJCTerm
// Method mosek.fusion.Term.toDJCTerm @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:257:6-53
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term :: __mosek_2fusion_2Term__toDJCTerm() { return mosek::fusion::p_Term::_get_impl(this)->__mosek_2fusion_2Term__toDJCTerm(); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term::toDJCTerm() { return __mosek_2fusion_2Term__toDJCTerm(); }
/* override: mosek.fusion.ExprDomain.toDJCTerm*/
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term::__mosek_2fusion_2ExprDomain__toDJCTerm() { return __mosek_2fusion_2Term__toDJCTerm(); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_Term::__mosek_2fusion_2Term__toDJCTerm()
{
  return ::mosek::fusion::Term::t(_pubthis);
}
// End mosek.fusion.Term.toDJCTerm

void mosek::fusion::p_Term::destroy()
{
  domains.reset();
  exprs.reset();
}
void mosek::fusion::Term::destroy() { mosek::fusion::p_Term::_get_impl(this)->destroy(); }
// } class Term
// class SimpleTerm { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:228:4-230:5
mosek::fusion::SimpleTerm::SimpleTerm(mosek::fusion::p_SimpleTerm *_impl) : mosek::fusion::Term(_impl) { /*std::cout << "SimpleTerm()" << this << std::endl; */ }
mosek::fusion::SimpleTerm::~SimpleTerm() {  /* std::cout << "~SimpleTerm()" << this << std::endl << std::flush; */ }
mosek::fusion::p_SimpleTerm::p_SimpleTerm
  (::mosek::fusion::SimpleTerm * _pubthis) :     mosek::fusion::p_Term(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.SimpleTerm.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:229:6-64
mosek::fusion::SimpleTerm::t mosek::fusion::p_SimpleTerm::_new_SimpleTerm
  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d)
{ return new mosek::fusion::SimpleTerm(_41_e,_42_d); }
mosek::fusion::SimpleTerm::SimpleTerm
  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d) : 
   SimpleTerm(new mosek::fusion::p_SimpleTerm(this))
{ mosek::fusion::p_SimpleTerm::_get_impl(this)->_initialize(_41_e,_42_d); }
void mosek::fusion::p_SimpleTerm::_initialize  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Term::_initialize(_41_e,_42_d);
  }
}
// mosek.fusion.SimpleTerm.ctor

void mosek::fusion::p_SimpleTerm::destroy()
{
}
void mosek::fusion::SimpleTerm::destroy() { mosek::fusion::p_SimpleTerm::_get_impl(this)->destroy(); }
// } class SimpleTerm
// class DJCDomain { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:171:4-217:5
mosek::fusion::DJCDomain::DJCDomain(mosek::fusion::p_DJCDomain *_impl) : _impl(_impl) { /*std::cout << "DJCDomain() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DJCDomain::~DJCDomain() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DJCDomain::p_DJCDomain
  (::mosek::fusion::DJCDomain * _pubthis) :     _pubthis(_pubthis)
{}
mosek::fusion::DJCDomainType mosek::fusion::DJCDomain::get_dom() { return _impl->dom; }
void mosek::fusion::DJCDomain::set_dom(mosek::fusion::DJCDomainType _val) { _impl->dom = _val; }
int32_t mosek::fusion::DJCDomain::get_conedim() { return _impl->conedim; }
void mosek::fusion::DJCDomain::set_conedim(int32_t _val) { _impl->conedim = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::DJCDomain::get_shape() { return _impl->shape; }
void mosek::fusion::DJCDomain::set_shape(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->shape = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::DJCDomain::get_par() { return _impl->par; }
void mosek::fusion::DJCDomain::set_par(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->par = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::DJCDomain::get_b() { return _impl->b; }
void mosek::fusion::DJCDomain::set_b(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->b = _val; }
// mosek.fusion.DJCDomain.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:190:6-199:7
mosek::fusion::DJCDomain::t mosek::fusion::p_DJCDomain::_new_DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_)
{ return new mosek::fusion::DJCDomain(_43_b_,_44_par_,_45_shape_,_46_dom_); }
mosek::fusion::DJCDomain::DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_) : 
   DJCDomain(new mosek::fusion::p_DJCDomain(this))
{ mosek::fusion::p_DJCDomain::_get_impl(this)->_initialize(_43_b_,_44_par_,_45_shape_,_46_dom_); }
void mosek::fusion::p_DJCDomain::_initialize  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<double> _ptr__43_b_(_43_b_ ? _43_b_->raw() : nullptr, _43_b_ ? _43_b_->size(0) : 0);
    _checked_ptr_1<double> _ptr__44_par_(_44_par_ ? _44_par_->raw() : nullptr, _44_par_ ? _44_par_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__45_shape_(_45_shape_ ? _45_shape_->raw() : nullptr, _45_shape_ ? _45_shape_->size(0) : 0);
    b = _43_b_;
    par = _44_par_;
    shape = _45_shape_;
    dom = _46_dom_;
    conedim = ((int)((_45_shape_)->size(0)) - (int)1);
  }
}
// mosek.fusion.DJCDomain.ctor

// mosek.fusion.DJCDomain.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:178:6-188:7
mosek::fusion::DJCDomain::t mosek::fusion::p_DJCDomain::_new_DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_)
{ return new mosek::fusion::DJCDomain(_47_b_,_48_par_,_49_shape_,_50_conedim_,_51_dom_); }
mosek::fusion::DJCDomain::DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_) : 
   DJCDomain(new mosek::fusion::p_DJCDomain(this))
{ mosek::fusion::p_DJCDomain::_get_impl(this)->_initialize(_47_b_,_48_par_,_49_shape_,_50_conedim_,_51_dom_); }
void mosek::fusion::p_DJCDomain::_initialize  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<double> _ptr__47_b_(_47_b_ ? _47_b_->raw() : nullptr, _47_b_ ? _47_b_->size(0) : 0);
    _checked_ptr_1<double> _ptr__48_par_(_48_par_ ? _48_par_->raw() : nullptr, _48_par_ ? _48_par_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__49_shape_(_49_shape_ ? _49_shape_->raw() : nullptr, _49_shape_ ? _49_shape_->size(0) : 0);
    b = _47_b_;
    par = _48_par_;
    shape = _49_shape_;
    dom = _51_dom_;
    conedim = _50_conedim_;
  }
}
// mosek.fusion.DJCDomain.ctor

// Begin mosek.fusion.DJCDomain.numaccterms
// Method mosek.fusion.DJCDomain.numaccterms @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:211:6-216:7
int32_t mosek::fusion::p_DJCDomain::numaccterms()
{
  if ((dom == ::mosek::fusion::DJCDomainType::InRange))
  {
    {
      return (int)2;
    }
  }
  else
  {
    {
      return (int)1;
    }
  }
}
// End mosek.fusion.DJCDomain.numaccterms

// Begin mosek.fusion.DJCDomain.numaccrows
// Method mosek.fusion.DJCDomain.numaccrows @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:205:6-210:7
int32_t mosek::fusion::p_DJCDomain::numaccrows()
{
  if ((dom == ::mosek::fusion::DJCDomainType::InRange))
  {
    {
      return safe_mul( _pubthis->size(),(int)2 );
    }
  }
  else
  {
    {
      return _pubthis->size();
    }
  }
}
// End mosek.fusion.DJCDomain.numaccrows

// Begin mosek.fusion.DJCDomain.size
// Method mosek.fusion.DJCDomain.size @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:201:6-204:7
int32_t mosek::fusion::DJCDomain :: size() { return mosek::fusion::p_DJCDomain::_get_impl(this)->size(); }
int32_t mosek::fusion::p_DJCDomain::size()
{
  int32_t _52_s = (int)1;
  int32_t _9295_ = (int)0;
  int32_t _9296_ = (int)((shape)->size(0));
  for (int32_t _53_i = _9295_; _53_i < _9296_; ++_53_i)
  {
    {
      _52_s *= ((*shape)(_53_i));
    }
  }
  return _52_s;
}
// End mosek.fusion.DJCDomain.size

void mosek::fusion::p_DJCDomain::destroy()
{
  shape.reset();
  par.reset();
  b.reset();
}
void mosek::fusion::DJCDomain::destroy() { mosek::fusion::p_DJCDomain::_get_impl(this)->destroy(); }
// } class DJCDomain
// class DJC { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:32:4-139:5
mosek::fusion::DJC::DJC(mosek::fusion::p_DJC *_impl) : _impl(_impl) { /*std::cout << "DJC() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DJC::~DJC() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DJC::p_DJC
  (::mosek::fusion::DJC * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.DJC.ANDFromTerms
// Method mosek.fusion.DJC.ANDFromTerms @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:121:6-137:7
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::ANDFromTerms
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _54_tlist) { return mosek::fusion::p_DJC::ANDFromTerms(_54_tlist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::ANDFromTerms(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _54_tlist)
{
  int32_t _55_n = (int)0;
  int32_t _9297_ = (int)0;
  int32_t _9298_ = (int)((_54_tlist)->size(0));
  for (int32_t _56_i = _9297_; _56_i < _9298_; ++_56_i)
  {
    {
      _55_n += mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_56_i)).get())->num();
    }
  }
  std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _57_el = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_55_n)));
  std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _58_dl = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_55_n)));
  _55_n = (int)0;
  int32_t _9299_ = (int)0;
  int32_t _9300_ = (int)((_54_tlist)->size(0));
  for (int32_t _59_i = _9299_; _59_i < _9300_; ++_59_i)
  {
    {
      int32_t _9301_ = (int)0;
      int32_t _9302_ = mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)).get())->num();
      for (int32_t _60_j = _9301_; _60_j < _9302_; ++_60_j)
      {
        {
          ((*_57_el)(_55_n)) = ((*mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)))->exprs)(_60_j));
          ((*_58_dl)(_55_n)) = ((*mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)))->domains)(_60_j));
          _55_n += (int)1;
        }
      }
    }
  }
  return ::mosek::fusion::p_Term::_new_Term(_57_el,_58_dl);
}
// End mosek.fusion.DJC.ANDFromTerms

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:116:6-136
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _61_elist) { return mosek::fusion::p_DJC::AND(_61_elist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _61_elist)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9303_;
  for (int32_t _62_i = (int)0, _9304_ = (int)((_61_elist)->size(0)); _62_i < _9304_ ; ++_62_i)
  {
    _9303_.push_back(((*_61_elist)(_62_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9305_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9303_.size()),_9303_.begin(),_9303_.end()));
  return ::mosek::fusion::p_DJC::ANDFromTerms(_9305_);
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:114:6-114
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _63_s1,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _64_s2,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _65_s3) { return mosek::fusion::p_DJC::AND(_63_s1,_64_s2,_65_s3); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _63_s1,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _64_s2,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _65_s3)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_63_s1.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_64_s2.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_65_s3.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:113:6-96
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _66_s1,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _67_s2) { return mosek::fusion::p_DJC::AND(_66_s1,_67_s2); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _66_s1,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _67_s2)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_66_s1.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_67_s2.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:112:6-78
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _68_s1) { return mosek::fusion::p_DJC::AND(_68_s1); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _68_s1)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_68_s1.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:111:6-84
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _69_slist) { return mosek::fusion::p_DJC::AND(_69_slist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _69_slist)
{
  return ::mosek::fusion::p_Term::_new_Term(_69_slist);
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:80:6-109:7
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Expression > _70_expr,
    monty::rc_ptr< ::mosek::fusion::RangeDomain > _71_dom) { return mosek::fusion::p_DJC::term(_70_expr,_71_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Expression > _70_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _71_dom)
{
  if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->sparsity.get()))
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Sparse domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  if (mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _72_domlb = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->lb;
  _checked_ptr_1<double> _ptr__72_domlb(_72_domlb ? _72_domlb->raw() : nullptr,_72_domlb ? _72_domlb->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _73_domub = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->ub;
  _checked_ptr_1<double> _ptr__73_domub(_73_domub ? _73_domub->raw() : nullptr,_73_domub ? _73_domub->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _74_eshape = _70_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__74_eshape(_74_eshape ? _74_eshape->raw() : nullptr,_74_eshape ? _74_eshape->size(0) : 0);
  int64_t _75_esize = _70_expr->getSize();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _76_shape = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__76_shape(_76_shape ? _76_shape->raw() : nullptr,_76_shape ? _76_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _77_bl;
  _checked_ptr_1<double> _ptr__77_bl(_77_bl ? _77_bl->raw() : nullptr,_77_bl ? _77_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _78_bu;
  _checked_ptr_1<double> _ptr__78_bu(_78_bu ? _78_bu->raw() : nullptr,_78_bu ? _78_bu->size(0) : 0);
  if ((NULL == _76_shape.get()))
  {
    {
      _76_shape = _74_eshape;
      _ptr__76_shape.update(_76_shape ? _76_shape->raw() : nullptr, _76_shape ? _76_shape->size(0) : 0);
    }
  }
  else
  {
    {
      if (((int)((_76_shape)->size(0)) != (int)((_74_eshape)->size(0))))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
        }
      }
      else
      {
        {
          int32_t _9306_ = (int)0;
          int32_t _9307_ = (int)((_76_shape)->size(0));
          for (int32_t _79_i = _9306_; _79_i < _9307_; ++_79_i)
          {
            {
              if ((_ptr__76_shape[_79_i] != _ptr__74_eshape[_79_i]))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  if (mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->scalable)
  {
    {
      std::vector<double> _9308_;
      for (int64_t _80_i = (int)0, _9309_ = _75_esize; _80_i < _9309_ ; ++_80_i)
      {
        _9308_.push_back(_ptr__72_domlb[(int)0]);
      }
      auto _9310_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9308_.size()),_9308_.begin(),_9308_.end()));
      _77_bl = _9310_;
      _ptr__77_bl.update(_77_bl ? _77_bl->raw() : nullptr, _77_bl ? _77_bl->size(0) : 0);
      std::vector<double> _9311_;
      for (int64_t _81_i = (int)0, _9312_ = _75_esize; _81_i < _9312_ ; ++_81_i)
      {
        _9311_.push_back(_ptr__73_domub[(int)0]);
      }
      auto _9313_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9311_.size()),_9311_.begin(),_9311_.end()));
      _78_bu = _9313_;
      _ptr__78_bu.update(_78_bu ? _78_bu->raw() : nullptr, _78_bu ? _78_bu->size(0) : 0);
    }
  }
  else
  {
    {
      _77_bl = _72_domlb;
      _ptr__77_bl.update(_77_bl ? _77_bl->raw() : nullptr, _77_bl ? _77_bl->size(0) : 0);
      _78_bu = _73_domub;
      _ptr__78_bu.update(_78_bu ? _78_bu->raw() : nullptr, _78_bu ? _78_bu->size(0) : 0);
    }
  }
  return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_70_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_77_bl,_78_bu,_76_shape,::mosek::fusion::DJCDomainType::InRange));
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:78:6-105
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Variable > _82_x,
    monty::rc_ptr< ::mosek::fusion::RangeDomain > _83_dom) { return mosek::fusion::p_DJC::term(_82_x,_83_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Variable > _82_x,monty::rc_ptr< ::mosek::fusion::RangeDomain > _83_dom)
{
  return ::mosek::fusion::p_DJC::term(_82_x->__mosek_2fusion_2Variable__asExpr(),_83_dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:76:6-110
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _84_exprdom) { return mosek::fusion::p_DJC::term(_84_exprdom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _84_exprdom)
{
  return ::mosek::fusion::p_DJC::term(mosek::fusion::p_ExprRangeDomain::_get_impl(_84_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_84_exprdom)->dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:41:6-73:7
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Expression > _85_expr,
    monty::rc_ptr< ::mosek::fusion::LinearDomain > _86_dom) { return mosek::fusion::p_DJC::term(_85_expr,_86_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Expression > _85_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _86_dom)
{
  if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->sparsity.get()))
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Sparse domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _87_dombnd = mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->bnd;
  _checked_ptr_1<double> _ptr__87_dombnd(_87_dombnd ? _87_dombnd->raw() : nullptr,_87_dombnd ? _87_dombnd->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _88_eshape = _85_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__88_eshape(_88_eshape ? _88_eshape->raw() : nullptr,_88_eshape ? _88_eshape->size(0) : 0);
  int64_t _89_esize = _85_expr->getSize();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _90_shape = mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__90_shape(_90_shape ? _90_shape->raw() : nullptr,_90_shape ? _90_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _91_b;
  _checked_ptr_1<double> _ptr__91_b(_91_b ? _91_b->raw() : nullptr,_91_b ? _91_b->size(0) : 0);
  if ((NULL == _90_shape.get()))
  {
    {
      _90_shape = _88_eshape;
      _ptr__90_shape.update(_90_shape ? _90_shape->raw() : nullptr, _90_shape ? _90_shape->size(0) : 0);
    }
  }
  else
  {
    {
      if (((int)((_90_shape)->size(0)) != (int)((_88_eshape)->size(0))))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
        }
      }
      else
      {
        {
          int32_t _9314_ = (int)0;
          int32_t _9315_ = (int)((_90_shape)->size(0));
          for (int32_t _92_i = _9314_; _92_i < _9315_; ++_92_i)
          {
            {
              if ((_ptr__90_shape[_92_i] != _ptr__88_eshape[_92_i]))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  if ((NULL == mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->bnd.get()))
  {
    {
      _91_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_89_esize)));
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else if(mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->scalable)
  {
    {
      std::vector<double> _9316_;
      for (int64_t _93_i = (int)0, _9317_ = _89_esize; _93_i < _9317_ ; ++_93_i)
      {
        _9316_.push_back(_ptr__87_dombnd[(int)0]);
      }
      auto _9318_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9316_.size()),_9316_.begin(),_9316_.end()));
      _91_b = _9318_;
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else if(mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->empty)
  {
    {
      _91_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_89_esize)));
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else
  {
    {
      _91_b = _87_dombnd;
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  if ((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::EqualTo));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::LessThan));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::GreaterThan));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::IsFree));
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Invalid linear domain"));
    }
  }
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:36:6-111
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _94_exprdom) { return mosek::fusion::p_DJC::term(_94_exprdom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _94_exprdom)
{
  return ::mosek::fusion::p_DJC::term(mosek::fusion::p_ExprLinearDomain::_get_impl(_94_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_94_exprdom)->dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Disjunction.mbi:33:6-105
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Variable > _95_x,
    monty::rc_ptr< ::mosek::fusion::LinearDomain > _96_dom) { return mosek::fusion::p_DJC::term(_95_x,_96_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Variable > _95_x,monty::rc_ptr< ::mosek::fusion::LinearDomain > _96_dom)
{
  return ::mosek::fusion::p_DJC::term(_95_x->__mosek_2fusion_2Variable__asExpr(),_96_dom);
}
// End mosek.fusion.DJC.term

void mosek::fusion::p_DJC::destroy()
{
}
void mosek::fusion::DJC::destroy() { mosek::fusion::p_DJC::_get_impl(this)->destroy(); }
// } class DJC
// class Sort { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:6:4-262:5
mosek::fusion::Sort::Sort(mosek::fusion::p_Sort *_impl) : _impl(_impl) { /*std::cout << "Sort() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Sort::~Sort() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Sort::p_Sort
  (::mosek::fusion::Sort * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.Sort.argTransposeSort
// Method mosek.fusion.Sort.argTransposeSort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:249:6-261:7
void mosek::fusion::Sort::argTransposeSort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _309_perm,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _310_ptrb,
    int32_t _311_m,
    int32_t _312_n,
    int32_t _313_p,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _314_val) { mosek::fusion::p_Sort::argTransposeSort(_309_perm,_310_ptrb,_311_m,_312_n,_313_p,_314_val); }
void mosek::fusion::p_Sort::argTransposeSort(std::shared_ptr< monty::ndarray< int64_t,1 > > _309_perm,std::shared_ptr< monty::ndarray< int64_t,1 > > _310_ptrb,int32_t _311_m,int32_t _312_n,int32_t _313_p,std::shared_ptr< monty::ndarray< int64_t,1 > > _314_val)
{
  _checked_ptr_1<int64_t> _ptr__309_perm(_309_perm ? _309_perm->raw() : nullptr, _309_perm ? _309_perm->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__310_ptrb(_310_ptrb ? _310_ptrb->raw() : nullptr, _310_ptrb ? _310_ptrb->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__314_val(_314_val ? _314_val->raw() : nullptr, _314_val ? _314_val->size(0) : 0);
  int32_t _9319_ = (int)0;
  int32_t _9320_ = safe_add( safe_mul( _311_m,_313_p ),(int)1 );
  for (int32_t _315_q = _9319_; _315_q < _9320_; ++_315_q)
  {
    {
      _ptr__310_ptrb[_315_q] = (int)0;
    }
  }
  int32_t _9321_ = (int)0;
  int32_t _9322_ = (int)((_314_val)->size(0));
  for (int32_t _316_q = _9321_; _316_q < _9322_; ++_316_q)
  {
    {
      ++ _ptr__310_ptrb[safe_add( safe_add( safe_mul( (_ptr__314_val[_ptr__309_perm[_316_q]] / safe_mul( _312_n,_313_p )),_313_p ),(_ptr__314_val[_ptr__309_perm[_316_q]] % _313_p) ),(int)1 )];
    }
  }
  int32_t _9323_ = (int)0;
  int32_t _9324_ = safe_mul( _311_m,_313_p );
  for (int32_t _317_i = _9323_; _317_i < _9324_; ++_317_i)
  {
    {
      _ptr__310_ptrb[safe_add( _317_i,(int)1 )] = safe_add( _ptr__310_ptrb[safe_add( _317_i,(int)1 )],_ptr__310_ptrb[_317_i] );
    }
  }
  std::vector<int64_t> _9325_;
  for (int32_t _319_i = (int)0, _9326_ = (int)((_309_perm)->size(0)); _319_i < _9326_ ; ++_319_i)
  {
    _9325_.push_back(_ptr__309_perm[_319_i]);
  }
  auto _9327_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9325_.size()),_9325_.begin(),_9325_.end()));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _318_per2 = _9327_;
  _checked_ptr_1<int64_t> _ptr__318_per2(_318_per2 ? _318_per2->raw() : nullptr,_318_per2 ? _318_per2->size(0) : 0);
  int64_t _9328_ = (int64_t)0;
  int64_t _9329_ = (int64_t)((_314_val)->size(0));
  for (int64_t _320_q = _9328_; _320_q < _9329_; ++_320_q)
  {
    {
      int64_t _321_k = safe_add( safe_mul( (_ptr__314_val[_ptr__318_per2[_320_q]] / safe_mul( _312_n,_313_p )),_313_p ),(_ptr__314_val[_ptr__318_per2[_320_q]] % _313_p) );
      _ptr__309_perm[_ptr__310_ptrb[_321_k]] = _ptr__318_per2[_320_q];
      ++ _ptr__310_ptrb[_321_k];
    }
  }
}
// End mosek.fusion.Sort.argTransposeSort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:207:6-210:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _322_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _323_vals1) { mosek::fusion::p_Sort::argsort(_322_idx,_323_vals1); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _322_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _323_vals1)
{
  _checked_ptr_1<int64_t> _ptr__322_idx(_322_idx ? _322_idx->raw() : nullptr, _322_idx ? _322_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__323_vals1(_323_vals1 ? _323_vals1->raw() : nullptr, _323_vals1 ? _323_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_322_idx,_323_vals1,(int64_t)((int)0),(int64_t)((int)((_322_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:202:6-205:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _324_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _325_vals1) { mosek::fusion::p_Sort::argsort(_324_idx,_325_vals1); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _324_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _325_vals1)
{
  _checked_ptr_1<int64_t> _ptr__324_idx(_324_idx ? _324_idx->raw() : nullptr, _324_idx ? _324_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__325_vals1(_325_vals1 ? _325_vals1->raw() : nullptr, _325_vals1 ? _325_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_324_idx,_325_vals1,(int64_t)((int)0),(int64_t)((int)((_324_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:197:6-200:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _326_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _327_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _328_vals2) { mosek::fusion::p_Sort::argsort(_326_idx,_327_vals1,_328_vals2); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _326_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _327_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _328_vals2)
{
  _checked_ptr_1<int64_t> _ptr__326_idx(_326_idx ? _326_idx->raw() : nullptr, _326_idx ? _326_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__327_vals1(_327_vals1 ? _327_vals1->raw() : nullptr, _327_vals1 ? _327_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__328_vals2(_328_vals2 ? _328_vals2->raw() : nullptr, _328_vals2 ? _328_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_326_idx,_327_vals1,_328_vals2,(int64_t)((int)0),(int64_t)((int)((_326_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:193:6-196:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _329_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _330_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _331_vals2) { mosek::fusion::p_Sort::argsort(_329_idx,_330_vals1,_331_vals2); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _329_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _330_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _331_vals2)
{
  _checked_ptr_1<int64_t> _ptr__329_idx(_329_idx ? _329_idx->raw() : nullptr, _329_idx ? _329_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__330_vals1(_330_vals1 ? _330_vals1->raw() : nullptr, _330_vals1 ? _330_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__331_vals2(_331_vals2 ? _331_vals2->raw() : nullptr, _331_vals2 ? _331_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_329_idx,_330_vals1,_331_vals2,(int64_t)((int)0),(int64_t)((int)((_329_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:188:6-191:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _332_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _333_vals1,
    int64_t _334_first,
    int64_t _335_last) { mosek::fusion::p_Sort::argsort(_332_idx,_333_vals1,_334_first,_335_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _332_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _333_vals1,int64_t _334_first,int64_t _335_last)
{
  _checked_ptr_1<int64_t> _ptr__332_idx(_332_idx ? _332_idx->raw() : nullptr, _332_idx ? _332_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__333_vals1(_333_vals1 ? _333_vals1->raw() : nullptr, _333_vals1 ? _333_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_332_idx,_333_vals1,_334_first,_335_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:183:6-186:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _336_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _337_vals1,
    int64_t _338_first,
    int64_t _339_last) { mosek::fusion::p_Sort::argsort(_336_idx,_337_vals1,_338_first,_339_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _336_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _337_vals1,int64_t _338_first,int64_t _339_last)
{
  _checked_ptr_1<int64_t> _ptr__336_idx(_336_idx ? _336_idx->raw() : nullptr, _336_idx ? _336_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__337_vals1(_337_vals1 ? _337_vals1->raw() : nullptr, _337_vals1 ? _337_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_336_idx,_337_vals1,_338_first,_339_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:178:6-181:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _340_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _341_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _342_vals2,
    int64_t _343_first,
    int64_t _344_last) { mosek::fusion::p_Sort::argsort(_340_idx,_341_vals1,_342_vals2,_343_first,_344_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _340_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _341_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _342_vals2,int64_t _343_first,int64_t _344_last)
{
  _checked_ptr_1<int64_t> _ptr__340_idx(_340_idx ? _340_idx->raw() : nullptr, _340_idx ? _340_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__341_vals1(_341_vals1 ? _341_vals1->raw() : nullptr, _341_vals1 ? _341_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__342_vals2(_342_vals2 ? _342_vals2->raw() : nullptr, _342_vals2 ? _342_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_340_idx,_341_vals1,_342_vals2,_343_first,_344_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:174:6-177:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _345_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _346_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _347_vals2,
    int64_t _348_first,
    int64_t _349_last) { mosek::fusion::p_Sort::argsort(_345_idx,_346_vals1,_347_vals2,_348_first,_349_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _345_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _346_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _347_vals2,int64_t _348_first,int64_t _349_last)
{
  _checked_ptr_1<int64_t> _ptr__345_idx(_345_idx ? _345_idx->raw() : nullptr, _345_idx ? _345_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__346_vals1(_346_vals1 ? _346_vals1->raw() : nullptr, _346_vals1 ? _346_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__347_vals2(_347_vals2 ? _347_vals2->raw() : nullptr, _347_vals2 ? _347_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_345_idx,_346_vals1,_347_vals2,_348_first,_349_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:156:6-171:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _350_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _351_vals1,
    int64_t _352_first,
    int64_t _353_last,
    bool _354_check) { mosek::fusion::p_Sort::argsort(_350_idx,_351_vals1,_352_first,_353_last,_354_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _350_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _351_vals1,int64_t _352_first,int64_t _353_last,bool _354_check)
{
  _checked_ptr_1<int64_t> _ptr__350_idx(_350_idx ? _350_idx->raw() : nullptr, _350_idx ? _350_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__351_vals1(_351_vals1 ? _351_vals1->raw() : nullptr, _351_vals1 ? _351_vals1->size(0) : 0);
  if (((!_354_check) || ((_353_last - _352_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_350_idx,_351_vals1,_352_first,_353_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_350_idx,_351_vals1,_352_first,_353_last,_354_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _355_minmax = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int64_t> _ptr__355_minmax(_355_minmax ? _355_minmax->raw() : nullptr,_355_minmax ? _355_minmax->size(0) : 0);
      int32_t _356_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_350_idx,_351_vals1,nullptr,_352_first,_353_last,_355_minmax);
      if (((_ptr__355_minmax[(int)1] - _ptr__355_minmax[(int)0]) < safe_mul( _356_scale,(_353_last - _352_first) )))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_350_idx,_351_vals1,_352_first,_353_last,_ptr__355_minmax[(int)0],_ptr__355_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_350_idx,_351_vals1,_352_first,_353_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:137:6-154:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _357_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _358_vals1,
    int64_t _359_first,
    int64_t _360_last,
    bool _361_check) { mosek::fusion::p_Sort::argsort(_357_idx,_358_vals1,_359_first,_360_last,_361_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _357_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _358_vals1,int64_t _359_first,int64_t _360_last,bool _361_check)
{
  _checked_ptr_1<int64_t> _ptr__357_idx(_357_idx ? _357_idx->raw() : nullptr, _357_idx ? _357_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__358_vals1(_358_vals1 ? _358_vals1->raw() : nullptr, _358_vals1 ? _358_vals1->size(0) : 0);
  if (((!_361_check) || ((_360_last - _359_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_357_idx,_358_vals1,_359_first,_360_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_357_idx,_358_vals1,_359_first,_360_last,_361_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _362_minmax = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int32_t> _ptr__362_minmax(_362_minmax ? _362_minmax->raw() : nullptr,_362_minmax ? _362_minmax->size(0) : 0);
      int32_t _363_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_357_idx,_358_vals1,nullptr,_359_first,_360_last,_362_minmax);
      if (((_ptr__362_minmax[(int)1] - _ptr__362_minmax[(int)0]) < safe_mul( _363_scale,(_360_last - _359_first) )))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_357_idx,_358_vals1,_359_first,_360_last,_ptr__362_minmax[(int)0],_ptr__362_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_357_idx,_358_vals1,_359_first,_360_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:116:6-135:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _364_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _365_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _366_vals2,
    int64_t _367_first,
    int64_t _368_last,
    bool _369_check) { mosek::fusion::p_Sort::argsort(_364_idx,_365_vals1,_366_vals2,_367_first,_368_last,_369_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _364_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _365_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _366_vals2,int64_t _367_first,int64_t _368_last,bool _369_check)
{
  _checked_ptr_1<int64_t> _ptr__364_idx(_364_idx ? _364_idx->raw() : nullptr, _364_idx ? _364_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__365_vals1(_365_vals1 ? _365_vals1->raw() : nullptr, _365_vals1 ? _365_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__366_vals2(_366_vals2 ? _366_vals2->raw() : nullptr, _366_vals2 ? _366_vals2->size(0) : 0);
  if (((!_369_check) || ((_368_last - _367_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_364_idx,_365_vals1,_366_vals2,_367_first,_368_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_364_idx,_365_vals1,_366_vals2,_367_first,_368_last,_369_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _370_minmax = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int64_t> _ptr__370_minmax(_370_minmax ? _370_minmax->raw() : nullptr,_370_minmax ? _370_minmax->size(0) : 0);
      int32_t _371_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_364_idx,_365_vals1,_366_vals2,_367_first,_368_last,_370_minmax);
      if ((((_ptr__370_minmax[(int)1] - _ptr__370_minmax[(int)0]) < safe_mul( _371_scale,(_368_last - _367_first) )) && ((_ptr__370_minmax[(int)3] - _ptr__370_minmax[(int)2]) < safe_mul( _371_scale,(_368_last - _367_first) ))))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_364_idx,_366_vals2,_367_first,_368_last,_ptr__370_minmax[(int)2],_ptr__370_minmax[(int)3]);
          ::mosek::fusion::p_Sort::argbucketsort(_364_idx,_365_vals1,_367_first,_368_last,_ptr__370_minmax[(int)0],_ptr__370_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_364_idx,_365_vals1,_366_vals2,_367_first,_368_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:95:6-114:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _372_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _373_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _374_vals2,
    int64_t _375_first,
    int64_t _376_last,
    bool _377_check) { mosek::fusion::p_Sort::argsort(_372_idx,_373_vals1,_374_vals2,_375_first,_376_last,_377_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _372_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _373_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _374_vals2,int64_t _375_first,int64_t _376_last,bool _377_check)
{
  _checked_ptr_1<int64_t> _ptr__372_idx(_372_idx ? _372_idx->raw() : nullptr, _372_idx ? _372_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__373_vals1(_373_vals1 ? _373_vals1->raw() : nullptr, _373_vals1 ? _373_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__374_vals2(_374_vals2 ? _374_vals2->raw() : nullptr, _374_vals2 ? _374_vals2->size(0) : 0);
  if (((!_377_check) || ((_376_last - _375_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_372_idx,_373_vals1,_374_vals2,_375_first,_376_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_372_idx,_373_vals1,_374_vals2,_375_first,_376_last,_377_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _378_minmax = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int32_t> _ptr__378_minmax(_378_minmax ? _378_minmax->raw() : nullptr,_378_minmax ? _378_minmax->size(0) : 0);
      int32_t _379_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_372_idx,_373_vals1,_374_vals2,_375_first,_376_last,_378_minmax);
      if ((((_ptr__378_minmax[(int)1] - _ptr__378_minmax[(int)0]) < safe_mul( _379_scale,(_376_last - _375_first) )) && ((_ptr__378_minmax[(int)3] - _ptr__378_minmax[(int)2]) < safe_mul( _379_scale,(_376_last - _375_first) ))))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_372_idx,_374_vals2,_375_first,_376_last,_ptr__378_minmax[(int)2],_ptr__378_minmax[(int)3]);
          ::mosek::fusion::p_Sort::argbucketsort(_372_idx,_373_vals1,_375_first,_376_last,_ptr__378_minmax[(int)0],_ptr__378_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_372_idx,_373_vals1,_374_vals2,_375_first,_376_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argbucketsort
// Method mosek.fusion.Sort.argbucketsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:90:6-93:7
void mosek::fusion::Sort::argbucketsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _380_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _381_vals,
    int64_t _382_first,
    int64_t _383_last,
    int64_t _384_minv,
    int64_t _385_maxv) { mosek::fusion::p_Sort::argbucketsort(_380_idx,_381_vals,_382_first,_383_last,_384_minv,_385_maxv); }
void mosek::fusion::p_Sort::argbucketsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _380_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _381_vals,int64_t _382_first,int64_t _383_last,int64_t _384_minv,int64_t _385_maxv)
{
  _checked_ptr_1<int64_t> _ptr__380_idx(_380_idx ? _380_idx->raw() : nullptr, _380_idx ? _380_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__381_vals(_381_vals ? _381_vals->raw() : nullptr, _381_vals ? _381_vals->size(0) : 0);
  ::mosek::fusion::Utils::Tools::bucketsort(_380_idx,_382_first,_383_last,_381_vals,_384_minv,_385_maxv);
}
// End mosek.fusion.Sort.argbucketsort

// Begin mosek.fusion.Sort.argbucketsort
// Method mosek.fusion.Sort.argbucketsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:85:6-88:7
void mosek::fusion::Sort::argbucketsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _386_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _387_vals,
    int64_t _388_first,
    int64_t _389_last,
    int32_t _390_minv,
    int32_t _391_maxv) { mosek::fusion::p_Sort::argbucketsort(_386_idx,_387_vals,_388_first,_389_last,_390_minv,_391_maxv); }
void mosek::fusion::p_Sort::argbucketsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _386_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _387_vals,int64_t _388_first,int64_t _389_last,int32_t _390_minv,int32_t _391_maxv)
{
  _checked_ptr_1<int64_t> _ptr__386_idx(_386_idx ? _386_idx->raw() : nullptr, _386_idx ? _386_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__387_vals(_387_vals ? _387_vals->raw() : nullptr, _387_vals ? _387_vals->size(0) : 0);
  ::mosek::fusion::Utils::Tools::bucketsort(_386_idx,_388_first,_389_last,_387_vals,_390_minv,_391_maxv);
}
// End mosek.fusion.Sort.argbucketsort

// Begin mosek.fusion.Sort.getminmax
// Method mosek.fusion.Sort.getminmax @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:61:6-80:7
void mosek::fusion::Sort::getminmax
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _392_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _393_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _394_vals2,
    int64_t _395_first,
    int64_t _396_last,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _397_res) { mosek::fusion::p_Sort::getminmax(_392_idx,_393_vals1,_394_vals2,_395_first,_396_last,_397_res); }
void mosek::fusion::p_Sort::getminmax(std::shared_ptr< monty::ndarray< int64_t,1 > > _392_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _393_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _394_vals2,int64_t _395_first,int64_t _396_last,std::shared_ptr< monty::ndarray< int64_t,1 > > _397_res)
{
  _checked_ptr_1<int64_t> _ptr__392_idx(_392_idx ? _392_idx->raw() : nullptr, _392_idx ? _392_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__393_vals1(_393_vals1 ? _393_vals1->raw() : nullptr, _393_vals1 ? _393_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__394_vals2(_394_vals2 ? _394_vals2->raw() : nullptr, _394_vals2 ? _394_vals2->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__397_res(_397_res ? _397_res->raw() : nullptr, _397_res ? _397_res->size(0) : 0);
  _ptr__397_res[(int)0] = _ptr__393_vals1[_ptr__392_idx[_395_first]];
  _ptr__397_res[(int)1] = _ptr__393_vals1[_ptr__392_idx[_395_first]];
  int64_t _9330_ = safe_add( _395_first,(int)1 );
  int64_t _9331_ = _396_last;
  for (int64_t _398_i = _9330_; _398_i < _9331_; ++_398_i)
  {
    {
      if ((_ptr__397_res[(int)0] > _ptr__393_vals1[_ptr__392_idx[_398_i]]))
      {
        {
          _ptr__397_res[(int)0] = _ptr__393_vals1[_ptr__392_idx[_398_i]];
        }
      }
      {}
      if ((_ptr__397_res[(int)1] < _ptr__393_vals1[_ptr__392_idx[_398_i]]))
      {
        {
          _ptr__397_res[(int)1] = _ptr__393_vals1[_ptr__392_idx[_398_i]];
        }
      }
      {}
    }
  }
  if ((NULL != _394_vals2.get()))
  {
    {
      _ptr__397_res[(int)2] = _ptr__394_vals2[_ptr__392_idx[_395_first]];
      _ptr__397_res[(int)3] = _ptr__394_vals2[_ptr__392_idx[_395_first]];
      int64_t _9332_ = safe_add( _395_first,(int)1 );
      int64_t _9333_ = _396_last;
      for (int64_t _399_i = _9332_; _399_i < _9333_; ++_399_i)
      {
        {
          if ((_ptr__397_res[(int)2] > _ptr__394_vals2[_ptr__392_idx[_399_i]]))
          {
            {
              _ptr__397_res[(int)2] = _ptr__394_vals2[_ptr__392_idx[_399_i]];
            }
          }
          {}
          if ((_ptr__397_res[(int)3] < _ptr__394_vals2[_ptr__392_idx[_399_i]]))
          {
            {
              _ptr__397_res[(int)3] = _ptr__394_vals2[_ptr__392_idx[_399_i]];
            }
          }
          {}
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.getminmax

// Begin mosek.fusion.Sort.getminmax
// Method mosek.fusion.Sort.getminmax @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:40:6-59:7
void mosek::fusion::Sort::getminmax
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _400_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _401_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _402_vals2,
    int64_t _403_first,
    int64_t _404_last,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _405_res) { mosek::fusion::p_Sort::getminmax(_400_idx,_401_vals1,_402_vals2,_403_first,_404_last,_405_res); }
void mosek::fusion::p_Sort::getminmax(std::shared_ptr< monty::ndarray< int64_t,1 > > _400_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _401_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _402_vals2,int64_t _403_first,int64_t _404_last,std::shared_ptr< monty::ndarray< int32_t,1 > > _405_res)
{
  _checked_ptr_1<int64_t> _ptr__400_idx(_400_idx ? _400_idx->raw() : nullptr, _400_idx ? _400_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__401_vals1(_401_vals1 ? _401_vals1->raw() : nullptr, _401_vals1 ? _401_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__402_vals2(_402_vals2 ? _402_vals2->raw() : nullptr, _402_vals2 ? _402_vals2->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__405_res(_405_res ? _405_res->raw() : nullptr, _405_res ? _405_res->size(0) : 0);
  _ptr__405_res[(int)0] = _ptr__401_vals1[_ptr__400_idx[_403_first]];
  _ptr__405_res[(int)1] = _ptr__401_vals1[_ptr__400_idx[_403_first]];
  int64_t _9334_ = safe_add( _403_first,(int)1 );
  int64_t _9335_ = _404_last;
  for (int64_t _406_i = _9334_; _406_i < _9335_; ++_406_i)
  {
    {
      if ((_ptr__405_res[(int)0] > _ptr__401_vals1[_ptr__400_idx[_406_i]]))
      {
        {
          _ptr__405_res[(int)0] = _ptr__401_vals1[_ptr__400_idx[_406_i]];
        }
      }
      {}
      if ((_ptr__405_res[(int)1] < _ptr__401_vals1[_ptr__400_idx[_406_i]]))
      {
        {
          _ptr__405_res[(int)1] = _ptr__401_vals1[_ptr__400_idx[_406_i]];
        }
      }
      {}
    }
  }
  if ((NULL != _402_vals2.get()))
  {
    {
      _ptr__405_res[(int)2] = _ptr__402_vals2[_ptr__400_idx[_403_first]];
      _ptr__405_res[(int)3] = _ptr__402_vals2[_ptr__400_idx[_403_first]];
      int64_t _9336_ = safe_add( _403_first,(int)1 );
      int64_t _9337_ = _404_last;
      for (int64_t _407_i = _9336_; _407_i < _9337_; ++_407_i)
      {
        {
          if ((_ptr__405_res[(int)2] > _ptr__402_vals2[_ptr__400_idx[_407_i]]))
          {
            {
              _ptr__405_res[(int)2] = _ptr__402_vals2[_ptr__400_idx[_407_i]];
            }
          }
          {}
          if ((_ptr__405_res[(int)3] < _ptr__402_vals2[_ptr__400_idx[_407_i]]))
          {
            {
              _ptr__405_res[(int)3] = _ptr__402_vals2[_ptr__400_idx[_407_i]];
            }
          }
          {}
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.getminmax

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:32:6-38:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _408_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _409_vals1,
    int64_t _410_first,
    int64_t _411_last,
    bool _412_check) { return mosek::fusion::p_Sort::issorted(_408_idx,_409_vals1,_410_first,_411_last,_412_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _408_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _409_vals1,int64_t _410_first,int64_t _411_last,bool _412_check)
{
  _checked_ptr_1<int64_t> _ptr__408_idx(_408_idx ? _408_idx->raw() : nullptr, _408_idx ? _408_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__409_vals1(_409_vals1 ? _409_vals1->raw() : nullptr, _409_vals1 ? _409_vals1->size(0) : 0);
  int64_t _9338_ = _410_first;
  int64_t _9339_ = (_411_last - (int)1);
  for (int64_t _413_i = _9338_; _413_i < _9339_; ++_413_i)
  {
    {
      if ((_ptr__409_vals1[_ptr__408_idx[_413_i]] > _ptr__409_vals1[_ptr__408_idx[safe_add( _413_i,(int)1 )]]))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:24:6-30:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _414_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _415_vals1,
    int64_t _416_first,
    int64_t _417_last,
    bool _418_check) { return mosek::fusion::p_Sort::issorted(_414_idx,_415_vals1,_416_first,_417_last,_418_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _414_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _415_vals1,int64_t _416_first,int64_t _417_last,bool _418_check)
{
  _checked_ptr_1<int64_t> _ptr__414_idx(_414_idx ? _414_idx->raw() : nullptr, _414_idx ? _414_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__415_vals1(_415_vals1 ? _415_vals1->raw() : nullptr, _415_vals1 ? _415_vals1->size(0) : 0);
  int64_t _9340_ = _416_first;
  int64_t _9341_ = (_417_last - (int)1);
  for (int64_t _419_i = _9340_; _419_i < _9341_; ++_419_i)
  {
    {
      if ((_ptr__415_vals1[_ptr__414_idx[_419_i]] > _ptr__415_vals1[_ptr__414_idx[safe_add( _419_i,(int)1 )]]))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:16:6-22:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _420_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _421_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _422_vals2,
    int64_t _423_first,
    int64_t _424_last,
    bool _425_check) { return mosek::fusion::p_Sort::issorted(_420_idx,_421_vals1,_422_vals2,_423_first,_424_last,_425_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _420_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _421_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _422_vals2,int64_t _423_first,int64_t _424_last,bool _425_check)
{
  _checked_ptr_1<int64_t> _ptr__420_idx(_420_idx ? _420_idx->raw() : nullptr, _420_idx ? _420_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__421_vals1(_421_vals1 ? _421_vals1->raw() : nullptr, _421_vals1 ? _421_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__422_vals2(_422_vals2 ? _422_vals2->raw() : nullptr, _422_vals2 ? _422_vals2->size(0) : 0);
  int64_t _9342_ = _423_first;
  int64_t _9343_ = (_424_last - (int)1);
  for (int64_t _426_i = _9342_; _426_i < _9343_; ++_426_i)
  {
    {
      if (((_ptr__421_vals1[_ptr__420_idx[_426_i]] > _ptr__421_vals1[_ptr__420_idx[safe_add( _426_i,(int)1 )]]) || ((_ptr__421_vals1[_ptr__420_idx[safe_add( _426_i,(int)1 )]] == _ptr__421_vals1[_ptr__420_idx[_426_i]]) && (_ptr__422_vals2[_ptr__420_idx[_426_i]] > _ptr__422_vals2[_ptr__420_idx[safe_add( _426_i,(int)1 )]]))))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Sort.mbi:8:6-14:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _427_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _428_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _429_vals2,
    int64_t _430_first,
    int64_t _431_last,
    bool _432_check) { return mosek::fusion::p_Sort::issorted(_427_idx,_428_vals1,_429_vals2,_430_first,_431_last,_432_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _427_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _428_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _429_vals2,int64_t _430_first,int64_t _431_last,bool _432_check)
{
  _checked_ptr_1<int64_t> _ptr__427_idx(_427_idx ? _427_idx->raw() : nullptr, _427_idx ? _427_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__428_vals1(_428_vals1 ? _428_vals1->raw() : nullptr, _428_vals1 ? _428_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__429_vals2(_429_vals2 ? _429_vals2->raw() : nullptr, _429_vals2 ? _429_vals2->size(0) : 0);
  int64_t _9344_ = _430_first;
  int64_t _9345_ = (_431_last - (int)1);
  for (int64_t _433_i = _9344_; _433_i < _9345_; ++_433_i)
  {
    {
      if (((_ptr__428_vals1[_ptr__427_idx[_433_i]] > _ptr__428_vals1[_ptr__427_idx[safe_add( _433_i,(int)1 )]]) || ((_ptr__428_vals1[_ptr__427_idx[_433_i]] == _ptr__428_vals1[_ptr__427_idx[safe_add( _433_i,(int)1 )]]) && (_ptr__429_vals2[_ptr__427_idx[_433_i]] > _ptr__429_vals2[_ptr__427_idx[safe_add( _433_i,(int)1 )]]))))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

void mosek::fusion::p_Sort::destroy()
{
}
void mosek::fusion::Sort::destroy() { mosek::fusion::p_Sort::_get_impl(this)->destroy(); }
// } class Sort
// class IndexCounter { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:446:4-523:5
mosek::fusion::IndexCounter::IndexCounter(mosek::fusion::p_IndexCounter *_impl) : _impl(_impl) { /*std::cout << "IndexCounter() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::IndexCounter::~IndexCounter() { delete _impl; _impl = nullptr; }
mosek::fusion::p_IndexCounter::p_IndexCounter
  (::mosek::fusion::IndexCounter * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.IndexCounter.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:486:6-496:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _434_shape)
{ return new mosek::fusion::IndexCounter(_434_shape); }
mosek::fusion::IndexCounter::IndexCounter
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _434_shape) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_434_shape); }
void mosek::fusion::p_IndexCounter::_initialize  (std::shared_ptr< monty::ndarray< int32_t,1 > > _434_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__434_shape(_434_shape ? _434_shape->raw() : nullptr, _434_shape ? _434_shape->size(0) : 0);
    n = (int)((_434_shape)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    start = (int64_t)0;
    strides = ::mosek::fusion::p_Set::strides(_434_shape);
    std::vector<int32_t> _9346_;
    for (int32_t _435_i = (int)0, _9347_ = (int)((_434_shape)->size(0)); _435_i < _9347_ ; ++_435_i)
    {
      _9346_.push_back(_ptr__434_shape[_435_i]);
    }
    auto _9348_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9346_.size()),_9346_.begin(),_9346_.end()));
    dims = _9348_;
  }
}
// mosek.fusion.IndexCounter.ctor

// mosek.fusion.IndexCounter.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:471:6-484:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (int64_t _436_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _437_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _438_shape)
{ return new mosek::fusion::IndexCounter(_436_start_,_437_dims_,_438_shape); }
mosek::fusion::IndexCounter::IndexCounter
  (int64_t _436_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _437_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _438_shape) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_436_start_,_437_dims_,_438_shape); }
void mosek::fusion::p_IndexCounter::_initialize  (int64_t _436_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _437_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _438_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__437_dims_(_437_dims_ ? _437_dims_->raw() : nullptr, _437_dims_ ? _437_dims_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__438_shape(_438_shape ? _438_shape->raw() : nullptr, _438_shape ? _438_shape->size(0) : 0);
    int32_t _439_nd = (int)((_438_shape)->size(0));
    n = (int)((_437_dims_)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    int32_t _9349_ = (int)0;
    int32_t _9350_ = n;
    for (int32_t _440_i = _9349_; _440_i < _9350_; ++_440_i)
    {
      {
        ((*st)(_440_i)) = _436_start_;
      }
    }
    strides = ::mosek::fusion::p_Set::strides(_438_shape);
    dims = _437_dims_;
    start = _436_start_;
  }
}
// mosek.fusion.IndexCounter.ctor

// mosek.fusion.IndexCounter.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:457:6-469:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (int64_t _441_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _442_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _443_strides_)
{ return new mosek::fusion::IndexCounter(_441_start_,_442_dims_,_443_strides_); }
mosek::fusion::IndexCounter::IndexCounter
  (int64_t _441_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _442_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _443_strides_) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_441_start_,_442_dims_,_443_strides_); }
void mosek::fusion::p_IndexCounter::_initialize  (int64_t _441_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _442_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _443_strides_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__442_dims_(_442_dims_ ? _442_dims_->raw() : nullptr, _442_dims_ ? _442_dims_->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__443_strides_(_443_strides_ ? _443_strides_->raw() : nullptr, _443_strides_ ? _443_strides_->size(0) : 0);
    n = (int)((_442_dims_)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    int32_t _9351_ = (int)0;
    int32_t _9352_ = n;
    for (int32_t _444_i = _9351_; _444_i < _9352_; ++_444_i)
    {
      {
        ((*st)(_444_i)) = _441_start_;
      }
    }
    strides = _443_strides_;
    dims = _442_dims_;
    start = _441_start_;
  }
}
// mosek.fusion.IndexCounter.ctor

// Begin mosek.fusion.IndexCounter.atEnd
// Method mosek.fusion.IndexCounter.atEnd @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:522:6-66
bool mosek::fusion::IndexCounter :: atEnd() { return mosek::fusion::p_IndexCounter::_get_impl(this)->atEnd(); }
bool mosek::fusion::p_IndexCounter::atEnd()
{
  return (((*ii)((n - (int)1))) >= ((*dims)((n - (int)1))));
}
// End mosek.fusion.IndexCounter.atEnd

// Begin mosek.fusion.IndexCounter.getIndex
// Method mosek.fusion.IndexCounter.getIndex @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:520:6-54
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::IndexCounter :: getIndex() { return mosek::fusion::p_IndexCounter::_get_impl(this)->getIndex(); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_IndexCounter::getIndex()
{
  return ii;
}
// End mosek.fusion.IndexCounter.getIndex

// Begin mosek.fusion.IndexCounter.next
// Method mosek.fusion.IndexCounter.next @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:519:6-68
int64_t mosek::fusion::IndexCounter :: next() { return mosek::fusion::p_IndexCounter::_get_impl(this)->next(); }
int64_t mosek::fusion::p_IndexCounter::next()
{
  int64_t _445_r = _pubthis->get();
  _pubthis->inc();
  return _445_r;
}
// End mosek.fusion.IndexCounter.next

// Begin mosek.fusion.IndexCounter.get
// Method mosek.fusion.IndexCounter.get @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:517:6-49
int64_t mosek::fusion::IndexCounter :: get() { return mosek::fusion::p_IndexCounter::_get_impl(this)->get(); }
int64_t mosek::fusion::p_IndexCounter::get()
{
  return ((*st)((int)0));
}
// End mosek.fusion.IndexCounter.get

// Begin mosek.fusion.IndexCounter.inc
// Method mosek.fusion.IndexCounter.inc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:503:6-515:7
void mosek::fusion::IndexCounter :: inc() { mosek::fusion::p_IndexCounter::_get_impl(this)->inc(); }
void mosek::fusion::p_IndexCounter::inc()
{
  ++ ((*ii)((int)0));
  ((*st)((int)0)) = safe_add( ((*st)((int)0)),((*strides)((n - (int)1))) );
  int32_t _446_i;
  {
    _446_i = (int)0;
    for(;((_446_i < (n - (int)1)) && (((*ii)(_446_i)) == ((*dims)(((n - _446_i) - (int)1)))));++ _446_i)
    {
      {
        ((*ii)(_446_i)) = (int)0;
        ++ ((*ii)(safe_add( _446_i,(int)1 )));
        ((*st)(safe_add( _446_i,(int)1 ))) = safe_add( ((*st)(safe_add( _446_i,(int)1 ))),((*strides)(((n - _446_i) - (int)2))) );
      }
    }
  }
  int32_t _9353_ = (int)0;
  int32_t _9354_ = _446_i;
  for (int32_t _447_j = _9353_; _447_j < _9354_; ++_447_j)
  {
    {
      ((*st)(_447_j)) = ((*st)(_446_i));
    }
  }
}
// End mosek.fusion.IndexCounter.inc

// Begin mosek.fusion.IndexCounter.reset
// Method mosek.fusion.IndexCounter.reset @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:498:6-501:7
void mosek::fusion::IndexCounter :: reset() { mosek::fusion::p_IndexCounter::_get_impl(this)->reset(); }
void mosek::fusion::p_IndexCounter::reset()
{
  st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
  int32_t _9355_ = (int)0;
  int32_t _9356_ = n;
  for (int32_t _448_i = _9355_; _448_i < _9356_; ++_448_i)
  {
    {
      ((*st)(_448_i)) = start;
    }
  }
}
// End mosek.fusion.IndexCounter.reset

void mosek::fusion::p_IndexCounter::destroy()
{
  dims.reset();
  strides.reset();
  st.reset();
  ii.reset();
}
void mosek::fusion::IndexCounter::destroy() { mosek::fusion::p_IndexCounter::_get_impl(this)->destroy(); }
// } class IndexCounter
// class CommonTools { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:6:4-443:5
mosek::fusion::CommonTools::CommonTools(mosek::fusion::p_CommonTools *_impl) : _impl(_impl) { /*std::cout << "CommonTools() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::CommonTools::~CommonTools() { delete _impl; _impl = nullptr; }
mosek::fusion::p_CommonTools::p_CommonTools
  (::mosek::fusion::CommonTools * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:431:6-442:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _449_values,
    int32_t _450_newsize) { return mosek::fusion::p_CommonTools::resize(_449_values,_450_newsize); }
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< int64_t,1 > > _449_values,int32_t _450_newsize)
{
  _checked_ptr_1<int64_t> _ptr__449_values(_449_values ? _449_values->raw() : nullptr, _449_values ? _449_values->size(0) : 0);
  if ((_450_newsize == (int)((_449_values)->size(0))))
  {
    {
      return _449_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _451_newvals = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_450_newsize)));
      _checked_ptr_1<int64_t> _ptr__451_newvals(_451_newvals ? _451_newvals->raw() : nullptr,_451_newvals ? _451_newvals->size(0) : 0);
      if ((_450_newsize > (int)((_449_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_449_values,(int)0,_451_newvals,(int)0,(int)((_449_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_449_values,(int)0,_451_newvals,(int)0,_450_newsize);
        }
      }
      return _451_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:419:6-430:7
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _452_values,
    int32_t _453_newsize) { return mosek::fusion::p_CommonTools::resize(_452_values,_453_newsize); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< int32_t,1 > > _452_values,int32_t _453_newsize)
{
  _checked_ptr_1<int32_t> _ptr__452_values(_452_values ? _452_values->raw() : nullptr, _452_values ? _452_values->size(0) : 0);
  if ((_453_newsize == (int)((_452_values)->size(0))))
  {
    {
      return _452_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _454_newvals = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_453_newsize)));
      _checked_ptr_1<int32_t> _ptr__454_newvals(_454_newvals ? _454_newvals->raw() : nullptr,_454_newvals ? _454_newvals->size(0) : 0);
      if ((_453_newsize > (int)((_452_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_452_values,(int)0,_454_newvals,(int)0,(int)((_452_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_452_values,(int)0,_454_newvals,(int)0,_453_newsize);
        }
      }
      return _454_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:407:6-418:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< double,1 > > _455_values,
    int32_t _456_newsize) { return mosek::fusion::p_CommonTools::resize(_455_values,_456_newsize); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< double,1 > > _455_values,int32_t _456_newsize)
{
  _checked_ptr_1<double> _ptr__455_values(_455_values ? _455_values->raw() : nullptr, _455_values ? _455_values->size(0) : 0);
  if ((_456_newsize == (int)((_455_values)->size(0))))
  {
    {
      return _455_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _457_newvals = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_456_newsize)));
      _checked_ptr_1<double> _ptr__457_newvals(_457_newvals ? _457_newvals->raw() : nullptr,_457_newvals ? _457_newvals->size(0) : 0);
      if ((_456_newsize > (int)((_455_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_455_values,(int)0,_457_newvals,(int)0,(int)((_455_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_455_values,(int)0,_457_newvals,(int)0,_456_newsize);
        }
      }
      return _457_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.binarySearch
// Method mosek.fusion.CommonTools.binarySearch @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:389:6-405:7
int32_t mosek::fusion::CommonTools::binarySearch
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _458_values,
    int32_t _459_target) { return mosek::fusion::p_CommonTools::binarySearch(_458_values,_459_target); }
int32_t mosek::fusion::p_CommonTools::binarySearch(std::shared_ptr< monty::ndarray< int32_t,1 > > _458_values,int32_t _459_target)
{
  _checked_ptr_1<int32_t> _ptr__458_values(_458_values ? _458_values->raw() : nullptr, _458_values ? _458_values->size(0) : 0);
  int32_t _460_i0 = (int)0;
  int32_t _461_i1 = (int)((_458_values)->size(0));
  if ((_ptr__458_values[_460_i0] == _459_target))
  {
    {
      return _460_i0;
    }
  }
  else if((_ptr__458_values[(_461_i1 - (int)1)] == _459_target))
  {
    {
      return (_461_i1 - (int)1);
    }
  }
  {}
  while ( (_460_i0 < (_461_i1 - (int)1)) )
  {
    {
      int32_t _462_p = ((_461_i1 - _460_i0) / (int)2);
      if ((_459_target < _ptr__458_values[_462_p]))
      {
        {
          _460_i0 = safe_add( _462_p,(int)1 );
        }
      }
      else if((_459_target > _ptr__458_values[_462_p]))
      {
        {
          _461_i1 = _462_p;
        }
      }
      else
      {
        {
          return _462_p;
        }
      }
    }
  }
  return (-(int)1);
}
// End mosek.fusion.CommonTools.binarySearch

// Begin mosek.fusion.CommonTools.binarySearch
// Method mosek.fusion.CommonTools.binarySearch @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:381:6-387:7
int32_t mosek::fusion::CommonTools::binarySearch
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _463_values,
    int64_t _464_target) { return mosek::fusion::p_CommonTools::binarySearch(_463_values,_464_target); }
int32_t mosek::fusion::p_CommonTools::binarySearch(std::shared_ptr< monty::ndarray< int64_t,1 > > _463_values,int64_t _464_target)
{
  _checked_ptr_1<int64_t> _ptr__463_values(_463_values ? _463_values->raw() : nullptr, _463_values ? _463_values->size(0) : 0);
  int32_t _465_i = ::mosek::fusion::p_CommonTools::binarySearchL(_463_values,_464_target);
  if (((_465_i < (int)((_463_values)->size(0))) && (_ptr__463_values[_465_i] == _464_target)))
  {
    {
      return _465_i;
    }
  }
  else
  {
    {
      return (-(int)1);
    }
  }
}
// End mosek.fusion.CommonTools.binarySearch

// Begin mosek.fusion.CommonTools.binarySearchR
// Method mosek.fusion.CommonTools.binarySearchR @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:365:6-379:7
int32_t mosek::fusion::CommonTools::binarySearchR
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _466_values,
    int64_t _467_target) { return mosek::fusion::p_CommonTools::binarySearchR(_466_values,_467_target); }
int32_t mosek::fusion::p_CommonTools::binarySearchR(std::shared_ptr< monty::ndarray< int64_t,1 > > _466_values,int64_t _467_target)
{
  _checked_ptr_1<int64_t> _ptr__466_values(_466_values ? _466_values->raw() : nullptr, _466_values ? _466_values->size(0) : 0);
  int32_t _468_i0 = (int)0;
  int32_t _469_i1 = (int)((_466_values)->size(0));
  if ((_ptr__466_values[_468_i0] > _467_target))
  {
    {
      return _468_i0;
    }
  }
  {}
  while ( (_468_i0 < (_469_i1 - (int)1)) )
  {
    {
      int32_t _470_p = (safe_add( _469_i1,_468_i0 ) / (int)2);
      if ((_467_target >= _ptr__466_values[_470_p]))
      {
        {
          _468_i0 = _470_p;
        }
      }
      else
      {
        {
          _469_i1 = _470_p;
        }
      }
    }
  }
  return _469_i1;
}
// End mosek.fusion.CommonTools.binarySearchR

// Begin mosek.fusion.CommonTools.binarySearchL
// Method mosek.fusion.CommonTools.binarySearchL @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:348:6-362:7
int32_t mosek::fusion::CommonTools::binarySearchL
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _471_values,
    int64_t _472_target) { return mosek::fusion::p_CommonTools::binarySearchL(_471_values,_472_target); }
int32_t mosek::fusion::p_CommonTools::binarySearchL(std::shared_ptr< monty::ndarray< int64_t,1 > > _471_values,int64_t _472_target)
{
  _checked_ptr_1<int64_t> _ptr__471_values(_471_values ? _471_values->raw() : nullptr, _471_values ? _471_values->size(0) : 0);
  int32_t _473_i0 = (int)0;
  int32_t _474_i1 = (int)((_471_values)->size(0));
  if ((_472_target <= _ptr__471_values[_473_i0]))
  {
    {
      return _473_i0;
    }
  }
  {}
  while ( (_473_i0 < (_474_i1 - (int)1)) )
  {
    {
      int32_t _475_p = (safe_add( _474_i1,_473_i0 ) / (int)2);
      if ((_ptr__471_values[_475_p] < _472_target))
      {
        {
          _473_i0 = _475_p;
        }
      }
      else
      {
        {
          _474_i1 = _475_p;
        }
      }
    }
  }
  return _474_i1;
}
// End mosek.fusion.CommonTools.binarySearchL

// Begin mosek.fusion.CommonTools.ndIncr
// Method mosek.fusion.CommonTools.ndIncr @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:335:6-344:7
void mosek::fusion::CommonTools::ndIncr
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _476_ndidx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _477_first,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _478_last) { mosek::fusion::p_CommonTools::ndIncr(_476_ndidx,_477_first,_478_last); }
void mosek::fusion::p_CommonTools::ndIncr(std::shared_ptr< monty::ndarray< int32_t,1 > > _476_ndidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _477_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _478_last)
{
  _checked_ptr_1<int32_t> _ptr__476_ndidx(_476_ndidx ? _476_ndidx->raw() : nullptr, _476_ndidx ? _476_ndidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__477_first(_477_first ? _477_first->raw() : nullptr, _477_first ? _477_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__478_last(_478_last ? _478_last->raw() : nullptr, _478_last ? _478_last->size(0) : 0);
  int32_t _479_i = ((int)((_476_ndidx)->size(0)) - (int)1);
  ++ _ptr__476_ndidx[_479_i];
  while ( ((_479_i > (int)0) && (_ptr__476_ndidx[_479_i] >= _ptr__478_last[_479_i])) )
  {
    {
      _ptr__476_ndidx[_479_i] = _ptr__477_first[_479_i];
      ++ _ptr__476_ndidx[(_479_i - (int)1)];
    }
  }
}
// End mosek.fusion.CommonTools.ndIncr

// Begin mosek.fusion.CommonTools.transposeTriplets
// Method mosek.fusion.CommonTools.transposeTriplets @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:288:6-333:7
void mosek::fusion::CommonTools::transposeTriplets
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _480_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _481_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _482_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _483_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _484_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _485_tval_,
    int64_t _486_nelm,
    int32_t _487_dimi,
    int32_t _488_dimj) { mosek::fusion::p_CommonTools::transposeTriplets(_480_subi,_481_subj,_482_val,_483_tsubi_,_484_tsubj_,_485_tval_,_486_nelm,_487_dimi,_488_dimj); }
void mosek::fusion::p_CommonTools::transposeTriplets(std::shared_ptr< monty::ndarray< int32_t,1 > > _480_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _481_subj,std::shared_ptr< monty::ndarray< double,1 > > _482_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _483_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _484_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _485_tval_,int64_t _486_nelm,int32_t _487_dimi,int32_t _488_dimj)
{
  _checked_ptr_1<int32_t> _ptr__480_subi(_480_subi ? _480_subi->raw() : nullptr, _480_subi ? _480_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__481_subj(_481_subj ? _481_subj->raw() : nullptr, _481_subj ? _481_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__482_val(_482_val ? _482_val->raw() : nullptr, _482_val ? _482_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _489_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _487_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__489_rptrb(_489_rptrb ? _489_rptrb->raw() : nullptr,_489_rptrb ? _489_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _490_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _488_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__490_cptrb(_490_cptrb ? _490_cptrb->raw() : nullptr,_490_cptrb ? _490_cptrb->size(0) : 0);
  {
    int64_t _491_i = (int64_t)0;
    for(;(_491_i < _486_nelm);++ _491_i)
    {
      {
        _ptr__490_cptrb[safe_add( _ptr__481_subj[_491_i],(int)2 )] = safe_add( _ptr__490_cptrb[safe_add( _ptr__481_subj[_491_i],(int)2 )],(int)1 );
        _ptr__489_rptrb[safe_add( _ptr__480_subi[_491_i],(int)2 )] = safe_add( _ptr__489_rptrb[safe_add( _ptr__480_subi[_491_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _492_i = (int)1;
    for(;(_492_i < _487_dimi);++ _492_i)
    {
      {
        _ptr__489_rptrb[safe_add( _492_i,(int)1 )] = safe_add( _ptr__489_rptrb[safe_add( _492_i,(int)1 )],_ptr__489_rptrb[_492_i] );
      }
    }
  }
  {
    int32_t _493_i = (int)1;
    for(;(_493_i < _488_dimj);++ _493_i)
    {
      {
        _ptr__490_cptrb[safe_add( _493_i,(int)1 )] = safe_add( _ptr__490_cptrb[safe_add( _493_i,(int)1 )],_ptr__490_cptrb[_493_i] );
      }
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _494_tsubi = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_486_nelm)));
  _checked_ptr_1<int64_t> _ptr__494_tsubi(_494_tsubi ? _494_tsubi->raw() : nullptr,_494_tsubi ? _494_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _495_tsubj = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_486_nelm)));
  _checked_ptr_1<int64_t> _ptr__495_tsubj(_495_tsubj ? _495_tsubj->raw() : nullptr,_495_tsubj ? _495_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _496_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_486_nelm)));
  _checked_ptr_1<double> _ptr__496_tval(_496_tval ? _496_tval->raw() : nullptr,_496_tval ? _496_tval->size(0) : 0);
  int32_t _9357_ = (int)0;
  int32_t _9358_ = _487_dimi;
  for (int32_t _497_i = _9357_; _497_i < _9358_; ++_497_i)
  {
    {
      {
        int64_t _498_k = _ptr__489_rptrb[_497_i];
        for(;(_498_k < _ptr__489_rptrb[safe_add( _497_i,(int)1 )]);++ _498_k)
        {
          {
            int32_t _499_j = _ptr__481_subj[_498_k];
            int64_t _500_cidx = _ptr__490_cptrb[_499_j];
            _ptr__494_tsubi[_500_cidx] = _497_i;
            _ptr__495_tsubj[_500_cidx] = _499_j;
            _ptr__496_tval[_500_cidx] = _ptr__482_val[_498_k];
            _ptr__490_cptrb[_499_j] = safe_add( _500_cidx,(int)1 );
          }
        }
      }
    }
  }
  ((*_483_tsubi_)((int)0)) = _495_tsubj;
  ((*_484_tsubj_)((int)0)) = _494_tsubi;
  ((*_485_tval_)((int)0)) = _496_tval;
}
// End mosek.fusion.CommonTools.transposeTriplets

// Begin mosek.fusion.CommonTools.transposeTriplets
// Method mosek.fusion.CommonTools.transposeTriplets @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:240:6-285:7
void mosek::fusion::CommonTools::transposeTriplets
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _501_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _502_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _503_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _504_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _505_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _506_tval_,
    int64_t _507_nelm,
    int32_t _508_dimi,
    int32_t _509_dimj) { mosek::fusion::p_CommonTools::transposeTriplets(_501_subi,_502_subj,_503_val,_504_tsubi_,_505_tsubj_,_506_tval_,_507_nelm,_508_dimi,_509_dimj); }
void mosek::fusion::p_CommonTools::transposeTriplets(std::shared_ptr< monty::ndarray< int32_t,1 > > _501_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _502_subj,std::shared_ptr< monty::ndarray< double,1 > > _503_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _504_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _505_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _506_tval_,int64_t _507_nelm,int32_t _508_dimi,int32_t _509_dimj)
{
  _checked_ptr_1<int32_t> _ptr__501_subi(_501_subi ? _501_subi->raw() : nullptr, _501_subi ? _501_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__502_subj(_502_subj ? _502_subj->raw() : nullptr, _502_subj ? _502_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__503_val(_503_val ? _503_val->raw() : nullptr, _503_val ? _503_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _510_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _508_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__510_rptrb(_510_rptrb ? _510_rptrb->raw() : nullptr,_510_rptrb ? _510_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _511_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _509_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__511_cptrb(_511_cptrb ? _511_cptrb->raw() : nullptr,_511_cptrb ? _511_cptrb->size(0) : 0);
  {
    int64_t _512_i = (int64_t)0;
    for(;(_512_i < _507_nelm);++ _512_i)
    {
      {
        _ptr__511_cptrb[safe_add( _ptr__502_subj[_512_i],(int)2 )] = safe_add( _ptr__511_cptrb[safe_add( _ptr__502_subj[_512_i],(int)2 )],(int)1 );
        _ptr__510_rptrb[safe_add( _ptr__501_subi[_512_i],(int)2 )] = safe_add( _ptr__510_rptrb[safe_add( _ptr__501_subi[_512_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _513_i = (int)1;
    for(;(_513_i < _508_dimi);++ _513_i)
    {
      {
        _ptr__510_rptrb[safe_add( _513_i,(int)1 )] = safe_add( _ptr__510_rptrb[safe_add( _513_i,(int)1 )],_ptr__510_rptrb[_513_i] );
      }
    }
  }
  {
    int32_t _514_i = (int)1;
    for(;(_514_i < _509_dimj);++ _514_i)
    {
      {
        _ptr__511_cptrb[safe_add( _514_i,(int)1 )] = safe_add( _ptr__511_cptrb[safe_add( _514_i,(int)1 )],_ptr__511_cptrb[_514_i] );
      }
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _515_tsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_507_nelm)));
  _checked_ptr_1<int32_t> _ptr__515_tsubi(_515_tsubi ? _515_tsubi->raw() : nullptr,_515_tsubi ? _515_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _516_tsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_507_nelm)));
  _checked_ptr_1<int32_t> _ptr__516_tsubj(_516_tsubj ? _516_tsubj->raw() : nullptr,_516_tsubj ? _516_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _517_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_507_nelm)));
  _checked_ptr_1<double> _ptr__517_tval(_517_tval ? _517_tval->raw() : nullptr,_517_tval ? _517_tval->size(0) : 0);
  {
    int32_t _518_i = (int)0;
    for(;(_518_i < _508_dimi);++ _518_i)
    {
      {
        int64_t _9359_ = _ptr__510_rptrb[_518_i];
        int64_t _9360_ = _ptr__510_rptrb[safe_add( _518_i,(int)1 )];
        for (int64_t _519_k = _9359_; _519_k < _9360_; ++_519_k)
        {
          {
            int32_t _520_j = _ptr__502_subj[_519_k];
            int64_t _521_cidx = _ptr__511_cptrb[_520_j];
            _ptr__515_tsubi[_521_cidx] = _518_i;
            _ptr__516_tsubj[_521_cidx] = _520_j;
            _ptr__517_tval[_521_cidx] = _ptr__503_val[_519_k];
            _ptr__511_cptrb[_520_j] = safe_add( _521_cidx,(int)1 );
          }
        }
      }
    }
  }
  ((*_504_tsubi_)((int)0)) = _516_tsubj;
  ((*_505_tsubj_)((int)0)) = _515_tsubi;
  ((*_506_tval_)((int)0)) = _517_tval;
}
// End mosek.fusion.CommonTools.transposeTriplets

// Begin mosek.fusion.CommonTools.tripletSort
// Method mosek.fusion.CommonTools.tripletSort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:114:6-213:7
void mosek::fusion::CommonTools::tripletSort
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _522_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _523_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _524_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _525_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _526_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _527_tval_,
    int64_t _528_nelm,
    int32_t _529_dimi,
    int32_t _530_dimj) { mosek::fusion::p_CommonTools::tripletSort(_522_subi,_523_subj,_524_val,_525_tsubi_,_526_tsubj_,_527_tval_,_528_nelm,_529_dimi,_530_dimj); }
void mosek::fusion::p_CommonTools::tripletSort(std::shared_ptr< monty::ndarray< int32_t,1 > > _522_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _523_subj,std::shared_ptr< monty::ndarray< double,1 > > _524_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _525_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _526_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _527_tval_,int64_t _528_nelm,int32_t _529_dimi,int32_t _530_dimj)
{
  _checked_ptr_1<int32_t> _ptr__522_subi(_522_subi ? _522_subi->raw() : nullptr, _522_subi ? _522_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__523_subj(_523_subj ? _523_subj->raw() : nullptr, _523_subj ? _523_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__524_val(_524_val ? _524_val->raw() : nullptr, _524_val ? _524_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _531_cols = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_528_nelm)));
  _checked_ptr_1<int32_t> _ptr__531_cols(_531_cols ? _531_cols->raw() : nullptr,_531_cols ? _531_cols->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _532_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _530_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__532_cptrb(_532_cptrb ? _532_cptrb->raw() : nullptr,_532_cptrb ? _532_cptrb->size(0) : 0);
  int64_t _9361_ = (int64_t)0;
  int64_t _9362_ = _528_nelm;
  for (int64_t _533_i = _9361_; _533_i < _9362_; ++_533_i)
  {
    {
      _ptr__532_cptrb[safe_add( _ptr__523_subj[_533_i],(int)2 )] = safe_add( _ptr__532_cptrb[safe_add( _ptr__523_subj[_533_i],(int)2 )],(int)1 );
    }
  }
  int32_t _9363_ = (int)1;
  int32_t _9364_ = _530_dimj;
  for (int32_t _534_i = _9363_; _534_i < _9364_; ++_534_i)
  {
    {
      _ptr__532_cptrb[safe_add( _534_i,(int)1 )] = safe_add( _ptr__532_cptrb[safe_add( _534_i,(int)1 )],_ptr__532_cptrb[_534_i] );
    }
  }
  int32_t _9365_ = (int)0;
  int32_t _9366_ = (int32_t)_528_nelm;
  for (int32_t _535_i = _9365_; _535_i < _9366_; ++_535_i)
  {
    {
      int32_t _536_cidx = _ptr__523_subj[_535_i];
      _ptr__531_cols[_ptr__532_cptrb[safe_add( _536_cidx,(int)1 )]] = _535_i;
      _ptr__532_cptrb[safe_add( _536_cidx,(int)1 )] = safe_add( _ptr__532_cptrb[safe_add( _536_cidx,(int)1 )],(int)1 );
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _537_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _529_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__537_rptrb(_537_rptrb ? _537_rptrb->raw() : nullptr,_537_rptrb ? _537_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _538_rows = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_528_nelm)));
  _checked_ptr_1<int32_t> _ptr__538_rows(_538_rows ? _538_rows->raw() : nullptr,_538_rows ? _538_rows->size(0) : 0);
  {
    int32_t _539_i = (int)0;
    for(;(_539_i < _528_nelm);++ _539_i)
    {
      {
        _ptr__537_rptrb[safe_add( _ptr__522_subi[_539_i],(int)2 )] = safe_add( _ptr__537_rptrb[safe_add( _ptr__522_subi[_539_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _540_i = (int)1;
    for(;(_540_i < _529_dimi);++ _540_i)
    {
      {
        _ptr__537_rptrb[safe_add( _540_i,(int)1 )] = safe_add( _ptr__537_rptrb[safe_add( _540_i,(int)1 )],_ptr__537_rptrb[_540_i] );
      }
    }
  }
  {
    int32_t _541_j = (int)0;
    for(;(_541_j < _530_dimj);++ _541_j)
    {
      {
        {
          int64_t _542_i = _ptr__532_cptrb[_541_j];
          for(;(_542_i < _ptr__532_cptrb[safe_add( _541_j,(int)1 )]);++ _542_i)
          {
            {
              int32_t _543_ridx = _ptr__522_subi[_ptr__531_cols[_542_i]];
              _ptr__538_rows[_ptr__537_rptrb[safe_add( _543_ridx,(int)1 )]] = _ptr__531_cols[_542_i];
              _ptr__537_rptrb[safe_add( _543_ridx,(int)1 )] = safe_add( _ptr__537_rptrb[safe_add( _543_ridx,(int)1 )],(int)1 );
            }
          }
        }
      }
    }
  }
  int32_t _544_nunique = (int)1;
  {
    int32_t _545_si = _ptr__522_subi[_ptr__538_rows[(int)0]];
    int32_t _546_sj = _ptr__523_subj[_ptr__538_rows[(int)0]];
    {
      int32_t _547_i = (int)1;
      for(;(_547_i < _528_nelm);++ _547_i)
      {
        {
          int32_t _548_idx = _ptr__538_rows[_547_i];
          if (((_545_si != _ptr__522_subi[_548_idx]) || (_546_sj != _ptr__523_subj[_548_idx])))
          {
            {
              _545_si = _ptr__522_subi[_548_idx];
              _546_sj = _ptr__523_subj[_548_idx];
              ++ _544_nunique;
            }
          }
          {}
        }
      }
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _549_tsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_544_nunique)));
  _checked_ptr_1<int32_t> _ptr__549_tsubi(_549_tsubi ? _549_tsubi->raw() : nullptr,_549_tsubi ? _549_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _550_tsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_544_nunique)));
  _checked_ptr_1<int32_t> _ptr__550_tsubj(_550_tsubj ? _550_tsubj->raw() : nullptr,_550_tsubj ? _550_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _551_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_544_nunique)));
  _checked_ptr_1<double> _ptr__551_tval(_551_tval ? _551_tval->raw() : nullptr,_551_tval ? _551_tval->size(0) : 0);
  {
    int32_t _552_idx = _ptr__538_rows[(int)0];
    _ptr__549_tsubi[(int)0] = _ptr__522_subi[_552_idx];
    _ptr__550_tsubj[(int)0] = _ptr__523_subj[_552_idx];
    _ptr__551_tval[(int)0] = _ptr__524_val[_552_idx];
  }
  int32_t _553_di = (int)1;
  int32_t _9367_ = (int)1;
  int64_t _9368_ = _528_nelm;
  for (int64_t _554_i = _9367_; _554_i < _9368_; ++_554_i)
  {
    {
      int32_t _555_idx = _ptr__538_rows[_554_i];
      if (((_ptr__549_tsubi[(_553_di - (int)1)] == _ptr__522_subi[_555_idx]) && (_ptr__550_tsubj[(_553_di - (int)1)] == _ptr__523_subj[_555_idx])))
      {
        {
          _ptr__551_tval[(_553_di - (int)1)] = (_ptr__551_tval[(_553_di - (int)1)] + _ptr__524_val[_555_idx]);
        }
      }
      else
      {
        {
          _ptr__549_tsubi[_553_di] = _ptr__522_subi[_555_idx];
          _ptr__550_tsubj[_553_di] = _ptr__523_subj[_555_idx];
          _ptr__551_tval[_553_di] = _ptr__524_val[_555_idx];
          ++ _553_di;
        }
      }
    }
  }
  ((*_525_tsubi_)((int)0)) = _549_tsubi;
  ((*_526_tsubj_)((int)0)) = _550_tsubj;
  ((*_527_tval_)((int)0)) = _551_tval;
}
// End mosek.fusion.CommonTools.tripletSort

// Begin mosek.fusion.CommonTools.argMSort
// Method mosek.fusion.CommonTools.argMSort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:67:6-87:7
void mosek::fusion::CommonTools::argMSort
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _556_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _557_vals) { mosek::fusion::p_CommonTools::argMSort(_556_idx,_557_vals); }
void mosek::fusion::p_CommonTools::argMSort(std::shared_ptr< monty::ndarray< int32_t,1 > > _556_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _557_vals)
{
  _checked_ptr_1<int32_t> _ptr__556_idx(_556_idx ? _556_idx->raw() : nullptr, _556_idx ? _556_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__557_vals(_557_vals ? _557_vals->raw() : nullptr, _557_vals ? _557_vals->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _558_temp = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)((_556_idx)->size(0)))));
  _checked_ptr_1<int32_t> _ptr__558_temp(_558_temp ? _558_temp->raw() : nullptr,_558_temp ? _558_temp->size(0) : 0);
  bool _559_alt = false;
  {
    int32_t _560_intvlen = (int)1;
    for(;(_560_intvlen < (int)((_556_idx)->size(0)));_560_intvlen *= (int)2)
    {
      {
        _559_alt = (!_559_alt);
        int32_t _9369_ = (int)0;
        int32_t _9370_ = (int)((_556_idx)->size(0));
        int32_t _9371_ = safe_mul( _560_intvlen,(int)2 );
        for (int32_t _561_i = _9369_; _561_i < _9370_; _561_i += _9371_)
        {
          {
            if (_559_alt)
            {
              {
                ::mosek::fusion::p_CommonTools::mergeInto(_556_idx,_558_temp,_557_vals,_561_i,safe_add( _561_i,_560_intvlen ),safe_add( _561_i,safe_mul( (int)2,_560_intvlen ) ));
              }
            }
            else
            {
              {
                ::mosek::fusion::p_CommonTools::mergeInto(_558_temp,_556_idx,_557_vals,_561_i,safe_add( _561_i,_560_intvlen ),safe_add( _561_i,safe_mul( (int)2,_560_intvlen ) ));
              }
            }
          }
        }
      }
    }
  }
  if (_559_alt)
  {
    {
      ::mosek::fusion::Utils::Tools::arraycopy(_558_temp,(int)0,_556_idx,(int)0,(int)((_556_idx)->size(0)));
    }
  }
  {}
}
// End mosek.fusion.CommonTools.argMSort

// Begin mosek.fusion.CommonTools.mergeInto
// Method mosek.fusion.CommonTools.mergeInto @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:29:6-54:7
void mosek::fusion::p_CommonTools::mergeInto(std::shared_ptr< monty::ndarray< int32_t,1 > > _562_src,std::shared_ptr< monty::ndarray< int32_t,1 > > _563_tgt,std::shared_ptr< monty::ndarray< int32_t,1 > > _564_vals,int32_t _565_si0,int32_t _566_si1_,int32_t _567_si2_)
{
  _checked_ptr_1<int32_t> _ptr__562_src(_562_src ? _562_src->raw() : nullptr, _562_src ? _562_src->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__563_tgt(_563_tgt ? _563_tgt->raw() : nullptr, _563_tgt ? _563_tgt->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__564_vals(_564_vals ? _564_vals->raw() : nullptr, _564_vals ? _564_vals->size(0) : 0);
  int32_t _9372_;
  bool _9373_ = (_567_si2_ > (int)((_562_src)->size(0)));
  if (_9373_)
  {
    _9372_ = (int)((_562_src)->size(0));
  }
  else
  {
    _9372_ = _567_si2_;
  }
  int32_t _568_si2 = _9372_;
  int32_t _9374_;
  bool _9375_ = (_566_si1_ > _568_si2);
  if (_9375_)
  {
    _9374_ = _568_si2;
  }
  else
  {
    _9374_ = _566_si1_;
  }
  int32_t _569_si1 = _9374_;
  if ((_569_si1 == _568_si2))
  {
    {
      ::mosek::fusion::Utils::Tools::arraycopy(_562_src,_565_si0,_563_tgt,_565_si0,(_569_si1 - _565_si0));
    }
  }
  else
  {
    {
      int32_t _570_i0 = _565_si0;
      int32_t _571_i1 = _569_si1;
      int32_t _572_i;
      {
        _572_i = _565_si0;
        for(;((_570_i0 < _569_si1) && (_571_i1 < _568_si2));++ _572_i)
        {
          {
            if ((_ptr__564_vals[_ptr__562_src[_570_i0]] < _ptr__564_vals[_ptr__562_src[_571_i1]]))
            {
              {
                _ptr__563_tgt[_572_i] = _ptr__562_src[_570_i0];
                ++ _570_i0;
              }
            }
            else
            {
              {
                _ptr__563_tgt[_572_i] = _ptr__562_src[_571_i1];
                ++ _571_i1;
              }
            }
          }
        }
      }
      while ( (_570_i0 < _569_si1) )
      {
        {
          _ptr__563_tgt[_572_i] = _ptr__562_src[_570_i0];
          ++ _570_i0;
          ++ _572_i;
        }
      }
      while ( (_571_i1 < _568_si2) )
      {
        {
          _ptr__563_tgt[_572_i] = _ptr__562_src[_571_i1];
          ++ _571_i1;
          ++ _572_i;
        }
      }
    }
  }
}
// End mosek.fusion.CommonTools.mergeInto

// Begin mosek.fusion.CommonTools.argQsort
// Method mosek.fusion.CommonTools.argQsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:16:6-22:7
void mosek::fusion::CommonTools::argQsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _573_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _574_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _575_vals2,
    int64_t _576_first,
    int64_t _577_last) { mosek::fusion::p_CommonTools::argQsort(_573_idx,_574_vals1,_575_vals2,_576_first,_577_last); }
void mosek::fusion::p_CommonTools::argQsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _573_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _574_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _575_vals2,int64_t _576_first,int64_t _577_last)
{
  _checked_ptr_1<int64_t> _ptr__573_idx(_573_idx ? _573_idx->raw() : nullptr, _573_idx ? _573_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__574_vals1(_574_vals1 ? _574_vals1->raw() : nullptr, _574_vals1 ? _574_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__575_vals2(_575_vals2 ? _575_vals2->raw() : nullptr, _575_vals2 ? _575_vals2->size(0) : 0);
  if ((NULL == _575_vals2.get()))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_573_idx,_574_vals1,_576_first,_577_last);
    }
  }
  else
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_573_idx,_574_vals1,_575_vals2,_576_first,_577_last);
    }
  }
}
// End mosek.fusion.CommonTools.argQsort

// Begin mosek.fusion.CommonTools.argQsort
// Method mosek.fusion.CommonTools.argQsort @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/CommonUtil.mbi:8:6-14:7
void mosek::fusion::CommonTools::argQsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _578_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _579_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _580_vals2,
    int64_t _581_first,
    int64_t _582_last) { mosek::fusion::p_CommonTools::argQsort(_578_idx,_579_vals1,_580_vals2,_581_first,_582_last); }
void mosek::fusion::p_CommonTools::argQsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _578_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _579_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _580_vals2,int64_t _581_first,int64_t _582_last)
{
  _checked_ptr_1<int64_t> _ptr__578_idx(_578_idx ? _578_idx->raw() : nullptr, _578_idx ? _578_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__579_vals1(_579_vals1 ? _579_vals1->raw() : nullptr, _579_vals1 ? _579_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__580_vals2(_580_vals2 ? _580_vals2->raw() : nullptr, _580_vals2 ? _580_vals2->size(0) : 0);
  if ((NULL == _580_vals2.get()))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_578_idx,_579_vals1,_581_first,_582_last);
    }
  }
  else
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_578_idx,_579_vals1,_580_vals2,_581_first,_582_last);
    }
  }
}
// End mosek.fusion.CommonTools.argQsort

void mosek::fusion::p_CommonTools::destroy()
{
}
void mosek::fusion::CommonTools::destroy() { mosek::fusion::p_CommonTools::_get_impl(this)->destroy(); }
// } class CommonTools
// class SolutionStruct { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5469:4-5804:5
mosek::fusion::SolutionStruct::SolutionStruct(mosek::fusion::p_SolutionStruct *_impl) : _impl(_impl) { /*std::cout << "SolutionStruct() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::SolutionStruct::~SolutionStruct() { delete _impl; _impl = nullptr; }
mosek::fusion::p_SolutionStruct::p_SolutionStruct
  (::mosek::fusion::SolutionStruct * _pubthis) :     _pubthis(_pubthis)
{}
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_accy() { return _impl->accy; }
void mosek::fusion::SolutionStruct::set_accy(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->accy = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_accx() { return _impl->accx; }
void mosek::fusion::SolutionStruct::set_accx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->accx = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::SolutionStruct::get_accptr() { return _impl->accptr; }
void mosek::fusion::SolutionStruct::set_accptr(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->accptr = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_yx() { return _impl->yx; }
void mosek::fusion::SolutionStruct::set_yx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->yx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_sux() { return _impl->sux; }
void mosek::fusion::SolutionStruct::set_sux(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->sux = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_slx() { return _impl->slx; }
void mosek::fusion::SolutionStruct::set_slx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->slx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_bars() { return _impl->bars; }
void mosek::fusion::SolutionStruct::set_bars(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->bars = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_barx() { return _impl->barx; }
void mosek::fusion::SolutionStruct::set_barx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->barx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_y() { return _impl->y; }
void mosek::fusion::SolutionStruct::set_y(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->y = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_suc() { return _impl->suc; }
void mosek::fusion::SolutionStruct::set_suc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->suc = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_slc() { return _impl->slc; }
void mosek::fusion::SolutionStruct::set_slc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->slc = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_xx() { return _impl->xx; }
void mosek::fusion::SolutionStruct::set_xx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->xx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_xc() { return _impl->xc; }
void mosek::fusion::SolutionStruct::set_xc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->xc = _val; }
double mosek::fusion::SolutionStruct::get_dobj() { return _impl->dobj; }
void mosek::fusion::SolutionStruct::set_dobj(double _val) { _impl->dobj = _val; }
double mosek::fusion::SolutionStruct::get_pobj() { return _impl->pobj; }
void mosek::fusion::SolutionStruct::set_pobj(double _val) { _impl->pobj = _val; }
mosek::fusion::ProblemStatus mosek::fusion::SolutionStruct::get_probstatus() { return _impl->probstatus; }
void mosek::fusion::SolutionStruct::set_probstatus(mosek::fusion::ProblemStatus _val) { _impl->probstatus = _val; }
mosek::fusion::SolutionStatus mosek::fusion::SolutionStruct::get_dstatus() { return _impl->dstatus; }
void mosek::fusion::SolutionStruct::set_dstatus(mosek::fusion::SolutionStatus _val) { _impl->dstatus = _val; }
mosek::fusion::SolutionStatus mosek::fusion::SolutionStruct::get_pstatus() { return _impl->pstatus; }
void mosek::fusion::SolutionStruct::set_pstatus(mosek::fusion::SolutionStatus _val) { _impl->pstatus = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numbarvar() { return _impl->sol_numbarvar; }
void mosek::fusion::SolutionStruct::set_sol_numbarvar(int32_t _val) { _impl->sol_numbarvar = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numaccelm() { return _impl->sol_numaccelm; }
void mosek::fusion::SolutionStruct::set_sol_numaccelm(int32_t _val) { _impl->sol_numaccelm = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numacc() { return _impl->sol_numacc; }
void mosek::fusion::SolutionStruct::set_sol_numacc(int32_t _val) { _impl->sol_numacc = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numvar() { return _impl->sol_numvar; }
void mosek::fusion::SolutionStruct::set_sol_numvar(int32_t _val) { _impl->sol_numvar = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numcon() { return _impl->sol_numcon; }
void mosek::fusion::SolutionStruct::set_sol_numcon(int32_t _val) { _impl->sol_numcon = _val; }
// mosek.fusion.SolutionStruct.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5744:6-5803:7
mosek::fusion::SolutionStruct::t mosek::fusion::p_SolutionStruct::_new_SolutionStruct
  (int32_t _583_numvar,
    int32_t _584_numcon,
    int32_t _585_numbarvar,
    int32_t _586_numacc,
    int32_t _587_numaccelm)
{ return new mosek::fusion::SolutionStruct(_583_numvar,_584_numcon,_585_numbarvar,_586_numacc,_587_numaccelm); }
mosek::fusion::SolutionStruct::SolutionStruct
  (int32_t _583_numvar,
    int32_t _584_numcon,
    int32_t _585_numbarvar,
    int32_t _586_numacc,
    int32_t _587_numaccelm) : 
   SolutionStruct(new mosek::fusion::p_SolutionStruct(this))
{ mosek::fusion::p_SolutionStruct::_get_impl(this)->_initialize(_583_numvar,_584_numcon,_585_numbarvar,_586_numacc,_587_numaccelm); }
void mosek::fusion::p_SolutionStruct::_initialize  (int32_t _583_numvar,
    int32_t _584_numcon,
    int32_t _585_numbarvar,
    int32_t _586_numacc,
    int32_t _587_numaccelm)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    pobj = 0.0;
    dobj = 0.0;
    pstatus = ::mosek::fusion::SolutionStatus::Unknown;
    dstatus = ::mosek::fusion::SolutionStatus::Unknown;
    probstatus = ::mosek::fusion::ProblemStatus::Unknown;
    sol_numvar = _583_numvar;
    sol_numcon = _584_numcon;
    sol_numbarvar = _585_numbarvar;
    sol_numacc = _586_numacc;
    sol_numaccelm = _587_numaccelm;
    slx = nullptr;
    sux = nullptr;
    xx = nullptr;
    yx = nullptr;
    slc = nullptr;
    suc = nullptr;
    y = nullptr;
    xc = nullptr;
    barx = nullptr;
    bars = nullptr;
    accx = nullptr;
    accy = nullptr;
    accptr = nullptr;
    if ((_583_numvar > (int)0))
    {
      {
        slx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_583_numvar)));
        sux = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_583_numvar)));
        xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_583_numvar)));
        yx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_583_numvar)));
      }
    }
    {}
    if ((_585_numbarvar > (int)0))
    {
      {
        barx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_585_numbarvar)));
        bars = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_585_numbarvar)));
      }
    }
    {}
    if ((_584_numcon > (int)0))
    {
      {
        slc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_584_numcon)));
        suc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_584_numcon)));
        y = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_584_numcon)));
        xc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_584_numcon)));
      }
    }
    {}
    if ((_586_numacc > (int)0))
    {
      {
        accx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_587_numaccelm)));
        accy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_587_numaccelm)));
        accptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _586_numacc,(int)1 ))));
      }
    }
    {}
  }
}
// mosek.fusion.SolutionStruct.ctor

// mosek.fusion.SolutionStruct.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5711:6-5741:7
mosek::fusion::SolutionStruct::t mosek::fusion::p_SolutionStruct::_new_SolutionStruct
  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _588_that)
{ return new mosek::fusion::SolutionStruct(_588_that); }
mosek::fusion::SolutionStruct::SolutionStruct
  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _588_that) : 
   SolutionStruct(new mosek::fusion::p_SolutionStruct(this))
{ mosek::fusion::p_SolutionStruct::_get_impl(this)->_initialize(_588_that); }
void mosek::fusion::p_SolutionStruct::_initialize  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _588_that)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    sol_numcon = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sol_numcon;
    sol_numvar = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sol_numvar;
    sol_numbarvar = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sol_numbarvar;
    sol_numacc = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sol_numacc;
    sol_numaccelm = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sol_numaccelm;
    pstatus = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->pstatus;
    dstatus = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->dstatus;
    probstatus = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->probstatus;
    pobj = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->pobj;
    dobj = mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->dobj;
    std::shared_ptr< monty::ndarray< double,1 > > _9376_;
    bool _9377_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->xc.get());
    if (_9377_)
    {
      _9376_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->xc);
    }
    else
    {
      _9376_ = nullptr;
    }
    xc = _9376_;
    std::shared_ptr< monty::ndarray< double,1 > > _9378_;
    bool _9379_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->xx.get());
    if (_9379_)
    {
      _9378_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->xx);
    }
    else
    {
      _9378_ = nullptr;
    }
    xx = _9378_;
    std::shared_ptr< monty::ndarray< double,1 > > _9380_;
    bool _9381_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->yx.get());
    if (_9381_)
    {
      _9380_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->yx);
    }
    else
    {
      _9380_ = nullptr;
    }
    yx = _9380_;
    std::shared_ptr< monty::ndarray< double,1 > > _9382_;
    bool _9383_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->slc.get());
    if (_9383_)
    {
      _9382_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->slc);
    }
    else
    {
      _9382_ = nullptr;
    }
    slc = _9382_;
    std::shared_ptr< monty::ndarray< double,1 > > _9384_;
    bool _9385_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->suc.get());
    if (_9385_)
    {
      _9384_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->suc);
    }
    else
    {
      _9384_ = nullptr;
    }
    suc = _9384_;
    std::shared_ptr< monty::ndarray< double,1 > > _9386_;
    bool _9387_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->y.get());
    if (_9387_)
    {
      _9386_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->y);
    }
    else
    {
      _9386_ = nullptr;
    }
    y = _9386_;
    std::shared_ptr< monty::ndarray< double,1 > > _9388_;
    bool _9389_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->barx.get());
    if (_9389_)
    {
      _9388_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->barx);
    }
    else
    {
      _9388_ = nullptr;
    }
    barx = _9388_;
    std::shared_ptr< monty::ndarray< double,1 > > _9390_;
    bool _9391_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->bars.get());
    if (_9391_)
    {
      _9390_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->bars);
    }
    else
    {
      _9390_ = nullptr;
    }
    bars = _9390_;
    std::shared_ptr< monty::ndarray< double,1 > > _9392_;
    bool _9393_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->slx.get());
    if (_9393_)
    {
      _9392_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->slx);
    }
    else
    {
      _9392_ = nullptr;
    }
    slx = _9392_;
    std::shared_ptr< monty::ndarray< double,1 > > _9394_;
    bool _9395_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sux.get());
    if (_9395_)
    {
      _9394_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->sux);
    }
    else
    {
      _9394_ = nullptr;
    }
    sux = _9394_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9396_;
    bool _9397_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accptr.get());
    if (_9397_)
    {
      _9396_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accptr);
    }
    else
    {
      _9396_ = nullptr;
    }
    accptr = _9396_;
    std::shared_ptr< monty::ndarray< double,1 > > _9398_;
    bool _9399_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accx.get());
    if (_9399_)
    {
      _9398_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accx);
    }
    else
    {
      _9398_ = nullptr;
    }
    accx = _9398_;
    std::shared_ptr< monty::ndarray< double,1 > > _9400_;
    bool _9401_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accy.get());
    if (_9401_)
    {
      _9400_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_588_that)->accy);
    }
    else
    {
      _9400_ = nullptr;
    }
    accy = _9400_;
  }
}
// mosek.fusion.SolutionStruct.ctor

// Begin mosek.fusion.SolutionStruct.clone
// Method mosek.fusion.SolutionStruct.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5742:6-79
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::SolutionStruct :: __mosek_2fusion_2SolutionStruct__clone() { return mosek::fusion::p_SolutionStruct::_get_impl(this)->__mosek_2fusion_2SolutionStruct__clone(); }
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::SolutionStruct::clone() { return __mosek_2fusion_2SolutionStruct__clone(); }
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_SolutionStruct::__mosek_2fusion_2SolutionStruct__clone()
{
  return ::mosek::fusion::p_SolutionStruct::_new_SolutionStruct(::mosek::fusion::SolutionStruct::t(_pubthis));
}
// End mosek.fusion.SolutionStruct.clone

// Begin mosek.fusion.SolutionStruct.resize
// Method mosek.fusion.SolutionStruct.resize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5536:6-5706:7
void mosek::fusion::SolutionStruct :: resize(int32_t _589_numvar,int32_t _590_numcon,int32_t _591_numbarvar,int32_t _592_numacc,int32_t _593_numaccelm) { mosek::fusion::p_SolutionStruct::_get_impl(this)->resize(_589_numvar,_590_numcon,_591_numbarvar,_592_numacc,_593_numaccelm); }
void mosek::fusion::p_SolutionStruct::resize(int32_t _589_numvar,int32_t _590_numcon,int32_t _591_numbarvar,int32_t _592_numacc,int32_t _593_numaccelm)
{
  if ((_592_numacc <= (int)0))
  {
    {
      sol_numacc = (int)0;
    }
  }
  else if((_592_numacc <= ((int)((accptr)->size(0)) - (int)1)))
  {
    {
      sol_numacc = _592_numacc;
    }
  }
  else
  {
    {
      int32_t _594_newnum = safe_mul( ((int)((accptr)->size(0)) - (int)1),(int)2 );
      if ((_594_newnum < _592_numacc))
      {
        {
          _594_newnum = _592_numacc;
        }
      }
      {}
      std::shared_ptr< monty::ndarray< int32_t,1 > > _595_new_accptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _594_newnum,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__595_new_accptr(_595_new_accptr ? _595_new_accptr->raw() : nullptr,_595_new_accptr ? _595_new_accptr->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(accptr,(int)0,_595_new_accptr,(int)0,safe_add( _592_numacc,(int)1 ));
      accptr = _595_new_accptr;
      sol_numacc = _592_numacc;
    }
  }
  if ((_593_numaccelm <= (int)0))
  {
    {
      sol_numaccelm = (int)0;
    }
  }
  else if((_593_numaccelm > (int)((accx)->size(0))))
  {
    {
      sol_numaccelm = _593_numaccelm;
    }
  }
  else
  {
    {
      int32_t _596_newnum = safe_mul( (int)((accx)->size(0)),(int)2 );
      if ((_596_newnum < _593_numaccelm))
      {
        {
          _596_newnum = _593_numaccelm;
        }
      }
      {}
      std::shared_ptr< monty::ndarray< double,1 > > _597_new_accx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_596_newnum)));
      _checked_ptr_1<double> _ptr__597_new_accx(_597_new_accx ? _597_new_accx->raw() : nullptr,_597_new_accx ? _597_new_accx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _598_new_accy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_596_newnum)));
      _checked_ptr_1<double> _ptr__598_new_accy(_598_new_accy ? _598_new_accy->raw() : nullptr,_598_new_accy ? _598_new_accy->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(accx,(int)0,_597_new_accx,(int)0,_593_numaccelm);
      ::mosek::fusion::Utils::Tools::arraycopy(accy,(int)0,_598_new_accy,(int)0,_593_numaccelm);
      accx = _597_new_accx;
      accy = _598_new_accy;
      sol_numaccelm = _593_numaccelm;
    }
  }
  if ((_591_numbarvar < (int)0))
  {
    {
    }
  }
  else if((_591_numbarvar <= sol_numbarvar))
  {
    {
      sol_numbarvar = _591_numbarvar;
    }
  }
  else if((_591_numbarvar <= (int)((barx)->size(0))))
  {
    {
    }
  }
  else
  {
    {
      int32_t _599_newsize;
      if ((safe_mul( sol_numbarvar,(int)2 ) > safe_add( _591_numbarvar,(int)100 )))
      {
        {
          _599_newsize = safe_mul( sol_numbarvar,(int)2 );
        }
      }
      else
      {
        {
          _599_newsize = safe_add( _591_numbarvar,(int)100 );
        }
      }
      barx = ::mosek::fusion::p_CommonTools::resize(barx,_599_newsize);
      bars = ::mosek::fusion::p_CommonTools::resize(bars,_599_newsize);
      sol_numbarvar = _599_newsize;
    }
  }
  if ((_589_numvar < (int)0))
  {
    {
    }
  }
  else if((_589_numvar <= sol_numvar))
  {
    {
      sol_numvar = _589_numvar;
    }
  }
  else if((_589_numvar <= (int)((xx)->size(0))))
  {
    {
      int32_t _9402_ = sol_numvar;
      int32_t _9403_ = _589_numvar;
      for (int32_t _600_i = _9402_; _600_i < _9403_; ++_600_i)
      {
        {
          ((*slx)(_600_i)) = 0.0;
          ((*sux)(_600_i)) = 0.0;
          ((*xx)(_600_i)) = 0.0;
          ((*yx)(_600_i)) = 0.0;
        }
      }
      sol_numvar = _589_numvar;
    }
  }
  else
  {
    {
      int32_t _601_newsize;
      if ((safe_mul( sol_numvar,(int)2 ) > safe_add( _589_numvar,(int)100 )))
      {
        {
          _601_newsize = safe_mul( sol_numvar,(int)2 );
        }
      }
      else
      {
        {
          _601_newsize = safe_add( _589_numvar,(int)100 );
        }
      }
      std::shared_ptr< monty::ndarray< double,1 > > _602_newslx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_601_newsize)));
      _checked_ptr_1<double> _ptr__602_newslx(_602_newslx ? _602_newslx->raw() : nullptr,_602_newslx ? _602_newslx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(slx,(int)0,_602_newslx,(int)0,(int)((slx)->size(0)));
      slx = _602_newslx;
      std::shared_ptr< monty::ndarray< double,1 > > _603_newsux = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_601_newsize)));
      _checked_ptr_1<double> _ptr__603_newsux(_603_newsux ? _603_newsux->raw() : nullptr,_603_newsux ? _603_newsux->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(sux,(int)0,_603_newsux,(int)0,(int)((sux)->size(0)));
      sux = _603_newsux;
      std::shared_ptr< monty::ndarray< double,1 > > _604_newxx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_601_newsize)));
      _checked_ptr_1<double> _ptr__604_newxx(_604_newxx ? _604_newxx->raw() : nullptr,_604_newxx ? _604_newxx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(xx,(int)0,_604_newxx,(int)0,(int)((xx)->size(0)));
      xx = _604_newxx;
      std::shared_ptr< monty::ndarray< double,1 > > _605_newyx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_601_newsize)));
      _checked_ptr_1<double> _ptr__605_newyx(_605_newyx ? _605_newyx->raw() : nullptr,_605_newyx ? _605_newyx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(yx,(int)0,_605_newyx,(int)0,(int)((yx)->size(0)));
      yx = _605_newyx;
      {
        int32_t _606_i = sol_numvar;
        for(;(_606_i < _589_numvar);++ _606_i)
        {
          {
            ((*slx)(_606_i)) = 0.0;
            ((*sux)(_606_i)) = 0.0;
            ((*xx)(_606_i)) = 0.0;
          }
        }
      }
      sol_numvar = _589_numvar;
    }
  }
  if ((_590_numcon < (int)0))
  {
    {
    }
  }
  else if((_590_numcon <= sol_numcon))
  {
    {
      sol_numcon = _590_numcon;
    }
  }
  else if((_590_numcon <= (int)((xx)->size(0))))
  {
    {
      int32_t _9404_ = sol_numcon;
      int32_t _9405_ = _590_numcon;
      for (int32_t _607_i = _9404_; _607_i < _9405_; ++_607_i)
      {
        {
          ((*slc)(_607_i)) = 0.0;
          ((*suc)(_607_i)) = 0.0;
          ((*xc)(_607_i)) = 0.0;
          ((*y)(_607_i)) = 0.0;
        }
      }
      sol_numcon = _590_numcon;
    }
  }
  else
  {
    {
      int32_t _608_newsize;
      if ((safe_mul( sol_numcon,(int)2 ) > safe_add( _590_numcon,(int)100 )))
      {
        {
          _608_newsize = safe_mul( sol_numcon,(int)2 );
        }
      }
      else
      {
        {
          _608_newsize = safe_add( _590_numcon,(int)100 );
        }
      }
      std::shared_ptr< monty::ndarray< double,1 > > _609_newslc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_608_newsize)));
      _checked_ptr_1<double> _ptr__609_newslc(_609_newslc ? _609_newslc->raw() : nullptr,_609_newslc ? _609_newslc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(slc,(int)0,_609_newslc,(int)0,(int)((slc)->size(0)));
      slc = _609_newslc;
      std::shared_ptr< monty::ndarray< double,1 > > _610_newsuc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_608_newsize)));
      _checked_ptr_1<double> _ptr__610_newsuc(_610_newsuc ? _610_newsuc->raw() : nullptr,_610_newsuc ? _610_newsuc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(suc,(int)0,_610_newsuc,(int)0,(int)((suc)->size(0)));
      suc = _610_newsuc;
      std::shared_ptr< monty::ndarray< double,1 > > _611_newxc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_608_newsize)));
      _checked_ptr_1<double> _ptr__611_newxc(_611_newxc ? _611_newxc->raw() : nullptr,_611_newxc ? _611_newxc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(xc,(int)0,_611_newxc,(int)0,(int)((xc)->size(0)));
      xc = _611_newxc;
      std::shared_ptr< monty::ndarray< double,1 > > _612_newy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_608_newsize)));
      _checked_ptr_1<double> _ptr__612_newy(_612_newy ? _612_newy->raw() : nullptr,_612_newy ? _612_newy->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(y,(int)0,_612_newy,(int)0,(int)((y)->size(0)));
      y = _612_newy;
      int32_t _9406_ = sol_numcon;
      int32_t _9407_ = _590_numcon;
      for (int32_t _613_i = _9406_; _613_i < _9407_; ++_613_i)
      {
        {
          ((*slc)(_613_i)) = 0.0;
          ((*suc)(_613_i)) = 0.0;
          ((*xc)(_613_i)) = 0.0;
          ((*y)(_613_i)) = 0.0;
        }
      }
      sol_numcon = _590_numcon;
    }
  }
}
// End mosek.fusion.SolutionStruct.resize

// Begin mosek.fusion.SolutionStruct.isDualAcceptable
// Method mosek.fusion.SolutionStruct.isDualAcceptable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5531:6-5534:7
bool mosek::fusion::SolutionStruct :: isDualAcceptable(mosek::fusion::AccSolutionStatus _614_acceptable_sol) { return mosek::fusion::p_SolutionStruct::_get_impl(this)->isDualAcceptable(_614_acceptable_sol); }
bool mosek::fusion::p_SolutionStruct::isDualAcceptable(mosek::fusion::AccSolutionStatus _614_acceptable_sol)
{
  return isAcceptable(dstatus,_614_acceptable_sol);
}
// End mosek.fusion.SolutionStruct.isDualAcceptable

// Begin mosek.fusion.SolutionStruct.isPrimalAcceptable
// Method mosek.fusion.SolutionStruct.isPrimalAcceptable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5526:6-5529:7
bool mosek::fusion::SolutionStruct :: isPrimalAcceptable(mosek::fusion::AccSolutionStatus _615_acceptable_sol) { return mosek::fusion::p_SolutionStruct::_get_impl(this)->isPrimalAcceptable(_615_acceptable_sol); }
bool mosek::fusion::p_SolutionStruct::isPrimalAcceptable(mosek::fusion::AccSolutionStatus _615_acceptable_sol)
{
  return isAcceptable(pstatus,_615_acceptable_sol);
}
// End mosek.fusion.SolutionStruct.isPrimalAcceptable

// Begin mosek.fusion.SolutionStruct.isAcceptable
// Method mosek.fusion.SolutionStruct.isAcceptable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5500:6-5524:7
bool mosek::fusion::p_SolutionStruct::isAcceptable(mosek::fusion::SolutionStatus _616_stat,mosek::fusion::AccSolutionStatus _617_accstat)
{
  if ((_617_accstat == ::mosek::fusion::AccSolutionStatus::Anything))
  {
    {
      return true;
    }
  }
  else if((_617_accstat == ::mosek::fusion::AccSolutionStatus::Optimal))
  {
    {
      return (_616_stat == ::mosek::fusion::SolutionStatus::Optimal);
    }
  }
  else if((_617_accstat == ::mosek::fusion::AccSolutionStatus::Feasible))
  {
    {
      return ((_616_stat == ::mosek::fusion::SolutionStatus::Optimal) || (_616_stat == ::mosek::fusion::SolutionStatus::Feasible));
    }
  }
  else if((_617_accstat == ::mosek::fusion::AccSolutionStatus::Certificate))
  {
    {
      return (_616_stat == ::mosek::fusion::SolutionStatus::Certificate);
    }
  }
  {}
  return false;
}
// End mosek.fusion.SolutionStruct.isAcceptable

void mosek::fusion::p_SolutionStruct::destroy()
{
  accy.reset();
  accx.reset();
  accptr.reset();
  yx.reset();
  sux.reset();
  slx.reset();
  bars.reset();
  barx.reset();
  y.reset();
  suc.reset();
  slc.reset();
  xx.reset();
  xc.reset();
}
void mosek::fusion::SolutionStruct::destroy() { mosek::fusion::p_SolutionStruct::_get_impl(this)->destroy(); }
// } class SolutionStruct
// class RowBlockManager { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4944:4-5464:5
mosek::fusion::RowBlockManager::RowBlockManager(mosek::fusion::p_RowBlockManager *_impl) : _impl(_impl) { /*std::cout << "RowBlockManager() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::RowBlockManager::~RowBlockManager() { delete _impl; _impl = nullptr; }
mosek::fusion::p_RowBlockManager::p_RowBlockManager
  (::mosek::fusion::RowBlockManager * _pubthis) :     _pubthis(_pubthis)
{}
int32_t mosek::fusion::RowBlockManager::get_varidx_used() { return _impl->varidx_used; }
void mosek::fusion::RowBlockManager::set_varidx_used(int32_t _val) { _impl->varidx_used = _val; }
int32_t mosek::fusion::RowBlockManager::get_code_used() { return _impl->code_used; }
void mosek::fusion::RowBlockManager::set_code_used(int32_t _val) { _impl->code_used = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::RowBlockManager::get_cconst() { return _impl->cconst; }
void mosek::fusion::RowBlockManager::set_cconst(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->cconst = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_code() { return _impl->code; }
void mosek::fusion::RowBlockManager::set_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->code = _val; }
int32_t mosek::fusion::RowBlockManager::get_first_free_codeitem() { return _impl->first_free_codeitem; }
void mosek::fusion::RowBlockManager::set_first_free_codeitem(int32_t _val) { _impl->first_free_codeitem = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_param_code_sizes() { return _impl->param_code_sizes; }
void mosek::fusion::RowBlockManager::set_param_code_sizes(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->param_code_sizes = _val; }
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::RowBlockManager::get_param_varidx() { return _impl->param_varidx; }
void mosek::fusion::RowBlockManager::set_param_varidx(std::shared_ptr< monty::ndarray< int64_t,1 > > _val) { _impl->param_varidx = _val; }
int32_t mosek::fusion::RowBlockManager::get_first_free_entry() { return _impl->first_free_entry; }
void mosek::fusion::RowBlockManager::set_first_free_entry(int32_t _val) { _impl->first_free_entry = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_code_ptr() { return _impl->row_code_ptr; }
void mosek::fusion::RowBlockManager::set_row_code_ptr(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_code_ptr = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_param_ptre() { return _impl->row_param_ptre; }
void mosek::fusion::RowBlockManager::set_row_param_ptre(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_param_ptre = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_param_ptrb() { return _impl->row_param_ptrb; }
void mosek::fusion::RowBlockManager::set_row_param_ptrb(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_param_ptrb = _val; }
monty::rc_ptr< ::mosek::fusion::LinkedBlocks > mosek::fusion::RowBlockManager::get_blocks() { return _impl->blocks; }
void mosek::fusion::RowBlockManager::set_blocks(monty::rc_ptr< ::mosek::fusion::LinkedBlocks > _val) { _impl->blocks = _val; }
// mosek.fusion.RowBlockManager.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5007:6-5023:7
mosek::fusion::RowBlockManager::t mosek::fusion::p_RowBlockManager::_new_RowBlockManager
  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _618_that)
{ return new mosek::fusion::RowBlockManager(_618_that); }
mosek::fusion::RowBlockManager::RowBlockManager
  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _618_that) : 
   RowBlockManager(new mosek::fusion::p_RowBlockManager(this))
{ mosek::fusion::p_RowBlockManager::_get_impl(this)->_initialize(_618_that); }
void mosek::fusion::p_RowBlockManager::_initialize  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _618_that)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    blocks = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->blocks);
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9408_;
    bool _9409_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_param_ptrb.get());
    if (_9409_)
    {
      _9408_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_param_ptrb);
    }
    else
    {
      _9408_ = nullptr;
    }
    row_param_ptrb = _9408_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9410_;
    bool _9411_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_param_ptre.get());
    if (_9411_)
    {
      _9410_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_param_ptre);
    }
    else
    {
      _9410_ = nullptr;
    }
    row_param_ptre = _9410_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9412_;
    bool _9413_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_code_ptr.get());
    if (_9413_)
    {
      _9412_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->row_code_ptr);
    }
    else
    {
      _9412_ = nullptr;
    }
    row_code_ptr = _9412_;
    first_free_entry = mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->first_free_entry;
    std::shared_ptr< monty::ndarray< int64_t,1 > > _9414_;
    bool _9415_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->param_varidx.get());
    if (_9415_)
    {
      _9414_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->param_varidx);
    }
    else
    {
      _9414_ = nullptr;
    }
    param_varidx = _9414_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9416_;
    bool _9417_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->param_code_sizes.get());
    if (_9417_)
    {
      _9416_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->param_code_sizes);
    }
    else
    {
      _9416_ = nullptr;
    }
    param_code_sizes = _9416_;
    first_free_codeitem = mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->first_free_codeitem;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9418_;
    bool _9419_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->code.get());
    if (_9419_)
    {
      _9418_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->code);
    }
    else
    {
      _9418_ = nullptr;
    }
    code = _9418_;
    std::shared_ptr< monty::ndarray< double,1 > > _9420_;
    bool _9421_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->cconst.get());
    if (_9421_)
    {
      _9420_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->cconst);
    }
    else
    {
      _9420_ = nullptr;
    }
    cconst = _9420_;
    code_used = mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->code_used;
    varidx_used = mosek::fusion::p_RowBlockManager::_get_impl(_618_that)->varidx_used;
  }
}
// mosek.fusion.RowBlockManager.ctor

// mosek.fusion.RowBlockManager.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4991:6-5005:7
mosek::fusion::RowBlockManager::t mosek::fusion::p_RowBlockManager::_new_RowBlockManager
  ()
{ return new mosek::fusion::RowBlockManager(); }
mosek::fusion::RowBlockManager::RowBlockManager
  () : 
   RowBlockManager(new mosek::fusion::p_RowBlockManager(this))
{ mosek::fusion::p_RowBlockManager::_get_impl(this)->_initialize(); }
void mosek::fusion::p_RowBlockManager::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    blocks = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    row_param_ptrb = nullptr;
    row_param_ptre = nullptr;
    row_code_ptr = nullptr;
    first_free_entry = (int)0;
    param_varidx = nullptr;
    param_code_sizes = nullptr;
    first_free_codeitem = (int)0;
    code = nullptr;
    cconst = nullptr;
    code_used = (int)0;
    varidx_used = (int)0;
  }
}
// mosek.fusion.RowBlockManager.ctor

// Begin mosek.fusion.RowBlockManager.num_parameterized
// Method mosek.fusion.RowBlockManager.num_parameterized @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5345:6-69
int32_t mosek::fusion::RowBlockManager :: num_parameterized() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->num_parameterized(); }
int32_t mosek::fusion::p_RowBlockManager::num_parameterized()
{
  return varidx_used;
}
// End mosek.fusion.RowBlockManager.num_parameterized

// Begin mosek.fusion.RowBlockManager.is_parameterized
// Method mosek.fusion.RowBlockManager.is_parameterized @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5344:6-69
bool mosek::fusion::RowBlockManager :: is_parameterized() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->is_parameterized(); }
bool mosek::fusion::p_RowBlockManager::is_parameterized()
{
  return (code_used > (int)0);
}
// End mosek.fusion.RowBlockManager.is_parameterized

// Begin mosek.fusion.RowBlockManager.blocksize
// Method mosek.fusion.RowBlockManager.blocksize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5343:6-79
int32_t mosek::fusion::RowBlockManager :: blocksize(int32_t _619_id) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->blocksize(_619_id); }
int32_t mosek::fusion::p_RowBlockManager::blocksize(int32_t _619_id)
{
  return blocks->blocksize(_619_id);
}
// End mosek.fusion.RowBlockManager.blocksize

// Begin mosek.fusion.RowBlockManager.block_capacity
// Method mosek.fusion.RowBlockManager.block_capacity @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5342:6-79
int32_t mosek::fusion::RowBlockManager :: block_capacity() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->block_capacity(); }
int32_t mosek::fusion::p_RowBlockManager::block_capacity()
{
  return blocks->block_capacity();
}
// End mosek.fusion.RowBlockManager.block_capacity

// Begin mosek.fusion.RowBlockManager.capacity
// Method mosek.fusion.RowBlockManager.capacity @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5341:6-67
int32_t mosek::fusion::RowBlockManager :: capacity() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->capacity(); }
int32_t mosek::fusion::p_RowBlockManager::capacity()
{
  return blocks->capacity();
}
// End mosek.fusion.RowBlockManager.capacity

// Begin mosek.fusion.RowBlockManager.get
// Method mosek.fusion.RowBlockManager.get @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5340:6-102
void mosek::fusion::RowBlockManager :: get(int32_t _620_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _621_target,int32_t _622_offset) { mosek::fusion::p_RowBlockManager::_get_impl(this)->get(_620_id,_621_target,_622_offset); }
void mosek::fusion::p_RowBlockManager::get(int32_t _620_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _621_target,int32_t _622_offset)
{
  _checked_ptr_1<int32_t> _ptr__621_target(_621_target ? _621_target->raw() : nullptr, _621_target ? _621_target->size(0) : 0);
  blocks->get(_620_id,_621_target,_622_offset);
}
// End mosek.fusion.RowBlockManager.get

// Begin mosek.fusion.RowBlockManager.evaluate
// Method mosek.fusion.RowBlockManager.evaluate @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5272:6-5338:7
void mosek::fusion::RowBlockManager :: evaluate(monty::rc_ptr< ::mosek::fusion::WorkStack > _623_xs,std::shared_ptr< monty::ndarray< double,1 > > _624_param_value,std::shared_ptr< monty::ndarray< int32_t,1 > > _625_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _626_subj,std::shared_ptr< monty::ndarray< double,1 > > _627_val) { mosek::fusion::p_RowBlockManager::_get_impl(this)->evaluate(_623_xs,_624_param_value,_625_subi,_626_subj,_627_val); }
void mosek::fusion::p_RowBlockManager::evaluate(monty::rc_ptr< ::mosek::fusion::WorkStack > _623_xs,std::shared_ptr< monty::ndarray< double,1 > > _624_param_value,std::shared_ptr< monty::ndarray< int32_t,1 > > _625_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _626_subj,std::shared_ptr< monty::ndarray< double,1 > > _627_val)
{
  _checked_ptr_1<double> _ptr__624_param_value(_624_param_value ? _624_param_value->raw() : nullptr, _624_param_value ? _624_param_value->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__625_subi(_625_subi ? _625_subi->raw() : nullptr, _625_subi ? _625_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__626_subj(_626_subj ? _626_subj->raw() : nullptr, _626_subj ? _626_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__627_val(_627_val ? _627_val->raw() : nullptr, _627_val ? _627_val->size(0) : 0);
  int32_t _628_stack = _623_xs->allocf64(code_used);
  int32_t _629_stackp = (int)0;
  std::shared_ptr< monty::ndarray< double,1 > > _630_xf64 = mosek::fusion::p_WorkStack::_get_impl(_623_xs)->f64;
  _checked_ptr_1<double> _ptr__630_xf64(_630_xf64 ? _630_xf64->raw() : nullptr,_630_xf64 ? _630_xf64->size(0) : 0);
  int32_t _631_idx = (int)0;
  int32_t _9422_ = (int)0;
  int32_t _9423_ = (int)((row_param_ptrb)->size(0));
  for (int32_t _632_i = _9422_; _632_i < _9423_; ++_632_i)
  {
    {
      if ((((*row_param_ptrb)(_632_i)) < ((*row_param_ptre)(_632_i))))
      {
        {
          int32_t _633_k = ((*row_code_ptr)(_632_i));
          int32_t _9424_ = ((*row_param_ptrb)(_632_i));
          int32_t _9425_ = ((*row_param_ptre)(_632_i));
          for (int32_t _634_j = _9424_; _634_j < _9425_; ++_634_j)
          {
            {
              int32_t _9426_ = (int)0;
              int32_t _9427_ = ((*param_code_sizes)(_634_j));
              for (int32_t _635_l = _9426_; _635_l < _9427_; ++_635_l)
              {
                {
                  if ((((*code)(_633_k)) == 0))
                  {
                    {
                    }
                  }
                  else
                  {
                    {
                      if ((((*code)(_633_k)) == 1))
                      {
                        {
                          _ptr__630_xf64[safe_add( _628_stack,_629_stackp )] = _ptr__624_param_value[(int32_t)((*cconst)(_633_k))];
                          ++ _629_stackp;
                        }
                      }
                      else
                      {
                        {
                          if ((((*code)(_633_k)) == 2))
                          {
                            {
                              _ptr__630_xf64[safe_add( _628_stack,_629_stackp )] = ((*cconst)(_633_k));
                              ++ _629_stackp;
                            }
                          }
                          else
                          {
                            {
                              if ((((*code)(_633_k)) == 3))
                              {
                                {
                                  _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)2)] += _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)];
                                  -- _629_stackp;
                                }
                              }
                              else
                              {
                                {
                                  if ((((*code)(_633_k)) == 4))
                                  {
                                    {
                                      _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)] = (-_ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)]);
                                    }
                                  }
                                  else
                                  {
                                    {
                                      if ((((*code)(_633_k)) == 5))
                                      {
                                        {
                                          _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)2)] *= _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)];
                                          -- _629_stackp;
                                        }
                                      }
                                      else
                                      {
                                        {
                                          if ((((*code)(_633_k)) == 6))
                                          {
                                            {
                                              _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)] = (1.0 / _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)]);
                                            }
                                          }
                                          else
                                          {
                                            {
                                              if ((((*code)(_633_k)) == 8))
                                              {
                                                {
                                                  double _636_v = 0.0;
                                                  int32_t _9428_ = (int)0;
                                                  int32_t _9429_ = (int32_t)((*cconst)(_633_k));
                                                  for (int32_t _637_i = _9428_; _637_i < _9429_; ++_637_i)
                                                  {
                                                    {
                                                      _636_v += _ptr__630_xf64[(safe_add( _628_stack,_629_stackp ) - (int)1)];
                                                      -- _629_stackp;
                                                    }
                                                  }
                                                  _ptr__630_xf64[safe_add( _628_stack,_629_stackp )] = _636_v;
                                                  ++ _629_stackp;
                                                }
                                              }
                                              else
                                              {
                                                {
                                                }
                                              }
                                            }
                                          }
                                        }
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                  ++ _633_k;
                }
              }
              _ptr__625_subi[_631_idx] = _632_i;
              _ptr__626_subj[_631_idx] = (int32_t)((*param_varidx)(_634_j));
              _ptr__627_val[_631_idx] = _ptr__630_xf64[_628_stack];
              _629_stackp = (int)0;
              ++ _631_idx;
            }
          }
        }
      }
      {}
    }
  }
  _623_xs->popf64(code_used);
}
// End mosek.fusion.RowBlockManager.evaluate

// Begin mosek.fusion.RowBlockManager.replace_row_code
// Method mosek.fusion.RowBlockManager.replace_row_code @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5167:6-5270:7
void mosek::fusion::RowBlockManager :: replace_row_code(monty::rc_ptr< ::mosek::fusion::WorkStack > _638_rs,std::shared_ptr< monty::ndarray< int32_t,1 > > _639_nativeidxs,int32_t _640_ptr,int32_t _641_nidxs,int32_t _642_codeptr,int32_t _643_code_p,int32_t _644_cconst_p) { mosek::fusion::p_RowBlockManager::_get_impl(this)->replace_row_code(_638_rs,_639_nativeidxs,_640_ptr,_641_nidxs,_642_codeptr,_643_code_p,_644_cconst_p); }
void mosek::fusion::p_RowBlockManager::replace_row_code(monty::rc_ptr< ::mosek::fusion::WorkStack > _638_rs,std::shared_ptr< monty::ndarray< int32_t,1 > > _639_nativeidxs,int32_t _640_ptr,int32_t _641_nidxs,int32_t _642_codeptr,int32_t _643_code_p,int32_t _644_cconst_p)
{
  _checked_ptr_1<int32_t> _ptr__639_nativeidxs(_639_nativeidxs ? _639_nativeidxs->raw() : nullptr, _639_nativeidxs ? _639_nativeidxs->size(0) : 0);
  int32_t _645_nelem = (int)((_639_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _646_ri32 = mosek::fusion::p_WorkStack::_get_impl(_638_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__646_ri32(_646_ri32 ? _646_ri32->raw() : nullptr,_646_ri32 ? _646_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _647_ri64 = mosek::fusion::p_WorkStack::_get_impl(_638_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__647_ri64(_647_ri64 ? _647_ri64->raw() : nullptr,_647_ri64 ? _647_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _648_rf64 = mosek::fusion::p_WorkStack::_get_impl(_638_rs)->f64;
  _checked_ptr_1<double> _ptr__648_rf64(_648_rf64 ? _648_rf64->raw() : nullptr,_648_rf64 ? _648_rf64->size(0) : 0);
  bool _649_ok = true;
  {
    int32_t _650_numop = (int)0;
    int32_t _651_nentry = (int)0;
    {
      int32_t _9430_ = (int)0;
      int32_t _9431_ = _645_nelem;
      for (int32_t _652_i = _9430_; _652_i < _9431_; ++_652_i)
      {
        {
          int32_t _9432_ = _ptr__646_ri32[safe_add( _640_ptr,_652_i )];
          int32_t _9433_ = _ptr__646_ri32[safe_add( safe_add( _640_ptr,_652_i ),(int)1 )];
          for (int32_t _653_j = _9432_; _653_j < _9433_; ++_653_j)
          {
            {
              if ((((_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_653_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_653_j )]) == (int)1) && ((_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_653_j )] )] == 10) || (_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_653_j )] )] == 2))))
              {
                {
                }
              }
              else
              {
                {
                  ++ _651_nentry;
                  _650_numop += (_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_653_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_653_j )]);
                }
              }
            }
          }
        }
      }
      _pubthis->clear_row_code(_639_nativeidxs);
    }
    ensure_code_cap(_651_nentry,_650_numop);
    code_used += _650_numop;
    varidx_used += _651_nentry;
  }
  if ((first_free_codeitem > safe_mul( code_used,(int)2 )))
  {
    {
      compress();
    }
  }
  {}
  {
    int32_t _9434_ = (int)0;
    int32_t _9435_ = _645_nelem;
    for (int32_t _654_i = _9434_; _654_i < _9435_; ++_654_i)
    {
      {
        int32_t _655_subi = _ptr__639_nativeidxs[_654_i];
        int32_t _656_cnnz = (int)0;
        int32_t _657_rncodeatom = (int)0;
        int32_t _9436_ = _ptr__646_ri32[safe_add( _640_ptr,_654_i )];
        int32_t _9437_ = _ptr__646_ri32[safe_add( safe_add( _640_ptr,_654_i ),(int)1 )];
        for (int32_t _658_j = _9436_; _658_j < _9437_; ++_658_j)
        {
          {
            if ((((_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_658_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_658_j )]) == (int)1) && ((_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_658_j )] )] == 10) || (_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_658_j )] )] == 2))))
            {
              {
              }
            }
            else
            {
              {
                ++ _656_cnnz;
                _657_rncodeatom += (_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_658_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_658_j )]);
              }
            }
          }
        }
        int32_t _659_ni = _ptr__639_nativeidxs[_654_i];
        int32_t _660_n = (((*row_param_ptre)(_659_ni)) - ((*row_param_ptrb)(_659_ni)));
        if ((_656_cnnz > (int)0))
        {
          {
            ((*row_param_ptrb)(_655_subi)) = first_free_entry;
            ((*row_param_ptre)(_655_subi)) = safe_add( first_free_entry,_656_cnnz );
            ((*row_code_ptr)(_655_subi)) = first_free_codeitem;
            first_free_entry += _656_cnnz;
            first_free_codeitem += _657_rncodeatom;
          }
        }
        {}
      }
    }
  }
  int32_t _9438_ = (int)0;
  int32_t _9439_ = _645_nelem;
  for (int32_t _661_i = _9438_; _661_i < _9439_; ++_661_i)
  {
    {
      int32_t _662_subi = _ptr__639_nativeidxs[_661_i];
      int32_t _663_codei = ((*row_code_ptr)(_662_subi));
      int32_t _664_elmi = ((*row_param_ptrb)(_662_subi));
      int32_t _9440_ = _ptr__646_ri32[safe_add( _640_ptr,_661_i )];
      int32_t _9441_ = _ptr__646_ri32[safe_add( safe_add( _640_ptr,_661_i ),(int)1 )];
      for (int32_t _665_j = _9440_; _665_j < _9441_; ++_665_j)
      {
        {
          if ((((_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_665_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_665_j )]) == (int)1) && ((_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_665_j )] )] == 10) || (_ptr__646_ri32[safe_add( _643_code_p,_ptr__646_ri32[safe_add( _642_codeptr,_665_j )] )] == 2))))
          {
            {
            }
          }
          else
          {
            {
              int32_t _9442_ = _ptr__646_ri32[safe_add( _642_codeptr,_665_j )];
              int32_t _9443_ = _ptr__646_ri32[safe_add( safe_add( _642_codeptr,_665_j ),(int)1 )];
              for (int32_t _666_k = _9442_; _666_k < _9443_; ++_666_k)
              {
                {
                  ((*code)(_663_codei)) = _ptr__646_ri32[safe_add( _643_code_p,_666_k )];
                  ((*cconst)(_663_codei)) = _ptr__648_rf64[safe_add( _644_cconst_p,_666_k )];
                  ++ _663_codei;
                }
              }
              ((*param_code_sizes)(_664_elmi)) = (_ptr__646_ri32[safe_add( safe_add( _642_codeptr,_665_j ),(int)1 )] - _ptr__646_ri32[safe_add( _642_codeptr,_665_j )]);
              ((*param_varidx)(_664_elmi)) = _ptr__647_ri64[safe_add( _641_nidxs,_665_j )];
              _649_ok = (_649_ok && (_ptr__647_ri64[safe_add( _641_nidxs,_665_j )] >= (int)0));
              ++ _664_elmi;
            }
          }
        }
      }
    }
  }
  if ((!_649_ok))
  {
    {
      throw ::mosek::fusion::ParameterError(std::string ("Expression contains parameterized SDP non-zero"));
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.replace_row_code

// Begin mosek.fusion.RowBlockManager.clear_row_code
// Method mosek.fusion.RowBlockManager.clear_row_code @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5140:6-5156:7
void mosek::fusion::RowBlockManager :: clear_row_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _667_nativeidxs) { mosek::fusion::p_RowBlockManager::_get_impl(this)->clear_row_code(_667_nativeidxs); }
void mosek::fusion::p_RowBlockManager::clear_row_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _667_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__667_nativeidxs(_667_nativeidxs ? _667_nativeidxs->raw() : nullptr, _667_nativeidxs ? _667_nativeidxs->size(0) : 0);
  if ((NULL != row_param_ptrb.get()))
  {
    {
      int32_t _9444_ = (int)0;
      int32_t _9445_ = (int)((_667_nativeidxs)->size(0));
      for (int32_t _668_i = _9444_; _668_i < _9445_; ++_668_i)
      {
        {
          int32_t _9446_ = ((*row_param_ptrb)(_ptr__667_nativeidxs[_668_i]));
          int32_t _9447_ = ((*row_param_ptre)(_ptr__667_nativeidxs[_668_i]));
          for (int32_t _669_j = _9446_; _669_j < _9447_; ++_669_j)
          {
            {
              code_used -= ((*param_code_sizes)(_669_j));
            }
          }
          varidx_used -= (((*row_param_ptre)(_ptr__667_nativeidxs[_668_i])) - ((*row_param_ptrb)(_ptr__667_nativeidxs[_668_i])));
          ((*row_param_ptrb)(_ptr__667_nativeidxs[_668_i])) = (int)0;
          ((*row_param_ptre)(_ptr__667_nativeidxs[_668_i])) = (int)0;
        }
      }
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.clear_row_code

// Begin mosek.fusion.RowBlockManager.compress
// Method mosek.fusion.RowBlockManager.compress @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5096:6-5138:7
void mosek::fusion::p_RowBlockManager::compress()
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _670_newcode = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(code_used)));
  _checked_ptr_1<int32_t> _ptr__670_newcode(_670_newcode ? _670_newcode->raw() : nullptr,_670_newcode ? _670_newcode->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _671_newcconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(code_used)));
  _checked_ptr_1<double> _ptr__671_newcconst(_671_newcconst ? _671_newcconst->raw() : nullptr,_671_newcconst ? _671_newcconst->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _672_newvaridx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(first_free_entry)));
  _checked_ptr_1<int64_t> _ptr__672_newvaridx(_672_newvaridx ? _672_newvaridx->raw() : nullptr,_672_newvaridx ? _672_newvaridx->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _673_newcodesizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(first_free_entry)));
  _checked_ptr_1<int32_t> _ptr__673_newcodesizes(_673_newcodesizes ? _673_newcodesizes->raw() : nullptr,_673_newcodesizes ? _673_newcodesizes->size(0) : 0);
  int32_t _674_tgtptr = (int)0;
  int32_t _675_tgtcptr = (int)0;
  int32_t _9448_ = (int)0;
  int32_t _9449_ = (int)((row_param_ptrb)->size(0));
  for (int32_t _676_i = _9448_; _676_i < _9449_; ++_676_i)
  {
    {
      int32_t _677_codesize = (int)0;
      int32_t _678_n = (((*row_param_ptre)(_676_i)) - ((*row_param_ptrb)(_676_i)));
      ::mosek::fusion::Utils::Tools::arraycopy(param_varidx,((*row_param_ptrb)(_676_i)),_672_newvaridx,_674_tgtptr,_678_n);
      ::mosek::fusion::Utils::Tools::arraycopy(param_code_sizes,((*row_param_ptrb)(_676_i)),_673_newcodesizes,_674_tgtptr,_678_n);
      int32_t _9450_ = ((*row_param_ptrb)(_676_i));
      int32_t _9451_ = ((*row_param_ptre)(_676_i));
      for (int32_t _679_j = _9450_; _679_j < _9451_; ++_679_j)
      {
        {
          _677_codesize += ((*param_code_sizes)(_679_j));
        }
      }
      ::mosek::fusion::Utils::Tools::arraycopy(code,((*row_code_ptr)(_676_i)),_670_newcode,_675_tgtcptr,_677_codesize);
      ::mosek::fusion::Utils::Tools::arraycopy(cconst,((*row_code_ptr)(_676_i)),_671_newcconst,_675_tgtcptr,_677_codesize);
      ((*row_param_ptrb)(_676_i)) = _674_tgtptr;
      ((*row_param_ptre)(_676_i)) = safe_add( _674_tgtptr,_678_n );
      ((*row_code_ptr)(_676_i)) = _675_tgtcptr;
      _675_tgtcptr += _677_codesize;
      _674_tgtptr += _678_n;
    }
  }
  ::mosek::fusion::Utils::Tools::arraycopy(_672_newvaridx,(int)0,param_varidx,(int)0,_674_tgtptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_673_newcodesizes,(int)0,param_code_sizes,(int)0,_674_tgtptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_670_newcode,(int)0,code,(int)0,_675_tgtcptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_671_newcconst,(int)0,cconst,(int)0,_675_tgtcptr);
  first_free_codeitem = _675_tgtcptr;
  first_free_entry = _674_tgtptr;
}
// End mosek.fusion.RowBlockManager.compress

// Begin mosek.fusion.RowBlockManager.ensure_code_cap
// Method mosek.fusion.RowBlockManager.ensure_code_cap @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5060:6-5094:7
void mosek::fusion::p_RowBlockManager::ensure_code_cap(int32_t _680_nentry,int32_t _681_codesize)
{
  if ((NULL == row_code_ptr.get()))
  {
    {
      int32_t _682_n = (int)128;
      while ( (_682_n < _680_nentry) )
      {
        {
          _682_n *= (int)2;
        }
      }
      int32_t _683_m = (int)128;
      while ( (_683_m < _681_codesize) )
      {
        {
          _683_m *= (int)2;
        }
      }
      row_param_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      row_param_ptre = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      row_code_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      param_varidx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_682_n)));
      param_code_sizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_682_n)));
      code = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_683_m)));
      cconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_683_m)));
    }
  }
  else
  {
    {
      if ((_680_nentry > ((int)((param_varidx)->size(0)) - first_free_entry)))
      {
        {
          int32_t _684_n = (int)((param_varidx)->size(0));
          while ( ((_684_n - first_free_entry) < _680_nentry) )
          {
            {
              _684_n *= (int)2;
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _685_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_684_n)));
          _checked_ptr_1<int32_t> _ptr__685_ptr(_685_ptr ? _685_ptr->raw() : nullptr,_685_ptr ? _685_ptr->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _686_varidx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_684_n)));
          _checked_ptr_1<int64_t> _ptr__686_varidx(_686_varidx ? _686_varidx->raw() : nullptr,_686_varidx ? _686_varidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _687_code_sizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_684_n)));
          _checked_ptr_1<int32_t> _ptr__687_code_sizes(_687_code_sizes ? _687_code_sizes->raw() : nullptr,_687_code_sizes ? _687_code_sizes->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(param_varidx,(int)0,_686_varidx,(int)0,first_free_entry);
          ::mosek::fusion::Utils::Tools::arraycopy(param_code_sizes,(int)0,_687_code_sizes,(int)0,first_free_entry);
          param_varidx = _686_varidx;
          param_code_sizes = _687_code_sizes;
        }
      }
      {}
      if ((_681_codesize > ((int)((code)->size(0)) - first_free_codeitem)))
      {
        {
          int32_t _688_n = (int)((code)->size(0));
          while ( (_681_codesize > (_688_n - first_free_codeitem)) )
          {
            {
              _688_n *= (int)2;
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _689_tcode = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_688_n)));
          _checked_ptr_1<int32_t> _ptr__689_tcode(_689_tcode ? _689_tcode->raw() : nullptr,_689_tcode ? _689_tcode->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(code,(int)0,_689_tcode,(int)0,first_free_codeitem);
          std::shared_ptr< monty::ndarray< double,1 > > _690_tcconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_688_n)));
          _checked_ptr_1<double> _ptr__690_tcconst(_690_tcconst ? _690_tcconst->raw() : nullptr,_690_tcconst ? _690_tcconst->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(cconst,(int)0,_690_tcconst,(int)0,first_free_codeitem);
          code = _689_tcode;
          cconst = _690_tcconst;
        }
      }
      {}
    }
  }
}
// End mosek.fusion.RowBlockManager.ensure_code_cap

// Begin mosek.fusion.RowBlockManager.release
// Method mosek.fusion.RowBlockManager.release @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5047:6-5058:7
void mosek::fusion::RowBlockManager :: release(int32_t _691_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _692_nativeidxs) { mosek::fusion::p_RowBlockManager::_get_impl(this)->release(_691_id,_692_nativeidxs); }
void mosek::fusion::p_RowBlockManager::release(int32_t _691_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _692_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__692_nativeidxs(_692_nativeidxs ? _692_nativeidxs->raw() : nullptr, _692_nativeidxs ? _692_nativeidxs->size(0) : 0);
  int32_t _693_num = blocks->blocksize(_691_id);
  blocks->get(_691_id,_692_nativeidxs,(int)0);
  blocks->free(_691_id);
  if ((NULL != row_code_ptr.get()))
  {
    {
      int32_t _9452_ = (int)0;
      int32_t _9453_ = _693_num;
      for (int32_t _694_i = _9452_; _694_i < _9453_; ++_694_i)
      {
        {
          int32_t _695_ni = _ptr__692_nativeidxs[_694_i];
          code_used -= (((*row_param_ptre)(_695_ni)) - ((*row_param_ptrb)(_695_ni)));
          ((*row_param_ptre)(_695_ni)) = (int)0;
          ((*row_param_ptrb)(_695_ni)) = (int)0;
        }
      }
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.release

// Begin mosek.fusion.RowBlockManager.allocate
// Method mosek.fusion.RowBlockManager.allocate @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5029:6-5045:7
int32_t mosek::fusion::RowBlockManager :: allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _696_nativeidxs) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->allocate(_696_nativeidxs); }
int32_t mosek::fusion::p_RowBlockManager::allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _696_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__696_nativeidxs(_696_nativeidxs ? _696_nativeidxs->raw() : nullptr, _696_nativeidxs ? _696_nativeidxs->size(0) : 0);
  int32_t _697_num = (int)((_696_nativeidxs)->size(0));
  int32_t _698_id = blocks->alloc(_697_num);
  blocks->get(_698_id,_696_nativeidxs,(int)0);
  if (((NULL != row_code_ptr.get()) && (blocks->capacity() > (int)((row_param_ptrb)->size(0)))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _699_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__699_ptrb(_699_ptrb ? _699_ptrb->raw() : nullptr,_699_ptrb ? _699_ptrb->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_param_ptrb,(int)0,_699_ptrb,(int)0,(int)((row_param_ptrb)->size(0)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _700_ptre = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__700_ptre(_700_ptre ? _700_ptre->raw() : nullptr,_700_ptre ? _700_ptre->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_param_ptre,(int)0,_700_ptre,(int)0,(int)((row_param_ptre)->size(0)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _701_cptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__701_cptr(_701_cptr ? _701_cptr->raw() : nullptr,_701_cptr ? _701_cptr->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_code_ptr,(int)0,_701_cptr,(int)0,(int)((row_code_ptr)->size(0)));
      row_param_ptrb = _699_ptrb;
      row_param_ptre = _700_ptre;
      row_code_ptr = _701_cptr;
    }
  }
  {}
  return _698_id;
}
// End mosek.fusion.RowBlockManager.allocate

// Begin mosek.fusion.RowBlockManager.row_is_parameterized
// Method mosek.fusion.RowBlockManager.row_is_parameterized @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:5025:6-5027:7
bool mosek::fusion::RowBlockManager :: row_is_parameterized(int32_t _702_i) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->row_is_parameterized(_702_i); }
bool mosek::fusion::p_RowBlockManager::row_is_parameterized(int32_t _702_i)
{
  return ((NULL != row_param_ptrb.get()) && (((*row_param_ptrb)(_702_i)) < ((*row_param_ptre)(_702_i))));
}
// End mosek.fusion.RowBlockManager.row_is_parameterized

void mosek::fusion::p_RowBlockManager::destroy()
{
  cconst.reset();
  code.reset();
  param_code_sizes.reset();
  param_varidx.reset();
  row_code_ptr.reset();
  row_param_ptre.reset();
  row_param_ptrb.reset();
  blocks.reset();
}
void mosek::fusion::RowBlockManager::destroy() { mosek::fusion::p_RowBlockManager::_get_impl(this)->destroy(); }
// } class RowBlockManager
// class Model { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:365:4-4941:5
mosek::fusion::Model::Model(mosek::fusion::p_Model *_impl) : mosek::fusion::BaseModel(_impl) { /*std::cout << "Model()" << this << std::endl; */ }
mosek::fusion::Model::~Model() {  /* std::cout << "~Model()" << this << std::endl << std::flush; */ }
mosek::fusion::p_Model::p_Model
  (::mosek::fusion::Model * _pubthis) :     mosek::fusion::p_BaseModel(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.Model.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:593:6-661:7
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (monty::rc_ptr< ::mosek::fusion::Model > _703_m)
{ return new mosek::fusion::Model(_703_m); }
mosek::fusion::Model::Model
  (monty::rc_ptr< ::mosek::fusion::Model > _703_m) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_703_m); }
void mosek::fusion::p_Model::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _703_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseModel::_initialize(_703_m);
    monty::rc_ptr< ::mosek::fusion::Model > _704_self = ::mosek::fusion::Model::t(_pubthis);
    model_name = mosek::fusion::p_Model::_get_impl(_703_m)->model_name;
    acceptable_sol = mosek::fusion::p_Model::_get_impl(_703_m)->acceptable_sol;
    solutionptr = mosek::fusion::p_Model::_get_impl(_703_m)->solutionptr;
    cache_evaluations = mosek::fusion::p_Model::_get_impl(_703_m)->cache_evaluations;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9454_;
    bool _9455_ = (NULL != mosek::fusion::p_Model::_get_impl(_703_m)->sol_itr.get());
    if (_9455_)
    {
      _9454_ = mosek::fusion::p_Model::_get_impl(_703_m)->sol_itr->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9454_ = nullptr;
    }
    sol_itr = _9454_;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9456_;
    bool _9457_ = (NULL != mosek::fusion::p_Model::_get_impl(_703_m)->sol_itg.get());
    if (_9457_)
    {
      _9456_ = mosek::fusion::p_Model::_get_impl(_703_m)->sol_itg->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9456_ = nullptr;
    }
    sol_itg = _9456_;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9458_;
    bool _9459_ = (NULL != mosek::fusion::p_Model::_get_impl(_703_m)->sol_bas.get());
    if (_9459_)
    {
      _9458_ = mosek::fusion::p_Model::_get_impl(_703_m)->sol_bas->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9458_ = nullptr;
    }
    sol_bas = _9458_;
    num_task_afe = mosek::fusion::p_Model::_get_impl(_703_m)->num_task_afe;
    num_task_con = mosek::fusion::p_Model::_get_impl(_703_m)->num_task_con;
    num_task_acc = mosek::fusion::p_Model::_get_impl(_703_m)->num_task_acc;
    con_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_703_m)->con_blocks);
    afe_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_703_m)->afe_blocks);
    obj_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_703_m)->obj_blocks);
    acc_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_703_m)->acc_block_map);
    acon_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_703_m)->acon_block_map);
    acon_acc = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_acc);
    acon_afe = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_afe);
    acon_elm_accid = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_elm_accid);
    acon_elm_scale = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_elm_scale);
    acon_elm_ofs = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_elm_ofs);
    acon_elm_afe = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->acon_elm_afe);
    task_numaferow = mosek::fusion::p_Model::_get_impl(_703_m)->task_numaferow;
    var_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_703_m)->var_block_map);
    var_block_acc_id = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->var_block_acc_id);
    var_elm_acc_idx = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->var_elm_acc_idx);
    var_elm_acc_ofs = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->var_elm_acc_ofs);
    barvar_num = (int)0;
    barvar_dim = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_dim);
    barvar_block_ptr = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_ptr);
    barvar_block_dim = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_dim);
    barvar_block_elm_ptr = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_elm_ptr);
    barvar_block_elm_barj = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_elm_barj);
    barvar_block_elm_i = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_elm_i);
    barvar_block_elm_j = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->barvar_block_elm_j);
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9460_;
    for (int32_t _705_i = (int)0, _9461_ = (int)((mosek::fusion::p_Model::_get_impl(_703_m)->vars)->size(0)); _705_i < _9461_ ; ++_705_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelVariable > _9462_;
      bool _9463_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_703_m)->vars)(_705_i)).get());
      if (_9463_)
      {
        _9462_ = mosek::fusion::p_ModelVariable::_get_impl(((*mosek::fusion::p_Model::_get_impl(_703_m)->vars)(_705_i)).get())->__mosek_2fusion_2ModelVariable__clone(_704_self);
      }
      else
      {
        _9462_ = nullptr;
      }
      _9460_.push_back(_9462_);
    }
    auto _9464_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9460_.size()),_9460_.begin(),_9460_.end()));
    vars = _9464_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9465_;
    for (int32_t _706_i = (int)0, _9466_ = (int)((mosek::fusion::p_Model::_get_impl(_703_m)->barvars)->size(0)); _706_i < _9466_ ; ++_706_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelVariable > _9467_;
      bool _9468_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_703_m)->barvars)(_706_i)).get());
      if (_9468_)
      {
        _9467_ = mosek::fusion::p_ModelVariable::_get_impl(((*mosek::fusion::p_Model::_get_impl(_703_m)->barvars)(_706_i)).get())->__mosek_2fusion_2ModelVariable__clone(_704_self);
      }
      else
      {
        _9467_ = nullptr;
      }
      _9465_.push_back(_9467_);
    }
    auto _9469_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9465_.size()),_9465_.begin(),_9465_.end()));
    barvars = _9469_;
    var_map = mosek::fusion::p_Model::_get_impl(_703_m)->var_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9470_;
    for (int32_t _707_i = (int)0, _9471_ = (int)((mosek::fusion::p_Model::_get_impl(_703_m)->cons)->size(0)); _707_i < _9471_ ; ++_707_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelConstraint > _9472_;
      bool _9473_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_703_m)->cons)(_707_i)).get());
      if (_9473_)
      {
        _9472_ = mosek::fusion::p_ModelConstraint::_get_impl(((*mosek::fusion::p_Model::_get_impl(_703_m)->cons)(_707_i)).get())->__mosek_2fusion_2ModelConstraint__clone(_704_self);
      }
      else
      {
        _9472_ = nullptr;
      }
      _9470_.push_back(_9472_);
    }
    auto _9474_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9470_.size()),_9470_.begin(),_9470_.end()));
    cons = _9474_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9475_;
    for (int32_t _708_i = (int)0, _9476_ = (int)((mosek::fusion::p_Model::_get_impl(_703_m)->acons)->size(0)); _708_i < _9476_ ; ++_708_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelConstraint > _9477_;
      bool _9478_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_703_m)->acons)(_708_i)).get());
      if (_9478_)
      {
        _9477_ = mosek::fusion::p_ModelConstraint::_get_impl(((*mosek::fusion::p_Model::_get_impl(_703_m)->acons)(_708_i)).get())->__mosek_2fusion_2ModelConstraint__clone(_704_self);
      }
      else
      {
        _9477_ = nullptr;
      }
      _9475_.push_back(_9477_);
    }
    auto _9479_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9475_.size()),_9475_.begin(),_9475_.end()));
    acons = _9479_;
    con_map = mosek::fusion::p_Model::_get_impl(_703_m)->con_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    numparameter = mosek::fusion::p_Model::_get_impl(_703_m)->numparameter;
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > > _9480_;
    bool _9481_ = (NULL == mosek::fusion::p_Model::_get_impl(_703_m)->parameters.get());
    if (_9481_)
    {
      _9480_ = nullptr;
    }
    else
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::Parameter >> _9482_;
      for (int32_t _709_i = (int)0, _9483_ = (int)((mosek::fusion::p_Model::_get_impl(_703_m)->parameters)->size(0)); _709_i < _9483_ ; ++_709_i)
      {
        monty::rc_ptr< ::mosek::fusion::Parameter > _9484_;
        bool _9485_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_703_m)->parameters)(_709_i)).get());
        if (_9485_)
        {
          _9484_ = ((*mosek::fusion::p_Model::_get_impl(_703_m)->parameters)(_709_i))->__mosek_2fusion_2Parameter__clone(_704_self);
        }
        else
        {
          _9484_ = nullptr;
        }
        _9482_.push_back(_9484_);
      }
      auto _9486_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 >(monty::shape(_9482_.size()),_9482_.begin(),_9482_.end()));
      _9480_ = _9486_;
    }
    parameters = _9480_;
    par_map = mosek::fusion::p_Model::_get_impl(_703_m)->par_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    param_num = mosek::fusion::p_Model::_get_impl(_703_m)->param_num;
    std::shared_ptr< monty::ndarray< double,1 > > _9487_;
    bool _9488_ = (NULL != mosek::fusion::p_Model::_get_impl(_703_m)->param_value.get());
    if (_9488_)
    {
      _9487_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_703_m)->param_value);
    }
    else
    {
      _9487_ = nullptr;
    }
    param_value = _9487_;
    initsol_xx = nullptr;
    initsol_xx_flag = nullptr;
    bfixidx = mosek::fusion::p_Model::_get_impl(_703_m)->bfixidx;
    rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    xs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:514:6-590:7
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (const std::string &  _710_name,
    int32_t _711_basesize,
    bool _712_cache_eval)
{ return new mosek::fusion::Model(_710_name,_711_basesize,_712_cache_eval); }
mosek::fusion::Model::Model
  (const std::string &  _710_name,
    int32_t _711_basesize,
    bool _712_cache_eval) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_710_name,_711_basesize,_712_cache_eval); }
void mosek::fusion::p_Model::_initialize  (const std::string &  _710_name,
    int32_t _711_basesize,
    bool _712_cache_eval)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseModel::_initialize(_710_name,std::string (""));
    model_name = _710_name;
    acceptable_sol = ::mosek::fusion::AccSolutionStatus::Optimal;
    solutionptr = ::mosek::fusion::SolutionType::Default;
    cache_evaluations = _712_cache_eval;
    sol_itr = nullptr;
    sol_itg = nullptr;
    sol_bas = nullptr;
    num_task_afe = (int)0;
    num_task_con = (int)0;
    num_task_acc = (int)0;
    con_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    afe_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    obj_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    acc_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    acon_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    task_numaferow = (int)0;
    acon_acc = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->block_capacity())));
    acon_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->block_capacity())));
    acon_elm_accid = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    std::vector<double> _9489_;
    for (int32_t _713_i = (int)0, _9490_ = acon_block_map->capacity(); _713_i < _9490_ ; ++_713_i)
    {
      _9489_.push_back(1.0);
    }
    auto _9491_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9489_.size()),_9489_.begin(),_9489_.end()));
    acon_elm_scale = _9491_;
    acon_elm_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    acon_elm_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    var_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    var_block_acc_id = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->block_capacity())));
    var_elm_acc_idx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->capacity())));
    var_elm_acc_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->capacity())));
    barvar_num = (int)0;
    barvar_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_711_basesize)));
    barvar_block_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _711_basesize,(int)1 ))));
    barvar_block_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_711_basesize)));
    barvar_block_elm_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _711_basesize,(int)1 ))));
    barvar_block_elm_barj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_711_basesize)));
    barvar_block_elm_i = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_711_basesize)));
    barvar_block_elm_j = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_711_basesize)));
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9492_;
    for (int32_t _714_i = (int)0, _9493_ = safe_mul( _711_basesize,(int)8 ); _714_i < _9493_ ; ++_714_i)
    {
      _9492_.push_back(nullptr);
    }
    auto _9494_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9492_.size()),_9492_.begin(),_9492_.end()));
    vars = _9494_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9495_;
    for (int32_t _715_i = (int)0, _9496_ = _711_basesize; _715_i < _9496_ ; ++_715_i)
    {
      _9495_.push_back(nullptr);
    }
    auto _9497_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9495_.size()),_9495_.begin(),_9495_.end()));
    barvars = _9497_;
    var_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    initsol_xx = nullptr;
    initsol_xx_flag = nullptr;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9498_;
    for (int32_t _716_i = (int)0, _9499_ = safe_mul( _711_basesize,(int)8 ); _716_i < _9499_ ; ++_716_i)
    {
      _9498_.push_back(nullptr);
    }
    auto _9500_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9498_.size()),_9498_.begin(),_9498_.end()));
    cons = _9500_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9501_;
    for (int32_t _717_i = (int)0, _9502_ = safe_mul( _711_basesize,(int)8 ); _717_i < _9502_ ; ++_717_i)
    {
      _9501_.push_back(nullptr);
    }
    auto _9503_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9501_.size()),_9501_.begin(),_9501_.end()));
    acons = _9503_;
    con_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    numparameter = (int)0;
    parameters = nullptr;
    par_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    param_num = (int)0;
    param_value = nullptr;
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _718_natvarbfixidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)1)));
      _checked_ptr_1<int32_t> _ptr__718_natvarbfixidx(_718_natvarbfixidx ? _718_natvarbfixidx->raw() : nullptr,_718_natvarbfixidx ? _718_natvarbfixidx->size(0) : 0);
      int32_t _719_bfixvarid = linearvar_alloc((int)1,_718_natvarbfixidx);
      bfixidx = _ptr__718_natvarbfixidx[(int)0];
    }
    task_var_putboundlist_fx(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(bfixidx)}),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >{(double)(1.0)}));
    task_var_name(bfixidx,std::string ("1.0"));
    task_append_domain_empty();
    rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    xs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:506:6-88
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (int32_t _720_basesize,
    bool _721_cache_eval)
{ return new mosek::fusion::Model(_720_basesize,_721_cache_eval); }
mosek::fusion::Model::Model
  (int32_t _720_basesize,
    bool _721_cache_eval) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_720_basesize,_721_cache_eval); }
void mosek::fusion::p_Model::_initialize  (int32_t _720_basesize,
    bool _721_cache_eval)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(std::string (""),_720_basesize,_721_cache_eval);
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:499:6-60
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (const std::string &  _722_name)
{ return new mosek::fusion::Model(_722_name); }
mosek::fusion::Model::Model
  (const std::string &  _722_name) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_722_name); }
void mosek::fusion::p_Model::_initialize  (const std::string &  _722_name)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(_722_name,(int)128,false);
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:493:6-45
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  ()
{ return new mosek::fusion::Model(); }
mosek::fusion::Model::Model
  () : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(); }
void mosek::fusion::p_Model::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(std::string (""),(int)128,false);
  }
}
// mosek.fusion.Model.ctor

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4822:6-4940:7
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _723_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _724_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_723_name,_724_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _723_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _724_terms) { return __mosek_2fusion_2Model__disjunction(_723_name,_724_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _723_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _724_terms)
{
  int32_t _725_n = (int)((_724_terms)->size(0));
  int32_t _726_nrows = (int)0;
  int32_t _727_nterms = (int)0;
  int32_t _728_naccterms = (int)0;
  int32_t _729_naccrows = (int)0;
  int32_t _9504_ = (int)0;
  int32_t _9505_ = _725_n;
  for (int32_t _730_i = _9504_; _730_i < _9505_; ++_730_i)
  {
    {
      _726_nrows += ((*_724_terms)(_730_i))->size();
      _727_nterms += mosek::fusion::p_Term::_get_impl(((*_724_terms)(_730_i)).get())->num();
      _729_naccrows += mosek::fusion::p_Term::_get_impl(((*_724_terms)(_730_i)).get())->numaccrows();
      _728_naccterms += mosek::fusion::p_Term::_get_impl(((*_724_terms)(_730_i)).get())->numaccterms();
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _731_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_726_nrows)));
  _checked_ptr_1<int32_t> _ptr__731_afeidxs(_731_afeidxs ? _731_afeidxs->raw() : nullptr,_731_afeidxs ? _731_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _732_accafeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_729_naccrows)));
  _checked_ptr_1<int32_t> _ptr__732_accafeidxs(_732_accafeidxs ? _732_accafeidxs->raw() : nullptr,_732_accafeidxs ? _732_accafeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _733_accb = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_729_naccrows)));
  _checked_ptr_1<double> _ptr__733_accb(_733_accb ? _733_accb->raw() : nullptr,_733_accb ? _733_accb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _734_domidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_728_naccterms)));
  _checked_ptr_1<int64_t> _ptr__734_domidxs(_734_domidxs ? _734_domidxs->raw() : nullptr,_734_domidxs ? _734_domidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _735_termsize = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_725_n)));
  _checked_ptr_1<int64_t> _ptr__735_termsize(_735_termsize ? _735_termsize->raw() : nullptr,_735_termsize ? _735_termsize->size(0) : 0);
  int32_t _736_afeblockid = afe_allocate(_731_afeidxs);
  int32_t _737_term_index = (int)0;
  int32_t _738_acctermptr = (int)0;
  int32_t _739_afetgtptr = (int)0;
  int32_t _740_afeptr = (int)0;
  int32_t _9506_ = (int)0;
  int32_t _9507_ = _725_n;
  for (int32_t _741_term_i = _9506_; _741_term_i < _9507_; ++_741_term_i)
  {
    {
      int32_t _9508_ = (int)0;
      int32_t _9509_ = mosek::fusion::p_Term::_get_impl(((*_724_terms)(_741_term_i)).get())->num();
      for (int32_t _742_simpterm_i = _9508_; _742_simpterm_i < _9509_; ++_742_simpterm_i)
      {
        {
          monty::rc_ptr< ::mosek::fusion::DJCDomain > _743_dom = ((*mosek::fusion::p_Term::_get_impl(((*_724_terms)(_741_term_i)))->domains)(_742_simpterm_i));
          monty::rc_ptr< ::mosek::fusion::ExprDense > _744_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(((*mosek::fusion::p_Term::_get_impl(((*_724_terms)(_741_term_i)))->exprs)(_742_simpterm_i)))).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
          _744_e->recursive_eval(rs,ws,xs,cache_evaluations);
          rs->pop_expr();
          std::shared_ptr< monty::ndarray< int32_t,1 > > _745_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
          _checked_ptr_1<int32_t> _ptr__745_ri32(_745_ri32 ? _745_ri32->raw() : nullptr,_745_ri32 ? _745_ri32->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _746_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
          _checked_ptr_1<int64_t> _ptr__746_ri64(_746_ri64 ? _746_ri64->raw() : nullptr,_746_ri64 ? _746_ri64->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _747_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
          _checked_ptr_1<double> _ptr__747_rf64(_747_rf64 ? _747_rf64->raw() : nullptr,_747_rf64 ? _747_rf64->size(0) : 0);
          int32_t _748_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
          int32_t _749_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
          int32_t _750_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
          int32_t _751_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
          int32_t _752_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
          bool _753_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
          int32_t _754_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
          int32_t _755_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
          int32_t _756_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
          int32_t _757_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
          int32_t _758_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
          int32_t _759_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
          std::vector<int32_t> _9510_;
          for (int32_t _761_i = (int)0, _9511_ = _748_nd; _761_i < _9511_ ; ++_761_i)
          {
            _9510_.push_back(_ptr__745_ri32[safe_add( _749_shape_base,_761_i )]);
          }
          auto _9512_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9510_.size()),_9510_.begin(),_9510_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _760_shape = _9512_;
          _checked_ptr_1<int32_t> _ptr__760_shape(_760_shape ? _760_shape->raw() : nullptr,_760_shape ? _760_shape->size(0) : 0);
          putfrows(_731_afeidxs,_740_afeptr,rs,_750_nelem,_751_nnz,_754_ptr_base,_755_nidxs_base,_756_cof_base);
          if ((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::InRange))
          {
            {
              std::shared_ptr< monty::ndarray< double,1 > > _762_lb = mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->b;
              _checked_ptr_1<double> _ptr__762_lb(_762_lb ? _762_lb->raw() : nullptr,_762_lb ? _762_lb->size(0) : 0);
              std::shared_ptr< monty::ndarray< double,1 > > _763_ub = mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->par;
              _checked_ptr_1<double> _ptr__763_ub(_763_ub ? _763_ub->raw() : nullptr,_763_ub ? _763_ub->size(0) : 0);
              int32_t _9513_ = (int)0;
              int32_t _9514_ = _750_nelem;
              for (int32_t _764_i = _9513_; _764_i < _9514_; ++_764_i)
              {
                {
                  _ptr__732_accafeidxs[safe_add( _739_afetgtptr,_764_i )] = _ptr__731_afeidxs[safe_add( _740_afeptr,_764_i )];
                  _ptr__732_accafeidxs[safe_add( safe_add( _739_afetgtptr,_750_nelem ),_764_i )] = _ptr__731_afeidxs[safe_add( _740_afeptr,_764_i )];
                  _ptr__733_accb[safe_add( _739_afetgtptr,_764_i )] = _ptr__762_lb[_764_i];
                  _ptr__733_accb[safe_add( safe_add( _739_afetgtptr,_750_nelem ),_764_i )] = _ptr__763_ub[_764_i];
                }
              }
              _ptr__734_domidxs[_738_acctermptr] = task_append_domain_rpos(_750_nelem);
              _ptr__734_domidxs[safe_add( _738_acctermptr,(int)1 )] = task_append_domain_rneg(_750_nelem);
              _ptr__735_termsize[_741_term_i] += (int)2;
              _738_acctermptr += (int)2;
              _739_afetgtptr += safe_mul( _750_nelem,(int)2 );
            }
          }
          else if(((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::EqualTo) || ((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::LessThan) || ((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::GreaterThan) || (mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::IsFree)))))
          {
            {
              std::shared_ptr< monty::ndarray< double,1 > > _765_b = mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->b;
              _checked_ptr_1<double> _ptr__765_b(_765_b ? _765_b->raw() : nullptr,_765_b ? _765_b->size(0) : 0);
              int32_t _9515_ = (int)0;
              int32_t _9516_ = _750_nelem;
              for (int32_t _766_i = _9515_; _766_i < _9516_; ++_766_i)
              {
                {
                  _ptr__732_accafeidxs[safe_add( _739_afetgtptr,_766_i )] = _ptr__731_afeidxs[safe_add( _740_afeptr,_766_i )];
                  _ptr__733_accb[safe_add( _739_afetgtptr,_766_i )] = _ptr__765_b[_766_i];
                }
              }
              if ((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::EqualTo))
              {
                {
                  _ptr__734_domidxs[_738_acctermptr] = task_append_domain_rzero(_750_nelem);
                }
              }
              else if((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::LessThan))
              {
                {
                  _ptr__734_domidxs[_738_acctermptr] = task_append_domain_rneg(_750_nelem);
                }
              }
              else if((mosek::fusion::p_DJCDomain::_get_impl(_743_dom)->dom == ::mosek::fusion::DJCDomainType::GreaterThan))
              {
                {
                  _ptr__734_domidxs[_738_acctermptr] = task_append_domain_rpos(_750_nelem);
                }
              }
              else
              {
                {
                  _ptr__734_domidxs[_738_acctermptr] = task_append_domain_r(_750_nelem);
                }
              }
              _ptr__735_termsize[_741_term_i] += (int)1;
              _738_acctermptr += (int)1;
              _739_afetgtptr += _750_nelem;
            }
          }
          else
          {
            {
              throw ::mosek::fusion::UnexpectedError(std::string ("Not implemented: Non-linear conic DJC constraints"));
            }
          }
          if ((_752_ncodeatom > (int)0))
          {
            {
              std::vector<int32_t> _9517_;
              for (int32_t _767_i = (int)0, _9518_ = _750_nelem; _767_i < _9518_ ; ++_767_i)
              {
                _9517_.push_back(_ptr__731_afeidxs[safe_add( _740_afeptr,_767_i )]);
              }
              auto _9519_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9517_.size()),_9517_.begin(),_9517_.end()));
              afe_blocks->replace_row_code(rs,_9519_,_754_ptr_base,_755_nidxs_base,_758_codeptr,_757_code,_759_cconst);
            }
          }
          {}
          _740_afeptr += _750_nelem;
          ++ _737_term_index;
        }
      }
    }
  }
  int64_t _768_djcidx = task_append_djc((int64_t)((int)1));
  std::vector<int64_t> _9520_;
  for (int32_t _769_i = (int)0, _9521_ = (int)((_732_accafeidxs)->size(0)); _769_i < _9521_ ; ++_769_i)
  {
    _9520_.push_back(_ptr__732_accafeidxs[_769_i]);
  }
  auto _9522_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9520_.size()),_9520_.begin(),_9520_.end()));
  task_putdjcslice(_768_djcidx,safe_add( _768_djcidx,(int)1 ),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >{(int64_t)((int64_t)_725_n)}),_735_termsize,_734_domidxs,_9522_,_733_accb);
  if (((int)((_723_name).size()) > (int)0))
  {
    {
      task_djc_name(_768_djcidx,::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(_723_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["))->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"))->toString());
    }
  }
  {}
  return ::mosek::fusion::p_Disjunction::_new_Disjunction(_768_djcidx);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4821:6-92
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _770_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_770_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _770_terms) { return __mosek_2fusion_2Model__disjunction(_770_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _770_terms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),_770_terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4819:6-111
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _771_djcterms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_771_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _771_djcterms) { return __mosek_2fusion_2Model__disjunction(_771_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _771_djcterms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(mosek::fusion::p_DisjunctionTerms::_get_impl(_771_djcterms)->terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4817:6-129
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _772_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _773_djcterms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_772_name,_773_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _772_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _773_djcterms) { return __mosek_2fusion_2Model__disjunction(_772_name,_773_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _772_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _773_djcterms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_772_name,mosek::fusion::p_DisjunctionTerms::_get_impl(_773_djcterms)->terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4815:6-157
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _774_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _775_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_774_name,_775_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _774_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _775_terms) { return __mosek_2fusion_2Model__disjunction(_774_name,_775_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _774_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _775_terms)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9523_;
  for (int32_t _776_i = (int)0, _9524_ = (int)((_775_terms)->size(0)); _776_i < _9524_ ; ++_776_i)
  {
    _9523_.push_back(((*_775_terms)(_776_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9525_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9523_.size()),_9523_.begin(),_9523_.end()));
  return _pubthis->__mosek_2fusion_2Model__disjunction(_774_name,_9525_);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4813:6-142
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _777_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_777_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _777_terms) { return __mosek_2fusion_2Model__disjunction(_777_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _777_terms)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9526_;
  for (int32_t _778_i = (int)0, _9527_ = (int)((_777_terms)->size(0)); _778_i < _9527_ ; ++_778_i)
  {
    _9526_.push_back(((*_777_terms)(_778_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9528_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9526_.size()),_9526_.begin(),_9526_.end()));
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),_9528_);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4811:6-128
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _779_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _780_term) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_779_name,_780_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _779_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _780_term) { return __mosek_2fusion_2Model__disjunction(_779_name,_780_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _779_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _780_term)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_779_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_780_term->__mosek_2fusion_2ExprDomain__toDJCTerm().get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4809:6-113
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _781_term) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_781_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _781_term) { return __mosek_2fusion_2Model__disjunction(_781_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _781_term)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_781_term->__mosek_2fusion_2ExprDomain__toDJCTerm().get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4806:6-115
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _782_t1,monty::rc_ptr< ::mosek::fusion::Term > _783_t2,monty::rc_ptr< ::mosek::fusion::Term > _784_t3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_782_t1,_783_t2,_784_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _782_t1,monty::rc_ptr< ::mosek::fusion::Term > _783_t2,monty::rc_ptr< ::mosek::fusion::Term > _784_t3) { return __mosek_2fusion_2Model__disjunction(_782_t1,_783_t2,_784_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _782_t1,monty::rc_ptr< ::mosek::fusion::Term > _783_t2,monty::rc_ptr< ::mosek::fusion::Term > _784_t3)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_782_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_783_t2.get()),::monty::rc_ptr<::mosek::fusion::Term>(_784_t3.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4805:6-103
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_785_t1,_786_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2) { return __mosek_2fusion_2Model__disjunction(_785_t1,_786_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_785_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_786_t2.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4804:6-91
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _787_t1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_787_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _787_t1) { return __mosek_2fusion_2Model__disjunction(_787_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _787_t1)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_787_t1.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4802:6-129
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _788_name,monty::rc_ptr< ::mosek::fusion::Term > _789_t1,monty::rc_ptr< ::mosek::fusion::Term > _790_t2,monty::rc_ptr< ::mosek::fusion::Term > _791_t3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_788_name,_789_t1,_790_t2,_791_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _788_name,monty::rc_ptr< ::mosek::fusion::Term > _789_t1,monty::rc_ptr< ::mosek::fusion::Term > _790_t2,monty::rc_ptr< ::mosek::fusion::Term > _791_t3) { return __mosek_2fusion_2Model__disjunction(_788_name,_789_t1,_790_t2,_791_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _788_name,monty::rc_ptr< ::mosek::fusion::Term > _789_t1,monty::rc_ptr< ::mosek::fusion::Term > _790_t2,monty::rc_ptr< ::mosek::fusion::Term > _791_t3)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_788_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_789_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_790_t2.get()),::monty::rc_ptr<::mosek::fusion::Term>(_791_t3.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4801:6-117
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _792_name,monty::rc_ptr< ::mosek::fusion::Term > _793_t1,monty::rc_ptr< ::mosek::fusion::Term > _794_t2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_792_name,_793_t1,_794_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _792_name,monty::rc_ptr< ::mosek::fusion::Term > _793_t1,monty::rc_ptr< ::mosek::fusion::Term > _794_t2) { return __mosek_2fusion_2Model__disjunction(_792_name,_793_t1,_794_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _792_name,monty::rc_ptr< ::mosek::fusion::Term > _793_t1,monty::rc_ptr< ::mosek::fusion::Term > _794_t2)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_792_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_793_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_794_t2.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4800:6-105
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _795_name,monty::rc_ptr< ::mosek::fusion::Term > _796_t1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_795_name,_796_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _795_name,monty::rc_ptr< ::mosek::fusion::Term > _796_t1) { return __mosek_2fusion_2Model__disjunction(_795_name,_796_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _795_name,monty::rc_ptr< ::mosek::fusion::Term > _796_t1)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_795_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_796_t1.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.formstConstr
// Method mosek.fusion.Model.formstConstr @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4795:6-4798:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_Model::__mosek_2fusion_2Model__formstConstr(monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _797_sb,std::shared_ptr< monty::ndarray< int32_t,1 > > _798_shape,std::shared_ptr< monty::ndarray< int32_t,1 > > _799_idxs)
{
  _checked_ptr_1<int32_t> _ptr__798_shape(_798_shape ? _798_shape->raw() : nullptr, _798_shape ? _798_shape->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__799_idxs(_799_idxs ? _799_idxs->raw() : nullptr, _799_idxs ? _799_idxs->size(0) : 0);
  return _797_sb;
}
// End mosek.fusion.Model.formstConstr

// Begin mosek.fusion.Model.acon_release
// Method mosek.fusion.Model.acon_release @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4685:6-4704:7
void mosek::fusion::p_Model::acon_release(int32_t _800_id)
{
  int32_t _801_num = acon_block_map->blocksize(_800_id);
  if ((_801_num >= (int)0))
  {
    {
      int32_t _802_numcone = acc_block_map->blocksize(((*acon_acc)(_800_id)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _803_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_802_numcone)));
      _checked_ptr_1<int32_t> _ptr__803_accidxs(_803_accidxs ? _803_accidxs->raw() : nullptr,_803_accidxs ? _803_accidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _804_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_801_num)));
      _checked_ptr_1<int32_t> _ptr__804_afeidxs(_804_afeidxs ? _804_afeidxs->raw() : nullptr,_804_afeidxs ? _804_afeidxs->size(0) : 0);
      acc_block_map->get(((*acon_acc)(_800_id)),_803_accidxs,(int)0);
      acc_block_map->free(((*acon_acc)(_800_id)));
      afe_blocks->get(((*acon_afe)(_800_id)),_804_afeidxs,(int)0);
      afe_release(((*acon_afe)(_800_id)));
      acon_block_map->free(_800_id);
      std::vector<int64_t> _9529_;
      for (int32_t _805_i = (int)0, _9530_ = _801_num; _805_i < _9530_ ; ++_805_i)
      {
        _9529_.push_back(_ptr__804_afeidxs[_805_i]);
      }
      auto _9531_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9529_.size()),_9529_.begin(),_9529_.end()));
      task_clearafelist(_9531_);
      std::vector<int64_t> _9532_;
      for (int32_t _806_i = (int)0, _9533_ = _802_numcone; _806_i < _9533_ ; ++_806_i)
      {
        _9532_.push_back(_ptr__803_accidxs[_806_i]);
      }
      auto _9534_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9532_.size()),_9532_.begin(),_9532_.end()));
      std::vector<int64_t> _9535_;
      for (int32_t _807_i = (int)0, _9536_ = _802_numcone; _807_i < _9536_ ; ++_807_i)
      {
        _9535_.push_back((int)0);
      }
      auto _9537_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9535_.size()),_9535_.begin(),_9535_.end()));
      task_putacclist(_9534_,_9537_,std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)0))));
    }
  }
  {}
}
// End mosek.fusion.Model.acon_release

// Begin mosek.fusion.Model.acon_allocate
// Method mosek.fusion.Model.acon_allocate @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4604:6-4683:7
int32_t mosek::fusion::p_Model::acon_allocate(std::shared_ptr< monty::ndarray< int64_t,1 > > _808_domidxs,int32_t _809_conesize,int32_t _810_numcone,std::shared_ptr< monty::ndarray< double,1 > > _811_g,std::shared_ptr< monty::ndarray< int32_t,1 > > _812_afeidxs,std::shared_ptr< monty::ndarray< int32_t,1 > > _813_accidxs)
{
  _checked_ptr_1<int64_t> _ptr__808_domidxs(_808_domidxs ? _808_domidxs->raw() : nullptr, _808_domidxs ? _808_domidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__811_g(_811_g ? _811_g->raw() : nullptr, _811_g ? _811_g->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__812_afeidxs(_812_afeidxs ? _812_afeidxs->raw() : nullptr, _812_afeidxs ? _812_afeidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__813_accidxs(_813_accidxs ? _813_accidxs->raw() : nullptr, _813_accidxs ? _813_accidxs->size(0) : 0);
  int32_t _814_n = safe_mul( _810_numcone,_809_conesize );
  std::shared_ptr< monty::ndarray< int32_t,1 > > _815_aconidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_814_n)));
  _checked_ptr_1<int32_t> _ptr__815_aconidxs(_815_aconidxs ? _815_aconidxs->raw() : nullptr,_815_aconidxs ? _815_aconidxs->size(0) : 0);
  int32_t _816_aconid = acon_block_map->alloc(_814_n);
  int32_t _817_accid = acc_block_map->alloc(_810_numcone);
  int32_t _818_afeid = afe_allocate(_812_afeidxs);
  acon_block_map->get(_816_aconid,_815_aconidxs,(int)0);
  acc_block_map->get(_817_accid,_813_accidxs,(int)0);
  int32_t _819_acon_block_cap = acon_block_map->block_capacity();
  int32_t _820_acon_cap = acon_block_map->capacity();
  if ((_819_acon_block_cap > (int)((acon_acc)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _821_tmpacc = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_819_acon_block_cap)));
      _checked_ptr_1<int32_t> _ptr__821_tmpacc(_821_tmpacc ? _821_tmpacc->raw() : nullptr,_821_tmpacc ? _821_tmpacc->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _822_tmpafe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_819_acon_block_cap)));
      _checked_ptr_1<int32_t> _ptr__822_tmpafe(_822_tmpafe ? _822_tmpafe->raw() : nullptr,_822_tmpafe ? _822_tmpafe->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(acon_acc,(int)0,_821_tmpacc,(int)0,(int)((acon_acc)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_afe,(int)0,_822_tmpafe,(int)0,(int)((acon_afe)->size(0)));
      acon_acc = _821_tmpacc;
      acon_afe = _822_tmpafe;
    }
  }
  {}
  ((*acon_acc)(_816_aconid)) = _817_accid;
  ((*acon_afe)(_816_aconid)) = _818_afeid;
  if ((_816_aconid >= (int)((acons)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9538_;
      for (int32_t _824_i = (int)0, _9539_ = safe_mul( (int)((acons)->size(0)),(int)2 ); _824_i < _9539_ ; ++_824_i)
      {
        _9538_.push_back(nullptr);
      }
      auto _9540_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9538_.size()),_9538_.begin(),_9538_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > > _823_newacons = _9540_;
      int32_t _9541_ = (int)0;
      int32_t _9542_ = (int)((acons)->size(0));
      for (int32_t _825_i = _9541_; _825_i < _9542_; ++_825_i)
      {
        {
          ((*_823_newacons)(_825_i)) = ((*acons)(_825_i));
        }
      }
      acons = _823_newacons;
    }
  }
  {}
  if ((_820_acon_cap > (int)((acon_elm_accid)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _826_tmp_accid = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_820_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__826_tmp_accid(_826_tmp_accid ? _826_tmp_accid->raw() : nullptr,_826_tmp_accid ? _826_tmp_accid->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _827_tmp_scale = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_820_acon_cap)));
      _checked_ptr_1<double> _ptr__827_tmp_scale(_827_tmp_scale ? _827_tmp_scale->raw() : nullptr,_827_tmp_scale ? _827_tmp_scale->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _828_tmp_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_820_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__828_tmp_ofs(_828_tmp_ofs ? _828_tmp_ofs->raw() : nullptr,_828_tmp_ofs ? _828_tmp_ofs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _829_tmp_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_820_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__829_tmp_afe(_829_tmp_afe ? _829_tmp_afe->raw() : nullptr,_829_tmp_afe ? _829_tmp_afe->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_accid,(int)0,_826_tmp_accid,(int)0,(int)((acon_elm_accid)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_scale,(int)0,_827_tmp_scale,(int)0,(int)((acon_elm_scale)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_ofs,(int)0,_828_tmp_ofs,(int)0,(int)((acon_elm_ofs)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_afe,(int)0,_829_tmp_afe,(int)0,(int)((acon_elm_afe)->size(0)));
      acon_elm_accid = _826_tmp_accid;
      acon_elm_ofs = _828_tmp_ofs;
      acon_elm_afe = _829_tmp_afe;
      acon_elm_scale = _827_tmp_scale;
    }
  }
  {}
  int32_t _830_maxaccidx = (int)0;
  int32_t _9543_ = (int)0;
  int32_t _9544_ = (int)((_813_accidxs)->size(0));
  for (int32_t _831_i = _9543_; _831_i < _9544_; ++_831_i)
  {
    {
      if ((_ptr__813_accidxs[_831_i] > _830_maxaccidx))
      {
        {
          _830_maxaccidx = _ptr__813_accidxs[_831_i];
        }
      }
      {}
    }
  }
  _830_maxaccidx += (int)1;
  int32_t _832_nacc = task_numacc();
  if ((_832_nacc < _830_maxaccidx))
  {
    {
      task_append_accs((int64_t)0,(_830_maxaccidx - _832_nacc),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)0))));
    }
  }
  {}
  std::vector<int64_t> _9545_;
  for (int32_t _833_i = (int)0, _9546_ = (int)((_813_accidxs)->size(0)); _833_i < _9546_ ; ++_833_i)
  {
    _9545_.push_back(_ptr__813_accidxs[_833_i]);
  }
  auto _9547_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9545_.size()),_9545_.begin(),_9545_.end()));
  std::vector<int64_t> _9548_;
  for (int32_t _834_i = (int)0, _9549_ = (int)((_812_afeidxs)->size(0)); _834_i < _9549_ ; ++_834_i)
  {
    _9548_.push_back(_ptr__812_afeidxs[_834_i]);
  }
  auto _9550_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9548_.size()),_9548_.begin(),_9548_.end()));
  task_putacclist(_9547_,_808_domidxs,_9550_,_811_g);
  {
    int32_t _835_k = (int)0;
    int32_t _9551_ = (int)0;
    int32_t _9552_ = _810_numcone;
    for (int32_t _836_i = _9551_; _836_i < _9552_; ++_836_i)
    {
      {
        int32_t _9553_ = (int)0;
        int32_t _9554_ = _809_conesize;
        for (int32_t _837_j = _9553_; _837_j < _9554_; ++_837_j)
        {
          {
            ((*acon_elm_accid)(_ptr__815_aconidxs[_835_k])) = _ptr__813_accidxs[_836_i];
            ((*acon_elm_ofs)(_ptr__815_aconidxs[_835_k])) = _837_j;
            ((*acon_elm_afe)(_ptr__815_aconidxs[_835_k])) = _ptr__812_afeidxs[_835_k];
            ((*acon_elm_scale)(_ptr__815_aconidxs[_835_k])) = 1.0;
            ++ _835_k;
          }
        }
      }
    }
  }
  return _816_aconid;
}
// End mosek.fusion.Model.acon_allocate

// Begin mosek.fusion.Model.afe_release
// Method mosek.fusion.Model.afe_release @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4596:6-4602:7
void mosek::fusion::p_Model::afe_release(int32_t _838_id)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _839_nidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(afe_blocks->blocksize(_838_id))));
  _checked_ptr_1<int32_t> _ptr__839_nidxs(_839_nidxs ? _839_nidxs->raw() : nullptr,_839_nidxs ? _839_nidxs->size(0) : 0);
  afe_blocks->release(_838_id,_839_nidxs);
  std::vector<int64_t> _9555_;
  for (int32_t _840_i = (int)0, _9556_ = (int)((_839_nidxs)->size(0)); _840_i < _9556_ ; ++_840_i)
  {
    _9555_.push_back(_ptr__839_nidxs[_840_i]);
  }
  auto _9557_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9555_.size()),_9555_.begin(),_9555_.end()));
  task_clearafelist(_9557_);
}
// End mosek.fusion.Model.afe_release

// Begin mosek.fusion.Model.afe_allocate
// Method mosek.fusion.Model.afe_allocate @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4581:6-4594:7
int32_t mosek::fusion::p_Model::afe_allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _841_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__841_nativeidxs(_841_nativeidxs ? _841_nativeidxs->raw() : nullptr, _841_nativeidxs ? _841_nativeidxs->size(0) : 0);
  int32_t _842_afeid = afe_blocks->allocate(_841_nativeidxs);
  if (((int)((_841_nativeidxs)->size(0)) > (int)0))
  {
    {
      int32_t _843_tasknum = task_numafe();
      int32_t _844_maxidx = _ptr__841_nativeidxs[(int)0];
      int32_t _9558_ = (int)1;
      int32_t _9559_ = (int)((_841_nativeidxs)->size(0));
      for (int32_t _845_i = _9558_; _845_i < _9559_; ++_845_i)
      {
        {
          if ((_ptr__841_nativeidxs[_845_i] > _844_maxidx))
          {
            {
              _844_maxidx = _ptr__841_nativeidxs[_845_i];
            }
          }
          {}
        }
      }
      if ((safe_add( _844_maxidx,(int)1 ) > _843_tasknum))
      {
        {
          int32_t _846_n = (safe_add( _844_maxidx,(int)1 ) - _843_tasknum);
          task_append_afes((int64_t)(_846_n));
          task_numaferow += _846_n;
        }
      }
      {}
    }
  }
  {}
  return _842_afeid;
}
// End mosek.fusion.Model.afe_allocate

// Begin mosek.fusion.Model.con_release
// Method mosek.fusion.Model.con_release @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4566:6-4576:7
void mosek::fusion::p_Model::con_release(int32_t _847_id)
{
  int32_t _848_num = con_blocks->blocksize(_847_id);
  if ((_848_num >= (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _849_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_848_num)));
      _checked_ptr_1<int32_t> _ptr__849_nativeidxs(_849_nativeidxs ? _849_nativeidxs->raw() : nullptr,_849_nativeidxs ? _849_nativeidxs->size(0) : 0);
      con_blocks->release(_847_id,_849_nativeidxs);
      ((*cons)(_847_id)) = nullptr;
      task_con_putboundlist_fr(_849_nativeidxs);
    }
  }
  {}
}
// End mosek.fusion.Model.con_release

// Begin mosek.fusion.Model.con_allocate
// Method mosek.fusion.Model.con_allocate @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4546:6-4564:7
int32_t mosek::fusion::p_Model::con_allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _850_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__850_nativeidxs(_850_nativeidxs ? _850_nativeidxs->raw() : nullptr, _850_nativeidxs ? _850_nativeidxs->size(0) : 0);
  int32_t _851_conid = con_blocks->allocate(_850_nativeidxs);
  if (((int)((_850_nativeidxs)->size(0)) > (int)0))
  {
    {
      int32_t _852_maxidx = _ptr__850_nativeidxs[(int)0];
      int32_t _9560_ = (int)1;
      int32_t _9561_ = (int)((_850_nativeidxs)->size(0));
      for (int32_t _853_i = _9560_; _853_i < _9561_; ++_853_i)
      {
        {
          if ((_ptr__850_nativeidxs[_853_i] > _852_maxidx))
          {
            {
              _852_maxidx = _ptr__850_nativeidxs[_853_i];
            }
          }
          {}
        }
      }
      if ((safe_add( _852_maxidx,(int)1 ) > num_task_con))
      {
        {
          int32_t _854_n = (safe_add( _852_maxidx,(int)1 ) - num_task_con);
          task_append_con(_854_n);
          num_task_con += _854_n;
        }
      }
      {}
    }
  }
  {}
  if ((_851_conid >= (int)((cons)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9562_;
      for (int32_t _856_i = (int)0, _9563_ = safe_mul( (int)((cons)->size(0)),(int)2 ); _856_i < _9563_ ; ++_856_i)
      {
        _9562_.push_back(nullptr);
      }
      auto _9564_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9562_.size()),_9562_.begin(),_9562_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > > _855_newcons = _9564_;
      int32_t _9565_ = (int)0;
      int32_t _9566_ = (int)((cons)->size(0));
      for (int32_t _857_i = _9565_; _857_i < _9566_; ++_857_i)
      {
        {
          ((*_855_newcons)(_857_i)) = ((*cons)(_857_i));
        }
      }
      cons = _855_newcons;
    }
  }
  {}
  return _851_conid;
}
// End mosek.fusion.Model.con_allocate

// Begin mosek.fusion.Model.barvar_alloc
// Method mosek.fusion.Model.barvar_alloc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4439:6-4540:7
int32_t mosek::fusion::p_Model::barvar_alloc(int32_t _858_conedim,int32_t _859_numcone,std::shared_ptr< monty::ndarray< int32_t,1 > > _860_barvaridxs,std::shared_ptr< monty::ndarray< int64_t,1 > > _861_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__860_barvaridxs(_860_barvaridxs ? _860_barvaridxs->raw() : nullptr, _860_barvaridxs ? _860_barvaridxs->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__861_nativeidxs(_861_nativeidxs ? _861_nativeidxs->raw() : nullptr, _861_nativeidxs ? _861_nativeidxs->size(0) : 0);
  int32_t _862_barvarid = barvar_num;
  ++ barvar_num;
  int32_t _863_conesize = (safe_mul( _858_conedim,safe_add( _858_conedim,(int)1 ) ) / (int)2);
  int32_t _864_domsize = safe_mul( _863_conesize,_859_numcone );
  int32_t _865_block_num = ((*barvar_block_ptr)(_862_barvarid));
  int32_t _866_block_cap = (int)((barvar_block_dim)->size(0));
  int32_t _867_block_elm_num = ((*barvar_block_elm_ptr)(_862_barvarid));
  int32_t _868_block_elm_cap = (int)((barvar_block_elm_barj)->size(0));
  if ((_866_block_cap < safe_add( _865_block_num,(int)1 )))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _869_new_barvar_block_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ))));
      _checked_ptr_1<int32_t> _ptr__869_new_barvar_block_dim(_869_new_barvar_block_dim ? _869_new_barvar_block_dim->raw() : nullptr,_869_new_barvar_block_dim ? _869_new_barvar_block_dim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _870_new_barvar_block_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__870_new_barvar_block_ptr(_870_new_barvar_block_ptr ? _870_new_barvar_block_ptr->raw() : nullptr,_870_new_barvar_block_ptr ? _870_new_barvar_block_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _871_new_barvar_block_elm_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__871_new_barvar_block_elm_ptr(_871_new_barvar_block_elm_ptr ? _871_new_barvar_block_elm_ptr->raw() : nullptr,_871_new_barvar_block_elm_ptr ? _871_new_barvar_block_elm_ptr->size(0) : 0);
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9567_;
      for (int32_t _873_i = (int)0, _9568_ = safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ); _873_i < _9568_ ; ++_873_i)
      {
        _9567_.push_back(nullptr);
      }
      auto _9569_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9567_.size()),_9567_.begin(),_9567_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > > _872_new_barvars = _9569_;
      int32_t _9570_ = (int)0;
      int32_t _9571_ = (int)((barvars)->size(0));
      for (int32_t _874_i = _9570_; _874_i < _9571_; ++_874_i)
      {
        {
          ((*_872_new_barvars)(_874_i)) = ((*barvars)(_874_i));
        }
      }
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_dim,(int)0,_869_new_barvar_block_dim,(int)0,(int)((barvar_block_dim)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_ptr,(int)0,_870_new_barvar_block_ptr,(int)0,(int)((barvar_block_ptr)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_ptr,(int)0,_871_new_barvar_block_elm_ptr,(int)0,(int)((barvar_block_elm_ptr)->size(0)));
      barvar_block_dim = _869_new_barvar_block_dim;
      barvar_block_ptr = _870_new_barvar_block_ptr;
      barvar_block_elm_ptr = _871_new_barvar_block_elm_ptr;
      barvars = _872_new_barvars;
    }
  }
  {}
  if ((_868_block_elm_cap < safe_add( _867_block_elm_num,_864_domsize )))
  {
    {
      int32_t _875_addnum = (safe_add( _868_block_elm_cap,_864_domsize ) - _867_block_elm_num);
      if ((_875_addnum < _868_block_elm_cap))
      {
        {
          _875_addnum = _868_block_elm_cap;
        }
      }
      {}
      int32_t _876_newlen = safe_add( _868_block_elm_cap,_875_addnum );
      std::shared_ptr< monty::ndarray< int32_t,1 > > _877_new_barvar_block_elm_barj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_876_newlen)));
      _checked_ptr_1<int32_t> _ptr__877_new_barvar_block_elm_barj(_877_new_barvar_block_elm_barj ? _877_new_barvar_block_elm_barj->raw() : nullptr,_877_new_barvar_block_elm_barj ? _877_new_barvar_block_elm_barj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _878_new_barvar_block_elm_i = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_876_newlen)));
      _checked_ptr_1<int32_t> _ptr__878_new_barvar_block_elm_i(_878_new_barvar_block_elm_i ? _878_new_barvar_block_elm_i->raw() : nullptr,_878_new_barvar_block_elm_i ? _878_new_barvar_block_elm_i->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _879_new_barvar_block_elm_j = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_876_newlen)));
      _checked_ptr_1<int32_t> _ptr__879_new_barvar_block_elm_j(_879_new_barvar_block_elm_j ? _879_new_barvar_block_elm_j->raw() : nullptr,_879_new_barvar_block_elm_j ? _879_new_barvar_block_elm_j->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_barj,(int)0,_877_new_barvar_block_elm_barj,(int)0,(int)((barvar_block_elm_barj)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_i,(int)0,_878_new_barvar_block_elm_i,(int)0,(int)((barvar_block_elm_i)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_j,(int)0,_879_new_barvar_block_elm_j,(int)0,(int)((barvar_block_elm_j)->size(0)));
      barvar_block_elm_barj = _877_new_barvar_block_elm_barj;
      barvar_block_elm_i = _878_new_barvar_block_elm_i;
      barvar_block_elm_j = _879_new_barvar_block_elm_j;
    }
  }
  {}
  int32_t _880_firstcone = task_numbarvar();
  int32_t _9572_ = (int)0;
  int32_t _9573_ = _859_numcone;
  for (int32_t _881_i = _9572_; _881_i < _9573_; ++_881_i)
  {
    {
      _ptr__860_barvaridxs[_881_i] = safe_add( _880_firstcone,_881_i );
    }
  }
  if ((safe_add( _880_firstcone,_859_numcone ) >= (int)((barvar_dim)->size(0))))
  {
    {
      int32_t _882_addnum = (safe_add( _880_firstcone,_859_numcone ) - (int)((barvar_dim)->size(0)));
      if ((_882_addnum < (int)((barvar_dim)->size(0))))
      {
        {
          _882_addnum = (int)((barvar_dim)->size(0));
        }
      }
      {}
      int32_t _883_newnum = safe_add( _882_addnum,(int)((barvar_dim)->size(0)) );
      std::shared_ptr< monty::ndarray< int32_t,1 > > _884_new_barvar_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_883_newnum)));
      _checked_ptr_1<int32_t> _ptr__884_new_barvar_dim(_884_new_barvar_dim ? _884_new_barvar_dim->raw() : nullptr,_884_new_barvar_dim ? _884_new_barvar_dim->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_dim,(int)0,_884_new_barvar_dim,(int)0,(int)((barvar_dim)->size(0)));
      barvar_dim = _884_new_barvar_dim;
    }
  }
  {}
  ((*barvar_block_dim)(_862_barvarid)) = _858_conedim;
  ((*barvar_block_ptr)(safe_add( _862_barvarid,(int)1 ))) = safe_add( ((*barvar_block_ptr)(_862_barvarid)),_859_numcone );
  ((*barvar_block_elm_ptr)(safe_add( _862_barvarid,(int)1 ))) = safe_add( ((*barvar_block_elm_ptr)(_862_barvarid)),_864_domsize );
  {
    int32_t _885_k = (int)0;
    int32_t _886_b = ((*barvar_block_elm_ptr)(_862_barvarid));
    int32_t _9574_ = (int)0;
    int32_t _9575_ = _859_numcone;
    for (int32_t _887_i = _9574_; _887_i < _9575_; ++_887_i)
    {
      {
        int32_t _9576_ = (int)0;
        int32_t _9577_ = _858_conedim;
        for (int32_t _888_j1 = _9576_; _888_j1 < _9577_; ++_888_j1)
        {
          {
            int32_t _9578_ = _888_j1;
            int32_t _9579_ = _858_conedim;
            for (int32_t _889_j0 = _9578_; _889_j0 < _9579_; ++_889_j0)
            {
              {
                ((*barvar_block_elm_barj)(safe_add( _886_b,_885_k ))) = safe_add( _880_firstcone,_887_i );
                ((*barvar_block_elm_i)(safe_add( _886_b,_885_k ))) = _889_j0;
                ((*barvar_block_elm_j)(safe_add( _886_b,_885_k ))) = _888_j1;
                _ptr__861_nativeidxs[_885_k] = safe_add( _886_b,_885_k );
                ++ _885_k;
              }
            }
          }
        }
      }
    }
    int32_t _9580_ = (int)0;
    int32_t _9581_ = _859_numcone;
    for (int32_t _890_i = _9580_; _890_i < _9581_; ++_890_i)
    {
      {
        ((*barvar_dim)(safe_add( _880_firstcone,_890_i ))) = _858_conedim;
      }
    }
  }
  int32_t _891_barvaridx = task_append_barvar(_858_conedim,_859_numcone);
  return _862_barvarid;
}
// End mosek.fusion.Model.barvar_alloc

// Begin mosek.fusion.Model.conicvar_alloc
// Method mosek.fusion.Model.conicvar_alloc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4406:6-4437:7
int32_t mosek::fusion::p_Model::conicvar_alloc(std::shared_ptr< monty::ndarray< int64_t,1 > > _892_domidxs,int32_t _893_conesize,int32_t _894_numcone,std::shared_ptr< monty::ndarray< int32_t,1 > > _895_accidxs,std::shared_ptr< monty::ndarray< int32_t,1 > > _896_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__892_domidxs(_892_domidxs ? _892_domidxs->raw() : nullptr, _892_domidxs ? _892_domidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__895_accidxs(_895_accidxs ? _895_accidxs->raw() : nullptr, _895_accidxs ? _895_accidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__896_nativeidxs(_896_nativeidxs ? _896_nativeidxs->raw() : nullptr, _896_nativeidxs ? _896_nativeidxs->size(0) : 0);
  int32_t _897_n = safe_mul( _893_conesize,_894_numcone );
  int32_t _898_varid = linearvar_alloc(_897_n,_896_nativeidxs);
  std::shared_ptr< monty::ndarray< double,1 > > _899_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_897_n)));
  _checked_ptr_1<double> _ptr__899_g(_899_g ? _899_g->raw() : nullptr,_899_g ? _899_g->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _900_conelmidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_897_n)));
  _checked_ptr_1<int32_t> _ptr__900_conelmidx(_900_conelmidx ? _900_conelmidx->raw() : nullptr,_900_conelmidx ? _900_conelmidx->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _901_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_897_n)));
  _checked_ptr_1<int32_t> _ptr__901_afeidxs(_901_afeidxs ? _901_afeidxs->raw() : nullptr,_901_afeidxs ? _901_afeidxs->size(0) : 0);
  int32_t _902_conid = acon_allocate(_892_domidxs,_893_conesize,_894_numcone,_899_g,_901_afeidxs,_895_accidxs);
  ((*var_block_acc_id)(_898_varid)) = _902_conid;
  int32_t _9582_ = (int)0;
  int32_t _9583_ = _897_n;
  for (int32_t _903_i = _9582_; _903_i < _9583_; ++_903_i)
  {
    {
      ((*var_elm_acc_idx)(_ptr__896_nativeidxs[_903_i])) = _ptr__895_accidxs[(_903_i / _893_conesize)];
      ((*var_elm_acc_ofs)(_ptr__896_nativeidxs[_903_i])) = (_903_i % _893_conesize);
    }
  }
  std::vector<int64_t> _9584_;
  for (int32_t _904_i = (int)0, _9585_ = (int)((_901_afeidxs)->size(0)); _904_i < _9585_ ; ++_904_i)
  {
    _9584_.push_back(_ptr__901_afeidxs[_904_i]);
  }
  auto _9586_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9584_.size()),_9584_.begin(),_9584_.end()));
  std::vector<int32_t> _9587_;
  for (int32_t _905_i = (int)0, _9588_ = safe_add( _897_n,(int)1 ); _905_i < _9588_ ; ++_905_i)
  {
    _9587_.push_back(_905_i);
  }
  auto _9589_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9587_.size()),_9587_.begin(),_9587_.end()));
  std::vector<double> _9590_;
  for (int32_t _906_i = (int)0, _9591_ = _897_n; _906_i < _9591_ ; ++_906_i)
  {
    _9590_.push_back(1.0);
  }
  auto _9592_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9590_.size()),_9590_.begin(),_9590_.end()));
  std::vector<double> _9593_;
  for (int32_t _907_i = (int)0, _9594_ = _897_n; _907_i < _9594_ ; ++_907_i)
  {
    _9593_.push_back(0.0);
  }
  auto _9595_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9593_.size()),_9593_.begin(),_9593_.end()));
  task_putafeflist(_9586_,_9589_,_896_nativeidxs,_9592_,_9595_);
  return _898_varid;
}
// End mosek.fusion.Model.conicvar_alloc

// Begin mosek.fusion.Model.linearvar_alloc
// Method mosek.fusion.Model.linearvar_alloc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4368:6-4404:7
int32_t mosek::fusion::p_Model::linearvar_alloc(int32_t _908_n,std::shared_ptr< monty::ndarray< int32_t,1 > > _909_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__909_nativeidxs(_909_nativeidxs ? _909_nativeidxs->raw() : nullptr, _909_nativeidxs ? _909_nativeidxs->size(0) : 0);
  int32_t _910_varid = var_block_map->alloc(_908_n);
  int32_t _911_maxvaridx = safe_add( var_block_map->maxidx(_910_varid),(int)1 );
  int32_t _912_cap = var_block_map->capacity();
  int32_t _913_numblocks = var_block_map->block_capacity();
  var_block_map->get(_910_varid,_909_nativeidxs,(int)0);
  if ((task_numvar() < _911_maxvaridx))
  {
    {
      task_append_var((_911_maxvaridx - task_numvar()));
    }
  }
  {}
  if ((_910_varid >= (int)((vars)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9596_;
      for (int32_t _915_i = (int)0, _9597_ = safe_mul( (int)((vars)->size(0)),(int)2 ); _915_i < _9597_ ; ++_915_i)
      {
        _9596_.push_back(nullptr);
      }
      auto _9598_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9596_.size()),_9596_.begin(),_9596_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > > _914_newvars = _9598_;
      int32_t _9599_ = (int)0;
      int32_t _9600_ = (int)((vars)->size(0));
      for (int32_t _916_i = _9599_; _916_i < _9600_; ++_916_i)
      {
        {
          ((*_914_newvars)(_916_i)) = ((*vars)(_916_i));
        }
      }
      vars = _914_newvars;
    }
  }
  {}
  if ((_913_numblocks > (int)((var_block_acc_id)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _917_new_var_block_acc_id = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_913_numblocks)));
      _checked_ptr_1<int32_t> _ptr__917_new_var_block_acc_id(_917_new_var_block_acc_id ? _917_new_var_block_acc_id->raw() : nullptr,_917_new_var_block_acc_id ? _917_new_var_block_acc_id->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_block_acc_id,(int)0,_917_new_var_block_acc_id,(int)0,(int)((var_block_acc_id)->size(0)));
      var_block_acc_id = _917_new_var_block_acc_id;
    }
  }
  {}
  if ((_912_cap > (int)((var_elm_acc_idx)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _918_new_var_elm_acc_idx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_912_cap)));
      _checked_ptr_1<int32_t> _ptr__918_new_var_elm_acc_idx(_918_new_var_elm_acc_idx ? _918_new_var_elm_acc_idx->raw() : nullptr,_918_new_var_elm_acc_idx ? _918_new_var_elm_acc_idx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_elm_acc_idx,(int)0,_918_new_var_elm_acc_idx,(int)0,(int)((var_elm_acc_idx)->size(0)));
      var_elm_acc_idx = _918_new_var_elm_acc_idx;
      std::shared_ptr< monty::ndarray< int32_t,1 > > _919_new_var_elm_acc_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_912_cap)));
      _checked_ptr_1<int32_t> _ptr__919_new_var_elm_acc_ofs(_919_new_var_elm_acc_ofs ? _919_new_var_elm_acc_ofs->raw() : nullptr,_919_new_var_elm_acc_ofs ? _919_new_var_elm_acc_ofs->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_elm_acc_ofs,(int)0,_919_new_var_elm_acc_ofs,(int)0,(int)((var_elm_acc_ofs)->size(0)));
      var_elm_acc_ofs = _919_new_var_elm_acc_ofs;
    }
  }
  {}
  ((*var_block_acc_id)(_910_varid)) = (-(int)1);
  int32_t _9601_ = (int)0;
  int32_t _9602_ = _908_n;
  for (int32_t _920_i = _9601_; _920_i < _9602_; ++_920_i)
  {
    {
      ((*var_elm_acc_idx)(_ptr__909_nativeidxs[_920_i])) = (-(int)1);
    }
  }
  return _910_varid;
}
// End mosek.fusion.Model.linearvar_alloc

// Begin mosek.fusion.Model.make_continuous
// Method mosek.fusion.Model.make_continuous @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4354:6-4365:7
void mosek::fusion::p_Model::make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _921_idxs)
{
  _checked_ptr_1<int64_t> _ptr__921_idxs(_921_idxs ? _921_idxs->raw() : nullptr, _921_idxs ? _921_idxs->size(0) : 0);
  int32_t _922_count = (int)0;
  int32_t _9603_ = (int)0;
  int32_t _9604_ = (int)((_921_idxs)->size(0));
  for (int32_t _923_i = _9603_; _923_i < _9604_; ++_923_i)
  {
    {
      if ((_ptr__921_idxs[_923_i] > (int)0))
      {
        {
          ++ _922_count;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _924_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_922_count)));
  _checked_ptr_1<int32_t> _ptr__924_nativeidxs(_924_nativeidxs ? _924_nativeidxs->raw() : nullptr,_924_nativeidxs ? _924_nativeidxs->size(0) : 0);
  int32_t _925_k = (int)0;
  int32_t _9605_ = (int)0;
  int32_t _9606_ = (int)((_921_idxs)->size(0));
  for (int32_t _926_i = _9605_; _926_i < _9606_; ++_926_i)
  {
    {
      if ((_ptr__921_idxs[_926_i] > (int)0))
      {
        {
          _ptr__924_nativeidxs[_925_k] = (int32_t)_ptr__921_idxs[_926_i];
          ++ _925_k;
        }
      }
      {}
    }
  }
  task_var_putcontlist(_924_nativeidxs);
}
// End mosek.fusion.Model.make_continuous

// Begin mosek.fusion.Model.make_integer
// Method mosek.fusion.Model.make_integer @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4342:6-4352:7
void mosek::fusion::p_Model::make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _927_idxs)
{
  _checked_ptr_1<int64_t> _ptr__927_idxs(_927_idxs ? _927_idxs->raw() : nullptr, _927_idxs ? _927_idxs->size(0) : 0);
  int32_t _928_count = (int)0;
  int32_t _9607_ = (int)0;
  int32_t _9608_ = (int)((_927_idxs)->size(0));
  for (int32_t _929_i = _9607_; _929_i < _9608_; ++_929_i)
  {
    {
      if ((_ptr__927_idxs[_929_i] > (int)0))
      {
        {
          ++ _928_count;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _930_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_928_count)));
  _checked_ptr_1<int32_t> _ptr__930_nativeidxs(_930_nativeidxs ? _930_nativeidxs->raw() : nullptr,_930_nativeidxs ? _930_nativeidxs->size(0) : 0);
  int32_t _931_k = (int)0;
  int32_t _9609_ = (int)0;
  int32_t _9610_ = (int)((_927_idxs)->size(0));
  for (int32_t _932_i = _9609_; _932_i < _9610_; ++_932_i)
  {
    {
      if ((_ptr__927_idxs[_932_i] > (int)0))
      {
        {
          _ptr__930_nativeidxs[_931_k] = (int32_t)_ptr__927_idxs[_932_i];
          ++ _931_k;
        }
      }
      {}
    }
  }
  task_var_putintlist(_930_nativeidxs);
}
// End mosek.fusion.Model.make_integer

// Begin mosek.fusion.Model.putlicensewait
// Method mosek.fusion.Model.putlicensewait @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4340:6-87
void mosek::fusion::Model::putlicensewait
  ( bool _933_wait) { mosek::fusion::p_Model::putlicensewait(_933_wait); }
void mosek::fusion::p_Model::putlicensewait(bool _933_wait)
{
  ::mosek::fusion::p_BaseModel::env_putlicensewait(_933_wait);
}
// End mosek.fusion.Model.putlicensewait

// Begin mosek.fusion.Model.putlicensepath
// Method mosek.fusion.Model.putlicensepath @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4337:6-90
void mosek::fusion::Model::putlicensepath
  ( const std::string & _934_licfile) { mosek::fusion::p_Model::putlicensepath(_934_licfile); }
void mosek::fusion::p_Model::putlicensepath(const std::string & _934_licfile)
{
  ::mosek::fusion::p_BaseModel::env_putlicensepath(_934_licfile);
}
// End mosek.fusion.Model.putlicensepath

// Begin mosek.fusion.Model.putlicensecode
// Method mosek.fusion.Model.putlicensecode @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4333:6-87
void mosek::fusion::Model::putlicensecode
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _935_code) { mosek::fusion::p_Model::putlicensecode(_935_code); }
void mosek::fusion::p_Model::putlicensecode(std::shared_ptr< monty::ndarray< int32_t,1 > > _935_code)
{
  _checked_ptr_1<int32_t> _ptr__935_code(_935_code ? _935_code->raw() : nullptr, _935_code ? _935_code->size(0) : 0);
  ::mosek::fusion::p_BaseModel::env_putlicensecode(_935_code);
}
// End mosek.fusion.Model.putlicensecode

// Begin mosek.fusion.Model.dispose
// Method mosek.fusion.Model.dispose @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4318:6-4327:7
void mosek::fusion::Model :: dispose() { mosek::fusion::p_Model::_get_impl(this)->dispose(); }
void mosek::fusion::p_Model::dispose()
{
  int32_t _9611_ = (int)0;
  int32_t _9612_ = (int)((vars)->size(0));
  for (int32_t _936_i = _9611_; _936_i < _9612_; ++_936_i)
  {
    {
      ((*vars)(_936_i)) = nullptr;
    }
  }
  int32_t _9613_ = (int)0;
  int32_t _9614_ = (int)((cons)->size(0));
  for (int32_t _937_i = _9613_; _937_i < _9614_; ++_937_i)
  {
    {
      ((*cons)(_937_i)) = nullptr;
    }
  }
  int32_t _9615_ = (int)0;
  int32_t _9616_ = (int)((acons)->size(0));
  for (int32_t _938_i = _9615_; _938_i < _9616_; ++_938_i)
  {
    {
      ((*acons)(_938_i)) = nullptr;
    }
  }
  int32_t _9617_ = (int)0;
  int32_t _9618_ = (int)((barvars)->size(0));
  for (int32_t _939_i = _9617_; _939_i < _9618_; ++_939_i)
  {
    {
      ((*barvars)(_939_i)) = nullptr;
    }
  }
  if ((NULL != parameters.get()))
  {
    {
      int32_t _9619_ = (int)0;
      int32_t _9620_ = (int)((parameters)->size(0));
      for (int32_t _940_i = _9619_; _940_i < _9620_; ++_940_i)
      {
        {
          ((*parameters)(_940_i)) = nullptr;
        }
      }
    }
  }
  {}
  ::mosek::fusion::p_BaseModel::dispose();
}
// End mosek.fusion.Model.dispose

// Begin mosek.fusion.Model.getTask
// Method mosek.fusion.Model.getTask @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4298:6-64
MSKtask_t mosek::fusion::Model :: __mosek_2fusion_2Model__getTask() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getTask(); }
MSKtask_t mosek::fusion::Model::getTask() { return __mosek_2fusion_2Model__getTask(); }
MSKtask_t mosek::fusion::p_Model::__mosek_2fusion_2Model__getTask()
{
  return __mosek_2fusion_2BaseModel__task_get();
}
// End mosek.fusion.Model.getTask

// Begin mosek.fusion.Model.getConstraintDuals
// Method mosek.fusion.Model.getConstraintDuals @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4272:6-4287:7
void mosek::fusion::Model :: getConstraintDuals(bool _941_lower,std::shared_ptr< monty::ndarray< int32_t,1 > > _942_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _943_res,int32_t _944_offset) { mosek::fusion::p_Model::_get_impl(this)->getConstraintDuals(_941_lower,_942_nativeidxs,_943_res,_944_offset); }
void mosek::fusion::p_Model::getConstraintDuals(bool _941_lower,std::shared_ptr< monty::ndarray< int32_t,1 > > _942_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _943_res,int32_t _944_offset)
{
  _checked_ptr_1<int32_t> _ptr__942_nativeidxs(_942_nativeidxs ? _942_nativeidxs->raw() : nullptr, _942_nativeidxs ? _942_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__943_res(_943_res ? _943_res->raw() : nullptr, _943_res ? _943_res->size(0) : 0);
  int32_t _945_n = (int)((_942_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< double,1 > > _946_slc = getSolution_slc(::mosek::fusion::SolutionType::Default);
  _checked_ptr_1<double> _ptr__946_slc(_946_slc ? _946_slc->raw() : nullptr,_946_slc ? _946_slc->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _947_suc = getSolution_suc(::mosek::fusion::SolutionType::Default);
  _checked_ptr_1<double> _ptr__947_suc(_947_suc ? _947_suc->raw() : nullptr,_947_suc ? _947_suc->size(0) : 0);
  int32_t _9621_ = (int)0;
  int32_t _9622_ = _945_n;
  for (int32_t _948_p = _9621_; _948_p < _9622_; ++_948_p)
  {
    {
      if ((_ptr__942_nativeidxs[_948_p] >= (int)0))
      {
        {
          if (_941_lower)
          {
            {
              _ptr__943_res[safe_add( _944_offset,_948_p )] = _ptr__946_slc[_ptr__942_nativeidxs[_948_p]];
            }
          }
          else
          {
            {
              _ptr__943_res[safe_add( _944_offset,_948_p )] = (-_ptr__947_suc[_ptr__942_nativeidxs[_948_p]]);
            }
          }
        }
      }
      else
      {
        {
          throw ::mosek::fusion::SolutionError(std::string ("Requested invalid solution item"));
        }
      }
    }
  }
}
// End mosek.fusion.Model.getConstraintDuals

// Begin mosek.fusion.Model.getConstraintValues
// Method mosek.fusion.Model.getConstraintValues @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4238:6-4269:7
void mosek::fusion::Model :: getConstraintValues(bool _949_primal,std::shared_ptr< monty::ndarray< int32_t,1 > > _950_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _951_res,int32_t _952_offset) { mosek::fusion::p_Model::_get_impl(this)->getConstraintValues(_949_primal,_950_nativeidxs,_951_res,_952_offset); }
void mosek::fusion::p_Model::getConstraintValues(bool _949_primal,std::shared_ptr< monty::ndarray< int32_t,1 > > _950_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _951_res,int32_t _952_offset)
{
  _checked_ptr_1<int32_t> _ptr__950_nativeidxs(_950_nativeidxs ? _950_nativeidxs->raw() : nullptr, _950_nativeidxs ? _950_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__951_res(_951_res ? _951_res->raw() : nullptr, _951_res ? _951_res->size(0) : 0);
  int32_t _953_n = (int)((_950_nativeidxs)->size(0));
  if (_949_primal)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _954_xc = getSolution_xc(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__954_xc(_954_xc ? _954_xc->raw() : nullptr,_954_xc ? _954_xc->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _955_accx = getSolution_accx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__955_accx(_955_accx ? _955_accx->raw() : nullptr,_955_accx ? _955_accx->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _956_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__956_accptr(_956_accptr ? _956_accptr->raw() : nullptr,_956_accptr ? _956_accptr->size(0) : 0);
      int32_t _9623_ = (int)0;
      int32_t _9624_ = _953_n;
      for (int32_t _957_p = _9623_; _957_p < _9624_; ++_957_p)
      {
        {
          if ((_ptr__950_nativeidxs[_957_p] >= (int)0))
          {
            {
              _ptr__951_res[safe_add( _952_offset,_957_p )] = _ptr__954_xc[_ptr__950_nativeidxs[_957_p]];
            }
          }
          else
          {
            {
              int32_t _958_i = (-safe_add( (int)1,_ptr__950_nativeidxs[_957_p] ));
              _ptr__951_res[safe_add( _952_offset,_957_p )] = (_ptr__955_accx[safe_add( _ptr__956_accptr[((*acon_elm_accid)(_958_i))],((*acon_elm_ofs)(_958_i)) )] * ((*acon_elm_scale)(_958_i)));
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _959_y = getSolution_y(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__959_y(_959_y ? _959_y->raw() : nullptr,_959_y ? _959_y->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _960_accy = getSolution_accy(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__960_accy(_960_accy ? _960_accy->raw() : nullptr,_960_accy ? _960_accy->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _961_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__961_accptr(_961_accptr ? _961_accptr->raw() : nullptr,_961_accptr ? _961_accptr->size(0) : 0);
      int32_t _9625_ = (int)0;
      int32_t _9626_ = _953_n;
      for (int32_t _962_p = _9625_; _962_p < _9626_; ++_962_p)
      {
        {
          if ((_ptr__950_nativeidxs[_962_p] >= (int)0))
          {
            {
              _ptr__951_res[safe_add( _952_offset,_962_p )] = _ptr__959_y[_ptr__950_nativeidxs[_962_p]];
            }
          }
          else
          {
            {
              int32_t _963_i = (-safe_add( (int)1,_ptr__950_nativeidxs[_962_p] ));
              _ptr__951_res[safe_add( _952_offset,_962_p )] = (_ptr__960_accy[safe_add( _ptr__961_accptr[((*acon_elm_accid)(_963_i))],((*acon_elm_ofs)(_963_i)) )] * ((*acon_elm_scale)(_963_i)));
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getConstraintValues

// Begin mosek.fusion.Model.getVariableDuals
// Method mosek.fusion.Model.getVariableDuals @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4208:6-4235:7
void mosek::fusion::Model :: getVariableDuals(bool _964_lower,std::shared_ptr< monty::ndarray< int64_t,1 > > _965_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _966_res,int32_t _967_offset) { mosek::fusion::p_Model::_get_impl(this)->getVariableDuals(_964_lower,_965_nativeidxs,_966_res,_967_offset); }
void mosek::fusion::p_Model::getVariableDuals(bool _964_lower,std::shared_ptr< monty::ndarray< int64_t,1 > > _965_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _966_res,int32_t _967_offset)
{
  _checked_ptr_1<int64_t> _ptr__965_nativeidxs(_965_nativeidxs ? _965_nativeidxs->raw() : nullptr, _965_nativeidxs ? _965_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__966_res(_966_res ? _966_res->raw() : nullptr, _966_res ? _966_res->size(0) : 0);
  int32_t _968_n = (int)((_965_nativeidxs)->size(0));
  if (_964_lower)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _969_xx = getSolution_slx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__969_xx(_969_xx ? _969_xx->raw() : nullptr,_969_xx ? _969_xx->size(0) : 0);
      int32_t _9627_ = (int)0;
      int32_t _9628_ = _968_n;
      for (int32_t _970_p = _9627_; _970_p < _9628_; ++_970_p)
      {
        {
          if ((_ptr__965_nativeidxs[_970_p] > (int)0))
          {
            {
              _ptr__966_res[safe_add( _967_offset,_970_p )] = _ptr__969_xx[_ptr__965_nativeidxs[_970_p]];
            }
          }
          else
          {
            {
              _ptr__966_res[safe_add( _967_offset,_970_p )] = (int)0;
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _971_xx = getSolution_sux(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__971_xx(_971_xx ? _971_xx->raw() : nullptr,_971_xx ? _971_xx->size(0) : 0);
      int32_t _9629_ = (int)0;
      int32_t _9630_ = _968_n;
      for (int32_t _972_p = _9629_; _972_p < _9630_; ++_972_p)
      {
        {
          if ((_ptr__965_nativeidxs[_972_p] > (int)0))
          {
            {
              _ptr__966_res[safe_add( _967_offset,_972_p )] = (-_ptr__971_xx[_ptr__965_nativeidxs[_972_p]]);
            }
          }
          else
          {
            {
              _ptr__966_res[safe_add( _967_offset,_972_p )] = (int)0;
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getVariableDuals

// Begin mosek.fusion.Model.getVariableValues
// Method mosek.fusion.Model.getVariableValues @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4167:6-4204:7
void mosek::fusion::Model :: getVariableValues(bool _973_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _974_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _975_res,int32_t _976_offset) { mosek::fusion::p_Model::_get_impl(this)->getVariableValues(_973_primal,_974_nativeidxs,_975_res,_976_offset); }
void mosek::fusion::p_Model::getVariableValues(bool _973_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _974_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _975_res,int32_t _976_offset)
{
  _checked_ptr_1<int64_t> _ptr__974_nativeidxs(_974_nativeidxs ? _974_nativeidxs->raw() : nullptr, _974_nativeidxs ? _974_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__975_res(_975_res ? _975_res->raw() : nullptr, _975_res ? _975_res->size(0) : 0);
  int32_t _977_n = (int)((_974_nativeidxs)->size(0));
  barvar_num = task_numbarvar();
  if (_973_primal)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _978_xx = getSolution_xx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__978_xx(_978_xx ? _978_xx->raw() : nullptr,_978_xx ? _978_xx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _9631_;
      bool _9632_ = (barvar_num > (int)0);
      if (_9632_)
      {
        _9631_ = getSolution_barx(::mosek::fusion::SolutionType::Default);
      }
      else
      {
        _9631_ = nullptr;
      }
      std::shared_ptr< monty::ndarray< double,1 > > _979_barx = _9631_;
      _checked_ptr_1<double> _ptr__979_barx(_979_barx ? _979_barx->raw() : nullptr,_979_barx ? _979_barx->size(0) : 0);
      int32_t _9633_ = (int)0;
      int32_t _9634_ = _977_n;
      for (int32_t _980_p = _9633_; _980_p < _9634_; ++_980_p)
      {
        {
          if ((_ptr__974_nativeidxs[_980_p] > (int)0))
          {
            {
              _ptr__975_res[safe_add( _976_offset,_980_p )] = _ptr__978_xx[_ptr__974_nativeidxs[_980_p]];
            }
          }
          else if((_ptr__974_nativeidxs[_980_p] < (int)0))
          {
            {
              int64_t _981_idx = (-safe_add( _ptr__974_nativeidxs[_980_p],(int)1 ));
              _ptr__975_res[safe_add( _976_offset,_980_p )] = _ptr__979_barx[_981_idx];
            }
          }
          else
          {
            {
              _ptr__975_res[safe_add( _976_offset,_980_p )] = (int)0;
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _982_yx = getSolution_yx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__982_yx(_982_yx ? _982_yx->raw() : nullptr,_982_yx ? _982_yx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _9635_;
      bool _9636_ = (barvar_num > (int)0);
      if (_9636_)
      {
        _9635_ = getSolution_bars(::mosek::fusion::SolutionType::Default);
      }
      else
      {
        _9635_ = nullptr;
      }
      std::shared_ptr< monty::ndarray< double,1 > > _983_bars = _9635_;
      _checked_ptr_1<double> _ptr__983_bars(_983_bars ? _983_bars->raw() : nullptr,_983_bars ? _983_bars->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _984_accy = getSolution_accy(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__984_accy(_984_accy ? _984_accy->raw() : nullptr,_984_accy ? _984_accy->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _985_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__985_accptr(_985_accptr ? _985_accptr->raw() : nullptr,_985_accptr ? _985_accptr->size(0) : 0);
      int32_t _9637_ = (int)0;
      int32_t _9638_ = _977_n;
      for (int32_t _986_p = _9637_; _986_p < _9638_; ++_986_p)
      {
        {
          if ((_ptr__974_nativeidxs[_986_p] < (int)0))
          {
            {
              int64_t _987_idx = (-safe_add( _ptr__974_nativeidxs[_986_p],(int)1 ));
              _ptr__975_res[safe_add( _976_offset,_986_p )] = _ptr__983_bars[_987_idx];
            }
          }
          else if((_ptr__974_nativeidxs[_986_p] == (int)0))
          {
            {
              _ptr__975_res[safe_add( _976_offset,_986_p )] = (int)0;
            }
          }
          else if((((*var_elm_acc_idx)(_ptr__974_nativeidxs[_986_p])) >= (int)0))
          {
            {
              _ptr__975_res[safe_add( _976_offset,_986_p )] = _ptr__984_accy[safe_add( _ptr__985_accptr[((*var_elm_acc_idx)(_ptr__974_nativeidxs[_986_p]))],((*var_elm_acc_ofs)(_ptr__974_nativeidxs[_986_p])) )];
            }
          }
          else
          {
            {
              _ptr__975_res[safe_add( _976_offset,_986_p )] = _ptr__982_yx[_ptr__974_nativeidxs[_986_p]];
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getVariableValues

// Begin mosek.fusion.Model.setVariableValues
// Method mosek.fusion.Model.setVariableValues @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4144:6-4164:7
void mosek::fusion::Model :: setVariableValues(bool _988_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _989_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _990_values) { mosek::fusion::p_Model::_get_impl(this)->setVariableValues(_988_primal,_989_nativeidxs,_990_values); }
void mosek::fusion::p_Model::setVariableValues(bool _988_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _989_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _990_values)
{
  _checked_ptr_1<int64_t> _ptr__989_nativeidxs(_989_nativeidxs ? _989_nativeidxs->raw() : nullptr, _989_nativeidxs ? _989_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__990_values(_990_values ? _990_values->raw() : nullptr, _990_values ? _990_values->size(0) : 0);
  if (_988_primal)
  {
    {
      int32_t _991_numnz = (int)0;
      int32_t _9639_ = (int)0;
      int32_t _9640_ = (int)((_989_nativeidxs)->size(0));
      for (int32_t _992_i = _9639_; _992_i < _9640_; ++_992_i)
      {
        {
          if ((_ptr__989_nativeidxs[_992_i] >= (int)0))
          {
            {
              ++ _991_numnz;
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _993_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_991_numnz)));
      _checked_ptr_1<int32_t> _ptr__993_subj(_993_subj ? _993_subj->raw() : nullptr,_993_subj ? _993_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _994_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_991_numnz)));
      _checked_ptr_1<double> _ptr__994_val(_994_val ? _994_val->raw() : nullptr,_994_val ? _994_val->size(0) : 0);
      int32_t _995_k = (int)0;
      int32_t _9641_ = (int)0;
      int32_t _9642_ = (int)((_989_nativeidxs)->size(0));
      for (int32_t _996_i = _9641_; _996_i < _9642_; ++_996_i)
      {
        {
          if ((_ptr__989_nativeidxs[_996_i] >= (int)0))
          {
            {
              _ptr__993_subj[_995_k] = (int32_t)_ptr__989_nativeidxs[_996_i];
              _ptr__994_val[_995_k] = _ptr__990_values[_996_i];
              ++ _995_k;
            }
          }
          {}
        }
      }
      setSolution_xx(_993_subj,_994_val);
    }
  }
  else
  {
    {
    }
  }
}
// End mosek.fusion.Model.setVariableValues

// Begin mosek.fusion.Model.writeTaskNoFlush
// Method mosek.fusion.Model.writeTaskNoFlush @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4133:6-4136:7
void mosek::fusion::Model :: writeTaskNoFlush(const std::string & _997_filename) { mosek::fusion::p_Model::_get_impl(this)->writeTaskNoFlush(_997_filename); }
void mosek::fusion::p_Model::writeTaskNoFlush(const std::string & _997_filename)
{
  task_write(_997_filename);
}
// End mosek.fusion.Model.writeTaskNoFlush

// Begin mosek.fusion.Model.writeTaskStream
// Method mosek.fusion.Model.writeTaskStream @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4126:6-4130:7
void mosek::fusion::Model :: writeTaskStream(const std::string & _998_ext,std::ostream& _999_stream) { mosek::fusion::p_Model::_get_impl(this)->writeTaskStream(_998_ext,_999_stream); }
void mosek::fusion::p_Model::writeTaskStream(const std::string & _998_ext,std::ostream& _999_stream)
{
  flush_parameters();
  task_write_stream(_998_ext,_999_stream);
}
// End mosek.fusion.Model.writeTaskStream

// Begin mosek.fusion.Model.expressionCache
// Method mosek.fusion.Model.expressionCache @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4093:6-4096:7
void mosek::fusion::Model :: expressionCache(bool _1000_cache) { mosek::fusion::p_Model::_get_impl(this)->expressionCache(_1000_cache); }
void mosek::fusion::p_Model::expressionCache(bool _1000_cache)
{
  cache_evaluations = _1000_cache;
}
// End mosek.fusion.Model.expressionCache

// Begin mosek.fusion.Model.dataReport
// Method mosek.fusion.Model.dataReport @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4091:6-58
void mosek::fusion::Model :: dataReport() { mosek::fusion::p_Model::_get_impl(this)->dataReport(); }
void mosek::fusion::p_Model::dataReport()
{
  _pubthis->dataReport((int)10);
}
// End mosek.fusion.Model.dataReport

// Begin mosek.fusion.Model.dataReport
// Method mosek.fusion.Model.dataReport @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4086:6-4090:7
void mosek::fusion::Model :: dataReport(int32_t _1001_detail) { mosek::fusion::p_Model::_get_impl(this)->dataReport(_1001_detail); }
void mosek::fusion::p_Model::dataReport(int32_t _1001_detail)
{
  flush_parameters();
  task_analyze_problem(_1001_detail);
}
// End mosek.fusion.Model.dataReport

// Begin mosek.fusion.Model.writeTask
// Method mosek.fusion.Model.writeTask @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4080:6-4084:7
void mosek::fusion::Model :: writeTask(const std::string & _1002_filename) { mosek::fusion::p_Model::_get_impl(this)->writeTask(_1002_filename); }
void mosek::fusion::p_Model::writeTask(const std::string & _1002_filename)
{
  flush_parameters();
  task_write(_1002_filename);
}
// End mosek.fusion.Model.writeTask

// Begin mosek.fusion.Model.getSolverLIntInfo
// Method mosek.fusion.Model.getSolverLIntInfo @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4072:6-93
int64_t mosek::fusion::Model :: getSolverLIntInfo(const std::string & _1003_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverLIntInfo(_1003_name); }
int64_t mosek::fusion::p_Model::getSolverLIntInfo(const std::string & _1003_name)
{
  return task_get_liinf(_1003_name);
}
// End mosek.fusion.Model.getSolverLIntInfo

// Begin mosek.fusion.Model.getSolverIntInfo
// Method mosek.fusion.Model.getSolverIntInfo @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4062:6-92
int32_t mosek::fusion::Model :: getSolverIntInfo(const std::string & _1004_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverIntInfo(_1004_name); }
int32_t mosek::fusion::p_Model::getSolverIntInfo(const std::string & _1004_name)
{
  return task_get_iinf(_1004_name);
}
// End mosek.fusion.Model.getSolverIntInfo

// Begin mosek.fusion.Model.getSolverDoubleInfo
// Method mosek.fusion.Model.getSolverDoubleInfo @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4052:6-92
double mosek::fusion::Model :: getSolverDoubleInfo(const std::string & _1005_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverDoubleInfo(_1005_name); }
double mosek::fusion::p_Model::getSolverDoubleInfo(const std::string & _1005_name)
{
  return task_get_dinf(_1005_name);
}
// End mosek.fusion.Model.getSolverDoubleInfo

// Begin mosek.fusion.Model.setCallbackHandler
// Method mosek.fusion.Model.setCallbackHandler @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4042:6-110
void mosek::fusion::Model :: setCallbackHandler(mosek::cbhandler_t _1006_h) { mosek::fusion::p_Model::_get_impl(this)->setCallbackHandler(_1006_h); }
void mosek::fusion::p_Model::setCallbackHandler(mosek::cbhandler_t _1006_h)
{
  task_setCallbackHandler(_1006_h);
}
// End mosek.fusion.Model.setCallbackHandler

// Begin mosek.fusion.Model.setDataCallbackHandler
// Method mosek.fusion.Model.setDataCallbackHandler @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:4021:6-114
void mosek::fusion::Model :: setDataCallbackHandler(mosek::datacbhandler_t _1007_h) { mosek::fusion::p_Model::_get_impl(this)->setDataCallbackHandler(_1007_h); }
void mosek::fusion::p_Model::setDataCallbackHandler(mosek::datacbhandler_t _1007_h)
{
  task_setDataCallbackHandler(_1007_h);
}
// End mosek.fusion.Model.setDataCallbackHandler

// Begin mosek.fusion.Model.setLogHandler
// Method mosek.fusion.Model.setLogHandler @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3999:6-89
void mosek::fusion::Model :: setLogHandler(mosek::msghandler_t _1008_h) { mosek::fusion::p_Model::_get_impl(this)->setLogHandler(_1008_h); }
void mosek::fusion::p_Model::setLogHandler(mosek::msghandler_t _1008_h)
{
  task_setLogHandler(_1008_h);
}
// End mosek.fusion.Model.setLogHandler

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3972:6-132
void mosek::fusion::Model :: setSolverParam(const std::string & _1009_name,double _1010_floatval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1009_name,_1010_floatval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1009_name,double _1010_floatval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1009_name,_1010_floatval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3960:6-128
void mosek::fusion::Model :: setSolverParam(const std::string & _1011_name,int32_t _1012_intval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1011_name,_1012_intval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1011_name,int32_t _1012_intval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1011_name,_1012_intval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3948:6-128
void mosek::fusion::Model :: setSolverParam(const std::string & _1013_name,const std::string & _1014_strval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1013_name,_1014_strval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1013_name,const std::string & _1014_strval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1013_name,_1014_strval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.breakSolver
// Method mosek.fusion.Model.breakSolver @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3932:6-3935:7
void mosek::fusion::Model :: breakSolver() { mosek::fusion::p_Model::_get_impl(this)->breakSolver(); }
void mosek::fusion::p_Model::breakSolver()
{
  task_break_solve();
}
// End mosek.fusion.Model.breakSolver

// Begin mosek.fusion.Model.optserverHost
// Method mosek.fusion.Model.optserverHost @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3918:6-3921:7
void mosek::fusion::Model :: optserverHost(const std::string & _1015_addr) { mosek::fusion::p_Model::_get_impl(this)->optserverHost(_1015_addr); }
void mosek::fusion::p_Model::optserverHost(const std::string & _1015_addr)
{
  task_putoptserver_host(_1015_addr);
}
// End mosek.fusion.Model.optserverHost

// Begin mosek.fusion.Model.report_solution
// Method mosek.fusion.Model.report_solution @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3837:6-3913:7
void mosek::fusion::p_Model::report_solution(mosek::fusion::SolutionType _1016_soltype,mosek::fusion::ProblemStatus _1017_prosta,mosek::fusion::SolutionStatus _1018_psolsta,mosek::fusion::SolutionStatus _1019_dsolsta,double _1020_pobj,double _1021_dobj,int32_t _1022_numvar,int32_t _1023_numcon,int32_t _1024_numbarelm,int32_t _1025_numacc,int32_t _1026_numaccelm,bool _1027_hasprimal,bool _1028_hasdual)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1029_sol;
  if ((_1016_soltype == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1029_sol = sol_itr;
    }
  }
  else if((_1016_soltype == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1029_sol = sol_bas;
    }
  }
  else if((_1016_soltype == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1029_sol = sol_itg;
    }
  }
  else
  {
    {
      _1029_sol = nullptr;
    }
  }
  if ((NULL == _1029_sol.get()))
  {
    {
      _1029_sol = ::mosek::fusion::p_SolutionStruct::_new_SolutionStruct(_1022_numvar,_1023_numcon,_1024_numbarelm,_1025_numacc,_1026_numaccelm);
    }
  }
  else
  {
    {
      _1029_sol->resize(_1022_numvar,_1023_numcon,_1024_numbarelm,_1025_numacc,_1026_numaccelm);
    }
  }
  mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->pobj = _1020_pobj;
  mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->dobj = _1021_dobj;
  mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->pstatus = _1018_psolsta;
  mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->dstatus = _1019_dsolsta;
  mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->probstatus = _1017_prosta;
  if ((_1022_numvar > (int)0))
  {
    {
      if (_1027_hasprimal)
      {
        {
          report_solution_get_xx(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->xx);
        }
      }
      {}
      if (_1028_hasdual)
      {
        {
          report_solution_get_slx(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->slx);
          report_solution_get_sux(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->sux);
          std::shared_ptr< monty::ndarray< double,1 > > _1030_yx = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->yx;
          _checked_ptr_1<double> _ptr__1030_yx(_1030_yx ? _1030_yx->raw() : nullptr,_1030_yx ? _1030_yx->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1031_slx = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->slx;
          _checked_ptr_1<double> _ptr__1031_slx(_1031_slx ? _1031_slx->raw() : nullptr,_1031_slx ? _1031_slx->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1032_sux = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->sux;
          _checked_ptr_1<double> _ptr__1032_sux(_1032_sux ? _1032_sux->raw() : nullptr,_1032_sux ? _1032_sux->size(0) : 0);
          int32_t _9643_ = (int)0;
          int32_t _9644_ = _1022_numvar;
          for (int32_t _1033_i = _9643_; _1033_i < _9644_; ++_1033_i)
          {
            {
              _ptr__1030_yx[_1033_i] = (_ptr__1031_slx[_1033_i] - _ptr__1032_sux[_1033_i]);
            }
          }
        }
      }
      {}
    }
  }
  {}
  if ((_1023_numcon > (int)0))
  {
    {
      if (_1027_hasprimal)
      {
        {
          report_solution_get_xc(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->xc);
        }
      }
      {}
      if (_1028_hasdual)
      {
        {
          report_solution_get_slc(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->slc);
          report_solution_get_suc(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->suc);
          std::shared_ptr< monty::ndarray< double,1 > > _1034_y = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->y;
          _checked_ptr_1<double> _ptr__1034_y(_1034_y ? _1034_y->raw() : nullptr,_1034_y ? _1034_y->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1035_slc = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->slc;
          _checked_ptr_1<double> _ptr__1035_slc(_1035_slc ? _1035_slc->raw() : nullptr,_1035_slc ? _1035_slc->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1036_suc = mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->suc;
          _checked_ptr_1<double> _ptr__1036_suc(_1036_suc ? _1036_suc->raw() : nullptr,_1036_suc ? _1036_suc->size(0) : 0);
          int32_t _9645_ = (int)0;
          int32_t _9646_ = _1023_numcon;
          for (int32_t _1037_i = _9645_; _1037_i < _9646_; ++_1037_i)
          {
            {
              _ptr__1034_y[_1037_i] = (_ptr__1035_slc[_1037_i] - _ptr__1036_suc[_1037_i]);
            }
          }
        }
      }
      {}
    }
  }
  {}
  if ((_1024_numbarelm > (int)0))
  {
    {
      if (_1027_hasprimal)
      {
        {
          report_solution_get_barx(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->barx);
        }
      }
      {}
      if (_1028_hasdual)
      {
        {
          report_solution_get_bars(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->bars);
        }
      }
      {}
    }
  }
  {}
  if ((task_numacc() > (int)0))
  {
    {
      report_solution_get_accptr(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->accptr);
      if (_1027_hasprimal)
      {
        {
          report_solution_get_accx(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->accx);
        }
      }
      {}
      if (_1028_hasdual)
      {
        {
          report_solution_get_accy(mosek::fusion::p_SolutionStruct::_get_impl(_1029_sol)->accy);
        }
      }
      {}
    }
  }
  {}
  if ((_1016_soltype == ::mosek::fusion::SolutionType::Interior))
  {
    {
      sol_itr = _1029_sol;
    }
  }
  else if((_1016_soltype == ::mosek::fusion::SolutionType::Basic))
  {
    {
      sol_bas = _1029_sol;
    }
  }
  else if((_1016_soltype == ::mosek::fusion::SolutionType::Integer))
  {
    {
      sol_itg = _1029_sol;
    }
  }
  {}
}
// End mosek.fusion.Model.report_solution

// Begin mosek.fusion.Model.clear_solutions
// Method mosek.fusion.Model.clear_solutions @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3831:6-3835:7
void mosek::fusion::p_Model::clear_solutions()
{
  sol_itr = nullptr;
  sol_itg = nullptr;
  sol_bas = nullptr;
}
// End mosek.fusion.Model.clear_solutions

// Begin mosek.fusion.Model.solveBatch
// Method mosek.fusion.Model.solveBatch @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3806:6-3829:7
std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > mosek::fusion::Model::solveBatch
  ( bool _1038_israce,
    double _1039_maxtime,
    int32_t _1040_numthreads,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Model >,1 > > _1041_models) { return mosek::fusion::p_Model::solveBatch(_1038_israce,_1039_maxtime,_1040_numthreads,_1041_models); }
std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > mosek::fusion::p_Model::solveBatch(bool _1038_israce,double _1039_maxtime,int32_t _1040_numthreads,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Model >,1 > > _1041_models)
{
  int32_t _1042_n = (int)((_1041_models)->size(0));
  int32_t _9647_ = (int)0;
  int32_t _9648_ = _1042_n;
  for (int32_t _1043_i = _9647_; _1043_i < _9648_; ++_1043_i)
  {
    {
      if ((NULL == ((*_1041_models)(_1043_i)).get()))
      {
        {
          throw ::mosek::fusion::OptimizeError(std::string ("One of the models is a null object"));
        }
      }
      {}
    }
  }
  int32_t _9649_ = (int)0;
  int32_t _9650_ = _1042_n;
  for (int32_t _1044_i = _9649_; _1044_i < _9650_; ++_1044_i)
  {
    {
      ((*_1041_models)(_1044_i))->flushSolutions();
      mosek::fusion::p_Model::_get_impl(((*_1041_models)(_1044_i)).get())->flush_parameters();
    }
  }
  std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > _1045_wasOK = ::mosek::fusion::p_BaseModel::env_solve_batch(_1038_israce,_1039_maxtime,_1040_numthreads,_1041_models);
  int32_t _9651_ = (int)0;
  int32_t _9652_ = _1042_n;
  for (int32_t _1046_i = _9651_; _1046_i < _9652_; ++_1046_i)
  {
    {
      mosek::fusion::p_Model::_get_impl(((*_1041_models)(_1046_i)).get())->task_post_solve();
    }
  }
  return _1045_wasOK;
}
// End mosek.fusion.Model.solveBatch

// Begin mosek.fusion.Model.solve
// Method mosek.fusion.Model.solve @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3797:6-3804:7
void mosek::fusion::Model :: solve(const std::string & _1047_addr,const std::string & _1048_accesstoken) { mosek::fusion::p_Model::_get_impl(this)->solve(_1047_addr,_1048_accesstoken); }
void mosek::fusion::p_Model::solve(const std::string & _1047_addr,const std::string & _1048_accesstoken)
{
  _pubthis->flushSolutions();
  flush_parameters();
  task_solve(true,_1047_addr,_1048_accesstoken);
  task_post_solve();
}
// End mosek.fusion.Model.solve

// Begin mosek.fusion.Model.solve
// Method mosek.fusion.Model.solve @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3788:6-3795:7
void mosek::fusion::Model :: solve() { mosek::fusion::p_Model::_get_impl(this)->solve(); }
void mosek::fusion::p_Model::solve()
{
  _pubthis->flushSolutions();
  flush_parameters();
  task_solve(false,std::string (""),std::string (""));
  task_post_solve();
}
// End mosek.fusion.Model.solve

// Begin mosek.fusion.Model.flush_parameters
// Method mosek.fusion.Model.flush_parameters @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3709:6-3756:7
void mosek::fusion::p_Model::flush_parameters()
{
  int32_t _1049_num_con_param_nz = con_blocks->num_parameterized();
  int32_t _1050_num_afe_param_nz = afe_blocks->num_parameterized();
  int32_t _1051_num_obj_param_nz = obj_blocks->num_parameterized();
  if ((_1049_num_con_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1052_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1049_num_con_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1052_subi(_1052_subi ? _1052_subi->raw() : nullptr,_1052_subi ? _1052_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1053_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1049_num_con_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1053_subj(_1053_subj ? _1053_subj->raw() : nullptr,_1053_subj ? _1053_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1054_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1049_num_con_param_nz)));
      _checked_ptr_1<double> _ptr__1054_val(_1054_val ? _1054_val->raw() : nullptr,_1054_val ? _1054_val->size(0) : 0);
      con_blocks->evaluate(xs,param_value,_1052_subi,_1053_subj,_1054_val);
      task_putaijlist(_1052_subi,_1053_subj,_1054_val,(int64_t)(_1049_num_con_param_nz));
    }
  }
  {}
  if ((_1050_num_afe_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1055_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1050_num_afe_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1055_subi(_1055_subi ? _1055_subi->raw() : nullptr,_1055_subi ? _1055_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1056_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1050_num_afe_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1056_subj(_1056_subj ? _1056_subj->raw() : nullptr,_1056_subj ? _1056_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1057_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1050_num_afe_param_nz)));
      _checked_ptr_1<double> _ptr__1057_val(_1057_val ? _1057_val->raw() : nullptr,_1057_val ? _1057_val->size(0) : 0);
      afe_blocks->evaluate(xs,param_value,_1055_subi,_1056_subj,_1057_val);
      task_putafefijlist(_1055_subi,_1056_subj,_1057_val);
    }
  }
  {}
  if ((_1051_num_obj_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1058_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1051_num_obj_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1058_subi(_1058_subi ? _1058_subi->raw() : nullptr,_1058_subi ? _1058_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1059_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1051_num_obj_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1059_subj(_1059_subj ? _1059_subj->raw() : nullptr,_1059_subj ? _1059_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1060_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1051_num_obj_param_nz)));
      _checked_ptr_1<double> _ptr__1060_val(_1060_val ? _1060_val->raw() : nullptr,_1060_val ? _1060_val->size(0) : 0);
      obj_blocks->evaluate(xs,param_value,_1058_subi,_1059_subj,_1060_val);
      task_putclist(_1059_subj,_1060_val);
    }
  }
  {}
}
// End mosek.fusion.Model.flush_parameters

// Begin mosek.fusion.Model.flushParameters
// Method mosek.fusion.Model.flushParameters @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3707:6-61
void mosek::fusion::Model :: flushParameters() { mosek::fusion::p_Model::_get_impl(this)->flushParameters(); }
void mosek::fusion::p_Model::flushParameters()
{
  flush_parameters();
}
// End mosek.fusion.Model.flushParameters

// Begin mosek.fusion.Model.evaluate_parameterized
// Method mosek.fusion.Model.evaluate_parameterized @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3638:6-3701:7
void mosek::fusion::p_Model::evaluate_parameterized(monty::rc_ptr< ::mosek::fusion::WorkStack > _1061_xs,int32_t _1062_numrow,std::shared_ptr< monty::ndarray< int32_t,1 > > _1063_rowptrb,std::shared_ptr< monty::ndarray< int32_t,1 > > _1064_rowptre,std::shared_ptr< monty::ndarray< int64_t,1 > > _1065_codenidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _1066_codeptr,std::shared_ptr< monty::ndarray< int32_t,1 > > _1067_codesizes,std::shared_ptr< monty::ndarray< int32_t,1 > > _1068_code,std::shared_ptr< monty::ndarray< double,1 > > _1069_cconst,std::shared_ptr< monty::ndarray< int32_t,1 > > _1070_subj,std::shared_ptr< monty::ndarray< double,1 > > _1071_val)
{
  _checked_ptr_1<int32_t> _ptr__1063_rowptrb(_1063_rowptrb ? _1063_rowptrb->raw() : nullptr, _1063_rowptrb ? _1063_rowptrb->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1064_rowptre(_1064_rowptre ? _1064_rowptre->raw() : nullptr, _1064_rowptre ? _1064_rowptre->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1065_codenidx(_1065_codenidx ? _1065_codenidx->raw() : nullptr, _1065_codenidx ? _1065_codenidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1066_codeptr(_1066_codeptr ? _1066_codeptr->raw() : nullptr, _1066_codeptr ? _1066_codeptr->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1067_codesizes(_1067_codesizes ? _1067_codesizes->raw() : nullptr, _1067_codesizes ? _1067_codesizes->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1068_code(_1068_code ? _1068_code->raw() : nullptr, _1068_code ? _1068_code->size(0) : 0);
  _checked_ptr_1<double> _ptr__1069_cconst(_1069_cconst ? _1069_cconst->raw() : nullptr, _1069_cconst ? _1069_cconst->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1070_subj(_1070_subj ? _1070_subj->raw() : nullptr, _1070_subj ? _1070_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__1071_val(_1071_val ? _1071_val->raw() : nullptr, _1071_val ? _1071_val->size(0) : 0);
  int32_t _1072_stack = _1061_xs->allocf64((int)((_1068_code)->size(0)));
  int32_t _1073_stackp = (int)0;
  std::shared_ptr< monty::ndarray< double,1 > > _1074_xf64 = mosek::fusion::p_WorkStack::_get_impl(_1061_xs)->f64;
  _checked_ptr_1<double> _ptr__1074_xf64(_1074_xf64 ? _1074_xf64->raw() : nullptr,_1074_xf64 ? _1074_xf64->size(0) : 0);
  int32_t _1075_idx = (int)0;
  int32_t _9653_ = (int)0;
  int32_t _9654_ = _1062_numrow;
  for (int32_t _1076_i = _9653_; _1076_i < _9654_; ++_1076_i)
  {
    {
      int32_t _1077_k = _ptr__1066_codeptr[_1076_i];
      int32_t _9655_ = _ptr__1063_rowptrb[_1076_i];
      int32_t _9656_ = _ptr__1064_rowptre[_1076_i];
      for (int32_t _1078_j = _9655_; _1078_j < _9656_; ++_1078_j)
      {
        {
          int32_t _9657_ = (int)0;
          int32_t _9658_ = _ptr__1067_codesizes[_1078_j];
          for (int32_t _1079_l = _9657_; _1079_l < _9658_; ++_1079_l)
          {
            {
              if ((_ptr__1068_code[_1077_k] == 0))
              {
                {
                }
              }
              else
              {
                {
                  if ((_ptr__1068_code[_1077_k] == 1))
                  {
                    {
                      _ptr__1074_xf64[safe_add( _1072_stack,_1073_stackp )] = ((*param_value)((int32_t)_ptr__1069_cconst[_1077_k]));
                      ++ _1073_stackp;
                    }
                  }
                  else
                  {
                    {
                      if ((_ptr__1068_code[_1077_k] == 2))
                      {
                        {
                          _ptr__1074_xf64[safe_add( _1072_stack,_1073_stackp )] = _ptr__1069_cconst[_1077_k];
                          ++ _1073_stackp;
                        }
                      }
                      else
                      {
                        {
                          if ((_ptr__1068_code[_1077_k] == 3))
                          {
                            {
                              _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)2)] += _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)];
                              -- _1073_stackp;
                            }
                          }
                          else
                          {
                            {
                              if ((_ptr__1068_code[_1077_k] == 4))
                              {
                                {
                                  _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)] = (-_ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)]);
                                }
                              }
                              else
                              {
                                {
                                  if ((_ptr__1068_code[_1077_k] == 5))
                                  {
                                    {
                                      _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)2)] *= _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)];
                                      -- _1073_stackp;
                                    }
                                  }
                                  else
                                  {
                                    {
                                      if ((_ptr__1068_code[_1077_k] == 6))
                                      {
                                        {
                                          _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)] = (1.0 / _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)]);
                                        }
                                      }
                                      else
                                      {
                                        {
                                          if ((_ptr__1068_code[_1077_k] == 8))
                                          {
                                            {
                                              double _1080_v = 0.0;
                                              int32_t _9659_ = (int)0;
                                              int32_t _9660_ = (int32_t)_ptr__1069_cconst[_1077_k];
                                              for (int32_t _1081_i = _9659_; _1081_i < _9660_; ++_1081_i)
                                              {
                                                {
                                                  _1080_v += _ptr__1074_xf64[(safe_add( _1072_stack,_1073_stackp ) - (int)1)];
                                                  -- _1073_stackp;
                                                }
                                              }
                                              _ptr__1074_xf64[safe_add( _1072_stack,_1073_stackp )] = _1080_v;
                                              ++ _1073_stackp;
                                            }
                                          }
                                          else
                                          {
                                            {
                                            }
                                          }
                                        }
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
              ++ _1077_k;
            }
          }
          _ptr__1070_subj[_1075_idx] = (int32_t)_ptr__1065_codenidx[_1078_j];
          _ptr__1071_val[_1075_idx] = _ptr__1074_xf64[_1072_stack];
          _1073_stackp = (int)0;
          ++ _1075_idx;
        }
      }
    }
  }
  _1061_xs->popf64((int)((_1068_code)->size(0)));
}
// End mosek.fusion.Model.evaluate_parameterized

// Begin mosek.fusion.Model.flushSolutions
// Method mosek.fusion.Model.flushSolutions @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3631:6-3636:7
void mosek::fusion::Model :: flushSolutions() { mosek::fusion::p_Model::_get_impl(this)->flushSolutions(); }
void mosek::fusion::p_Model::flushSolutions()
{
  flush_initsol(::mosek::fusion::SolutionType::Interior);
  flush_initsol(::mosek::fusion::SolutionType::Integer);
  flush_initsol(::mosek::fusion::SolutionType::Basic);
}
// End mosek.fusion.Model.flushSolutions

// Begin mosek.fusion.Model.flush_initsol
// Method mosek.fusion.Model.flush_initsol @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3605:6-3627:7
void mosek::fusion::p_Model::flush_initsol(mosek::fusion::SolutionType _1082_which)
{
  if ((NULL != initsol_xx.get()))
  {
    {
      int32_t _1083_numvar = task_numvar();
      std::shared_ptr< monty::ndarray< double,1 > > _1084_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1083_numvar)));
      _checked_ptr_1<double> _ptr__1084_xx(_1084_xx ? _1084_xx->raw() : nullptr,_1084_xx ? _1084_xx->size(0) : 0);
      /* Try @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3610:10-3616:11 */
      try
      {
        {
          monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1086_sol = __mosek_2fusion_2Model__get_sol_cache(_1082_which,true);
          if ((NULL != mosek::fusion::p_SolutionStruct::_get_impl(_1086_sol)->xx.get()))
          {
            {
              int32_t _9661_;
              bool _9662_ = ((int)((mosek::fusion::p_SolutionStruct::_get_impl(_1086_sol)->xx)->size(0)) <= _1083_numvar);
              if (_9662_)
              {
                _9661_ = (int)((mosek::fusion::p_SolutionStruct::_get_impl(_1086_sol)->xx)->size(0));
              }
              else
              {
                _9661_ = _1083_numvar;
              }
              ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_1086_sol)->xx,(int)0,_1084_xx,(int)0,_9661_);
            }
          }
          {}
        }
      }
      catch (mosek::fusion::SolutionError _1085_e)
      {
        {
        }
      }
      int32_t _9663_;
      bool _9664_ = ((int)((initsol_xx)->size(0)) <= _1083_numvar);
      if (_9664_)
      {
        _9663_ = (int)((initsol_xx)->size(0));
      }
      else
      {
        _9663_ = _1083_numvar;
      }
      int32_t _1087_n = _9663_;
      int32_t _9665_ = (int)0;
      int32_t _9666_ = _1087_n;
      for (int32_t _1088_i = _9665_; _1088_i < _9666_; ++_1088_i)
      {
        {
          if (((*initsol_xx_flag)(_1088_i)))
          {
            {
              _ptr__1084_xx[_1088_i] = ((*initsol_xx)(_1088_i));
            }
          }
          {}
        }
      }
      _ptr__1084_xx[bfixidx] = 1.0;
      task_putxx_slice(_1082_which,(int)0,_1083_numvar,_1084_xx);
    }
  }
  {}
}
// End mosek.fusion.Model.flush_initsol

// Begin mosek.fusion.Model.getDualSolutionStatus
// Method mosek.fusion.Model.getDualSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3600:6-3603:7
mosek::fusion::SolutionStatus mosek::fusion::Model :: getDualSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getDualSolutionStatus(); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getDualSolutionStatus()
{
  return _pubthis->getDualSolutionStatus(solutionptr);
}
// End mosek.fusion.Model.getDualSolutionStatus

// Begin mosek.fusion.Model.getProblemStatus
// Method mosek.fusion.Model.getProblemStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3594:6-3597:7
mosek::fusion::ProblemStatus mosek::fusion::Model :: getProblemStatus() { return mosek::fusion::p_Model::_get_impl(this)->getProblemStatus(); }
mosek::fusion::ProblemStatus mosek::fusion::p_Model::getProblemStatus()
{
  return _pubthis->getProblemStatus(solutionptr);
}
// End mosek.fusion.Model.getProblemStatus

// Begin mosek.fusion.Model.getPrimalSolutionStatus
// Method mosek.fusion.Model.getPrimalSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3587:6-3590:7
mosek::fusion::SolutionStatus mosek::fusion::Model :: getPrimalSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getPrimalSolutionStatus(); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getPrimalSolutionStatus()
{
  return getPrimalSolutionStatus(solutionptr);
}
// End mosek.fusion.Model.getPrimalSolutionStatus

// Begin mosek.fusion.Model.dualObjValue
// Method mosek.fusion.Model.dualObjValue @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3580:6-3583:7
double mosek::fusion::Model :: dualObjValue() { return mosek::fusion::p_Model::_get_impl(this)->dualObjValue(); }
double mosek::fusion::p_Model::dualObjValue()
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(::mosek::fusion::SolutionType::Default,false))->dobj;
}
// End mosek.fusion.Model.dualObjValue

// Begin mosek.fusion.Model.primalObjValue
// Method mosek.fusion.Model.primalObjValue @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3574:6-3577:7
double mosek::fusion::Model :: primalObjValue() { return mosek::fusion::p_Model::_get_impl(this)->primalObjValue(); }
double mosek::fusion::p_Model::primalObjValue()
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(::mosek::fusion::SolutionType::Default,true))->pobj;
}
// End mosek.fusion.Model.primalObjValue

// Begin mosek.fusion.Model.get_sol_cache
// Method mosek.fusion.Model.get_sol_cache @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3500:6-3571:7
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_Model::__mosek_2fusion_2Model__get_sol_cache(mosek::fusion::SolutionType _1089_which_,bool _1090_primal,bool _1091_nothrow)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1092_res = nullptr;
  mosek::fusion::SolutionType _1093_which = _1089_which_;
  if ((_1093_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      _1093_which = solutionptr;
    }
  }
  {}
  if ((_1093_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      if ((NULL != sol_itg.get()))
      {
        {
          _1092_res = sol_itg;
        }
      }
      else if((NULL != sol_bas.get()))
      {
        {
          _1092_res = sol_bas;
        }
      }
      else
      {
        {
          _1092_res = sol_itr;
        }
      }
    }
  }
  else if((_1093_which == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1092_res = sol_itr;
    }
  }
  else if((_1093_which == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1092_res = sol_bas;
    }
  }
  else if((_1093_which == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1092_res = sol_itg;
    }
  }
  {}
  if ((NULL == _1092_res.get()))
  {
    {
      throw ::mosek::fusion::SolutionError(std::string ("Solution not available"));
    }
  }
  else
  {
    {
      if ((_1091_nothrow || ((_1090_primal && _1092_res->isPrimalAcceptable(acceptable_sol)) || ((!_1090_primal) && _1092_res->isDualAcceptable(acceptable_sol)))))
      {
        {
          return _1092_res;
        }
      }
      else if(((_1092_res == sol_itg) && (!_1090_primal)))
      {
        {
          throw ::mosek::fusion::SolutionError(std::string ("Dual solution does not exists for mixed-integer problems."));
        }
      }
      else
      {
        {
          monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1094_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
          mosek::fusion::SolutionStatus _1095_solsta = mosek::fusion::p_SolutionStruct::_get_impl(_1092_res)->pstatus;
          mosek::fusion::ProblemStatus _1096_probsta = mosek::fusion::p_SolutionStruct::_get_impl(_1092_res)->probstatus;
          if ((!_1090_primal))
          {
            {
              _1095_solsta = mosek::fusion::p_SolutionStruct::_get_impl(_1092_res)->dstatus;
            }
          }
          {}
          _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Solution status is "));
          if ((_1095_solsta == ::mosek::fusion::SolutionStatus::Undefined))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Undefined"));
            }
          }
          else if((_1095_solsta == ::mosek::fusion::SolutionStatus::Unknown))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Unknown"));
            }
          }
          else if((_1095_solsta == ::mosek::fusion::SolutionStatus::Optimal))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Optimal"));
            }
          }
          else if((_1095_solsta == ::mosek::fusion::SolutionStatus::Feasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Feasible"));
            }
          }
          else if((_1095_solsta == ::mosek::fusion::SolutionStatus::Certificate))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Certificate"));
            }
          }
          else if((_1095_solsta == ::mosek::fusion::SolutionStatus::IllposedCert))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("IllposedCert"));
            }
          }
          {}
          _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" but "));
          if ((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Anything))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Anything"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Optimal))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Optimal"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Feasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("at least Feasible"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Certificate))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Certificate"));
            }
          }
          {}
          _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" is expected. Reason: Accessing "));
          if ((_1092_res == sol_itr))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("interior-point solution"));
            }
          }
          else if((_1092_res == sol_bas))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("basic solution"));
            }
          }
          else if((_1092_res == sol_itg))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("integer solution"));
            }
          }
          {}
          _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" whose problem status is "));
          if ((_1096_probsta == ::mosek::fusion::ProblemStatus::Unknown))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Unknown"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::PrimalAndDualFeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalAndDualFeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::PrimalFeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalFeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::DualFeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("DualFeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::PrimalInfeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalInfeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::DualInfeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("DualInfeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::PrimalAndDualInfeasible))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalAndDualInfeasible"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::IllPosed))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("IllPosed"));
            }
          }
          else if((_1096_probsta == ::mosek::fusion::ProblemStatus::PrimalInfeasibleOrUnbounded))
          {
            {
              _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalInfeasibleOrUnbounded"));
            }
          }
          {}
          _1094_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("."));
          throw ::mosek::fusion::SolutionError(_1094_sb->toString());
        }
      }
    }
  }
}
// End mosek.fusion.Model.get_sol_cache

// Begin mosek.fusion.Model.get_sol_cache
// Method mosek.fusion.Model.get_sol_cache @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3493:6-3498:7
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_Model::__mosek_2fusion_2Model__get_sol_cache(mosek::fusion::SolutionType _1097_which_,bool _1098_primal)
{
  return __mosek_2fusion_2Model__get_sol_cache(_1097_which_,_1098_primal,false);
}
// End mosek.fusion.Model.get_sol_cache

// Begin mosek.fusion.Model.setSolution_xx
// Method mosek.fusion.Model.setSolution_xx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3478:6-3486:7
void mosek::fusion::p_Model::setSolution_xx(std::shared_ptr< monty::ndarray< int32_t,1 > > _1099_subj,std::shared_ptr< monty::ndarray< double,1 > > _1100_val)
{
  _checked_ptr_1<int32_t> _ptr__1099_subj(_1099_subj ? _1099_subj->raw() : nullptr, _1099_subj ? _1099_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__1100_val(_1100_val ? _1100_val->raw() : nullptr, _1100_val ? _1100_val->size(0) : 0);
  ensure_initsol_xx();
  int32_t _9667_ = (int)0;
  int32_t _9668_ = (int)((_1099_subj)->size(0));
  for (int32_t _1101_i = _9667_; _1101_i < _9668_; ++_1101_i)
  {
    {
      ((*initsol_xx)(_ptr__1099_subj[_1101_i])) = _ptr__1100_val[_1101_i];
      ((*initsol_xx_flag)(_ptr__1099_subj[_1101_i])) = true;
    }
  }
}
// End mosek.fusion.Model.setSolution_xx

// Begin mosek.fusion.Model.ensure_initsol_xx
// Method mosek.fusion.Model.ensure_initsol_xx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3461:6-3476:7
void mosek::fusion::p_Model::ensure_initsol_xx()
{
  int32_t _1102_natvarmap_num = task_numvar();
  if ((NULL == initsol_xx.get()))
  {
    {
      initsol_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1102_natvarmap_num)));
      std::vector<bool> _9669_;
      for (int32_t _1103_i = (int)0, _9670_ = _1102_natvarmap_num; _1103_i < _9670_ ; ++_1103_i)
      {
        _9669_.push_back(false);
      }
      auto _9671_ = std::shared_ptr< monty::ndarray< bool,1 > >(new monty::ndarray< bool,1 >(monty::shape(_9669_.size()),_9669_.begin(),_9669_.end()));
      initsol_xx_flag = _9671_;
    }
  }
  else if(((int)((initsol_xx)->size(0)) < _1102_natvarmap_num))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _1104_tmp = initsol_xx;
      _checked_ptr_1<double> _ptr__1104_tmp(_1104_tmp ? _1104_tmp->raw() : nullptr,_1104_tmp ? _1104_tmp->size(0) : 0);
      std::shared_ptr< monty::ndarray< bool,1 > > _1105_tmp_flag = initsol_xx_flag;
      initsol_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1102_natvarmap_num)));
      initsol_xx_flag = std::shared_ptr< monty::ndarray< bool,1 > >(new monty::ndarray< bool,1 >(monty::shape(_1102_natvarmap_num)));
      ::mosek::fusion::Utils::Tools::arraycopy(_1104_tmp,(int)0,initsol_xx,(int)0,(int)((_1104_tmp)->size(0)));
      int32_t _9672_ = (int)0;
      int32_t _9673_ = (int)((_1104_tmp)->size(0));
      for (int32_t _1106_i = _9672_; _1106_i < _9673_; ++_1106_i)
      {
        {
          ((*initsol_xx_flag)(_1106_i)) = ((*_1105_tmp_flag)(_1106_i));
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Model.ensure_initsol_xx

// Begin mosek.fusion.Model.getSolution_accptr
// Method mosek.fusion.Model.getSolution_accptr @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3454:6-3458:7
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_Model::getSolution_accptr(mosek::fusion::SolutionType _1107_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1107_which,true,true))->accptr;
}
// End mosek.fusion.Model.getSolution_accptr

// Begin mosek.fusion.Model.getSolution_accy
// Method mosek.fusion.Model.getSolution_accy @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3448:6-3452:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_accy(mosek::fusion::SolutionType _1108_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1108_which,false))->accy;
}
// End mosek.fusion.Model.getSolution_accy

// Begin mosek.fusion.Model.getSolution_accx
// Method mosek.fusion.Model.getSolution_accx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3442:6-3446:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_accx(mosek::fusion::SolutionType _1109_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1109_which,true))->accx;
}
// End mosek.fusion.Model.getSolution_accx

// Begin mosek.fusion.Model.getSolution_bars
// Method mosek.fusion.Model.getSolution_bars @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3435:6-3439:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_bars(mosek::fusion::SolutionType _1110_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1110_which,false))->bars;
}
// End mosek.fusion.Model.getSolution_bars

// Begin mosek.fusion.Model.getSolution_barx
// Method mosek.fusion.Model.getSolution_barx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3429:6-3433:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_barx(mosek::fusion::SolutionType _1111_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1111_which,true))->barx;
}
// End mosek.fusion.Model.getSolution_barx

// Begin mosek.fusion.Model.getSolution_y
// Method mosek.fusion.Model.getSolution_y @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3423:6-3427:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_y(mosek::fusion::SolutionType _1112_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1112_which,false))->y;
}
// End mosek.fusion.Model.getSolution_y

// Begin mosek.fusion.Model.getSolution_xc
// Method mosek.fusion.Model.getSolution_xc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3417:6-3421:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_xc(mosek::fusion::SolutionType _1113_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1113_which,true))->xc;
}
// End mosek.fusion.Model.getSolution_xc

// Begin mosek.fusion.Model.getSolution_suc
// Method mosek.fusion.Model.getSolution_suc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3411:6-3415:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_suc(mosek::fusion::SolutionType _1114_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1114_which,false))->suc;
}
// End mosek.fusion.Model.getSolution_suc

// Begin mosek.fusion.Model.getSolution_slc
// Method mosek.fusion.Model.getSolution_slc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3405:6-3409:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_slc(mosek::fusion::SolutionType _1115_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1115_which,false))->slc;
}
// End mosek.fusion.Model.getSolution_slc

// Begin mosek.fusion.Model.getSolution_sux
// Method mosek.fusion.Model.getSolution_sux @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3399:6-3403:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_sux(mosek::fusion::SolutionType _1116_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1116_which,false))->sux;
}
// End mosek.fusion.Model.getSolution_sux

// Begin mosek.fusion.Model.getSolution_slx
// Method mosek.fusion.Model.getSolution_slx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3393:6-3397:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_slx(mosek::fusion::SolutionType _1117_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1117_which,false))->slx;
}
// End mosek.fusion.Model.getSolution_slx

// Begin mosek.fusion.Model.getSolution_yx
// Method mosek.fusion.Model.getSolution_yx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3387:6-3391:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_yx(mosek::fusion::SolutionType _1118_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1118_which,false))->yx;
}
// End mosek.fusion.Model.getSolution_yx

// Begin mosek.fusion.Model.getSolution_xx
// Method mosek.fusion.Model.getSolution_xx @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3382:6-3386:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_xx(mosek::fusion::SolutionType _1119_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1119_which,true))->xx;
}
// End mosek.fusion.Model.getSolution_xx

// Begin mosek.fusion.Model.selectedSolution
// Method mosek.fusion.Model.selectedSolution @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3377:6-3380:7
void mosek::fusion::Model :: selectedSolution(mosek::fusion::SolutionType _1120_soltype) { mosek::fusion::p_Model::_get_impl(this)->selectedSolution(_1120_soltype); }
void mosek::fusion::p_Model::selectedSolution(mosek::fusion::SolutionType _1120_soltype)
{
  solutionptr = _1120_soltype;
}
// End mosek.fusion.Model.selectedSolution

// Begin mosek.fusion.Model.getAcceptedSolutionStatus
// Method mosek.fusion.Model.getAcceptedSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3364:6-3367:7
mosek::fusion::AccSolutionStatus mosek::fusion::Model :: getAcceptedSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getAcceptedSolutionStatus(); }
mosek::fusion::AccSolutionStatus mosek::fusion::p_Model::getAcceptedSolutionStatus()
{
  return acceptable_sol;
}
// End mosek.fusion.Model.getAcceptedSolutionStatus

// Begin mosek.fusion.Model.acceptedSolutionStatus
// Method mosek.fusion.Model.acceptedSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3358:6-3361:7
void mosek::fusion::Model :: acceptedSolutionStatus(mosek::fusion::AccSolutionStatus _1121_what) { mosek::fusion::p_Model::_get_impl(this)->acceptedSolutionStatus(_1121_what); }
void mosek::fusion::p_Model::acceptedSolutionStatus(mosek::fusion::AccSolutionStatus _1121_what)
{
  acceptable_sol = _1121_what;
}
// End mosek.fusion.Model.acceptedSolutionStatus

// Begin mosek.fusion.Model.getProblemStatus
// Method mosek.fusion.Model.getProblemStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3327:6-3345:7
mosek::fusion::ProblemStatus mosek::fusion::Model :: getProblemStatus(mosek::fusion::SolutionType _1122_which) { return mosek::fusion::p_Model::_get_impl(this)->getProblemStatus(_1122_which); }
mosek::fusion::ProblemStatus mosek::fusion::p_Model::getProblemStatus(mosek::fusion::SolutionType _1122_which)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1123_res = nullptr;
  if ((_1122_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      if ((NULL != sol_itg.get()))
      {
        {
          _1123_res = sol_itg;
        }
      }
      else if((NULL != sol_bas.get()))
      {
        {
          _1123_res = sol_bas;
        }
      }
      else
      {
        {
          _1123_res = sol_itr;
        }
      }
    }
  }
  else if((_1122_which == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1123_res = sol_itr;
    }
  }
  else if((_1122_which == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1123_res = sol_bas;
    }
  }
  else if((_1122_which == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1123_res = sol_itg;
    }
  }
  {}
  if ((NULL == _1123_res.get()))
  {
    {
      return ::mosek::fusion::ProblemStatus::Unknown;
    }
  }
  else
  {
    {
      return mosek::fusion::p_SolutionStruct::_get_impl(_1123_res)->probstatus;
    }
  }
}
// End mosek.fusion.Model.getProblemStatus

// Begin mosek.fusion.Model.getDualSolutionStatus
// Method mosek.fusion.Model.getDualSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3321:6-122
mosek::fusion::SolutionStatus mosek::fusion::Model :: getDualSolutionStatus(mosek::fusion::SolutionType _1124_which) { return mosek::fusion::p_Model::_get_impl(this)->getDualSolutionStatus(_1124_which); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getDualSolutionStatus(mosek::fusion::SolutionType _1124_which)
{
  return getSolutionStatus(_1124_which,false);
}
// End mosek.fusion.Model.getDualSolutionStatus

// Begin mosek.fusion.Model.getPrimalSolutionStatus
// Method mosek.fusion.Model.getPrimalSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3315:6-128
mosek::fusion::SolutionStatus mosek::fusion::Model :: getPrimalSolutionStatus(mosek::fusion::SolutionType _1125_which) { return mosek::fusion::p_Model::_get_impl(this)->getPrimalSolutionStatus(_1125_which); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getPrimalSolutionStatus(mosek::fusion::SolutionType _1125_which)
{
  return getSolutionStatus(_1125_which,true);
}
// End mosek.fusion.Model.getPrimalSolutionStatus

// Begin mosek.fusion.Model.getSolutionStatus
// Method mosek.fusion.Model.getSolutionStatus @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3298:6-3309:7
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getSolutionStatus(mosek::fusion::SolutionType _1126_which,bool _1127_primal)
{
  /* Try @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3299:8-3308:9 */
  try
  {
    {
      monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1129_sol = __mosek_2fusion_2Model__get_sol_cache(_1126_which,_1127_primal,true);
      if (_1127_primal)
      {
        {
          return mosek::fusion::p_SolutionStruct::_get_impl(_1129_sol)->pstatus;
        }
      }
      else
      {
        {
          return mosek::fusion::p_SolutionStruct::_get_impl(_1129_sol)->dstatus;
        }
      }
    }
  }
  catch (mosek::fusion::SolutionError _1128_e)
  {
    {
      return ::mosek::fusion::SolutionStatus::Undefined;
    }
  }
}
// End mosek.fusion.Model.getSolutionStatus

// Begin mosek.fusion.Model.update
// Method mosek.fusion.Model.update @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:3160:6-3294:7
void mosek::fusion::p_Model::update(std::shared_ptr< monty::ndarray< int32_t,1 > > _1130_conidxs,monty::rc_ptr< ::mosek::fusion::Expression > _1131_expr)
{
  _checked_ptr_1<int32_t> _ptr__1130_conidxs(_1130_conidxs ? _1130_conidxs->raw() : nullptr, _1130_conidxs ? _1130_conidxs->size(0) : 0);
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1131_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>())->recursive_eval(ws,rs,xs,cache_evaluations);
  ws->pop_expr();
  int32_t _1132_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1133_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1134_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1135_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1136_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1137_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1138_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1139_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1140_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1141_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1142_code = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1143_codeptr = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1144_cconst = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1145_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1145_wi32(_1145_wi32 ? _1145_wi32->raw() : nullptr,_1145_wi32 ? _1145_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1146_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1146_wi64(_1146_wi64 ? _1146_wi64->raw() : nullptr,_1146_wi64 ? _1146_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1147_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1147_wf64(_1147_wf64 ? _1147_wf64->raw() : nullptr,_1147_wf64 ? _1147_wf64->size(0) : 0);
  int32_t _1148_num_lincon = (int)0;
  int32_t _9674_ = (int)0;
  int32_t _9675_ = _1134_nelem;
  for (int32_t _1149_i = _9674_; _1149_i < _9675_; ++_1149_i)
  {
    {
      if ((_ptr__1130_conidxs[_1149_i] >= (int)0))
      {
        {
          ++ _1148_num_lincon;
        }
      }
      {}
    }
  }
  int32_t _1150_num_concon = (_1134_nelem - _1148_num_lincon);
  if ((_1148_num_lincon == _1134_nelem))
  {
    {
      putarows(_1130_conidxs,ws,_1134_nelem,_1135_nnz,_1138_ptr_base,_1140_nidxs_base,_1141_cof_base);
      if ((_1137_ncodeatom > (int)0))
      {
        {
          con_blocks->replace_row_code(rs,_1130_conidxs,_1138_ptr_base,_1140_nidxs_base,_1143_codeptr,_1142_code,_1144_cconst);
        }
      }
      else
      {
        {
          con_blocks->clear_row_code(_1130_conidxs);
        }
      }
    }
  }
  else if((_1150_num_concon == _1134_nelem))
  {
    {
      std::vector<int32_t> _9676_;
      for (int32_t _1151_i = (int)0, _9677_ = _1134_nelem; _1151_i < _9677_ ; ++_1151_i)
      {
        _9676_.push_back((-safe_add( (int)1,_ptr__1130_conidxs[_1151_i] )));
      }
      auto _9678_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9676_.size()),_9676_.begin(),_9676_.end()));
      putfrows(_9678_,(int)0,ws,_1134_nelem,_1135_nnz,_1138_ptr_base,_1140_nidxs_base,_1141_cof_base);
      if ((_1137_ncodeatom > (int)0))
      {
        {
          std::vector<int32_t> _9679_;
          for (int32_t _1152_i = (int)0, _9680_ = _1134_nelem; _1152_i < _9680_ ; ++_1152_i)
          {
            _9679_.push_back((-safe_add( (int)1,_ptr__1130_conidxs[_1152_i] )));
          }
          auto _9681_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9679_.size()),_9679_.begin(),_9679_.end()));
          afe_blocks->replace_row_code(rs,_9681_,_1138_ptr_base,_1140_nidxs_base,_1143_codeptr,_1142_code,_1144_cconst);
        }
      }
      else
      {
        {
          std::vector<int32_t> _9682_;
          for (int32_t _1153_i = (int)0, _9683_ = _1134_nelem; _1153_i < _9683_ ; ++_1153_i)
          {
            _9682_.push_back((-safe_add( (int)1,_ptr__1130_conidxs[_1153_i] )));
          }
          auto _9684_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9682_.size()),_9682_.begin(),_9682_.end()));
          afe_blocks->clear_row_code(_9684_);
        }
      }
    }
  }
  else
  {
    {
      int32_t _1154_cncodeelm = (int)0;
      int32_t _1155_lncodeelm = (int)0;
      int32_t _1156_cnnz = (int)0;
      int32_t _1157_lnnz = (int)0;
      int32_t _9685_ = (int)0;
      int32_t _9686_ = _1134_nelem;
      for (int32_t _1158_i = _9685_; _1158_i < _9686_; ++_1158_i)
      {
        {
          if ((_ptr__1130_conidxs[_1158_i] >= (int)0))
          {
            {
              _1157_lnnz += (_ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1158_i ),(int)1 )] - _ptr__1145_wi32[safe_add( _1138_ptr_base,_1158_i )]);
            }
          }
          else
          {
            {
              _1156_cnnz += (_ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1158_i ),(int)1 )] - _ptr__1145_wi32[safe_add( _1138_ptr_base,_1158_i )]);
            }
          }
        }
      }
      int32_t _1159_lrowidx = xs->alloci32(_1148_num_lincon);
      int32_t _1160_crowidx = xs->alloci32(_1150_num_concon);
      int32_t _1161_lptr = xs->alloci32(safe_add( _1148_num_lincon,(int)1 ));
      int32_t _1162_cptr = xs->alloci32(safe_add( _1150_num_concon,(int)1 ));
      int32_t _1163_lnidxs = xs->alloci64(_1157_lnnz);
      int32_t _1164_cnidxs = xs->alloci64(_1156_cnnz);
      int32_t _1165_lcof = xs->allocf64(_1157_lnnz);
      int32_t _1166_ccof = xs->allocf64(_1156_cnnz);
      std::shared_ptr< monty::ndarray< double,1 > > _1167_xf64 = mosek::fusion::p_WorkStack::_get_impl(xs)->f64;
      _checked_ptr_1<double> _ptr__1167_xf64(_1167_xf64 ? _1167_xf64->raw() : nullptr,_1167_xf64 ? _1167_xf64->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1168_xi64 = mosek::fusion::p_WorkStack::_get_impl(xs)->i64;
      _checked_ptr_1<int64_t> _ptr__1168_xi64(_1168_xi64 ? _1168_xi64->raw() : nullptr,_1168_xi64 ? _1168_xi64->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1169_xi32 = mosek::fusion::p_WorkStack::_get_impl(xs)->i32;
      _checked_ptr_1<int32_t> _ptr__1169_xi32(_1169_xi32 ? _1169_xi32->raw() : nullptr,_1169_xi32 ? _1169_xi32->size(0) : 0);
      _ptr__1169_xi32[_1161_lptr] = (int)0;
      _ptr__1169_xi32[_1162_cptr] = (int)0;
      int32_t _1170_ci = (int)0;
      int32_t _1171_li = (int)0;
      int32_t _1172_cnzi = (int)0;
      int32_t _1173_lnzi = (int)0;
      int32_t _9687_ = (int)0;
      int32_t _9688_ = _1134_nelem;
      for (int32_t _1174_i = _9687_; _1174_i < _9688_; ++_1174_i)
      {
        {
          int32_t _1175_pb = _ptr__1145_wi32[safe_add( _1138_ptr_base,_1174_i )];
          int32_t _1176_pe = _ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1174_i ),(int)1 )];
          if ((_ptr__1130_conidxs[_1174_i] >= (int)0))
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1146_wi64,safe_add( _1140_nidxs_base,_1175_pb ),_1168_xi64,safe_add( _1163_lnidxs,_1173_lnzi ),(_1176_pe - _1175_pb));
              ::mosek::fusion::Utils::Tools::arraycopy(_1147_wf64,safe_add( _1141_cof_base,_1175_pb ),_1167_xf64,safe_add( _1165_lcof,_1173_lnzi ),(_1176_pe - _1175_pb));
              _1173_lnzi += (_1176_pe - _1175_pb);
              _ptr__1169_xi32[safe_add( _1159_lrowidx,_1171_li )] = _ptr__1130_conidxs[_1174_i];
              _ptr__1169_xi32[safe_add( safe_add( _1161_lptr,_1171_li ),(int)1 )] = _1173_lnzi;
              ++ _1171_li;
            }
          }
          else
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1146_wi64,safe_add( _1140_nidxs_base,_1175_pb ),_1168_xi64,safe_add( _1164_cnidxs,_1172_cnzi ),(_1176_pe - _1175_pb));
              ::mosek::fusion::Utils::Tools::arraycopy(_1147_wf64,safe_add( _1141_cof_base,_1175_pb ),_1167_xf64,safe_add( _1166_ccof,_1172_cnzi ),(_1176_pe - _1175_pb));
              _1172_cnzi += (_1176_pe - _1175_pb);
              _ptr__1169_xi32[safe_add( _1160_crowidx,_1170_ci )] = (-safe_add( (int)1,_ptr__1130_conidxs[_1174_i] ));
              _ptr__1169_xi32[safe_add( safe_add( _1162_cptr,_1170_ci ),(int)1 )] = _1172_cnzi;
              ++ _1170_ci;
            }
          }
        }
      }
      std::vector<int32_t> _9689_;
      for (int32_t _1177_i = (int)0, _9690_ = _1148_num_lincon; _1177_i < _9690_ ; ++_1177_i)
      {
        _9689_.push_back(_ptr__1169_xi32[safe_add( _1159_lrowidx,_1177_i )]);
      }
      auto _9691_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9689_.size()),_9689_.begin(),_9689_.end()));
      putarows(_9691_,xs,_1148_num_lincon,_1157_lnnz,_1161_lptr,_1163_lnidxs,_1165_lcof);
      putfrows(_1169_xi32,_1160_crowidx,xs,_1150_num_concon,_1156_cnnz,_1162_cptr,_1164_cnidxs,_1166_ccof);
      if ((_1137_ncodeatom > (int)0))
      {
        {
          int32_t _9692_ = (int)0;
          int32_t _9693_ = _1134_nelem;
          for (int32_t _1178_i = _9692_; _1178_i < _9693_; ++_1178_i)
          {
            {
              if ((_ptr__1130_conidxs[_1178_i] >= (int)0))
              {
                {
                  _1155_lncodeelm += (_ptr__1145_wi32[safe_add( _1143_codeptr,_ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1178_i ),(int)1 )] )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_ptr__1145_wi32[safe_add( _1138_ptr_base,_1178_i )] )]);
                }
              }
              else
              {
                {
                  _1154_cncodeelm += (_ptr__1145_wi32[safe_add( _1143_codeptr,_ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1178_i ),(int)1 )] )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_ptr__1145_wi32[safe_add( _1138_ptr_base,_1178_i )] )]);
                }
              }
            }
          }
          int32_t _1179_lcodeptr = xs->alloci32(safe_add( _1157_lnnz,(int)1 ));
          int32_t _1180_ccodeptr = xs->alloci32(safe_add( _1156_cnnz,(int)1 ));
          int32_t _1181_lcode = xs->alloci32(_1155_lncodeelm);
          int32_t _1182_ccode = xs->alloci32(_1154_cncodeelm);
          int32_t _1183_lcconst = xs->allocf64(_1155_lncodeelm);
          int32_t _1184_ccconst = xs->allocf64(_1154_cncodeelm);
          _ptr__1169_xi32[_1179_lcodeptr] = (int)0;
          _ptr__1169_xi32[_1180_ccodeptr] = (int)0;
          int32_t _1185_ccodei = (int)0;
          int32_t _1186_lcodei = (int)0;
          int32_t _1187_rnzi = (int)0;
          int32_t _1188_lnzi = (int)0;
          int32_t _9694_ = (int)0;
          int32_t _9695_ = _1134_nelem;
          for (int32_t _1189_i = _9694_; _1189_i < _9695_; ++_1189_i)
          {
            {
              int32_t _1190_pb = _ptr__1145_wi32[safe_add( _1138_ptr_base,_1189_i )];
              int32_t _1191_pe = _ptr__1145_wi32[safe_add( safe_add( _1138_ptr_base,_1189_i ),(int)1 )];
              if ((_ptr__1130_conidxs[_1189_i] >= (int)0))
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1145_wi32,safe_add( _1142_code,_ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )] ),_1169_xi32,safe_add( _1181_lcode,_1186_lcodei ),(_ptr__1145_wi32[safe_add( _1143_codeptr,_1191_pe )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1147_wf64,safe_add( _1144_cconst,_ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )] ),_1167_xf64,safe_add( _1183_lcconst,_1186_lcodei ),(_ptr__1145_wi32[safe_add( _1143_codeptr,_1191_pe )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )]));
                  int32_t _9696_ = _1190_pb;
                  int32_t _9697_ = _1191_pe;
                  for (int32_t _1192_j = _9696_; _1192_j < _9697_; ++_1192_j)
                  {
                    {
                      _1186_lcodei += (_ptr__1145_wi32[safe_add( safe_add( _1143_codeptr,_1192_j ),(int)1 )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1192_j )]);
                      _ptr__1169_xi32[safe_add( safe_add( _1179_lcodeptr,_1188_lnzi ),(int)1 )] = _1186_lcodei;
                      ++ _1188_lnzi;
                    }
                  }
                }
              }
              else
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1145_wi32,safe_add( _1142_code,_ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )] ),_1169_xi32,safe_add( _1182_ccode,_1185_ccodei ),(_ptr__1145_wi32[safe_add( _1143_codeptr,_1191_pe )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1147_wf64,safe_add( _1144_cconst,_ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )] ),_1167_xf64,safe_add( _1184_ccconst,_1185_ccodei ),(_ptr__1145_wi32[safe_add( _1143_codeptr,_1191_pe )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1190_pb )]));
                  int32_t _9698_ = _1190_pb;
                  int32_t _9699_ = _1191_pe;
                  for (int32_t _1193_j = _9698_; _1193_j < _9699_; ++_1193_j)
                  {
                    {
                      _1185_ccodei += (_ptr__1145_wi32[safe_add( safe_add( _1143_codeptr,_1193_j ),(int)1 )] - _ptr__1145_wi32[safe_add( _1143_codeptr,_1193_j )]);
                      _ptr__1169_xi32[safe_add( safe_add( _1180_ccodeptr,_1172_cnzi ),(int)1 )] = _1185_ccodei;
                      ++ _1172_cnzi;
                    }
                  }
                }
              }
            }
          }
          std::vector<int32_t> _9700_;
          for (int32_t _1194_i = (int)0, _9701_ = _1148_num_lincon; _1194_i < _9701_ ; ++_1194_i)
          {
            _9700_.push_back(_ptr__1169_xi32[safe_add( _1159_lrowidx,_1194_i )]);
          }
          auto _9702_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9700_.size()),_9700_.begin(),_9700_.end()));
          con_blocks->replace_row_code(xs,_9702_,_1161_lptr,_1163_lnidxs,_1179_lcodeptr,_1181_lcode,_1183_lcconst);
          std::vector<int32_t> _9703_;
          for (int32_t _1195_i = (int)0, _9704_ = _1150_num_concon; _1195_i < _9704_ ; ++_1195_i)
          {
            _9703_.push_back((int32_t)_ptr__1168_xi64[safe_add( _1160_crowidx,_1195_i )]);
          }
          auto _9705_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9703_.size()),_9703_.begin(),_9703_.end()));
          afe_blocks->replace_row_code(xs,_9705_,_1162_cptr,_1164_cnidxs,_1180_ccodeptr,_1182_ccode,_1184_ccconst);
        }
      }
      else
      {
        {
          std::vector<int32_t> _9706_;
          for (int32_t _1196_i = (int)0, _9707_ = _1148_num_lincon; _1196_i < _9707_ ; ++_1196_i)
          {
            _9706_.push_back(_ptr__1169_xi32[safe_add( _1159_lrowidx,_1196_i )]);
          }
          auto _9708_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9706_.size()),_9706_.begin(),_9706_.end()));
          con_blocks->clear_row_code(_9708_);
          std::vector<int32_t> _9709_;
          for (int32_t _1197_i = (int)0, _9710_ = _1150_num_concon; _1197_i < _9710_ ; ++_1197_i)
          {
            _9709_.push_back((int32_t)_ptr__1168_xi64[safe_add( _1160_crowidx,_1197_i )]);
          }
          auto _9711_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9709_.size()),_9709_.begin(),_9709_.end()));
          afe_blocks->clear_row_code(_9711_);
        }
      }
    }
  }
  xs->clear();
}
// End mosek.fusion.Model.update

// Begin mosek.fusion.Model.update
// Method mosek.fusion.Model.update @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2945:6-3156:7
void mosek::fusion::p_Model::update(std::shared_ptr< monty::ndarray< int32_t,1 > > _1198_conidxs,monty::rc_ptr< ::mosek::fusion::Expression > _1199_expr,std::shared_ptr< monty::ndarray< int32_t,1 > > _1200_varidxs)
{
  _checked_ptr_1<int32_t> _ptr__1198_conidxs(_1198_conidxs ? _1198_conidxs->raw() : nullptr, _1198_conidxs ? _1198_conidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1200_varidxs(_1200_varidxs ? _1200_varidxs->raw() : nullptr, _1200_varidxs ? _1200_varidxs->size(0) : 0);
  int32_t _9712_ = (int)0;
  int32_t _9713_ = (int)((_1198_conidxs)->size(0));
  for (int32_t _1201_i = _9712_; _1201_i < _9713_; ++_1201_i)
  {
    {
      if (((_ptr__1198_conidxs[_1201_i] >= (int)0) && con_blocks->row_is_parameterized(_1201_i)))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
        }
      }
      else if(((_ptr__1198_conidxs[_1201_i] < (int)0) && con_blocks->row_is_parameterized((-safe_add( _1201_i,(int)1 )))))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
        }
      }
      {}
    }
  }
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1199_expr)).as<::mosek::fusion::Expression>())->recursive_eval(ws,rs,xs,cache_evaluations);
  ws->pop_expr();
  int32_t _1202_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1203_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1204_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1205_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1206_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1207_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1208_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1209_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1210_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1211_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1212_code = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1213_codeptr = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1214_cconst = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1215_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1215_wi32(_1215_wi32 ? _1215_wi32->raw() : nullptr,_1215_wi32 ? _1215_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1216_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1216_wi64(_1216_wi64 ? _1216_wi64->raw() : nullptr,_1216_wi64 ? _1216_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1217_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1217_wf64(_1217_wf64 ? _1217_wf64->raw() : nullptr,_1217_wf64 ? _1217_wf64->size(0) : 0);
  int32_t _1218_num_lincon = (int)0;
  int32_t _9714_ = (int)0;
  int32_t _9715_ = _1203_nelem;
  for (int32_t _1219_i = _9714_; _1219_i < _9715_; ++_1219_i)
  {
    {
      if ((_ptr__1198_conidxs[_1219_i] >= (int)0))
      {
        {
          ++ _1218_num_lincon;
        }
      }
      {}
    }
  }
  int32_t _1220_num_concon = (_1203_nelem - _1218_num_lincon);
  if (((_1218_num_lincon == _1203_nelem) || (_1220_num_concon == _1203_nelem)))
  {
    {
      int32_t _1221_N = safe_mul( (int)((_1198_conidxs)->size(0)),(int)((_1200_varidxs)->size(0)) );
      std::shared_ptr< monty::ndarray< double,1 > > _1222_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1221_N)));
      _checked_ptr_1<double> _ptr__1222_cof(_1222_cof ? _1222_cof->raw() : nullptr,_1222_cof ? _1222_cof->size(0) : 0);
      int32_t _1223_nzi = (int)0;
      int32_t _9716_ = (int)0;
      int32_t _9717_ = _1203_nelem;
      for (int32_t _1224_i = _9716_; _1224_i < _9717_; ++_1224_i)
      {
        {
          int32_t _1225_p0 = _ptr__1215_wi32[safe_add( _1208_ptr_base,_1224_i )];
          int32_t _1226_p1 = _ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1224_i ),(int)1 )];
          int32_t _1227_p = _1225_p0;
          int32_t _1228_l = (int)0;
          while ( ((_1228_l < (int)((_1200_varidxs)->size(0))) && (_1227_p < _1226_p1)) )
          {
            {
              if ((_ptr__1200_varidxs[_1228_l] < _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1227_p )]))
              {
                {
                  ++ _1228_l;
                }
              }
              else if((_ptr__1200_varidxs[_1228_l] > _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1227_p )]))
              {
                {
                  ++ _1227_p;
                }
              }
              else
              {
                {
                  _ptr__1222_cof[safe_add( _1223_nzi,_1228_l )] = _ptr__1217_wf64[safe_add( _1211_cof_base,_1227_p )];
                  ++ _1228_l;
                  ++ _1227_p;
                }
              }
            }
          }
          _1223_nzi += (int)((_1200_varidxs)->size(0));
        }
      }
      std::vector<int32_t> _9718_;
      for (int32_t _1230_i = (int)0, _9719_ = _1203_nelem; _1230_i < _9719_ ; ++_1230_i)
      {
        for (int32_t _1231_j = (int)0, _9720_ = (int)((_1200_varidxs)->size(0)); _1231_j < _9720_ ; ++_1231_j)
        {
          _9718_.push_back(_ptr__1200_varidxs[_1231_j]);
        }
      }
      auto _9721_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9718_.size()),_9718_.begin(),_9718_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1229_subj = _9721_;
      _checked_ptr_1<int32_t> _ptr__1229_subj(_1229_subj ? _1229_subj->raw() : nullptr,_1229_subj ? _1229_subj->size(0) : 0);
      if ((_1218_num_lincon == _1203_nelem))
      {
        {
          std::vector<int32_t> _9722_;
          for (int32_t _1233_i = (int)0, _9723_ = _1203_nelem; _1233_i < _9723_ ; ++_1233_i)
          {
            for (int32_t _1234_j = (int)0, _9724_ = (int)((_1200_varidxs)->size(0)); _1234_j < _9724_ ; ++_1234_j)
            {
              _9722_.push_back(_ptr__1198_conidxs[_1233_i]);
            }
          }
          auto _9725_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9722_.size()),_9722_.begin(),_9722_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1232_subi = _9725_;
          _checked_ptr_1<int32_t> _ptr__1232_subi(_1232_subi ? _1232_subi->raw() : nullptr,_1232_subi ? _1232_subi->size(0) : 0);
          task_putaijlist(_1232_subi,_1229_subj,_1222_cof,(int64_t)(_1223_nzi));
          if ((_1206_ncodeatom > (int)0))
          {
            {
              con_blocks->replace_row_code(rs,_1198_conidxs,_1208_ptr_base,_1210_nidxs_base,_1213_codeptr,_1212_code,_1214_cconst);
            }
          }
          {}
        }
      }
      else
      {
        {
          std::vector<int32_t> _9726_;
          for (int32_t _1236_i = (int)0, _9727_ = _1203_nelem; _1236_i < _9727_ ; ++_1236_i)
          {
            for (int32_t _1237_j = (int)0, _9728_ = (int)((_1200_varidxs)->size(0)); _1237_j < _9728_ ; ++_1237_j)
            {
              _9726_.push_back((-safe_add( _ptr__1198_conidxs[_1236_i],(int)1 )));
            }
          }
          auto _9729_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9726_.size()),_9726_.begin(),_9726_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1235_subi = _9729_;
          _checked_ptr_1<int32_t> _ptr__1235_subi(_1235_subi ? _1235_subi->raw() : nullptr,_1235_subi ? _1235_subi->size(0) : 0);
          task_putafefijlist(_1235_subi,_1229_subj,_1222_cof);
          if ((_1206_ncodeatom > (int)0))
          {
            {
              std::vector<int32_t> _9730_;
              for (int32_t _1238_i = (int)0, _9731_ = _1203_nelem; _1238_i < _9731_ ; ++_1238_i)
              {
                _9730_.push_back((-safe_add( (int)1,_ptr__1198_conidxs[_1238_i] )));
              }
              auto _9732_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9730_.size()),_9730_.begin(),_9730_.end()));
              afe_blocks->replace_row_code(rs,_9732_,_1208_ptr_base,_1210_nidxs_base,_1213_codeptr,_1212_code,_1214_cconst);
            }
          }
          {}
        }
      }
    }
  }
  else
  {
    {
      int32_t _1239_cN = safe_mul( _1220_num_concon,(int)((_1200_varidxs)->size(0)) );
      int32_t _1240_lN = safe_mul( _1218_num_lincon,(int)((_1200_varidxs)->size(0)) );
      std::shared_ptr< monty::ndarray< double,1 > > _1241_lcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1240_lN)));
      _checked_ptr_1<double> _ptr__1241_lcof(_1241_lcof ? _1241_lcof->raw() : nullptr,_1241_lcof ? _1241_lcof->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1242_ccof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1239_cN)));
      _checked_ptr_1<double> _ptr__1242_ccof(_1242_ccof ? _1242_ccof->raw() : nullptr,_1242_ccof ? _1242_ccof->size(0) : 0);
      std::vector<int32_t> _9733_;
      for (int32_t _1244_i = (int)0, _9734_ = _1218_num_lincon; _1244_i < _9734_ ; ++_1244_i)
      {
        for (int32_t _1245_j = (int)0, _9735_ = (int)((_1200_varidxs)->size(0)); _1245_j < _9735_ ; ++_1245_j)
        {
          _9733_.push_back(_ptr__1200_varidxs[_1245_j]);
        }
      }
      auto _9736_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9733_.size()),_9733_.begin(),_9733_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1243_lsubj = _9736_;
      _checked_ptr_1<int32_t> _ptr__1243_lsubj(_1243_lsubj ? _1243_lsubj->raw() : nullptr,_1243_lsubj ? _1243_lsubj->size(0) : 0);
      std::vector<int32_t> _9737_;
      for (int32_t _1247_i = (int)0, _9738_ = _1220_num_concon; _1247_i < _9738_ ; ++_1247_i)
      {
        for (int32_t _1248_j = (int)0, _9739_ = (int)((_1200_varidxs)->size(0)); _1248_j < _9739_ ; ++_1248_j)
        {
          _9737_.push_back(_ptr__1200_varidxs[_1248_j]);
        }
      }
      auto _9740_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9737_.size()),_9737_.begin(),_9737_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1246_csubj = _9740_;
      _checked_ptr_1<int32_t> _ptr__1246_csubj(_1246_csubj ? _1246_csubj->raw() : nullptr,_1246_csubj ? _1246_csubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1249_lsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1218_num_lincon)));
      _checked_ptr_1<int32_t> _ptr__1249_lsubi(_1249_lsubi ? _1249_lsubi->raw() : nullptr,_1249_lsubi ? _1249_lsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1250_csubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1220_num_concon)));
      _checked_ptr_1<int32_t> _ptr__1250_csubi(_1250_csubi ? _1250_csubi->raw() : nullptr,_1250_csubi ? _1250_csubi->size(0) : 0);
      int32_t _1251_crowi = (int)0;
      int32_t _1252_lrowi = (int)0;
      int32_t _1253_cncodeelm = (int)0;
      int32_t _1254_lncodeelm = (int)0;
      int32_t _1255_cnnz = (int)0;
      int32_t _1256_lnnz = (int)0;
      int32_t _9741_ = (int)0;
      int32_t _9742_ = _1203_nelem;
      for (int32_t _1257_i = _9741_; _1257_i < _9742_; ++_1257_i)
      {
        {
          int32_t _1258_p0 = _ptr__1215_wi32[safe_add( _1208_ptr_base,_1257_i )];
          int32_t _1259_p1 = _ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1257_i ),(int)1 )];
          int32_t _1260_p = _1258_p0;
          int32_t _1261_l = (int)0;
          if ((_ptr__1198_conidxs[_1257_i] >= (int)0))
          {
            {
              while ( ((_1261_l < (int)((_1200_varidxs)->size(0))) && (_1260_p < _1259_p1)) )
              {
                {
                  if ((_ptr__1200_varidxs[_1261_l] < _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1260_p )]))
                  {
                    {
                      ++ _1261_l;
                    }
                  }
                  else if((_ptr__1200_varidxs[_1261_l] > _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1260_p )]))
                  {
                    {
                      ++ _1260_p;
                    }
                  }
                  else
                  {
                    {
                      _ptr__1241_lcof[safe_add( _1256_lnnz,_1261_l )] = _ptr__1217_wf64[safe_add( _1211_cof_base,_1260_p )];
                      ++ _1261_l;
                      ++ _1260_p;
                    }
                  }
                }
              }
              int32_t _9743_ = (int)0;
              int32_t _9744_ = (int)((_1200_varidxs)->size(0));
              for (int32_t _1262_j = _9743_; _1262_j < _9744_; ++_1262_j)
              {
                {
                  _ptr__1249_lsubi[safe_add( _1256_lnnz,_1262_j )] = _ptr__1198_conidxs[_1257_i];
                }
              }
              _1256_lnnz += (int)((_1200_varidxs)->size(0));
            }
          }
          else
          {
            {
              while ( ((_1261_l < (int)((_1200_varidxs)->size(0))) && (_1260_p < _1259_p1)) )
              {
                {
                  if ((_ptr__1200_varidxs[_1261_l] < _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1260_p )]))
                  {
                    {
                      ++ _1261_l;
                    }
                  }
                  else if((_ptr__1200_varidxs[_1261_l] > _ptr__1216_wi64[safe_add( _1210_nidxs_base,_1260_p )]))
                  {
                    {
                      ++ _1260_p;
                    }
                  }
                  else
                  {
                    {
                      _ptr__1242_ccof[safe_add( _1255_cnnz,_1261_l )] = _ptr__1217_wf64[safe_add( _1211_cof_base,_1260_p )];
                      ++ _1261_l;
                      ++ _1260_p;
                    }
                  }
                }
              }
              int32_t _9745_ = (int)0;
              int32_t _9746_ = (int)((_1200_varidxs)->size(0));
              for (int32_t _1263_j = _9745_; _1263_j < _9746_; ++_1263_j)
              {
                {
                  _ptr__1250_csubi[safe_add( _1255_cnnz,_1263_j )] = (-safe_add( (int)1,_ptr__1198_conidxs[_1257_i] ));
                }
              }
              _1255_cnnz += (int)((_1200_varidxs)->size(0));
            }
          }
        }
      }
      task_putaijlist(_1249_lsubi,_1243_lsubj,_1241_lcof,(int64_t)(_1256_lnnz));
      task_putafefijlist(_1250_csubi,_1246_csubj,_1242_ccof);
      if ((_1206_ncodeatom > (int)0))
      {
        {
          int32_t _9747_ = (int)0;
          int32_t _9748_ = _1203_nelem;
          for (int32_t _1264_i = _9747_; _1264_i < _9748_; ++_1264_i)
          {
            {
              if ((_ptr__1198_conidxs[_1264_i] >= (int)0))
              {
                {
                  _1256_lnnz += (_ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1264_i ),(int)1 )] - _ptr__1215_wi32[safe_add( _1208_ptr_base,_1264_i )]);
                }
              }
              else
              {
                {
                  _1255_cnnz += (_ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1264_i ),(int)1 )] - _ptr__1215_wi32[safe_add( _1208_ptr_base,_1264_i )]);
                }
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1265_lrowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1218_num_lincon)));
          _checked_ptr_1<int32_t> _ptr__1265_lrowidx(_1265_lrowidx ? _1265_lrowidx->raw() : nullptr,_1265_lrowidx ? _1265_lrowidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1266_crowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1220_num_concon)));
          _checked_ptr_1<int32_t> _ptr__1266_crowidx(_1266_crowidx ? _1266_crowidx->raw() : nullptr,_1266_crowidx ? _1266_crowidx->size(0) : 0);
          int32_t _1267_lptr = xs->alloci32(safe_add( _1218_num_lincon,(int)1 ));
          int32_t _1268_cptr = xs->alloci32(safe_add( _1220_num_concon,(int)1 ));
          int32_t _1269_lnidxs = xs->alloci64(_1256_lnnz);
          int32_t _1270_cnidxs = xs->alloci64(_1255_cnnz);
          std::shared_ptr< monty::ndarray< double,1 > > _1271_xf64 = mosek::fusion::p_WorkStack::_get_impl(xs)->f64;
          _checked_ptr_1<double> _ptr__1271_xf64(_1271_xf64 ? _1271_xf64->raw() : nullptr,_1271_xf64 ? _1271_xf64->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _1272_xi64 = mosek::fusion::p_WorkStack::_get_impl(xs)->i64;
          _checked_ptr_1<int64_t> _ptr__1272_xi64(_1272_xi64 ? _1272_xi64->raw() : nullptr,_1272_xi64 ? _1272_xi64->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1273_xi32 = mosek::fusion::p_WorkStack::_get_impl(xs)->i32;
          _checked_ptr_1<int32_t> _ptr__1273_xi32(_1273_xi32 ? _1273_xi32->raw() : nullptr,_1273_xi32 ? _1273_xi32->size(0) : 0);
          _ptr__1273_xi32[_1267_lptr] = (int)0;
          _ptr__1273_xi32[_1268_cptr] = (int)0;
          int32_t _1274_ci = (int)0;
          int32_t _1275_li = (int)0;
          int32_t _1276_cnzi = (int)0;
          int32_t _1277_lnzi = (int)0;
          int32_t _9749_ = (int)0;
          int32_t _9750_ = _1203_nelem;
          for (int32_t _1278_i = _9749_; _1278_i < _9750_; ++_1278_i)
          {
            {
              int32_t _1279_pb = _ptr__1215_wi32[safe_add( _1208_ptr_base,_1278_i )];
              int32_t _1280_pe = _ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1278_i ),(int)1 )];
              if ((_ptr__1198_conidxs[_1278_i] >= (int)0))
              {
                {
                  _1277_lnzi += (_1280_pe - _1279_pb);
                  _ptr__1265_lrowidx[_1275_li] = _ptr__1198_conidxs[_1278_i];
                  _ptr__1273_xi32[safe_add( safe_add( _1267_lptr,_1275_li ),(int)1 )] = _1277_lnzi;
                  ++ _1275_li;
                }
              }
              else
              {
                {
                  _1276_cnzi += (_1280_pe - _1279_pb);
                  _ptr__1266_crowidx[_1274_ci] = (-safe_add( (int)1,_ptr__1198_conidxs[_1278_i] ));
                  _ptr__1273_xi32[safe_add( safe_add( _1268_cptr,_1274_ci ),(int)1 )] = _1276_cnzi;
                  ++ _1274_ci;
                }
              }
            }
          }
          int32_t _9751_ = (int)0;
          int32_t _9752_ = _1203_nelem;
          for (int32_t _1281_i = _9751_; _1281_i < _9752_; ++_1281_i)
          {
            {
              if ((_ptr__1198_conidxs[_1281_i] >= (int)0))
              {
                {
                  _1254_lncodeelm += (_ptr__1215_wi32[safe_add( _1213_codeptr,_ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1281_i ),(int)1 )] )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_ptr__1215_wi32[safe_add( _1208_ptr_base,_1281_i )] )]);
                }
              }
              else
              {
                {
                  _1253_cncodeelm += (_ptr__1215_wi32[safe_add( _1213_codeptr,_ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1281_i ),(int)1 )] )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_ptr__1215_wi32[safe_add( _1208_ptr_base,_1281_i )] )]);
                }
              }
            }
          }
          int32_t _1282_lcodeptr = xs->alloci32(safe_add( _1256_lnnz,(int)1 ));
          int32_t _1283_ccodeptr = xs->alloci32(safe_add( _1255_cnnz,(int)1 ));
          int32_t _1284_lcode = xs->alloci32(_1254_lncodeelm);
          int32_t _1285_ccode = xs->alloci32(_1253_cncodeelm);
          int32_t _1286_lcconst = xs->allocf64(_1254_lncodeelm);
          int32_t _1287_ccconst = xs->allocf64(_1253_cncodeelm);
          _ptr__1273_xi32[_1282_lcodeptr] = (int)0;
          _ptr__1273_xi32[_1283_ccodeptr] = (int)0;
          int32_t _1288_ccodei = (int)0;
          int32_t _1289_lcodei = (int)0;
          int32_t _1290_rnzi = (int)0;
          int32_t _1291_lnzi = (int)0;
          int32_t _9753_ = (int)0;
          int32_t _9754_ = _1203_nelem;
          for (int32_t _1292_i = _9753_; _1292_i < _9754_; ++_1292_i)
          {
            {
              int32_t _1293_pb = _ptr__1215_wi32[safe_add( _1208_ptr_base,_1292_i )];
              int32_t _1294_pe = _ptr__1215_wi32[safe_add( safe_add( _1208_ptr_base,_1292_i ),(int)1 )];
              if ((_ptr__1198_conidxs[_1292_i] >= (int)0))
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1215_wi32,safe_add( _1212_code,_ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )] ),_1273_xi32,safe_add( _1284_lcode,_1289_lcodei ),(_ptr__1215_wi32[safe_add( _1213_codeptr,_1294_pe )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1217_wf64,safe_add( _1214_cconst,_ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )] ),_1271_xf64,safe_add( _1286_lcconst,_1289_lcodei ),(_ptr__1215_wi32[safe_add( _1213_codeptr,_1294_pe )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )]));
                  int32_t _9755_ = _1293_pb;
                  int32_t _9756_ = _1294_pe;
                  for (int32_t _1295_j = _9755_; _1295_j < _9756_; ++_1295_j)
                  {
                    {
                      _1289_lcodei += (_ptr__1215_wi32[safe_add( safe_add( _1213_codeptr,_1295_j ),(int)1 )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1295_j )]);
                      _ptr__1273_xi32[safe_add( safe_add( _1282_lcodeptr,_1291_lnzi ),(int)1 )] = _1289_lcodei;
                      ++ _1291_lnzi;
                    }
                  }
                }
              }
              else
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1215_wi32,safe_add( _1212_code,_ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )] ),_1273_xi32,safe_add( _1285_ccode,_1288_ccodei ),(_ptr__1215_wi32[safe_add( _1213_codeptr,_1294_pe )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1217_wf64,safe_add( _1214_cconst,_ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )] ),_1271_xf64,safe_add( _1287_ccconst,_1288_ccodei ),(_ptr__1215_wi32[safe_add( _1213_codeptr,_1294_pe )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1293_pb )]));
                  int32_t _9757_ = _1293_pb;
                  int32_t _9758_ = _1294_pe;
                  for (int32_t _1296_j = _9757_; _1296_j < _9758_; ++_1296_j)
                  {
                    {
                      _1288_ccodei += (_ptr__1215_wi32[safe_add( safe_add( _1213_codeptr,_1296_j ),(int)1 )] - _ptr__1215_wi32[safe_add( _1213_codeptr,_1296_j )]);
                      _ptr__1273_xi32[safe_add( safe_add( _1283_ccodeptr,_1276_cnzi ),(int)1 )] = _1288_ccodei;
                      ++ _1276_cnzi;
                    }
                  }
                }
              }
            }
          }
          con_blocks->replace_row_code(xs,_1265_lrowidx,_1267_lptr,_1269_lnidxs,_1282_lcodeptr,_1284_lcode,_1286_lcconst);
          afe_blocks->replace_row_code(xs,_1266_crowidx,_1268_cptr,_1270_cnidxs,_1283_ccodeptr,_1285_ccode,_1287_ccconst);
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1297_lrowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1218_num_lincon)));
          _checked_ptr_1<int32_t> _ptr__1297_lrowidx(_1297_lrowidx ? _1297_lrowidx->raw() : nullptr,_1297_lrowidx ? _1297_lrowidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1298_crowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1220_num_concon)));
          _checked_ptr_1<int32_t> _ptr__1298_crowidx(_1298_crowidx ? _1298_crowidx->raw() : nullptr,_1298_crowidx ? _1298_crowidx->size(0) : 0);
          int32_t _1299_ci = (int)0;
          int32_t _1300_li = (int)0;
          int32_t _9759_ = (int)0;
          int32_t _9760_ = _1203_nelem;
          for (int32_t _1301_i = _9759_; _1301_i < _9760_; ++_1301_i)
          {
            {
              if ((_ptr__1198_conidxs[_1301_i] >= (int)0))
              {
                {
                  _ptr__1297_lrowidx[_1300_li] = _ptr__1198_conidxs[_1301_i];
                  ++ _1300_li;
                }
              }
              else
              {
                {
                  _ptr__1298_crowidx[_1299_ci] = (-safe_add( (int)1,_ptr__1198_conidxs[_1301_i] ));
                  ++ _1299_ci;
                }
              }
            }
          }
          con_blocks->clear_row_code(_1297_lrowidx);
          afe_blocks->clear_row_code(_1298_crowidx);
        }
      }
    }
  }
  xs->clear();
}
// End mosek.fusion.Model.update

// Begin mosek.fusion.Model.updateObjective
// Method mosek.fusion.Model.updateObjective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2857:6-2940:7
void mosek::fusion::Model :: updateObjective(monty::rc_ptr< ::mosek::fusion::Expression > _1302_expr,monty::rc_ptr< ::mosek::fusion::Variable > _1303_x) { mosek::fusion::p_Model::_get_impl(this)->updateObjective(_1302_expr,_1303_x); }
void mosek::fusion::p_Model::updateObjective(monty::rc_ptr< ::mosek::fusion::Expression > _1302_expr,monty::rc_ptr< ::mosek::fusion::Variable > _1303_x)
{
  if ((_1302_expr->getSize() != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid expression size for objective"));
    }
  }
  {}
  if ((mosek::fusion::p_RowBlockManager::_get_impl(obj_blocks)->code_used > (int)0))
  {
    {
      throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
    }
  }
  {}
  int32_t _1304_ni = _1303_x->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1305_varidxs_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1304_ni)));
  _checked_ptr_1<int64_t> _ptr__1305_varidxs_(_1305_varidxs_ ? _1305_varidxs_->raw() : nullptr,_1305_varidxs_ ? _1305_varidxs_->size(0) : 0);
  if ((_1304_ni < _1303_x->getSize()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1306_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1304_ni)));
      _checked_ptr_1<int64_t> _ptr__1306_sp(_1306_sp ? _1306_sp->raw() : nullptr,_1306_sp ? _1306_sp->size(0) : 0);
      _1303_x->inst((int)0,_1306_sp,(int)0,_1305_varidxs_);
    }
  }
  else
  {
    {
      _1303_x->inst((int)0,_1305_varidxs_);
    }
  }
  int32_t _9761_ = (int)0;
  int32_t _9762_ = _1304_ni;
  for (int32_t _1307_i = _9761_; _1307_i < _9762_; ++_1307_i)
  {
    {
      if ((_ptr__1305_varidxs_[_1307_i] < (int)0))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Updating semidefinite terms is currently not possible"));
        }
      }
      {}
    }
  }
  bool _1308_varidxs_issorted = true;
  int32_t _9763_ = (int)0;
  int32_t _9764_ = (_1304_ni - (int)1);
  for (int32_t _1309_i = _9763_; _1309_i < _9764_; ++_1309_i)
  {
    {
      _1308_varidxs_issorted = (_1308_varidxs_issorted && (_ptr__1305_varidxs_[_1309_i] < _ptr__1305_varidxs_[(_1309_i - (int)1)]));
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1310_varidxs = nullptr;
  _checked_ptr_1<int32_t> _ptr__1310_varidxs(_1310_varidxs ? _1310_varidxs->raw() : nullptr,_1310_varidxs ? _1310_varidxs->size(0) : 0);
  if (_1308_varidxs_issorted)
  {
    {
      std::vector<int32_t> _9765_;
      for (int32_t _1311_i = (int)0, _9766_ = _1304_ni; _1311_i < _9766_ ; ++_1311_i)
      {
        _9765_.push_back((int32_t)_ptr__1305_varidxs_[_1311_i]);
      }
      auto _9767_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9765_.size()),_9765_.begin(),_9765_.end()));
      _1310_varidxs = _9767_;
      _ptr__1310_varidxs.update(_1310_varidxs ? _1310_varidxs->raw() : nullptr, _1310_varidxs ? _1310_varidxs->size(0) : 0);
    }
  }
  else
  {
    {
      std::vector<int64_t> _9768_;
      for (int32_t _1313_i = (-(int)1), _9769_ = _1304_ni; _1313_i < _9769_ ; ++_1313_i)
      {
        _9768_.push_back(_1313_i);
      }
      auto _9770_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9768_.size()),_9768_.begin(),_9768_.end()));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1312_perm = _9770_;
      _checked_ptr_1<int64_t> _ptr__1312_perm(_1312_perm ? _1312_perm->raw() : nullptr,_1312_perm ? _1312_perm->size(0) : 0);
      ::mosek::fusion::p_CommonTools::argQsort(_1312_perm,_1305_varidxs_,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_1304_ni));
      std::vector<int32_t> _9771_;
      for (int32_t _1314_i = (int)0, _9772_ = _1304_ni; _1314_i < _9772_ ; ++_1314_i)
      {
        _9771_.push_back((int32_t)_ptr__1305_varidxs_[_ptr__1312_perm[_1314_i]]);
      }
      auto _9773_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9771_.size()),_9771_.begin(),_9771_.end()));
      _1310_varidxs = _9773_;
      _ptr__1310_varidxs.update(_1310_varidxs ? _1310_varidxs->raw() : nullptr, _1310_varidxs ? _1310_varidxs->size(0) : 0);
    }
  }
  int32_t _1315_N = (int)((_1310_varidxs)->size(0));
  std::shared_ptr< monty::ndarray< double,1 > > _1316_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1315_N)));
  _checked_ptr_1<double> _ptr__1316_cof(_1316_cof ? _1316_cof->raw() : nullptr,_1316_cof ? _1316_cof->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1317_subj = _1310_varidxs;
  _checked_ptr_1<int32_t> _ptr__1317_subj(_1317_subj ? _1317_subj->raw() : nullptr,_1317_subj ? _1317_subj->size(0) : 0);
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1302_expr)).as<::mosek::fusion::Expression>())->recursive_eval(ws,rs,xs,cache_evaluations);
  ws->pop_expr();
  {
  }
  int32_t _1318_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1319_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1320_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1321_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1322_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1323_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1324_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1325_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1326_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1327_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1328_code_base = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1329_codeptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1330_cconst_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  {
  }
  if ((_1322_ncodeatom > (int)0))
  {
    {
      throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized objective"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1331_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1331_wi32(_1331_wi32 ? _1331_wi32->raw() : nullptr,_1331_wi32 ? _1331_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1332_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1332_wi64(_1332_wi64 ? _1332_wi64->raw() : nullptr,_1332_wi64 ? _1332_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1333_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1333_wf64(_1333_wf64 ? _1333_wf64->raw() : nullptr,_1333_wf64 ? _1333_wf64->size(0) : 0);
  int32_t _1334_k = (int)0;
  int32_t _9774_ = (int)0;
  int32_t _9775_ = _1319_nelem;
  for (int32_t _1335_i = _9774_; _1335_i < _9775_; ++_1335_i)
  {
    {
      int32_t _1336_p0 = _ptr__1331_wi32[safe_add( _1324_ptr_base,_1335_i )];
      int32_t _1337_p1 = _ptr__1331_wi32[safe_add( safe_add( _1324_ptr_base,_1335_i ),(int)1 )];
      int32_t _1338_p = _1336_p0;
      int32_t _1339_l = (int)0;
      while ( ((_1339_l < (int)((_1310_varidxs)->size(0))) && (_1338_p < _1337_p1)) )
      {
        {
          if ((_ptr__1310_varidxs[_1339_l] < _ptr__1332_wi64[safe_add( _1326_nidxs_base,_1338_p )]))
          {
            {
              ++ _1339_l;
            }
          }
          else if((_ptr__1310_varidxs[_1339_l] > _ptr__1332_wi64[safe_add( _1326_nidxs_base,_1338_p )]))
          {
            {
              ++ _1338_p;
            }
          }
          else
          {
            {
              _ptr__1316_cof[safe_add( _1334_k,_1339_l )] = _ptr__1333_wf64[safe_add( _1327_cof_base,_1338_p )];
              ++ _1339_l;
              ++ _1338_p;
            }
          }
        }
      }
      _1334_k += (int)((_1310_varidxs)->size(0));
    }
  }
  task_putclist(_1317_subj,_1316_cof);
  if ((_1322_ncodeatom > (int)0))
  {
    {
      obj_blocks->replace_row_code(rs,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0)}),_1324_ptr_base,_1326_nidxs_base,_1329_codeptr_base,_1328_code_base,_1330_cconst_base);
    }
  }
  {}
}
// End mosek.fusion.Model.updateObjective

// Begin mosek.fusion.Model.parameter_unchecked
// Method mosek.fusion.Model.parameter_unchecked @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2809:6-2842:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_unchecked(const std::string & _1340_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1341_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1342_sp)
{
  _checked_ptr_1<int32_t> _ptr__1341_shape(_1341_shape ? _1341_shape->raw() : nullptr, _1341_shape ? _1341_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1342_sp(_1342_sp ? _1342_sp->raw() : nullptr, _1342_sp ? _1342_sp->size(0) : 0);
  if ((((int)((_1340_name).size()) > (int)0) && par_map->hasItem(_1340_name)))
  {
    {
      throw ::mosek::fusion::NameError(std::string ("Duplicate parameter name"));
    }
  }
  {}
  int64_t _1343_sz = ::mosek::fusion::p_Set::size(_1341_shape);
  int32_t _9776_;
  bool _9777_ = (NULL != _1342_sp.get());
  if (_9777_)
  {
    _9776_ = (int)((_1342_sp)->size(0));
  }
  else
  {
    _9776_ = (int32_t)_1343_sz;
  }
  int32_t _1344_nelem = _9776_;
  if ((NULL == param_value.get()))
  {
    {
      param_value = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1344_nelem)));
    }
  }
  else
  {
    {
      if ((safe_add( param_num,_1344_nelem ) > (int)((param_value)->size(0))))
      {
        {
          std::shared_ptr< monty::ndarray< double,1 > > _1345_old_param_value = param_value;
          _checked_ptr_1<double> _ptr__1345_old_param_value(_1345_old_param_value ? _1345_old_param_value->raw() : nullptr,_1345_old_param_value ? _1345_old_param_value->size(0) : 0);
          int32_t _9778_;
          bool _9779_ = (safe_mul( (int)2,(int)((_1345_old_param_value)->size(0)) ) > safe_add( (int)((_1345_old_param_value)->size(0)),_1344_nelem ));
          if (_9779_)
          {
            _9778_ = safe_mul( (int)2,(int)((_1345_old_param_value)->size(0)) );
          }
          else
          {
            _9778_ = safe_add( (int)((_1345_old_param_value)->size(0)),_1344_nelem );
          }
          int32_t _1346_new_len = _9778_;
          param_value = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1346_new_len)));
          if ((param_num > (int)0))
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1345_old_param_value,(int)0,param_value,(int)0,param_num);
            }
          }
          {}
        }
      }
      {}
    }
  }
  std::vector<int32_t> _9780_;
  for (int32_t _1348_i = (int)0, _9781_ = (int32_t)_1344_nelem; _1348_i < _9781_ ; ++_1348_i)
  {
    _9780_.push_back(safe_add( param_num,_1348_i ));
  }
  auto _9782_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9780_.size()),_9780_.begin(),_9780_.end()));
  monty::rc_ptr< ::mosek::fusion::ParameterImpl > _1347_r = ::mosek::fusion::p_ParameterImpl::_new_ParameterImpl(::mosek::fusion::Model::t(_pubthis),_1341_shape,_1342_sp,_9782_);
  param_num += _1344_nelem;
  if (((int)((_1340_name).size()) > (int)0))
  {
    {
      if (((NULL == parameters.get()) || (numparameter == (int)((parameters)->size(0)))))
      {
        {
          std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > > _1349_tmppar = parameters;
          std::vector<monty::rc_ptr< ::mosek::fusion::Parameter >> _9783_;
          for (int32_t _1350_i = (int)0, _9784_ = safe_add( safe_mul( numparameter,(int)2 ),(int)64 ); _1350_i < _9784_ ; ++_1350_i)
          {
            _9783_.push_back(nullptr);
          }
          auto _9785_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 >(monty::shape(_9783_.size()),_9783_.begin(),_9783_.end()));
          parameters = _9785_;
          int32_t _9786_ = (int)0;
          int32_t _9787_ = numparameter;
          for (int32_t _1351_i = _9786_; _1351_i < _9787_; ++_1351_i)
          {
            {
              ((*parameters)(_1351_i)) = ((*_1349_tmppar)(_1351_i));
            }
          }
        }
      }
      {}
      ((*parameters)(numparameter)) = (_1347_r).as<::mosek::fusion::Parameter>();
      par_map->setItem(_1340_name,numparameter);
      numparameter += (int)1;
    }
  }
  {}
  return (_1347_r).as<::mosek::fusion::Parameter>();
}
// End mosek.fusion.Model.parameter_unchecked

// Begin mosek.fusion.Model.parameter_
// Method mosek.fusion.Model.parameter_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2793:6-2807:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_(const std::string & _1352_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1353_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1354_sp)
{
  _checked_ptr_1<int32_t> _ptr__1353_shape(_1353_shape ? _1353_shape->raw() : nullptr, _1353_shape ? _1353_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1354_sp(_1354_sp ? _1354_sp->raw() : nullptr, _1354_sp ? _1354_sp->size(0) : 0);
  bool _1355_sorted = true;
  int32_t _1356_nd = (int)((_1353_shape)->size(0));
  if (((NULL != _1354_sp.get()) && ((int)((_1354_sp)->size(0)) > (int)0)))
  {
    {
      bool _1357_ok = (_ptr__1354_sp[((int)((_1354_sp)->size(0)) - (int)1)] < ::mosek::fusion::p_Set::size(_1353_shape));
      int32_t _9788_ = (int)1;
      int32_t _9789_ = (int)((_1354_sp)->size(0));
      for (int32_t _1358_i = _9788_; _1358_i < _9789_; ++_1358_i)
      {
        {
          _1357_ok = (_1357_ok && (_ptr__1354_sp[(_1358_i - (int)1)] < _ptr__1354_sp[_1358_i]));
        }
      }
      if ((!_1357_ok))
      {
        {
          throw ::mosek::fusion::IndexError(std::string ("Unordered sparsity pattern"));
        }
      }
      {}
    }
  }
  {}
  return __mosek_2fusion_2Model__parameter_unchecked(_1352_name,_1353_shape,_1354_sp);
}
// End mosek.fusion.Model.parameter_

// Begin mosek.fusion.Model.parameter_
// Method mosek.fusion.Model.parameter_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2770:6-2791:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_(const std::string & _1359_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1360_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1361_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1360_shape(_1360_shape ? _1360_shape->raw() : nullptr, _1360_shape ? _1360_shape->size(0) : 0);
  int32_t _1362_n = (int)((_1361_sparsity)->size(0));
  int32_t _1363_m = (int)((_1361_sparsity)->size(1));
  if (((int)((_1360_shape)->size(0)) != _1363_m))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid sparsity"));
    }
  }
  {}
  if ((NULL == _1361_sparsity.get()))
  {
    {
      return __mosek_2fusion_2Model__parameter_unchecked(_1359_name,_1360_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr));
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1364_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1362_n)));
      _checked_ptr_1<int64_t> _ptr__1364_sp(_1364_sp ? _1364_sp->raw() : nullptr,_1364_sp ? _1364_sp->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1365_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1363_m)));
      _checked_ptr_1<int64_t> _ptr__1365_strides(_1365_strides ? _1365_strides->raw() : nullptr,_1365_strides ? _1365_strides->size(0) : 0);
      _ptr__1365_strides[(_1363_m - (int)1)] = (int)1;
      int32_t _9790_ = (int)1;
      int32_t _9791_ = _1363_m;
      for (int32_t _1366_i = _9790_; _1366_i < _9791_; ++_1366_i)
      {
        {
          _ptr__1365_strides[((_1363_m - (int)1) - _1366_i)] = safe_mul( _ptr__1365_strides[(_1363_m - _1366_i)],_ptr__1360_shape[(_1363_m - _1366_i)] );
        }
      }
      int32_t _9792_ = (int)0;
      int32_t _9793_ = _1362_n;
      for (int32_t _1367_i = _9792_; _1367_i < _9793_; ++_1367_i)
      {
        {
          _ptr__1364_sp[_1367_i] = (int)0;
          int32_t _9794_ = (int)0;
          int32_t _9795_ = _1363_m;
          for (int32_t _1368_j = _9794_; _1368_j < _9795_; ++_1368_j)
          {
            {
              _ptr__1364_sp[_1367_i] += safe_mul( ((*_1361_sparsity)(_1367_i,_1368_j)),_ptr__1365_strides[_1368_j] );
            }
          }
        }
      }
      return __mosek_2fusion_2Model__parameter_(_1359_name,_1360_shape,_1364_sp);
    }
  }
}
// End mosek.fusion.Model.parameter_

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2768:6-139
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1369_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1369_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1369_name) { return __mosek_2fusion_2Model__parameter(_1369_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1369_name)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1370_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1370_sp(_1370_sp ? _1370_sp->raw() : nullptr,_1370_sp ? _1370_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1369_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),_1370_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2767:6-159
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1371_name,int32_t _1372_d1,int32_t _1373_d2,int32_t _1374_d3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1371_name,_1372_d1,_1373_d2,_1374_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1371_name,int32_t _1372_d1,int32_t _1373_d2,int32_t _1374_d3) { return __mosek_2fusion_2Model__parameter(_1371_name,_1372_d1,_1373_d2,_1374_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1371_name,int32_t _1372_d1,int32_t _1373_d2,int32_t _1374_d3)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1375_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1375_sp(_1375_sp ? _1375_sp->raw() : nullptr,_1375_sp ? _1375_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1371_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1372_d1),(int32_t)(_1373_d2),(int32_t)(_1374_d3)}),_1375_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2766:6-146
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1376_name,int32_t _1377_d1,int32_t _1378_d2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1376_name,_1377_d1,_1378_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1376_name,int32_t _1377_d1,int32_t _1378_d2) { return __mosek_2fusion_2Model__parameter(_1376_name,_1377_d1,_1378_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1376_name,int32_t _1377_d1,int32_t _1378_d2)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1379_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1379_sp(_1379_sp ? _1379_sp->raw() : nullptr,_1379_sp ? _1379_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1376_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1377_d1),(int32_t)(_1378_d2)}),_1379_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2765:6-143
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1380_name,int32_t _1381_d1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1380_name,_1381_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1380_name,int32_t _1381_d1) { return __mosek_2fusion_2Model__parameter(_1380_name,_1381_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1380_name,int32_t _1381_d1)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1382_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1382_sp(_1382_sp ? _1382_sp->raw() : nullptr,_1382_sp ? _1382_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1380_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1381_d1)}),_1382_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2764:6-134
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1383_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1384_shape) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1383_name,_1384_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1383_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1384_shape) { return __mosek_2fusion_2Model__parameter(_1383_name,_1384_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1383_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1384_shape)
{
  _checked_ptr_1<int32_t> _ptr__1384_shape(_1384_shape ? _1384_shape->raw() : nullptr, _1384_shape ? _1384_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1385_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1385_sp(_1385_sp ? _1385_sp->raw() : nullptr,_1385_sp ? _1385_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1383_name,_1384_shape,_1385_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2763:6-123
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1386_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1387_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1388_sp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1386_name,_1387_shape,_1388_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1386_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1387_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1388_sp) { return __mosek_2fusion_2Model__parameter(_1386_name,_1387_shape,_1388_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1386_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1387_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1388_sp)
{
  _checked_ptr_1<int32_t> _ptr__1387_shape(_1387_shape ? _1387_shape->raw() : nullptr, _1387_shape ? _1387_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1388_sp(_1388_sp ? _1388_sp->raw() : nullptr, _1388_sp ? _1388_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1386_name,_1387_shape,_1388_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2762:6-136
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1389_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1390_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1391_sparsity) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1389_name,_1390_shape,_1391_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1389_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1390_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1391_sparsity) { return __mosek_2fusion_2Model__parameter(_1389_name,_1390_shape,_1391_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1389_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1390_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1391_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1390_shape(_1390_shape ? _1390_shape->raw() : nullptr, _1390_shape ? _1390_shape->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1389_name,_1390_shape,_1391_sparsity);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2760:6-126
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter() { return __mosek_2fusion_2Model__parameter(); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter()
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1392_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1392_sp(_1392_sp ? _1392_sp->raw() : nullptr,_1392_sp ? _1392_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),_1392_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2759:6-145
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1393_d1,int32_t _1394_d2,int32_t _1395_d3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1393_d1,_1394_d2,_1395_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1393_d1,int32_t _1394_d2,int32_t _1395_d3) { return __mosek_2fusion_2Model__parameter(_1393_d1,_1394_d2,_1395_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1393_d1,int32_t _1394_d2,int32_t _1395_d3)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1396_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1396_sp(_1396_sp ? _1396_sp->raw() : nullptr,_1396_sp ? _1396_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1393_d1),(int32_t)(_1394_d2),(int32_t)(_1395_d3)}),_1396_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2758:6-132
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1397_d1,int32_t _1398_d2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1397_d1,_1398_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1397_d1,int32_t _1398_d2) { return __mosek_2fusion_2Model__parameter(_1397_d1,_1398_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1397_d1,int32_t _1398_d2)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1399_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1399_sp(_1399_sp ? _1399_sp->raw() : nullptr,_1399_sp ? _1399_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1397_d1),(int32_t)(_1398_d2)}),_1399_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2757:6-129
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1400_d1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1400_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1400_d1) { return __mosek_2fusion_2Model__parameter(_1400_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1400_d1)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1401_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1401_sp(_1401_sp ? _1401_sp->raw() : nullptr,_1401_sp ? _1401_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1400_d1)}),_1401_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2756:6-120
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1402_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape) { return __mosek_2fusion_2Model__parameter(_1402_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape)
{
  _checked_ptr_1<int32_t> _ptr__1402_shape(_1402_shape ? _1402_shape->raw() : nullptr, _1402_shape ? _1402_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1403_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1403_sp(_1403_sp ? _1403_sp->raw() : nullptr,_1403_sp ? _1403_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1402_shape,_1403_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2755:6-109
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1404_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1405_sp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1404_shape,_1405_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1404_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1405_sp) { return __mosek_2fusion_2Model__parameter(_1404_shape,_1405_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1404_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1405_sp)
{
  _checked_ptr_1<int32_t> _ptr__1404_shape(_1404_shape ? _1404_shape->raw() : nullptr, _1404_shape ? _1404_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1405_sp(_1405_sp ? _1405_sp->raw() : nullptr, _1405_sp ? _1405_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1404_shape,_1405_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2754:6-122
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1406_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1407_sparsity) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1406_shape,_1407_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1406_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1407_sparsity) { return __mosek_2fusion_2Model__parameter(_1406_shape,_1407_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1406_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1407_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1406_shape(_1406_shape ? _1406_shape->raw() : nullptr, _1406_shape ? _1406_shape->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1406_shape,_1407_sparsity);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.objective_
// Method mosek.fusion.Model.objective_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2658:6-2747:7
void mosek::fusion::p_Model::objective_(const std::string & _1408_name,mosek::fusion::ObjectiveSense _1409_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1410_expr)
{
  task_putobjectivename(_1408_name);
  monty::rc_ptr< ::mosek::fusion::WorkStack > _1411_rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  monty::rc_ptr< ::mosek::fusion::WorkStack > _1412_ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1413_obje = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1410_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1413_obje->recursive_eval(_1411_rs,_1412_ws,xs,cache_evaluations);
  _1411_rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1414_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1414_ri32(_1414_ri32 ? _1414_ri32->raw() : nullptr,_1414_ri32 ? _1414_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1415_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1415_ri64(_1415_ri64 ? _1415_ri64->raw() : nullptr,_1415_ri64 ? _1415_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1416_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->f64;
  _checked_ptr_1<double> _ptr__1416_rf64(_1416_rf64 ? _1416_rf64->raw() : nullptr,_1416_rf64 ? _1416_rf64->size(0) : 0);
  int32_t _1417_nd = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->nd;
  int32_t _1418_nelem = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->nelem;
  int32_t _1419_nnz = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->nnz;
  bool _1420_hassp = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->hassp;
  int32_t _1421_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->ncodeatom;
  int32_t _1422_ptr_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->ptr_base;
  int32_t _1423_shape_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->shape_base;
  int32_t _1424_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->nidxs_base;
  int32_t _1425_cof_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->cof_base;
  int32_t _1426_code_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->code_base;
  int32_t _1427_codeptr_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->codeptr_base;
  int32_t _1428_cconst_base = mosek::fusion::p_WorkStack::_get_impl(_1411_rs)->cconst_base;
  int64_t _1429_totalsize = (int64_t)1;
  int32_t _9796_ = (int)0;
  int32_t _9797_ = _1417_nd;
  for (int32_t _1430_i = _9796_; _1430_i < _9797_; ++_1430_i)
  {
    {
      _1417_nd *= _ptr__1414_ri32[safe_add( _1423_shape_base,_1430_i )];
    }
  }
  if ((_1429_totalsize != (int)1))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid expression size for objective"));
    }
  }
  {}
  if ((_1418_nelem != (int)1))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid expression for objective"));
    }
  }
  {}
  int32_t _1431_linnz = (int)0;
  int32_t _1432_psdnz = (int)0;
  int32_t _9798_ = (int)0;
  int32_t _9799_ = _1419_nnz;
  for (int32_t _1433_i = _9798_; _1433_i < _9799_; ++_1433_i)
  {
    {
      if ((_ptr__1415_ri64[safe_add( _1424_nidxs_base,_1433_i )] < (int)0))
      {
        {
          ++ _1432_psdnz;
        }
      }
      else if((_ptr__1415_ri64[safe_add( _1424_nidxs_base,_1433_i )] >= (int)0))
      {
        {
          ++ _1431_linnz;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1434_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1431_linnz)));
  _checked_ptr_1<int32_t> _ptr__1434_subj(_1434_subj ? _1434_subj->raw() : nullptr,_1434_subj ? _1434_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1435_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1431_linnz)));
  _checked_ptr_1<double> _ptr__1435_cof(_1435_cof ? _1435_cof->raw() : nullptr,_1435_cof ? _1435_cof->size(0) : 0);
  {
    int32_t _1436_k = (int)0;
    int32_t _9800_ = (int)0;
    int32_t _9801_ = _1419_nnz;
    for (int32_t _1437_j = _9800_; _1437_j < _9801_; ++_1437_j)
    {
      {
        if ((_ptr__1415_ri64[safe_add( _1424_nidxs_base,_1437_j )] >= (int)0))
        {
          {
            _ptr__1434_subj[_1436_k] = (int32_t)_ptr__1415_ri64[safe_add( _1424_nidxs_base,_1437_j )];
            _ptr__1435_cof[_1436_k] = _ptr__1416_rf64[safe_add( _1425_cof_base,_1437_j )];
            ++ _1436_k;
          }
        }
        {}
      }
    }
  }
  task_putobjective((_1409_sense == ::mosek::fusion::ObjectiveSense::Maximize),_1434_subj,_1435_cof,0.0);
  if ((_1432_psdnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1438_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1432_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1438_subj(_1438_subj ? _1438_subj->raw() : nullptr,_1438_subj ? _1438_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1439_subk = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1432_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1439_subk(_1439_subk ? _1439_subk->raw() : nullptr,_1439_subk ? _1439_subk->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1440_subl = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1432_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1440_subl(_1440_subl ? _1440_subl->raw() : nullptr,_1440_subl ? _1440_subl->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1441_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1432_psdnz)));
      _checked_ptr_1<double> _ptr__1441_val(_1441_val ? _1441_val->raw() : nullptr,_1441_val ? _1441_val->size(0) : 0);
      int32_t _9802_ = (int)0;
      int32_t _9803_ = _1432_psdnz;
      for (int32_t _1442_i = _9802_; _1442_i < _9803_; ++_1442_i)
      {
        {
          int64_t _1443_varidx = (-safe_add( _ptr__1415_ri64[safe_add( _1424_nidxs_base,_1442_i )],(int)1 ));
          _ptr__1438_subj[_1442_i] = ((*barvar_block_elm_barj)(_1443_varidx));
          _ptr__1439_subk[_1442_i] = ((*barvar_block_elm_i)(_1443_varidx));
          _ptr__1440_subl[_1442_i] = ((*barvar_block_elm_j)(_1443_varidx));
          double _9804_;
          bool _9805_ = (_ptr__1439_subk[_1442_i] == _ptr__1440_subl[_1442_i]);
          if (_9805_)
          {
            _9804_ = _ptr__1416_rf64[safe_add( _1425_cof_base,_1442_i )];
          }
          else
          {
            _9804_ = (0.5 * _ptr__1416_rf64[safe_add( _1425_cof_base,_1442_i )]);
          }
          _ptr__1441_val[_1442_i] = _9804_;
        }
      }
      task_putbarc(_1438_subj,_1439_subk,_1440_subl,_1441_val);
    }
  }
  {}
  if ((_1421_ncodeatom > (int)0))
  {
    {
      obj_blocks->replace_row_code(_1411_rs,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0)}),_1422_ptr_base,_1424_nidxs_base,_1427_codeptr_base,_1426_code_base,_1428_cconst_base);
    }
  }
  {}
}
// End mosek.fusion.Model.objective_

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2655:6-163
void mosek::fusion::Model :: objective(double _1444_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1444_c); }
void mosek::fusion::p_Model::objective(double _1444_c)
{
  objective_(std::string (""),::mosek::fusion::ObjectiveSense::Minimize,::mosek::fusion::p_Expr::constTerm(_1444_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2654:6-145
void mosek::fusion::Model :: objective(mosek::fusion::ObjectiveSense _1445_sense,double _1446_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1445_sense,_1446_c); }
void mosek::fusion::p_Model::objective(mosek::fusion::ObjectiveSense _1445_sense,double _1446_c)
{
  objective_(std::string (""),_1445_sense,::mosek::fusion::p_Expr::constTerm(_1446_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2653:6-132
void mosek::fusion::Model :: objective(mosek::fusion::ObjectiveSense _1447_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1448_expr) { mosek::fusion::p_Model::_get_impl(this)->objective(_1447_sense,_1448_expr); }
void mosek::fusion::p_Model::objective(mosek::fusion::ObjectiveSense _1447_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1448_expr)
{
  objective_(std::string (""),_1447_sense,_1448_expr);
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2652:6-165
void mosek::fusion::Model :: objective(const std::string & _1449_name,double _1450_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1449_name,_1450_c); }
void mosek::fusion::p_Model::objective(const std::string & _1449_name,double _1450_c)
{
  objective_(_1449_name,::mosek::fusion::ObjectiveSense::Minimize,::mosek::fusion::p_Expr::constTerm(_1450_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2651:6-147
void mosek::fusion::Model :: objective(const std::string & _1451_name,mosek::fusion::ObjectiveSense _1452_sense,double _1453_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1451_name,_1452_sense,_1453_c); }
void mosek::fusion::p_Model::objective(const std::string & _1451_name,mosek::fusion::ObjectiveSense _1452_sense,double _1453_c)
{
  objective_(_1451_name,_1452_sense,::mosek::fusion::p_Expr::constTerm(_1453_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2650:6-134
void mosek::fusion::Model :: objective(const std::string & _1454_name,mosek::fusion::ObjectiveSense _1455_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1456_expr) { mosek::fusion::p_Model::_get_impl(this)->objective(_1454_name,_1455_sense,_1456_expr); }
void mosek::fusion::p_Model::objective(const std::string & _1454_name,mosek::fusion::ObjectiveSense _1455_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1456_expr)
{
  objective_(_1454_name,_1455_sense,_1456_expr);
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2648:6-159
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1457_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1457_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1457_exprdom) { return __mosek_2fusion_2Model__constraint(_1457_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1457_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprRangeDomain::_get_impl(_1457_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_1457_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2646:6-159
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1458_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1459_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1458_name,_1459_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(const std::string &  _1458_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1459_exprdom) { return __mosek_2fusion_2Model__constraint(_1458_name,_1459_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1458_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1459_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1458_name,mosek::fusion::p_ExprRangeDomain::_get_impl(_1459_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_1459_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2644:6-144
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1460_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1461_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1460_expr,_1461_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1460_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1461_rdom) { return __mosek_2fusion_2Model__constraint(_1460_expr,_1461_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1460_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1461_rdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1460_expr,_1461_rdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2643:6-144
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1462_name,monty::rc_ptr< ::mosek::fusion::Expression > _1463_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1464_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1462_name,_1463_expr,_1464_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(const std::string &  _1462_name,monty::rc_ptr< ::mosek::fusion::Expression > _1463_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1464_rdom) { return __mosek_2fusion_2Model__constraint(_1462_name,_1463_expr,_1464_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1462_name,monty::rc_ptr< ::mosek::fusion::Expression > _1463_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1464_rdom)
{
  return __mosek_2fusion_2Model__constraint_(_1462_name,_1463_expr,_1464_rdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2641:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1465_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1465_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1465_exprdom) { return __mosek_2fusion_2Model__constraint(_1465_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1465_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprConicDomain::_get_impl(_1465_exprdom)->expr,mosek::fusion::p_ExprConicDomain::_get_impl(_1465_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2639:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1466_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1467_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1466_name,_1467_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1466_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1467_exprdom) { return __mosek_2fusion_2Model__constraint(_1466_name,_1467_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1466_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1467_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1466_name,mosek::fusion::p_ExprConicDomain::_get_impl(_1467_exprdom)->expr,mosek::fusion::p_ExprConicDomain::_get_impl(_1467_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2637:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1468_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1469_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1468_expr,_1469_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1468_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1469_qdom) { return __mosek_2fusion_2Model__constraint(_1468_expr,_1469_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1468_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1469_qdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1468_expr,_1469_qdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2636:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1470_name,monty::rc_ptr< ::mosek::fusion::Expression > _1471_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1472_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1470_name,_1471_expr,_1472_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1470_name,monty::rc_ptr< ::mosek::fusion::Expression > _1471_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1472_qdom) { return __mosek_2fusion_2Model__constraint(_1470_name,_1471_expr,_1472_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1470_name,monty::rc_ptr< ::mosek::fusion::Expression > _1471_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1472_qdom)
{
  return __mosek_2fusion_2Model__constraint_(_1470_name,_1471_expr,_1472_qdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2634:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1473_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1473_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1473_exprdom) { return __mosek_2fusion_2Model__constraint(_1473_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1473_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprLinearDomain::_get_impl(_1473_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_1473_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2632:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1474_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1475_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1474_name,_1475_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1474_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1475_exprdom) { return __mosek_2fusion_2Model__constraint(_1474_name,_1475_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1474_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1475_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1474_name,mosek::fusion::p_ExprLinearDomain::_get_impl(_1475_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_1475_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2630:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1476_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1477_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1476_expr,_1477_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1476_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1477_ldom) { return __mosek_2fusion_2Model__constraint(_1476_expr,_1477_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1476_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1477_ldom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1476_expr,_1477_ldom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2629:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1478_name,monty::rc_ptr< ::mosek::fusion::Expression > _1479_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1480_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1478_name,_1479_expr,_1480_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1478_name,monty::rc_ptr< ::mosek::fusion::Expression > _1479_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1480_ldom) { return __mosek_2fusion_2Model__constraint(_1478_name,_1479_expr,_1480_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1478_name,monty::rc_ptr< ::mosek::fusion::Expression > _1479_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1480_ldom)
{
  return __mosek_2fusion_2Model__constraint_(_1478_name,_1479_expr,_1480_ldom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2627:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1481_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1481_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1481_exprdom) { return __mosek_2fusion_2Model__constraint(_1481_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1481_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprPSDDomain::_get_impl(_1481_exprdom)->expr,mosek::fusion::p_ExprPSDDomain::_get_impl(_1481_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2625:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1482_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1483_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1482_name,_1483_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1482_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1483_exprdom) { return __mosek_2fusion_2Model__constraint(_1482_name,_1483_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1482_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1483_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1482_name,mosek::fusion::p_ExprPSDDomain::_get_impl(_1483_exprdom)->expr,mosek::fusion::p_ExprPSDDomain::_get_impl(_1483_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2623:6-141
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1484_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1484_expr,_1485_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1484_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom) { return __mosek_2fusion_2Model__constraint(_1484_expr,_1485_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1484_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1484_expr,_1485_psddom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2622:6-141
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1486_name,monty::rc_ptr< ::mosek::fusion::Expression > _1487_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1486_name,_1487_expr,_1488_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1486_name,monty::rc_ptr< ::mosek::fusion::Expression > _1487_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom) { return __mosek_2fusion_2Model__constraint(_1486_name,_1487_expr,_1488_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1486_name,monty::rc_ptr< ::mosek::fusion::Expression > _1487_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom)
{
  return __mosek_2fusion_2Model__constraint_(_1486_name,_1487_expr,_1488_psddom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2611:6-170
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1489_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1489_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1489_psddom) { return __mosek_2fusion_2Model__variable(_1489_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1489_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),mosek::fusion::p_PSDDomain::_get_impl(_1489_psddom)->shape,_1489_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2610:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1490_n,int32_t _1491_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1490_n,_1491_m,_1492_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1490_n,int32_t _1491_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom) { return __mosek_2fusion_2Model__variable(_1490_n,_1491_m,_1492_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1490_n,int32_t _1491_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1491_m),(int32_t)(_1490_n),(int32_t)(_1490_n)}),_1492_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2609:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1493_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1494_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1493_n,_1494_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1493_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1494_psddom) { return __mosek_2fusion_2Model__variable(_1493_n,_1494_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1493_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1494_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1493_n),(int32_t)(_1493_n)}),_1494_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2608:6-170
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1495_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1495_name,_1496_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1495_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom) { return __mosek_2fusion_2Model__variable(_1495_name,_1496_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1495_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1495_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1496_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2607:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1497_name,int32_t _1498_n,int32_t _1499_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1500_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1497_name,_1498_n,_1499_m,_1500_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1497_name,int32_t _1498_n,int32_t _1499_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1500_psddom) { return __mosek_2fusion_2Model__variable(_1497_name,_1498_n,_1499_m,_1500_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1497_name,int32_t _1498_n,int32_t _1499_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1500_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1497_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1499_m),(int32_t)(_1498_n),(int32_t)(_1498_n)}),_1500_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2606:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1501_name,int32_t _1502_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1503_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1501_name,_1502_n,_1503_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1501_name,int32_t _1502_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1503_psddom) { return __mosek_2fusion_2Model__variable(_1501_name,_1502_n,_1503_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1501_name,int32_t _1502_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1503_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1501_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1502_n),(int32_t)(_1502_n)}),_1503_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2605:6-157
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1504_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1505_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1506_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1504_name,_1505_shp,_1506_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1504_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1505_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1506_psddom) { return __mosek_2fusion_2Model__variable(_1504_name,_1505_shp,_1506_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1504_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1505_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1506_psddom)
{
  _checked_ptr_1<int32_t> _ptr__1505_shp(_1505_shp ? _1505_shp->raw() : nullptr, _1505_shp ? _1505_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1504_name,_1505_shp,_1506_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2603:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1507_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom) { return __mosek_2fusion_2Model__variable(_1507_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1507_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2602:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1508_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1508_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1508_rdom) { return __mosek_2fusion_2Model__variable(_1508_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1508_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1508_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2601:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1509_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1509_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1509_ldom) { return __mosek_2fusion_2Model__variable(_1509_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1509_ldom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1509_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2600:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1511_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1510_shp,_1511_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1511_qdom) { return __mosek_2fusion_2Model__variable(_1510_shp,_1511_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1511_qdom)
{
  _checked_ptr_1<int32_t> _ptr__1510_shp(_1510_shp ? _1510_shp->raw() : nullptr, _1510_shp ? _1510_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1510_shp,_1511_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2599:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1513_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1512_shp,_1513_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1513_rdom) { return __mosek_2fusion_2Model__variable(_1512_shp,_1513_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1513_rdom)
{
  _checked_ptr_1<int32_t> _ptr__1512_shp(_1512_shp ? _1512_shp->raw() : nullptr, _1512_shp ? _1512_shp->size(0) : 0);
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),_1512_shp,_1513_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2598:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1514_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1515_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1514_shp,_1515_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1514_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1515_ldom) { return __mosek_2fusion_2Model__variable(_1514_shp,_1515_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1514_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1515_ldom)
{
  _checked_ptr_1<int32_t> _ptr__1514_shp(_1514_shp ? _1514_shp->raw() : nullptr, _1514_shp ? _1514_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1514_shp,_1515_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2597:6-162
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1516_shp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1516_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1516_shp) { return __mosek_2fusion_2Model__variable(_1516_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1516_shp)
{
  _checked_ptr_1<int32_t> _ptr__1516_shp(_1516_shp ? _1516_shp->raw() : nullptr, _1516_shp ? _1516_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1516_shp,::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2596:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1518_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1517_size,_1518_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1518_qdom) { return __mosek_2fusion_2Model__variable(_1517_size,_1518_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1518_qdom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1517_size)}),_1518_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2595:6-161
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1519_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1520_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1519_size,_1520_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(int32_t _1519_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1520_rdom) { return __mosek_2fusion_2Model__variable(_1519_size,_1520_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1519_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1520_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1519_size)}),_1520_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2594:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1521_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1522_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1521_size,_1522_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1521_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1522_ldom) { return __mosek_2fusion_2Model__variable(_1521_size,_1522_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1521_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1522_ldom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1521_size)}),_1522_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2593:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1523_size) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1523_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1523_size) { return __mosek_2fusion_2Model__variable(_1523_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1523_size)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1523_size)}),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2592:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable() { return __mosek_2fusion_2Model__variable(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable()
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2590:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1524_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1525_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1524_name,_1525_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1524_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1525_qdom) { return __mosek_2fusion_2Model__variable(_1524_name,_1525_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1524_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1525_qdom)
{
  return __mosek_2fusion_2Model__variable_(_1524_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1525_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2589:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1526_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1527_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1526_name,_1527_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1526_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1527_rdom) { return __mosek_2fusion_2Model__variable(_1526_name,_1527_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1526_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1527_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(_1526_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1527_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2588:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1528_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1529_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1528_name,_1529_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1528_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1529_ldom) { return __mosek_2fusion_2Model__variable(_1528_name,_1529_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1528_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1529_ldom)
{
  return __mosek_2fusion_2Model__variable_(_1528_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1529_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2587:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1530_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1531_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1532_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1530_name,_1531_shp,_1532_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1530_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1531_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1532_qdom) { return __mosek_2fusion_2Model__variable(_1530_name,_1531_shp,_1532_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1530_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1531_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1532_qdom)
{
  _checked_ptr_1<int32_t> _ptr__1531_shp(_1531_shp ? _1531_shp->raw() : nullptr, _1531_shp ? _1531_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1530_name,_1531_shp,_1532_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2586:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1533_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1534_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1535_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1533_name,_1534_shp,_1535_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1533_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1534_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1535_rdom) { return __mosek_2fusion_2Model__variable(_1533_name,_1534_shp,_1535_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1533_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1534_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1535_rdom)
{
  _checked_ptr_1<int32_t> _ptr__1534_shp(_1534_shp ? _1534_shp->raw() : nullptr, _1534_shp ? _1534_shp->size(0) : 0);
  return __mosek_2fusion_2Model__ranged_variable(_1533_name,_1534_shp,_1535_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2585:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1536_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1537_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1538_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1536_name,_1537_shp,_1538_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1536_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1537_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1538_ldom) { return __mosek_2fusion_2Model__variable(_1536_name,_1537_shp,_1538_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1536_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1537_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1538_ldom)
{
  _checked_ptr_1<int32_t> _ptr__1537_shp(_1537_shp ? _1537_shp->raw() : nullptr, _1537_shp ? _1537_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1536_name,_1537_shp,_1538_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2584:6-162
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1539_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1540_shp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1539_name,_1540_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1539_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1540_shp) { return __mosek_2fusion_2Model__variable(_1539_name,_1540_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1539_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1540_shp)
{
  _checked_ptr_1<int32_t> _ptr__1540_shp(_1540_shp ? _1540_shp->raw() : nullptr, _1540_shp ? _1540_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1539_name,_1540_shp,::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2583:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1541_name,int32_t _1542_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1543_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1541_name,_1542_size,_1543_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1541_name,int32_t _1542_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1543_qdom) { return __mosek_2fusion_2Model__variable(_1541_name,_1542_size,_1543_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1541_name,int32_t _1542_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1543_qdom)
{
  return __mosek_2fusion_2Model__variable_(_1541_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1542_size)}),_1543_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2582:6-161
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1544_name,int32_t _1545_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1546_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1544_name,_1545_size,_1546_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1544_name,int32_t _1545_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1546_rdom) { return __mosek_2fusion_2Model__variable(_1544_name,_1545_size,_1546_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1544_name,int32_t _1545_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1546_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(_1544_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1545_size)}),_1546_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2581:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1547_name,int32_t _1548_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1549_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1547_name,_1548_size,_1549_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1547_name,int32_t _1548_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1549_ldom) { return __mosek_2fusion_2Model__variable(_1547_name,_1548_size,_1549_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1547_name,int32_t _1548_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1549_ldom)
{
  return __mosek_2fusion_2Model__variable_(_1547_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1548_size)}),_1549_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2580:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1550_name,int32_t _1551_size) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1550_name,_1551_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1550_name,int32_t _1551_size) { return __mosek_2fusion_2Model__variable(_1550_name,_1551_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1550_name,int32_t _1551_size)
{
  return __mosek_2fusion_2Model__variable_(_1550_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1551_size)}),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2579:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1552_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1552_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1552_name) { return __mosek_2fusion_2Model__variable(_1552_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1552_name)
{
  return __mosek_2fusion_2Model__variable_(_1552_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.removeConstraintBlock
// Method mosek.fusion.Model.removeConstraintBlock @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2570:6-2577:7
void mosek::fusion::p_Model::removeConstraintBlock(int32_t _1553_conid)
{
  if ((_1553_conid >= (int)0))
  {
    {
      con_release(_1553_conid);
    }
  }
  else
  {
    {
      acon_release((-safe_add( (int)1,_1553_conid )));
    }
  }
}
// End mosek.fusion.Model.removeConstraintBlock

// Begin mosek.fusion.Model.removeVariableBlock
// Method mosek.fusion.Model.removeVariableBlock @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2542:6-2567:7
void mosek::fusion::p_Model::removeVariableBlock(int64_t _1554_varid64)
{
  if ((con_blocks->is_parameterized() || afe_blocks->is_parameterized()))
  {
    {
      throw ::mosek::fusion::DeletionError(std::string ("Cannot remove variables in parameterized model"));
    }
  }
  {}
  if ((_1554_varid64 < (int)0))
  {
    {
      throw ::mosek::fusion::DeletionError(std::string ("PSD variables cannot be removed"));
    }
  }
  else
  {
    {
      int32_t _1555_varid = (int32_t)_1554_varid64;
      int32_t _1556_sz = var_block_map->blocksize(_1555_varid);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1557_nidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1556_sz)));
      _checked_ptr_1<int32_t> _ptr__1557_nidxs(_1557_nidxs ? _1557_nidxs->raw() : nullptr,_1557_nidxs ? _1557_nidxs->size(0) : 0);
      var_block_map->get(_1555_varid,_1557_nidxs,(int)0);
      var_block_map->free(_1555_varid);
      int32_t _1558_conid = ((*var_block_acc_id)(_1555_varid));
      if ((_1558_conid >= (int)0))
      {
        {
          removeConstraintBlock(_1558_conid);
        }
      }
      {}
      task_var_putboundlist_fr(_1557_nidxs);
      task_clearacollist(_1557_nidxs);
    }
  }
}
// End mosek.fusion.Model.removeVariableBlock

// Begin mosek.fusion.Model.ranged_variable
// Method mosek.fusion.Model.ranged_variable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2313:6-2539:7
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__ranged_variable(const std::string & _1559_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1560_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1561_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1560_shp(_1560_shp ? _1560_shp->raw() : nullptr, _1560_shp ? _1560_shp->size(0) : 0);
  if ((((int)((_1559_name).size()) > (int)0) && hasVariable(_1559_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1559_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::RangeDomain > _1562_dom = mosek::fusion::p_RangeDomain::_get_impl(_1561_dom_.get())->__mosek_2fusion_2RangeDomain__finalize_and_validate(_1560_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1563_shape = mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1563_shape(_1563_shape ? _1563_shape->raw() : nullptr,_1563_shape ? _1563_shape->size(0) : 0);
  int32_t _1564_nd = (int)((_1563_shape)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1565_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1564_nd)));
  _checked_ptr_1<int32_t> _ptr__1565_strides(_1565_strides ? _1565_strides->raw() : nullptr,_1565_strides ? _1565_strides->size(0) : 0);
  if ((_1564_nd > (int)0))
  {
    {
      _ptr__1565_strides[(_1564_nd - (int)1)] = (int)1;
      int32_t _9806_ = (int)1;
      int32_t _9807_ = _1564_nd;
      for (int32_t _1566_i = _9806_; _1566_i < _9807_; ++_1566_i)
      {
        {
          _ptr__1565_strides[((_1564_nd - _1566_i) - (int)1)] = safe_mul( _ptr__1565_strides[(_1564_nd - _1566_i)],_ptr__1563_shape[(_1564_nd - _1566_i)] );
        }
      }
    }
  }
  {}
  int32_t _1567_domsize = (int)1;
  int32_t _9808_ = (int)0;
  int32_t _9809_ = (int)((_1563_shape)->size(0));
  for (int32_t _1568_i = _9808_; _1568_i < _9809_; ++_1568_i)
  {
    {
      _1567_domsize *= _ptr__1563_shape[_1568_i];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1569_axisptr = nullptr;
  _checked_ptr_1<int32_t> _ptr__1569_axisptr(_1569_axisptr ? _1569_axisptr->raw() : nullptr,_1569_axisptr ? _1569_axisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1570_nameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1570_nameptr(_1570_nameptr ? _1570_nameptr->raw() : nullptr,_1570_nameptr ? _1570_nameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1571_axisnames = nullptr;
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1572_accaxisptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1572_accaxisptr(_1572_accaxisptr ? _1572_accaxisptr->raw() : nullptr,_1572_accaxisptr ? _1572_accaxisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1573_accnameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1573_accnameptr(_1573_accnameptr ? _1573_accnameptr->raw() : nullptr,_1573_accnameptr ? _1573_accnameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1574_accaxisnames = nullptr;
  if ((((int)((_1559_name).size()) > (int)0) && (NULL != mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames.get())))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1575_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1576_accsb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      int32_t _1577_axisp = (int)0;
      int64_t _1578_namep = (int)0;
      int64_t _1579_numnames = (int)0;
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)->size(0)) != _1564_nd))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
        }
      }
      {}
      int32_t _9810_ = (int)0;
      int32_t _9811_ = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)->size(0));
      for (int32_t _1580_i = _9810_; _1580_i < _9811_; ++_1580_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)(_1580_i)).get()))
          {
            {
              if ((_ptr__1563_shape[_1580_i] != (int)((((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)(_1580_i)))->size(0))))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
                }
              }
              {}
              _1579_numnames += _ptr__1563_shape[_1580_i];
            }
          }
          {}
        }
      }
      _1570_nameptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1579_numnames,(int)1 ))));
      _ptr__1570_nameptr.update(_1570_nameptr ? _1570_nameptr->raw() : nullptr, _1570_nameptr ? _1570_nameptr->size(0) : 0);
      std::vector<int32_t> _9812_;
      for (int32_t _1581_i = (int)0, _9813_ = _1564_nd; _1581_i < _9813_ ; ++_1581_i)
      {
        _9812_.push_back((-(int)1));
      }
      auto _9814_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9812_.size()),_9812_.begin(),_9812_.end()));
      _1569_axisptr = _9814_;
      _ptr__1569_axisptr.update(_1569_axisptr ? _1569_axisptr->raw() : nullptr, _1569_axisptr ? _1569_axisptr->size(0) : 0);
      int32_t _9815_ = (int)0;
      int32_t _9816_ = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)->size(0));
      for (int32_t _1582_i = _9815_; _1582_i < _9816_; ++_1582_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)(_1582_i)).get()))
          {
            {
              _ptr__1569_axisptr[_1582_i] = _1577_axisp;
              int32_t _9817_ = (int)0;
              int32_t _9818_ = _ptr__1563_shape[_1582_i];
              for (int32_t _1583_j = _9817_; _1583_j < _9818_; ++_1583_j)
              {
                {
                  _1575_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)(_1582_i)))(_1583_j)));
                  ++ _1577_axisp;
                  _1578_namep += (int)((((*((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames)(_1582_i)))(_1583_j))).size());
                  _ptr__1570_nameptr[_1577_axisp] = _1578_namep;
                }
              }
            }
          }
          {}
        }
      }
      _1571_axisnames = _1575_sb;
      _1574_accaxisnames = _1576_accsb;
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1584_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__1584_sparsity(_1584_sparsity ? _1584_sparsity->raw() : nullptr,_1584_sparsity ? _1584_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1585_perm = nullptr;
  _checked_ptr_1<int32_t> _ptr__1585_perm(_1585_perm ? _1585_perm->raw() : nullptr,_1585_perm ? _1585_perm->size(0) : 0);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->empty)
  {
    {
      _1584_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
      _ptr__1584_sparsity.update(_1584_sparsity ? _1584_sparsity->raw() : nullptr, _1584_sparsity ? _1584_sparsity->size(0) : 0);
    }
  }
  else if((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->sparsity.get()))
  {
    {
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->sparsity)->size(1)) != _1564_nd))
      {
        {
          throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is not correct"));
        }
      }
      {}
      int32_t _1586_n = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->sparsity)->size(0));
      _1584_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1586_n)));
      _ptr__1584_sparsity.update(_1584_sparsity ? _1584_sparsity->raw() : nullptr, _1584_sparsity ? _1584_sparsity->size(0) : 0);
      {
        bool _1587_err = false;
        std::shared_ptr< monty::ndarray< int32_t,2 > > _1588_sp = mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->sparsity;
        int32_t _9819_ = (int)0;
        int32_t _9820_ = _1586_n;
        for (int32_t _1589_i = _9819_; _1589_i < _9820_; ++_1589_i)
        {
          {
            int32_t _9821_ = (int)0;
            int32_t _9822_ = (int)((_1588_sp)->size(1));
            for (int32_t _1590_j = _9821_; _1590_j < _9822_; ++_1590_j)
            {
              {
                _1587_err = (_1587_err || ((((*_1588_sp)(_1589_i,_1590_j)) < (int)0) || (((*_1588_sp)(_1589_i,_1590_j)) >= _ptr__1563_shape[_1590_j])));
              }
            }
            if ((!_1587_err))
            {
              {
                int64_t _1591_linidx = (int)0;
                int32_t _9823_ = (int)0;
                int32_t _9824_ = (int)((_1588_sp)->size(1));
                for (int32_t _1592_j = _9823_; _1592_j < _9824_; ++_1592_j)
                {
                  {
                    _1591_linidx += safe_mul( _ptr__1565_strides[_1592_j],((*_1588_sp)(_1589_i,_1592_j)) );
                  }
                }
                _ptr__1584_sparsity[_1589_i] = _1591_linidx;
              }
            }
            {}
          }
        }
        if (_1587_err)
        {
          {
            throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is out of bounds"));
          }
        }
        {}
      }
      std::shared_ptr< monty::ndarray< int32_t,2 > > _1593_midxs = mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->sparsity;
      int32_t _1594_maxdim = (int)0;
      int32_t _9825_ = (int)0;
      int32_t _9826_ = _1564_nd;
      for (int32_t _1595_i = _9825_; _1595_i < _9826_; ++_1595_i)
      {
        {
          if ((_1594_maxdim < _ptr__1563_shape[_1595_i]))
          {
            {
              _1594_maxdim = _ptr__1563_shape[_1595_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1596_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1586_n)));
      _checked_ptr_1<int32_t> _ptr__1596_curperm(_1596_curperm ? _1596_curperm->raw() : nullptr,_1596_curperm ? _1596_curperm->size(0) : 0);
      int32_t _9827_ = (int)0;
      int32_t _9828_ = _1586_n;
      for (int32_t _1597_i = _9827_; _1597_i < _9828_; ++_1597_i)
      {
        {
          _ptr__1596_curperm[_1597_i] = _1597_i;
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1598_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1586_n)));
      _checked_ptr_1<int32_t> _ptr__1598_tgtperm(_1598_tgtperm ? _1598_tgtperm->raw() : nullptr,_1598_tgtperm ? _1598_tgtperm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1599_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1594_maxdim,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1599_ptr(_1599_ptr ? _1599_ptr->raw() : nullptr,_1599_ptr ? _1599_ptr->size(0) : 0);
      int32_t _9829_ = (int)0;
      int32_t _9830_ = _1564_nd;
      for (int32_t _1600_dd = _9829_; _1600_dd < _9830_; ++_1600_dd)
      {
        {
          {
            int32_t _1601_d = ((_1564_nd - _1600_dd) - (int)1);
            int32_t _9831_ = (int)0;
            int32_t _9832_ = safe_add( _ptr__1563_shape[_1601_d],(int)1 );
            for (int32_t _1602_i = _9831_; _1602_i < _9832_; ++_1602_i)
            {
              {
                _ptr__1599_ptr[_1602_i] = (int)0;
              }
            }
            int32_t _9833_ = (int)0;
            int32_t _9834_ = _1586_n;
            for (int32_t _1603_i = _9833_; _1603_i < _9834_; ++_1603_i)
            {
              {
                ++ _ptr__1599_ptr[safe_add( ((*_1593_midxs)(_1603_i,_1601_d)),(int)1 )];
              }
            }
            int32_t _9835_ = (int)0;
            int32_t _9836_ = _ptr__1563_shape[_1601_d];
            for (int32_t _1604_i = _9835_; _1604_i < _9836_; ++_1604_i)
            {
              {
                _ptr__1599_ptr[safe_add( _1604_i,(int)1 )] += _ptr__1599_ptr[_1604_i];
              }
            }
            int32_t _9837_ = (int)0;
            int32_t _9838_ = _1586_n;
            for (int32_t _1605_ii = _9837_; _1605_ii < _9838_; ++_1605_ii)
            {
              {
                int32_t _1606_i = _ptr__1596_curperm[_1605_ii];
                _ptr__1598_tgtperm[_ptr__1599_ptr[((*_1593_midxs)(_1606_i,_1601_d))]] = _1606_i;
                ++ _ptr__1599_ptr[((*_1593_midxs)(_1606_i,_1601_d))];
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1607_tmp = _1596_curperm;
          _checked_ptr_1<int32_t> _ptr__1607_tmp(_1607_tmp ? _1607_tmp->raw() : nullptr,_1607_tmp ? _1607_tmp->size(0) : 0);
          _1596_curperm = _1598_tgtperm;
          _ptr__1596_curperm.update(_1596_curperm ? _1596_curperm->raw() : nullptr, _1596_curperm ? _1596_curperm->size(0) : 0);
          _1598_tgtperm = _1607_tmp;
          _ptr__1598_tgtperm.update(_1598_tgtperm ? _1598_tgtperm->raw() : nullptr, _1598_tgtperm ? _1598_tgtperm->size(0) : 0);
        }
      }
      int32_t _9839_ = (int)0;
      int32_t _9840_ = _1586_n;
      for (int32_t _1608_ii = _9839_; _1608_ii < _9840_; ++_1608_ii)
      {
        {
          int32_t _1609_i = _ptr__1596_curperm[_1608_ii];
          int64_t _1610_k = (int)0;
          int32_t _9841_ = (int)0;
          int32_t _9842_ = _1564_nd;
          for (int32_t _1611_j = _9841_; _1611_j < _9842_; ++_1611_j)
          {
            {
              _1610_k += safe_mul( _ptr__1565_strides[_1611_j],((*_1593_midxs)(_1609_i,_1611_j)) );
            }
          }
          _ptr__1584_sparsity[_1608_ii] = _1610_k;
        }
      }
    }
  }
  {}
  int32_t _9843_;
  bool _9844_ = (NULL != _1584_sparsity.get());
  if (_9844_)
  {
    _9843_ = (int)((_1584_sparsity)->size(0));
  }
  else
  {
    _9843_ = _1567_domsize;
  }
  int32_t _1612_nvaritems = _9843_;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1613_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1612_nvaritems)));
  _checked_ptr_1<int32_t> _ptr__1613_nativeidxs(_1613_nativeidxs ? _1613_nativeidxs->raw() : nullptr,_1613_nativeidxs ? _1613_nativeidxs->size(0) : 0);
  int32_t _1614_varblockid = linearvar_alloc(_1612_nvaritems,_1613_nativeidxs);
  if (((int)((_1559_name).size()) > (int)0))
  {
    {
      var_map->setItem(_1559_name,_1614_varblockid);
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1615_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1615_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1559_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1564_nd > (int)0))
      {
        {
          _1615_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9845_ = (int)1;
          int32_t _9846_ = _1564_nd;
          for (int32_t _1616_i = _9845_; _1616_i < _9846_; ++_1616_i)
          {
            {
              _1615_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1616_i);
            }
          }
        }
      }
      {}
      _1615_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames.get()))
      {
        {
          task_format_var_names(_1613_nativeidxs,_1615_sb->toString(),_1563_shape,_1584_sparsity,mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_var_names(_1613_nativeidxs,_1615_sb->toString(),_1563_shape,_1584_sparsity,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _1617_bl;
  _checked_ptr_1<double> _ptr__1617_bl(_1617_bl ? _1617_bl->raw() : nullptr,_1617_bl ? _1617_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1618_bu;
  _checked_ptr_1<double> _ptr__1618_bu(_1618_bu ? _1618_bu->raw() : nullptr,_1618_bu ? _1618_bu->size(0) : 0);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->scalable)
  {
    {
      std::vector<double> _9847_;
      for (int32_t _1619_i = (int)0, _9848_ = _1612_nvaritems; _1619_i < _9848_ ; ++_1619_i)
      {
        _9847_.push_back(((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->lb)((int)0)));
      }
      auto _9849_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9847_.size()),_9847_.begin(),_9847_.end()));
      _1617_bl = _9849_;
      _ptr__1617_bl.update(_1617_bl ? _1617_bl->raw() : nullptr, _1617_bl ? _1617_bl->size(0) : 0);
      std::vector<double> _9850_;
      for (int32_t _1620_i = (int)0, _9851_ = _1612_nvaritems; _1620_i < _9851_ ; ++_1620_i)
      {
        _9850_.push_back(((*mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->ub)((int)0)));
      }
      auto _9852_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9850_.size()),_9850_.begin(),_9850_.end()));
      _1618_bu = _9852_;
      _ptr__1618_bu.update(_1618_bu ? _1618_bu->raw() : nullptr, _1618_bu ? _1618_bu->size(0) : 0);
    }
  }
  else
  {
    {
      _1617_bl = mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->lb;
      _ptr__1617_bl.update(_1617_bl ? _1617_bl->raw() : nullptr, _1617_bl ? _1617_bl->size(0) : 0);
      _1618_bu = mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->ub;
      _ptr__1618_bu.update(_1618_bu ? _1618_bu->raw() : nullptr, _1618_bu ? _1618_bu->size(0) : 0);
    }
  }
  task_var_putboundlist_ra(_1613_nativeidxs,_1617_bl,_1618_bu);
  monty::rc_ptr< ::mosek::fusion::RangedVariable > _1621_res = ::mosek::fusion::p_RangedVariable::_new_RangedVariable(::mosek::fusion::Model::t(_pubthis),_1559_name,(int64_t)(_1614_varblockid),_1563_shape,_1584_sparsity,_1613_nativeidxs);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1562_dom)->cardinal_flag)
  {
    {
      task_var_putintlist(_1613_nativeidxs);
    }
  }
  {}
  ((*vars)(_1614_varblockid)) = (_1621_res).as<::mosek::fusion::ModelVariable>();
  return _1621_res;
}
// End mosek.fusion.Model.ranged_variable

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:2040:6-2307:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1622_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1623_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1624_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1623_shp(_1623_shp ? _1623_shp->raw() : nullptr, _1623_shp ? _1623_shp->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::ConeDomain > _1625_dom = mosek::fusion::p_ConeDomain::_get_impl(_1624_dom_.get())->__mosek_2fusion_2ConeDomain__finalize_and_validate(_1623_shp);
  if ((((int)((_1622_name).size()) > (int)0) && hasVariable(_1622_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1622_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1626_shape = mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1626_shape(_1626_shape ? _1626_shape->raw() : nullptr,_1626_shape ? _1626_shape->size(0) : 0);
  int32_t _1627_nd = (int)((_1626_shape)->size(0));
  int32_t _1628_coneaxis = _1625_dom->getAxis();
  int32_t _1629_conesize = _ptr__1626_shape[_1628_coneaxis];
  int32_t _1630_d1 = (int)1;
  int32_t _1631_d0 = (int)1;
  int32_t _1632_domsize = (int)1;
  int32_t _9853_ = (int)0;
  int32_t _9854_ = _1627_nd;
  for (int32_t _1633_i = _9853_; _1633_i < _9854_; ++_1633_i)
  {
    {
      _1632_domsize *= _ptr__1626_shape[_1633_i];
    }
  }
  int32_t _9855_ = (int)0;
  int32_t _9856_ = _1628_coneaxis;
  for (int32_t _1634_i = _9855_; _1634_i < _9856_; ++_1634_i)
  {
    {
      _1631_d0 *= _ptr__1626_shape[_1634_i];
    }
  }
  int32_t _9857_ = safe_add( _1628_coneaxis,(int)1 );
  int32_t _9858_ = _1627_nd;
  for (int32_t _1635_i = _9857_; _1635_i < _9858_; ++_1635_i)
  {
    {
      _1630_d1 *= _ptr__1626_shape[_1635_i];
    }
  }
  int32_t _1636_numcone = safe_mul( _1631_d0,_1630_d1 );
  int64_t _1637_domainidx = (-(int)1);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1638_domainidxlist = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
  _checked_ptr_1<int64_t> _ptr__1638_domainidxlist(_1638_domainidxlist ? _1638_domainidxlist->raw() : nullptr,_1638_domainidxlist ? _1638_domainidxlist->size(0) : 0);
  if ((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InQCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for quadratic cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_quad(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InRotatedQCone))
  {
    {
      if ((_1629_conesize < (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for rotated quadratic cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_rquad(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPExpCone))
  {
    {
      if ((_1629_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for exponential cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_pexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPPowCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_ppow(_1629_conesize,mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPPowConeSeq))
  {
    {
      _1638_domainidxlist = task_append_domain_ppow_seq(_1629_conesize,((int)((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha)->size(0)) / _1636_numcone),mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha);
      _ptr__1638_domainidxlist.update(_1638_domainidxlist ? _1638_domainidxlist->raw() : nullptr, _1638_domainidxlist ? _1638_domainidxlist->size(0) : 0);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InDExpCone))
  {
    {
      if ((_1629_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual exponential cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_dexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InDPowCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual power cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_dpow(_1629_conesize,mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InDPowConeSeq))
  {
    {
      _1638_domainidxlist = task_append_domain_dpow_seq(_1629_conesize,((int)((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha)->size(0)) / _1636_numcone),mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->alpha);
      _ptr__1638_domainidxlist.update(_1638_domainidxlist ? _1638_domainidxlist->raw() : nullptr, _1638_domainidxlist ? _1638_domainidxlist->size(0) : 0);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPGeoMeanCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for primal geometric mean cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_pgeomean(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InDGeoMeanCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual geometric mean cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_dgeomean(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::Positive))
  {
    {
      _1637_domainidx = task_append_domain_rpos(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InSVecPSDCone))
  {
    {
      int32_t _1639_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_1629_conesize ) ))) - (int)1) / (int)2);
      if (((safe_mul( _1639_d,safe_add( _1639_d,(int)1 ) ) / (int)2) != _1629_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for vector positive semidefinite cone"))->toString());
        }
      }
      {}
      _1637_domainidx = task_append_domain_svec_psd(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::Negative))
  {
    {
      _1637_domainidx = task_append_domain_rneg(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::Unbounded))
  {
    {
      _1637_domainidx = task_append_domain_r(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::Zero))
  {
    {
      _1637_domainidx = task_append_domain_rzero(_1629_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPSDCone))
  {
    {
      if ((_1629_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
        }
      }
      else
      {
        {
          int32_t _1640_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_1629_conesize ) ))) - (int)1) / (int)2);
          if (((safe_mul( _1640_d,safe_add( _1640_d,(int)1 ) ) / (int)2) != _1629_conesize))
          {
            {
              throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1629_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
            }
          }
          {}
        }
      }
      _1637_domainidx = task_append_domain_svec_psd(_1629_conesize);
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal error: Invalid cone type"));
    }
  }
  if (((mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InPPowConeSeq) || (mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->key == ::mosek::fusion::QConeKey::InDPowConeSeq)))
  {
    {
    }
  }
  else
  {
    {
      std::vector<int64_t> _9859_;
      for (int32_t _1641_i = (int)0, _9860_ = _1636_numcone; _1641_i < _9860_ ; ++_1641_i)
      {
        _9859_.push_back(_1637_domainidx);
      }
      auto _9861_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9859_.size()),_9859_.begin(),_9859_.end()));
      _1638_domainidxlist = _9861_;
      _ptr__1638_domainidxlist.update(_1638_domainidxlist ? _1638_domainidxlist->raw() : nullptr, _1638_domainidxlist ? _1638_domainidxlist->size(0) : 0);
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1642_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( _1636_numcone,_1629_conesize ))));
  _checked_ptr_1<int32_t> _ptr__1642_nativeidxs(_1642_nativeidxs ? _1642_nativeidxs->raw() : nullptr,_1642_nativeidxs ? _1642_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1643_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1636_numcone)));
  _checked_ptr_1<int32_t> _ptr__1643_accidxs(_1643_accidxs ? _1643_accidxs->raw() : nullptr,_1643_accidxs ? _1643_accidxs->size(0) : 0);
  int32_t _1644_varid = conicvar_alloc(_1638_domainidxlist,_1629_conesize,_1636_numcone,_1643_accidxs,_1642_nativeidxs);
  task_var_putboundlist_fr(_1642_nativeidxs);
  if (mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->int_flag)
  {
    {
      task_var_putintlist(_1642_nativeidxs);
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1645_permidxs = _1642_nativeidxs;
  _checked_ptr_1<int32_t> _ptr__1645_permidxs(_1645_permidxs ? _1645_permidxs->raw() : nullptr,_1645_permidxs ? _1645_permidxs->size(0) : 0);
  if ((_1628_coneaxis < ((int)((_1626_shape)->size(0)) - (int)1)))
  {
    {
      _1645_permidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)((_1642_nativeidxs)->size(0)))));
      _ptr__1645_permidxs.update(_1645_permidxs ? _1645_permidxs->raw() : nullptr, _1645_permidxs ? _1645_permidxs->size(0) : 0);
      int32_t _1646_p = (int)0;
      int32_t _9862_ = (int)0;
      int32_t _9863_ = _1631_d0;
      for (int32_t _1647_i0 = _9862_; _1647_i0 < _9863_; ++_1647_i0)
      {
        {
          int32_t _9864_ = (int)0;
          int32_t _9865_ = _1630_d1;
          for (int32_t _1648_i1 = _9864_; _1648_i1 < _9865_; ++_1648_i1)
          {
            {
              int32_t _9866_ = (int)0;
              int32_t _9867_ = _1629_conesize;
              for (int32_t _1649_k = _9866_; _1649_k < _9867_; ++_1649_k)
              {
                {
                  _ptr__1645_permidxs[_1646_p] = _ptr__1642_nativeidxs[safe_add( safe_mul( safe_add( safe_mul( _1647_i0,_1629_conesize ),_1649_k ),_1630_d1 ),_1648_i1 )];
                  ++ _1646_p;
                }
              }
            }
          }
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicVariable > _1650_res = ::mosek::fusion::p_ConicVariable::_new_ConicVariable(::mosek::fusion::Model::t(_pubthis),_1622_name,_1644_varid,_1626_shape,_1642_nativeidxs);
  ((*vars)(_1644_varid)) = (_1650_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1622_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1651_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      {
        _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1622_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
        if ((_1627_nd > (int)0))
        {
          {
            _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
            int32_t _9868_ = (int)1;
            int32_t _9869_ = _1627_nd;
            for (int32_t _1652_i = _9868_; _1652_i < _9869_; ++_1652_i)
            {
              {
                _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1652_i);
              }
            }
          }
        }
        {}
        _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
        if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->indexnames.get()))
        {
          {
            task_format_var_names(_1645_permidxs,_1651_sb->toString(),_1626_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->indexnames);
          }
        }
        else
        {
          {
            task_format_var_names(_1645_permidxs,_1651_sb->toString(),_1626_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
          }
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1653_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1636_numcone)));
      _checked_ptr_1<int32_t> _ptr__1653_accidxs(_1653_accidxs ? _1653_accidxs->raw() : nullptr,_1653_accidxs ? _1653_accidxs->size(0) : 0);
      acc_block_map->get(((*var_block_acc_id)(_1644_varid)),_1653_accidxs,(int)0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _1654_accshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_1626_shape)->size(0)) - (int)1))));
        _checked_ptr_1<int32_t> _ptr__1654_accshape(_1654_accshape ? _1654_accshape->raw() : nullptr,_1654_accshape ? _1654_accshape->size(0) : 0);
        int32_t _9870_ = (int)0;
        int32_t _9871_ = _1628_coneaxis;
        for (int32_t _1655_i = _9870_; _1655_i < _9871_; ++_1655_i)
        {
          {
            _ptr__1654_accshape[_1655_i] = _ptr__1626_shape[_1655_i];
          }
        }
        int32_t _9872_ = safe_add( _1628_coneaxis,(int)1 );
        int32_t _9873_ = _1627_nd;
        for (int32_t _1656_i = _9872_; _1656_i < _9873_; ++_1656_i)
        {
          {
            _ptr__1654_accshape[(_1656_i - (int)1)] = _ptr__1626_shape[_1656_i];
          }
        }
        monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1657_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
        _1657_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1622_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
        if (((_1627_nd - (int)1) > (int)0))
        {
          {
            _1657_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
            int32_t _9874_ = (int)1;
            int32_t _9875_ = (_1627_nd - (int)1);
            for (int32_t _1658_i = _9874_; _1658_i < _9875_; ++_1658_i)
            {
              {
                _1657_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1658_i);
              }
            }
          }
        }
        {}
        _1657_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
        if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->indexnames.get()))
        {
          {
            std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1659_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1627_nd - (int)1))));
            int32_t _9876_ = (int)0;
            int32_t _9877_ = _1628_coneaxis;
            for (int32_t _1660_i = _9876_; _1660_i < _9877_; ++_1660_i)
            {
              {
                ((*_1659_indexnames)(_1660_i)) = ((*mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->indexnames)(_1660_i));
              }
            }
            int32_t _9878_ = safe_add( _1628_coneaxis,(int)1 );
            int32_t _9879_ = _1627_nd;
            for (int32_t _1661_i = _9878_; _1661_i < _9879_; ++_1661_i)
            {
              {
                ((*_1659_indexnames)((_1661_i - (int)1))) = ((*mosek::fusion::p_ConeDomain::_get_impl(_1625_dom)->indexnames)(_1661_i));
              }
            }
            std::vector<int64_t> _9880_;
            for (int32_t _1662_i = (int)0, _9881_ = _1636_numcone; _1662_i < _9881_ ; ++_1662_i)
            {
              _9880_.push_back((int64_t)_ptr__1653_accidxs[_1662_i]);
            }
            auto _9882_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9880_.size()),_9880_.begin(),_9880_.end()));
            task_format_acc_names(_9882_,_1657_sb->toString(),_1654_accshape,_1659_indexnames);
          }
        }
        else
        {
          {
            std::vector<int64_t> _9883_;
            for (int32_t _1663_i = (int)0, _9884_ = _1636_numcone; _1663_i < _9884_ ; ++_1663_i)
            {
              _9883_.push_back((int64_t)_ptr__1653_accidxs[_1663_i]);
            }
            auto _9885_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9883_.size()),_9883_.begin(),_9883_.end()));
            task_format_acc_names(_9885_,_1657_sb->toString(),_1654_accshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
          }
        }
      }
      var_map->setItem(_1622_name,_1644_varid);
    }
  }
  {}
  return (_1650_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1849:6-2035:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1664_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1665_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1666_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1665_shp(_1665_shp ? _1665_shp->raw() : nullptr, _1665_shp ? _1665_shp->size(0) : 0);
  if ((((int)((_1664_name).size()) > (int)0) && hasVariable(_1664_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1664_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearDomain > _1667_dom = mosek::fusion::p_LinearDomain::_get_impl(_1666_dom_.get())->__mosek_2fusion_2LinearDomain__finalize_and_validate(_1665_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1668_shape = mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1668_shape(_1668_shape ? _1668_shape->raw() : nullptr,_1668_shape ? _1668_shape->size(0) : 0);
  int32_t _1669_nd = (int)((_1668_shape)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1670_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1669_nd)));
  _checked_ptr_1<int32_t> _ptr__1670_strides(_1670_strides ? _1670_strides->raw() : nullptr,_1670_strides ? _1670_strides->size(0) : 0);
  if ((_1669_nd > (int)0))
  {
    {
      _ptr__1670_strides[(_1669_nd - (int)1)] = (int)1;
      int32_t _9886_ = (int)1;
      int32_t _9887_ = _1669_nd;
      for (int32_t _1671_i = _9886_; _1671_i < _9887_; ++_1671_i)
      {
        {
          _ptr__1670_strides[((_1669_nd - _1671_i) - (int)1)] = safe_mul( _ptr__1670_strides[(_1669_nd - _1671_i)],_ptr__1668_shape[(_1669_nd - _1671_i)] );
        }
      }
    }
  }
  {}
  int32_t _1672_domsize = (int)1;
  int32_t _9888_ = (int)0;
  int32_t _9889_ = (int)((_1668_shape)->size(0));
  for (int32_t _1673_i = _9888_; _1673_i < _9889_; ++_1673_i)
  {
    {
      _1672_domsize *= _ptr__1668_shape[_1673_i];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1674_axisptr = nullptr;
  _checked_ptr_1<int32_t> _ptr__1674_axisptr(_1674_axisptr ? _1674_axisptr->raw() : nullptr,_1674_axisptr ? _1674_axisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1675_nameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1675_nameptr(_1675_nameptr ? _1675_nameptr->raw() : nullptr,_1675_nameptr ? _1675_nameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1676_axisnames = nullptr;
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1677_accaxisptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1677_accaxisptr(_1677_accaxisptr ? _1677_accaxisptr->raw() : nullptr,_1677_accaxisptr ? _1677_accaxisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1678_accnameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1678_accnameptr(_1678_accnameptr ? _1678_accnameptr->raw() : nullptr,_1678_accnameptr ? _1678_accnameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1679_accaxisnames = nullptr;
  if ((((int)((_1664_name).size()) > (int)0) && (NULL != mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames.get())))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1680_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1681_accsb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      int32_t _1682_axisp = (int)0;
      int64_t _1683_namep = (int)0;
      int64_t _1684_numnames = (int)0;
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)->size(0)) != _1669_nd))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
        }
      }
      {}
      int32_t _9890_ = (int)0;
      int32_t _9891_ = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)->size(0));
      for (int32_t _1685_i = _9890_; _1685_i < _9891_; ++_1685_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)(_1685_i)).get()))
          {
            {
              if ((_ptr__1668_shape[_1685_i] != (int)((((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)(_1685_i)))->size(0))))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
                }
              }
              {}
              _1684_numnames += _ptr__1668_shape[_1685_i];
            }
          }
          {}
        }
      }
      _1675_nameptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1684_numnames,(int)1 ))));
      _ptr__1675_nameptr.update(_1675_nameptr ? _1675_nameptr->raw() : nullptr, _1675_nameptr ? _1675_nameptr->size(0) : 0);
      std::vector<int32_t> _9892_;
      for (int32_t _1686_i = (int)0, _9893_ = _1669_nd; _1686_i < _9893_ ; ++_1686_i)
      {
        _9892_.push_back((-(int)1));
      }
      auto _9894_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9892_.size()),_9892_.begin(),_9892_.end()));
      _1674_axisptr = _9894_;
      _ptr__1674_axisptr.update(_1674_axisptr ? _1674_axisptr->raw() : nullptr, _1674_axisptr ? _1674_axisptr->size(0) : 0);
      int32_t _9895_ = (int)0;
      int32_t _9896_ = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)->size(0));
      for (int32_t _1687_i = _9895_; _1687_i < _9896_; ++_1687_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)(_1687_i)).get()))
          {
            {
              _ptr__1674_axisptr[_1687_i] = _1682_axisp;
              int32_t _9897_ = (int)0;
              int32_t _9898_ = _ptr__1668_shape[_1687_i];
              for (int32_t _1688_j = _9897_; _1688_j < _9898_; ++_1688_j)
              {
                {
                  _1680_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)(_1687_i)))(_1688_j)));
                  ++ _1682_axisp;
                  _1683_namep += (int)((((*((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames)(_1687_i)))(_1688_j))).size());
                  _ptr__1675_nameptr[_1682_axisp] = _1683_namep;
                }
              }
            }
          }
          {}
        }
      }
      _1676_axisnames = _1680_sb;
      _1679_accaxisnames = _1681_accsb;
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1689_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__1689_sparsity(_1689_sparsity ? _1689_sparsity->raw() : nullptr,_1689_sparsity ? _1689_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1690_perm = nullptr;
  _checked_ptr_1<int32_t> _ptr__1690_perm(_1690_perm ? _1690_perm->raw() : nullptr,_1690_perm ? _1690_perm->size(0) : 0);
  if (mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->empty)
  {
    {
      _1689_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
      _ptr__1689_sparsity.update(_1689_sparsity ? _1689_sparsity->raw() : nullptr, _1689_sparsity ? _1689_sparsity->size(0) : 0);
    }
  }
  else if((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->sparsity.get()))
  {
    {
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->sparsity)->size(1)) != _1669_nd))
      {
        {
          throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is not correct"));
        }
      }
      {}
      int32_t _1691_n = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->sparsity)->size(0));
      _1689_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1691_n)));
      _ptr__1689_sparsity.update(_1689_sparsity ? _1689_sparsity->raw() : nullptr, _1689_sparsity ? _1689_sparsity->size(0) : 0);
      {
        bool _1692_err = false;
        std::shared_ptr< monty::ndarray< int32_t,2 > > _1693_sp = mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->sparsity;
        int32_t _9899_ = (int)0;
        int32_t _9900_ = _1691_n;
        for (int32_t _1694_i = _9899_; _1694_i < _9900_; ++_1694_i)
        {
          {
            int32_t _9901_ = (int)0;
            int32_t _9902_ = (int)((_1693_sp)->size(1));
            for (int32_t _1695_j = _9901_; _1695_j < _9902_; ++_1695_j)
            {
              {
                _1692_err = (_1692_err || ((((*_1693_sp)(_1694_i,_1695_j)) < (int)0) || (((*_1693_sp)(_1694_i,_1695_j)) >= _ptr__1668_shape[_1695_j])));
              }
            }
            if ((!_1692_err))
            {
              {
                int64_t _1696_linidx = (int)0;
                int32_t _9903_ = (int)0;
                int32_t _9904_ = (int)((_1693_sp)->size(1));
                for (int32_t _1697_j = _9903_; _1697_j < _9904_; ++_1697_j)
                {
                  {
                    _1696_linidx += safe_mul( _ptr__1670_strides[_1697_j],((*_1693_sp)(_1694_i,_1697_j)) );
                  }
                }
                _ptr__1689_sparsity[_1694_i] = _1696_linidx;
              }
            }
            {}
          }
        }
        if (_1692_err)
        {
          {
            throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is out of bounds"));
          }
        }
        {}
      }
      std::shared_ptr< monty::ndarray< int32_t,2 > > _1698_midxs = mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->sparsity;
      int32_t _1699_maxdim = (int)0;
      int32_t _9905_ = (int)0;
      int32_t _9906_ = _1669_nd;
      for (int32_t _1700_i = _9905_; _1700_i < _9906_; ++_1700_i)
      {
        {
          if ((_1699_maxdim < _ptr__1668_shape[_1700_i]))
          {
            {
              _1699_maxdim = _ptr__1668_shape[_1700_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1701_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1691_n)));
      _checked_ptr_1<int32_t> _ptr__1701_curperm(_1701_curperm ? _1701_curperm->raw() : nullptr,_1701_curperm ? _1701_curperm->size(0) : 0);
      int32_t _9907_ = (int)0;
      int32_t _9908_ = _1691_n;
      for (int32_t _1702_i = _9907_; _1702_i < _9908_; ++_1702_i)
      {
        {
          _ptr__1701_curperm[_1702_i] = _1702_i;
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1703_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1691_n)));
      _checked_ptr_1<int32_t> _ptr__1703_tgtperm(_1703_tgtperm ? _1703_tgtperm->raw() : nullptr,_1703_tgtperm ? _1703_tgtperm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1704_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1699_maxdim,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1704_ptr(_1704_ptr ? _1704_ptr->raw() : nullptr,_1704_ptr ? _1704_ptr->size(0) : 0);
      int32_t _9909_ = (int)0;
      int32_t _9910_ = _1669_nd;
      for (int32_t _1705_dd = _9909_; _1705_dd < _9910_; ++_1705_dd)
      {
        {
          {
            int32_t _1706_d = ((_1669_nd - _1705_dd) - (int)1);
            int32_t _9911_ = (int)0;
            int32_t _9912_ = safe_add( _ptr__1668_shape[_1706_d],(int)1 );
            for (int32_t _1707_i = _9911_; _1707_i < _9912_; ++_1707_i)
            {
              {
                _ptr__1704_ptr[_1707_i] = (int)0;
              }
            }
            int32_t _9913_ = (int)0;
            int32_t _9914_ = _1691_n;
            for (int32_t _1708_i = _9913_; _1708_i < _9914_; ++_1708_i)
            {
              {
                ++ _ptr__1704_ptr[safe_add( ((*_1698_midxs)(_1708_i,_1706_d)),(int)1 )];
              }
            }
            int32_t _9915_ = (int)0;
            int32_t _9916_ = _ptr__1668_shape[_1706_d];
            for (int32_t _1709_i = _9915_; _1709_i < _9916_; ++_1709_i)
            {
              {
                _ptr__1704_ptr[safe_add( _1709_i,(int)1 )] += _ptr__1704_ptr[_1709_i];
              }
            }
            int32_t _9917_ = (int)0;
            int32_t _9918_ = _1691_n;
            for (int32_t _1710_ii = _9917_; _1710_ii < _9918_; ++_1710_ii)
            {
              {
                int32_t _1711_i = _ptr__1701_curperm[_1710_ii];
                _ptr__1703_tgtperm[_ptr__1704_ptr[((*_1698_midxs)(_1711_i,_1706_d))]] = _1711_i;
                ++ _ptr__1704_ptr[((*_1698_midxs)(_1711_i,_1706_d))];
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1712_tmp = _1701_curperm;
          _checked_ptr_1<int32_t> _ptr__1712_tmp(_1712_tmp ? _1712_tmp->raw() : nullptr,_1712_tmp ? _1712_tmp->size(0) : 0);
          _1701_curperm = _1703_tgtperm;
          _ptr__1701_curperm.update(_1701_curperm ? _1701_curperm->raw() : nullptr, _1701_curperm ? _1701_curperm->size(0) : 0);
          _1703_tgtperm = _1712_tmp;
          _ptr__1703_tgtperm.update(_1703_tgtperm ? _1703_tgtperm->raw() : nullptr, _1703_tgtperm ? _1703_tgtperm->size(0) : 0);
        }
      }
      int32_t _9919_ = (int)0;
      int32_t _9920_ = _1691_n;
      for (int32_t _1713_ii = _9919_; _1713_ii < _9920_; ++_1713_ii)
      {
        {
          int32_t _1714_i = _ptr__1701_curperm[_1713_ii];
          int64_t _1715_k = (int)0;
          int32_t _9921_ = (int)0;
          int32_t _9922_ = _1669_nd;
          for (int32_t _1716_j = _9921_; _1716_j < _9922_; ++_1716_j)
          {
            {
              _1715_k += safe_mul( _ptr__1670_strides[_1716_j],((*_1698_midxs)(_1714_i,_1716_j)) );
            }
          }
          _ptr__1689_sparsity[_1713_ii] = _1715_k;
        }
      }
    }
  }
  {}
  int32_t _9923_;
  bool _9924_ = (NULL != _1689_sparsity.get());
  if (_9924_)
  {
    _9923_ = (int)((_1689_sparsity)->size(0));
  }
  else
  {
    _9923_ = _1672_domsize;
  }
  int32_t _1717_nvaritems = _9923_;
  std::shared_ptr< monty::ndarray< double,1 > > _1718_bnds = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1717_nvaritems)));
  _checked_ptr_1<double> _ptr__1718_bnds(_1718_bnds ? _1718_bnds->raw() : nullptr,_1718_bnds ? _1718_bnds->size(0) : 0);
  if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->bnd.get()))
  {
    {
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->bnd)->size(0)) == (int)1))
      {
        {
          int32_t _9925_ = (int)0;
          int32_t _9926_ = _1717_nvaritems;
          for (int32_t _1719_i = _9925_; _1719_i < _9926_; ++_1719_i)
          {
            {
              _ptr__1718_bnds[_1719_i] = ((*mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->bnd)((int)0));
            }
          }
        }
      }
      else if(((int)((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->bnd)->size(0)) == _1717_nvaritems))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->bnd,(int)0,_1718_bnds,(int)0,_1717_nvaritems);
        }
      }
      else
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid right hand side shape in domain"));
        }
      }
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1720_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1717_nvaritems)));
  _checked_ptr_1<int32_t> _ptr__1720_nativeidxs(_1720_nativeidxs ? _1720_nativeidxs->raw() : nullptr,_1720_nativeidxs ? _1720_nativeidxs->size(0) : 0);
  int32_t _1721_varblockid = linearvar_alloc(_1717_nvaritems,_1720_nativeidxs);
  if ((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      task_var_putboundlist_fr(_1720_nativeidxs);
    }
  }
  {}
  if ((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      task_var_putboundlist_up(_1720_nativeidxs,_1718_bnds);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      task_var_putboundlist_lo(_1720_nativeidxs,_1718_bnds);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      task_var_putboundlist_fx(_1720_nativeidxs,_1718_bnds);
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearVariable > _1722_res = ::mosek::fusion::p_LinearVariable::_new_LinearVariable(::mosek::fusion::Model::t(_pubthis),_1664_name,(int64_t)(_1721_varblockid),_1668_shape,_1689_sparsity,_1720_nativeidxs);
  ((*vars)(_1721_varblockid)) = (_1722_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1664_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1723_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1723_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1664_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1669_nd > (int)0))
      {
        {
          _1723_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9927_ = (int)1;
          int32_t _9928_ = _1669_nd;
          for (int32_t _1724_i = _9927_; _1724_i < _9928_; ++_1724_i)
          {
            {
              _1723_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1724_i);
            }
          }
        }
      }
      {}
      _1723_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames.get()))
      {
        {
          task_format_var_names(_1720_nativeidxs,_1723_sb->toString(),_1668_shape,_1689_sparsity,mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_var_names(_1720_nativeidxs,_1723_sb->toString(),_1668_shape,_1689_sparsity,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
      var_map->setItem(_1664_name,_1721_varblockid);
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_1667_dom)->cardinal_flag)
  {
    {
      task_var_putintlist(_1720_nativeidxs);
    }
  }
  {}
  return (_1722_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1665:6-1787:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1725_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1726_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1727_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1726_shp(_1726_shp ? _1726_shp->raw() : nullptr, _1726_shp ? _1726_shp->size(0) : 0);
  if ((((int)((_1725_name).size()) > (int)0) && hasVariable(_1725_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1725_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::PSDDomain > _1728_dom = mosek::fusion::p_PSDDomain::_get_impl(_1727_dom_.get())->__mosek_2fusion_2PSDDomain__finalize_and_validate(_1726_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1729_shape = mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1729_shape(_1729_shape ? _1729_shape->raw() : nullptr,_1729_shape ? _1729_shape->size(0) : 0);
  int32_t _1730_conedim1 = mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->conedim1;
  int32_t _1731_conedim2 = mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->conedim2;
  int32_t _1732_nd = (int)((_1729_shape)->size(0));
  int32_t _1733_d = _ptr__1729_shape[_1730_conedim1];
  if ((_1730_conedim1 > _1731_conedim2))
  {
    {
      int32_t _1734_tmp = _1730_conedim1;
      _1730_conedim1 = _1731_conedim2;
      _1731_conedim2 = _1734_tmp;
    }
  }
  {}
  if (((_1730_conedim1 == _1731_conedim2) || ((_1730_conedim1 < (int)0) || (_1731_conedim2 >= _1732_nd))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid cone order dimensions for semidefinite variables"));
    }
  }
  {}
  if ((_ptr__1729_shape[_1730_conedim1] != _ptr__1729_shape[_1731_conedim2]))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching shape in cone order dimensions"));
    }
  }
  {}
  int32_t _1735_numcone = (int)1;
  int32_t _9929_ = (int)0;
  int32_t _9930_ = _1730_conedim1;
  for (int32_t _1736_i = _9929_; _1736_i < _9930_; ++_1736_i)
  {
    {
      _1735_numcone *= _ptr__1729_shape[_1736_i];
    }
  }
  int32_t _9931_ = safe_add( _1730_conedim1,(int)1 );
  int32_t _9932_ = _1731_conedim2;
  for (int32_t _1737_i = _9931_; _1737_i < _9932_; ++_1737_i)
  {
    {
      _1735_numcone *= _ptr__1729_shape[_1737_i];
    }
  }
  int32_t _9933_ = safe_add( _1731_conedim2,(int)1 );
  int32_t _9934_ = _1732_nd;
  for (int32_t _1738_i = _9933_; _1738_i < _9934_; ++_1738_i)
  {
    {
      _1735_numcone *= _ptr__1729_shape[_1738_i];
    }
  }
  int32_t _1739_numelm = (safe_mul( safe_mul( _1735_numcone,_1733_d ),safe_add( _1733_d,(int)1 ) ) / (int)2);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1740_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1739_numelm)));
  _checked_ptr_1<int64_t> _ptr__1740_nativeidxs(_1740_nativeidxs ? _1740_nativeidxs->raw() : nullptr,_1740_nativeidxs ? _1740_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1741_barvaridxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1735_numcone)));
  _checked_ptr_1<int32_t> _ptr__1741_barvaridxs(_1741_barvaridxs ? _1741_barvaridxs->raw() : nullptr,_1741_barvaridxs ? _1741_barvaridxs->size(0) : 0);
  int32_t _1742_varid = barvar_alloc(_1733_d,_1735_numcone,_1741_barvaridxs,_1740_nativeidxs);
  monty::rc_ptr< ::mosek::fusion::PSDVariable > _1743_res = ::mosek::fusion::p_PSDVariable::_new_PSDVariable(::mosek::fusion::Model::t(_pubthis),_1725_name,_1742_varid,_1729_shape,mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->conedim1,mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->conedim2,_1741_barvaridxs,_1740_nativeidxs);
  if (((int)((_1725_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1744_coneshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((_1732_nd - (int)2))));
      _checked_ptr_1<int32_t> _ptr__1744_coneshape(_1744_coneshape ? _1744_coneshape->raw() : nullptr,_1744_coneshape ? _1744_coneshape->size(0) : 0);
      int32_t _9935_;
      bool _9936_ = (_1730_conedim1 < _1731_conedim2);
      if (_9936_)
      {
        _9935_ = _1730_conedim1;
      }
      else
      {
        _9935_ = _1731_conedim2;
      }
      int32_t _1745_cdim1 = _9935_;
      int32_t _9937_;
      bool _9938_ = (_1730_conedim1 < _1731_conedim2);
      if (_9938_)
      {
        _9937_ = _1731_conedim2;
      }
      else
      {
        _9937_ = _1730_conedim1;
      }
      int32_t _1746_cdim2 = _9937_;
      int32_t _9939_ = (int)0;
      int32_t _9940_ = _1745_cdim1;
      for (int32_t _1747_i = _9939_; _1747_i < _9940_; ++_1747_i)
      {
        {
          _ptr__1744_coneshape[_1747_i] = _ptr__1729_shape[_1747_i];
        }
      }
      int32_t _9941_ = safe_add( _1745_cdim1,(int)1 );
      int32_t _9942_ = _1746_cdim2;
      for (int32_t _1748_i = _9941_; _1748_i < _9942_; ++_1748_i)
      {
        {
          _ptr__1744_coneshape[(_1748_i - (int)1)] = _ptr__1729_shape[_1748_i];
        }
      }
      int32_t _9943_ = safe_add( _1746_cdim2,(int)1 );
      int32_t _9944_ = _1732_nd;
      for (int32_t _1749_i = _9943_; _1749_i < _9944_; ++_1749_i)
      {
        {
          _ptr__1744_coneshape[(_1749_i - (int)2)] = _ptr__1729_shape[_1749_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1750_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1750_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1725_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((_1732_nd - (int)2) > (int)0))
      {
        {
          _1750_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9945_ = (int)1;
          int32_t _9946_ = (_1732_nd - (int)2);
          for (int32_t _1751_i = _9945_; _1751_i < _9946_; ++_1751_i)
          {
            {
              _1750_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1751_i);
            }
          }
        }
      }
      {}
      _1750_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1752_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1732_nd - (int)2))));
          int32_t _9947_ = (int)0;
          int32_t _9948_ = _1745_cdim1;
          for (int32_t _1753_i = _9947_; _1753_i < _9948_; ++_1753_i)
          {
            {
              ((*_1752_indexnames)(_1753_i)) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->indexnames)(_1753_i));
            }
          }
          int32_t _9949_ = safe_add( _1745_cdim1,(int)1 );
          int32_t _9950_ = _1746_cdim2;
          for (int32_t _1754_i = _9949_; _1754_i < _9950_; ++_1754_i)
          {
            {
              ((*_1752_indexnames)((_1754_i - (int)1))) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->indexnames)(_1754_i));
            }
          }
          int32_t _9951_ = safe_add( _1746_cdim2,(int)1 );
          int32_t _9952_ = _1732_nd;
          for (int32_t _1755_i = _9951_; _1755_i < _9952_; ++_1755_i)
          {
            {
              ((*_1752_indexnames)((_1755_i - (int)2))) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1728_dom)->indexnames)(_1755_i));
            }
          }
          task_format_barvar_names(_1741_barvaridxs,_1750_sb->toString(),_1744_coneshape,_1752_indexnames);
        }
      }
      else
      {
        {
          task_format_barvar_names(_1741_barvaridxs,_1750_sb->toString(),_1744_coneshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  ((*barvars)(_1742_varid)) = (_1743_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1725_name).size()) > (int)0))
  {
    {
      var_map->setItem(_1725_name,(-safe_add( _1742_varid,(int)1 )));
    }
  }
  {}
  return (_1743_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.putfrows
// Method mosek.fusion.Model.putfrows @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1534:6-1658:7
void mosek::fusion::p_Model::putfrows(std::shared_ptr< monty::ndarray< int32_t,1 > > _1756_nativeidxs,int32_t _1757_nativebaseptr,monty::rc_ptr< ::mosek::fusion::WorkStack > _1758_rs,int32_t _1759_nelem,int32_t _1760_nnz,int32_t _1761_ptr_base,int32_t _1762_nidxs_base,int32_t _1763_cof_base)
{
  _checked_ptr_1<int32_t> _ptr__1756_nativeidxs(_1756_nativeidxs ? _1756_nativeidxs->raw() : nullptr, _1756_nativeidxs ? _1756_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1764_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1758_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1764_ri32(_1764_ri32 ? _1764_ri32->raw() : nullptr,_1764_ri32 ? _1764_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1765_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1758_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1765_ri64(_1765_ri64 ? _1765_ri64->raw() : nullptr,_1765_ri64 ? _1765_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1766_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1758_rs)->f64;
  _checked_ptr_1<double> _ptr__1766_rf64(_1766_rf64 ? _1766_rf64->raw() : nullptr,_1766_rf64 ? _1766_rf64->size(0) : 0);
  int32_t _1767_linnz = (int)0;
  int32_t _1768_barnnz = (int)0;
  int32_t _1769_totalpsdelm = (int)0;
  int32_t _9953_ = (int)0;
  int32_t _9954_ = _1759_nelem;
  for (int32_t _1770_i = _9953_; _1770_i < _9954_; ++_1770_i)
  {
    {
      int32_t _1771_ps = _ptr__1764_ri32[safe_add( _1761_ptr_base,_1770_i )];
      int32_t _1772_pe = _ptr__1764_ri32[safe_add( safe_add( _1761_ptr_base,_1770_i ),(int)1 )];
      int32_t _1773_p = _1771_ps;
      while ( ((_1773_p < _1772_pe) && (_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1773_p )] < (int)0)) )
      {
        {
          int32_t _1774_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1773_p )],(int)1 ))));
          int32_t _1775_p0 = _1773_p;
          ++ _1773_p;
          while ( ((_1773_p < _1772_pe) && ((_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1773_p )] < (int)0) && (_1774_barj == ((*barvar_block_elm_barj)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1773_p )],(int)1 ))))))) )
          {
            {
              ++ _1773_p;
            }
          }
          ++ _1768_barnnz;
        }
      }
      _1769_totalpsdelm += (_1773_p - _1771_ps);
      while ( ((_1773_p < _1772_pe) && (_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1773_p )] == (int)0)) )
      {
        {
          ++ _1773_p;
        }
      }
      _1767_linnz += (_1772_pe - _1773_p);
    }
  }
  if ((_1759_nelem > ((int)((_1756_nativeidxs)->size(0)) - _1757_nativebaseptr)))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal: Mismatching sizes"));
    }
  }
  {}
  int32_t _1776_numcon = _1759_nelem;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1777_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1776_numcon,(int)1 ))));
  _checked_ptr_1<int32_t> _ptr__1777_ptrb(_1777_ptrb ? _1777_ptrb->raw() : nullptr,_1777_ptrb ? _1777_ptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1778_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1767_linnz)));
  _checked_ptr_1<int32_t> _ptr__1778_subj(_1778_subj ? _1778_subj->raw() : nullptr,_1778_subj ? _1778_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1779_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1767_linnz)));
  _checked_ptr_1<double> _ptr__1779_cof(_1779_cof ? _1779_cof->raw() : nullptr,_1779_cof ? _1779_cof->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1780_bfix = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1759_nelem)));
  _checked_ptr_1<double> _ptr__1780_bfix(_1780_bfix ? _1780_bfix->raw() : nullptr,_1780_bfix ? _1780_bfix->size(0) : 0);
  {
    int32_t _1781_k = (int)0;
    int32_t _9955_ = (int)0;
    int32_t _9956_ = _1759_nelem;
    for (int32_t _1782_i = _9955_; _1782_i < _9956_; ++_1782_i)
    {
      {
        int32_t _1783_knz = (int)0;
        int32_t _9957_ = _ptr__1764_ri32[safe_add( _1761_ptr_base,_1782_i )];
        int32_t _9958_ = _ptr__1764_ri32[safe_add( safe_add( _1761_ptr_base,_1782_i ),(int)1 )];
        for (int32_t _1784_j = _9957_; _1784_j < _9958_; ++_1784_j)
        {
          {
            if ((_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1784_j )] == (int)0))
            {
              {
                _ptr__1780_bfix[_1782_i] += _ptr__1766_rf64[safe_add( _1763_cof_base,_1784_j )];
              }
            }
            else if((_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1784_j )] > (int)0))
            {
              {
                _ptr__1778_subj[safe_add( _1781_k,_1783_knz )] = (int32_t)_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1784_j )];
                _ptr__1779_cof[safe_add( _1781_k,_1783_knz )] = _ptr__1766_rf64[safe_add( _1763_cof_base,_1784_j )];
                ++ _1783_knz;
              }
            }
            {}
          }
        }
        _ptr__1777_ptrb[safe_add( _1782_i,(int)1 )] = safe_add( _ptr__1777_ptrb[_1782_i],_1783_knz );
        _1781_k += _1783_knz;
      }
    }
  }
  std::vector<int64_t> _9959_;
  for (int32_t _1785_i = (int)0, _9960_ = _1759_nelem; _1785_i < _9960_ ; ++_1785_i)
  {
    _9959_.push_back(_ptr__1756_nativeidxs[safe_add( _1785_i,_1757_nativebaseptr )]);
  }
  auto _9961_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9959_.size()),_9959_.begin(),_9959_.end()));
  task_putafeflist(_9961_,_1777_ptrb,_1778_subj,_1779_cof,_1780_bfix);
  if ((_1768_barnnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1786_subi = _1756_nativeidxs;
      _checked_ptr_1<int32_t> _ptr__1786_subi(_1786_subi ? _1786_subi->raw() : nullptr,_1786_subi ? _1786_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1787_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1776_numcon,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1787_ptr(_1787_ptr ? _1787_ptr->raw() : nullptr,_1787_ptr ? _1787_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1788_barsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1768_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1788_barsubj(_1788_barsubj ? _1788_barsubj->raw() : nullptr,_1788_barsubj ? _1788_barsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1789_matdim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1768_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1789_matdim(_1789_matdim ? _1789_matdim->raw() : nullptr,_1789_matdim ? _1789_matdim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1790_matnnz = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1768_barnnz)));
      _checked_ptr_1<int64_t> _ptr__1790_matnnz(_1790_matnnz ? _1790_matnnz->raw() : nullptr,_1790_matnnz ? _1790_matnnz->size(0) : 0);
      std::vector<int32_t> _9962_;
      for (int32_t _1792_i = (int)0, _9963_ = _1769_totalpsdelm; _1792_i < _9963_ ; ++_1792_i)
      {
        _9962_.push_back((-(int)1));
      }
      auto _9964_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9962_.size()),_9962_.begin(),_9962_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1791_matsubi = _9964_;
      _checked_ptr_1<int32_t> _ptr__1791_matsubi(_1791_matsubi ? _1791_matsubi->raw() : nullptr,_1791_matsubi ? _1791_matsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1793_matsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1769_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1793_matsubj(_1793_matsubj ? _1793_matsubj->raw() : nullptr,_1793_matsubj ? _1793_matsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1794_matcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1769_totalpsdelm)));
      _checked_ptr_1<double> _ptr__1794_matcof(_1794_matcof ? _1794_matcof->raw() : nullptr,_1794_matcof ? _1794_matcof->size(0) : 0);
      int32_t _1795_barnzi = (int)0;
      int32_t _1796_barmatnzi = (int)0;
      int32_t _9965_ = (int)0;
      int32_t _9966_ = _1776_numcon;
      for (int32_t _1797_i = _9965_; _1797_i < _9966_; ++_1797_i)
      {
        {
          int32_t _1798_p = _ptr__1764_ri32[safe_add( _1761_ptr_base,_1797_i )];
          int32_t _1799_pe = _ptr__1764_ri32[safe_add( safe_add( _1761_ptr_base,_1797_i ),(int)1 )];
          while ( ((_1798_p < _1799_pe) && (_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )] < (int)0)) )
          {
            {
              int32_t _1800_p0 = _1798_p;
              int32_t _1801_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )],(int)1 ))));
              while ( ((_1798_p < _1799_pe) && ((_ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )] < (int)0) && (((*barvar_block_elm_barj)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )],(int)1 )))) == _1801_barj))) )
              {
                {
                  _ptr__1791_matsubi[_1796_barmatnzi] = ((*barvar_block_elm_i)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )],(int)1 ))));
                  _ptr__1793_matsubj[_1796_barmatnzi] = ((*barvar_block_elm_j)((-safe_add( _ptr__1765_ri64[safe_add( _1762_nidxs_base,_1798_p )],(int)1 ))));
                  if ((_ptr__1791_matsubi[_1796_barmatnzi] == _ptr__1793_matsubj[_1796_barmatnzi]))
                  {
                    {
                      _ptr__1794_matcof[_1796_barmatnzi] = _ptr__1766_rf64[safe_add( _1763_cof_base,_1798_p )];
                    }
                  }
                  else
                  {
                    {
                      _ptr__1794_matcof[_1796_barmatnzi] = (0.5 * _ptr__1766_rf64[safe_add( _1763_cof_base,_1798_p )]);
                    }
                  }
                  ++ _1796_barmatnzi;
                  ++ _1798_p;
                }
              }
              _ptr__1788_barsubj[_1795_barnzi] = _1801_barj;
              _ptr__1789_matdim[_1795_barnzi] = ((*barvar_dim)(_1801_barj));
              _ptr__1790_matnnz[_1795_barnzi] = (_1798_p - _1800_p0);
              ++ _1795_barnzi;
            }
          }
          _1798_p = _1799_pe;
          _ptr__1787_ptr[safe_add( _1797_i,(int)1 )] = _1795_barnzi;
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1802_symmatidx = task_appendsymmatlist(_1789_matdim,_1790_matnnz,_1791_matsubi,_1793_matsubj,_1794_matcof);
      _checked_ptr_1<int64_t> _ptr__1802_symmatidx(_1802_symmatidx ? _1802_symmatidx->raw() : nullptr,_1802_symmatidx ? _1802_symmatidx->size(0) : 0);
      task_putafebarfrowlist(_1786_subi,_1787_ptr,_1788_barsubj,_1802_symmatidx);
    }
  }
  {}
}
// End mosek.fusion.Model.putfrows

// Begin mosek.fusion.Model.putarows
// Method mosek.fusion.Model.putarows @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1417:6-1531:7
void mosek::fusion::p_Model::putarows(std::shared_ptr< monty::ndarray< int32_t,1 > > _1803_nativeidxs,monty::rc_ptr< ::mosek::fusion::WorkStack > _1804_rs,int32_t _1805_nelem,int32_t _1806_nnz,int32_t _1807_ptr_base,int32_t _1808_nidxs_base,int32_t _1809_cof_base)
{
  _checked_ptr_1<int32_t> _ptr__1803_nativeidxs(_1803_nativeidxs ? _1803_nativeidxs->raw() : nullptr, _1803_nativeidxs ? _1803_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1810_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1804_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1810_ri32(_1810_ri32 ? _1810_ri32->raw() : nullptr,_1810_ri32 ? _1810_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1811_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1804_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1811_ri64(_1811_ri64 ? _1811_ri64->raw() : nullptr,_1811_ri64 ? _1811_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1812_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1804_rs)->f64;
  _checked_ptr_1<double> _ptr__1812_rf64(_1812_rf64 ? _1812_rf64->raw() : nullptr,_1812_rf64 ? _1812_rf64->size(0) : 0);
  int32_t _1813_linnz = (int)0;
  int32_t _1814_barnnz = (int)0;
  int32_t _1815_totalpsdelm = (int)0;
  int32_t _9967_ = (int)0;
  int32_t _9968_ = _1805_nelem;
  for (int32_t _1816_i = _9967_; _1816_i < _9968_; ++_1816_i)
  {
    {
      int32_t _1817_ps = _ptr__1810_ri32[safe_add( _1807_ptr_base,_1816_i )];
      int32_t _1818_pe = _ptr__1810_ri32[safe_add( safe_add( _1807_ptr_base,_1816_i ),(int)1 )];
      int32_t _1819_p = _1817_ps;
      while ( ((_1819_p < _1818_pe) && (_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1819_p )] < (int)0)) )
      {
        {
          int32_t _1820_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1819_p )],(int)1 ))));
          int32_t _1821_p0 = _1819_p;
          ++ _1819_p;
          while ( ((_1819_p < _1818_pe) && ((_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1819_p )] < (int)0) && (_1820_barj == ((*barvar_block_elm_barj)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1819_p )],(int)1 ))))))) )
          {
            {
              ++ _1819_p;
            }
          }
          ++ _1814_barnnz;
        }
      }
      _1815_totalpsdelm += (_1819_p - _1817_ps);
      while ( (_1819_p < _1818_pe) )
      {
        {
          if ((0.0 != _ptr__1812_rf64[safe_add( _1809_cof_base,_1819_p )]))
          {
            {
              _1813_linnz += (int)1;
            }
          }
          {}
          ++ _1819_p;
        }
      }
    }
  }
  int32_t _1822_numcon = (int)((_1803_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1823_ptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1822_numcon,(int)1 ))));
  _checked_ptr_1<int64_t> _ptr__1823_ptrb(_1823_ptrb ? _1823_ptrb->raw() : nullptr,_1823_ptrb ? _1823_ptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1824_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1813_linnz)));
  _checked_ptr_1<int32_t> _ptr__1824_subj(_1824_subj ? _1824_subj->raw() : nullptr,_1824_subj ? _1824_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1825_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1813_linnz)));
  _checked_ptr_1<double> _ptr__1825_cof(_1825_cof ? _1825_cof->raw() : nullptr,_1825_cof ? _1825_cof->size(0) : 0);
  {
    int32_t _1826_k = (int)0;
    int32_t _9969_ = (int)0;
    int32_t _9970_ = _1805_nelem;
    for (int32_t _1827_i = _9969_; _1827_i < _9970_; ++_1827_i)
    {
      {
        int32_t _1828_knz = (int)0;
        int32_t _9971_ = _ptr__1810_ri32[safe_add( _1807_ptr_base,_1827_i )];
        int32_t _9972_ = _ptr__1810_ri32[safe_add( safe_add( _1807_ptr_base,_1827_i ),(int)1 )];
        for (int32_t _1829_j = _9971_; _1829_j < _9972_; ++_1829_j)
        {
          {
            if (((_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1829_j )] >= (int)0) && ((int)0 != _ptr__1812_rf64[safe_add( _1809_cof_base,_1829_j )])))
            {
              {
                _ptr__1824_subj[safe_add( _1826_k,_1828_knz )] = (int32_t)_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1829_j )];
                _ptr__1825_cof[safe_add( _1826_k,_1828_knz )] = _ptr__1812_rf64[safe_add( _1809_cof_base,_1829_j )];
                ++ _1828_knz;
              }
            }
            {}
          }
        }
        _ptr__1823_ptrb[safe_add( _1827_i,(int)1 )] = safe_add( _ptr__1823_ptrb[_1827_i],_1828_knz );
        _1826_k += _1828_knz;
      }
    }
  }
  task_putarowlist(_1803_nativeidxs,_1823_ptrb,_1824_subj,_1825_cof);
  if ((_1814_barnnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1830_subi = _1803_nativeidxs;
      _checked_ptr_1<int32_t> _ptr__1830_subi(_1830_subi ? _1830_subi->raw() : nullptr,_1830_subi ? _1830_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1831_ptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1822_numcon,(int)1 ))));
      _checked_ptr_1<int64_t> _ptr__1831_ptr(_1831_ptr ? _1831_ptr->raw() : nullptr,_1831_ptr ? _1831_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1832_barsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1814_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1832_barsubj(_1832_barsubj ? _1832_barsubj->raw() : nullptr,_1832_barsubj ? _1832_barsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1833_matdim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1814_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1833_matdim(_1833_matdim ? _1833_matdim->raw() : nullptr,_1833_matdim ? _1833_matdim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1834_matnnz = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1814_barnnz)));
      _checked_ptr_1<int64_t> _ptr__1834_matnnz(_1834_matnnz ? _1834_matnnz->raw() : nullptr,_1834_matnnz ? _1834_matnnz->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1835_matsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1815_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1835_matsubi(_1835_matsubi ? _1835_matsubi->raw() : nullptr,_1835_matsubi ? _1835_matsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1836_matsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1815_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1836_matsubj(_1836_matsubj ? _1836_matsubj->raw() : nullptr,_1836_matsubj ? _1836_matsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1837_matcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1815_totalpsdelm)));
      _checked_ptr_1<double> _ptr__1837_matcof(_1837_matcof ? _1837_matcof->raw() : nullptr,_1837_matcof ? _1837_matcof->size(0) : 0);
      int32_t _1838_barnzi = (int)0;
      int32_t _1839_barmatnzi = (int)0;
      int32_t _9973_ = (int)0;
      int32_t _9974_ = _1822_numcon;
      for (int32_t _1840_i = _9973_; _1840_i < _9974_; ++_1840_i)
      {
        {
          int32_t _1841_p = _ptr__1810_ri32[safe_add( _1807_ptr_base,_1840_i )];
          int32_t _1842_pe = _ptr__1810_ri32[safe_add( safe_add( _1807_ptr_base,_1840_i ),(int)1 )];
          while ( ((_1841_p < _1842_pe) && (_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )] < (int)0)) )
          {
            {
              int32_t _1843_p0 = _1841_p;
              int32_t _1844_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )],(int)1 ))));
              while ( ((_1841_p < _1842_pe) && ((_ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )] < (int)0) && (((*barvar_block_elm_barj)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )],(int)1 )))) == _1844_barj))) )
              {
                {
                  _ptr__1835_matsubi[_1839_barmatnzi] = ((*barvar_block_elm_i)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )],(int)1 ))));
                  _ptr__1836_matsubj[_1839_barmatnzi] = ((*barvar_block_elm_j)((-safe_add( _ptr__1811_ri64[safe_add( _1808_nidxs_base,_1841_p )],(int)1 ))));
                  if ((_ptr__1835_matsubi[_1839_barmatnzi] == _ptr__1836_matsubj[_1839_barmatnzi]))
                  {
                    {
                      _ptr__1837_matcof[_1839_barmatnzi] = _ptr__1812_rf64[safe_add( _1809_cof_base,_1841_p )];
                    }
                  }
                  else
                  {
                    {
                      _ptr__1837_matcof[_1839_barmatnzi] = (0.5 * _ptr__1812_rf64[safe_add( _1809_cof_base,_1841_p )]);
                    }
                  }
                  ++ _1839_barmatnzi;
                  ++ _1841_p;
                }
              }
              _ptr__1832_barsubj[_1838_barnzi] = _1844_barj;
              _ptr__1833_matdim[_1838_barnzi] = ((*barvar_dim)(_1844_barj));
              _ptr__1834_matnnz[_1838_barnzi] = (_1841_p - _1843_p0);
              ++ _1838_barnzi;
            }
          }
          _1841_p = _1842_pe;
          _ptr__1831_ptr[safe_add( _1840_i,(int)1 )] = _1838_barnzi;
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1845_symmatidx = task_appendsymmatlist(_1833_matdim,_1834_matnnz,_1835_matsubi,_1836_matsubj,_1837_matcof);
      _checked_ptr_1<int64_t> _ptr__1845_symmatidx(_1845_symmatidx ? _1845_symmatidx->raw() : nullptr,_1845_symmatidx ? _1845_symmatidx->size(0) : 0);
      task_putbararowlist(_1830_subi,_1831_ptr,_1832_barsubj,_1845_symmatidx);
    }
  }
  {}
}
// End mosek.fusion.Model.putarows

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1285:6-1401:7
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1846_name,monty::rc_ptr< ::mosek::fusion::Expression > _1847_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1848_dom_)
{
  if (((mosek::fusion::p_WorkStack::_get_impl(rs)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(rs)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(rs)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack rs not clean"));
    }
  }
  {}
  if (((mosek::fusion::p_WorkStack::_get_impl(ws)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(ws)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(ws)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack ws not clean"));
    }
  }
  {}
  xs->clear();
  if ((((int)((_1846_name).size()) > (int)0) && hasConstraint(_1846_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1846_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_RangeDomain::_get_impl(_1848_dom_)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1849_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1847_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1849_e->recursive_eval(rs,ws,xs,cache_evaluations);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1850_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1850_ri32(_1850_ri32 ? _1850_ri32->raw() : nullptr,_1850_ri32 ? _1850_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1851_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1851_ri64(_1851_ri64 ? _1851_ri64->raw() : nullptr,_1851_ri64 ? _1851_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1852_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1852_rf64(_1852_rf64 ? _1852_rf64->raw() : nullptr,_1852_rf64 ? _1852_rf64->size(0) : 0);
  int32_t _1853_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1854_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1855_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1856_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  int32_t _1857_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  bool _1858_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _1859_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _1860_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _1861_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _1862_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _1863_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _1864_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _9975_;
  for (int32_t _1866_i = (int)0, _9976_ = _1853_nd; _1866_i < _9976_ ; ++_1866_i)
  {
    _9975_.push_back(_ptr__1850_ri32[safe_add( _1854_shape_base,_1866_i )]);
  }
  auto _9977_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9975_.size()),_9975_.begin(),_9975_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1865_shape = _9977_;
  _checked_ptr_1<int32_t> _ptr__1865_shape(_1865_shape ? _1865_shape->raw() : nullptr,_1865_shape ? _1865_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::RangeDomain > _1867_dom = mosek::fusion::p_RangeDomain::_get_impl(_1848_dom_.get())->__mosek_2fusion_2RangeDomain__finalize_and_validate(_1865_shape);
  if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->shape.get()))
  {
    {
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->shape)->size(0)) != _1853_nd))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Mismatching expression and domain"));
        }
      }
      else
      {
        {
          int32_t _9978_ = (int)0;
          int32_t _9979_ = _1853_nd;
          for (int32_t _1868_i = _9978_; _1868_i < _9979_; ++_1868_i)
          {
            {
              if ((((*mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->shape)(_1868_i)) != _ptr__1850_ri32[safe_add( _1854_shape_base,_1868_i )]))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Mismatching expression and domain"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  {}
  int32_t _1869_numcon = (int)1;
  int32_t _9980_ = (int)0;
  int32_t _9981_ = _1853_nd;
  for (int32_t _1870_i = _9980_; _1870_i < _9981_; ++_1870_i)
  {
    {
      _1869_numcon *= _ptr__1850_ri32[safe_add( _1854_shape_base,_1870_i )];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1871_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1869_numcon)));
  _checked_ptr_1<int32_t> _ptr__1871_nativeidxs(_1871_nativeidxs ? _1871_nativeidxs->raw() : nullptr,_1871_nativeidxs ? _1871_nativeidxs->size(0) : 0);
  int32_t _1872_conid = con_allocate(_1871_nativeidxs);
  putarows(_1871_nativeidxs,rs,_1855_nelem,_1856_nnz,_1859_ptr_base,_1860_nidxs_base,_1861_cof_base);
  xs->clear();
  std::shared_ptr< monty::ndarray< double,1 > > _1873_bl = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1869_numcon)));
  _checked_ptr_1<double> _ptr__1873_bl(_1873_bl ? _1873_bl->raw() : nullptr,_1873_bl ? _1873_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1874_bu = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1869_numcon)));
  _checked_ptr_1<double> _ptr__1874_bu(_1874_bu ? _1874_bu->raw() : nullptr,_1874_bu ? _1874_bu->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1875_domlb = mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->lb;
  _checked_ptr_1<double> _ptr__1875_domlb(_1875_domlb ? _1875_domlb->raw() : nullptr,_1875_domlb ? _1875_domlb->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1876_domub = mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->ub;
  _checked_ptr_1<double> _ptr__1876_domub(_1876_domub ? _1876_domub->raw() : nullptr,_1876_domub ? _1876_domub->size(0) : 0);
  bool _1877_domempty = mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->empty;
  bool _1878_domsc = mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->scalable;
  std::shared_ptr< monty::ndarray< int32_t,2 > > _1879_domsp = mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->sparsity;
  {
    if (_1878_domsc)
    {
      {
        int32_t _9982_ = (int)0;
        int32_t _9983_ = _1869_numcon;
        for (int32_t _1880_i = _9982_; _1880_i < _9983_; ++_1880_i)
        {
          {
            _ptr__1873_bl[_1880_i] = _ptr__1875_domlb[(int)0];
            _ptr__1874_bu[_1880_i] = _ptr__1876_domub[(int)0];
          }
        }
      }
    }
    else if(_1877_domempty)
    {
      {
      }
    }
    else if((NULL != _1879_domsp.get()))
    {
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _1881_stride = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1853_nd)));
        _checked_ptr_1<int32_t> _ptr__1881_stride(_1881_stride ? _1881_stride->raw() : nullptr,_1881_stride ? _1881_stride->size(0) : 0);
        _ptr__1881_stride[(_1853_nd - (int)1)] = (int)1;
        int32_t _9984_ = (int)1;
        int32_t _9985_ = _1853_nd;
        for (int32_t _1882_i = _9984_; _1882_i < _9985_; ++_1882_i)
        {
          {
            _ptr__1881_stride[((_1853_nd - _1882_i) - (int)1)] = safe_mul( _ptr__1881_stride[(_1853_nd - _1882_i)],_ptr__1850_ri32[(safe_add( _1854_shape_base,_1853_nd ) - _1882_i)] );
          }
        }
        int32_t _9986_ = (int)0;
        int32_t _9987_ = (int)((_1875_domlb)->size(0));
        for (int32_t _1883_i = _9986_; _1883_i < _9987_; ++_1883_i)
        {
          {
            int32_t _1884_idx = (int)0;
            int32_t _9988_ = (int)0;
            int32_t _9989_ = _1853_nd;
            for (int32_t _1885_j = _9988_; _1885_j < _9989_; ++_1885_j)
            {
              {
                _1884_idx += safe_mul( _ptr__1881_stride[_1885_j],((*_1879_domsp)(_1883_i,_1885_j)) );
              }
            }
            _ptr__1873_bl[_1884_idx] = _ptr__1875_domlb[_1883_i];
            _ptr__1874_bu[_1884_idx] = _ptr__1876_domub[_1883_i];
          }
        }
      }
    }
    else
    {
      {
        int32_t _9990_ = (int)0;
        int32_t _9991_ = _1869_numcon;
        for (int32_t _1886_i = _9990_; _1886_i < _9991_; ++_1886_i)
        {
          {
            _ptr__1873_bl[_1886_i] = _ptr__1875_domlb[_1886_i];
            _ptr__1874_bu[_1886_i] = _ptr__1876_domub[_1886_i];
          }
        }
      }
    }
  }
  task_con_putboundlist_ra(_1871_nativeidxs,_1873_bl,_1874_bu);
  if ((_1857_ncodeatom > (int)0))
  {
    {
      con_blocks->replace_row_code(rs,_1871_nativeidxs,_1859_ptr_base,_1860_nidxs_base,_1862_codeptr,_1863_code,_1864_cconst);
    }
  }
  {}
  if (((int)((_1846_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1887_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1887_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1846_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((int)((_1865_shape)->size(0)) > (int)0))
      {
        {
          _1887_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
        }
      }
      {}
      int32_t _9992_ = (int)1;
      int32_t _9993_ = (int)((_1865_shape)->size(0));
      for (int32_t _1888_i = _9992_; _1888_i < _9993_; ++_1888_i)
      {
        {
          _1887_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1888_i);
        }
      }
      _1887_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->indexnames.get()))
      {
        {
          task_format_con_names(_1871_nativeidxs,_1887_sb->toString(),_1865_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_RangeDomain::_get_impl(_1867_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_con_names(_1871_nativeidxs,_1887_sb->toString(),_1865_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  std::vector<int32_t> _9994_;
  for (int32_t _1890_i = (int)0, _9995_ = _1853_nd; _1890_i < _9995_ ; ++_1890_i)
  {
    _9994_.push_back(_ptr__1850_ri32[safe_add( _1854_shape_base,_1890_i )]);
  }
  auto _9996_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9994_.size()),_9994_.begin(),_9994_.end()));
  monty::rc_ptr< ::mosek::fusion::RangedConstraint > _1889_res = ::mosek::fusion::p_RangedConstraint::_new_RangedConstraint(::mosek::fusion::Model::t(_pubthis),_1846_name,_9996_,_1871_nativeidxs,_1872_conid);
  ((*cons)(_1872_conid)) = (_1889_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1846_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1846_name,_1872_conid);
    }
  }
  {}
  return _1889_res;
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:1078:6-1265:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1891_name,monty::rc_ptr< ::mosek::fusion::Expression > _1892_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1893_dom_)
{
  if (((mosek::fusion::p_WorkStack::_get_impl(rs)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(rs)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(rs)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack rs not clean"));
    }
  }
  {}
  if (((mosek::fusion::p_WorkStack::_get_impl(ws)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(ws)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(ws)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack ws not clean"));
    }
  }
  {}
  xs->clear();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1894_shape = _1892_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__1894_shape(_1894_shape ? _1894_shape->raw() : nullptr,_1894_shape ? _1894_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::PSDDomain > _1895_dom = mosek::fusion::p_PSDDomain::_get_impl(_1893_dom_.get())->__mosek_2fusion_2PSDDomain__finalize_and_validate(_1894_shape);
  bool _1896_tril = (mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim2);
  int32_t _9997_;
  bool _9998_ = (mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim2);
  if (_9998_)
  {
    _9997_ = mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim2;
  }
  else
  {
    _9997_ = mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim1;
  }
  int32_t _1897_conedim1 = _9997_;
  int32_t _9999_;
  bool _10000_ = (mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim2);
  if (_10000_)
  {
    _9999_ = mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim1;
  }
  else
  {
    _9999_ = mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->conedim2;
  }
  int32_t _1898_conedim0 = _9999_;
  monty::rc_ptr< ::mosek::fusion::Expression > _10001_;
  bool _10002_ = (mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->key == ::mosek::fusion::PSDKey::IsSymPSD);
  if (_10002_)
  {
    _10001_ = ::mosek::fusion::p_ExprSymmetrize::_new_ExprSymmetrize(_1898_conedim0,_1897_conedim1,_1892_expr).as<::mosek::fusion::Expression>();
  }
  else
  {
    _10001_ = _1892_expr;
  }
  monty::rc_ptr< ::mosek::fusion::ExprDenseTril > _1899_e = ::mosek::fusion::p_ExprDenseTril::_new_ExprDenseTril(_1898_conedim0,_1897_conedim1,(::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress((::mosek::fusion::p_ExprScaleVecPSD::_new_ExprScaleVecPSD(_1898_conedim0,_1897_conedim1,(::mosek::fusion::p_ExprDenseTril::_new_ExprDenseTril(_1898_conedim0,_1897_conedim1,_10001_)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1899_e->recursive_eval(rs,ws,xs,cache_evaluations);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1900_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1900_ri32(_1900_ri32 ? _1900_ri32->raw() : nullptr,_1900_ri32 ? _1900_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1901_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1901_ri64(_1901_ri64 ? _1901_ri64->raw() : nullptr,_1901_ri64 ? _1901_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1902_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1902_rf64(_1902_rf64 ? _1902_rf64->raw() : nullptr,_1902_rf64 ? _1902_rf64->size(0) : 0);
  int32_t _1903_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1904_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1905_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  bool _1906_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _1907_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  int32_t _1908_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1909_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _1910_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _1911_sp_base = mosek::fusion::p_WorkStack::_get_impl(rs)->sp_base;
  int32_t _1912_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _1913_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _1914_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _1915_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  int32_t _1916_conedim = _ptr__1894_shape[_1897_conedim1];
  int32_t _1917_conesize = (safe_mul( _1916_conedim,safe_add( _1916_conedim,(int)1 ) ) / (int)2);
  int32_t _1918_numcone = (int)1;
  int32_t _10003_ = (int)0;
  int32_t _10004_ = _1903_nd;
  for (int32_t _1919_i = _10003_; _1919_i < _10004_; ++_1919_i)
  {
    {
      if (((_1919_i != _1898_conedim0) && (_1919_i != _1897_conedim1)))
      {
        {
          _1918_numcone *= _ptr__1894_shape[_1919_i];
        }
      }
      {}
    }
  }
  int64_t _1920_domainidx = task_append_domain_svec_psd(_1917_conesize);
  int32_t _1921_domsize = safe_mul( _1917_conesize,_1918_numcone );
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1922_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1921_domsize)));
  _checked_ptr_1<int32_t> _ptr__1922_afeidxs(_1922_afeidxs ? _1922_afeidxs->raw() : nullptr,_1922_afeidxs ? _1922_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1923_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1918_numcone)));
  _checked_ptr_1<int32_t> _ptr__1923_accidxs(_1923_accidxs ? _1923_accidxs->raw() : nullptr,_1923_accidxs ? _1923_accidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1924_conelmidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1921_domsize)));
  _checked_ptr_1<int32_t> _ptr__1924_conelmidx(_1924_conelmidx ? _1924_conelmidx->raw() : nullptr,_1924_conelmidx ? _1924_conelmidx->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1925_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1921_domsize)));
  _checked_ptr_1<double> _ptr__1925_g(_1925_g ? _1925_g->raw() : nullptr,_1925_g ? _1925_g->size(0) : 0);
  std::vector<int64_t> _10005_;
  for (int32_t _1927_i = (int)0, _10006_ = _1918_numcone; _1927_i < _10006_ ; ++_1927_i)
  {
    _10005_.push_back(_1920_domainidx);
  }
  auto _10007_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10005_.size()),_10005_.begin(),_10005_.end()));
  int32_t _1926_conid = acon_allocate(_10007_,_1917_conesize,_1918_numcone,_1925_g,_1922_afeidxs,_1923_accidxs);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1928_aconidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1921_domsize)));
  _checked_ptr_1<int32_t> _ptr__1928_aconidxs(_1928_aconidxs ? _1928_aconidxs->raw() : nullptr,_1928_aconidxs ? _1928_aconidxs->size(0) : 0);
  acon_block_map->get(_1926_conid,_1928_aconidxs,(int)0);
  int64_t _1929_d5 = (int)1;
  int64_t _1930_d4 = _ptr__1894_shape[_1897_conedim1];
  int64_t _1931_d3 = (int)1;
  int64_t _1932_d2 = _ptr__1894_shape[_1898_conedim0];
  int64_t _1933_d1 = (int)1;
  int32_t _10008_ = (int)0;
  int32_t _10009_ = _1898_conedim0;
  for (int32_t _1934_i = _10008_; _1934_i < _10009_; ++_1934_i)
  {
    {
      _1933_d1 *= _ptr__1894_shape[_1934_i];
    }
  }
  int32_t _10010_ = safe_add( _1898_conedim0,(int)1 );
  int32_t _10011_ = _1897_conedim1;
  for (int32_t _1935_i = _10010_; _1935_i < _10011_; ++_1935_i)
  {
    {
      _1931_d3 *= _ptr__1894_shape[_1935_i];
    }
  }
  int32_t _10012_ = safe_add( _1897_conedim1,(int)1 );
  int32_t _10013_ = _1903_nd;
  for (int32_t _1936_i = _10012_; _1936_i < _10013_; ++_1936_i)
  {
    {
      _1929_d5 *= _ptr__1894_shape[_1936_i];
    }
  }
  {
    int32_t _1937_k = (int)0;
    double _1938_s = (1.0 / ::mosek::fusion::Utils::Tools::sqrt((double)((int)2)));
    int32_t _10014_ = (int)0;
    int64_t _10015_ = safe_mul( safe_mul( _1933_d1,_1931_d3 ),_1929_d5 );
    for (int64_t _1939_i1 = _10014_; _1939_i1 < _10015_; ++_1939_i1)
    {
      {
        int32_t _10016_ = (int)0;
        int64_t _10017_ = _1932_d2;
        for (int64_t _1940_i2 = _10016_; _1940_i2 < _10017_; ++_1940_i2)
        {
          {
            ++ _1937_k;
            int64_t _10018_ = safe_add( _1940_i2,(int)1 );
            int64_t _10019_ = _1930_d4;
            for (int64_t _1941_i4 = _10018_; _1941_i4 < _10019_; ++_1941_i4)
            {
              {
                ((*acon_elm_scale)(_ptr__1928_aconidxs[_1937_k])) = _1938_s;
                ++ _1937_k;
              }
            }
          }
        }
      }
    }
  }
  {
    std::shared_ptr< monty::ndarray< int32_t,1 > > _1942_afetrilperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1904_nelem)));
    _checked_ptr_1<int32_t> _ptr__1942_afetrilperm(_1942_afetrilperm ? _1942_afetrilperm->raw() : nullptr,_1942_afetrilperm ? _1942_afetrilperm->size(0) : 0);
    int32_t _1943_k = (int)0;
    if (_1896_tril)
    {
      {
        int32_t _10020_ = (int)0;
        int64_t _10021_ = _1933_d1;
        for (int64_t _1944_i1 = _10020_; _1944_i1 < _10021_; ++_1944_i1)
        {
          {
            int32_t _10022_ = (int)0;
            int64_t _10023_ = _1930_d4;
            for (int64_t _1945_i4 = _10022_; _1945_i4 < _10023_; ++_1945_i4)
            {
              {
                int32_t _10024_ = (int)0;
                int64_t _10025_ = _1931_d3;
                for (int64_t _1946_i3 = _10024_; _1946_i3 < _10025_; ++_1946_i3)
                {
                  {
                    int64_t _10026_ = _1945_i4;
                    int64_t _10027_ = _1932_d2;
                    for (int64_t _1947_i2 = _10026_; _1947_i2 < _10027_; ++_1947_i2)
                    {
                      {
                        int32_t _10028_ = (int)0;
                        int64_t _10029_ = _1929_d5;
                        for (int64_t _1948_i5 = _10028_; _1948_i5 < _10029_; ++_1948_i5)
                        {
                          {
                            int64_t _1949_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1944_i1,_1931_d3 ),_1929_d5 ),safe_mul( _1946_i3,_1929_d5 ) ),_1948_i5 );
                            int64_t _1950_coneofs = safe_add( (safe_mul( _1947_i2,safe_add( _1947_i2,(int)1 ) ) / (int)2),_1945_i4 );
                            _ptr__1942_afetrilperm[safe_add( safe_mul( _1949_coneidx,_1917_conesize ),_1950_coneofs )] = _ptr__1922_afeidxs[_1943_k];
                            ++ _1943_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    else
    {
      {
        int32_t _10030_ = (int)0;
        int64_t _10031_ = _1933_d1;
        for (int64_t _1951_i1 = _10030_; _1951_i1 < _10031_; ++_1951_i1)
        {
          {
            int32_t _10032_ = (int)0;
            int64_t _10033_ = _1930_d4;
            for (int64_t _1952_i4 = _10032_; _1952_i4 < _10033_; ++_1952_i4)
            {
              {
                int32_t _10034_ = (int)0;
                int64_t _10035_ = _1931_d3;
                for (int64_t _1953_i3 = _10034_; _1953_i3 < _10035_; ++_1953_i3)
                {
                  {
                    int32_t _10036_ = (int)0;
                    int64_t _10037_ = safe_add( _1952_i4,(int)1 );
                    for (int64_t _1954_i2 = _10036_; _1954_i2 < _10037_; ++_1954_i2)
                    {
                      {
                        int32_t _10038_ = (int)0;
                        int64_t _10039_ = _1929_d5;
                        for (int64_t _1955_i5 = _10038_; _1955_i5 < _10039_; ++_1955_i5)
                        {
                          {
                            int64_t _1956_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1951_i1,_1931_d3 ),_1929_d5 ),safe_mul( _1953_i3,_1929_d5 ) ),_1955_i5 );
                            int64_t _1957_coneofs = safe_add( (safe_mul( _1954_i2,((safe_mul( (int)2,_1932_d2 ) - _1954_i2) - (int)1) ) / (int)2),_1952_i4 );
                            _ptr__1942_afetrilperm[safe_add( safe_mul( _1956_coneidx,_1917_conesize ),_1957_coneofs )] = _ptr__1922_afeidxs[_1943_k];
                            ++ _1943_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    putfrows(_1942_afetrilperm,(int)0,rs,_1904_nelem,_1905_nnz,_1909_ptr_base,_1910_nidxs_base,_1912_cof_base);
    if ((_1907_ncodeatom > (int)0))
    {
      {
        afe_blocks->replace_row_code(rs,_1942_afetrilperm,_1909_ptr_base,_1910_nidxs_base,_1913_codeptr,_1914_code,_1915_cconst);
      }
    }
    else
    {
      {
      }
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1958_pafeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( safe_mul( _1918_numcone,_1932_d2 ),_1932_d2 ))));
  _checked_ptr_1<int32_t> _ptr__1958_pafeidxs(_1958_pafeidxs ? _1958_pafeidxs->raw() : nullptr,_1958_pafeidxs ? _1958_pafeidxs->size(0) : 0);
  {
    int32_t _1959_k = (int)0;
    if (_1896_tril)
    {
      {
        int32_t _10040_ = (int)0;
        int64_t _10041_ = _1933_d1;
        for (int64_t _1960_i1 = _10040_; _1960_i1 < _10041_; ++_1960_i1)
        {
          {
            int32_t _10042_ = (int)0;
            int64_t _10043_ = _1932_d2;
            for (int64_t _1961_i2 = _10042_; _1961_i2 < _10043_; ++_1961_i2)
            {
              {
                int32_t _10044_ = (int)0;
                int64_t _10045_ = _1931_d3;
                for (int64_t _1962_i3 = _10044_; _1962_i3 < _10045_; ++_1962_i3)
                {
                  {
                    int32_t _10046_ = (int)0;
                    int64_t _10047_ = _1930_d4;
                    for (int64_t _1963_i4 = _10046_; _1963_i4 < _10047_; ++_1963_i4)
                    {
                      {
                        int32_t _10048_ = (int)0;
                        int64_t _10049_ = _1929_d5;
                        for (int64_t _1964_i5 = _10048_; _1964_i5 < _10049_; ++_1964_i5)
                        {
                          {
                            int64_t _1965_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1960_i1,_1931_d3 ),_1929_d5 ),safe_mul( _1962_i3,_1929_d5 ) ),_1964_i5 );
                            int64_t _10050_;
                            bool _10051_ = (_1961_i2 >= _1963_i4);
                            if (_10051_)
                            {
                              _10050_ = safe_add( (safe_mul( _1963_i4,((safe_mul( (int)2,_1932_d2 ) - _1963_i4) - (int)1) ) / (int)2),_1961_i2 );
                            }
                            else
                            {
                              _10050_ = safe_add( (safe_mul( _1961_i2,((safe_mul( (int)2,_1932_d2 ) - _1961_i2) - (int)1) ) / (int)2),_1963_i4 );
                            }
                            int64_t _1966_coneofs = _10050_;
                            _ptr__1958_pafeidxs[_1959_k] = _ptr__1922_afeidxs[safe_add( safe_mul( _1965_coneidx,_1917_conesize ),_1966_coneofs )];
                            ++ _1959_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    else
    {
      {
        int32_t _10052_ = (int)0;
        int64_t _10053_ = _1933_d1;
        for (int64_t _1967_i1 = _10052_; _1967_i1 < _10053_; ++_1967_i1)
        {
          {
            int32_t _10054_ = (int)0;
            int64_t _10055_ = _1932_d2;
            for (int64_t _1968_i2 = _10054_; _1968_i2 < _10055_; ++_1968_i2)
            {
              {
                int32_t _10056_ = (int)0;
                int64_t _10057_ = _1931_d3;
                for (int64_t _1969_i3 = _10056_; _1969_i3 < _10057_; ++_1969_i3)
                {
                  {
                    int32_t _10058_ = (int)0;
                    int64_t _10059_ = _1930_d4;
                    for (int64_t _1970_i4 = _10058_; _1970_i4 < _10059_; ++_1970_i4)
                    {
                      {
                        int32_t _10060_ = (int)0;
                        int64_t _10061_ = _1929_d5;
                        for (int64_t _1971_i5 = _10060_; _1971_i5 < _10061_; ++_1971_i5)
                        {
                          {
                            int64_t _1972_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1967_i1,_1931_d3 ),_1929_d5 ),safe_mul( _1969_i3,_1929_d5 ) ),_1971_i5 );
                            int64_t _10062_;
                            bool _10063_ = (_1968_i2 >= _1970_i4);
                            if (_10063_)
                            {
                              _10062_ = safe_add( (safe_mul( _1970_i4,((safe_mul( (int)2,_1932_d2 ) - _1970_i4) - (int)1) ) / (int)2),_1968_i2 );
                            }
                            else
                            {
                              _10062_ = safe_add( (safe_mul( _1968_i2,((safe_mul( (int)2,_1932_d2 ) - _1968_i2) - (int)1) ) / (int)2),_1970_i4 );
                            }
                            int64_t _1973_coneofs = _10062_;
                            _ptr__1958_pafeidxs[_1959_k] = _ptr__1922_afeidxs[safe_add( safe_mul( _1972_coneidx,_1917_conesize ),_1973_coneofs )];
                            ++ _1959_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  if (((int)((_1891_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1974_coneshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_1894_shape)->size(0)) - (int)2))));
      _checked_ptr_1<int32_t> _ptr__1974_coneshape(_1974_coneshape ? _1974_coneshape->raw() : nullptr,_1974_coneshape ? _1974_coneshape->size(0) : 0);
      int32_t _10064_ = (int)0;
      int32_t _10065_ = _1898_conedim0;
      for (int32_t _1975_i = _10064_; _1975_i < _10065_; ++_1975_i)
      {
        {
          _ptr__1974_coneshape[_1975_i] = _ptr__1894_shape[_1975_i];
        }
      }
      int32_t _10066_ = safe_add( _1898_conedim0,(int)1 );
      int32_t _10067_ = _1897_conedim1;
      for (int32_t _1976_i = _10066_; _1976_i < _10067_; ++_1976_i)
      {
        {
          _ptr__1974_coneshape[(_1976_i - (int)1)] = _ptr__1894_shape[_1976_i];
        }
      }
      int32_t _10068_ = safe_add( _1897_conedim1,(int)1 );
      int32_t _10069_ = _1903_nd;
      for (int32_t _1977_i = _10068_; _1977_i < _10069_; ++_1977_i)
      {
        {
          _ptr__1974_coneshape[(_1977_i - (int)2)] = _ptr__1894_shape[_1977_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1978_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1978_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1891_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1903_nd > (int)2))
      {
        {
          _1978_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _10070_ = (int)1;
          int32_t _10071_ = (_1903_nd - (int)2);
          for (int32_t _1979_i = _10070_; _1979_i < _10071_; ++_1979_i)
          {
            {
              _1978_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1979_i);
            }
          }
        }
      }
      {}
      _1978_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1980_dindexnames = mosek::fusion::p_PSDDomain::_get_impl(_1895_dom)->indexnames;
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1981_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1903_nd - (int)2))));
          int32_t _10072_ = (int)0;
          int32_t _10073_ = _1898_conedim0;
          for (int32_t _1982_i = _10072_; _1982_i < _10073_; ++_1982_i)
          {
            {
              ((*_1981_indexnames)(_1982_i)) = ((*_1980_dindexnames)(_1982_i));
            }
          }
          int32_t _10074_ = safe_add( _1898_conedim0,(int)1 );
          int32_t _10075_ = _1897_conedim1;
          for (int32_t _1983_i = _10074_; _1983_i < _10075_; ++_1983_i)
          {
            {
              ((*_1981_indexnames)((_1983_i - (int)1))) = ((*_1980_dindexnames)(_1983_i));
            }
          }
          int32_t _10076_ = safe_add( _1897_conedim1,(int)1 );
          int32_t _10077_ = _1903_nd;
          for (int32_t _1984_i = _10076_; _1984_i < _10077_; ++_1984_i)
          {
            {
              ((*_1981_indexnames)((_1984_i - (int)2))) = ((*_1980_dindexnames)(_1984_i));
            }
          }
          std::vector<int64_t> _10078_;
          for (int32_t _1985_i = (int)0, _10079_ = (int)((_1923_accidxs)->size(0)); _1985_i < _10079_ ; ++_1985_i)
          {
            _10078_.push_back(_ptr__1923_accidxs[_1985_i]);
          }
          auto _10080_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10078_.size()),_10078_.begin(),_10078_.end()));
          task_format_acc_names(_10080_,_1978_sb->toString(),_1974_coneshape,_1981_indexnames);
        }
      }
      else
      {
        {
          std::vector<int64_t> _10081_;
          for (int32_t _1986_i = (int)0, _10082_ = (int)((_1923_accidxs)->size(0)); _1986_i < _10082_ ; ++_1986_i)
          {
            _10081_.push_back(_ptr__1923_accidxs[_1986_i]);
          }
          auto _10083_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10081_.size()),_10081_.begin(),_10081_.end()));
          task_format_acc_names(_10083_,_1978_sb->toString(),_1974_coneshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicConstraint > _1987_res = ::mosek::fusion::p_ConicConstraint::_new_ConicConstraint(::mosek::fusion::Model::t(_pubthis),_1891_name,nullptr,_1894_shape,_1926_conid,_1958_pafeidxs,nullptr);
  ((*acons)(_1926_conid)) = (_1987_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1891_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1891_name,(-safe_add( (int)1,_1926_conid )));
    }
  }
  {}
  return (_1987_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:850:6-1075:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1988_name,monty::rc_ptr< ::mosek::fusion::Expression > _1989_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1990_dom_)
{
  if (((mosek::fusion::p_WorkStack::_get_impl(rs)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(rs)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(rs)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack rs not clean"));
    }
  }
  {}
  if (((mosek::fusion::p_WorkStack::_get_impl(ws)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(ws)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(ws)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack ws not clean"));
    }
  }
  {}
  xs->clear();
  if ((((int)((_1988_name).size()) > (int)0) && hasConstraint(_1988_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1988_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if ((((int)((_1988_name).size()) > (int)0) && hasConstraint(_1988_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1988_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_ConeDomain::_get_impl(_1990_dom_)->int_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1991_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1989_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1991_e->recursive_eval(rs,ws,xs,cache_evaluations);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1992_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1992_ri32(_1992_ri32 ? _1992_ri32->raw() : nullptr,_1992_ri32 ? _1992_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1993_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1993_ri64(_1993_ri64 ? _1993_ri64->raw() : nullptr,_1993_ri64 ? _1993_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1994_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1994_rf64(_1994_rf64 ? _1994_rf64->raw() : nullptr,_1994_rf64 ? _1994_rf64->size(0) : 0);
  int32_t _1995_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1996_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1997_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1998_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  int32_t _1999_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  bool _2000_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _2001_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _2002_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _2003_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _2004_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _2005_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _2006_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _10084_;
  for (int32_t _2008_i = (int)0, _10085_ = _1995_nd; _2008_i < _10085_ ; ++_2008_i)
  {
    _10084_.push_back(_ptr__1992_ri32[safe_add( _1996_shape_base,_2008_i )]);
  }
  auto _10086_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10084_.size()),_10084_.begin(),_10084_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2007_shape = _10086_;
  _checked_ptr_1<int32_t> _ptr__2007_shape(_2007_shape ? _2007_shape->raw() : nullptr,_2007_shape ? _2007_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::ConeDomain > _2009_dom = mosek::fusion::p_ConeDomain::_get_impl(_1990_dom_.get())->__mosek_2fusion_2ConeDomain__finalize_and_validate(_2007_shape);
  int32_t _10087_;
  bool _10088_ = _2009_dom->axisIsSet();
  if (_10088_)
  {
    _10087_ = _2009_dom->getAxis();
  }
  else
  {
    _10087_ = (_1995_nd - (int)1);
  }
  int32_t _2010_coneaxis = _10087_;
  int32_t _2011_conesize = _ptr__2007_shape[_2010_coneaxis];
  int32_t _2012_d1 = (int)1;
  int32_t _2013_d0 = (int)1;
  int32_t _2014_domsize = (int)1;
  int32_t _10089_ = (int)0;
  int32_t _10090_ = _1995_nd;
  for (int32_t _2015_i = _10089_; _2015_i < _10090_; ++_2015_i)
  {
    {
      _2014_domsize *= _ptr__2007_shape[_2015_i];
    }
  }
  int32_t _10091_ = (int)0;
  int32_t _10092_ = _2010_coneaxis;
  for (int32_t _2016_i = _10091_; _2016_i < _10092_; ++_2016_i)
  {
    {
      _2013_d0 *= _ptr__2007_shape[_2016_i];
    }
  }
  int32_t _10093_ = safe_add( _2010_coneaxis,(int)1 );
  int32_t _10094_ = _1995_nd;
  for (int32_t _2017_i = _10093_; _2017_i < _10094_; ++_2017_i)
  {
    {
      _2012_d1 *= _ptr__2007_shape[_2017_i];
    }
  }
  int32_t _2018_numcone = safe_mul( _2013_d0,_2012_d1 );
  int64_t _2019_domainidx = (-(int)1);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2020_domainidxlist = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
  _checked_ptr_1<int64_t> _ptr__2020_domainidxlist(_2020_domainidxlist ? _2020_domainidxlist->raw() : nullptr,_2020_domainidxlist ? _2020_domainidxlist->size(0) : 0);
  if ((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InQCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for quadratic cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_quad(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InRotatedQCone))
  {
    {
      if ((_2011_conesize < (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for rotated quadratic cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_rquad(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPExpCone))
  {
    {
      if ((_2011_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for exponential cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_pexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPPowCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      if (((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Power cone requires at least two parameters"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_ppow(_2011_conesize,mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPPowConeSeq))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      if ((((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) % _2018_numcone) != (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Incompatible number of cones and number of exponents alpha"))->toString());
        }
      }
      {}
      int32_t _2021_nleftcommon = ((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) / _2018_numcone);
      if ((_2021_nleftcommon > _2011_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid number of exponents ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2021_nleftcommon)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") per power cone of dimension "))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->toString());
        }
      }
      {}
      _2020_domainidxlist = task_append_domain_ppow_seq(_2011_conesize,_2021_nleftcommon,mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha);
      _ptr__2020_domainidxlist.update(_2020_domainidxlist ? _2020_domainidxlist->raw() : nullptr, _2020_domainidxlist ? _2020_domainidxlist->size(0) : 0);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InDExpCone))
  {
    {
      if ((_2011_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual exponential cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_dexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InDPowCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual power cone"))->toString());
        }
      }
      {}
      if (((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Power cone requires at least two parameters"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_dpow(_2011_conesize,mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InDPowConeSeq))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      if ((((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) % _2018_numcone) != (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Incompatible number of cones and number of exponents alpha"))->toString());
        }
      }
      {}
      int32_t _2022_nleftcommon = ((int)((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha)->size(0)) / _2018_numcone);
      if ((_2022_nleftcommon > _2011_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid number of exponents ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2022_nleftcommon)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") per power cone of dimension "))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->toString());
        }
      }
      {}
      _2020_domainidxlist = task_append_domain_dpow_seq(_2011_conesize,_2022_nleftcommon,mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->alpha);
      _ptr__2020_domainidxlist.update(_2020_domainidxlist ? _2020_domainidxlist->raw() : nullptr, _2020_domainidxlist ? _2020_domainidxlist->size(0) : 0);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPGeoMeanCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for primal geometric mean cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_pgeomean(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InDGeoMeanCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual geometric mean cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_dgeomean(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InSVecPSDCone))
  {
    {
      int32_t _2023_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_2011_conesize ) ))) - (int)1) / (int)2);
      if (((safe_mul( _2023_d,safe_add( _2023_d,(int)1 ) ) / (int)2) != _2011_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for vector positive semidefinite cone"))->toString());
        }
      }
      {}
      _2019_domainidx = task_append_domain_svec_psd(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::Positive))
  {
    {
      _2019_domainidx = task_append_domain_rpos(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::Negative))
  {
    {
      _2019_domainidx = task_append_domain_rneg(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::Unbounded))
  {
    {
      _2019_domainidx = task_append_domain_r(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::Zero))
  {
    {
      _2019_domainidx = task_append_domain_rzero(_2011_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPSDCone))
  {
    {
      if ((_2011_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
        }
      }
      else
      {
        {
          int32_t _2024_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_2011_conesize ) ))) - (int)1) / (int)2);
          if (((safe_mul( _2024_d,safe_add( _2024_d,(int)1 ) ) / (int)2) != _2011_conesize))
          {
            {
              throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2011_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
            }
          }
          {}
        }
      }
      _2019_domainidx = task_append_domain_svec_psd(_2011_conesize);
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal error: Invalid cone type"));
    }
  }
  std::shared_ptr< monty::ndarray< double,1 > > _2025_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_2014_domsize)));
  _checked_ptr_1<double> _ptr__2025_g(_2025_g ? _2025_g->raw() : nullptr,_2025_g ? _2025_g->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2026_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2014_domsize)));
  _checked_ptr_1<int32_t> _ptr__2026_afeidxs(_2026_afeidxs ? _2026_afeidxs->raw() : nullptr,_2026_afeidxs ? _2026_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2027_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2018_numcone)));
  _checked_ptr_1<int32_t> _ptr__2027_accidxs(_2027_accidxs ? _2027_accidxs->raw() : nullptr,_2027_accidxs ? _2027_accidxs->size(0) : 0);
  if (((mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InPPowConeSeq) || (mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->key == ::mosek::fusion::QConeKey::InDPowConeSeq)))
  {
    {
    }
  }
  else
  {
    {
      std::vector<int64_t> _10095_;
      for (int32_t _2028_i = (int)0, _10096_ = _2018_numcone; _2028_i < _10096_ ; ++_2028_i)
      {
        _10095_.push_back(_2019_domainidx);
      }
      auto _10097_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10095_.size()),_10095_.begin(),_10095_.end()));
      _2020_domainidxlist = _10097_;
      _ptr__2020_domainidxlist.update(_2020_domainidxlist ? _2020_domainidxlist->raw() : nullptr, _2020_domainidxlist ? _2020_domainidxlist->size(0) : 0);
    }
  }
  int32_t _2029_conid = acon_allocate(_2020_domainidxlist,_2011_conesize,_2018_numcone,_2025_g,_2026_afeidxs,_2027_accidxs);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2030_perm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2014_domsize)));
  _checked_ptr_1<int32_t> _ptr__2030_perm(_2030_perm ? _2030_perm->raw() : nullptr,_2030_perm ? _2030_perm->size(0) : 0);
  {
    int32_t _2031_idx = (int)0;
    std::shared_ptr< monty::ndarray< double,1 > > _2032_ofs = mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->domofs;
    _checked_ptr_1<double> _ptr__2032_ofs(_2032_ofs ? _2032_ofs->raw() : nullptr,_2032_ofs ? _2032_ofs->size(0) : 0);
    {
      int32_t _10098_ = (int)0;
      int32_t _10099_ = _2013_d0;
      for (int32_t _2033_i0 = _10098_; _2033_i0 < _10099_; ++_2033_i0)
      {
        {
          int32_t _10100_ = (int)0;
          int32_t _10101_ = _2012_d1;
          for (int32_t _2034_i1 = _10100_; _2034_i1 < _10101_; ++_2034_i1)
          {
            {
              int32_t _10102_ = (int)0;
              int32_t _10103_ = _2011_conesize;
              for (int32_t _2035_ic = _10102_; _2035_ic < _10103_; ++_2035_ic)
              {
                {
                  _ptr__2030_perm[safe_add( safe_add( safe_mul( safe_mul( _2033_i0,_2012_d1 ),_2011_conesize ),safe_mul( _2035_ic,_2012_d1 ) ),_2034_i1 )] = _2031_idx;
                  ++ _2031_idx;
                }
              }
            }
          }
        }
      }
      if ((NULL != _2032_ofs.get()))
      {
        {
          int32_t _10104_ = (int)0;
          int32_t _10105_ = _2014_domsize;
          for (int32_t _2036_i = _10104_; _2036_i < _10105_; ++_2036_i)
          {
            {
              _ptr__2025_g[_ptr__2030_perm[_2036_i]] = _ptr__2032_ofs[_2036_i];
            }
          }
        }
      }
      {}
    }
  }
  std::vector<int32_t> _10106_;
  for (int32_t _2038_i = (int)0, _10107_ = _2014_domsize; _2038_i < _10107_ ; ++_2038_i)
  {
    _10106_.push_back(_ptr__2026_afeidxs[_ptr__2030_perm[_2038_i]]);
  }
  auto _10108_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10106_.size()),_10106_.begin(),_10106_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2037_pafeidxs = _10108_;
  _checked_ptr_1<int32_t> _ptr__2037_pafeidxs(_2037_pafeidxs ? _2037_pafeidxs->raw() : nullptr,_2037_pafeidxs ? _2037_pafeidxs->size(0) : 0);
  putfrows(_2037_pafeidxs,(int)0,rs,_1997_nelem,_1998_nnz,_2001_ptr_base,_2002_nidxs_base,_2003_cof_base);
  if ((_1999_ncodeatom > (int)0))
  {
    {
      afe_blocks->replace_row_code(rs,_2037_pafeidxs,_2001_ptr_base,_2002_nidxs_base,_2005_codeptr,_2004_code,_2006_cconst);
    }
  }
  {}
  if (((int)((_1988_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2039_accshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_2007_shape)->size(0)) - (int)1))));
      _checked_ptr_1<int32_t> _ptr__2039_accshape(_2039_accshape ? _2039_accshape->raw() : nullptr,_2039_accshape ? _2039_accshape->size(0) : 0);
      int32_t _10109_ = (int)0;
      int32_t _10110_ = _2010_coneaxis;
      for (int32_t _2040_i = _10109_; _2040_i < _10110_; ++_2040_i)
      {
        {
          _ptr__2039_accshape[_2040_i] = _ptr__2007_shape[_2040_i];
        }
      }
      int32_t _10111_ = safe_add( _2010_coneaxis,(int)1 );
      int32_t _10112_ = _1995_nd;
      for (int32_t _2041_i = _10111_; _2041_i < _10112_; ++_2041_i)
      {
        {
          _ptr__2039_accshape[(_2041_i - (int)1)] = _ptr__2007_shape[_2041_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2042_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _2042_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1988_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1995_nd > (int)1))
      {
        {
          _2042_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _10113_ = (int)1;
          int32_t _10114_ = (_1995_nd - (int)1);
          for (int32_t _2043_i = _10113_; _2043_i < _10114_; ++_2043_i)
          {
            {
              _2042_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2043_i);
            }
          }
        }
      }
      {}
      _2042_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2044_dindexnames = mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->indexnames;
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2045_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1995_nd - (int)1))));
          int32_t _10115_ = (int)0;
          int32_t _10116_ = _2010_coneaxis;
          for (int32_t _2046_i = _10115_; _2046_i < _10116_; ++_2046_i)
          {
            {
              ((*_2045_indexnames)(_2046_i)) = ((*_2044_dindexnames)(_2046_i));
            }
          }
          int32_t _10117_ = safe_add( _2010_coneaxis,(int)1 );
          int32_t _10118_ = _1995_nd;
          for (int32_t _2047_i = _10117_; _2047_i < _10118_; ++_2047_i)
          {
            {
              ((*_2045_indexnames)((_2047_i - (int)1))) = ((*_2044_dindexnames)(_2047_i));
            }
          }
          std::vector<int64_t> _10119_;
          for (int32_t _2048_i = (int)0, _10120_ = (int)((_2027_accidxs)->size(0)); _2048_i < _10120_ ; ++_2048_i)
          {
            _10119_.push_back(_ptr__2027_accidxs[_2048_i]);
          }
          auto _10121_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10119_.size()),_10119_.begin(),_10119_.end()));
          task_format_acc_names(_10121_,_2042_sb->toString(),_2039_accshape,_2045_indexnames);
        }
      }
      else
      {
        {
          std::vector<int64_t> _10122_;
          for (int32_t _2049_i = (int)0, _10123_ = (int)((_2027_accidxs)->size(0)); _2049_i < _10123_ ; ++_2049_i)
          {
            _10122_.push_back(_ptr__2027_accidxs[_2049_i]);
          }
          auto _10124_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10122_.size()),_10122_.begin(),_10122_.end()));
          task_format_acc_names(_10124_,_2042_sb->toString(),_2039_accshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicConstraint > _2050_res = ::mosek::fusion::p_ConicConstraint::_new_ConicConstraint(::mosek::fusion::Model::t(_pubthis),_1988_name,_2009_dom,_2007_shape,_2029_conid,_2026_afeidxs,mosek::fusion::p_ConeDomain::_get_impl(_2009_dom)->indexnames);
  ((*acons)(_2029_conid)) = (_2050_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1988_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1988_name,(-safe_add( _2029_conid,(int)1 )));
    }
  }
  {}
  return (_2050_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:720:6-845:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _2051_name,monty::rc_ptr< ::mosek::fusion::Expression > _2052_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _2053_dom_)
{
  if (((mosek::fusion::p_WorkStack::_get_impl(rs)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(rs)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(rs)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack rs not clean"));
    }
  }
  {}
  if (((mosek::fusion::p_WorkStack::_get_impl(ws)->pi32 != (int)0) || ((mosek::fusion::p_WorkStack::_get_impl(ws)->pi64 != (int)0) || (mosek::fusion::p_WorkStack::_get_impl(ws)->pf64 != (int)0))))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Stack ws not clean"));
    }
  }
  {}
  xs->clear();
  if ((((int)((_2051_name).size()) > (int)0) && hasConstraint(_2051_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2051_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_2053_dom_)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _2054_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_2052_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _2054_e->recursive_eval(rs,ws,xs,cache_evaluations);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2055_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__2055_ri32(_2055_ri32 ? _2055_ri32->raw() : nullptr,_2055_ri32 ? _2055_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2056_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__2056_ri64(_2056_ri64 ? _2056_ri64->raw() : nullptr,_2056_ri64 ? _2056_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2057_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__2057_rf64(_2057_rf64 ? _2057_rf64->raw() : nullptr,_2057_rf64 ? _2057_rf64->size(0) : 0);
  int32_t _2058_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _2059_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _2060_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  bool _2061_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _2062_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  int32_t _2063_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _2064_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _2065_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _2066_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _2067_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _2068_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _2069_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _10125_;
  for (int32_t _2071_i = (int)0, _10126_ = _2058_nd; _2071_i < _10126_ ; ++_2071_i)
  {
    _10125_.push_back(_ptr__2055_ri32[safe_add( _2063_shape_base,_2071_i )]);
  }
  auto _10127_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10125_.size()),_10125_.begin(),_10125_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2070_shape = _10127_;
  _checked_ptr_1<int32_t> _ptr__2070_shape(_2070_shape ? _2070_shape->raw() : nullptr,_2070_shape ? _2070_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::LinearDomain > _2072_dom = mosek::fusion::p_LinearDomain::_get_impl(_2053_dom_.get())->__mosek_2fusion_2LinearDomain__finalize_and_validate(_2070_shape);
  int64_t _2073_domsize = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->domsize;
  std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2074_indexnames = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->indexnames;
  std::shared_ptr< monty::ndarray< double,1 > > _2075_b;
  _checked_ptr_1<double> _ptr__2075_b(_2075_b ? _2075_b->raw() : nullptr,_2075_b ? _2075_b->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2076_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2073_domsize)));
  _checked_ptr_1<int32_t> _ptr__2076_nativeidxs(_2076_nativeidxs ? _2076_nativeidxs->raw() : nullptr,_2076_nativeidxs ? _2076_nativeidxs->size(0) : 0);
  int32_t _2077_conid = con_allocate(_2076_nativeidxs);
  putarows(_2076_nativeidxs,rs,_2059_nelem,_2060_nnz,_2064_ptr_base,_2065_nidxs_base,_2066_cof_base);
  {
    std::shared_ptr< monty::ndarray< int32_t,2 > > _2078_domsp = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->sparsity;
    bool _2079_domempty = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->empty;
    std::shared_ptr< monty::ndarray< double,1 > > _2080_dombnd = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->bnd;
    _checked_ptr_1<double> _ptr__2080_dombnd(_2080_dombnd ? _2080_dombnd->raw() : nullptr,_2080_dombnd ? _2080_dombnd->size(0) : 0);
    bool _2081_domscalable = mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->scalable;
    {
      if ((NULL == _2080_dombnd.get()))
      {
        {
          _2075_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2076_nativeidxs)->size(0)))));
          _ptr__2075_b.update(_2075_b ? _2075_b->raw() : nullptr, _2075_b ? _2075_b->size(0) : 0);
        }
      }
      else if(_2081_domscalable)
      {
        {
          std::vector<double> _10128_;
          for (int32_t _2082_i = (int)0, _10129_ = (int)((_2076_nativeidxs)->size(0)); _2082_i < _10129_ ; ++_2082_i)
          {
            _10128_.push_back(_ptr__2080_dombnd[(int)0]);
          }
          auto _10130_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10128_.size()),_10128_.begin(),_10128_.end()));
          _2075_b = _10130_;
          _ptr__2075_b.update(_2075_b ? _2075_b->raw() : nullptr, _2075_b ? _2075_b->size(0) : 0);
        }
      }
      else if((NULL != _2078_domsp.get()))
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2083_stride = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2058_nd)));
          _checked_ptr_1<int32_t> _ptr__2083_stride(_2083_stride ? _2083_stride->raw() : nullptr,_2083_stride ? _2083_stride->size(0) : 0);
          _ptr__2083_stride[(_2058_nd - (int)1)] = (int)1;
          int32_t _10131_ = (int)1;
          int32_t _10132_ = _2058_nd;
          for (int32_t _2084_i = _10131_; _2084_i < _10132_; ++_2084_i)
          {
            {
              _ptr__2083_stride[((_2058_nd - _2084_i) - (int)1)] = safe_mul( _ptr__2083_stride[(_2058_nd - _2084_i)],_ptr__2070_shape[(_2058_nd - _2084_i)] );
            }
          }
          _2075_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2076_nativeidxs)->size(0)))));
          _ptr__2075_b.update(_2075_b ? _2075_b->raw() : nullptr, _2075_b ? _2075_b->size(0) : 0);
          int32_t _10133_ = (int)0;
          int32_t _10134_ = (int)((_2078_domsp)->size(0));
          for (int32_t _2085_i = _10133_; _2085_i < _10134_; ++_2085_i)
          {
            {
              int32_t _2086_idx = (int)0;
              int32_t _10135_ = (int)0;
              int32_t _10136_ = _2058_nd;
              for (int32_t _2087_j = _10135_; _2087_j < _10136_; ++_2087_j)
              {
                {
                  _2086_idx += safe_mul( _ptr__2083_stride[_2087_j],((*_2078_domsp)(_2085_i,_2087_j)) );
                }
              }
              _ptr__2075_b[_2086_idx] = _ptr__2080_dombnd[_2085_i];
            }
          }
        }
      }
      else if(_2079_domempty)
      {
        {
          _2075_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2076_nativeidxs)->size(0)))));
          _ptr__2075_b.update(_2075_b ? _2075_b->raw() : nullptr, _2075_b ? _2075_b->size(0) : 0);
        }
      }
      else
      {
        {
          std::vector<double> _10137_;
          for (int32_t _2088_i = (int)0, _10138_ = (int)((_2076_nativeidxs)->size(0)); _2088_i < _10138_ ; ++_2088_i)
          {
            _10137_.push_back(_ptr__2080_dombnd[_2088_i]);
          }
          auto _10139_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10137_.size()),_10137_.begin(),_10137_.end()));
          _2075_b = _10139_;
          _ptr__2075_b.update(_2075_b ? _2075_b->raw() : nullptr, _2075_b ? _2075_b->size(0) : 0);
        }
      }
    }
  }
  if ((mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      task_con_putboundlist_fr(_2076_nativeidxs);
    }
  }
  {}
  if ((mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      task_con_putboundlist_up(_2076_nativeidxs,_2075_b);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      task_con_putboundlist_lo(_2076_nativeidxs,_2075_b);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      task_con_putboundlist_fx(_2076_nativeidxs,_2075_b);
    }
  }
  {}
  if ((_2062_ncodeatom > (int)0))
  {
    {
      con_blocks->replace_row_code(rs,_2076_nativeidxs,_2064_ptr_base,_2065_nidxs_base,_2067_codeptr,_2068_code,_2069_cconst);
    }
  }
  {}
  if (((int)((_2051_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2089_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _2089_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_2051_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((int)((_2070_shape)->size(0)) > (int)0))
      {
        {
          _2089_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
        }
      }
      {}
      int32_t _10140_ = (int)1;
      int32_t _10141_ = (int)((_2070_shape)->size(0));
      for (int32_t _2090_i = _10140_; _2090_i < _10141_; ++_2090_i)
      {
        {
          _2089_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2090_i);
        }
      }
      _2089_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->indexnames.get()))
      {
        {
          task_format_con_names(_2076_nativeidxs,_2089_sb->toString(),_2070_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_LinearDomain::_get_impl(_2072_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_con_names(_2076_nativeidxs,_2089_sb->toString(),_2070_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearConstraint > _2091_res = ::mosek::fusion::p_LinearConstraint::_new_LinearConstraint(::mosek::fusion::Model::t(_pubthis),_2051_name,_2077_conid,_2070_shape,_2076_nativeidxs,_2074_indexnames);
  ((*cons)(_2077_conid)) = (_2091_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_2051_name).size()) > (int)0))
  {
    {
      con_map->setItem(_2051_name,_2077_conid);
    }
  }
  {}
  return (_2091_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.getVersion
// Method mosek.fusion.Model.getVersion @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:716:6-718:7
std::string mosek::fusion::Model::getVersion
  ( ) { return mosek::fusion::p_Model::getVersion(); }
std::string mosek::fusion::p_Model::getVersion()
{
  return ::mosek::fusion::p_BaseModel::env_getversion();
}
// End mosek.fusion.Model.getVersion

// Begin mosek.fusion.Model.hasParameter
// Method mosek.fusion.Model.hasParameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:711:6-93
bool mosek::fusion::Model :: hasParameter(const std::string & _2092_name) { return mosek::fusion::p_Model::_get_impl(this)->hasParameter(_2092_name); }
bool mosek::fusion::p_Model::hasParameter(const std::string & _2092_name)
{
  return par_map->hasItem(_2092_name);
}
// End mosek.fusion.Model.hasParameter

// Begin mosek.fusion.Model.hasConstraint
// Method mosek.fusion.Model.hasConstraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:710:6-93
bool mosek::fusion::Model :: hasConstraint(const std::string & _2093_name) { return mosek::fusion::p_Model::_get_impl(this)->hasConstraint(_2093_name); }
bool mosek::fusion::p_Model::hasConstraint(const std::string & _2093_name)
{
  return con_map->hasItem(_2093_name);
}
// End mosek.fusion.Model.hasConstraint

// Begin mosek.fusion.Model.hasVariable
// Method mosek.fusion.Model.hasVariable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:709:6-93
bool mosek::fusion::Model :: hasVariable(const std::string & _2094_name) { return mosek::fusion::p_Model::_get_impl(this)->hasVariable(_2094_name); }
bool mosek::fusion::p_Model::hasVariable(const std::string & _2094_name)
{
  return var_map->hasItem(_2094_name);
}
// End mosek.fusion.Model.hasVariable

// Begin mosek.fusion.Model.getParameter
// Method mosek.fusion.Model.getParameter @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:707:6-149
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__getParameter(const std::string & _2095_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getParameter(_2095_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::getParameter(const std::string &  _2095_name) { return __mosek_2fusion_2Model__getParameter(_2095_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__getParameter(const std::string & _2095_name)
{
  monty::rc_ptr< ::mosek::fusion::Parameter > _10142_;
  bool _10143_ = par_map->hasItem(_2095_name);
  if (_10143_)
  {
    _10142_ = ((*parameters)(par_map->getItem(_2095_name)));
  }
  else
  {
    _10142_ = nullptr;
  }
  return _10142_;
}
// End mosek.fusion.Model.getParameter

// Begin mosek.fusion.Model.getConstraint
// Method mosek.fusion.Model.getConstraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:706:6-167
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__getConstraint(int32_t _2096_index) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getConstraint(_2096_index); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::getConstraint(int32_t _2096_index) { return __mosek_2fusion_2Model__getConstraint(_2096_index); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__getConstraint(int32_t _2096_index)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2097_res;
  if ((_2096_index >= (int)0))
  {
    {
      _2097_res = (((*cons)(_2096_index))).as<::mosek::fusion::Constraint>();
    }
  }
  else
  {
    {
      _2097_res = (((*acons)((-safe_add( (int)1,_2096_index ))))).as<::mosek::fusion::Constraint>();
    }
  }
  return _2097_res;
}
// End mosek.fusion.Model.getConstraint

// Begin mosek.fusion.Model.getConstraint
// Method mosek.fusion.Model.getConstraint @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:696:6-705:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__getConstraint(const std::string & _2098_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getConstraint(_2098_name); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::getConstraint(const std::string &  _2098_name) { return __mosek_2fusion_2Model__getConstraint(_2098_name); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__getConstraint(const std::string & _2098_name)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2099_retval = nullptr;
  if (con_map->hasItem(_2098_name))
  {
    {
      int32_t _2100_idx = con_map->getItem(_2098_name);
      if ((_2100_idx >= (int)0))
      {
        {
          _2099_retval = (((*cons)(_2100_idx))).as<::mosek::fusion::Constraint>();
        }
      }
      else
      {
        {
          _2099_retval = (((*acons)((-safe_add( (int)1,_2100_idx ))))).as<::mosek::fusion::Constraint>();
        }
      }
    }
  }
  {}
  return _2099_retval;
}
// End mosek.fusion.Model.getConstraint

// Begin mosek.fusion.Model.getVariable
// Method mosek.fusion.Model.getVariable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:694:6-86
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__getVariable(int32_t _2101_index) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getVariable(_2101_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::getVariable(int32_t _2101_index) { return __mosek_2fusion_2Model__getVariable(_2101_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__getVariable(int32_t _2101_index)
{
  return (((*vars)(_2101_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.getVariable

// Begin mosek.fusion.Model.getVariable
// Method mosek.fusion.Model.getVariable @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:683:6-693:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__getVariable(const std::string & _2102_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getVariable(_2102_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::getVariable(const std::string &  _2102_name) { return __mosek_2fusion_2Model__getVariable(_2102_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__getVariable(const std::string & _2102_name)
{
  if (var_map->hasItem(_2102_name))
  {
    {
      int32_t _2103_varid = var_map->getItem(_2102_name);
      if ((_2103_varid < (int)0))
      {
        {
          return (((*barvars)((-safe_add( _2103_varid,(int)1 ))))).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          return (((*vars)(_2103_varid))).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
  else
  {
    {
      return nullptr;
    }
  }
}
// End mosek.fusion.Model.getVariable

// Begin mosek.fusion.Model.getName
// Method mosek.fusion.Model.getName @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:681:6-67
std::string mosek::fusion::Model :: getName() { return mosek::fusion::p_Model::_get_impl(this)->getName(); }
std::string mosek::fusion::p_Model::getName()
{
  return model_name;
}
// End mosek.fusion.Model.getName

// Begin mosek.fusion.Model.getParameterValue
// Method mosek.fusion.Model.getParameterValue @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:673:6-678:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getParameterValue(std::shared_ptr< monty::ndarray< int32_t,1 > > _2104_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2104_idxs(_2104_idxs ? _2104_idxs->raw() : nullptr, _2104_idxs ? _2104_idxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2105_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2104_idxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2105_res(_2105_res ? _2105_res->raw() : nullptr,_2105_res ? _2105_res->size(0) : 0);
  int32_t _10144_ = (int)0;
  int32_t _10145_ = (int)((_2104_idxs)->size(0));
  for (int32_t _2106_i = _10144_; _2106_i < _10145_; ++_2106_i)
  {
    {
      _ptr__2105_res[_2106_i] = ((*param_value)(_ptr__2104_idxs[_2106_i]));
    }
  }
  return _2105_res;
}
// End mosek.fusion.Model.getParameterValue

// Begin mosek.fusion.Model.setParameterValue
// Method mosek.fusion.Model.setParameterValue @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:668:6-671:7
void mosek::fusion::p_Model::setParameterValue(std::shared_ptr< monty::ndarray< int32_t,1 > > _2107_idxs,std::shared_ptr< monty::ndarray< double,1 > > _2108_vals)
{
  _checked_ptr_1<int32_t> _ptr__2107_idxs(_2107_idxs ? _2107_idxs->raw() : nullptr, _2107_idxs ? _2107_idxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__2108_vals(_2108_vals ? _2108_vals->raw() : nullptr, _2108_vals ? _2108_vals->size(0) : 0);
  int32_t _10146_ = (int)0;
  int32_t _10147_ = (int)((_2107_idxs)->size(0));
  for (int32_t _2109_i = _10146_; _2109_i < _10147_; ++_2109_i)
  {
    {
      ((*param_value)(_ptr__2107_idxs[_2109_i])) = _ptr__2108_vals[_2109_i];
    }
  }
}
// End mosek.fusion.Model.setParameterValue

// Begin mosek.fusion.Model.clone
// Method mosek.fusion.Model.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Model.mbi:663:6-666:7
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::Model :: __mosek_2fusion_2Model__clone() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__clone(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::Model::clone() { return __mosek_2fusion_2Model__clone(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::p_Model::__mosek_2fusion_2Model__clone()
{
  return ::mosek::fusion::p_Model::_new_Model(::mosek::fusion::Model::t(_pubthis));
}
// End mosek.fusion.Model.clone

void mosek::fusion::p_Model::destroy()
{
  xs.reset();
  ws.reset();
  rs.reset();
  sol_itg.reset();
  sol_bas.reset();
  sol_itr.reset();
  con_map.reset();
  acons.reset();
  cons.reset();
  param_value.reset();
  par_map.reset();
  parameters.reset();
  initsol_xx_flag.reset();
  initsol_xx.reset();
  var_map.reset();
  barvars.reset();
  vars.reset();
  barvar_block_elm_j.reset();
  barvar_block_elm_i.reset();
  barvar_block_elm_barj.reset();
  barvar_block_elm_ptr.reset();
  barvar_block_dim.reset();
  barvar_block_ptr.reset();
  barvar_dim.reset();
  var_elm_acc_ofs.reset();
  var_elm_acc_idx.reset();
  var_block_acc_id.reset();
  var_block_map.reset();
  acon_elm_afe.reset();
  acon_elm_ofs.reset();
  acon_elm_scale.reset();
  acon_elm_accid.reset();
  acon_afe.reset();
  acon_acc.reset();
  acon_block_map.reset();
  acc_block_map.reset();
  obj_blocks.reset();
  afe_blocks.reset();
  con_blocks.reset();
}
void mosek::fusion::Model::destroy() { mosek::fusion::p_Model::_get_impl(this)->destroy(); }
// } class Model
// class BoundInterfaceVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:29:4-85:5
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable(mosek::fusion::p_BoundInterfaceVariable *_impl) : mosek::fusion::SliceVariable(_impl) { /*std::cout << "BoundInterfaceVariable()" << this << std::endl; */ }
mosek::fusion::BoundInterfaceVariable::~BoundInterfaceVariable() {  /* std::cout << "~BoundInterfaceVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_BoundInterfaceVariable::p_BoundInterfaceVariable
  (::mosek::fusion::BoundInterfaceVariable * _pubthis) :     mosek::fusion::p_SliceVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.BoundInterfaceVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:41:6-50:7
mosek::fusion::BoundInterfaceVariable::t mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2110_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2112_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2113_nativeidxs,
    bool _2114_islower)
{ return new mosek::fusion::BoundInterfaceVariable(_2110_m,_2111_shape,_2112_sparsity,_2113_nativeidxs,_2114_islower); }
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2110_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2112_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2113_nativeidxs,
    bool _2114_islower) : 
   BoundInterfaceVariable(new mosek::fusion::p_BoundInterfaceVariable(this))
{ mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->_initialize(_2110_m,_2111_shape,_2112_sparsity,_2113_nativeidxs,_2114_islower); }
void mosek::fusion::p_BoundInterfaceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2110_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2112_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2113_nativeidxs,
    bool _2114_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2111_shape(_2111_shape ? _2111_shape->raw() : nullptr, _2111_shape ? _2111_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2112_sparsity(_2112_sparsity ? _2112_sparsity->raw() : nullptr, _2112_sparsity ? _2112_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2113_nativeidxs(_2113_nativeidxs ? _2113_nativeidxs->raw() : nullptr, _2113_nativeidxs ? _2113_nativeidxs->size(0) : 0);
    mosek::fusion::p_SliceVariable::_initialize(_2110_m,_2111_shape,_2112_sparsity,_2113_nativeidxs);
    mosek::fusion::p_BoundInterfaceVariable::_get_impl(::mosek::fusion::BoundInterfaceVariable::t(_pubthis))->islower = _2114_islower;
  }
}
// mosek.fusion.BoundInterfaceVariable.ctor

// mosek.fusion.BoundInterfaceVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:33:6-39:7
mosek::fusion::BoundInterfaceVariable::t mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2115_v,
    bool _2116_islower)
{ return new mosek::fusion::BoundInterfaceVariable(_2115_v,_2116_islower); }
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2115_v,
    bool _2116_islower) : 
   BoundInterfaceVariable(new mosek::fusion::p_BoundInterfaceVariable(this))
{ mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->_initialize(_2115_v,_2116_islower); }
void mosek::fusion::p_BoundInterfaceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2115_v,
    bool _2116_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_SliceVariable::_initialize(_2115_v);
    mosek::fusion::p_BoundInterfaceVariable::_get_impl(::mosek::fusion::BoundInterfaceVariable::t(_pubthis))->islower = _2116_islower;
  }
}
// mosek.fusion.BoundInterfaceVariable.ctor

// Begin mosek.fusion.BoundInterfaceVariable.dual
// Method mosek.fusion.BoundInterfaceVariable.dual @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:80:6-84:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BoundInterfaceVariable :: dual() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->dual(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BoundInterfaceVariable::dual()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2117_target = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
  _checked_ptr_1<double> _ptr__2117_target(_2117_target ? _2117_target->raw() : nullptr,_2117_target ? _2117_target->size(0) : 0);
  dual_lu((int)0,_2117_target,islower);
  return _2117_target;
}
// End mosek.fusion.BoundInterfaceVariable.dual

// Begin mosek.fusion.BoundInterfaceVariable.transpose
// Method mosek.fusion.BoundInterfaceVariable.transpose @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:78:6-78
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__transpose() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::transpose() { return __mosek_2fusion_2BoundInterfaceVariable__transpose(); }
/* override: mosek.fusion.BaseVariable.transpose*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__transpose() { return __mosek_2fusion_2BoundInterfaceVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__transpose()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__transpose())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.transpose

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:77:6-112
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2118_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2119_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2120_i2) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2118_i0,_2119_i1,_2120_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2118_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2119_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2120_i2) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2118_i0,_2119_i1,_2120_i2); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2118_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2119_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2120_i2) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2118_i0,_2119_i1,_2120_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2118_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2119_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2120_i2)
{
  _checked_ptr_1<int32_t> _ptr__2118_i0(_2118_i0 ? _2118_i0->raw() : nullptr, _2118_i0 ? _2118_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2119_i1(_2119_i1 ? _2119_i1->raw() : nullptr, _2119_i1 ? _2119_i1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2120_i2(_2120_i2 ? _2120_i2->raw() : nullptr, _2120_i2 ? _2120_i2->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2118_i0,_2119_i1,_2120_i2))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:76:6-98
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2121_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2122_i1) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2121_i0,_2122_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2121_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2122_i1) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2121_i0,_2122_i1); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2121_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2122_i1) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2121_i0,_2122_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2121_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2122_i1)
{
  _checked_ptr_1<int32_t> _ptr__2121_i0(_2121_i0 ? _2121_i0->raw() : nullptr, _2121_i0 ? _2121_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2122_i1(_2122_i1 ? _2122_i1->raw() : nullptr, _2122_i1 ? _2122_i1->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2121_i0,_2122_i1))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:75:6-91
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2123_midxs) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2123_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2123_midxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2123_midxs); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2123_midxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2123_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2123_midxs)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2123_midxs))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:74:6-88
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2124_idxs) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2124_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2124_idxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2124_idxs); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2124_idxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2124_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2124_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2124_idxs(_2124_idxs ? _2124_idxs->raw() : nullptr, _2124_idxs ? _2124_idxs->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2124_idxs))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.antidiag
// Method mosek.fusion.BoundInterfaceVariable.antidiag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:73:6-93
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__antidiag(int32_t _2125_index) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__antidiag(_2125_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::antidiag(int32_t _2125_index) { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(_2125_index); }
/* override: mosek.fusion.BaseVariable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__antidiag(int32_t _2125_index) { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(_2125_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__antidiag(int32_t _2125_index)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag(_2125_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.antidiag

// Begin mosek.fusion.BoundInterfaceVariable.antidiag
// Method mosek.fusion.BoundInterfaceVariable.antidiag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:72:6-76
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__antidiag() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::antidiag() { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
/* override: mosek.fusion.BaseVariable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__antidiag() { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__antidiag()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.antidiag

// Begin mosek.fusion.BoundInterfaceVariable.diag
// Method mosek.fusion.BoundInterfaceVariable.diag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:71:6-88
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__diag(int32_t _2126_index) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__diag(_2126_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::diag(int32_t _2126_index) { return __mosek_2fusion_2BoundInterfaceVariable__diag(_2126_index); }
/* override: mosek.fusion.BaseVariable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__diag(int32_t _2126_index) { return __mosek_2fusion_2BoundInterfaceVariable__diag(_2126_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__diag(int32_t _2126_index)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag(_2126_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.diag

// Begin mosek.fusion.BoundInterfaceVariable.diag
// Method mosek.fusion.BoundInterfaceVariable.diag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:70:6-72
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__diag() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::diag() { return __mosek_2fusion_2BoundInterfaceVariable__diag(); }
/* override: mosek.fusion.BaseVariable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__diag() { return __mosek_2fusion_2BoundInterfaceVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__diag()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.diag

// Begin mosek.fusion.BoundInterfaceVariable.slice
// Method mosek.fusion.BoundInterfaceVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:69:6-113
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2127_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2128_lasta) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__slice(_2127_firsta,_2128_lasta); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2127_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2128_lasta) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2127_firsta,_2128_lasta); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2127_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2128_lasta) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2127_firsta,_2128_lasta); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2127_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2128_lasta)
{
  _checked_ptr_1<int32_t> _ptr__2127_firsta(_2127_firsta ? _2127_firsta->raw() : nullptr, _2127_firsta ? _2127_firsta->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2128_lasta(_2128_lasta ? _2128_lasta->raw() : nullptr, _2128_lasta ? _2128_lasta->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(_2127_firsta,_2128_lasta))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.slice

// Begin mosek.fusion.BoundInterfaceVariable.slice
// Method mosek.fusion.BoundInterfaceVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:68:6-105
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__slice(int32_t _2129_first,int32_t _2130_last) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__slice(_2129_first,_2130_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::slice(int32_t _2129_first,int32_t _2130_last) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2129_first,_2130_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2129_first,int32_t _2130_last) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2129_first,_2130_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__slice(int32_t _2129_first,int32_t _2130_last)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(_2129_first,_2130_last))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.slice

// Begin mosek.fusion.BoundInterfaceVariable.from_
// Method mosek.fusion.BoundInterfaceVariable.from_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceVariable.mbi:52:6-66:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__from_(monty::rc_ptr< ::mosek::fusion::Variable > _2131_v)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2132_shape = _pubthis->getShape();
  _checked_ptr_1<int32_t> _ptr__2132_shape(_2132_shape ? _2132_shape->raw() : nullptr,_2132_shape ? _2132_shape->size(0) : 0);
  int32_t _2133_nnz = _2131_v->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2134_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2133_nnz)));
  _checked_ptr_1<int64_t> _ptr__2134_nativeidxs(_2134_nativeidxs ? _2134_nativeidxs->raw() : nullptr,_2134_nativeidxs ? _2134_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2135_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__2135_sparsity(_2135_sparsity ? _2135_sparsity->raw() : nullptr,_2135_sparsity ? _2135_sparsity->size(0) : 0);
  if ((_2131_v->getSize() == _2133_nnz))
  {
    {
      _2131_v->inst((int)0,_2134_nativeidxs);
    }
  }
  else
  {
    {
      _2135_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2133_nnz)));
      _ptr__2135_sparsity.update(_2135_sparsity ? _2135_sparsity->raw() : nullptr, _2135_sparsity ? _2135_sparsity->size(0) : 0);
      _2131_v->inst((int)0,_2135_sparsity,(int)0,_2134_nativeidxs);
    }
  }
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_2131_v->__mosek_2fusion_2Expression__getModel(),_2132_shape,_2135_sparsity,_2134_nativeidxs,islower);
}
// End mosek.fusion.BoundInterfaceVariable.from_

void mosek::fusion::p_BoundInterfaceVariable::destroy()
{
}
void mosek::fusion::BoundInterfaceVariable::destroy() { mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->destroy(); }
// } class BoundInterfaceVariable
// class SliceVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SliceVariable.mbi:18:4-66:5
mosek::fusion::SliceVariable::SliceVariable(mosek::fusion::p_SliceVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "SliceVariable()" << this << std::endl; */ }
mosek::fusion::SliceVariable::~SliceVariable() {  /* std::cout << "~SliceVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_SliceVariable::p_SliceVariable
  (::mosek::fusion::SliceVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.SliceVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SliceVariable.mbi:54:6-65:7
mosek::fusion::SliceVariable::t mosek::fusion::p_SliceVariable::_new_SliceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2136_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2138_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2139_nativeidxs)
{ return new mosek::fusion::SliceVariable(_2136_m,_2137_shape,_2138_sparsity,_2139_nativeidxs); }
mosek::fusion::SliceVariable::SliceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2136_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2138_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2139_nativeidxs) : 
   SliceVariable(new mosek::fusion::p_SliceVariable(this))
{ mosek::fusion::p_SliceVariable::_get_impl(this)->_initialize(_2136_m,_2137_shape,_2138_sparsity,_2139_nativeidxs); }
void mosek::fusion::p_SliceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2136_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2138_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2139_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2137_shape(_2137_shape ? _2137_shape->raw() : nullptr, _2137_shape ? _2137_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2138_sparsity(_2138_sparsity ? _2138_sparsity->raw() : nullptr, _2138_sparsity ? _2138_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2139_nativeidxs(_2139_nativeidxs ? _2139_nativeidxs->raw() : nullptr, _2139_nativeidxs ? _2139_nativeidxs->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(_2136_m,_2137_shape,_2138_sparsity,_2139_nativeidxs);
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->nativeidxs = _2139_nativeidxs;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->sparsity = _2138_sparsity;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->shape = _2137_shape;
  }
}
// mosek.fusion.SliceVariable.ctor

// mosek.fusion.SliceVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SliceVariable.mbi:47:6-52:7
mosek::fusion::SliceVariable::t mosek::fusion::p_SliceVariable::_new_SliceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2140_v)
{ return new mosek::fusion::SliceVariable(_2140_v); }
mosek::fusion::SliceVariable::SliceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2140_v) : 
   SliceVariable(new mosek::fusion::p_SliceVariable(this))
{ mosek::fusion::p_SliceVariable::_get_impl(this)->_initialize(_2140_v); }
void mosek::fusion::p_SliceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2140_v)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseVariable::_initialize(_2140_v,_2140_v->__mosek_2fusion_2BaseVariable__getModel());
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->nativeidxs = mosek::fusion::p_SliceVariable::_get_impl(_2140_v)->nativeidxs;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->sparsity = mosek::fusion::p_SliceVariable::_get_impl(_2140_v)->sparsity;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->shape = mosek::fusion::p_SliceVariable::_get_impl(_2140_v)->shape;
  }
}
// mosek.fusion.SliceVariable.ctor

void mosek::fusion::p_SliceVariable::destroy()
{
  shape.reset();
  sparsity.reset();
  nativeidxs.reset();
}
void mosek::fusion::SliceVariable::destroy() { mosek::fusion::p_SliceVariable::_get_impl(this)->destroy(); }
// } class SliceVariable
// class RangedVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:15:4-103:5
mosek::fusion::RangedVariable::RangedVariable(mosek::fusion::p_RangedVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "RangedVariable()" << this << std::endl; */ }
mosek::fusion::RangedVariable::~RangedVariable() {  /* std::cout << "~RangedVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_RangedVariable::p_RangedVariable
  (::mosek::fusion::RangedVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.RangedVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:53:6-61:7
mosek::fusion::RangedVariable::t mosek::fusion::p_RangedVariable::_new_RangedVariable
  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2141_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2142_m)
{ return new mosek::fusion::RangedVariable(_2141_v,_2142_m); }
mosek::fusion::RangedVariable::RangedVariable
  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2141_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2142_m) : 
   RangedVariable(new mosek::fusion::p_RangedVariable(this))
{ mosek::fusion::p_RangedVariable::_get_impl(this)->_initialize(_2141_v,_2142_m); }
void mosek::fusion::p_RangedVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2141_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2142_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2141_v,_2142_m);
    name = mosek::fusion::p_RangedVariable::_get_impl(_2141_v)->name;
    nativeidxs = mosek::fusion::p_RangedVariable::_get_impl(_2141_v)->nativeidxs;
    sparsity = mosek::fusion::p_RangedVariable::_get_impl(_2141_v)->sparsity;
    shape = mosek::fusion::p_RangedVariable::_get_impl(_2141_v)->shape;
  }
}
// mosek.fusion.RangedVariable.ctor

// mosek.fusion.RangedVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:38:6-51:7
mosek::fusion::RangedVariable::t mosek::fusion::p_RangedVariable::_new_RangedVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2143_model,
    const std::string &  _2144_name,
    int64_t _2145_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2146_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2147_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2148_nativeidxs)
{ return new mosek::fusion::RangedVariable(_2143_model,_2144_name,_2145_varid,_2146_shape,_2147_sparsity,_2148_nativeidxs); }
mosek::fusion::RangedVariable::RangedVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2143_model,
    const std::string &  _2144_name,
    int64_t _2145_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2146_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2147_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2148_nativeidxs) : 
   RangedVariable(new mosek::fusion::p_RangedVariable(this))
{ mosek::fusion::p_RangedVariable::_get_impl(this)->_initialize(_2143_model,_2144_name,_2145_varid,_2146_shape,_2147_sparsity,_2148_nativeidxs); }
void mosek::fusion::p_RangedVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2143_model,
    const std::string &  _2144_name,
    int64_t _2145_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2146_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2147_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2148_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2146_shape(_2146_shape ? _2146_shape->raw() : nullptr, _2146_shape ? _2146_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2147_sparsity(_2147_sparsity ? _2147_sparsity->raw() : nullptr, _2147_sparsity ? _2147_sparsity->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2148_nativeidxs(_2148_nativeidxs ? _2148_nativeidxs->raw() : nullptr, _2148_nativeidxs ? _2148_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2143_model,_2144_name,_2146_shape,_2145_varid,_2147_sparsity,::mosek::fusion::p_RangedVariable::globalNativeIndexes(_2148_nativeidxs));
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->nativeidxs = _2148_nativeidxs;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->sparsity = _2147_sparsity;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->name = _2144_name;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->shape = _2146_shape;
  }
}
// mosek.fusion.RangedVariable.ctor

// Begin mosek.fusion.RangedVariable.elementDesc
// Method mosek.fusion.RangedVariable.elementDesc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:97:6-102:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__elementDesc(int64_t _2149_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2150_sb) { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__elementDesc(_2149_index,_2150_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::RangedVariable::elementDesc(int64_t _2149_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2150_sb) { return __mosek_2fusion_2RangedVariable__elementDesc(_2149_index,_2150_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__elementDesc(int64_t _2149_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2150_sb)
{
  _pubthis->elementName(_2149_index,_2150_sb);
  _2150_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" : in range"));
  return _2150_sb;
}
// End mosek.fusion.RangedVariable.elementDesc

// Begin mosek.fusion.RangedVariable.dual_u
// Method mosek.fusion.RangedVariable.dual_u @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:88:6-93:7
void mosek::fusion::p_RangedVariable::dual_u(int32_t _2151_offset,std::shared_ptr< monty::ndarray< double,1 > > _2152_target)
{
  _checked_ptr_1<double> _ptr__2152_target(_2152_target ? _2152_target->raw() : nullptr, _2152_target ? _2152_target->size(0) : 0);
  dual_lu(_2151_offset,_2152_target,false);
}
// End mosek.fusion.RangedVariable.dual_u

// Begin mosek.fusion.RangedVariable.dual_l
// Method mosek.fusion.RangedVariable.dual_l @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:80:6-85:7
void mosek::fusion::p_RangedVariable::dual_l(int32_t _2153_offset,std::shared_ptr< monty::ndarray< double,1 > > _2154_target)
{
  _checked_ptr_1<double> _ptr__2154_target(_2154_target ? _2154_target->raw() : nullptr, _2154_target ? _2154_target->size(0) : 0);
  dual_lu(_2153_offset,_2154_target,true);
}
// End mosek.fusion.RangedVariable.dual_l

// Begin mosek.fusion.RangedVariable.upperBoundVar
// Method mosek.fusion.RangedVariable.upperBoundVar @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:76:6-78:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__upperBoundVar() { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__upperBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable::upperBoundVar() { return __mosek_2fusion_2RangedVariable__upperBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__upperBoundVar()
{
  std::vector<int64_t> _10148_;
  for (int32_t _2155_i = (int)0, _10149_ = (int)((nativeidxs)->size(0)); _2155_i < _10149_ ; ++_2155_i)
  {
    _10148_.push_back(((*nativeidxs)(_2155_i)));
  }
  auto _10150_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10148_.size()),_10148_.begin(),_10148_.end()));
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),shape,sparsity,_10150_,false);
}
// End mosek.fusion.RangedVariable.upperBoundVar

// Begin mosek.fusion.RangedVariable.lowerBoundVar
// Method mosek.fusion.RangedVariable.lowerBoundVar @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:70:6-72:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__lowerBoundVar() { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__lowerBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable::lowerBoundVar() { return __mosek_2fusion_2RangedVariable__lowerBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__lowerBoundVar()
{
  std::vector<int64_t> _10151_;
  for (int32_t _2156_i = (int)0, _10152_ = (int)((nativeidxs)->size(0)); _2156_i < _10152_ ; ++_2156_i)
  {
    _10151_.push_back(((*nativeidxs)(_2156_i)));
  }
  auto _10153_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10151_.size()),_10151_.begin(),_10151_.end()));
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),shape,sparsity,_10153_,true);
}
// End mosek.fusion.RangedVariable.lowerBoundVar

// Begin mosek.fusion.RangedVariable.clone
// Method mosek.fusion.RangedVariable.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:63:6-65:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2157_m)
{
  return (::mosek::fusion::p_RangedVariable::_new_RangedVariable(::mosek::fusion::RangedVariable::t(_pubthis),_2157_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.RangedVariable.clone

// Begin mosek.fusion.RangedVariable.globalNativeIndexes
// Method mosek.fusion.RangedVariable.globalNativeIndexes @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/RangedVariable.mbi:24:6-27:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_RangedVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2158_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2158_nativeidxs(_2158_nativeidxs ? _2158_nativeidxs->raw() : nullptr, _2158_nativeidxs ? _2158_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2159_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2158_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2159_r(_2159_r ? _2159_r->raw() : nullptr,_2159_r ? _2159_r->size(0) : 0);
  int32_t _10154_ = (int)0;
  int32_t _10155_ = (int)((_2158_nativeidxs)->size(0));
  for (int32_t _2160_i = _10154_; _2160_i < _10155_; ++_2160_i)
  {
    {
      _ptr__2159_r[_2160_i] = _ptr__2158_nativeidxs[_2160_i];
    }
  }
  return _2159_r;
}
// End mosek.fusion.RangedVariable.globalNativeIndexes

void mosek::fusion::p_RangedVariable::destroy()
{
  shape.reset();
  nativeidxs.reset();
  sparsity.reset();
}
void mosek::fusion::RangedVariable::destroy() { mosek::fusion::p_RangedVariable::_get_impl(this)->destroy(); }
// } class RangedVariable
// class LinearPSDVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:8:4-71:5
mosek::fusion::LinearPSDVariable::LinearPSDVariable(mosek::fusion::p_LinearPSDVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "LinearPSDVariable()" << this << std::endl; */ }
mosek::fusion::LinearPSDVariable::~LinearPSDVariable() {  /* std::cout << "~LinearPSDVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_LinearPSDVariable::p_LinearPSDVariable
  (::mosek::fusion::LinearPSDVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.LinearPSDVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:35:6-43:7
mosek::fusion::LinearPSDVariable::t mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2161_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2162_m)
{ return new mosek::fusion::LinearPSDVariable(_2161_v,_2162_m); }
mosek::fusion::LinearPSDVariable::LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2161_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2162_m) : 
   LinearPSDVariable(new mosek::fusion::p_LinearPSDVariable(this))
{ mosek::fusion::p_LinearPSDVariable::_get_impl(this)->_initialize(_2161_v,_2162_m); }
void mosek::fusion::p_LinearPSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2161_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2162_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2161_v,_2162_m);
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->conedim = mosek::fusion::p_LinearPSDVariable::_get_impl(_2161_v)->conedim;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->varid = mosek::fusion::p_LinearPSDVariable::_get_impl(_2161_v)->varid;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->name = mosek::fusion::p_LinearPSDVariable::_get_impl(_2161_v)->name;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->shape = mosek::fusion::p_LinearPSDVariable::_get_impl(_2161_v)->shape;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->nativeidxs = mosek::fusion::p_LinearPSDVariable::_get_impl(_2161_v)->nativeidxs;
  }
}
// mosek.fusion.LinearPSDVariable.ctor

// mosek.fusion.LinearPSDVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:19:6-33:7
mosek::fusion::LinearPSDVariable::t mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2163_model,
    const std::string &  _2164_name,
    int32_t _2165_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2166_shape,
    int32_t _2167_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2168_nativeidxs)
{ return new mosek::fusion::LinearPSDVariable(_2163_model,_2164_name,_2165_varid,_2166_shape,_2167_conedim,_2168_nativeidxs); }
mosek::fusion::LinearPSDVariable::LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2163_model,
    const std::string &  _2164_name,
    int32_t _2165_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2166_shape,
    int32_t _2167_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2168_nativeidxs) : 
   LinearPSDVariable(new mosek::fusion::p_LinearPSDVariable(this))
{ mosek::fusion::p_LinearPSDVariable::_get_impl(this)->_initialize(_2163_model,_2164_name,_2165_varid,_2166_shape,_2167_conedim,_2168_nativeidxs); }
void mosek::fusion::p_LinearPSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2163_model,
    const std::string &  _2164_name,
    int32_t _2165_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2166_shape,
    int32_t _2167_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2168_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2166_shape(_2166_shape ? _2166_shape->raw() : nullptr, _2166_shape ? _2166_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2168_nativeidxs(_2168_nativeidxs ? _2168_nativeidxs->raw() : nullptr, _2168_nativeidxs ? _2168_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2163_model,_2164_name,_2166_shape,_2165_varid,nullptr,::mosek::fusion::p_LinearPSDVariable::globalNativeIndexes(_2168_nativeidxs));
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->name = _2164_name;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->conedim = _2167_conedim;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->varid = _2165_varid;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->shape = _2166_shape;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->nativeidxs = _2168_nativeidxs;
  }
}
// mosek.fusion.LinearPSDVariable.ctor

// Begin mosek.fusion.LinearPSDVariable.toString
// Method mosek.fusion.LinearPSDVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:60:6-69:7
std::string mosek::fusion::LinearPSDVariable :: toString() { return mosek::fusion::p_LinearPSDVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_LinearPSDVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2169_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2169_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PSDVariable("));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2169_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2169_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10156_ = (int)1;
  int32_t _10157_ = (int)((shape)->size(0));
  for (int32_t _2170_i = _10156_; _2170_i < _10157_; ++_2170_i)
  {
    {
      _2169_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2170_i)));
    }
  }
  _2169_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2169_sb->toString();
}
// End mosek.fusion.LinearPSDVariable.toString

// Begin mosek.fusion.LinearPSDVariable.make_continuous
// Method mosek.fusion.LinearPSDVariable.make_continuous @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:56:6-58:7
void mosek::fusion::LinearPSDVariable :: make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _2171_idxs) { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->make_continuous(_2171_idxs); }
void mosek::fusion::p_LinearPSDVariable::make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _2171_idxs)
{
  _checked_ptr_1<int64_t> _ptr__2171_idxs(_2171_idxs ? _2171_idxs->raw() : nullptr, _2171_idxs ? _2171_idxs->size(0) : 0);
}
// End mosek.fusion.LinearPSDVariable.make_continuous

// Begin mosek.fusion.LinearPSDVariable.make_integer
// Method mosek.fusion.LinearPSDVariable.make_integer @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:51:6-54:7
void mosek::fusion::LinearPSDVariable :: make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _2172_idxs) { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->make_integer(_2172_idxs); }
void mosek::fusion::p_LinearPSDVariable::make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _2172_idxs)
{
  _checked_ptr_1<int64_t> _ptr__2172_idxs(_2172_idxs ? _2172_idxs->raw() : nullptr, _2172_idxs ? _2172_idxs->size(0) : 0);
  throw ::mosek::fusion::UnimplementedError(std::string ("Cannot make PSDVariable elements integer"));
}
// End mosek.fusion.LinearPSDVariable.make_integer

// Begin mosek.fusion.LinearPSDVariable.clone
// Method mosek.fusion.LinearPSDVariable.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:45:6-48:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_LinearPSDVariable::__mosek_2fusion_2LinearPSDVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2173_m)
{
  return (::mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable(::mosek::fusion::LinearPSDVariable::t(_pubthis),_2173_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.LinearPSDVariable.clone

// Begin mosek.fusion.LinearPSDVariable.globalNativeIndexes
// Method mosek.fusion.LinearPSDVariable.globalNativeIndexes @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearPSDVariable.monty:17:6-147
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_LinearPSDVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int64_t,1 > > _2174_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2174_nativeidxs(_2174_nativeidxs ? _2174_nativeidxs->raw() : nullptr, _2174_nativeidxs ? _2174_nativeidxs->size(0) : 0);
  std::vector<int64_t> _10158_;
  for (int32_t _2175_i = (int)0, _10159_ = (int)((_2174_nativeidxs)->size(0)); _2175_i < _10159_ ; ++_2175_i)
  {
    _10158_.push_back((-safe_add( _ptr__2174_nativeidxs[_2175_i],(int)1 )));
  }
  auto _10160_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10158_.size()),_10158_.begin(),_10158_.end()));
  return _10160_;
}
// End mosek.fusion.LinearPSDVariable.globalNativeIndexes

void mosek::fusion::p_LinearPSDVariable::destroy()
{
  shape.reset();
  nativeidxs.reset();
}
void mosek::fusion::LinearPSDVariable::destroy() { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->destroy(); }
// } class LinearPSDVariable
// class PSDVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:8:4-126:5
mosek::fusion::PSDVariable::PSDVariable(mosek::fusion::p_PSDVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "PSDVariable()" << this << std::endl; */ }
mosek::fusion::PSDVariable::~PSDVariable() {  /* std::cout << "~PSDVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_PSDVariable::p_PSDVariable
  (::mosek::fusion::PSDVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.PSDVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:86:6-98:7
mosek::fusion::PSDVariable::t mosek::fusion::p_PSDVariable::_new_PSDVariable
  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2176_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2177_m)
{ return new mosek::fusion::PSDVariable(_2176_v,_2177_m); }
mosek::fusion::PSDVariable::PSDVariable
  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2176_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2177_m) : 
   PSDVariable(new mosek::fusion::p_PSDVariable(this))
{ mosek::fusion::p_PSDVariable::_get_impl(this)->_initialize(_2176_v,_2177_m); }
void mosek::fusion::p_PSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2176_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2177_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2176_v,_2177_m);
    varid = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->varid;
    nativeidxs = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->nativeidxs;
    name = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->name;
    shape = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->shape;
    conedim1 = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->conedim1;
    conedim2 = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->conedim2;
    model = _2177_m;
    barvaridxs = mosek::fusion::p_PSDVariable::_get_impl(_2176_v)->barvaridxs;
  }
}
// mosek.fusion.PSDVariable.ctor

// mosek.fusion.PSDVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:64:6-84:7
mosek::fusion::PSDVariable::t mosek::fusion::p_PSDVariable::_new_PSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2178_model,
    const std::string &  _2179_name,
    int32_t _2180_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2181_shape,
    int32_t _2182_conedim1,
    int32_t _2183_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2184_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2185_nativeidxs)
{ return new mosek::fusion::PSDVariable(_2178_model,_2179_name,_2180_varid,_2181_shape,_2182_conedim1,_2183_conedim2,_2184_barvaridxs,_2185_nativeidxs); }
mosek::fusion::PSDVariable::PSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2178_model,
    const std::string &  _2179_name,
    int32_t _2180_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2181_shape,
    int32_t _2182_conedim1,
    int32_t _2183_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2184_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2185_nativeidxs) : 
   PSDVariable(new mosek::fusion::p_PSDVariable(this))
{ mosek::fusion::p_PSDVariable::_get_impl(this)->_initialize(_2178_model,_2179_name,_2180_varid,_2181_shape,_2182_conedim1,_2183_conedim2,_2184_barvaridxs,_2185_nativeidxs); }
void mosek::fusion::p_PSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2178_model,
    const std::string &  _2179_name,
    int32_t _2180_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2181_shape,
    int32_t _2182_conedim1,
    int32_t _2183_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2184_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2185_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2181_shape(_2181_shape ? _2181_shape->raw() : nullptr, _2181_shape ? _2181_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2184_barvaridxs(_2184_barvaridxs ? _2184_barvaridxs->raw() : nullptr, _2184_barvaridxs ? _2184_barvaridxs->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2185_nativeidxs(_2185_nativeidxs ? _2185_nativeidxs->raw() : nullptr, _2185_nativeidxs ? _2185_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2178_model,_2179_name,_2181_shape,_2180_varid,nullptr,::mosek::fusion::p_PSDVariable::fullnativeidxs(_2181_shape,_2182_conedim1,_2183_conedim2,_2185_nativeidxs));
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->varid = _2180_varid;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->nativeidxs = _2185_nativeidxs;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->name = _2179_name;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->shape = _2181_shape;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->conedim1 = _2182_conedim1;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->conedim2 = _2183_conedim2;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->model = _2178_model;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->barvaridxs = _2184_barvaridxs;
  }
}
// mosek.fusion.PSDVariable.ctor

// Begin mosek.fusion.PSDVariable.toString
// Method mosek.fusion.PSDVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:115:6-124:7
std::string mosek::fusion::PSDVariable :: toString() { return mosek::fusion::p_PSDVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_PSDVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2186_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2186_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PSDVariable("));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2186_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2186_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10161_ = (int)1;
  int32_t _10162_ = (int)((shape)->size(0));
  for (int32_t _2187_i = _10161_; _2187_i < _10162_; ++_2187_i)
  {
    {
      _2186_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2187_i)));
    }
  }
  _2186_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2186_sb->toString();
}
// End mosek.fusion.PSDVariable.toString

// Begin mosek.fusion.PSDVariable.elementDesc
// Method mosek.fusion.PSDVariable.elementDesc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:107:6-112:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::PSDVariable :: __mosek_2fusion_2PSDVariable__elementDesc(int64_t _2188_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2189_sb) { return mosek::fusion::p_PSDVariable::_get_impl(this)->__mosek_2fusion_2PSDVariable__elementDesc(_2188_index,_2189_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::PSDVariable::elementDesc(int64_t _2188_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2189_sb) { return __mosek_2fusion_2PSDVariable__elementDesc(_2188_index,_2189_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_PSDVariable::__mosek_2fusion_2PSDVariable__elementDesc(int64_t _2188_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2189_sb)
{
  _pubthis->elementName(_2188_index,_2189_sb);
  _2189_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" : element in positive semidefinite cone"));
  return _2189_sb;
}
// End mosek.fusion.PSDVariable.elementDesc

// Begin mosek.fusion.PSDVariable.clone
// Method mosek.fusion.PSDVariable.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:100:6-103:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_PSDVariable::__mosek_2fusion_2PSDVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2190_m)
{
  return (::mosek::fusion::p_PSDVariable::_new_PSDVariable(::mosek::fusion::PSDVariable::t(_pubthis),_2190_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.PSDVariable.clone

// Begin mosek.fusion.PSDVariable.fullnativeidxs
// Method mosek.fusion.PSDVariable.fullnativeidxs @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/SemidefiniteVariable.mbi:21:6-57:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_PSDVariable::fullnativeidxs(std::shared_ptr< monty::ndarray< int32_t,1 > > _2191_shape,int32_t _2192_conedim1,int32_t _2193_conedim2,std::shared_ptr< monty::ndarray< int64_t,1 > > _2194_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2191_shape(_2191_shape ? _2191_shape->raw() : nullptr, _2191_shape ? _2191_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2194_nativeidxs(_2194_nativeidxs ? _2194_nativeidxs->raw() : nullptr, _2194_nativeidxs ? _2194_nativeidxs->size(0) : 0);
  int32_t _2195_nd = (int)((_2191_shape)->size(0));
  bool _2196_tril = (_2192_conedim1 < _2193_conedim2);
  int32_t _10163_;
  bool _10164_ = _2196_tril;
  if (_10164_)
  {
    _10163_ = _2192_conedim1;
  }
  else
  {
    _10163_ = _2193_conedim2;
  }
  int32_t _2197_cd1 = _10163_;
  int32_t _10165_;
  bool _10166_ = (!_2196_tril);
  if (_10166_)
  {
    _10165_ = _2192_conedim1;
  }
  else
  {
    _10165_ = _2193_conedim2;
  }
  int32_t _2198_cd2 = _10165_;
  int32_t _2199_d0 = (int)1;
  int32_t _10167_ = (int)0;
  int32_t _10168_ = _2197_cd1;
  for (int32_t _2200_i = _10167_; _2200_i < _10168_; ++_2200_i)
  {
    {
      _2199_d0 *= _ptr__2191_shape[_2200_i];
    }
  }
  int32_t _2201_d1 = _ptr__2191_shape[_2197_cd1];
  int32_t _2202_d2 = (int)1;
  int32_t _10169_ = safe_add( _2197_cd1,(int)1 );
  int32_t _10170_ = _2198_cd2;
  for (int32_t _2203_i = _10169_; _2203_i < _10170_; ++_2203_i)
  {
    {
      _2202_d2 *= _ptr__2191_shape[_2203_i];
    }
  }
  int32_t _2204_d3 = _ptr__2191_shape[_2198_cd2];
  int32_t _2205_d4 = (int)1;
  int32_t _10171_ = safe_add( _2198_cd2,(int)1 );
  int32_t _10172_ = _2195_nd;
  for (int32_t _2206_i = _10171_; _2206_i < _10172_; ++_2206_i)
  {
    {
      _2205_d4 *= _ptr__2191_shape[_2206_i];
    }
  }
  int32_t _2207_size = safe_mul( safe_mul( safe_mul( safe_mul( _2199_d0,_2201_d1 ),_2202_d2 ),_2204_d3 ),_2205_d4 );
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2208_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2207_size)));
  _checked_ptr_1<int64_t> _ptr__2208_residxs(_2208_residxs ? _2208_residxs->raw() : nullptr,_2208_residxs ? _2208_residxs->size(0) : 0);
  int32_t _2209_s3 = _2205_d4;
  int32_t _2210_s2 = safe_mul( _2209_s3,_2204_d3 );
  int32_t _2211_s1 = safe_mul( _2210_s2,_2202_d2 );
  int32_t _2212_s0 = safe_mul( _2211_s1,_2211_s1 );
  {
    int32_t _2213_k = (int)0;
    int32_t _10173_ = (int)0;
    int32_t _10174_ = _2199_d0;
    for (int32_t _2214_i0 = _10173_; _2214_i0 < _10174_; ++_2214_i0)
    {
      {
        int32_t _10175_ = (int)0;
        int32_t _10176_ = _2204_d3;
        for (int32_t _2215_i3 = _10175_; _2215_i3 < _10176_; ++_2215_i3)
        {
          {
            int32_t _10177_ = (int)0;
            int32_t _10178_ = _2202_d2;
            for (int32_t _2216_i2 = _10177_; _2216_i2 < _10178_; ++_2216_i2)
            {
              {
                int32_t _10179_ = _2215_i3;
                int32_t _10180_ = _2201_d1;
                for (int32_t _2217_i1 = _10179_; _2217_i1 < _10180_; ++_2217_i1)
                {
                  {
                    int32_t _10181_ = (int)0;
                    int32_t _10182_ = _2205_d4;
                    for (int32_t _2218_i4 = _10181_; _2218_i4 < _10182_; ++_2218_i4)
                    {
                      {
                        _ptr__2208_residxs[safe_add( safe_add( safe_add( safe_add( safe_mul( _2214_i0,_2212_s0 ),safe_mul( _2217_i1,_2211_s1 ) ),safe_mul( _2216_i2,_2210_s2 ) ),safe_mul( _2215_i3,_2209_s3 ) ),_2218_i4 )] = (-safe_add( _ptr__2194_nativeidxs[_2213_k],(int)1 ));
                        _ptr__2208_residxs[safe_add( safe_add( safe_add( safe_add( safe_mul( _2214_i0,_2212_s0 ),safe_mul( _2215_i3,_2211_s1 ) ),safe_mul( _2216_i2,_2210_s2 ) ),safe_mul( _2217_i1,_2209_s3 ) ),_2218_i4 )] = (-safe_add( _ptr__2194_nativeidxs[_2213_k],(int)1 ));
                        ++ _2213_k;
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  return _2208_residxs;
}
// End mosek.fusion.PSDVariable.fullnativeidxs

void mosek::fusion::p_PSDVariable::destroy()
{
  model.reset();
  barvaridxs.reset();
  shape.reset();
  nativeidxs.reset();
}
void mosek::fusion::PSDVariable::destroy() { mosek::fusion::p_PSDVariable::_get_impl(this)->destroy(); }
// } class PSDVariable
// class LinearVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:21:4-81:5
mosek::fusion::LinearVariable::LinearVariable(mosek::fusion::p_LinearVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "LinearVariable()" << this << std::endl; */ }
mosek::fusion::LinearVariable::~LinearVariable() {  /* std::cout << "~LinearVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_LinearVariable::p_LinearVariable
  (::mosek::fusion::LinearVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.LinearVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:51:6-59:7
mosek::fusion::LinearVariable::t mosek::fusion::p_LinearVariable::_new_LinearVariable
  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2219_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2220_m)
{ return new mosek::fusion::LinearVariable(_2219_v,_2220_m); }
mosek::fusion::LinearVariable::LinearVariable
  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2219_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2220_m) : 
   LinearVariable(new mosek::fusion::p_LinearVariable(this))
{ mosek::fusion::p_LinearVariable::_get_impl(this)->_initialize(_2219_v,_2220_m); }
void mosek::fusion::p_LinearVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2219_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2220_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2219_v,_2220_m);
    nativeidxs = mosek::fusion::p_LinearVariable::_get_impl(_2219_v)->nativeidxs;
    sparsity = mosek::fusion::p_LinearVariable::_get_impl(_2219_v)->sparsity;
    shape = mosek::fusion::p_LinearVariable::_get_impl(_2219_v)->shape;
    name = mosek::fusion::p_LinearVariable::_get_impl(_2219_v)->name;
  }
}
// mosek.fusion.LinearVariable.ctor

// mosek.fusion.LinearVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:35:6-49:7
mosek::fusion::LinearVariable::t mosek::fusion::p_LinearVariable::_new_LinearVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2221_model,
    const std::string &  _2222_name,
    int64_t _2223_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2224_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2225_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2226_nativeidxs)
{ return new mosek::fusion::LinearVariable(_2221_model,_2222_name,_2223_varid,_2224_shape,_2225_sparsity,_2226_nativeidxs); }
mosek::fusion::LinearVariable::LinearVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2221_model,
    const std::string &  _2222_name,
    int64_t _2223_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2224_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2225_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2226_nativeidxs) : 
   LinearVariable(new mosek::fusion::p_LinearVariable(this))
{ mosek::fusion::p_LinearVariable::_get_impl(this)->_initialize(_2221_model,_2222_name,_2223_varid,_2224_shape,_2225_sparsity,_2226_nativeidxs); }
void mosek::fusion::p_LinearVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2221_model,
    const std::string &  _2222_name,
    int64_t _2223_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2224_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2225_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2226_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2224_shape(_2224_shape ? _2224_shape->raw() : nullptr, _2224_shape ? _2224_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2225_sparsity(_2225_sparsity ? _2225_sparsity->raw() : nullptr, _2225_sparsity ? _2225_sparsity->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2226_nativeidxs(_2226_nativeidxs ? _2226_nativeidxs->raw() : nullptr, _2226_nativeidxs ? _2226_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2221_model,_2222_name,_2224_shape,_2223_varid,_2225_sparsity,::mosek::fusion::p_LinearVariable::globalNativeIndexes(_2226_nativeidxs));
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->name = _2222_name;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->nativeidxs = _2226_nativeidxs;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->sparsity = _2225_sparsity;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->shape = _2224_shape;
  }
}
// mosek.fusion.LinearVariable.ctor

// Begin mosek.fusion.LinearVariable.toString
// Method mosek.fusion.LinearVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:65:6-79:7
std::string mosek::fusion::LinearVariable :: toString() { return mosek::fusion::p_LinearVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_LinearVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2227_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("LinearVariable(("));
  if (((int)((shape)->size(0)) > (int)0))
  {
    {
      _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
      int32_t _10183_ = (int)1;
      int32_t _10184_ = (int)((shape)->size(0));
      for (int32_t _2228_i = _10183_; _2228_i < _10184_; ++_2228_i)
      {
        {
          _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2228_i)));
        }
      }
    }
  }
  {}
  _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  if ((NULL != sparsity.get()))
  {
    {
      _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",sparse"));
    }
  }
  {}
  _2227_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2227_sb->toString();
}
// End mosek.fusion.LinearVariable.toString

// Begin mosek.fusion.LinearVariable.clone
// Method mosek.fusion.LinearVariable.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:61:6-63:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_LinearVariable::__mosek_2fusion_2LinearVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2229_m)
{
  return (::mosek::fusion::p_LinearVariable::_new_LinearVariable(::mosek::fusion::LinearVariable::t(_pubthis),_2229_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.LinearVariable.clone

// Begin mosek.fusion.LinearVariable.globalNativeIndexes
// Method mosek.fusion.LinearVariable.globalNativeIndexes @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/LinearVariable.mbi:30:6-33:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_LinearVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2230_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2230_nativeidxs(_2230_nativeidxs ? _2230_nativeidxs->raw() : nullptr, _2230_nativeidxs ? _2230_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2231_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2230_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2231_r(_2231_r ? _2231_r->raw() : nullptr,_2231_r ? _2231_r->size(0) : 0);
  int32_t _10185_ = (int)0;
  int32_t _10186_ = (int)((_2230_nativeidxs)->size(0));
  for (int32_t _2232_i = _10185_; _2232_i < _10186_; ++_2232_i)
  {
    {
      _ptr__2231_r[_2232_i] = _ptr__2230_nativeidxs[_2232_i];
    }
  }
  return _2231_r;
}
// End mosek.fusion.LinearVariable.globalNativeIndexes

void mosek::fusion::p_LinearVariable::destroy()
{
  shape.reset();
  sparsity.reset();
  nativeidxs.reset();
}
void mosek::fusion::LinearVariable::destroy() { mosek::fusion::p_LinearVariable::_get_impl(this)->destroy(); }
// } class LinearVariable
// class ConicVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:23:4-79:5
mosek::fusion::ConicVariable::ConicVariable(mosek::fusion::p_ConicVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "ConicVariable()" << this << std::endl; */ }
mosek::fusion::ConicVariable::~ConicVariable() {  /* std::cout << "~ConicVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_ConicVariable::p_ConicVariable
  (::mosek::fusion::ConicVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.ConicVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:53:6-61:7
mosek::fusion::ConicVariable::t mosek::fusion::p_ConicVariable::_new_ConicVariable
  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2233_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2234_m)
{ return new mosek::fusion::ConicVariable(_2233_v,_2234_m); }
mosek::fusion::ConicVariable::ConicVariable
  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2233_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2234_m) : 
   ConicVariable(new mosek::fusion::p_ConicVariable(this))
{ mosek::fusion::p_ConicVariable::_get_impl(this)->_initialize(_2233_v,_2234_m); }
void mosek::fusion::p_ConicVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2233_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2234_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2233_v,_2234_m);
    shape = mosek::fusion::p_ConicVariable::_get_impl(_2233_v)->shape;
    nativeidxs = mosek::fusion::p_ConicVariable::_get_impl(_2233_v)->nativeidxs;
    shape = mosek::fusion::p_ConicVariable::_get_impl(_2233_v)->shape;
    varid = mosek::fusion::p_ConicVariable::_get_impl(_2233_v)->varid;
    name = mosek::fusion::p_ConicVariable::_get_impl(_2233_v)->name;
  }
}
// mosek.fusion.ConicVariable.ctor

// mosek.fusion.ConicVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:39:6-51:7
mosek::fusion::ConicVariable::t mosek::fusion::p_ConicVariable::_new_ConicVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2235_model,
    const std::string &  _2236_name,
    int32_t _2237_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2238_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2239_nativeidxs)
{ return new mosek::fusion::ConicVariable(_2235_model,_2236_name,_2237_varid,_2238_shape,_2239_nativeidxs); }
mosek::fusion::ConicVariable::ConicVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2235_model,
    const std::string &  _2236_name,
    int32_t _2237_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2238_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2239_nativeidxs) : 
   ConicVariable(new mosek::fusion::p_ConicVariable(this))
{ mosek::fusion::p_ConicVariable::_get_impl(this)->_initialize(_2235_model,_2236_name,_2237_varid,_2238_shape,_2239_nativeidxs); }
void mosek::fusion::p_ConicVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2235_model,
    const std::string &  _2236_name,
    int32_t _2237_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2238_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2239_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2238_shape(_2238_shape ? _2238_shape->raw() : nullptr, _2238_shape ? _2238_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2239_nativeidxs(_2239_nativeidxs ? _2239_nativeidxs->raw() : nullptr, _2239_nativeidxs ? _2239_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2235_model,_2236_name,_2238_shape,_2237_varid,nullptr,::mosek::fusion::p_ConicVariable::globalNativeIndexes(_2239_nativeidxs));
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->name = _2236_name;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->varid = _2237_varid;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->shape = _2238_shape;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->nativeidxs = _2239_nativeidxs;
  }
}
// mosek.fusion.ConicVariable.ctor

// Begin mosek.fusion.ConicVariable.toString
// Method mosek.fusion.ConicVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:68:6-78:7
std::string mosek::fusion::ConicVariable :: toString() { return mosek::fusion::p_ConicVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_ConicVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2240_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("ConicVariable( "));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("("));
  _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10187_ = (int)1;
  int32_t _10188_ = (int)((shape)->size(0));
  for (int32_t _2241_i = _10187_; _2241_i < _10188_; ++_2241_i)
  {
    {
      _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2241_i)));
    }
  }
  _2240_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") )"));
  return _2240_sb->toString();
}
// End mosek.fusion.ConicVariable.toString

// Begin mosek.fusion.ConicVariable.clone
// Method mosek.fusion.ConicVariable.clone @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:63:6-66:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_ConicVariable::__mosek_2fusion_2ConicVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2242_m)
{
  return (::mosek::fusion::p_ConicVariable::_new_ConicVariable(::mosek::fusion::ConicVariable::t(_pubthis),_2242_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.ConicVariable.clone

// Begin mosek.fusion.ConicVariable.globalNativeIndexes
// Method mosek.fusion.ConicVariable.globalNativeIndexes @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ConicVariable.mbi:34:6-37:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_ConicVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2243_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2243_nativeidxs(_2243_nativeidxs ? _2243_nativeidxs->raw() : nullptr, _2243_nativeidxs ? _2243_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2244_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2243_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2244_r(_2244_r ? _2244_r->raw() : nullptr,_2244_r ? _2244_r->size(0) : 0);
  int32_t _10189_ = (int)0;
  int32_t _10190_ = (int)((_2243_nativeidxs)->size(0));
  for (int32_t _2245_i = _10189_; _2245_i < _10190_; ++_2245_i)
  {
    {
      _ptr__2244_r[_2245_i] = _ptr__2243_nativeidxs[_2245_i];
    }
  }
  return _2244_r;
}
// End mosek.fusion.ConicVariable.globalNativeIndexes

void mosek::fusion::p_ConicVariable::destroy()
{
  nativeidxs.reset();
  shape.reset();
}
void mosek::fusion::ConicVariable::destroy() { mosek::fusion::p_ConicVariable::_get_impl(this)->destroy(); }
// } class ConicVariable
// class ModelVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ModelVariable.mbi:14:4-90:5
mosek::fusion::ModelVariable::ModelVariable(mosek::fusion::p_ModelVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "ModelVariable()" << this << std::endl; */ }
mosek::fusion::ModelVariable::~ModelVariable() {  /* std::cout << "~ModelVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_ModelVariable::p_ModelVariable
  (::mosek::fusion::ModelVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.ModelVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ModelVariable.mbi:62:6-72:7
void mosek::fusion::p_ModelVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::ModelVariable > _2246_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2247_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseVariable::_initialize(_2246_v,_2247_m);
    name = mosek::fusion::p_ModelVariable::_get_impl(_2246_v)->name;
    varid = mosek::fusion::p_ModelVariable::_get_impl(_2246_v)->varid;
    modelvar_nativeidxs = mosek::fusion::p_ModelVariable::_get_impl(_2246_v)->modelvar_nativeidxs;
    shape = mosek::fusion::p_ModelVariable::_get_impl(_2246_v)->shape;
    sparsity = mosek::fusion::p_ModelVariable::_get_impl(_2246_v)->sparsity;
  }
}
// mosek.fusion.ModelVariable.ctor

// mosek.fusion.ModelVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ModelVariable.mbi:44:6-58:7
void mosek::fusion::p_ModelVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2248_model,
    const std::string &  _2249_name,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2250_shape,
    int64_t _2251_varid,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2252_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2253_modelvar_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2250_shape(_2250_shape ? _2250_shape->raw() : nullptr, _2250_shape ? _2250_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2252_sparsity(_2252_sparsity ? _2252_sparsity->raw() : nullptr, _2252_sparsity ? _2252_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2253_modelvar_nativeidxs(_2253_modelvar_nativeidxs ? _2253_modelvar_nativeidxs->raw() : nullptr, _2253_modelvar_nativeidxs ? _2253_modelvar_nativeidxs->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(_2248_model,_2250_shape,_2252_sparsity,_2253_modelvar_nativeidxs);
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->name = _2249_name;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->varid = _2251_varid;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->shape = _2250_shape;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->sparsity = _2252_sparsity;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->modelvar_nativeidxs = _2253_modelvar_nativeidxs;
  }
}
// mosek.fusion.ModelVariable.ctor

// Begin mosek.fusion.ModelVariable.elementName
// Method mosek.fusion.ModelVariable.elementName @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ModelVariable.mbi:84:6-88:7
void mosek::fusion::ModelVariable :: elementName(int64_t _2254_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2255_sb) { mosek::fusion::p_ModelVariable::_get_impl(this)->elementName(_2254_index,_2255_sb); }
void mosek::fusion::p_ModelVariable::elementName(int64_t _2254_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2255_sb)
{
  std::string _10191_;
  bool _10192_ = ((int)((name).size()) == (int)0);
  if (_10192_)
  {
    _10191_ = std::string ("_");
  }
  else
  {
    _10191_ = name;
  }
  _2255_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_10191_)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["))->__mosek_2fusion_2Utils_2StringBuffer__a(::mosek::fusion::p_Set::indexToString(shape,_2254_index))->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
}
// End mosek.fusion.ModelVariable.elementName

// Begin mosek.fusion.ModelVariable.remove
// Method mosek.fusion.ModelVariable.remove @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/ModelVariable.mbi:78:6-80:7
void mosek::fusion::ModelVariable :: remove() { mosek::fusion::p_ModelVariable::_get_impl(this)->remove(); }
void mosek::fusion::p_ModelVariable::remove()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->removeVariableBlock(varid);
}
// End mosek.fusion.ModelVariable.remove

// Begin mosek.fusion.ModelVariable.clone
// End mosek.fusion.ModelVariable.clone

void mosek::fusion::p_ModelVariable::destroy()
{
  sparsity.reset();
  shape.reset();
  modelvar_nativeidxs.reset();
}
void mosek::fusion::ModelVariable::destroy() { mosek::fusion::p_ModelVariable::_get_impl(this)->destroy(); }
// } class ModelVariable
// class NilVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1013:4-1098:5
mosek::fusion::NilVariable::NilVariable(mosek::fusion::p_NilVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "NilVariable()" << this << std::endl; */ }
mosek::fusion::NilVariable::~NilVariable() {  /* std::cout << "~NilVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_NilVariable::p_NilVariable
  (::mosek::fusion::NilVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.NilVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1020:6-1024:7
mosek::fusion::NilVariable::t mosek::fusion::p_NilVariable::_new_NilVariable
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2257_shape)
{ return new mosek::fusion::NilVariable(_2257_shape); }
mosek::fusion::NilVariable::NilVariable
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2257_shape) : 
   NilVariable(new mosek::fusion::p_NilVariable(this))
{ mosek::fusion::p_NilVariable::_get_impl(this)->_initialize(_2257_shape); }
void mosek::fusion::p_NilVariable::_initialize  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2257_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2257_shape(_2257_shape ? _2257_shape->raw() : nullptr, _2257_shape ? _2257_shape->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(nullptr,::mosek::fusion::Utils::Tools::arraycopy(_2257_shape),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))));
    std::vector<int32_t> _10193_;
    for (int32_t _2258_i = (int)0, _10194_ = (int)((_2257_shape)->size(0)); _2258_i < _10194_ ; ++_2258_i)
    {
      _10193_.push_back(_ptr__2257_shape[_2258_i]);
    }
    auto _10195_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10193_.size()),_10193_.begin(),_10193_.end()));
    mosek::fusion::p_NilVariable::_get_impl(::mosek::fusion::NilVariable::t(_pubthis))->shape = _10195_;
  }
}
// mosek.fusion.NilVariable.ctor

// mosek.fusion.NilVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1018:6-45
mosek::fusion::NilVariable::t mosek::fusion::p_NilVariable::_new_NilVariable
  ()
{ return new mosek::fusion::NilVariable(); }
mosek::fusion::NilVariable::NilVariable
  () : 
   NilVariable(new mosek::fusion::p_NilVariable(this))
{ mosek::fusion::p_NilVariable::_get_impl(this)->_initialize(); }
void mosek::fusion::p_NilVariable::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_NilVariable::_initialize(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
  }
}
// mosek.fusion.NilVariable.ctor

// Begin mosek.fusion.NilVariable.elementDesc
// Method mosek.fusion.NilVariable.elementDesc @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1096:6-102
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__elementDesc(int64_t _2259_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2260_sb) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__elementDesc(_2259_index,_2260_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::NilVariable::elementDesc(int64_t _2259_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2260_sb) { return __mosek_2fusion_2NilVariable__elementDesc(_2259_index,_2260_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__elementDesc(int64_t _2259_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2260_sb)
{
  return _2260_sb;
}
// End mosek.fusion.NilVariable.elementDesc

// Begin mosek.fusion.NilVariable.elementName
// Method mosek.fusion.NilVariable.elementName @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1095:6-76
void mosek::fusion::NilVariable :: elementName(int64_t _2261_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2262_sb) { mosek::fusion::p_NilVariable::_get_impl(this)->elementName(_2261_index,_2262_sb); }
void mosek::fusion::p_NilVariable::elementName(int64_t _2261_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2262_sb)
{
}
// End mosek.fusion.NilVariable.elementName

// Begin mosek.fusion.NilVariable.numInst
// Method mosek.fusion.NilVariable.numInst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1091:6-1093:7
int32_t mosek::fusion::NilVariable :: numInst() { return mosek::fusion::p_NilVariable::_get_impl(this)->numInst(); }
int32_t mosek::fusion::p_NilVariable::numInst()
{
  return (int)0;
}
// End mosek.fusion.NilVariable.numInst

// Begin mosek.fusion.NilVariable.inst
// Method mosek.fusion.NilVariable.inst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1083:6-1089:7
int32_t mosek::fusion::NilVariable :: inst(int32_t _2263_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2264_sparsity,std::shared_ptr< monty::ndarray< int64_t,1 > > _2265_basevar_nativeidxs) { return mosek::fusion::p_NilVariable::_get_impl(this)->inst(_2263_offset,_2264_sparsity,_2265_basevar_nativeidxs); }
int32_t mosek::fusion::p_NilVariable::inst(int32_t _2263_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2264_sparsity,std::shared_ptr< monty::ndarray< int64_t,1 > > _2265_basevar_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2264_sparsity(_2264_sparsity ? _2264_sparsity->raw() : nullptr, _2264_sparsity ? _2264_sparsity->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2265_basevar_nativeidxs(_2265_basevar_nativeidxs ? _2265_basevar_nativeidxs->raw() : nullptr, _2265_basevar_nativeidxs ? _2265_basevar_nativeidxs->size(0) : 0);
  return (int)0;
}
// End mosek.fusion.NilVariable.inst

// Begin mosek.fusion.NilVariable.inst
// Method mosek.fusion.NilVariable.inst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1078:6-1081:8
void mosek::fusion::NilVariable :: inst(int32_t _2266_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2267_nindex) { mosek::fusion::p_NilVariable::_get_impl(this)->inst(_2266_offset,_2267_nindex); }
void mosek::fusion::p_NilVariable::inst(int32_t _2266_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2267_nindex)
{
  _checked_ptr_1<int64_t> _ptr__2267_nindex(_2267_nindex ? _2267_nindex->raw() : nullptr, _2267_nindex ? _2267_nindex->size(0) : 0);
}
// End mosek.fusion.NilVariable.inst

// Begin mosek.fusion.NilVariable.set_values
// Method mosek.fusion.NilVariable.set_values @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1073:6-1076:8
void mosek::fusion::NilVariable :: set_values(std::shared_ptr< monty::ndarray< double,1 > > _2268_target,bool _2269_primal) { mosek::fusion::p_NilVariable::_get_impl(this)->set_values(_2268_target,_2269_primal); }
void mosek::fusion::p_NilVariable::set_values(std::shared_ptr< monty::ndarray< double,1 > > _2268_target,bool _2269_primal)
{
  _checked_ptr_1<double> _ptr__2268_target(_2268_target ? _2268_target->raw() : nullptr, _2268_target ? _2268_target->size(0) : 0);
}
// End mosek.fusion.NilVariable.set_values

// Begin mosek.fusion.NilVariable.values
// Method mosek.fusion.NilVariable.values @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1067:6-1071:8
void mosek::fusion::NilVariable :: values(int32_t _2270_offset,std::shared_ptr< monty::ndarray< double,1 > > _2271_target,bool _2272_primal) { mosek::fusion::p_NilVariable::_get_impl(this)->values(_2270_offset,_2271_target,_2272_primal); }
void mosek::fusion::p_NilVariable::values(int32_t _2270_offset,std::shared_ptr< monty::ndarray< double,1 > > _2271_target,bool _2272_primal)
{
  _checked_ptr_1<double> _ptr__2271_target(_2271_target ? _2271_target->raw() : nullptr, _2271_target ? _2271_target->size(0) : 0);
}
// End mosek.fusion.NilVariable.values

// Begin mosek.fusion.NilVariable.make_continuous
// Method mosek.fusion.NilVariable.make_continuous @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1065:6-40
void mosek::fusion::NilVariable :: make_continuous() { mosek::fusion::p_NilVariable::_get_impl(this)->make_continuous(); }
void mosek::fusion::p_NilVariable::make_continuous()
{
}
// End mosek.fusion.NilVariable.make_continuous

// Begin mosek.fusion.NilVariable.make_integer
// Method mosek.fusion.NilVariable.make_integer @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1064:6-37
void mosek::fusion::NilVariable :: make_integer() { mosek::fusion::p_NilVariable::_get_impl(this)->make_integer(); }
void mosek::fusion::p_NilVariable::make_integer()
{
}
// End mosek.fusion.NilVariable.make_integer

// Begin mosek.fusion.NilVariable.toString
// Method mosek.fusion.NilVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1062:6-63
std::string mosek::fusion::NilVariable :: toString() { return mosek::fusion::p_NilVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_NilVariable::toString()
{
  return std::string ("NilVariable");
}
// End mosek.fusion.NilVariable.toString

// Begin mosek.fusion.NilVariable.index
// Method mosek.fusion.NilVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1056:6-1060:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2273_first) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__index(_2273_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2273_first) { return __mosek_2fusion_2NilVariable__index(_2273_first); }
/* override: mosek.fusion.BaseVariable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2273_first) { return __mosek_2fusion_2NilVariable__index(_2273_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2273_first)
{
  _checked_ptr_1<int32_t> _ptr__2273_first(_2273_first ? _2273_first->raw() : nullptr, _2273_first ? _2273_first->size(0) : 0);
  std::vector<int32_t> _10196_;
  for (int32_t _2274_i = (int)0, _10197_ = (int)((shape)->size(0)); _2274_i < _10197_ ; ++_2274_i)
  {
    _10196_.push_back((int)1);
  }
  auto _10198_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10196_.size()),_10196_.begin(),_10196_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10198_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.index

// Begin mosek.fusion.NilVariable.index
// Method mosek.fusion.NilVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1050:6-1054:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__index(int32_t _2275_first) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__index(_2275_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::index(int32_t _2275_first) { return __mosek_2fusion_2NilVariable__index(_2275_first); }
/* override: mosek.fusion.BaseVariable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2275_first) { return __mosek_2fusion_2NilVariable__index(_2275_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__index(int32_t _2275_first)
{
  std::vector<int32_t> _10199_;
  for (int32_t _2276_i = (int)0, _10200_ = (int)((shape)->size(0)); _2276_i < _10200_ ; ++_2276_i)
  {
    _10199_.push_back((int)1);
  }
  auto _10201_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10199_.size()),_10199_.begin(),_10199_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10201_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.index

// Begin mosek.fusion.NilVariable.slice
// Method mosek.fusion.NilVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1038:6-1048:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2277_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2278_last) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__slice(_2277_first,_2278_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2277_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2278_last) { return __mosek_2fusion_2NilVariable__slice(_2277_first,_2278_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2277_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2278_last) { return __mosek_2fusion_2NilVariable__slice(_2277_first,_2278_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2277_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2278_last)
{
  _checked_ptr_1<int32_t> _ptr__2277_first(_2277_first ? _2277_first->raw() : nullptr, _2277_first ? _2277_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2278_last(_2278_last ? _2278_last->raw() : nullptr, _2278_last ? _2278_last->size(0) : 0);
  if ((((int)((shape)->size(0)) != (int)((_2277_first)->size(0))) || ((int)((shape)->size(0)) != (int)((_2278_last)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching slice shape"));
    }
  }
  {}
  int32_t _10202_ = (int)0;
  int32_t _10203_ = (int)((_2277_first)->size(0));
  for (int32_t _2279_i = _10202_; _2279_i < _10203_; ++_2279_i)
  {
    {
      if (((_ptr__2278_last[_2279_i] > _ptr__2277_first[_2279_i]) || ((_ptr__2277_first[_2279_i] < (int)0) || (_ptr__2278_last[_2279_i] >= ((*shape)(_2279_i))))))
      {
        {
          throw ::mosek::fusion::SliceError(std::string ("Slice is out of bounds"));
        }
      }
      {}
    }
  }
  std::vector<int32_t> _10204_;
  for (int32_t _2280_i = (int)0, _10205_ = (int)((shape)->size(0)); _2280_i < _10205_ ; ++_2280_i)
  {
    _10204_.push_back((_ptr__2278_last[_2280_i] - _ptr__2277_first[_2280_i]));
  }
  auto _10206_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10204_.size()),_10204_.begin(),_10204_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10206_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.slice

// Begin mosek.fusion.NilVariable.slice
// Method mosek.fusion.NilVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:1026:6-1036:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__slice(int32_t _2281_first,int32_t _2282_last) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__slice(_2281_first,_2282_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::slice(int32_t _2281_first,int32_t _2282_last) { return __mosek_2fusion_2NilVariable__slice(_2281_first,_2282_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2281_first,int32_t _2282_last) { return __mosek_2fusion_2NilVariable__slice(_2281_first,_2282_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__slice(int32_t _2281_first,int32_t _2282_last)
{
  if (((_2282_last > _2281_first) || ((_2281_first < (int)0) || (_2282_last > ((*shape)((int)0))))))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Slice is out of bounds"));
    }
  }
  else if(((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching slice shape"));
    }
  }
  else
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((_2282_last - _2281_first))}))).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.NilVariable.slice

void mosek::fusion::p_NilVariable::destroy()
{
  shape.reset();
}
void mosek::fusion::NilVariable::destroy() { mosek::fusion::p_NilVariable::_get_impl(this)->destroy(); }
// } class NilVariable
// class BaseVariable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:53:4-982:5
mosek::fusion::BaseVariable::BaseVariable(mosek::fusion::p_BaseVariable *_impl) : _impl(_impl) { /*std::cout << "BaseVariable() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::BaseVariable::~BaseVariable() { delete _impl; _impl = nullptr; }
mosek::fusion::p_BaseVariable::p_BaseVariable
  (::mosek::fusion::BaseVariable * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.BaseVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:99:6-108:7
void mosek::fusion::p_BaseVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::BaseVariable > _2283_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2284_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    model = _2284_m;
    shape = mosek::fusion::p_BaseVariable::_get_impl(_2283_v)->shape;
    sparsity = mosek::fusion::p_BaseVariable::_get_impl(_2283_v)->sparsity;
    basevar_nativeidxs = mosek::fusion::p_BaseVariable::_get_impl(_2283_v)->basevar_nativeidxs;
  }
}
// mosek.fusion.BaseVariable.ctor

// mosek.fusion.BaseVariable.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:83:6-94:7
void mosek::fusion::p_BaseVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2285_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2286_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2287_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2288_basevar_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2286_shape(_2286_shape ? _2286_shape->raw() : nullptr, _2286_shape ? _2286_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2287_sparsity(_2287_sparsity ? _2287_sparsity->raw() : nullptr, _2287_sparsity ? _2287_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2288_basevar_nativeidxs(_2288_basevar_nativeidxs ? _2288_basevar_nativeidxs->raw() : nullptr, _2288_basevar_nativeidxs ? _2288_basevar_nativeidxs->size(0) : 0);
    model = _2285_m;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->shape = _2286_shape;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity = _2287_sparsity;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs = _2288_basevar_nativeidxs;
  }
}
// mosek.fusion.BaseVariable.ctor

// Begin mosek.fusion.BaseVariable.toString
// Method mosek.fusion.BaseVariable.toString @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:970:6-981:7
std::string mosek::fusion::BaseVariable :: toString() { return mosek::fusion::p_BaseVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_BaseVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2289_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2289_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Variable(("));
  if (((int)((shape)->size(0)) > (int)0))
  {
    {
      _2289_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
      int32_t _10207_ = (int)1;
      int32_t _10208_ = (int)((shape)->size(0));
      for (int32_t _2290_i = _10207_; _2290_i < _10208_; ++_2290_i)
      {
        {
          _2289_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2290_i)));
        }
      }
    }
  }
  {}
  _2289_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("))"));
  return _2289_sb->toString();
}
// End mosek.fusion.BaseVariable.toString

// Begin mosek.fusion.BaseVariable.eval
// Method mosek.fusion.BaseVariable.eval @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:939:6-967:7
void mosek::fusion::BaseVariable :: eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2291_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2292_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2293_xs,bool _2294_cache_eval) { mosek::fusion::p_BaseVariable::_get_impl(this)->eval(_2291_rs,_2292_ws,_2293_xs,_2294_cache_eval); }
void mosek::fusion::p_BaseVariable::eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2291_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2292_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2293_xs,bool _2294_cache_eval)
{
  int32_t _2295_nnz = (int)((basevar_nativeidxs)->size(0));
  int32_t _2296_nelem = _2295_nnz;
  int32_t _2297_nd = (int)((shape)->size(0));
  bool _2298_hassp = (NULL != sparsity.get());
  _2291_rs->alloc_expr(_2297_nd,_2296_nelem,_2295_nnz,_2298_hassp);
  int32_t _2299_ptr = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->ptr_base;
  int32_t _2300_nidxs = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->nidxs_base;
  int32_t _2301_sp = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->sp_base;
  int32_t _2302_cof = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->cof_base;
  int32_t _2303_shape_base = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->shape_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2304_ri32 = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__2304_ri32(_2304_ri32 ? _2304_ri32->raw() : nullptr,_2304_ri32 ? _2304_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2305_ri64 = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__2305_ri64(_2305_ri64 ? _2305_ri64->raw() : nullptr,_2305_ri64 ? _2305_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2306_rf64 = mosek::fusion::p_WorkStack::_get_impl(_2291_rs)->f64;
  _checked_ptr_1<double> _ptr__2306_rf64(_2306_rf64 ? _2306_rf64->raw() : nullptr,_2306_rf64 ? _2306_rf64->size(0) : 0);
  int32_t _10209_ = (int)0;
  int32_t _10210_ = _2297_nd;
  for (int32_t _2307_i = _10209_; _2307_i < _10210_; ++_2307_i)
  {
    {
      _ptr__2304_ri32[safe_add( _2303_shape_base,_2307_i )] = ((*shape)(_2307_i));
    }
  }
  {
    int32_t _10211_ = (int)0;
    int32_t _10212_ = safe_add( _2295_nnz,(int)1 );
    for (int32_t _2308_i = _10211_; _2308_i < _10212_; ++_2308_i)
    {
      {
        _ptr__2304_ri32[safe_add( _2299_ptr,_2308_i )] = _2308_i;
      }
    }
    int32_t _10213_ = (int)0;
    int32_t _10214_ = _2295_nnz;
    for (int32_t _2309_i = _10213_; _2309_i < _10214_; ++_2309_i)
    {
      {
        _ptr__2305_ri64[safe_add( _2300_nidxs,_2309_i )] = ((*basevar_nativeidxs)(_2309_i));
      }
    }
    if (_2298_hassp)
    {
      {
        int32_t _10215_ = (int)0;
        int32_t _10216_ = _2295_nnz;
        for (int32_t _2310_i = _10215_; _2310_i < _10216_; ++_2310_i)
        {
          {
            _ptr__2305_ri64[safe_add( _2301_sp,_2310_i )] = ((*sparsity)(_2310_i));
          }
        }
      }
    }
    {}
    int32_t _10217_ = (int)0;
    int32_t _10218_ = _2295_nnz;
    for (int32_t _2311_i = _10217_; _2311_i < _10218_; ++_2311_i)
    {
      {
        _ptr__2306_rf64[safe_add( _2302_cof,_2311_i )] = 1.0;
      }
    }
  }
}
// End mosek.fusion.BaseVariable.eval

// Begin mosek.fusion.BaseVariable.compress_when_caching
// Method mosek.fusion.BaseVariable.compress_when_caching @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:936:6-66
bool mosek::fusion::BaseVariable :: compress_when_caching() { return mosek::fusion::p_BaseVariable::_get_impl(this)->compress_when_caching(); }
bool mosek::fusion::p_BaseVariable::compress_when_caching()
{
  return false;
}
// End mosek.fusion.BaseVariable.compress_when_caching

// Begin mosek.fusion.BaseVariable.eval
// Method mosek.fusion.BaseVariable.eval @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:933:6-102
void mosek::fusion::BaseVariable :: eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2312_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2313_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2314_xs) { mosek::fusion::p_BaseVariable::_get_impl(this)->eval(_2312_rs,_2313_ws,_2314_xs); }
void mosek::fusion::p_BaseVariable::eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2312_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2313_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2314_xs)
{
  _pubthis->recursive_eval(_2312_rs,_2313_ws,_2314_xs,false);
}
// End mosek.fusion.BaseVariable.eval

// Begin mosek.fusion.BaseVariable.recursive_eval
// Method mosek.fusion.BaseVariable.recursive_eval @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:927:6-930:7
void mosek::fusion::BaseVariable :: recursive_eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2315_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2316_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2317_xs,bool _2318_cache_eval) { mosek::fusion::p_BaseVariable::_get_impl(this)->recursive_eval(_2315_rs,_2316_ws,_2317_xs,_2318_cache_eval); }
void mosek::fusion::p_BaseVariable::recursive_eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2315_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2316_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2317_xs,bool _2318_cache_eval)
{
  _pubthis->eval(_2315_rs,_2316_ws,_2317_xs,_2318_cache_eval);
}
// End mosek.fusion.BaseVariable.recursive_eval

// Begin mosek.fusion.BaseVariable.remove
// Method mosek.fusion.BaseVariable.remove @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:920:6-922:7
void mosek::fusion::BaseVariable :: remove() { mosek::fusion::p_BaseVariable::_get_impl(this)->remove(); }
void mosek::fusion::p_BaseVariable::remove()
{
  throw ::mosek::fusion::DeletionError(std::string ("Only original variables can be removed"));
}
// End mosek.fusion.BaseVariable.remove

// Begin mosek.fusion.BaseVariable.fromTril
// Method mosek.fusion.BaseVariable.fromTril @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:844:6-918:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__fromTril(int32_t _2319_dim0,int32_t _2320_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__fromTril(_2319_dim0,_2320_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::fromTril(int32_t _2319_dim0,int32_t _2320_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2319_dim0,_2320_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__fromTril(int32_t _2319_dim0,int32_t _2320_d)
{
  if (((_2319_dim0 < (int)0) || (_2319_dim0 >= (int)((shape)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid dimension"));
    }
  }
  else if((((*shape)(_2319_dim0)) != (safe_mul( _2320_d,safe_add( _2320_d,(int)1 ) ) / (int)2)))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid shape for operation"));
    }
  }
  {}
  int32_t _2321_d0 = (int)1;
  int32_t _10219_ = (int)0;
  int32_t _10220_ = _2319_dim0;
  for (int32_t _2322_i = _10219_; _2322_i < _10220_; ++_2322_i)
  {
    {
      _2321_d0 *= ((*shape)(_2322_i));
    }
  }
  int32_t _2323_d1 = ((*shape)(_2319_dim0));
  int32_t _2324_d2 = (int)1;
  int32_t _10221_ = safe_add( _2319_dim0,(int)1 );
  int32_t _10222_ = (int)((shape)->size(0));
  for (int32_t _2325_i = _10221_; _2325_i < _10222_; ++_2325_i)
  {
    {
      _2324_d2 *= ((*shape)(_2325_i));
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2326_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( (int)((shape)->size(0)),(int)1 ))));
  _checked_ptr_1<int32_t> _ptr__2326_rshape(_2326_rshape ? _2326_rshape->raw() : nullptr,_2326_rshape ? _2326_rshape->size(0) : 0);
  int32_t _10223_ = (int)0;
  int32_t _10224_ = _2319_dim0;
  for (int32_t _2327_i = _10223_; _2327_i < _10224_; ++_2327_i)
  {
    {
      _ptr__2326_rshape[_2327_i] = ((*shape)(_2327_i));
    }
  }
  _ptr__2326_rshape[_2319_dim0] = _2320_d;
  _ptr__2326_rshape[safe_add( _2319_dim0,(int)1 )] = _2320_d;
  int32_t _10225_ = safe_add( _2319_dim0,(int)1 );
  int32_t _10226_ = (int)((shape)->size(0));
  for (int32_t _2328_i = _10225_; _2328_i < _10226_; ++_2328_i)
  {
    {
      _ptr__2326_rshape[safe_add( _2328_i,(int)1 )] = ((*shape)(_2328_i));
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2329_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_mul( safe_mul( safe_mul( _2321_d0,_2320_d ),_2320_d ),_2324_d2 ))));
      _checked_ptr_1<int64_t> _ptr__2329_nidxs(_2329_nidxs ? _2329_nidxs->raw() : nullptr,_2329_nidxs ? _2329_nidxs->size(0) : 0);
      int32_t _2330_ofs = (int)0;
      int32_t _10227_ = (int)0;
      int32_t _10228_ = _2321_d0;
      for (int32_t _2331_i0 = _10227_; _2331_i0 < _10228_; ++_2331_i0)
      {
        {
          int32_t _10229_ = (int)0;
          int32_t _10230_ = _2320_d;
          for (int32_t _2332_i10 = _10229_; _2332_i10 < _10230_; ++_2332_i10)
          {
            {
              int32_t _10231_ = (int)0;
              int32_t _10232_ = _2320_d;
              for (int32_t _2333_i11 = _10231_; _2333_i11 < _10232_; ++_2333_i11)
              {
                {
                  int32_t _10233_ = (int)0;
                  int32_t _10234_ = _2324_d2;
                  for (int32_t _2334_i2 = _10233_; _2334_i2 < _10234_; ++_2334_i2)
                  {
                    {
                      if ((_2332_i10 >= _2333_i11))
                      {
                        {
                          _ptr__2329_nidxs[_2330_ofs] = ((*basevar_nativeidxs)(safe_add( safe_add( safe_mul( safe_mul( _2331_i0,_2323_d1 ),_2324_d2 ),safe_mul( safe_add( (safe_mul( _2332_i10,safe_add( _2332_i10,(int)1 ) ) / (int)2),_2333_i11 ),_2324_d2 ) ),_2334_i2 )));
                        }
                      }
                      else
                      {
                        {
                          _ptr__2329_nidxs[_2330_ofs] = ((*basevar_nativeidxs)(safe_add( safe_add( safe_mul( safe_mul( _2331_i0,_2323_d1 ),_2324_d2 ),safe_mul( safe_add( (safe_mul( _2333_i11,safe_add( _2333_i11,(int)1 ) ) / (int)2),_2332_i10 ),_2324_d2 ) ),_2334_i2 )));
                        }
                      }
                      ++ _2330_ofs;
                    }
                  }
                }
              }
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2326_rshape,nullptr,_2329_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2335_nelm = (int)0;
      int32_t _10235_ = (int)0;
      int32_t _10236_ = (int)((sparsity)->size(0));
      for (int32_t _2336_i = _10235_; _2336_i < _10236_; ++_2336_i)
      {
        {
          int64_t _2337_i1 = ((((*sparsity)(_2336_i)) / _2324_d2) % _2323_d1);
          int32_t _2338_ii = (int32_t)(((-(int)1) + ::mosek::fusion::Utils::Tools::sqrt((double)(safe_mul( safe_add( (int)1,safe_mul( (int)2,_2337_i1 ) ),safe_add( (int)1,safe_mul( (int)2,_2337_i1 ) ) )))) / (int)4);
          int64_t _2339_jj = (((*sparsity)(_2336_i)) - (safe_mul( _2338_ii,safe_add( _2338_ii,(int)1 ) ) / (int)2));
          if ((_2338_ii == _2339_jj))
          {
            {
              _2335_nelm += (int)1;
            }
          }
          else
          {
            {
              _2335_nelm += (int)2;
            }
          }
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2340_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2335_nelm)));
      _checked_ptr_1<int64_t> _ptr__2340_nidxs(_2340_nidxs ? _2340_nidxs->raw() : nullptr,_2340_nidxs ? _2340_nidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2341_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2335_nelm)));
      _checked_ptr_1<int64_t> _ptr__2341_sp(_2341_sp ? _2341_sp->raw() : nullptr,_2341_sp ? _2341_sp->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2342_perm = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2335_nelm)));
      _checked_ptr_1<int64_t> _ptr__2342_perm(_2342_perm ? _2342_perm->raw() : nullptr,_2342_perm ? _2342_perm->size(0) : 0);
      int32_t _10237_ = (int)0;
      int32_t _10238_ = _2335_nelm;
      for (int32_t _2343_i = _10237_; _2343_i < _10238_; ++_2343_i)
      {
        {
          _ptr__2342_perm[_2343_i] = _2343_i;
        }
      }
      int32_t _2344_elmi = (int)0;
      int32_t _10239_ = (int)0;
      int32_t _10240_ = (int)((sparsity)->size(0));
      for (int32_t _2345_i = _10239_; _2345_i < _10240_; ++_2345_i)
      {
        {
          int64_t _2346_i0 = (((*sparsity)(_2345_i)) / safe_mul( _2323_d1,_2324_d2 ));
          int64_t _2347_i1 = ((((*sparsity)(_2345_i)) / _2324_d2) % _2323_d1);
          int64_t _2348_i2 = (((*sparsity)(_2345_i)) % _2324_d2);
          int32_t _2349_ii1 = (int32_t)(((-(int)1) + ::mosek::fusion::Utils::Tools::sqrt((double)(safe_mul( safe_add( (int)1,safe_mul( (int)2,_2347_i1 ) ),safe_add( (int)1,safe_mul( (int)2,_2347_i1 ) ) )))) / (int)4);
          int64_t _2350_jj1 = (((*sparsity)(_2345_i)) - (safe_mul( _2349_ii1,safe_add( _2349_ii1,(int)1 ) ) / (int)2));
          if ((_2349_ii1 == _2350_jj1))
          {
            {
              _ptr__2340_nidxs[_2344_elmi] = ((*basevar_nativeidxs)(_2345_i));
              _ptr__2341_sp[_2344_elmi] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2346_i0,_2320_d ),_2320_d ),_2324_d2 ),safe_mul( safe_add( safe_mul( _2349_ii1,_2320_d ),_2350_jj1 ),_2324_d2 ) ),_2348_i2 );
              _2344_elmi += (int)1;
            }
          }
          else
          {
            {
              _ptr__2340_nidxs[_2344_elmi] = ((*basevar_nativeidxs)(_2345_i));
              _ptr__2340_nidxs[safe_add( _2344_elmi,(int)1 )] = ((*basevar_nativeidxs)(_2345_i));
              _ptr__2341_sp[_2344_elmi] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2346_i0,_2320_d ),_2320_d ),_2324_d2 ),safe_mul( safe_add( safe_mul( _2349_ii1,_2320_d ),_2350_jj1 ),_2324_d2 ) ),_2348_i2 );
              _ptr__2341_sp[safe_add( _2344_elmi,(int)1 )] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2346_i0,_2320_d ),_2320_d ),_2324_d2 ),safe_mul( safe_add( safe_mul( _2350_jj1,_2320_d ),_2349_ii1 ),_2324_d2 ) ),_2348_i2 );
              _2344_elmi += (int)2;
            }
          }
        }
      }
      ::mosek::fusion::p_CommonTools::argQsort(_2342_perm,_2340_nidxs,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2335_nelm));
      std::vector<int64_t> _10241_;
      for (int32_t _2351_i = (int)0, _10242_ = _2335_nelm; _2351_i < _10242_ ; ++_2351_i)
      {
        _10241_.push_back(_ptr__2341_sp[_ptr__2342_perm[_2351_i]]);
      }
      auto _10243_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10241_.size()),_10241_.begin(),_10241_.end()));
      std::vector<int64_t> _10244_;
      for (int32_t _2352_i = (int)0, _10245_ = _2335_nelm; _2352_i < _10245_ ; ++_2352_i)
      {
        _10244_.push_back(_ptr__2340_nidxs[_ptr__2342_perm[_2352_i]]);
      }
      auto _10246_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10244_.size()),_10244_.begin(),_10244_.end()));
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2326_rshape,_10243_,_10246_)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.fromTril

// Begin mosek.fusion.BaseVariable.fromTril
// Method mosek.fusion.BaseVariable.fromTril @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:827:6-842:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__fromTril(int32_t _2353_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__fromTril(_2353_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::fromTril(int32_t _2353_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2353_d); }
/* override: mosek.fusion.Variable.fromTril*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__fromTril(int32_t _2353_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2353_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__fromTril(int32_t _2353_d)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__fromTril((int)0,_2353_d);
}
// End mosek.fusion.BaseVariable.fromTril

// Begin mosek.fusion.BaseVariable.tril
// Method mosek.fusion.BaseVariable.tril @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:738:6-825:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__tril(int32_t _2354_dim1,int32_t _2355_dim2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__tril(_2354_dim1,_2355_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::tril(int32_t _2354_dim1,int32_t _2355_dim2) { return __mosek_2fusion_2BaseVariable__tril(_2354_dim1,_2355_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__tril(int32_t _2354_dim1,int32_t _2355_dim2)
{
  if (((_2354_dim1 < (int)0) || ((_2354_dim1 >= _2355_dim2) || (_2355_dim2 >= (int)((shape)->size(0))))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid dimension specification"));
    }
  }
  else
  {
    {
      if ((((*shape)(_2354_dim1)) != ((*shape)(_2355_dim2))))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Operation only valid for square variables"));
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2356_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((shape)->size(0)) - (int)1))));
  _checked_ptr_1<int32_t> _ptr__2356_rshape(_2356_rshape ? _2356_rshape->raw() : nullptr,_2356_rshape ? _2356_rshape->size(0) : 0);
  int32_t _10247_ = (int)0;
  int32_t _10248_ = _2354_dim1;
  for (int32_t _2357_i = _10247_; _2357_i < _10248_; ++_2357_i)
  {
    {
      _ptr__2356_rshape[_2357_i] = ((*shape)(_2357_i));
    }
  }
  _ptr__2356_rshape[_2354_dim1] = (safe_mul( ((*shape)(_2354_dim1)),safe_add( ((*shape)(_2354_dim1)),(int)1 ) ) / (int)2);
  int32_t _10249_ = safe_add( _2354_dim1,(int)1 );
  int32_t _10250_ = _2355_dim2;
  for (int32_t _2358_i = _10249_; _2358_i < _10250_; ++_2358_i)
  {
    {
      _ptr__2356_rshape[_2358_i] = ((*shape)(_2358_i));
    }
  }
  int32_t _10251_ = safe_add( _2355_dim2,(int)1 );
  int32_t _10252_ = (int)((shape)->size(0));
  for (int32_t _2359_i = _10251_; _2359_i < _10252_; ++_2359_i)
  {
    {
      _ptr__2356_rshape[(_2359_i - (int)1)] = ((*shape)(_2359_i));
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      int32_t _2360_d0 = (int)1;
      int32_t _10253_ = (int)0;
      int32_t _10254_ = _2354_dim1;
      for (int32_t _2361_i = _10253_; _2361_i < _10254_; ++_2361_i)
      {
        {
          _2360_d0 *= ((*shape)(_2361_i));
        }
      }
      int32_t _2362_d1 = ((*shape)(_2354_dim1));
      int32_t _2363_d2 = (int)1;
      int32_t _10255_ = safe_add( _2354_dim1,(int)1 );
      int32_t _10256_ = _2355_dim2;
      for (int32_t _2364_i = _10255_; _2364_i < _10256_; ++_2364_i)
      {
        {
          _2363_d2 *= ((*shape)(_2364_i));
        }
      }
      int32_t _2365_d3 = ((*shape)(_2355_dim2));
      int32_t _2366_d4 = (int)1;
      int32_t _10257_ = safe_add( _2355_dim2,(int)1 );
      int32_t _10258_ = (int)((shape)->size(0));
      for (int32_t _2367_i = _10257_; _2367_i < _10258_; ++_2367_i)
      {
        {
          _2366_d4 *= ((*shape)(_2367_i));
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2368_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((safe_mul( safe_mul( safe_mul( safe_mul( _2360_d0,_2363_d2 ),_2366_d4 ),_2362_d1 ),safe_add( _2362_d1,(int)1 ) ) / (int)2))));
      _checked_ptr_1<int64_t> _ptr__2368_nidxs(_2368_nidxs ? _2368_nidxs->raw() : nullptr,_2368_nidxs ? _2368_nidxs->size(0) : 0);
      int32_t _2369_i = (int)0;
      int32_t _10259_ = (int)0;
      int32_t _10260_ = _2360_d0;
      for (int32_t _2370_i0 = _10259_; _2370_i0 < _10260_; ++_2370_i0)
      {
        {
          int32_t _10261_ = (int)0;
          int32_t _10262_ = _2362_d1;
          for (int32_t _2371_i1 = _10261_; _2371_i1 < _10262_; ++_2371_i1)
          {
            {
              int32_t _10263_ = (int)0;
              int32_t _10264_ = safe_add( _2371_i1,(int)1 );
              for (int32_t _2372_i3 = _10263_; _2372_i3 < _10264_; ++_2372_i3)
              {
                {
                  int32_t _10265_ = (int)0;
                  int32_t _10266_ = _2363_d2;
                  for (int32_t _2373_i2 = _10265_; _2373_i2 < _10266_; ++_2373_i2)
                  {
                    {
                      int32_t _10267_ = (int)0;
                      int32_t _10268_ = _2366_d4;
                      for (int32_t _2374_i4 = _10267_; _2374_i4 < _10268_; ++_2374_i4)
                      {
                        {
                          _ptr__2368_nidxs[_2369_i] = ((*basevar_nativeidxs)(safe_add( _2374_i4,safe_mul( _2366_d4,safe_add( _2372_i3,safe_mul( _2365_d3,safe_add( _2373_i2,safe_mul( _2363_d2,safe_add( _2371_i1,safe_mul( _2362_d1,_2370_i0 ) ) ) ) ) ) ) )));
                          _2369_i += (int)1;
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2356_rshape,nullptr,_2368_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2375_d0 = (int)1;
      int32_t _10269_ = (int)0;
      int32_t _10270_ = _2354_dim1;
      for (int32_t _2376_i = _10269_; _2376_i < _10270_; ++_2376_i)
      {
        {
          _2375_d0 *= ((*shape)(_2376_i));
        }
      }
      int32_t _2377_d1 = ((*shape)(_2354_dim1));
      int32_t _2378_d2 = (int)1;
      int32_t _10271_ = safe_add( _2354_dim1,(int)1 );
      int32_t _10272_ = _2355_dim2;
      for (int32_t _2379_i = _10271_; _2379_i < _10272_; ++_2379_i)
      {
        {
          _2378_d2 *= ((*shape)(_2379_i));
        }
      }
      int32_t _2380_d3 = ((*shape)(_2355_dim2));
      int32_t _2381_d4 = (int)1;
      int32_t _10273_ = safe_add( _2355_dim2,(int)1 );
      int32_t _10274_ = (int)((shape)->size(0));
      for (int32_t _2382_i = _10273_; _2382_i < _10274_; ++_2382_i)
      {
        {
          _2381_d4 *= ((*shape)(_2382_i));
        }
      }
      int32_t _2383_s4 = (int)1;
      int32_t _2384_s3 = _2381_d4;
      int32_t _2385_s2 = safe_mul( _2384_s3,_2380_d3 );
      int32_t _2386_s1 = safe_mul( _2385_s2,_2378_d2 );
      int32_t _2387_s0 = safe_mul( _2386_s1,_2377_d1 );
      int32_t _2388_nelm = (int)0;
      int32_t _10275_ = (int)0;
      int32_t _10276_ = (int)((sparsity)->size(0));
      for (int32_t _2389_i = _10275_; _2389_i < _10276_; ++_2389_i)
      {
        {
          if ((((((*sparsity)(_2389_i)) / _2386_s1) % _2377_d1) >= ((((*sparsity)(_2389_i)) / _2384_s3) % _2380_d3)))
          {
            {
              ++ _2388_nelm;
            }
          }
          {}
        }
      }
      int32_t _2390_rs3 = (int)1;
      int32_t _2391_rs2 = _2381_d4;
      int32_t _2392_rs1 = safe_mul( _2391_rs2,_2378_d2 );
      int32_t _2393_rs0 = (safe_mul( safe_mul( _2392_rs1,_2377_d1 ),safe_add( _2377_d1,(int)1 ) ) / (int)2);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2394_perm = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2388_nelm)));
      _checked_ptr_1<int64_t> _ptr__2394_perm(_2394_perm ? _2394_perm->raw() : nullptr,_2394_perm ? _2394_perm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2395_rnidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2388_nelm)));
      _checked_ptr_1<int64_t> _ptr__2395_rnidxs(_2395_rnidxs ? _2395_rnidxs->raw() : nullptr,_2395_rnidxs ? _2395_rnidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2396_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2388_nelm)));
      _checked_ptr_1<int64_t> _ptr__2396_rsp(_2396_rsp ? _2396_rsp->raw() : nullptr,_2396_rsp ? _2396_rsp->size(0) : 0);
      int32_t _2397_elmi = (int)0;
      {
        int32_t _10277_ = (int)0;
        int32_t _10278_ = (int)((sparsity)->size(0));
        for (int32_t _2398_i = _10277_; _2398_i < _10278_; ++_2398_i)
        {
          {
            if ((((((*sparsity)(_2398_i)) / _2386_s1) % _2377_d1) >= ((((*sparsity)(_2398_i)) / _2384_s3) % _2380_d3)))
            {
              {
                int64_t _2399_i0 = (((*sparsity)(_2398_i)) / _2387_s0);
                int64_t _2400_i1 = ((((*sparsity)(_2398_i)) / _2386_s1) % _2377_d1);
                int64_t _2401_i2 = ((((*sparsity)(_2398_i)) / _2385_s2) % _2378_d2);
                int64_t _2402_i3 = ((((*sparsity)(_2398_i)) / _2384_s3) % _2380_d3);
                int64_t _2403_i4 = ((((*sparsity)(_2398_i)) / _2383_s4) % _2381_d4);
                _ptr__2395_rnidxs[_2397_elmi] = ((*basevar_nativeidxs)(_2398_i));
                _ptr__2396_rsp[_2397_elmi] = safe_add( safe_add( safe_add( safe_mul( _2393_rs0,_2399_i0 ),safe_mul( _2392_rs1,safe_add( (safe_mul( _2400_i1,safe_add( _2400_i1,(int)1 ) ) / (int)2),_2402_i3 ) ) ),safe_mul( _2391_rs2,_2401_i2 ) ),safe_mul( _2390_rs3,_2403_i4 ) );
                ++ _2397_elmi;
              }
            }
            {}
          }
        }
        int32_t _10279_ = (int)0;
        int32_t _10280_ = _2388_nelm;
        for (int32_t _2404_i = _10279_; _2404_i < _10280_; ++_2404_i)
        {
          {
            _ptr__2394_perm[_2404_i] = _2404_i;
          }
        }
      }
      ::mosek::fusion::p_CommonTools::argQsort(_2394_perm,_2395_rnidxs,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2388_nelm));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2405_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2388_nelm)));
      _checked_ptr_1<int64_t> _ptr__2405_nidxs(_2405_nidxs ? _2405_nidxs->raw() : nullptr,_2405_nidxs ? _2405_nidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2406_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2388_nelm)));
      _checked_ptr_1<int64_t> _ptr__2406_sp(_2406_sp ? _2406_sp->raw() : nullptr,_2406_sp ? _2406_sp->size(0) : 0);
      int32_t _10281_ = (int)0;
      int32_t _10282_ = _2388_nelm;
      for (int32_t _2407_i = _10281_; _2407_i < _10282_; ++_2407_i)
      {
        {
          _ptr__2405_nidxs[_2407_i] = _ptr__2395_rnidxs[_ptr__2394_perm[_2407_i]];
        }
      }
      int32_t _10283_ = (int)0;
      int32_t _10284_ = _2388_nelm;
      for (int32_t _2408_i = _10283_; _2408_i < _10284_; ++_2408_i)
      {
        {
          _ptr__2406_sp[_2408_i] = _ptr__2396_rsp[_ptr__2394_perm[_2408_i]];
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2356_rshape,_2406_sp,_2405_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.tril

// Begin mosek.fusion.BaseVariable.tril
// Method mosek.fusion.BaseVariable.tril @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:736:6-57
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__tril() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__tril(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::tril() { return __mosek_2fusion_2BaseVariable__tril(); }
/* override: mosek.fusion.Variable.tril*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__tril() { return __mosek_2fusion_2BaseVariable__tril(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__tril()
{
  return _pubthis->__mosek_2fusion_2BaseVariable__tril((int)0,(int)1);
}
// End mosek.fusion.BaseVariable.tril

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:734:6-114
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2409_dim0,int32_t _2410_dim1,int32_t _2411_dim2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2409_dim0,_2410_dim1,_2411_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2409_dim0,int32_t _2410_dim1,int32_t _2411_dim2) { return __mosek_2fusion_2BaseVariable__reshape(_2409_dim0,_2410_dim1,_2411_dim2); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2409_dim0,int32_t _2410_dim1,int32_t _2411_dim2) { return __mosek_2fusion_2BaseVariable__reshape(_2409_dim0,_2410_dim1,_2411_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2409_dim0,int32_t _2410_dim1,int32_t _2411_dim2)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2409_dim0),(int32_t)(_2410_dim1),(int32_t)(_2411_dim2)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:733:6-98
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2412_dim0,int32_t _2413_dim1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2412_dim0,_2413_dim1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2412_dim0,int32_t _2413_dim1) { return __mosek_2fusion_2BaseVariable__reshape(_2412_dim0,_2413_dim1); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2412_dim0,int32_t _2413_dim1) { return __mosek_2fusion_2BaseVariable__reshape(_2412_dim0,_2413_dim1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2412_dim0,int32_t _2413_dim1)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2412_dim0),(int32_t)(_2413_dim1)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:732:6-82
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2414_dim0) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2414_dim0); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2414_dim0) { return __mosek_2fusion_2BaseVariable__reshape(_2414_dim0); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2414_dim0) { return __mosek_2fusion_2BaseVariable__reshape(_2414_dim0); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2414_dim0)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2414_dim0)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:719:6-730:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2415_shape) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2415_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2415_shape) { return __mosek_2fusion_2BaseVariable__reshape(_2415_shape); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2415_shape) { return __mosek_2fusion_2BaseVariable__reshape(_2415_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2415_shape)
{
  _checked_ptr_1<int32_t> _ptr__2415_shape(_2415_shape ? _2415_shape->raw() : nullptr, _2415_shape ? _2415_shape->size(0) : 0);
  int64_t _2416_shapesize = (int)1;
  int32_t _10285_ = (int)0;
  int32_t _10286_ = (int)((_2415_shape)->size(0));
  for (int32_t _2417_i = _10285_; _2417_i < _10286_; ++_2417_i)
  {
    {
      if ((_ptr__2415_shape[_2417_i] < (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid shape"));
        }
      }
      {}
      _2416_shapesize *= _ptr__2415_shape[_2417_i];
    }
  }
  if ((_2416_shapesize != _pubthis->getSize()))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Shape size does not match variable size"));
    }
  }
  {}
  std::vector<int32_t> _10287_;
  for (int32_t _2418_i = (int)0, _10288_ = (int)((_2415_shape)->size(0)); _2418_i < _10288_ ; ++_2418_i)
  {
    _10287_.push_back(_ptr__2415_shape[_2418_i]);
  }
  auto _10289_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10287_.size()),_10287_.begin(),_10287_.end()));
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_10289_,sparsity,basevar_nativeidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.setLevel
// Method mosek.fusion.BaseVariable.setLevel @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:705:6-717:7
void mosek::fusion::BaseVariable :: setLevel(std::shared_ptr< monty::ndarray< double,1 > > _2419_v) { mosek::fusion::p_BaseVariable::_get_impl(this)->setLevel(_2419_v); }
void mosek::fusion::p_BaseVariable::setLevel(std::shared_ptr< monty::ndarray< double,1 > > _2419_v)
{
  _checked_ptr_1<double> _ptr__2419_v(_2419_v ? _2419_v->raw() : nullptr, _2419_v ? _2419_v->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      if (((int)((_2419_v)->size(0)) != (int)((basevar_nativeidxs)->size(0))))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Invalid length of solution values vector"));
        }
      }
      {}
      model->setVariableValues(true,basevar_nativeidxs,_2419_v);
    }
  }
  else
  {
    {
      if (((int)((_2419_v)->size(0)) != _pubthis->getSize()))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Invalid length of solution values vector"));
        }
      }
      {}
      std::vector<double> _10290_;
      for (int32_t _2421_i = (int)0, _10291_ = (int)((basevar_nativeidxs)->size(0)); _2421_i < _10291_ ; ++_2421_i)
      {
        _10290_.push_back(_ptr__2419_v[((*sparsity)(_2421_i))]);
      }
      auto _10292_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10290_.size()),_10290_.begin(),_10290_.end()));
      std::shared_ptr< monty::ndarray< double,1 > > _2420_v = _10292_;
      _checked_ptr_1<double> _ptr__2420_v(_2420_v ? _2420_v->raw() : nullptr,_2420_v ? _2420_v->size(0) : 0);
      model->setVariableValues(true,basevar_nativeidxs,_2420_v);
    }
  }
}
// End mosek.fusion.BaseVariable.setLevel

// Begin mosek.fusion.BaseVariable.getModel
// Method mosek.fusion.BaseVariable.getModel @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:703:6-55
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__getModel() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__getModel(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable::getModel() { return __mosek_2fusion_2BaseVariable__getModel(); }
/* override: mosek.fusion.Expression.getModel*/
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable::__mosek_2fusion_2Expression__getModel() { return __mosek_2fusion_2BaseVariable__getModel(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__getModel()
{
  return model;
}
// End mosek.fusion.BaseVariable.getModel

// Begin mosek.fusion.BaseVariable.getND
// Method mosek.fusion.BaseVariable.getND @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:702:6-56
int32_t mosek::fusion::BaseVariable :: getND() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getND(); }
int32_t mosek::fusion::p_BaseVariable::getND()
{
  return (int)((shape)->size(0));
}
// End mosek.fusion.BaseVariable.getND

// Begin mosek.fusion.BaseVariable.getDim
// Method mosek.fusion.BaseVariable.getDim @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:701:6-62
int32_t mosek::fusion::BaseVariable :: getDim(int32_t _2422_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->getDim(_2422_d); }
int32_t mosek::fusion::p_BaseVariable::getDim(int32_t _2422_d)
{
  return ((*shape)(_2422_d));
}
// End mosek.fusion.BaseVariable.getDim

// Begin mosek.fusion.BaseVariable.getShape
// Method mosek.fusion.BaseVariable.getShape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:700:6-79
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::BaseVariable :: getShape() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getShape(); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_BaseVariable::getShape()
{
  return ::mosek::fusion::Utils::Tools::arraycopy(shape);
}
// End mosek.fusion.BaseVariable.getShape

// Begin mosek.fusion.BaseVariable.getSize
// Method mosek.fusion.BaseVariable.getSize @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:699:6-107
int64_t mosek::fusion::BaseVariable :: getSize() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getSize(); }
int64_t mosek::fusion::p_BaseVariable::getSize()
{
  int64_t _2423_sz = (int64_t)1;
  int32_t _10293_ = (int)0;
  int32_t _10294_ = (int)((shape)->size(0));
  for (int32_t _2424_i = _10293_; _2424_i < _10294_; ++_2424_i)
  {
    {
      _2423_sz *= ((*shape)(_2424_i));
    }
  }
  return _2423_sz;
}
// End mosek.fusion.BaseVariable.getSize

// Begin mosek.fusion.BaseVariable.dual
// Method mosek.fusion.BaseVariable.dual @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:686:6-696:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BaseVariable :: dual() { return mosek::fusion::p_BaseVariable::_get_impl(this)->dual(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BaseVariable::dual()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2425_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2425_res(_2425_res ? _2425_res->raw() : nullptr,_2425_res ? _2425_res->size(0) : 0);
  model->getVariableValues(false,basevar_nativeidxs,_2425_res,(int)0);
  if ((NULL != sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _2426_spread_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<double> _ptr__2426_spread_res(_2426_spread_res ? _2426_spread_res->raw() : nullptr,_2426_spread_res ? _2426_spread_res->size(0) : 0);
      int32_t _10295_ = (int)0;
      int32_t _10296_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2427_i = _10295_; _2427_i < _10296_; ++_2427_i)
      {
        {
          _ptr__2426_spread_res[((*sparsity)(_2427_i))] = _ptr__2425_res[_2427_i];
        }
      }
      _2425_res = _2426_spread_res;
      _ptr__2425_res.update(_2425_res ? _2425_res->raw() : nullptr, _2425_res ? _2425_res->size(0) : 0);
    }
  }
  {}
  return _2425_res;
}
// End mosek.fusion.BaseVariable.dual

// Begin mosek.fusion.BaseVariable.level
// Method mosek.fusion.BaseVariable.level @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:674:6-684:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BaseVariable :: level() { return mosek::fusion::p_BaseVariable::_get_impl(this)->level(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BaseVariable::level()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2428_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2428_res(_2428_res ? _2428_res->raw() : nullptr,_2428_res ? _2428_res->size(0) : 0);
  model->getVariableValues(true,basevar_nativeidxs,_2428_res,(int)0);
  if ((NULL != sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _2429_spread_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<double> _ptr__2429_spread_res(_2429_spread_res ? _2429_spread_res->raw() : nullptr,_2429_spread_res ? _2429_spread_res->size(0) : 0);
      int32_t _10297_ = (int)0;
      int32_t _10298_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2430_i = _10297_; _2430_i < _10298_; ++_2430_i)
      {
        {
          _ptr__2429_spread_res[((*sparsity)(_2430_i))] = _ptr__2428_res[_2430_i];
        }
      }
      _2428_res = _2429_spread_res;
      _ptr__2428_res.update(_2428_res ? _2428_res->raw() : nullptr, _2428_res ? _2428_res->size(0) : 0);
    }
  }
  {}
  return _2428_res;
}
// End mosek.fusion.BaseVariable.level

// Begin mosek.fusion.BaseVariable.makeContinuous
// Method mosek.fusion.BaseVariable.makeContinuous @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:670:6-59
void mosek::fusion::BaseVariable :: makeContinuous() { mosek::fusion::p_BaseVariable::_get_impl(this)->makeContinuous(); }
void mosek::fusion::p_BaseVariable::makeContinuous()
{
  _pubthis->make_continuous();
}
// End mosek.fusion.BaseVariable.makeContinuous

// Begin mosek.fusion.BaseVariable.makeInteger
// Method mosek.fusion.BaseVariable.makeInteger @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:669:6-53
void mosek::fusion::BaseVariable :: makeInteger() { mosek::fusion::p_BaseVariable::_get_impl(this)->makeInteger(); }
void mosek::fusion::p_BaseVariable::makeInteger()
{
  _pubthis->make_integer();
}
// End mosek.fusion.BaseVariable.makeInteger

// Begin mosek.fusion.BaseVariable.transpose
// Method mosek.fusion.BaseVariable.transpose @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:630:6-665:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__transpose() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::transpose() { return __mosek_2fusion_2BaseVariable__transpose(); }
/* override: mosek.fusion.Variable.transpose*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__transpose() { return __mosek_2fusion_2BaseVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__transpose()
{
  if (((int)((shape)->size(0)) == (int)1))
  {
    {
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)(((*shape)((int)0)))}),sparsity,basevar_nativeidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else if(((int)((shape)->size(0)) > (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid shape for operation"));
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2431_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
      _checked_ptr_1<int64_t> _ptr__2431_idxs(_2431_idxs ? _2431_idxs->raw() : nullptr,_2431_idxs ? _2431_idxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _10299_;
      bool _10300_ = (NULL == sparsity.get());
      if (_10300_)
      {
        _10299_ = nullptr;
      }
      else
      {
        _10299_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2432_sp = _10299_;
      _checked_ptr_1<int64_t> _ptr__2432_sp(_2432_sp ? _2432_sp->raw() : nullptr,_2432_sp ? _2432_sp->size(0) : 0);
      if ((NULL == sparsity.get()))
      {
        {
          int32_t _2433_k = (int)0;
          int32_t _10301_ = (int)0;
          int32_t _10302_ = ((*shape)((int)1));
          for (int32_t _2434_j = _10301_; _2434_j < _10302_; ++_2434_j)
          {
            {
              int32_t _10303_ = (int)0;
              int32_t _10304_ = ((*shape)((int)0));
              for (int32_t _2435_i = _10303_; _2435_i < _10304_; ++_2435_i)
              {
                {
                  _ptr__2431_idxs[_2433_k] = ((*basevar_nativeidxs)(safe_add( safe_mul( _2435_i,((*shape)((int)1)) ),_2434_j )));
                  ++ _2433_k;
                }
              }
            }
          }
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2436_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( ((*shape)((int)1)),(int)1 ))));
          _checked_ptr_1<int32_t> _ptr__2436_ptr(_2436_ptr ? _2436_ptr->raw() : nullptr,_2436_ptr ? _2436_ptr->size(0) : 0);
          int32_t _2437_dim0 = ((*shape)((int)0));
          int32_t _2438_dim1 = ((*shape)((int)1));
          int32_t _10305_ = (int)0;
          int32_t _10306_ = (int)((sparsity)->size(0));
          for (int32_t _2439_i = _10305_; _2439_i < _10306_; ++_2439_i)
          {
            {
              ++ _ptr__2436_ptr[safe_add( (((*sparsity)(_2439_i)) % _2438_dim1),(int)1 )];
            }
          }
          int32_t _10307_ = (int)0;
          int32_t _10308_ = _2438_dim1;
          for (int32_t _2440_i = _10307_; _2440_i < _10308_; ++_2440_i)
          {
            {
              _ptr__2436_ptr[safe_add( _2440_i,(int)1 )] += _ptr__2436_ptr[_2440_i];
            }
          }
          int32_t _10309_ = (int)0;
          int32_t _10310_ = (int)((sparsity)->size(0));
          for (int32_t _2441_i = _10309_; _2441_i < _10310_; ++_2441_i)
          {
            {
              int64_t _2442_ii = (((*sparsity)(_2441_i)) % _2438_dim1);
              _ptr__2431_idxs[_ptr__2436_ptr[_2442_ii]] = ((*basevar_nativeidxs)(_2441_i));
              _ptr__2432_sp[_ptr__2436_ptr[_2442_ii]] = safe_add( (((*sparsity)(_2441_i)) / _2438_dim1),safe_mul( (((*sparsity)(_2441_i)) % _2438_dim1),_2437_dim0 ) );
              ++ _ptr__2436_ptr[_2442_ii];
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(((*shape)((int)1))),(int32_t)(((*shape)((int)0)))}),_2432_sp,_2431_idxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.transpose

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:619:6-622:40
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2443_i0,int32_t _2444_i1,int32_t _2445_i2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2443_i0,_2444_i1,_2445_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2443_i0,int32_t _2444_i1,int32_t _2445_i2) { return __mosek_2fusion_2BaseVariable__index(_2443_i0,_2444_i1,_2445_i2); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2443_i0,int32_t _2444_i1,int32_t _2445_i2) { return __mosek_2fusion_2BaseVariable__index(_2443_i0,_2444_i1,_2445_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2443_i0,int32_t _2444_i1,int32_t _2445_i2)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2443_i0),(int32_t)(_2444_i1),(int32_t)(_2445_i2)}));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:614:6-616:37
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2446_i0,int32_t _2447_i1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2446_i0,_2447_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2446_i0,int32_t _2447_i1) { return __mosek_2fusion_2BaseVariable__index(_2446_i0,_2447_i1); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2446_i0,int32_t _2447_i1) { return __mosek_2fusion_2BaseVariable__index(_2446_i0,_2447_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2446_i0,int32_t _2447_i1)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2446_i0),(int32_t)(_2447_i1)}));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:606:6-611:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2448_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2448_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2448_index) { return __mosek_2fusion_2BaseVariable__index(_2448_index); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2448_index) { return __mosek_2fusion_2BaseVariable__index(_2448_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2448_index)
{
  _checked_ptr_1<int32_t> _ptr__2448_index(_2448_index ? _2448_index->raw() : nullptr, _2448_index ? _2448_index->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2449_midx = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)1,(int)((_2448_index)->size(0)))));
  int32_t _10311_ = (int)0;
  int32_t _10312_ = (int)((_2448_index)->size(0));
  for (int32_t _2450_i = _10311_; _2450_i < _10312_; ++_2450_i)
  {
    {
      ((*_2449_midx)((int)0,_2450_i)) = _ptr__2448_index[_2450_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2449_midx)->__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:600:6-603:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2451_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2451_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2451_index) { return __mosek_2fusion_2BaseVariable__index(_2451_index); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2451_index) { return __mosek_2fusion_2BaseVariable__index(_2451_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2451_index)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2451_index)}))->__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:579:6-597:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2452_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2453_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2454_i2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2452_i0,_2453_i1,_2454_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2452_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2453_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2454_i2) { return __mosek_2fusion_2BaseVariable__pick(_2452_i0,_2453_i1,_2454_i2); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2452_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2453_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2454_i2) { return __mosek_2fusion_2BaseVariable__pick(_2452_i0,_2453_i1,_2454_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2452_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2453_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2454_i2)
{
  _checked_ptr_1<int32_t> _ptr__2452_i0(_2452_i0 ? _2452_i0->raw() : nullptr, _2452_i0 ? _2452_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2453_i1(_2453_i1 ? _2453_i1->raw() : nullptr, _2453_i1 ? _2453_i1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2454_i2(_2454_i2 ? _2454_i2->raw() : nullptr, _2454_i2 ? _2454_i2->size(0) : 0);
  if (((int)((shape)->size(0)) != (int)3))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  if ((((int)((_2452_i0)->size(0)) != (int)((_2453_i1)->size(0))) || ((int)((_2452_i0)->size(0)) != (int)((_2454_i2)->size(0)))))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Mismatching argument lengths"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2455_midxs = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)((_2452_i0)->size(0)),(int)3)));
  int32_t _10313_ = (int)0;
  int32_t _10314_ = (int)((_2452_i0)->size(0));
  for (int32_t _2456_i = _10313_; _2456_i < _10314_; ++_2456_i)
  {
    {
      ((*_2455_midxs)(_2456_i,(int)0)) = _ptr__2452_i0[_2456_i];
      ((*_2455_midxs)(_2456_i,(int)1)) = _ptr__2453_i1[_2456_i];
      ((*_2455_midxs)(_2456_i,(int)2)) = _ptr__2454_i2[_2456_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2455_midxs);
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:563:6-577:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2457_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2458_i1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2457_i0,_2458_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2457_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2458_i1) { return __mosek_2fusion_2BaseVariable__pick(_2457_i0,_2458_i1); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2457_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2458_i1) { return __mosek_2fusion_2BaseVariable__pick(_2457_i0,_2458_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2457_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2458_i1)
{
  _checked_ptr_1<int32_t> _ptr__2457_i0(_2457_i0 ? _2457_i0->raw() : nullptr, _2457_i0 ? _2457_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2458_i1(_2458_i1 ? _2458_i1->raw() : nullptr, _2458_i1 ? _2458_i1->size(0) : 0);
  if ((((int)((shape)->size(0)) != (int)2) || ((int)((_2457_i0)->size(0)) != (int)((_2458_i1)->size(0)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2459_midxs = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)((_2457_i0)->size(0)),(int)2)));
  int32_t _10315_ = (int)0;
  int32_t _10316_ = (int)((_2457_i0)->size(0));
  for (int32_t _2460_i = _10315_; _2460_i < _10316_; ++_2460_i)
  {
    {
      ((*_2459_midxs)(_2460_i,(int)0)) = _ptr__2457_i0[_2460_i];
      ((*_2459_midxs)(_2460_i,(int)1)) = _ptr__2458_i1[_2460_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2459_midxs);
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:502:6-561:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2461_midxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2461_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2461_midxs) { return __mosek_2fusion_2BaseVariable__pick(_2461_midxs); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2461_midxs) { return __mosek_2fusion_2BaseVariable__pick(_2461_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2461_midxs)
{
  if (((int)((shape)->size(0)) != (int)((_2461_midxs)->size(1))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  bool _2462_invalidindex = false;
  int32_t _10317_ = (int)0;
  int32_t _10318_ = (int)((_2461_midxs)->size(0));
  for (int32_t _2463_i = _10317_; _2463_i < _10318_; ++_2463_i)
  {
    {
      int32_t _10319_ = (int)0;
      int32_t _10320_ = (int)((_2461_midxs)->size(1));
      for (int32_t _2464_j = _10319_; _2464_j < _10320_; ++_2464_j)
      {
        {
          if (((((*_2461_midxs)(_2463_i,_2464_j)) < (int)0) || (((*_2461_midxs)(_2463_i,_2464_j)) >= ((*shape)(_2464_j)))))
          {
            {
              _2462_invalidindex = true;
            }
          }
          {}
        }
      }
    }
  }
  if (_2462_invalidindex)
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  int32_t _2465_nd = (int)((shape)->size(0));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2466_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2465_nd)));
  _checked_ptr_1<int64_t> _ptr__2466_strides(_2466_strides ? _2466_strides->raw() : nullptr,_2466_strides ? _2466_strides->size(0) : 0);
  _ptr__2466_strides[(_2465_nd - (int)1)] = (int)1;
  int32_t _10321_ = (int)1;
  int32_t _10322_ = _2465_nd;
  for (int32_t _2467_i = _10321_; _2467_i < _10322_; ++_2467_i)
  {
    {
      _ptr__2466_strides[((_2465_nd - _2467_i) - (int)1)] = safe_mul( _ptr__2466_strides[(_2465_nd - _2467_i)],((*shape)((_2465_nd - _2467_i))) );
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2468_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)((_2461_midxs)->size(0)))});
  _checked_ptr_1<int32_t> _ptr__2468_rshape(_2468_rshape ? _2468_rshape->raw() : nullptr,_2468_rshape ? _2468_rshape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2469_idxs;
  _checked_ptr_1<int64_t> _ptr__2469_idxs(_2469_idxs ? _2469_idxs->raw() : nullptr,_2469_idxs ? _2469_idxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2470_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__2470_sp(_2470_sp ? _2470_sp->raw() : nullptr,_2470_sp ? _2470_sp->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      _2469_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2461_midxs)->size(0)))));
      _ptr__2469_idxs.update(_2469_idxs ? _2469_idxs->raw() : nullptr, _2469_idxs ? _2469_idxs->size(0) : 0);
      {
        int32_t _10323_ = (int)0;
        int32_t _10324_ = (int)((_2461_midxs)->size(0));
        for (int32_t _2471_i = _10323_; _2471_i < _10324_; ++_2471_i)
        {
          {
            int64_t _2472_ii = (int64_t)0;
            int32_t _10325_ = (int)0;
            int32_t _10326_ = _2465_nd;
            for (int32_t _2473_j = _10325_; _2473_j < _10326_; ++_2473_j)
            {
              {
                _2472_ii += safe_mul( _ptr__2466_strides[_2473_j],((*_2461_midxs)(_2471_i,_2473_j)) );
              }
            }
            _ptr__2469_idxs[_2471_i] = ((*basevar_nativeidxs)(_2472_ii));
          }
        }
      }
    }
  }
  else
  {
    {
      int32_t _2474_nnz = (int)0;
      int32_t _10327_ = (int)0;
      int32_t _10328_ = (int)((_2461_midxs)->size(0));
      for (int32_t _2475_i = _10327_; _2475_i < _10328_; ++_2475_i)
      {
        {
          int64_t _2476_ii = (int64_t)0;
          int32_t _10329_ = (int)0;
          int32_t _10330_ = _2465_nd;
          for (int32_t _2477_j = _10329_; _2477_j < _10330_; ++_2477_j)
          {
            {
              _2476_ii += safe_mul( _ptr__2466_strides[_2477_j],((*_2461_midxs)(_2475_i,_2477_j)) );
            }
          }
          if ((::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2476_ii) >= (int)0))
          {
            {
              ++ _2474_nnz;
            }
          }
          {}
        }
      }
      _2469_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2474_nnz)));
      _ptr__2469_idxs.update(_2469_idxs ? _2469_idxs->raw() : nullptr, _2469_idxs ? _2469_idxs->size(0) : 0);
      _2470_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2474_nnz)));
      _ptr__2470_sp.update(_2470_sp ? _2470_sp->raw() : nullptr, _2470_sp ? _2470_sp->size(0) : 0);
      {
        int32_t _2478_k = (int)0;
        int32_t _10331_ = (int)0;
        int32_t _10332_ = (int)((_2461_midxs)->size(0));
        for (int32_t _2479_i = _10331_; _2479_i < _10332_; ++_2479_i)
        {
          {
            int64_t _2480_ii = (int64_t)0;
            int32_t _10333_ = (int)0;
            int32_t _10334_ = _2465_nd;
            for (int32_t _2481_j = _10333_; _2481_j < _10334_; ++_2481_j)
            {
              {
                _2480_ii += safe_mul( _ptr__2466_strides[_2481_j],((*_2461_midxs)(_2479_i,_2481_j)) );
              }
            }
            int32_t _2482_iidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2480_ii);
            if ((_2482_iidx >= (int)0))
            {
              {
                _ptr__2469_idxs[_2478_k] = ((*basevar_nativeidxs)(_2482_iidx));
                _ptr__2470_sp[_2478_k] = _2479_i;
                ++ _2478_k;
              }
            }
            {}
          }
        }
      }
      if ((_2474_nnz == (int)((_2461_midxs)->size(0))))
      {
        {
          _2470_sp = nullptr;
          _ptr__2470_sp.update(_2470_sp ? _2470_sp->raw() : nullptr, _2470_sp ? _2470_sp->size(0) : 0);
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2468_rshape,_2470_sp,_2469_idxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:460:6-500:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2483_idxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2483_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2483_idxs) { return __mosek_2fusion_2BaseVariable__pick(_2483_idxs); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2483_idxs) { return __mosek_2fusion_2BaseVariable__pick(_2483_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2483_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2483_idxs(_2483_idxs ? _2483_idxs->raw() : nullptr, _2483_idxs ? _2483_idxs->size(0) : 0);
  if (((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  bool _2484_invalidindex = false;
  int32_t _10335_ = (int)0;
  int32_t _10336_ = (int)((_2483_idxs)->size(0));
  for (int32_t _2485_i = _10335_; _2485_i < _10336_; ++_2485_i)
  {
    {
      if (((_ptr__2483_idxs[_2485_i] < (int)0) || (_ptr__2483_idxs[_2485_i] >= ((*shape)((int)0)))))
      {
        {
          _2484_invalidindex = true;
        }
      }
      {}
    }
  }
  if (_2484_invalidindex)
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2486_nidxs;
  _checked_ptr_1<int64_t> _ptr__2486_nidxs(_2486_nidxs ? _2486_nidxs->raw() : nullptr,_2486_nidxs ? _2486_nidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2487_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__2487_sp(_2487_sp ? _2487_sp->raw() : nullptr,_2487_sp ? _2487_sp->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      std::vector<int64_t> _10337_;
      for (int32_t _2488_i = (int)0, _10338_ = (int)((_2483_idxs)->size(0)); _2488_i < _10338_ ; ++_2488_i)
      {
        _10337_.push_back(((*basevar_nativeidxs)(_ptr__2483_idxs[_2488_i])));
      }
      auto _10339_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10337_.size()),_10337_.begin(),_10337_.end()));
      _2486_nidxs = _10339_;
      _ptr__2486_nidxs.update(_2486_nidxs ? _2486_nidxs->raw() : nullptr, _2486_nidxs ? _2486_nidxs->size(0) : 0);
    }
  }
  else
  {
    {
      int32_t _2489_nnz = (int)0;
      int32_t _10340_ = (int)0;
      int32_t _10341_ = (int)((_2483_idxs)->size(0));
      for (int32_t _2490_i = _10340_; _2490_i < _10341_; ++_2490_i)
      {
        {
          if ((::mosek::fusion::p_CommonTools::binarySearch(sparsity,(int64_t)(_ptr__2483_idxs[_2490_i])) >= (int)0))
          {
            {
              ++ _2489_nnz;
            }
          }
          {}
        }
      }
      _2486_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2489_nnz)));
      _ptr__2486_nidxs.update(_2486_nidxs ? _2486_nidxs->raw() : nullptr, _2486_nidxs ? _2486_nidxs->size(0) : 0);
      _2487_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2489_nnz)));
      _ptr__2487_sp.update(_2487_sp ? _2487_sp->raw() : nullptr, _2487_sp ? _2487_sp->size(0) : 0);
      int32_t _2491_k = (int)0;
      int32_t _10342_ = (int)0;
      int32_t _10343_ = (int)((_2483_idxs)->size(0));
      for (int32_t _2492_i = _10342_; _2492_i < _10343_; ++_2492_i)
      {
        {
          int32_t _2493_j = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,(int64_t)(_ptr__2483_idxs[_2492_i]));
          if ((_2493_j >= (int)0))
          {
            {
              _ptr__2487_sp[_2491_k] = _2492_i;
              _ptr__2486_nidxs[_2491_k] = ((*basevar_nativeidxs)(_2493_j));
              ++ _2491_k;
            }
          }
          {}
        }
      }
      if ((_2489_nnz == (int)((_2483_idxs)->size(0))))
      {
        {
          _2487_sp = nullptr;
          _ptr__2487_sp.update(_2487_sp ? _2487_sp->raw() : nullptr, _2487_sp ? _2487_sp->size(0) : 0);
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)((_2483_idxs)->size(0)))}),_2487_sp,_2486_nidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.antidiag
// Method mosek.fusion.BaseVariable.antidiag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:446:6-457:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__antidiag(int32_t _2494_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__antidiag(_2494_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::antidiag(int32_t _2494_index) { return __mosek_2fusion_2BaseVariable__antidiag(_2494_index); }
/* override: mosek.fusion.Variable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__antidiag(int32_t _2494_index) { return __mosek_2fusion_2BaseVariable__antidiag(_2494_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag(int32_t _2494_index)
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("antidiag() only works on two-dimensional objects"));
    }
  }
  else if(((_2494_index >= ((*shape)((int)1))) || ((-_2494_index) >= ((*shape)((int)2)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Diagonal index out of bounds"));
    }
  }
  {}
  if ((_2494_index >= (int)0))
  {
    {
      int32_t _10344_;
      bool _10345_ = (((*shape)((int)0)) < (((*shape)((int)1)) - _2494_index));
      if (_10345_)
      {
        _10344_ = ((*shape)((int)0));
      }
      else
      {
        _10344_ = (((*shape)((int)1)) - _2494_index);
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)(((((*shape)((int)1)) - (int)1) - _2494_index))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10344_);
    }
  }
  else
  {
    {
      int32_t _10346_;
      bool _10347_ = (((*shape)((int)1)) < safe_add( ((*shape)((int)0)),_2494_index ));
      if (_10347_)
      {
        _10346_ = ((*shape)((int)1));
      }
      else
      {
        _10346_ = safe_add( ((*shape)((int)0)),_2494_index );
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((-_2494_index)),(int32_t)((((*shape)((int)1)) - (int)1))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10346_);
    }
  }
}
// End mosek.fusion.BaseVariable.antidiag

// Begin mosek.fusion.BaseVariable.antidiag
// Method mosek.fusion.BaseVariable.antidiag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:438:6-444:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__antidiag() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::antidiag() { return __mosek_2fusion_2BaseVariable__antidiag(); }
/* override: mosek.fusion.Variable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__antidiag() { return __mosek_2fusion_2BaseVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag()
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("antidiag() only works on two-dimensional objects"));
    }
  }
  {}
  int32_t _10348_;
  bool _10349_ = (((*shape)((int)0)) < ((*shape)((int)1)));
  if (_10349_)
  {
    _10348_ = ((*shape)((int)0));
  }
  else
  {
    _10348_ = ((*shape)((int)1));
  }
  return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)((((*shape)((int)1)) - (int)1))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10348_);
}
// End mosek.fusion.BaseVariable.antidiag

// Begin mosek.fusion.BaseVariable.diag
// Method mosek.fusion.BaseVariable.diag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:425:6-436:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__diag(int32_t _2495_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__diag(_2495_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::diag(int32_t _2495_index) { return __mosek_2fusion_2BaseVariable__diag(_2495_index); }
/* override: mosek.fusion.Variable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__diag(int32_t _2495_index) { return __mosek_2fusion_2BaseVariable__diag(_2495_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag(int32_t _2495_index)
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("diag() only works on two-dimensional objects"));
    }
  }
  else if(((_2495_index >= ((*shape)((int)1))) || ((-_2495_index) >= ((*shape)((int)2)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Diagonal index out of bounds"));
    }
  }
  {}
  if ((_2495_index >= (int)0))
  {
    {
      int32_t _10350_;
      bool _10351_ = (((*shape)((int)0)) < (((*shape)((int)1)) - _2495_index));
      if (_10351_)
      {
        _10350_ = ((*shape)((int)0));
      }
      else
      {
        _10350_ = (((*shape)((int)1)) - _2495_index);
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)(_2495_index)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10350_);
    }
  }
  else
  {
    {
      int32_t _10352_;
      bool _10353_ = (((*shape)((int)1)) < safe_add( ((*shape)((int)0)),_2495_index ));
      if (_10353_)
      {
        _10352_ = ((*shape)((int)1));
      }
      else
      {
        _10352_ = safe_add( ((*shape)((int)0)),_2495_index );
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((-_2495_index)),(int32_t)((int)0)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10352_);
    }
  }
}
// End mosek.fusion.BaseVariable.diag

// Begin mosek.fusion.BaseVariable.diag
// Method mosek.fusion.BaseVariable.diag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:418:6-423:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__diag() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::diag() { return __mosek_2fusion_2BaseVariable__diag(); }
/* override: mosek.fusion.Variable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__diag() { return __mosek_2fusion_2BaseVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag()
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("diag() only works on two-dimensional objects"));
    }
  }
  {}
  int32_t _10354_;
  bool _10355_ = (((*shape)((int)0)) < ((*shape)((int)1)));
  if (_10355_)
  {
    _10354_ = ((*shape)((int)0));
  }
  else
  {
    _10354_ = ((*shape)((int)1));
  }
  return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)((int)0)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10354_);
}
// End mosek.fusion.BaseVariable.diag

// Begin mosek.fusion.BaseVariable.general_diag
// Method mosek.fusion.BaseVariable.general_diag @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:368:6-415:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > > _2496_firstidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _2497_step,int32_t _2498_num)
{
  _checked_ptr_1<int32_t> _ptr__2496_firstidx(_2496_firstidx ? _2496_firstidx->raw() : nullptr, _2496_firstidx ? _2496_firstidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2497_step(_2497_step ? _2497_step->raw() : nullptr, _2497_step ? _2497_step->size(0) : 0);
  int32_t _2499_nd = (int)((shape)->size(0));
  if (((_2499_nd != (int)((_2496_firstidx)->size(0))) || (_2499_nd != (int)((_2497_step)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid start or step definition"));
    }
  }
  {}
  int32_t _10356_ = (int)0;
  int32_t _10357_ = _2499_nd;
  for (int32_t _2500_i = _10356_; _2500_i < _10357_; ++_2500_i)
  {
    {
      if (((_ptr__2496_firstidx[_2500_i] < (int)0) || (_ptr__2496_firstidx[_2500_i] >= ((*shape)(_2500_i)))))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Start element is out of bounds"));
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2501_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2499_nd)));
  _checked_ptr_1<int64_t> _ptr__2501_strides(_2501_strides ? _2501_strides->raw() : nullptr,_2501_strides ? _2501_strides->size(0) : 0);
  _ptr__2501_strides[(_2499_nd - (int)1)] = (int)1;
  int32_t _10358_ = (int)1;
  int32_t _10359_ = _2499_nd;
  for (int32_t _2502_i = _10358_; _2502_i < _10359_; ++_2502_i)
  {
    {
      _ptr__2501_strides[((_2499_nd - _2502_i) - (int)1)] = safe_mul( _ptr__2501_strides[(_2499_nd - _2502_i)],((*shape)((_2499_nd - _2502_i))) );
    }
  }
  int64_t _2503_startidx = (int64_t)0;
  int32_t _10360_ = (int)0;
  int32_t _10361_ = (int)((_2497_step)->size(0));
  for (int32_t _2504_i = _10360_; _2504_i < _10361_; ++_2504_i)
  {
    {
      _2503_startidx += safe_mul( _ptr__2496_firstidx[_2504_i],_ptr__2501_strides[_2504_i] );
    }
  }
  int64_t _2505_stepstride = (int64_t)0;
  int32_t _10362_ = (int)0;
  int32_t _10363_ = (int)((_2497_step)->size(0));
  for (int32_t _2506_i = _10362_; _2506_i < _10363_; ++_2506_i)
  {
    {
      _2505_stepstride += safe_mul( _ptr__2497_step[_2506_i],_ptr__2501_strides[_2506_i] );
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2507_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2498_num)));
      _checked_ptr_1<int64_t> _ptr__2507_residxs(_2507_residxs ? _2507_residxs->raw() : nullptr,_2507_residxs ? _2507_residxs->size(0) : 0);
      int32_t _10364_ = (int)0;
      int32_t _10365_ = _2498_num;
      for (int32_t _2508_i = _10364_; _2508_i < _10365_; ++_2508_i)
      {
        {
          _ptr__2507_residxs[_2508_i] = ((*basevar_nativeidxs)(safe_add( _2503_startidx,safe_mul( _2505_stepstride,_2508_i ) )));
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2498_num)}),nullptr,_2507_residxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2509_numnz = (int)0;
      int32_t _10366_ = (int)0;
      int32_t _10367_ = _2498_num;
      for (int32_t _2510_i = _10366_; _2510_i < _10367_; ++_2510_i)
      {
        {
          int64_t _2511_residx = safe_add( _2503_startidx,safe_mul( _2505_stepstride,_2510_i ) );
          int32_t _2512_spidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2511_residx);
          if ((_2512_spidx >= (int)0))
          {
            {
              ++ _2509_numnz;
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2513_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2509_numnz)));
      _checked_ptr_1<int64_t> _ptr__2513_residxs(_2513_residxs ? _2513_residxs->raw() : nullptr,_2513_residxs ? _2513_residxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2514_ressp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2509_numnz)));
      _checked_ptr_1<int64_t> _ptr__2514_ressp(_2514_ressp ? _2514_ressp->raw() : nullptr,_2514_ressp ? _2514_ressp->size(0) : 0);
      int32_t _2515_k = (int)0;
      int32_t _10368_ = (int)0;
      int32_t _10369_ = _2498_num;
      for (int32_t _2516_i = _10368_; _2516_i < _10369_; ++_2516_i)
      {
        {
          _ptr__2513_residxs[_2516_i] = ((*basevar_nativeidxs)(safe_add( _2503_startidx,safe_mul( _2505_stepstride,_2516_i ) )));
          int64_t _2517_residx = safe_add( _2503_startidx,safe_mul( _2505_stepstride,_2516_i ) );
          int32_t _2518_spidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2517_residx);
          if ((_2518_spidx >= (int)0))
          {
            {
              _ptr__2514_ressp[_2515_k] = _2516_i;
              _ptr__2513_residxs[_2515_k] = ((*basevar_nativeidxs)(_2518_spidx));
              ++ _2515_k;
            }
          }
          {}
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2498_num)}),_2514_ressp,_2513_residxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.general_diag

// Begin mosek.fusion.BaseVariable.slice
// Method mosek.fusion.BaseVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:260:6-357:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2519_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2520_last) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__slice(_2519_first,_2520_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2519_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2520_last) { return __mosek_2fusion_2BaseVariable__slice(_2519_first,_2520_last); }
/* override: mosek.fusion.Variable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2519_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2520_last) { return __mosek_2fusion_2BaseVariable__slice(_2519_first,_2520_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2519_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2520_last)
{
  _checked_ptr_1<int32_t> _ptr__2519_first(_2519_first ? _2519_first->raw() : nullptr, _2519_first ? _2519_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2520_last(_2520_last ? _2520_last->raw() : nullptr, _2520_last ? _2520_last->size(0) : 0);
  if (((int)((_2519_first)->size(0)) != (int)((_2520_last)->size(0))))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
    }
  }
  else if(((int)((_2519_first)->size(0)) != (int)((shape)->size(0))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Variable shape and slice do not match"));
    }
  }
  else
  {
    {
      int32_t _10370_ = (int)0;
      int32_t _10371_ = (int)((shape)->size(0));
      for (int32_t _2521_i = _10370_; _2521_i < _10371_; ++_2521_i)
      {
        {
          if (((_ptr__2519_first[_2521_i] < (int)0) || (_ptr__2520_last[_2521_i] > ((*shape)(_2521_i)))))
          {
            {
              throw ::mosek::fusion::IndexError(std::string ("Slice is out of bounds"));
            }
          }
          else if((_ptr__2519_first[_2521_i] > _ptr__2520_last[_2521_i]))
          {
            {
              throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
            }
          }
          {}
        }
      }
      int32_t _2522_nd = (int)((shape)->size(0));
      std::vector<int32_t> _10372_;
      for (int32_t _2524_i = (int)0, _10373_ = _2522_nd; _2524_i < _10373_ ; ++_2524_i)
      {
        _10372_.push_back((_ptr__2520_last[_2524_i] - _ptr__2519_first[_2524_i]));
      }
      auto _10374_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10372_.size()),_10372_.begin(),_10372_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2523_sliceshape = _10374_;
      _checked_ptr_1<int32_t> _ptr__2523_sliceshape(_2523_sliceshape ? _2523_sliceshape->raw() : nullptr,_2523_sliceshape ? _2523_sliceshape->size(0) : 0);
      int32_t _2525_slicesize = (int)1;
      int32_t _10375_ = (int)0;
      int32_t _10376_ = _2522_nd;
      for (int32_t _2526_i = _10375_; _2526_i < _10376_; ++_2526_i)
      {
        {
          _2525_slicesize *= (_ptr__2520_last[_2526_i] - _ptr__2519_first[_2526_i]);
        }
      }
      if ((_2525_slicesize == (int)0))
      {
        {
          return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
        }
      }
      else if((NULL != sparsity.get()))
      {
        {
          int32_t _2527_nnz = (int)0;
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2528_key = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2522_nd)));
          _checked_ptr_1<int32_t> _ptr__2528_key(_2528_key ? _2528_key->raw() : nullptr,_2528_key ? _2528_key->size(0) : 0);
          int32_t _10377_ = (int)0;
          int32_t _10378_ = (int)((sparsity)->size(0));
          for (int32_t _2529_i = _10377_; _2529_i < _10378_; ++_2529_i)
          {
            {
              int64_t _2530_rem = ((*sparsity)(_2529_i));
              int32_t _10379_ = (int)1;
              int32_t _10380_ = _2522_nd;
              for (int32_t _2531_k = _10379_; _2531_k < _10380_; ++_2531_k)
              {
                {
                  _ptr__2528_key[(_2522_nd - _2531_k)] = (int32_t)(_2530_rem % ((*shape)((_2522_nd - _2531_k))));
                  _2530_rem /= ((*shape)((_2522_nd - _2531_k)));
                }
              }
              _ptr__2528_key[(int)0] = (int32_t)_2530_rem;
              int32_t _2532_k = (int)0;
              while ( ((_ptr__2528_key[_2532_k] >= _ptr__2519_first[_2532_k]) && (_ptr__2528_key[_2532_k] < _ptr__2520_last[_2532_k])) )
              {
                {
                  ++ _2532_k;
                }
              }
              if ((_2532_k == _2522_nd))
              {
                {
                  ++ _2527_nnz;
                }
              }
              {}
            }
          }
          int32_t _2533_idx = (int)0;
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2534_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2527_nnz)));
          _checked_ptr_1<int64_t> _ptr__2534_idxs(_2534_idxs ? _2534_idxs->raw() : nullptr,_2534_idxs ? _2534_idxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2535_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2527_nnz)));
          _checked_ptr_1<int64_t> _ptr__2535_sp(_2535_sp ? _2535_sp->raw() : nullptr,_2535_sp ? _2535_sp->size(0) : 0);
          int32_t _10381_ = (int)0;
          int32_t _10382_ = (int)((sparsity)->size(0));
          for (int32_t _2536_i = _10381_; _2536_i < _10382_; ++_2536_i)
          {
            {
              int64_t _2537_rem = ((*sparsity)(_2536_i));
              int32_t _10383_ = (int)1;
              int32_t _10384_ = _2522_nd;
              for (int32_t _2538_k = _10383_; _2538_k < _10384_; ++_2538_k)
              {
                {
                  _ptr__2528_key[(_2522_nd - _2538_k)] = (int32_t)(_2537_rem % ((*shape)((_2522_nd - _2538_k))));
                  _2537_rem /= ((*shape)((_2522_nd - _2538_k)));
                }
              }
              _ptr__2528_key[(int)0] = (int32_t)_2537_rem;
              int32_t _2539_k = (int)0;
              while ( ((_ptr__2528_key[_2539_k] >= _ptr__2519_first[_2539_k]) && (_ptr__2528_key[_2539_k] < _ptr__2520_last[_2539_k])) )
              {
                {
                  ++ _2539_k;
                }
              }
              if ((_2539_k == _2522_nd))
              {
                {
                  _ptr__2534_idxs[_2533_idx] = ((*basevar_nativeidxs)(_2536_i));
                  int32_t _2540_spidx = (int)0;
                  int32_t _10385_ = (int)0;
                  int32_t _10386_ = _2522_nd;
                  for (int32_t _2541_i = _10385_; _2541_i < _10386_; ++_2541_i)
                  {
                    {
                      _2540_spidx = safe_add( safe_mul( _2540_spidx,_ptr__2523_sliceshape[_2541_i] ),(_ptr__2528_key[_2541_i] - _ptr__2519_first[_2541_i]) );
                    }
                  }
                  _ptr__2535_sp[_2533_idx] = _2540_spidx;
                  ++ _2533_idx;
                }
              }
              {}
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2523_sliceshape,_2535_sp,_2534_idxs)).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2542_sliceidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2525_slicesize)));
          _checked_ptr_1<int64_t> _ptr__2542_sliceidxs(_2542_sliceidxs ? _2542_sliceidxs->raw() : nullptr,_2542_sliceidxs ? _2542_sliceidxs->size(0) : 0);
          {
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2543_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2522_nd)));
            _checked_ptr_1<int32_t> _ptr__2543_strides(_2543_strides ? _2543_strides->raw() : nullptr,_2543_strides ? _2543_strides->size(0) : 0);
            _ptr__2543_strides[(_2522_nd - (int)1)] = (int)1;
            int32_t _10387_ = (int)1;
            int32_t _10388_ = _2522_nd;
            for (int32_t _2544_i = _10387_; _2544_i < _10388_; ++_2544_i)
            {
              {
                _ptr__2543_strides[((_2522_nd - _2544_i) - (int)1)] = safe_mul( _ptr__2543_strides[(_2522_nd - _2544_i)],((*shape)((_2522_nd - _2544_i))) );
              }
            }
            int32_t _2545_start = (int)0;
            int32_t _10389_ = (int)0;
            int32_t _10390_ = _2522_nd;
            for (int32_t _2546_i = _10389_; _2546_i < _10390_; ++_2546_i)
            {
              {
                _2545_start += safe_mul( _ptr__2519_first[_2546_i],_ptr__2543_strides[_2546_i] );
              }
            }
            std::vector<int32_t> _10391_;
            for (int32_t _2548_i = (int)0, _10392_ = _2522_nd; _2548_i < _10392_ ; ++_2548_i)
            {
              _10391_.push_back(_2545_start);
            }
            auto _10393_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10391_.size()),_10391_.begin(),_10391_.end()));
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2547_pp = _10393_;
            _checked_ptr_1<int32_t> _ptr__2547_pp(_2547_pp ? _2547_pp->raw() : nullptr,_2547_pp ? _2547_pp->size(0) : 0);
            std::vector<int32_t> _10394_;
            for (int32_t _2550_i = (int)0, _10395_ = _2522_nd; _2550_i < _10395_ ; ++_2550_i)
            {
              _10394_.push_back(_ptr__2519_first[_2550_i]);
            }
            auto _10396_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10394_.size()),_10394_.begin(),_10394_.end()));
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2549_ii = _10396_;
            _checked_ptr_1<int32_t> _ptr__2549_ii(_2549_ii ? _2549_ii->raw() : nullptr,_2549_ii ? _2549_ii->size(0) : 0);
            int32_t _10397_ = (int)0;
            int32_t _10398_ = _2525_slicesize;
            for (int32_t _2551_i = _10397_; _2551_i < _10398_; ++_2551_i)
            {
              {
                _ptr__2542_sliceidxs[_2551_i] = ((*basevar_nativeidxs)(_ptr__2547_pp[(_2522_nd - (int)1)]));
                _ptr__2549_ii[(_2522_nd - (int)1)] += (int)1;
                _ptr__2547_pp[(_2522_nd - (int)1)] += _ptr__2543_strides[(_2522_nd - (int)1)];
                if ((_ptr__2549_ii[(_2522_nd - (int)1)] >= _ptr__2520_last[(_2522_nd - (int)1)]))
                {
                  {
                    int32_t _2552_k = (_2522_nd - (int)1);
                    while ( ((_2552_k > (int)0) && (_ptr__2549_ii[_2552_k] >= _ptr__2520_last[_2552_k])) )
                    {
                      {
                        _ptr__2549_ii[_2552_k] = _ptr__2519_first[_2552_k];
                        _ptr__2549_ii[(_2552_k - (int)1)] += (int)1;
                        _ptr__2547_pp[(_2552_k - (int)1)] += _ptr__2543_strides[(_2552_k - (int)1)];
                        -- _2552_k;
                      }
                    }
                    int32_t _10399_ = safe_add( _2552_k,(int)1 );
                    int32_t _10400_ = _2522_nd;
                    for (int32_t _2553_k2 = _10399_; _2553_k2 < _10400_; ++_2553_k2)
                    {
                      {
                        _ptr__2547_pp[_2553_k2] = _ptr__2547_pp[_2552_k];
                      }
                    }
                  }
                }
                {}
              }
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2523_sliceshape,nullptr,_2542_sliceidxs)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.BaseVariable.slice

// Begin mosek.fusion.BaseVariable.slice
// Method mosek.fusion.BaseVariable.slice @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:228:6-257:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__slice(int32_t _2554_first,int32_t _2555_last) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__slice(_2554_first,_2555_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::slice(int32_t _2554_first,int32_t _2555_last) { return __mosek_2fusion_2BaseVariable__slice(_2554_first,_2555_last); }
/* override: mosek.fusion.Variable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__slice(int32_t _2554_first,int32_t _2555_last) { return __mosek_2fusion_2BaseVariable__slice(_2554_first,_2555_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2554_first,int32_t _2555_last)
{
  if (((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Variable shape and slice do not match"));
    }
  }
  {}
  if (((_2554_first == (int)0) && (_2555_last == ((*shape)((int)0)))))
  {
    {
      return (::mosek::fusion::BaseVariable::t(_pubthis)).as<::mosek::fusion::Variable>();
    }
  }
  else if(((_2554_first < (int)0) || (_2555_last > ((*shape)((int)0)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Slice is out of bounds"));
    }
  }
  else if((_2554_first > _2555_last))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
    }
  }
  else if((_2554_first == _2555_last))
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
    }
  }
  else if((NULL != sparsity.get()))
  {
    {
      int32_t _2556_slicesize = (_2555_last - _2554_first);
      int32_t _2557_nnz = (int)0;
      int32_t _2558_i0 = ::mosek::fusion::p_CommonTools::binarySearchL(sparsity,(int64_t)(_2554_first));
      int32_t _2559_i1 = ::mosek::fusion::p_CommonTools::binarySearchL(sparsity,(int64_t)(_2555_last));
      std::vector<int64_t> _10401_;
      for (int32_t _2560_i = _2558_i0, _10402_ = _2559_i1; _2560_i < _10402_ ; ++_2560_i)
      {
        _10401_.push_back((((*sparsity)(_2560_i)) - _2554_first));
      }
      auto _10403_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10401_.size()),_10401_.begin(),_10401_.end()));
      std::vector<int64_t> _10404_;
      for (int32_t _2561_i = _2558_i0, _10405_ = _2559_i1; _2561_i < _10405_ ; ++_2561_i)
      {
        _10404_.push_back(((*basevar_nativeidxs)(_2561_i)));
      }
      auto _10406_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10404_.size()),_10404_.begin(),_10404_.end()));
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2556_slicesize)}),_10403_,_10406_)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2562_slicesize = (_2555_last - _2554_first);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2563_sliceidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2562_slicesize)));
      _checked_ptr_1<int64_t> _ptr__2563_sliceidxs(_2563_sliceidxs ? _2563_sliceidxs->raw() : nullptr,_2563_sliceidxs ? _2563_sliceidxs->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(basevar_nativeidxs,_2554_first,_2563_sliceidxs,(int)0,_2562_slicesize);
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2562_slicesize)}),nullptr,_2563_sliceidxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.slice

// Begin mosek.fusion.BaseVariable.asExpr
// Method mosek.fusion.BaseVariable.asExpr @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:226:6-74
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__asExpr() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__asExpr(); }
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable::asExpr() { return __mosek_2fusion_2BaseVariable__asExpr(); }
/* override: mosek.fusion.Variable.asExpr*/
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__asExpr() { return __mosek_2fusion_2BaseVariable__asExpr(); }
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__asExpr()
{
  return (::mosek::fusion::p_ExprFromVar::_new_ExprFromVar((::mosek::fusion::BaseVariable::t(_pubthis)).as<::mosek::fusion::Variable>())).as<::mosek::fusion::Expression>();
}
// End mosek.fusion.BaseVariable.asExpr

// Begin mosek.fusion.BaseVariable.inst
// Method mosek.fusion.BaseVariable.inst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:185:6-207:7
int32_t mosek::fusion::BaseVariable :: inst(int32_t _2564_spoffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2565_sparsity,int32_t _2566_nioffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2567_basevar_nativeidxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->inst(_2564_spoffset,_2565_sparsity,_2566_nioffset,_2567_basevar_nativeidxs); }
int32_t mosek::fusion::p_BaseVariable::inst(int32_t _2564_spoffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2565_sparsity,int32_t _2566_nioffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2567_basevar_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2565_sparsity(_2565_sparsity ? _2565_sparsity->raw() : nullptr, _2565_sparsity ? _2565_sparsity->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2567_basevar_nativeidxs(_2567_basevar_nativeidxs ? _2567_basevar_nativeidxs->raw() : nullptr, _2567_basevar_nativeidxs ? _2567_basevar_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2568_thisnidxs = mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs;
  _checked_ptr_1<int64_t> _ptr__2568_thisnidxs(_2568_thisnidxs ? _2568_thisnidxs->raw() : nullptr,_2568_thisnidxs ? _2568_thisnidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2569_thissp = mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity;
  _checked_ptr_1<int64_t> _ptr__2569_thissp(_2569_thissp ? _2569_thissp->raw() : nullptr,_2569_thissp ? _2569_thissp->size(0) : 0);
  if ((NULL == _2569_thissp.get()))
  {
    {
      int32_t _10407_ = (int)0;
      int32_t _10408_ = (int)((_2568_thisnidxs)->size(0));
      for (int32_t _2570_i = _10407_; _2570_i < _10408_; ++_2570_i)
      {
        {
          _ptr__2565_sparsity[safe_add( _2570_i,_2564_spoffset )] = _2570_i;
          _ptr__2567_basevar_nativeidxs[safe_add( _2570_i,_2566_nioffset )] = _ptr__2568_thisnidxs[_2570_i];
        }
      }
    }
  }
  else
  {
    {
      int32_t _10409_ = (int)0;
      int32_t _10410_ = (int)((mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity)->size(0));
      for (int32_t _2571_i = _10409_; _2571_i < _10410_; ++_2571_i)
      {
        {
          _ptr__2565_sparsity[safe_add( _2564_spoffset,_2571_i )] = _ptr__2569_thissp[_2571_i];
          _ptr__2567_basevar_nativeidxs[safe_add( _2571_i,_2566_nioffset )] = _ptr__2568_thisnidxs[_2571_i];
        }
      }
    }
  }
  return (int)((mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs)->size(0));
}
// End mosek.fusion.BaseVariable.inst

// Begin mosek.fusion.BaseVariable.numInst
// Method mosek.fusion.BaseVariable.numInst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:181:6-183:7
int32_t mosek::fusion::BaseVariable :: numInst() { return mosek::fusion::p_BaseVariable::_get_impl(this)->numInst(); }
int32_t mosek::fusion::p_BaseVariable::numInst()
{
  return (int)((basevar_nativeidxs)->size(0));
}
// End mosek.fusion.BaseVariable.numInst

// Begin mosek.fusion.BaseVariable.inst
// Method mosek.fusion.BaseVariable.inst @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:165:6-179:7
void mosek::fusion::BaseVariable :: inst(int32_t _2572_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2573_nindex) { mosek::fusion::p_BaseVariable::_get_impl(this)->inst(_2572_offset,_2573_nindex); }
void mosek::fusion::p_BaseVariable::inst(int32_t _2572_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2573_nindex)
{
  _checked_ptr_1<int64_t> _ptr__2573_nindex(_2573_nindex ? _2573_nindex->raw() : nullptr, _2573_nindex ? _2573_nindex->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      int32_t _10411_ = (int)0;
      int32_t _10412_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2574_i = _10411_; _2574_i < _10412_; ++_2574_i)
      {
        {
          _ptr__2573_nindex[safe_add( _2572_offset,_2574_i )] = ((*basevar_nativeidxs)(_2574_i));
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2575_shape = _pubthis->getShape();
      _checked_ptr_1<int32_t> _ptr__2575_shape(_2575_shape ? _2575_shape->raw() : nullptr,_2575_shape ? _2575_shape->size(0) : 0);
      int32_t _2576_domsize = (int)1;
      int32_t _10413_ = (int)0;
      int32_t _10414_ = (int)((_2575_shape)->size(0));
      for (int32_t _2577_i = _10413_; _2577_i < _10414_; ++_2577_i)
      {
        {
          _2576_domsize *= _ptr__2575_shape[_2577_i];
        }
      }
      int32_t _10415_ = (int)0;
      int32_t _10416_ = _2576_domsize;
      for (int32_t _2578_i = _10415_; _2578_i < _10416_; ++_2578_i)
      {
        {
          _ptr__2573_nindex[safe_add( _2572_offset,_2578_i )] = (int)0;
        }
      }
      int32_t _10417_ = (int)0;
      int32_t _10418_ = (int)((sparsity)->size(0));
      for (int32_t _2579_i = _10417_; _2579_i < _10418_; ++_2579_i)
      {
        {
          _ptr__2573_nindex[safe_add( _2572_offset,((*sparsity)(_2579_i)) )] = ((*basevar_nativeidxs)(_2579_i));
        }
      }
    }
  }
}
// End mosek.fusion.BaseVariable.inst

// Begin mosek.fusion.BaseVariable.set_values
// Method mosek.fusion.BaseVariable.set_values @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:147:6-163:7
void mosek::fusion::BaseVariable :: set_values(std::shared_ptr< monty::ndarray< double,1 > > _2580_values,bool _2581_primal) { mosek::fusion::p_BaseVariable::_get_impl(this)->set_values(_2580_values,_2581_primal); }
void mosek::fusion::p_BaseVariable::set_values(std::shared_ptr< monty::ndarray< double,1 > > _2580_values,bool _2581_primal)
{
  _checked_ptr_1<double> _ptr__2580_values(_2580_values ? _2580_values->raw() : nullptr, _2580_values ? _2580_values->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      _pubthis->__mosek_2fusion_2BaseVariable__getModel()->setVariableValues(_2581_primal,basevar_nativeidxs,_2580_values);
    }
  }
  else
  {
    {
      int32_t _2582_varsize = (int)1;
      int32_t _10419_ = (int)0;
      int32_t _10420_ = (int)((shape)->size(0));
      for (int32_t _2583_i = _10419_; _2583_i < _10420_; ++_2583_i)
      {
        {
          _2582_varsize *= ((*shape)(_2583_i));
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2584_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2582_varsize)));
      _checked_ptr_1<int64_t> _ptr__2584_idxs(_2584_idxs ? _2584_idxs->raw() : nullptr,_2584_idxs ? _2584_idxs->size(0) : 0);
      int32_t _10421_ = (int)0;
      int32_t _10422_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2585_i = _10421_; _2585_i < _10422_; ++_2585_i)
      {
        {
          _ptr__2584_idxs[((*sparsity)(_2585_i))] = ((*basevar_nativeidxs)(_2585_i));
        }
      }
      _pubthis->__mosek_2fusion_2BaseVariable__getModel()->setVariableValues(_2581_primal,_2584_idxs,_2580_values);
    }
  }
}
// End mosek.fusion.BaseVariable.set_values

// Begin mosek.fusion.BaseVariable.dual_lu
// Method mosek.fusion.BaseVariable.dual_lu @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:131:6-144:7
void mosek::fusion::p_BaseVariable::dual_lu(int32_t _2586_offset,std::shared_ptr< monty::ndarray< double,1 > > _2587_target,bool _2588_lower)
{
  _checked_ptr_1<double> _ptr__2587_target(_2587_target ? _2587_target->raw() : nullptr, _2587_target ? _2587_target->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      model->getVariableDuals(_2588_lower,basevar_nativeidxs,_2587_target,_2586_offset);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2589_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<int64_t> _ptr__2589_idxs(_2589_idxs ? _2589_idxs->raw() : nullptr,_2589_idxs ? _2589_idxs->size(0) : 0);
      int32_t _10423_ = (int)0;
      int32_t _10424_ = (int)((sparsity)->size(0));
      for (int32_t _2590_i = _10423_; _2590_i < _10424_; ++_2590_i)
      {
        {
          _ptr__2589_idxs[((*sparsity)(_2590_i))] = ((*basevar_nativeidxs)(_2590_i));
        }
      }
      model->getVariableDuals(_2588_lower,_2589_idxs,_2587_target,_2586_offset);
    }
  }
}
// End mosek.fusion.BaseVariable.dual_lu

// Begin mosek.fusion.BaseVariable.values
// Method mosek.fusion.BaseVariable.values @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:116:6-129:7
void mosek::fusion::BaseVariable :: values(int32_t _2591_offset,std::shared_ptr< monty::ndarray< double,1 > > _2592_target,bool _2593_primal) { mosek::fusion::p_BaseVariable::_get_impl(this)->values(_2591_offset,_2592_target,_2593_primal); }
void mosek::fusion::p_BaseVariable::values(int32_t _2591_offset,std::shared_ptr< monty::ndarray< double,1 > > _2592_target,bool _2593_primal)
{
  _checked_ptr_1<double> _ptr__2592_target(_2592_target ? _2592_target->raw() : nullptr, _2592_target ? _2592_target->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      model->getVariableValues(_2593_primal,basevar_nativeidxs,_2592_target,_2591_offset);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2594_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<int64_t> _ptr__2594_idxs(_2594_idxs ? _2594_idxs->raw() : nullptr,_2594_idxs ? _2594_idxs->size(0) : 0);
      int32_t _10425_ = (int)0;
      int32_t _10426_ = (int)((sparsity)->size(0));
      for (int32_t _2595_i = _10425_; _2595_i < _10426_; ++_2595_i)
      {
        {
          _ptr__2594_idxs[((*sparsity)(_2595_i))] = ((*basevar_nativeidxs)(_2595_i));
        }
      }
      model->getVariableValues(_2593_primal,_2594_idxs,_2592_target,_2591_offset);
    }
  }
}
// End mosek.fusion.BaseVariable.values

// Begin mosek.fusion.BaseVariable.make_continuous
// Method mosek.fusion.BaseVariable.make_continuous @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:114:6-89
void mosek::fusion::BaseVariable :: make_continuous() { mosek::fusion::p_BaseVariable::_get_impl(this)->make_continuous(); }
void mosek::fusion::p_BaseVariable::make_continuous()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->make_continuous(basevar_nativeidxs);
}
// End mosek.fusion.BaseVariable.make_continuous

// Begin mosek.fusion.BaseVariable.make_integer
// Method mosek.fusion.BaseVariable.make_integer @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BaseVariable.monty:113:6-83
void mosek::fusion::BaseVariable :: make_integer() { mosek::fusion::p_BaseVariable::_get_impl(this)->make_integer(); }
void mosek::fusion::p_BaseVariable::make_integer()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->make_integer(basevar_nativeidxs);
}
// End mosek.fusion.BaseVariable.make_integer

void mosek::fusion::p_BaseVariable::destroy()
{
  sparsity.reset();
  basevar_nativeidxs.reset();
  model.reset();
  shape.reset();
}
void mosek::fusion::BaseVariable::destroy() { mosek::fusion::p_BaseVariable::_get_impl(this)->destroy(); }
// } class BaseVariable
// class Variable { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Variable.mbi:187:4-423:5
// Begin mosek.fusion.Variable.toString
// End mosek.fusion.Variable.toString

// Begin mosek.fusion.Variable.numInst
// End mosek.fusion.Variable.numInst

// Begin mosek.fusion.Variable.inst
// End mosek.fusion.Variable.inst

// Begin mosek.fusion.Variable.inst
// End mosek.fusion.Variable.inst

// Begin mosek.fusion.Variable.remove
// End mosek.fusion.Variable.remove

// Begin mosek.fusion.Variable.getND
// End mosek.fusion.Variable.getND

// Begin mosek.fusion.Variable.getShape
// End mosek.fusion.Variable.getShape

// Begin mosek.fusion.Variable.getSize
// End mosek.fusion.Variable.getSize

// Begin mosek.fusion.Variable.setLevel
// End mosek.fusion.Variable.setLevel

// Begin mosek.fusion.Variable.fromTril
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::fromTril(int32_t _2603_dim) { return __mosek_2fusion_2Variable__fromTril(_2603_dim); }
// End mosek.fusion.Variable.fromTril

// Begin mosek.fusion.Variable.tril
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::tril() { return __mosek_2fusion_2Variable__tril(); }
// End mosek.fusion.Variable.tril

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2604_dim0,int32_t _2605_dim1,int32_t _2606_dim2) { return __mosek_2fusion_2Variable__reshape(_2604_dim0,_2605_dim1,_2606_dim2); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2607_dim0,int32_t _2608_dim1) { return __mosek_2fusion_2Variable__reshape(_2607_dim0,_2608_dim1); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2609_dim0) { return __mosek_2fusion_2Variable__reshape(_2609_dim0); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2610_shape) { return __mosek_2fusion_2Variable__reshape(_2610_shape); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.set_values
// End mosek.fusion.Variable.set_values

// Begin mosek.fusion.Variable.dual
// End mosek.fusion.Variable.dual

// Begin mosek.fusion.Variable.level
// End mosek.fusion.Variable.level

// Begin mosek.fusion.Variable.values
// End mosek.fusion.Variable.values

// Begin mosek.fusion.Variable.make_continuous
// End mosek.fusion.Variable.make_continuous

// Begin mosek.fusion.Variable.make_integer
// End mosek.fusion.Variable.make_integer

// Begin mosek.fusion.Variable.makeContinuous
// End mosek.fusion.Variable.makeContinuous

// Begin mosek.fusion.Variable.makeInteger
// End mosek.fusion.Variable.makeInteger

// Begin mosek.fusion.Variable.transpose
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::transpose() { return __mosek_2fusion_2Variable__transpose(); }
// End mosek.fusion.Variable.transpose

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2616_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2617_i2,std::shared_ptr< monty::ndarray< int32_t,1 > > _2618_i3) { return __mosek_2fusion_2Variable__pick(_2616_i1,_2617_i2,_2618_i3); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2619_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2620_i2) { return __mosek_2fusion_2Variable__pick(_2619_i1,_2620_i2); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2621_midxs) { return __mosek_2fusion_2Variable__pick(_2621_midxs); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2621_midxs) { return __mosek_2fusion_2Variable__pick(_2621_midxs); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2622_idxs) { return __mosek_2fusion_2Variable__pick(_2622_idxs); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2622_idxs) { return __mosek_2fusion_2Variable__pick(_2622_idxs); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.antidiag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::antidiag() { return __mosek_2fusion_2Variable__antidiag(); }
// End mosek.fusion.Variable.antidiag

// Begin mosek.fusion.Variable.antidiag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::antidiag(int32_t _2623_index) { return __mosek_2fusion_2Variable__antidiag(_2623_index); }
// End mosek.fusion.Variable.antidiag

// Begin mosek.fusion.Variable.diag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::diag() { return __mosek_2fusion_2Variable__diag(); }
// End mosek.fusion.Variable.diag

// Begin mosek.fusion.Variable.diag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::diag(int32_t _2624_index) { return __mosek_2fusion_2Variable__diag(_2624_index); }
// End mosek.fusion.Variable.diag

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2625_idx) { return __mosek_2fusion_2Variable__index(_2625_idx); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2625_idx) { return __mosek_2fusion_2Variable__index(_2625_idx); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2626_i1,int32_t _2627_i2,int32_t _2628_i3) { return __mosek_2fusion_2Variable__index(_2626_i1,_2627_i2,_2628_i3); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2629_i1,int32_t _2630_i2) { return __mosek_2fusion_2Variable__index(_2629_i1,_2630_i2); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__index(int32_t _2631_i1) { return __mosek_2fusion_2Variable__index(_2631_i1); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2631_i1) { return __mosek_2fusion_2Variable__index(_2631_i1); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.slice
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2632_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2633_lasta) { return __mosek_2fusion_2Variable__slice(_2632_firsta,_2633_lasta); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2632_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2633_lasta) { return __mosek_2fusion_2Variable__slice(_2632_firsta,_2633_lasta); }
// End mosek.fusion.Variable.slice

// Begin mosek.fusion.Variable.slice
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__slice(int32_t _2634_first,int32_t _2635_last) { return __mosek_2fusion_2Variable__slice(_2634_first,_2635_last); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::slice(int32_t _2634_first,int32_t _2635_last) { return __mosek_2fusion_2Variable__slice(_2634_first,_2635_last); }
// End mosek.fusion.Variable.slice

// Begin mosek.fusion.Variable.asExpr
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::asExpr() { return __mosek_2fusion_2Variable__asExpr(); }
// End mosek.fusion.Variable.asExpr

// } class Variable
// class Var { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:144:4-833:5
mosek::fusion::Var::Var(mosek::fusion::p_Var *_impl) : _impl(_impl) { /*std::cout << "Var() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Var::~Var() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Var::p_Var
  (::mosek::fusion::Var * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.Var.empty
// Method mosek.fusion.Var.empty @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:827:6-832:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::empty
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _2636_shape) { return mosek::fusion::p_Var::empty(_2636_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::empty(std::shared_ptr< monty::ndarray< int32_t,1 > > _2636_shape)
{
  _checked_ptr_1<int32_t> _ptr__2636_shape(_2636_shape ? _2636_shape->raw() : nullptr, _2636_shape ? _2636_shape->size(0) : 0);
  int32_t _10427_ = (int)0;
  int32_t _10428_ = (int)((_2636_shape)->size(0));
  for (int32_t _2637_i = _10427_; _2637_i < _10428_; ++_2637_i)
  {
    {
      if ((_ptr__2636_shape[_2637_i] < (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid shape"));
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_2636_shape)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Var.empty

// Begin mosek.fusion.Var.compress
// Method mosek.fusion.Var.compress @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:801:6-819:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::compress
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2638_v) { return mosek::fusion::p_Var::compress(_2638_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::compress(monty::rc_ptr< ::mosek::fusion::Variable > _2638_v)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2639_shape = _2638_v->getShape();
  _checked_ptr_1<int32_t> _ptr__2639_shape(_2639_shape ? _2639_shape->raw() : nullptr,_2639_shape ? _2639_shape->size(0) : 0);
  int32_t _2640_nd = (int)((_2639_shape)->size(0));
  int32_t _2641_realnd = (int)0;
  int32_t _10429_ = (int)0;
  int32_t _10430_ = _2640_nd;
  for (int32_t _2642_i = _10429_; _2642_i < _10430_; ++_2642_i)
  {
    {
      if ((_ptr__2639_shape[_2642_i] != (int)1))
      {
        {
          ++ _2641_realnd;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2643_realshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2641_realnd)));
  _checked_ptr_1<int32_t> _ptr__2643_realshape(_2643_realshape ? _2643_realshape->raw() : nullptr,_2643_realshape ? _2643_realshape->size(0) : 0);
  {
    int32_t _2644_k = (int)0;
    int32_t _10431_ = (int)0;
    int32_t _10432_ = _2640_nd;
    for (int32_t _2645_i = _10431_; _2645_i < _10432_; ++_2645_i)
    {
      {
        if ((_ptr__2639_shape[_2645_i] != (int)1))
        {
          {
            _ptr__2643_realshape[_2644_k] = _ptr__2639_shape[_2645_i];
            ++ _2644_k;
          }
        }
        {}
      }
    }
  }
  return ::mosek::fusion::p_Var::reshape(_2638_v,_2643_realshape);
}
// End mosek.fusion.Var.compress

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:784:6-787:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2646_v,
    int32_t _2647_d1) { return mosek::fusion::p_Var::reshape(_2646_v,_2647_d1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2646_v,int32_t _2647_d1)
{
  return ::mosek::fusion::p_Var::reshape(_2646_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2647_d1)}));
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:770:6-773:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2648_v,
    int32_t _2649_d1,
    int32_t _2650_d2) { return mosek::fusion::p_Var::reshape(_2648_v,_2649_d1,_2650_d2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2648_v,int32_t _2649_d1,int32_t _2650_d2)
{
  return ::mosek::fusion::p_Var::reshape(_2648_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2649_d1),(int32_t)(_2650_d2)}));
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.flatten
// Method mosek.fusion.Var.flatten @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:757:6-760:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::flatten
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2651_v) { return mosek::fusion::p_Var::flatten(_2651_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::flatten(monty::rc_ptr< ::mosek::fusion::Variable > _2651_v)
{
  return ::mosek::fusion::p_Var::reshape(_2651_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int32_t)::mosek::fusion::p_Set::size(_2651_v->getShape()))}));
}
// End mosek.fusion.Var.flatten

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:726:6-728:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2652_v,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2653_shape) { return mosek::fusion::p_Var::reshape(_2652_v,_2653_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2652_v,std::shared_ptr< monty::ndarray< int32_t,1 > > _2653_shape)
{
  _checked_ptr_1<int32_t> _ptr__2653_shape(_2653_shape ? _2653_shape->raw() : nullptr, _2653_shape ? _2653_shape->size(0) : 0);
  return _2652_v->__mosek_2fusion_2Variable__reshape(_2653_shape);
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.index_permute_
// Method mosek.fusion.Var.index_permute_ @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:638:6-691:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::index_permute_(monty::rc_ptr< ::mosek::fusion::Variable > _2654_v,std::shared_ptr< monty::ndarray< int32_t,1 > > _2655_perm)
{
  _checked_ptr_1<int32_t> _ptr__2655_perm(_2655_perm ? _2655_perm->raw() : nullptr, _2655_perm ? _2655_perm->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2656_sparsity;
  _checked_ptr_1<int64_t> _ptr__2656_sparsity(_2656_sparsity ? _2656_sparsity->raw() : nullptr,_2656_sparsity ? _2656_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2657_nativeidxs;
  _checked_ptr_1<int64_t> _ptr__2657_nativeidxs(_2657_nativeidxs ? _2657_nativeidxs->raw() : nullptr,_2657_nativeidxs ? _2657_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2658_shape = _2654_v->getShape();
  _checked_ptr_1<int32_t> _ptr__2658_shape(_2658_shape ? _2658_shape->raw() : nullptr,_2658_shape ? _2658_shape->size(0) : 0);
  int32_t _2659_nd = (int)((_2658_shape)->size(0));
  int32_t _2660_shapesize = (int)1;
  int32_t _10433_ = (int)0;
  int32_t _10434_ = _2659_nd;
  for (int32_t _2661_i = _10433_; _2661_i < _10434_; ++_2661_i)
  {
    {
      _2660_shapesize *= _ptr__2658_shape[_2661_i];
    }
  }
  int32_t _2662_nnz = _2654_v->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2663_pstrides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2659_nd)));
  _checked_ptr_1<int64_t> _ptr__2663_pstrides(_2663_pstrides ? _2663_pstrides->raw() : nullptr,_2663_pstrides ? _2663_pstrides->size(0) : 0);
  _ptr__2663_pstrides[(_2659_nd - (int)1)] = (int)1;
  int32_t _10435_ = (int)1;
  int32_t _10436_ = _2659_nd;
  for (int32_t _2664_i = _10435_; _2664_i < _10436_; ++_2664_i)
  {
    {
      _ptr__2663_pstrides[((_2659_nd - _2664_i) - (int)1)] = safe_mul( _ptr__2663_pstrides[(_2659_nd - _2664_i)],_ptr__2658_shape[_ptr__2655_perm[(_2659_nd - _2664_i)]] );
    }
  }
  if ((_2662_nnz < _2660_shapesize))
  {
    {
      _2656_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2662_nnz)));
      _ptr__2656_sparsity.update(_2656_sparsity ? _2656_sparsity->raw() : nullptr, _2656_sparsity ? _2656_sparsity->size(0) : 0);
      _2657_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2662_nnz)));
      _ptr__2657_nativeidxs.update(_2657_nativeidxs ? _2657_nativeidxs->raw() : nullptr, _2657_nativeidxs ? _2657_nativeidxs->size(0) : 0);
      _2654_v->inst((int)0,_2656_sparsity,(int)0,_2657_nativeidxs);
    }
  }
  else
  {
    {
      _2656_sparsity = nullptr;
      _ptr__2656_sparsity.update(_2656_sparsity ? _2656_sparsity->raw() : nullptr, _2656_sparsity ? _2656_sparsity->size(0) : 0);
      _2657_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2662_nnz)));
      _ptr__2657_nativeidxs.update(_2657_nativeidxs ? _2657_nativeidxs->raw() : nullptr, _2657_nativeidxs ? _2657_nativeidxs->size(0) : 0);
      _2654_v->inst((int)0,_2657_nativeidxs);
    }
  }
  if ((NULL == _2656_sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2665_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2662_nnz)));
      _checked_ptr_1<int64_t> _ptr__2665_residxs(_2665_residxs ? _2665_residxs->raw() : nullptr,_2665_residxs ? _2665_residxs->size(0) : 0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _2666_ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2659_nd)));
        _checked_ptr_1<int32_t> _ptr__2666_ii(_2666_ii ? _2666_ii->raw() : nullptr,_2666_ii ? _2666_ii->size(0) : 0);
        int32_t _10437_ = (int)0;
        int32_t _10438_ = _2662_nnz;
        for (int32_t _2667_i = _10437_; _2667_i < _10438_; ++_2667_i)
        {
          {
            int32_t _2668_rem = _2667_i;
            int32_t _10439_ = (int)0;
            int32_t _10440_ = _2659_nd;
            for (int32_t _2669_j = _10439_; _2669_j < _10440_; ++_2669_j)
            {
              {
                _ptr__2666_ii[((_2659_nd - _2667_i) - (int)1)] = (_2668_rem % _ptr__2658_shape[((_2659_nd - _2667_i) - (int)1)]);
                _2668_rem /= _ptr__2658_shape[((_2659_nd - _2667_i) - (int)1)];
              }
            }
            int32_t _2670_newi = (int)0;
            int32_t _10441_ = (int)0;
            int32_t _10442_ = _2659_nd;
            for (int32_t _2671_j = _10441_; _2671_j < _10442_; ++_2671_j)
            {
              {
                _2670_newi += (int32_t)safe_mul( _ptr__2666_ii[_ptr__2655_perm[_2671_j]],_ptr__2663_pstrides[_2671_j] );
              }
            }
            _ptr__2665_residxs[_2670_newi] = _ptr__2657_nativeidxs[_2667_i];
          }
        }
      }
      _2657_nativeidxs = _2665_residxs;
      _ptr__2657_nativeidxs.update(_2657_nativeidxs ? _2657_nativeidxs->raw() : nullptr, _2657_nativeidxs ? _2657_nativeidxs->size(0) : 0);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2672_ressp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2662_nnz)));
      _checked_ptr_1<int64_t> _ptr__2672_ressp(_2672_ressp ? _2672_ressp->raw() : nullptr,_2672_ressp ? _2672_ressp->size(0) : 0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _2673_ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2659_nd)));
        _checked_ptr_1<int32_t> _ptr__2673_ii(_2673_ii ? _2673_ii->raw() : nullptr,_2673_ii ? _2673_ii->size(0) : 0);
        int32_t _10443_ = (int)0;
        int32_t _10444_ = _2662_nnz;
        for (int32_t _2674_i = _10443_; _2674_i < _10444_; ++_2674_i)
        {
          {
            int32_t _2675_rem = _2674_i;
            int32_t _10445_ = (int)0;
            int32_t _10446_ = _2659_nd;
            for (int32_t _2676_j = _10445_; _2676_j < _10446_; ++_2676_j)
            {
              {
                _ptr__2673_ii[((_2659_nd - _2674_i) - (int)1)] = (_2675_rem % _ptr__2658_shape[((_2659_nd - _2674_i) - (int)1)]);
                _2675_rem /= _ptr__2658_shape[((_2659_nd - _2674_i) - (int)1)];
              }
            }
            int32_t _2677_newi = (int)0;
            int32_t _10447_ = (int)0;
            int32_t _10448_ = _2659_nd;
            for (int32_t _2678_j = _10447_; _2678_j < _10448_; ++_2678_j)
            {
              {
                _2677_newi += (int32_t)safe_mul( _ptr__2673_ii[_ptr__2655_perm[_2678_j]],_ptr__2663_pstrides[_2678_j] );
              }
            }
            _ptr__2672_ressp[_2674_i] = _2677_newi;
          }
        }
      }
      std::vector<int64_t> _10449_;
      for (int32_t _2680_i = (int)0, _10450_ = _2662_nnz; _2680_i < _10450_ ; ++_2680_i)
      {
        _10449_.push_back(_2680_i);
      }
      auto _10451_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10449_.size()),_10449_.begin(),_10449_.end()));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2679_spperm = _10451_;
      _checked_ptr_1<int64_t> _ptr__2679_spperm(_2679_spperm ? _2679_spperm->raw() : nullptr,_2679_spperm ? _2679_spperm->size(0) : 0);
      ::mosek::fusion::p_CommonTools::argQsort(_2679_spperm,_2672_ressp,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2662_nnz));
      std::vector<int64_t> _10452_;
      for (int32_t _2681_i = (int)0, _10453_ = _2662_nnz; _2681_i < _10453_ ; ++_2681_i)
      {
        _10452_.push_back(_ptr__2656_sparsity[_ptr__2679_spperm[_2681_i]]);
      }
      auto _10454_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10452_.size()),_10452_.begin(),_10452_.end()));
      _2656_sparsity = _10454_;
      _ptr__2656_sparsity.update(_2656_sparsity ? _2656_sparsity->raw() : nullptr, _2656_sparsity ? _2656_sparsity->size(0) : 0);
      std::vector<int64_t> _10455_;
      for (int32_t _2682_i = (int)0, _10456_ = _2662_nnz; _2682_i < _10456_ ; ++_2682_i)
      {
        _10455_.push_back(_ptr__2657_nativeidxs[_ptr__2679_spperm[_2682_i]]);
      }
      auto _10457_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10455_.size()),_10455_.begin(),_10455_.end()));
      _2657_nativeidxs = _10457_;
      _ptr__2657_nativeidxs.update(_2657_nativeidxs ? _2657_nativeidxs->raw() : nullptr, _2657_nativeidxs ? _2657_nativeidxs->size(0) : 0);
    }
  }
  std::vector<int32_t> _10458_;
  for (int32_t _2683_i = (int)0, _10459_ = _2659_nd; _2683_i < _10459_ ; ++_2683_i)
  {
    _10458_.push_back(_ptr__2658_shape[_ptr__2655_perm[_2683_i]]);
  }
  auto _10460_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10458_.size()),_10458_.begin(),_10458_.end()));
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2654_v->__mosek_2fusion_2Expression__getModel(),_10460_,_2656_sparsity,_2657_nativeidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Var.index_permute_

// Begin mosek.fusion.Var.hrepeat
// Method mosek.fusion.Var.hrepeat @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:630:6-106
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hrepeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2684_v,
    int32_t _2685_n) { return mosek::fusion::p_Var::hrepeat(_2684_v,_2685_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hrepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2684_v,int32_t _2685_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2684_v,(int)1,_2685_n);
}
// End mosek.fusion.Var.hrepeat

// Begin mosek.fusion.Var.vrepeat
// Method mosek.fusion.Var.vrepeat @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:629:6-106
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vrepeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2686_v,
    int32_t _2687_n) { return mosek::fusion::p_Var::vrepeat(_2686_v,_2687_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vrepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2686_v,int32_t _2687_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2686_v,(int)0,_2687_n);
}
// End mosek.fusion.Var.vrepeat

// Begin mosek.fusion.Var.repeat
// Method mosek.fusion.Var.repeat @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:628:6-106
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::repeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2688_v,
    int32_t _2689_n) { return mosek::fusion::p_Var::repeat(_2688_v,_2689_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::repeat(monty::rc_ptr< ::mosek::fusion::Variable > _2688_v,int32_t _2689_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2688_v,(int)0,_2689_n);
}
// End mosek.fusion.Var.repeat

// Begin mosek.fusion.Var.repeat
// Method mosek.fusion.Var.repeat @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:627:6-106
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::repeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2690_v,
    int32_t _2691_n,
    int32_t _2692_dim) { return mosek::fusion::p_Var::repeat(_2690_v,_2691_n,_2692_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::repeat(monty::rc_ptr< ::mosek::fusion::Variable > _2690_v,int32_t _2691_n,int32_t _2692_dim)
{
  return ::mosek::fusion::p_Var::drepeat(_2690_v,_2692_dim,_2691_n);
}
// End mosek.fusion.Var.repeat

// Begin mosek.fusion.Var.drepeat
// Method mosek.fusion.Var.drepeat @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:472:6-625:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::drepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2693_v,int32_t _2694_dim,int32_t _2695_n)
{
  if ((_2695_n < (int)0))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Cannot repeat less than 0 times"));
    }
  }
  else if((_2695_n == (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2696_vshape = _2693_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2696_vshape(_2696_vshape ? _2696_vshape->raw() : nullptr,_2696_vshape ? _2696_vshape->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2697_shape = _2696_vshape;
      _checked_ptr_1<int32_t> _ptr__2697_shape(_2697_shape ? _2697_shape->raw() : nullptr,_2697_shape ? _2697_shape->size(0) : 0);
      if (((int)((_2696_vshape)->size(0)) == _2694_dim))
      {
        {
          _2697_shape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2694_dim,(int)1 ))));
          _ptr__2697_shape.update(_2697_shape ? _2697_shape->raw() : nullptr, _2697_shape ? _2697_shape->size(0) : 0);
          int32_t _10461_ = (int)0;
          int32_t _10462_ = _2694_dim;
          for (int32_t _2698_i = _10461_; _2698_i < _10462_; ++_2698_i)
          {
            {
              _ptr__2697_shape[_2698_i] = _ptr__2696_vshape[_2698_i];
            }
          }
        }
      }
      {}
      _ptr__2697_shape[_2694_dim] = (int)0;
      ::mosek::fusion::p_Debug::o()->__mosek_2fusion_2Debug__p(std::string ("drepeat dim="))->__mosek_2fusion_2Debug__p(_2694_dim)->__mosek_2fusion_2Debug__p(std::string (", n="))->__mosek_2fusion_2Debug__p(_2695_n)->__mosek_2fusion_2Debug__p(std::string (", shape="))->__mosek_2fusion_2Debug__p(_2693_v->getShape())->__mosek_2fusion_2Debug__p(std::string (" -> "))->__mosek_2fusion_2Debug__p(_2697_shape)->__mosek_2fusion_2Debug__lf();
      return (::mosek::fusion::p_NilVariable::_new_NilVariable(_2697_shape)).as<::mosek::fusion::Variable>();
    }
  }
  else if((_2695_n == (int)1))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2699_vshape = _2693_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2699_vshape(_2699_vshape ? _2699_vshape->raw() : nullptr,_2699_vshape ? _2699_vshape->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2700_shape = _2699_vshape;
      _checked_ptr_1<int32_t> _ptr__2700_shape(_2700_shape ? _2700_shape->raw() : nullptr,_2700_shape ? _2700_shape->size(0) : 0);
      if (((int)((_2699_vshape)->size(0)) == _2694_dim))
      {
        {
          _2700_shape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2694_dim,(int)1 ))));
          _ptr__2700_shape.update(_2700_shape ? _2700_shape->raw() : nullptr, _2700_shape ? _2700_shape->size(0) : 0);
          int32_t _10463_ = (int)0;
          int32_t _10464_ = _2694_dim;
          for (int32_t _2701_i = _10463_; _2701_i < _10464_; ++_2701_i)
          {
            {
              _ptr__2700_shape[_2701_i] = _ptr__2699_vshape[_2701_i];
            }
          }
          _ptr__2700_shape[_2694_dim] = (int)1;
        }
      }
      {}
      ::mosek::fusion::p_Debug::o()->__mosek_2fusion_2Debug__p(std::string ("drepeat dim="))->__mosek_2fusion_2Debug__p(_2694_dim)->__mosek_2fusion_2Debug__p(std::string (", n="))->__mosek_2fusion_2Debug__p(_2695_n)->__mosek_2fusion_2Debug__p(std::string (", shape="))->__mosek_2fusion_2Debug__p(_2693_v->getShape())->__mosek_2fusion_2Debug__p(std::string (" -> "))->__mosek_2fusion_2Debug__p(_2700_shape)->__mosek_2fusion_2Debug__lf();
      return _2693_v;
    }
  }
  else
  {
    {
      monty::rc_ptr< ::mosek::fusion::Model > _2702_model = _2693_v->__mosek_2fusion_2Expression__getModel();
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2703_shape = _2693_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2703_shape(_2703_shape ? _2703_shape->raw() : nullptr,_2703_shape ? _2703_shape->size(0) : 0);
      int32_t _2704_nd = (int)((_2703_shape)->size(0));
      if (((_2694_dim < (int)0) || (_2694_dim > _2704_nd)))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid stacking dimension"));
        }
      }
      {}
      int64_t _2705_domsize = ::mosek::fusion::p_Set::size(_2703_shape);
      int32_t _2706_nnz = _2693_v->numInst();
      int32_t _2707_rnnz = safe_mul( _2706_nnz,_2695_n );
      int32_t _2708_d0 = (int)1;
      int32_t _10465_ = (int)0;
      int32_t _10466_ = _2694_dim;
      for (int32_t _2709_i = _10465_; _2709_i < _10466_; ++_2709_i)
      {
        {
          _2708_d0 *= _ptr__2703_shape[_2709_i];
        }
      }
      int32_t _10467_;
      bool _10468_ = (_2694_dim < _2704_nd);
      if (_10468_)
      {
        _10467_ = _ptr__2703_shape[_2694_dim];
      }
      else
      {
        _10467_ = (int)1;
      }
      int32_t _2710_d1 = _10467_;
      int32_t _2711_nd1 = safe_mul( _2710_d1,_2695_n );
      int32_t _2712_d2 = (int)1;
      int32_t _10469_ = safe_add( _2694_dim,(int)1 );
      int32_t _10470_ = _2704_nd;
      for (int32_t _2713_i = _10469_; _2713_i < _10470_; ++_2713_i)
      {
        {
          _2712_d2 *= _ptr__2703_shape[_2713_i];
        }
      }
      int32_t _10471_;
      bool _10472_ = (_2694_dim < _2704_nd);
      if (_10472_)
      {
        _10471_ = _2704_nd;
      }
      else
      {
        _10471_ = safe_add( _2694_dim,(int)1 );
      }
      int32_t _2714_rnd = _10471_;
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2715_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2714_rnd)));
      _checked_ptr_1<int32_t> _ptr__2715_rshape(_2715_rshape ? _2715_rshape->raw() : nullptr,_2715_rshape ? _2715_rshape->size(0) : 0);
      int32_t _10473_ = (int)0;
      int32_t _10474_ = _2704_nd;
      for (int32_t _2716_i = _10473_; _2716_i < _10474_; ++_2716_i)
      {
        {
          _ptr__2715_rshape[_2716_i] = _ptr__2703_shape[_2716_i];
        }
      }
      int32_t _10475_ = _2704_nd;
      int32_t _10476_ = _2714_rnd;
      for (int32_t _2717_i = _10475_; _2717_i < _10476_; ++_2717_i)
      {
        {
          _ptr__2715_rshape[_2717_i] = (int)1;
        }
      }
      _ptr__2715_rshape[_2694_dim] = _2711_nd1;
      if ((_2706_nnz < _2705_domsize))
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2718_tnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2706_nnz)));
          _checked_ptr_1<int64_t> _ptr__2718_tnativeidxs(_2718_tnativeidxs ? _2718_tnativeidxs->raw() : nullptr,_2718_tnativeidxs ? _2718_tnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2719_tsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2706_nnz)));
          _checked_ptr_1<int64_t> _ptr__2719_tsp(_2719_tsp ? _2719_tsp->raw() : nullptr,_2719_tsp ? _2719_tsp->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2720_rnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2707_rnnz)));
          _checked_ptr_1<int64_t> _ptr__2720_rnativeidxs(_2720_rnativeidxs ? _2720_rnativeidxs->raw() : nullptr,_2720_rnativeidxs ? _2720_rnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2721_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2707_rnnz)));
          _checked_ptr_1<int64_t> _ptr__2721_rsp(_2721_rsp ? _2721_rsp->raw() : nullptr,_2721_rsp ? _2721_rsp->size(0) : 0);
          _2693_v->inst((int)0,_2719_tsp,(int)0,_2718_tnativeidxs);
          {
            int32_t _2722_ns0 = safe_mul( _2711_nd1,_2712_d2 );
            int32_t _2723_s0 = safe_mul( _2710_d1,_2712_d2 );
            int32_t _2724_s1 = _2712_d2;
            if ((_2694_dim == (int)0))
            {
              {
                int32_t _2725_k = (int)0;
                int32_t _10477_ = (int)0;
                int32_t _10478_ = _2695_n;
                for (int32_t _2726_i = _10477_; _2726_i < _10478_; ++_2726_i)
                {
                  {
                    int32_t _10479_ = (int)0;
                    int32_t _10480_ = _2706_nnz;
                    for (int32_t _2727_j = _10479_; _2727_j < _10480_; ++_2727_j)
                    {
                      {
                        _ptr__2721_rsp[_2725_k] = safe_add( _ptr__2719_tsp[_2727_j],safe_mul( _2726_i,_2705_domsize ) );
                        _ptr__2720_rnativeidxs[_2725_k] = _ptr__2718_tnativeidxs[_2727_j];
                        ++ _2725_k;
                      }
                    }
                  }
                }
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2702_model,_2715_rshape,_2721_rsp,_2720_rnativeidxs)).as<::mosek::fusion::Variable>();
              }
            }
            else if((_2694_dim >= _2704_nd))
            {
              {
                int32_t _2728_k = (int)0;
                int32_t _10481_ = (int)0;
                int32_t _10482_ = _2706_nnz;
                for (int32_t _2729_j = _10481_; _2729_j < _10482_; ++_2729_j)
                {
                  {
                    int32_t _10483_ = (int)0;
                    int32_t _10484_ = _2695_n;
                    for (int32_t _2730_i = _10483_; _2730_i < _10484_; ++_2730_i)
                    {
                      {
                        _ptr__2721_rsp[_2728_k] = safe_add( safe_mul( _ptr__2719_tsp[_2729_j],_2695_n ),_2730_i );
                        _ptr__2720_rnativeidxs[_2728_k] = _ptr__2718_tnativeidxs[_2729_j];
                        ++ _2728_k;
                      }
                    }
                  }
                }
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2702_model,_2715_rshape,_2721_rsp,_2720_rnativeidxs)).as<::mosek::fusion::Variable>();
              }
            }
            else
            {
              {
                int32_t _2731_k = (int)0;
                int32_t _10485_ = (int)0;
                int32_t _10486_ = _2706_nnz;
                for (int32_t _2732_i = _10485_; _2732_i < _10486_; ++_2732_i)
                {
                  {
                    int64_t _2733_i0 = (_ptr__2719_tsp[_2732_i] / _2723_s0);
                    int64_t _2734_i1 = ((_ptr__2719_tsp[_2732_i] / _2724_s1) % _2710_d1);
                    int64_t _2735_i2 = (_ptr__2719_tsp[_2732_i] % _2712_d2);
                    int32_t _10487_ = (int)0;
                    int32_t _10488_ = _2695_n;
                    for (int32_t _2736_j = _10487_; _2736_j < _10488_; ++_2736_j)
                    {
                      {
                        _ptr__2721_rsp[_2731_k] = safe_add( safe_add( safe_mul( _2733_i0,_2722_ns0 ),safe_mul( safe_add( _2734_i1,safe_mul( _2710_d1,_2736_j ) ),_2712_d2 ) ),_2735_i2 );
                        _ptr__2720_rnativeidxs[_2731_k] = _ptr__2718_tnativeidxs[_2732_i];
                        ++ _2731_k;
                      }
                    }
                  }
                }
                int32_t _2737_maxdim = (int)1;
                int32_t _10489_ = (int)0;
                int32_t _10490_ = _2714_rnd;
                for (int32_t _2738_i = _10489_; _2738_i < _10490_; ++_2738_i)
                {
                  {
                    if ((_2737_maxdim < _ptr__2715_rshape[_2738_i]))
                    {
                      {
                        _2737_maxdim = _ptr__2715_rshape[_2738_i];
                      }
                    }
                    {}
                  }
                }
                std::shared_ptr< monty::ndarray< int32_t,1 > > _2739_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2707_rnnz)));
                _checked_ptr_1<int32_t> _ptr__2739_curperm(_2739_curperm ? _2739_curperm->raw() : nullptr,_2739_curperm ? _2739_curperm->size(0) : 0);
                std::shared_ptr< monty::ndarray< int32_t,1 > > _2740_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2707_rnnz)));
                _checked_ptr_1<int32_t> _ptr__2740_tgtperm(_2740_tgtperm ? _2740_tgtperm->raw() : nullptr,_2740_tgtperm ? _2740_tgtperm->size(0) : 0);
                {
                  std::shared_ptr< monty::ndarray< int32_t,1 > > _2741_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2737_maxdim,(int)1 ))));
                  _checked_ptr_1<int32_t> _ptr__2741_ptr(_2741_ptr ? _2741_ptr->raw() : nullptr,_2741_ptr ? _2741_ptr->size(0) : 0);
                  std::shared_ptr< monty::ndarray< int64_t,1 > > _2742_s = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2714_rnd)));
                  _checked_ptr_1<int64_t> _ptr__2742_s(_2742_s ? _2742_s->raw() : nullptr,_2742_s ? _2742_s->size(0) : 0);
                  _ptr__2742_s[(_2714_rnd - (int)1)] = (int)1;
                  int32_t _10491_ = (int)1;
                  int32_t _10492_ = _2714_rnd;
                  for (int32_t _2743_i = _10491_; _2743_i < _10492_; ++_2743_i)
                  {
                    {
                      _ptr__2742_s[((_2714_rnd - _2743_i) - (int)1)] = safe_mul( _ptr__2742_s[(_2714_rnd - _2743_i)],_ptr__2715_rshape[(_2714_rnd - _2743_i)] );
                    }
                  }
                  int32_t _10493_ = (int)0;
                  int32_t _10494_ = _2707_rnnz;
                  for (int32_t _2744_i = _10493_; _2744_i < _10494_; ++_2744_i)
                  {
                    {
                      _ptr__2739_curperm[_2744_i] = _2744_i;
                    }
                  }
                  int32_t _10495_ = (int)0;
                  int32_t _10496_ = _2714_rnd;
                  for (int32_t _2745_ii = _10495_; _2745_ii < _10496_; ++_2745_ii)
                  {
                    {
                      int32_t _2746_i = ((_2714_rnd - _2745_ii) - (int)1);
                      int32_t _10497_ = (int)0;
                      int32_t _10498_ = safe_add( _ptr__2715_rshape[_2746_i],(int)1 );
                      for (int32_t _2747_j = _10497_; _2747_j < _10498_; ++_2747_j)
                      {
                        {
                          _ptr__2741_ptr[_2747_j] = (int)0;
                        }
                      }
                      int32_t _10499_ = (int)0;
                      int32_t _10500_ = _2707_rnnz;
                      for (int32_t _2748_j = _10499_; _2748_j < _10500_; ++_2748_j)
                      {
                        {
                          ++ _ptr__2741_ptr[safe_add( ((_ptr__2721_rsp[_2748_j] / _ptr__2742_s[_2746_i]) % _ptr__2715_rshape[_2746_i]),(int)1 )];
                        }
                      }
                      int32_t _10501_ = (int)0;
                      int32_t _10502_ = _ptr__2715_rshape[_2746_i];
                      for (int32_t _2749_j = _10501_; _2749_j < _10502_; ++_2749_j)
                      {
                        {
                          _ptr__2741_ptr[safe_add( _2749_j,(int)1 )] += _ptr__2741_ptr[_2749_j];
                        }
                      }
                      int32_t _10503_ = (int)0;
                      int32_t _10504_ = _2707_rnnz;
                      for (int32_t _2750_jj = _10503_; _2750_jj < _10504_; ++_2750_jj)
                      {
                        {
                          int32_t _2751_j = _ptr__2739_curperm[_2750_jj];
                          int64_t _2752_idx = ((_ptr__2721_rsp[_2751_j] / _ptr__2742_s[_2746_i]) % _ptr__2715_rshape[_2746_i]);
                          _ptr__2740_tgtperm[_ptr__2741_ptr[_2752_idx]] = _2751_j;
                          ++ _ptr__2741_ptr[_2752_idx];
                        }
                      }
                      std::shared_ptr< monty::ndarray< int32_t,1 > > _2753_tmp = _2739_curperm;
                      _checked_ptr_1<int32_t> _ptr__2753_tmp(_2753_tmp ? _2753_tmp->raw() : nullptr,_2753_tmp ? _2753_tmp->size(0) : 0);
                      _2739_curperm = _2740_tgtperm;
                      _ptr__2739_curperm.update(_2739_curperm ? _2739_curperm->raw() : nullptr, _2739_curperm ? _2739_curperm->size(0) : 0);
                      _2740_tgtperm = _2753_tmp;
                      _ptr__2740_tgtperm.update(_2740_tgtperm ? _2740_tgtperm->raw() : nullptr, _2740_tgtperm ? _2740_tgtperm->size(0) : 0);
                    }
                  }
                }
                std::vector<int64_t> _10505_;
                for (int32_t _2754_i = (int)0, _10506_ = _2707_rnnz; _2754_i < _10506_ ; ++_2754_i)
                {
                  _10505_.push_back(_ptr__2721_rsp[_ptr__2739_curperm[_2754_i]]);
                }
                auto _10507_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10505_.size()),_10505_.begin(),_10505_.end()));
                std::vector<int64_t> _10508_;
                for (int32_t _2755_i = (int)0, _10509_ = _2707_rnnz; _2755_i < _10509_ ; ++_2755_i)
                {
                  _10508_.push_back(_ptr__2720_rnativeidxs[_ptr__2739_curperm[_2755_i]]);
                }
                auto _10510_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10508_.size()),_10508_.begin(),_10508_.end()));
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2702_model,_2715_rshape,_10507_,_10510_)).as<::mosek::fusion::Variable>();
              }
            }
          }
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2756_tnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2706_nnz)));
          _checked_ptr_1<int64_t> _ptr__2756_tnativeidxs(_2756_tnativeidxs ? _2756_tnativeidxs->raw() : nullptr,_2756_tnativeidxs ? _2756_tnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2757_rnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_mul( _2706_nnz,_2695_n ))));
          _checked_ptr_1<int64_t> _ptr__2757_rnativeidxs(_2757_rnativeidxs ? _2757_rnativeidxs->raw() : nullptr,_2757_rnativeidxs ? _2757_rnativeidxs->size(0) : 0);
          _2693_v->inst((int)0,_2756_tnativeidxs);
          int32_t _2758_s0 = safe_mul( _2711_nd1,_2712_d2 );
          int32_t _2759_s1 = _2712_d2;
          int32_t _2760_k = (int)0;
          int32_t _10511_ = (int)0;
          int32_t _10512_ = _2708_d0;
          for (int32_t _2761_i0 = _10511_; _2761_i0 < _10512_; ++_2761_i0)
          {
            {
              int32_t _10513_ = (int)0;
              int32_t _10514_ = _2710_d1;
              for (int32_t _2762_i1 = _10513_; _2762_i1 < _10514_; ++_2762_i1)
              {
                {
                  int32_t _10515_ = (int)0;
                  int32_t _10516_ = _2712_d2;
                  for (int32_t _2763_i2 = _10515_; _2763_i2 < _10516_; ++_2763_i2)
                  {
                    {
                      int32_t _10517_ = (int)0;
                      int32_t _10518_ = _2695_n;
                      for (int32_t _2764_j = _10517_; _2764_j < _10518_; ++_2764_j)
                      {
                        {
                          _ptr__2757_rnativeidxs[safe_add( safe_add( safe_mul( _2761_i0,_2758_s0 ),safe_mul( safe_add( _2762_i1,safe_mul( _2764_j,_2710_d1 ) ),_2759_s1 ) ),_2763_i2 )] = _ptr__2756_tnativeidxs[_2760_k];
                        }
                      }
                      ++ _2760_k;
                    }
                  }
                }
              }
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2702_model,_2715_rshape,nullptr,_2757_rnativeidxs)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.Var.drepeat

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:450:6-454:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >,1 > > _2765_vlist) { return mosek::fusion::p_Var::stack(_2765_vlist); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >,1 > > _2765_vlist)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Variable >> _10519_;
  for (int32_t _2766_i = (int)0, _10520_ = (int)((_2765_vlist)->size(0)); _2766_i < _10520_ ; ++_2766_i)
  {
    _10519_.push_back(::mosek::fusion::p_Var::dstack(((*_2765_vlist)(_2766_i)),(int)1));
  }
  auto _10521_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >(monty::shape(_10519_.size()),_10519_.begin(),_10519_.end()));
  return ::mosek::fusion::p_Var::dstack(_10521_,(int)0);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:414:6-420:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2767_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2768_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2769_v3) { return mosek::fusion::p_Var::vstack(_2767_v1,_2768_v2,_2769_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(monty::rc_ptr< ::mosek::fusion::Variable > _2767_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2768_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2769_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2767_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2768_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2769_v3.get())}),(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:396:6-401:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2770_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2771_v2) { return mosek::fusion::p_Var::vstack(_2770_v1,_2771_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(monty::rc_ptr< ::mosek::fusion::Variable > _2770_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2771_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2770_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2771_v2.get())}),(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:377:6-381:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2772_v) { return mosek::fusion::p_Var::vstack(_2772_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2772_v)
{
  return ::mosek::fusion::p_Var::dstack(_2772_v,(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:357:6-363:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2773_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2774_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2775_v3) { return mosek::fusion::p_Var::hstack(_2773_v1,_2774_v2,_2775_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(monty::rc_ptr< ::mosek::fusion::Variable > _2773_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2774_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2775_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2773_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2774_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2775_v3.get())}),(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:336:6-341:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2776_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2777_v2) { return mosek::fusion::p_Var::hstack(_2776_v1,_2777_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(monty::rc_ptr< ::mosek::fusion::Variable > _2776_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2777_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2776_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2777_v2.get())}),(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:315:6-319:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2778_v) { return mosek::fusion::p_Var::hstack(_2778_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2778_v)
{
  return ::mosek::fusion::p_Var::dstack(_2778_v,(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:298:6-91
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2779_v,
    int32_t _2780_dim) { return mosek::fusion::p_Var::stack(_2779_v,_2780_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2779_v,int32_t _2780_dim)
{
  return ::mosek::fusion::p_Var::stack(_2780_dim,_2779_v);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:297:6-130
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2781_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2782_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2783_v3,
    int32_t _2784_dim) { return mosek::fusion::p_Var::stack(_2781_v1,_2782_v2,_2783_v3,_2784_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(monty::rc_ptr< ::mosek::fusion::Variable > _2781_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2782_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2783_v3,int32_t _2784_dim)
{
  return ::mosek::fusion::p_Var::stack(_2784_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2781_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2782_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2783_v3.get())}));
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:296:6-114
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2785_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2786_v2,
    int32_t _2787_dim) { return mosek::fusion::p_Var::stack(_2785_v1,_2786_v2,_2787_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(monty::rc_ptr< ::mosek::fusion::Variable > _2785_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2786_v2,int32_t _2787_dim)
{
  return ::mosek::fusion::p_Var::stack(_2787_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2785_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2786_v2.get())}));
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:290:6-293:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2788_dim,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2789_v) { return mosek::fusion::p_Var::stack(_2788_dim,_2789_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2788_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2789_v)
{
  int32_t _2790_maxnd = _2788_dim;
  int32_t _10522_ = (int)0;
  int32_t _10523_ = (int)((_2789_v)->size(0));
  for (int32_t _2791_i = _10522_; _2791_i < _10523_; ++_2791_i)
  {
    {
      if ((_2790_maxnd < ((*_2789_v)(_2791_i))->getND()))
      {
        {
          _2790_maxnd = ((*_2789_v)(_2791_i))->getND();
        }
      }
      {}
    }
  }
  return ::mosek::fusion::p_Var::dstack(_2789_v,_2788_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:289:6-131
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2792_dim,
    monty::rc_ptr< ::mosek::fusion::Variable > _2793_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2794_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2795_v3) { return mosek::fusion::p_Var::stack(_2792_dim,_2793_v1,_2794_v2,_2795_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2792_dim,monty::rc_ptr< ::mosek::fusion::Variable > _2793_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2794_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2795_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2793_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2794_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2795_v3.get())}),_2792_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:288:6-115
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2796_dim,
    monty::rc_ptr< ::mosek::fusion::Variable > _2797_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2798_v2) { return mosek::fusion::p_Var::stack(_2796_dim,_2797_v1,_2798_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2796_dim,monty::rc_ptr< ::mosek::fusion::Variable > _2797_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2798_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2797_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2798_v2.get())}),_2796_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.promote
// Method mosek.fusion.Var.promote @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:275:6-285:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::promote
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2799_v,
    int32_t _2800_nd) { return mosek::fusion::p_Var::promote(_2799_v,_2800_nd); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::promote(monty::rc_ptr< ::mosek::fusion::Variable > _2799_v,int32_t _2800_nd)
{
  if ((_2800_nd > _2799_v->getND()))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2801_shape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2800_nd)));
      _checked_ptr_1<int32_t> _ptr__2801_shape(_2801_shape ? _2801_shape->raw() : nullptr,_2801_shape ? _2801_shape->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2802_sh = _2799_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2802_sh(_2802_sh ? _2802_sh->raw() : nullptr,_2802_sh ? _2802_sh->size(0) : 0);
      int32_t _10524_ = (int)0;
      int32_t _10525_ = (int)((_2802_sh)->size(0));
      for (int32_t _2803_i = _10524_; _2803_i < _10525_; ++_2803_i)
      {
        {
          _ptr__2801_shape[_2803_i] = _ptr__2802_sh[_2803_i];
        }
      }
      int32_t _10526_ = (int)((_2802_sh)->size(0));
      int32_t _10527_ = _2800_nd;
      for (int32_t _2804_i = _10526_; _2804_i < _10527_; ++_2804_i)
      {
        {
          _ptr__2801_shape[_2804_i] = (int)1;
        }
      }
      return _2799_v->__mosek_2fusion_2Variable__reshape(_2801_shape);
    }
  }
  else
  {
    {
      return _2799_v;
    }
  }
}
// End mosek.fusion.Var.promote

// Begin mosek.fusion.Var.dstack
// Method mosek.fusion.Var.dstack @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/Var.monty:151:6-267:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2805_v,int32_t _2806_dim)
{
  if ((_2806_dim < (int)0))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid stacking dimension"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::Model > _2807_m0 = nullptr;
  {
    int32_t _10528_ = (int)0;
    int32_t _10529_ = (int)((_2805_v)->size(0));
    for (int32_t _2808_i = _10528_; _2808_i < _10529_; ++_2808_i)
    {
      {
        if ((NULL == _2807_m0.get()))
        {
          {
            _2807_m0 = ((*_2805_v)(_2808_i))->__mosek_2fusion_2Expression__getModel();
          }
        }
        else if(((NULL != ((*_2805_v)(_2808_i))->__mosek_2fusion_2Expression__getModel().get()) && (((*_2805_v)(_2808_i))->__mosek_2fusion_2Expression__getModel() != _2807_m0)))
        {
          {
            monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2809_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
            std::string _10530_;
            bool _10531_ = ((int)((mosek::fusion::p_Model::_get_impl(_2807_m0.get())->getName()).size()) > (int)0);
            if (_10531_)
            {
              _10530_ = mosek::fusion::p_Model::_get_impl(_2807_m0.get())->getName();
            }
            else
            {
              _10530_ = std::string ("?");
            }
            std::string _2810_m1name = _10530_;
            std::string _10532_;
            bool _10533_ = ((int)((mosek::fusion::p_Model::_get_impl(((*_2805_v)(_2808_i))->__mosek_2fusion_2Expression__getModel().get())->getName()).size()) > (int)0);
            if (_10533_)
            {
              _10532_ = mosek::fusion::p_Model::_get_impl(((*_2805_v)(_2808_i))->__mosek_2fusion_2Expression__getModel().get())->getName();
            }
            else
            {
              _10532_ = std::string ("?");
            }
            std::string _2811_m2name = _10532_;
            _2809_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Variables belong to different models: '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2810_m1name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("' and '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2811_m2name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"));
            throw ::mosek::fusion::ModelError(_2809_sb->toString());
          }
        }
        {}
      }
    }
  }
  if (((int)((_2805_v)->size(0)) == (int)0))
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
    }
  }
  else if(((int)((_2805_v)->size(0)) == (int)1))
  {
    {
      return ((*_2805_v)((int)0));
    }
  }
  else
  {
    {
      int32_t _2812_n = (int)((_2805_v)->size(0));
      std::vector<int32_t> _10534_;
      for (int32_t _2814_i = (int)0, _10535_ = _2812_n; _2814_i < _10535_ ; ++_2814_i)
      {
        _10534_.push_back(((*_2805_v)(_2814_i))->numInst());
      }
      auto _10536_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10534_.size()),_10534_.begin(),_10534_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2813_numnz = _10536_;
      _checked_ptr_1<int32_t> _ptr__2813_numnz(_2813_numnz ? _2813_numnz->raw() : nullptr,_2813_numnz ? _2813_numnz->size(0) : 0);
      std::vector<std::shared_ptr< monty::ndarray< int32_t,1 > >> _10537_;
      for (int32_t _2816_i = (int)0, _10538_ = _2812_n; _2816_i < _10538_ ; ++_2816_i)
      {
        _10537_.push_back(((*_2805_v)(_2816_i))->getShape());
      }
      auto _10539_ = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 >(monty::shape(_10537_.size()),_10537_.begin(),_10537_.end()));
      std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _2815_shapes = _10539_;
      std::vector<int32_t> _10540_;
      for (int32_t _2818_i = (int)0, _10541_ = _2812_n; _2818_i < _10541_ ; ++_2818_i)
      {
        _10540_.push_back((int)((((*_2815_shapes)(_2818_i)))->size(0)));
      }
      auto _10542_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10540_.size()),_10540_.begin(),_10540_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2817_nds = _10542_;
      _checked_ptr_1<int32_t> _ptr__2817_nds(_2817_nds ? _2817_nds->raw() : nullptr,_2817_nds ? _2817_nds->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2819_shape0 = ((*_2815_shapes)((int)0));
      _checked_ptr_1<int32_t> _ptr__2819_shape0(_2819_shape0 ? _2819_shape0->raw() : nullptr,_2819_shape0 ? _2819_shape0->size(0) : 0);
      int32_t _2820_nd0 = _ptr__2817_nds[(int)0];
      int32_t _2821_rnnz = (int)0;
      int32_t _10543_ = (int)0;
      int32_t _10544_ = _2812_n;
      for (int32_t _2822_i = _10543_; _2822_i < _10544_; ++_2822_i)
      {
        {
          _2821_rnnz += _ptr__2813_numnz[_2822_i];
        }
      }
      int32_t _2823_rnd = safe_add( _2806_dim,(int)1 );
      int32_t _10545_ = (int)0;
      int32_t _10546_ = _2812_n;
      for (int32_t _2824_i = _10545_; _2824_i < _10546_; ++_2824_i)
      {
        {
          if ((_2823_rnd < _ptr__2817_nds[_2824_i]))
          {
            {
              _2823_rnd = _ptr__2817_nds[_2824_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2825_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2823_rnd)));
      _checked_ptr_1<int32_t> _ptr__2825_rshape(_2825_rshape ? _2825_rshape->raw() : nullptr,_2825_rshape ? _2825_rshape->size(0) : 0);
      int32_t _10547_ = (int)0;
      int32_t _10548_ = _2820_nd0;
      for (int32_t _2826_i = _10547_; _2826_i < _10548_; ++_2826_i)
      {
        {
          _ptr__2825_rshape[_2826_i] = _ptr__2819_shape0[_2826_i];
        }
      }
      int32_t _10549_ = _2820_nd0;
      int32_t _10550_ = _2823_rnd;
      for (int32_t _2827_i = _10549_; _2827_i < _10550_; ++_2827_i)
      {
        {
          _ptr__2825_rshape[_2827_i] = (int)1;
        }
      }
      _ptr__2825_rshape[_2806_dim] = (int)0;
      int32_t _10551_ = (int)0;
      int32_t _10552_ = _2812_n;
      for (int32_t _2828_i = _10551_; _2828_i < _10552_; ++_2828_i)
      {
        {
          int32_t _10553_;
          bool _10554_ = ((int)((((*_2815_shapes)(_2828_i)))->size(0)) > _2806_dim);
          if (_10554_)
          {
            _10553_ = ((*((*_2815_shapes)(_2828_i)))(_2806_dim));
          }
          else
          {
            _10553_ = (int)1;
          }
          _ptr__2825_rshape[_2806_dim] += _10553_;
        }
      }
      int32_t _10555_ = (int)0;
      int32_t _10556_ = _2812_n;
      for (int32_t _2829_i = _10555_; _2829_i < _10556_; ++_2829_i)
      {
        {
          int32_t _10557_ = (int)0;
          int32_t _10558_ = (int)((((*_2815_shapes)(_2829_i)))->size(0));
          for (int32_t _2830_j = _10557_; _2830_j < _10558_; ++_2830_j)
          {
            {
              if (((_2830_j != _2806_dim) && (((*((*_2815_shapes)(_2829_i)))(_2830_j)) != _ptr__2825_rshape[_2830_j])))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Variable dimensions do not match"));
                }
              }
              {}
            }
          }
          int32_t _10559_ = (int)((((*_2815_shapes)(_2829_i)))->size(0));
          int32_t _10560_ = _2823_rnd;
          for (int32_t _2831_j = _10559_; _2831_j < _10560_; ++_2831_j)
          {
            {
              if (((_2831_j != _2806_dim) && ((int)1 != _ptr__2825_rshape[_2831_j])))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Variable dimensions do not match"));
                }
              }
              {}
            }
          }
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2832_rnidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2821_rnnz)));
      _checked_ptr_1<int64_t> _ptr__2832_rnidxs(_2832_rnidxs ? _2832_rnidxs->raw() : nullptr,_2832_rnidxs ? _2832_rnidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2833_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2821_rnnz)));
      _checked_ptr_1<int64_t> _ptr__2833_rsp(_2833_rsp ? _2833_rsp->raw() : nullptr,_2833_rsp ? _2833_rsp->size(0) : 0);
      std::vector<int32_t> _10561_;
      for (int32_t _2835_i = (int)0, _10562_ = (int)((_2805_v)->size(0)); _2835_i < _10562_ ; ++_2835_i)
      {
        int32_t _10563_;
        bool _10564_ = (_2806_dim < (int)((((*_2815_shapes)(_2835_i)))->size(0)));
        if (_10564_)
        {
          _10563_ = ((*((*_2815_shapes)(_2835_i)))(_2806_dim));
        }
        else
        {
          _10563_ = (int)1;
        }
        _10561_.push_back(_10563_);
      }
      auto _10565_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10561_.size()),_10561_.begin(),_10561_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2834_sdims = _10565_;
      _checked_ptr_1<int32_t> _ptr__2834_sdims(_2834_sdims ? _2834_sdims->raw() : nullptr,_2834_sdims ? _2834_sdims->size(0) : 0);
      int32_t _2836_d0 = (int)1;
      int32_t _10566_ = (int)0;
      int32_t _10567_ = _2806_dim;
      for (int32_t _2837_i = _10566_; _2837_i < _10567_; ++_2837_i)
      {
        {
          int32_t _10568_;
          bool _10569_ = (_2837_i < (int)((_2819_shape0)->size(0)));
          if (_10569_)
          {
            _10568_ = _ptr__2819_shape0[_2837_i];
          }
          else
          {
            _10568_ = (int)1;
          }
          _2836_d0 *= _10568_;
        }
      }
      int32_t _2838_d1 = (int)0;
      int32_t _10570_ = (int)0;
      int32_t _10571_ = _2812_n;
      for (int32_t _2839_i = _10570_; _2839_i < _10571_; ++_2839_i)
      {
        {
          _2838_d1 += _ptr__2834_sdims[_2839_i];
        }
      }
      int32_t _2840_d2 = (int)1;
      int32_t _10572_ = safe_add( _2806_dim,(int)1 );
      int32_t _10573_ = (int)((_2819_shape0)->size(0));
      for (int32_t _2841_i = _10572_; _2841_i < _10573_; ++_2841_i)
      {
        {
          _2840_d2 *= _ptr__2819_shape0[_2841_i];
        }
      }
      if ((_2836_d0 == (int)1))
      {
        {
          int32_t _2842_k = (int)0;
          int64_t _2843_kspofs = (int)0;
          int32_t _10574_ = (int)0;
          int32_t _10575_ = _2812_n;
          for (int32_t _2844_i = _10574_; _2844_i < _10575_; ++_2844_i)
          {
            {
              ((*_2805_v)(_2844_i))->inst(_2842_k,_2833_rsp,_2842_k,_2832_rnidxs);
              int32_t _10576_ = (int)0;
              int32_t _10577_ = _ptr__2813_numnz[_2844_i];
              for (int32_t _2845_j = _10576_; _2845_j < _10577_; ++_2845_j)
              {
                {
                  _ptr__2833_rsp[_2842_k] += _2843_kspofs;
                  ++ _2842_k;
                }
              }
              _2843_kspofs += ::mosek::fusion::p_Set::size(((*_2815_shapes)(_2844_i)));
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _10578_;
          bool _10579_ = (_2821_rnnz < safe_mul( safe_mul( _2836_d0,_2838_d1 ),_2840_d2 ));
          if (_10579_)
          {
            _10578_ = _2833_rsp;
          }
          else
          {
            _10578_ = nullptr;
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2807_m0,_2825_rshape,_10578_,_2832_rnidxs)).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          int32_t _2846_b = (int)0;
          int32_t _10580_ = (int)0;
          int32_t _10581_ = _2812_n;
          for (int32_t _2847_i = _10580_; _2847_i < _10581_; ++_2847_i)
          {
            {
              _2846_b += ((*_2805_v)(_2847_i))->inst(_2846_b,_2833_rsp,_2846_b,_2832_rnidxs);
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2848_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2821_rnnz)));
          _checked_ptr_1<int32_t> _ptr__2848_curperm(_2848_curperm ? _2848_curperm->raw() : nullptr,_2848_curperm ? _2848_curperm->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2849_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2821_rnnz)));
          _checked_ptr_1<int32_t> _ptr__2849_tgtperm(_2849_tgtperm ? _2849_tgtperm->raw() : nullptr,_2849_tgtperm ? _2849_tgtperm->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2850_offset = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2812_n)));
          _checked_ptr_1<int32_t> _ptr__2850_offset(_2850_offset ? _2850_offset->raw() : nullptr,_2850_offset ? _2850_offset->size(0) : 0);
          int32_t _10582_ = (int)0;
          int32_t _10583_ = (_2812_n - (int)1);
          for (int32_t _2851_i = _10582_; _2851_i < _10583_; ++_2851_i)
          {
            {
              _ptr__2850_offset[safe_add( _2851_i,(int)1 )] = safe_add( _ptr__2850_offset[_2851_i],_ptr__2834_sdims[_2851_i] );
            }
          }
          int32_t _10584_ = (int)0;
          int32_t _10585_ = _2821_rnnz;
          for (int32_t _2852_i = _10584_; _2852_i < _10585_; ++_2852_i)
          {
            {
              _ptr__2848_curperm[_2852_i] = _2852_i;
            }
          }
          int32_t _2853_k = (int)0;
          int32_t _10586_ = (int)0;
          int32_t _10587_ = _2812_n;
          for (int32_t _2854_i = _10586_; _2854_i < _10587_; ++_2854_i)
          {
            {
              int32_t _10588_ = (int)0;
              int32_t _10589_ = _ptr__2813_numnz[_2854_i];
              for (int32_t _2855_j = _10588_; _2855_j < _10589_; ++_2855_j)
              {
                {
                  int64_t _2856_i0 = (_ptr__2833_rsp[_2853_k] / safe_mul( _ptr__2834_sdims[_2854_i],_2840_d2 ));
                  int64_t _2857_i1 = ((_ptr__2833_rsp[_2853_k] / _2840_d2) % _ptr__2834_sdims[_2854_i]);
                  int64_t _2858_i2 = (_ptr__2833_rsp[_2853_k] % _2840_d2);
                  _ptr__2833_rsp[_2853_k] = safe_add( safe_mul( safe_add( safe_add( safe_mul( _2856_i0,_2838_d1 ),_2857_i1 ),_ptr__2850_offset[_2854_i] ),_2840_d2 ),_2858_i2 );
                  ++ _2853_k;
                }
              }
            }
          }
          int32_t _2859_maxdim = (int)1;
          int32_t _10590_ = (int)0;
          int32_t _10591_ = _2823_rnd;
          for (int32_t _2860_i = _10590_; _2860_i < _10591_; ++_2860_i)
          {
            {
              int32_t _10592_;
              bool _10593_ = (_2859_maxdim < _ptr__2825_rshape[_2860_i]);
              if (_10593_)
              {
                _10592_ = _ptr__2825_rshape[_2860_i];
              }
              else
              {
                _10592_ = _2859_maxdim;
              }
              _2859_maxdim = _10592_;
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2861_s = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2823_rnd)));
          _checked_ptr_1<int64_t> _ptr__2861_s(_2861_s ? _2861_s->raw() : nullptr,_2861_s ? _2861_s->size(0) : 0);
          _ptr__2861_s[(_2823_rnd - (int)1)] = (int)1;
          int32_t _10594_ = (int)1;
          int32_t _10595_ = _2823_rnd;
          for (int32_t _2862_i = _10594_; _2862_i < _10595_; ++_2862_i)
          {
            {
              _ptr__2861_s[((_2823_rnd - _2862_i) - (int)1)] = safe_mul( _ptr__2861_s[(_2823_rnd - _2862_i)],_ptr__2825_rshape[(_2823_rnd - _2862_i)] );
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2863_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2859_maxdim,(int)1 ))));
          _checked_ptr_1<int32_t> _ptr__2863_ptr(_2863_ptr ? _2863_ptr->raw() : nullptr,_2863_ptr ? _2863_ptr->size(0) : 0);
          int32_t _10596_ = (int)0;
          int32_t _10597_ = _2823_rnd;
          for (int32_t _2864_ii = _10596_; _2864_ii < _10597_; ++_2864_ii)
          {
            {
              int32_t _2865_i = ((_2823_rnd - _2864_ii) - (int)1);
              int32_t _10598_ = (int)0;
              int32_t _10599_ = safe_add( _ptr__2825_rshape[_2865_i],(int)1 );
              for (int32_t _2866_j = _10598_; _2866_j < _10599_; ++_2866_j)
              {
                {
                  _ptr__2863_ptr[_2866_j] = (int)0;
                }
              }
              int32_t _10600_ = (int)0;
              int32_t _10601_ = _2821_rnnz;
              for (int32_t _2867_j = _10600_; _2867_j < _10601_; ++_2867_j)
              {
                {
                  ++ _ptr__2863_ptr[safe_add( ((_ptr__2833_rsp[_2867_j] / _ptr__2861_s[_2865_i]) % _ptr__2825_rshape[_2865_i]),(int)1 )];
                }
              }
              int32_t _10602_ = (int)0;
              int32_t _10603_ = _ptr__2825_rshape[_2865_i];
              for (int32_t _2868_j = _10602_; _2868_j < _10603_; ++_2868_j)
              {
                {
                  _ptr__2863_ptr[safe_add( _2868_j,(int)1 )] += _ptr__2863_ptr[_2868_j];
                }
              }
              int32_t _10604_ = (int)0;
              int32_t _10605_ = _2821_rnnz;
              for (int32_t _2869_jj = _10604_; _2869_jj < _10605_; ++_2869_jj)
              {
                {
                  int32_t _2870_j = _ptr__2848_curperm[_2869_jj];
                  int64_t _2871_idx = ((_ptr__2833_rsp[_2870_j] / _ptr__2861_s[_2865_i]) % _ptr__2825_rshape[_2865_i]);
                  _ptr__2849_tgtperm[_ptr__2863_ptr[_2871_idx]] = _2870_j;
                  ++ _ptr__2863_ptr[_2871_idx];
                }
              }
              std::shared_ptr< monty::ndarray< int32_t,1 > > _2872_tmp = _2848_curperm;
              _checked_ptr_1<int32_t> _ptr__2872_tmp(_2872_tmp ? _2872_tmp->raw() : nullptr,_2872_tmp ? _2872_tmp->size(0) : 0);
              _2848_curperm = _2849_tgtperm;
              _ptr__2848_curperm.update(_2848_curperm ? _2848_curperm->raw() : nullptr, _2848_curperm ? _2848_curperm->size(0) : 0);
              _2849_tgtperm = _2872_tmp;
              _ptr__2849_tgtperm.update(_2849_tgtperm ? _2849_tgtperm->raw() : nullptr, _2849_tgtperm ? _2849_tgtperm->size(0) : 0);
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _10606_;
          bool _10607_ = (_2821_rnnz < safe_mul( safe_mul( _2836_d0,_2838_d1 ),_2840_d2 ));
          if (_10607_)
          {
            std::vector<int64_t> _10608_;
            for (int32_t _2873_i = (int)0, _10609_ = _2821_rnnz; _2873_i < _10609_ ; ++_2873_i)
            {
              _10608_.push_back(_ptr__2833_rsp[_ptr__2848_curperm[_2873_i]]);
            }
            auto _10610_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10608_.size()),_10608_.begin(),_10608_.end()));
            _10606_ = _10610_;
          }
          else
          {
            _10606_ = nullptr;
          }
          std::vector<int64_t> _10611_;
          for (int32_t _2874_i = (int)0, _10612_ = _2821_rnnz; _2874_i < _10612_ ; ++_2874_i)
          {
            _10611_.push_back(_ptr__2832_rnidxs[_ptr__2848_curperm[_2874_i]]);
          }
          auto _10613_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10611_.size()),_10611_.begin(),_10611_.end()));
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2807_m0,_2825_rshape,_10606_,_10613_)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.Var.dstack

void mosek::fusion::p_Var::destroy()
{
}
void mosek::fusion::Var::destroy() { mosek::fusion::p_Var::_get_impl(this)->destroy(); }
// } class Var
// class BoundInterfaceConstraint { // @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceConstraint.mbi:26:4-69:5
mosek::fusion::BoundInterfaceConstraint::BoundInterfaceConstraint(mosek::fusion::p_BoundInterfaceConstraint *_impl) : mosek::fusion::SliceConstraint(_impl) { /*std::cout << "BoundInterfaceConstraint()" << this << std::endl; */ }
mosek::fusion::BoundInterfaceConstraint::~BoundInterfaceConstraint() {  /* std::cout << "~BoundInterfaceConstraint()" << this << std::endl << std::flush; */ }
mosek::fusion::p_BoundInterfaceConstraint::p_BoundInterfaceConstraint
  (::mosek::fusion::BoundInterfaceConstraint * _pubthis) :     mosek::fusion::p_SliceConstraint(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.BoundInterfaceConstraint.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceConstraint.mbi:39:6-47:7
mosek::fusion::BoundInterfaceConstraint::t mosek::fusion::p_BoundInterfaceConstraint::_new_BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2875_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2876_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    bool _2878_islower)
{ return new mosek::fusion::BoundInterfaceConstraint(_2875_m,_2876_shape,_2877_nativeidxs,_2878_islower); }
mosek::fusion::BoundInterfaceConstraint::BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2875_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2876_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    bool _2878_islower) : 
   BoundInterfaceConstraint(new mosek::fusion::p_BoundInterfaceConstraint(this))
{ mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->_initialize(_2875_m,_2876_shape,_2877_nativeidxs,_2878_islower); }
void mosek::fusion::p_BoundInterfaceConstraint::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2875_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2876_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    bool _2878_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2876_shape(_2876_shape ? _2876_shape->raw() : nullptr, _2876_shape ? _2876_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2877_nativeidxs(_2877_nativeidxs ? _2877_nativeidxs->raw() : nullptr, _2877_nativeidxs ? _2877_nativeidxs->size(0) : 0);
    mosek::fusion::p_SliceConstraint::_initialize(_2875_m,_2876_shape,_2877_nativeidxs);
    mosek::fusion::p_BoundInterfaceConstraint::_get_impl(::mosek::fusion::BoundInterfaceConstraint::t(_pubthis))->islower = _2878_islower;
  }
}
// mosek.fusion.BoundInterfaceConstraint.ctor

// mosek.fusion.BoundInterfaceConstraint.ctor @ bld/docker-rockylinux-aarch64-8/final/default/src/fusion/BoundInterfaceConstraint.mbi:31:6-37:7
mosek::fusion::BoundInterfaceConstraint::t mosek::fusion::p_BoundInterfaceConstraint::_new_BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::SliceC