#include "monty.h"
#include "fusion_p.h"

template<typename T> struct _checked_ptr_1
{
  T * ptr;
  size_t sz;
  _checked_ptr_1<T>(T * ptr, size_t sz) : ptr(ptr), sz(sz) {}
  T & operator[](ptrdiff_t i) { assert(i >= 0 && i < sz); return ptr[i]; }
  const T & operator[](ptrdiff_t i) const { assert(i >= 0 && i < sz); return ptr[i]; }
  void update(T * ptr_, size_t sz_) { ptr = ptr_; sz = sz_; }
};

template<typename T> struct _checked_ptr_2
{
  T * ptr;
  size_t sz0, sz1;
  _checked_ptr_2<T>(T * ptr, size_t sz0, size_t sz1) : ptr(ptr), sz0(sz0),sz1(sz1) {}
  T & operator()(ptrdiff_t i0, ptrdiff_t i1) { assert(i0 >= 0 && i0 < sz0 && i1 >= 0 && i1 < sz1); return ptr[i0 * sz1 + i1]; }
  const T & operator()(ptrdiff_t i0, ptrdiff_t i1) const { assert(i0 >= 0 && i0 < sz0 && i1 >= 0 && i1 < sz1); return ptr[i0 * sz1 + i1]; }
  void update(T * ptr_, size_t sz0_, size_t sz1_) { ptr = ptr_; sz0 = sz0_; sz1 = sz1_; }
};

// See http://www.cplusplus.com/articles/DE18T05o/
//                 A      && ( (       B      && B1 && B2      ) || (      C      &&   C1 &&   C2  ) )
// A a and b a have different signs
// B, B1, B2 a and b are both non-negative, r is less than both a and b
// C, C1, C2 a and b are both negative, r is greater than both a and b

struct InternalNumericOverflow {};

template<typename T>
static inline T checked_add(T a, T b, bool & overflow) {
  int r = a+b;
  overflow = ((a ^ b) >= 0) && ( (( (a & b) >= 0) && r < a && r < b) || ( ((a & b) < 0) && a < r && b < r) );
  return r;
}

template<typename T>
static inline T safe_add_(T a, T b) {
  bool overflow;
  T r = checked_add(a,b,overflow);
  if (overflow)
    throw InternalNumericOverflow();
  else
    return r;
}

static inline int      safe_add(int32_t a, int32_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int64_t a, int64_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int32_t a, int64_t b) { return safe_add_<int64_t>(a,b); }
static inline int64_t  safe_add(int64_t a, int32_t b) { return safe_add_<int64_t>(a,b); }

template<typename T>
static inline T checked_mul(T a, T b, bool & overflow) {
  T r = a * b;
  overflow = b != 0 && (((a ^ b) >= 0 && a > std::numeric_limits<T>::max() / b) || ((a ^ b) < 0 && a < std::numeric_limits<T>::min() / b));
  return r;
}

template<typename T>
static inline T safe_mul_(T a, T b) {
  bool overflow;
  T r = checked_mul(a,b,overflow);
  if (overflow)
    throw InternalNumericOverflow();
  else
    return r;
}

static inline           int safe_mul(int a, int b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int64_t a, int64_t b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int a, int64_t b) { return safe_mul_<int64_t>(a,b); }
static inline int64_t safe_mul(int64_t a, int b) { return safe_mul_<int64_t>(a,b); }

std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::DJCDomainType val)
{
  switch (val)
  {
    case DJCDomainType::EqualTo: os << "DJCDomainType::EqualTo"; break;
    case DJCDomainType::LessThan: os << "DJCDomainType::LessThan"; break;
    case DJCDomainType::GreaterThan: os << "DJCDomainType::GreaterThan"; break;
    case DJCDomainType::IsFree: os << "DJCDomainType::IsFree"; break;
    case DJCDomainType::InRange: os << "DJCDomainType::InRange"; break;
    case DJCDomainType::InQCone: os << "DJCDomainType::InQCone"; break;
    case DJCDomainType::InRotatedQCone: os << "DJCDomainType::InRotatedQCone"; break;
    case DJCDomainType::InPExpCone: os << "DJCDomainType::InPExpCone"; break;
    case DJCDomainType::InPPowCone: os << "DJCDomainType::InPPowCone"; break;
    case DJCDomainType::InDExpCone: os << "DJCDomainType::InDExpCone"; break;
    case DJCDomainType::InDPowCone: os << "DJCDomainType::InDPowCone"; break;
    case DJCDomainType::InOneNormCone: os << "DJCDomainType::InOneNormCone"; break;
    case DJCDomainType::InInfNormCone: os << "DJCDomainType::InInfNormCone"; break;
    case DJCDomainType::InPGeoMeanCone: os << "DJCDomainType::InPGeoMeanCone"; break;
    case DJCDomainType::InDGeoMeanCone: os << "DJCDomainType::InDGeoMeanCone"; break;
    case DJCDomainType::InPSDCone: os << "DJCDomainType::InPSDCone"; break;
    default: os << "DJCDomainType.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::StatusKey val)
{
  switch (val)
  {
    case StatusKey::Unknown: os << "StatusKey::Unknown"; break;
    case StatusKey::Basic: os << "StatusKey::Basic"; break;
    case StatusKey::SuperBasic: os << "StatusKey::SuperBasic"; break;
    case StatusKey::OnBound: os << "StatusKey::OnBound"; break;
    case StatusKey::Infinity: os << "StatusKey::Infinity"; break;
    default: os << "StatusKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolutionType val)
{
  switch (val)
  {
    case SolutionType::Default: os << "SolutionType::Default"; break;
    case SolutionType::Basic: os << "SolutionType::Basic"; break;
    case SolutionType::Interior: os << "SolutionType::Interior"; break;
    case SolutionType::Integer: os << "SolutionType::Integer"; break;
    default: os << "SolutionType.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolverStatus val)
{
  switch (val)
  {
    case SolverStatus::OK: os << "SolverStatus::OK"; break;
    case SolverStatus::Error: os << "SolverStatus::Error"; break;
    case SolverStatus::LostRace: os << "SolverStatus::LostRace"; break;
    default: os << "SolverStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::ProblemStatus val)
{
  switch (val)
  {
    case ProblemStatus::Unknown: os << "ProblemStatus::Unknown"; break;
    case ProblemStatus::PrimalAndDualFeasible: os << "ProblemStatus::PrimalAndDualFeasible"; break;
    case ProblemStatus::PrimalFeasible: os << "ProblemStatus::PrimalFeasible"; break;
    case ProblemStatus::DualFeasible: os << "ProblemStatus::DualFeasible"; break;
    case ProblemStatus::PrimalInfeasible: os << "ProblemStatus::PrimalInfeasible"; break;
    case ProblemStatus::DualInfeasible: os << "ProblemStatus::DualInfeasible"; break;
    case ProblemStatus::PrimalAndDualInfeasible: os << "ProblemStatus::PrimalAndDualInfeasible"; break;
    case ProblemStatus::IllPosed: os << "ProblemStatus::IllPosed"; break;
    case ProblemStatus::PrimalInfeasibleOrUnbounded: os << "ProblemStatus::PrimalInfeasibleOrUnbounded"; break;
    default: os << "ProblemStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::AccSolutionStatus val)
{
  switch (val)
  {
    case AccSolutionStatus::Anything: os << "AccSolutionStatus::Anything"; break;
    case AccSolutionStatus::Optimal: os << "AccSolutionStatus::Optimal"; break;
    case AccSolutionStatus::Feasible: os << "AccSolutionStatus::Feasible"; break;
    case AccSolutionStatus::Certificate: os << "AccSolutionStatus::Certificate"; break;
    default: os << "AccSolutionStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::SolutionStatus val)
{
  switch (val)
  {
    case SolutionStatus::Undefined: os << "SolutionStatus::Undefined"; break;
    case SolutionStatus::Unknown: os << "SolutionStatus::Unknown"; break;
    case SolutionStatus::Optimal: os << "SolutionStatus::Optimal"; break;
    case SolutionStatus::Feasible: os << "SolutionStatus::Feasible"; break;
    case SolutionStatus::Certificate: os << "SolutionStatus::Certificate"; break;
    case SolutionStatus::IllposedCert: os << "SolutionStatus::IllposedCert"; break;
    default: os << "SolutionStatus.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::ObjectiveSense val)
{
  switch (val)
  {
    case ObjectiveSense::Undefined: os << "ObjectiveSense::Undefined"; break;
    case ObjectiveSense::Minimize: os << "ObjectiveSense::Minimize"; break;
    case ObjectiveSense::Maximize: os << "ObjectiveSense::Maximize"; break;
    default: os << "ObjectiveSense.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::QConeKey val)
{
  switch (val)
  {
    case QConeKey::InQCone: os << "QConeKey::InQCone"; break;
    case QConeKey::InRotatedQCone: os << "QConeKey::InRotatedQCone"; break;
    case QConeKey::InPExpCone: os << "QConeKey::InPExpCone"; break;
    case QConeKey::InPPowCone: os << "QConeKey::InPPowCone"; break;
    case QConeKey::InDExpCone: os << "QConeKey::InDExpCone"; break;
    case QConeKey::InDPowCone: os << "QConeKey::InDPowCone"; break;
    case QConeKey::InPGeoMeanCone: os << "QConeKey::InPGeoMeanCone"; break;
    case QConeKey::InDGeoMeanCone: os << "QConeKey::InDGeoMeanCone"; break;
    case QConeKey::Positive: os << "QConeKey::Positive"; break;
    case QConeKey::Negative: os << "QConeKey::Negative"; break;
    case QConeKey::Unbounded: os << "QConeKey::Unbounded"; break;
    case QConeKey::Zero: os << "QConeKey::Zero"; break;
    case QConeKey::InPSDCone: os << "QConeKey::InPSDCone"; break;
    case QConeKey::InSVecPSDCone: os << "QConeKey::InSVecPSDCone"; break;
    default: os << "QConeKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::PSDKey val)
{
  switch (val)
  {
    case PSDKey::IsSymPSD: os << "PSDKey::IsSymPSD"; break;
    case PSDKey::IsTrilPSD: os << "PSDKey::IsTrilPSD"; break;
    default: os << "PSDKey.<invalid>"; break;
  }
  return os;
}
std::ostream & mosek::fusion::operator<<(std::ostream & os,mosek::fusion::RelationKey val)
{
  switch (val)
  {
    case RelationKey::EqualsTo: os << "RelationKey::EqualsTo"; break;
    case RelationKey::LessThan: os << "RelationKey::LessThan"; break;
    case RelationKey::GreaterThan: os << "RelationKey::GreaterThan"; break;
    case RelationKey::IsFree: os << "RelationKey::IsFree"; break;
    case RelationKey::InRange: os << "RelationKey::InRange"; break;
    default: os << "RelationKey.<invalid>"; break;
  }
  return os;
}
// class Disjunction { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:350:4-355:5
mosek::fusion::Disjunction::Disjunction(mosek::fusion::p_Disjunction *_impl) : _impl(_impl) { /*std::cout << "Disjunction() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Disjunction::~Disjunction() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Disjunction::p_Disjunction
  (::mosek::fusion::Disjunction * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.Disjunction.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:352:6-354:7
mosek::fusion::Disjunction::t mosek::fusion::p_Disjunction::_new_Disjunction
  (int64_t _7_id)
{ return new mosek::fusion::Disjunction(_7_id); }
mosek::fusion::Disjunction::Disjunction
  (int64_t _7_id) : 
   Disjunction(new mosek::fusion::p_Disjunction(this))
{ mosek::fusion::p_Disjunction::_get_impl(this)->_initialize(_7_id); }
void mosek::fusion::p_Disjunction::_initialize  (int64_t _7_id)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Disjunction::_get_impl(::mosek::fusion::Disjunction::t(_pubthis))->id = _7_id;
  }
}
// mosek.fusion.Disjunction.ctor

void mosek::fusion::p_Disjunction::destroy()
{
}
void mosek::fusion::Disjunction::destroy() { mosek::fusion::p_Disjunction::_get_impl(this)->destroy(); }
// } class Disjunction
// class DisjunctionTerms { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:321:4-345:5
mosek::fusion::DisjunctionTerms::DisjunctionTerms(mosek::fusion::p_DisjunctionTerms *_impl) : _impl(_impl) { /*std::cout << "DisjunctionTerms() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DisjunctionTerms::~DisjunctionTerms() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DisjunctionTerms::p_DisjunctionTerms
  (::mosek::fusion::DisjunctionTerms * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:344:6-97
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term)
{ return new mosek::fusion::DisjunctionTerms(_8_terms1,_9_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_8_terms1,_9_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _8_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _9_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_8_terms1,::mosek::fusion::p_DJC::term(_9_term));
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:343:6-98
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term)
{ return new mosek::fusion::DisjunctionTerms(_10_terms1,_11_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_10_terms1,_11_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _10_terms1,
    monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _11_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_10_terms1,::mosek::fusion::p_DJC::term(_11_term));
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:340:6-342:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2)
{ return new mosek::fusion::DisjunctionTerms(_12_terms1,_13_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_12_terms1,_13_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _12_terms1,
    monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _13_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_DisjunctionTerms::_initialize(_12_terms1,mosek::fusion::p_DisjunctionTerms::_get_impl(_13_term2)->terms);
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:334:6-339:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2)
{ return new mosek::fusion::DisjunctionTerms(_14_term1,_15_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_14_term1,_15_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _14_term1,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _15_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _16_n1 = (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0));
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(safe_add( (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0)),(int)((_15_term2)->size(0)) ))));
    {
      int32_t _17_i = (int)0;
      for(;(_17_i < (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)->size(0)));++ _17_i)
      {
        {
          ((*terms)(_17_i)) = ((*mosek::fusion::p_DisjunctionTerms::_get_impl(_14_term1)->terms)(_17_i));
        }
      }
    }
    {
      int32_t _18_i = (int)0;
      for(;(_18_i < (int)((_15_term2)->size(0)));++ _18_i)
      {
        {
          ((*terms)(safe_add( _18_i,_16_n1 ))) = ((*_15_term2)(_18_i));
        }
      }
    }
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:329:6-333:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2)
{ return new mosek::fusion::DisjunctionTerms(_19_term1,_20_term2); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_19_term1,_20_term2); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _19_term1,
    monty::rc_ptr< ::mosek::fusion::Term > _20_term2)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(safe_add( (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)),(int)1 ))));
    {
      int32_t _21_i = (int)0;
      for(;(_21_i < (int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)));++ _21_i)
      {
        {
          ((*terms)(_21_i)) = ((*mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)(_21_i));
        }
      }
    }
    ((*terms)((int)((mosek::fusion::p_DisjunctionTerms::_get_impl(_19_term1)->terms)->size(0)))) = _20_term2;
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:326:6-328:7
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms)
{ return new mosek::fusion::DisjunctionTerms(_22_terms); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_22_terms); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _22_terms)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9189_;
    for (int32_t _23_i = (int)0, _9190_ = (int)((_22_terms)->size(0)); _23_i < _9190_ ; ++_23_i)
    {
      _9189_.push_back(((*_22_terms)(_23_i)));
    }
    auto _9191_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9189_.size()),_9189_.begin(),_9189_.end()));
    mosek::fusion::p_DisjunctionTerms::_get_impl(::mosek::fusion::DisjunctionTerms::t(_pubthis))->terms = _9191_;
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:325:6-79
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term)
{ return new mosek::fusion::DisjunctionTerms(_24_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_24_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _24_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(::mosek::fusion::p_DJC::term(_24_term).get())});
  }
}
// mosek.fusion.DisjunctionTerms.ctor

// mosek.fusion.DisjunctionTerms.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:323:6-80
mosek::fusion::DisjunctionTerms::t mosek::fusion::p_DisjunctionTerms::_new_DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term)
{ return new mosek::fusion::DisjunctionTerms(_25_term); }
mosek::fusion::DisjunctionTerms::DisjunctionTerms
  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term) : 
   DisjunctionTerms(new mosek::fusion::p_DisjunctionTerms(this))
{ mosek::fusion::p_DisjunctionTerms::_get_impl(this)->_initialize(_25_term); }
void mosek::fusion::p_DisjunctionTerms::_initialize  (monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _25_term)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    terms = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(::mosek::fusion::p_DJC::term(_25_term).get())});
  }
}
// mosek.fusion.DisjunctionTerms.ctor

void mosek::fusion::p_DisjunctionTerms::destroy()
{
  terms.reset();
}
void mosek::fusion::DisjunctionTerms::destroy() { mosek::fusion::p_DisjunctionTerms::_get_impl(this)->destroy(); }
// } class DisjunctionTerms
// class Term { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:243:4-317:5
mosek::fusion::Term::Term(mosek::fusion::p_Term *_impl) : _impl(_impl) { /*std::cout << "Term() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Term::~Term() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Term::p_Term
  (::mosek::fusion::Term * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.Term.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:261:6-269:7
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist)
{ return new mosek::fusion::Term(_26_elist,_27_dlist); }
mosek::fusion::Term::Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_26_elist,_27_dlist); }
void mosek::fusion::p_Term::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _26_elist,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _27_dlist)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _28_n = (int)((_26_elist)->size(0));
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_28_n)));
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_28_n)));
    int32_t _9192_ = (int)0;
    int32_t _9193_ = _28_n;
    for (int32_t _29_i = _9192_; _29_i < _9193_; ++_29_i)
    {
      {
        ((*exprs)(_29_i)) = ((*_26_elist)(_29_i));
        ((*domains)(_29_i)) = ((*_27_dlist)(_29_i));
      }
    }
  }
}
// mosek.fusion.Term.ctor

// mosek.fusion.Term.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:259:6-90
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d)
{ return new mosek::fusion::Term(_30_e,_31_d); }
mosek::fusion::Term::Term
  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_30_e,_31_d); }
void mosek::fusion::p_Term::_initialize  (monty::rc_ptr< ::mosek::fusion::Expression > _30_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _31_d)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >{::monty::rc_ptr<::mosek::fusion::Expression>(_30_e.get())});
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >{::monty::rc_ptr<::mosek::fusion::DJCDomain>(_31_d.get())});
  }
}
// mosek.fusion.Term.ctor

// mosek.fusion.Term.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:247:6-255:7
mosek::fusion::Term::t mosek::fusion::p_Term::_new_Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t)
{ return new mosek::fusion::Term(_32_t); }
mosek::fusion::Term::Term
  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t) : 
   Term(new mosek::fusion::p_Term(this))
{ mosek::fusion::p_Term::_get_impl(this)->_initialize(_32_t); }
void mosek::fusion::p_Term::_initialize  (std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _32_t)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    int32_t _33_n = (int)((_32_t)->size(0));
    exprs = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_33_n)));
    domains = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_33_n)));
    int32_t _9194_ = (int)0;
    int32_t _9195_ = _33_n;
    for (int32_t _34_i = _9194_; _34_i < _9195_; ++_34_i)
    {
      {
        ((*exprs)(_34_i)) = ((*mosek::fusion::p_Term::_get_impl(((*_32_t)(_34_i)))->exprs)((int)0));
        ((*domains)(_34_i)) = ((*mosek::fusion::p_Term::_get_impl(((*_32_t)(_34_i)))->domains)((int)0));
      }
    }
  }
}
// mosek.fusion.Term.ctor

// Begin mosek.fusion.Term.numaccterms
// Method mosek.fusion.Term.numaccterms @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:316:6-128
int32_t mosek::fusion::p_Term::numaccterms()
{
  int32_t _35_n = (int)0;
  int32_t _9196_ = (int)0;
  int32_t _9197_ = (int)((domains)->size(0));
  for (int32_t _36_i = _9196_; _36_i < _9197_; ++_36_i)
  {
    {
      _35_n += mosek::fusion::p_DJCDomain::_get_impl(((*domains)(_36_i)).get())->numaccterms();
    }
  }
  return _35_n;
}
// End mosek.fusion.Term.numaccterms

// Begin mosek.fusion.Term.numaccrows
// Method mosek.fusion.Term.numaccrows @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:315:6-127
int32_t mosek::fusion::p_Term::numaccrows()
{
  int32_t _37_n = (int)0;
  int32_t _9198_ = (int)0;
  int32_t _9199_ = (int)((domains)->size(0));
  for (int32_t _38_i = _9198_; _38_i < _9199_; ++_38_i)
  {
    {
      _37_n += mosek::fusion::p_DJCDomain::_get_impl(((*domains)(_38_i)).get())->numaccrows();
    }
  }
  return _37_n;
}
// End mosek.fusion.Term.numaccrows

// Begin mosek.fusion.Term.num
// Method mosek.fusion.Term.num @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:314:6-67
int32_t mosek::fusion::p_Term::num()
{
  return (int)((domains)->size(0));
}
// End mosek.fusion.Term.num

// Begin mosek.fusion.Term.size
// Method mosek.fusion.Term.size @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:313:6-121
int32_t mosek::fusion::Term :: size() { return mosek::fusion::p_Term::_get_impl(this)->size(); }
int32_t mosek::fusion::p_Term::size()
{
  int32_t _39_s = (int)0;
  int32_t _9200_ = (int)0;
  int32_t _9201_ = (int)((domains)->size(0));
  for (int32_t _40_i = _9200_; _40_i < _9201_; ++_40_i)
  {
    {
      _39_s += ((*domains)(_40_i))->size();
    }
  }
  return _39_s;
}
// End mosek.fusion.Term.size

// Begin mosek.fusion.Term.toDJCTerm
// Method mosek.fusion.Term.toDJCTerm @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:257:6-53
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term :: __mosek_2fusion_2Term__toDJCTerm() { return mosek::fusion::p_Term::_get_impl(this)->__mosek_2fusion_2Term__toDJCTerm(); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term::toDJCTerm() { return __mosek_2fusion_2Term__toDJCTerm(); }
/* override: mosek.fusion.ExprDomain.toDJCTerm*/
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::Term::__mosek_2fusion_2ExprDomain__toDJCTerm() { return __mosek_2fusion_2Term__toDJCTerm(); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_Term::__mosek_2fusion_2Term__toDJCTerm()
{
  return ::mosek::fusion::Term::t(_pubthis);
}
// End mosek.fusion.Term.toDJCTerm

void mosek::fusion::p_Term::destroy()
{
  domains.reset();
  exprs.reset();
}
void mosek::fusion::Term::destroy() { mosek::fusion::p_Term::_get_impl(this)->destroy(); }
// } class Term
// class SimpleTerm { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:228:4-230:5
mosek::fusion::SimpleTerm::SimpleTerm(mosek::fusion::p_SimpleTerm *_impl) : mosek::fusion::Term(_impl) { /*std::cout << "SimpleTerm()" << this << std::endl; */ }
mosek::fusion::SimpleTerm::~SimpleTerm() {  /* std::cout << "~SimpleTerm()" << this << std::endl << std::flush; */ }
mosek::fusion::p_SimpleTerm::p_SimpleTerm
  (::mosek::fusion::SimpleTerm * _pubthis) :     mosek::fusion::p_Term(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.SimpleTerm.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:229:6-64
mosek::fusion::SimpleTerm::t mosek::fusion::p_SimpleTerm::_new_SimpleTerm
  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d)
{ return new mosek::fusion::SimpleTerm(_41_e,_42_d); }
mosek::fusion::SimpleTerm::SimpleTerm
  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d) : 
   SimpleTerm(new mosek::fusion::p_SimpleTerm(this))
{ mosek::fusion::p_SimpleTerm::_get_impl(this)->_initialize(_41_e,_42_d); }
void mosek::fusion::p_SimpleTerm::_initialize  (monty::rc_ptr< ::mosek::fusion::Expression > _41_e,
    monty::rc_ptr< ::mosek::fusion::DJCDomain > _42_d)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Term::_initialize(_41_e,_42_d);
  }
}
// mosek.fusion.SimpleTerm.ctor

void mosek::fusion::p_SimpleTerm::destroy()
{
}
void mosek::fusion::SimpleTerm::destroy() { mosek::fusion::p_SimpleTerm::_get_impl(this)->destroy(); }
// } class SimpleTerm
// class DJCDomain { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:171:4-217:5
mosek::fusion::DJCDomain::DJCDomain(mosek::fusion::p_DJCDomain *_impl) : _impl(_impl) { /*std::cout << "DJCDomain() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DJCDomain::~DJCDomain() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DJCDomain::p_DJCDomain
  (::mosek::fusion::DJCDomain * _pubthis) :     _pubthis(_pubthis)
{}
mosek::fusion::DJCDomainType mosek::fusion::DJCDomain::get_dom() { return _impl->dom; }
void mosek::fusion::DJCDomain::set_dom(mosek::fusion::DJCDomainType _val) { _impl->dom = _val; }
int32_t mosek::fusion::DJCDomain::get_conedim() { return _impl->conedim; }
void mosek::fusion::DJCDomain::set_conedim(int32_t _val) { _impl->conedim = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::DJCDomain::get_shape() { return _impl->shape; }
void mosek::fusion::DJCDomain::set_shape(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->shape = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::DJCDomain::get_par() { return _impl->par; }
void mosek::fusion::DJCDomain::set_par(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->par = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::DJCDomain::get_b() { return _impl->b; }
void mosek::fusion::DJCDomain::set_b(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->b = _val; }
// mosek.fusion.DJCDomain.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:190:6-199:7
mosek::fusion::DJCDomain::t mosek::fusion::p_DJCDomain::_new_DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_)
{ return new mosek::fusion::DJCDomain(_43_b_,_44_par_,_45_shape_,_46_dom_); }
mosek::fusion::DJCDomain::DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_) : 
   DJCDomain(new mosek::fusion::p_DJCDomain(this))
{ mosek::fusion::p_DJCDomain::_get_impl(this)->_initialize(_43_b_,_44_par_,_45_shape_,_46_dom_); }
void mosek::fusion::p_DJCDomain::_initialize  (std::shared_ptr< monty::ndarray< double,1 > > _43_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _44_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _45_shape_,
    mosek::fusion::DJCDomainType _46_dom_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<double> _ptr__43_b_(_43_b_ ? _43_b_->raw() : nullptr, _43_b_ ? _43_b_->size(0) : 0);
    _checked_ptr_1<double> _ptr__44_par_(_44_par_ ? _44_par_->raw() : nullptr, _44_par_ ? _44_par_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__45_shape_(_45_shape_ ? _45_shape_->raw() : nullptr, _45_shape_ ? _45_shape_->size(0) : 0);
    b = _43_b_;
    par = _44_par_;
    shape = _45_shape_;
    dom = _46_dom_;
    conedim = ((int)((_45_shape_)->size(0)) - (int)1);
  }
}
// mosek.fusion.DJCDomain.ctor

// mosek.fusion.DJCDomain.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:178:6-188:7
mosek::fusion::DJCDomain::t mosek::fusion::p_DJCDomain::_new_DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_)
{ return new mosek::fusion::DJCDomain(_47_b_,_48_par_,_49_shape_,_50_conedim_,_51_dom_); }
mosek::fusion::DJCDomain::DJCDomain
  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_) : 
   DJCDomain(new mosek::fusion::p_DJCDomain(this))
{ mosek::fusion::p_DJCDomain::_get_impl(this)->_initialize(_47_b_,_48_par_,_49_shape_,_50_conedim_,_51_dom_); }
void mosek::fusion::p_DJCDomain::_initialize  (std::shared_ptr< monty::ndarray< double,1 > > _47_b_,
    std::shared_ptr< monty::ndarray< double,1 > > _48_par_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _49_shape_,
    int32_t _50_conedim_,
    mosek::fusion::DJCDomainType _51_dom_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<double> _ptr__47_b_(_47_b_ ? _47_b_->raw() : nullptr, _47_b_ ? _47_b_->size(0) : 0);
    _checked_ptr_1<double> _ptr__48_par_(_48_par_ ? _48_par_->raw() : nullptr, _48_par_ ? _48_par_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__49_shape_(_49_shape_ ? _49_shape_->raw() : nullptr, _49_shape_ ? _49_shape_->size(0) : 0);
    b = _47_b_;
    par = _48_par_;
    shape = _49_shape_;
    dom = _51_dom_;
    conedim = _50_conedim_;
  }
}
// mosek.fusion.DJCDomain.ctor

// Begin mosek.fusion.DJCDomain.numaccterms
// Method mosek.fusion.DJCDomain.numaccterms @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:211:6-216:7
int32_t mosek::fusion::p_DJCDomain::numaccterms()
{
  if ((dom == ::mosek::fusion::DJCDomainType::InRange))
  {
    {
      return (int)2;
    }
  }
  else
  {
    {
      return (int)1;
    }
  }
}
// End mosek.fusion.DJCDomain.numaccterms

// Begin mosek.fusion.DJCDomain.numaccrows
// Method mosek.fusion.DJCDomain.numaccrows @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:205:6-210:7
int32_t mosek::fusion::p_DJCDomain::numaccrows()
{
  if ((dom == ::mosek::fusion::DJCDomainType::InRange))
  {
    {
      return safe_mul( _pubthis->size(),(int)2 );
    }
  }
  else
  {
    {
      return _pubthis->size();
    }
  }
}
// End mosek.fusion.DJCDomain.numaccrows

// Begin mosek.fusion.DJCDomain.size
// Method mosek.fusion.DJCDomain.size @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:201:6-204:7
int32_t mosek::fusion::DJCDomain :: size() { return mosek::fusion::p_DJCDomain::_get_impl(this)->size(); }
int32_t mosek::fusion::p_DJCDomain::size()
{
  int32_t _52_s = (int)1;
  int32_t _9202_ = (int)0;
  int32_t _9203_ = (int)((shape)->size(0));
  for (int32_t _53_i = _9202_; _53_i < _9203_; ++_53_i)
  {
    {
      _52_s *= ((*shape)(_53_i));
    }
  }
  return _52_s;
}
// End mosek.fusion.DJCDomain.size

void mosek::fusion::p_DJCDomain::destroy()
{
  shape.reset();
  par.reset();
  b.reset();
}
void mosek::fusion::DJCDomain::destroy() { mosek::fusion::p_DJCDomain::_get_impl(this)->destroy(); }
// } class DJCDomain
// class DJC { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:32:4-139:5
mosek::fusion::DJC::DJC(mosek::fusion::p_DJC *_impl) : _impl(_impl) { /*std::cout << "DJC() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::DJC::~DJC() { delete _impl; _impl = nullptr; }
mosek::fusion::p_DJC::p_DJC
  (::mosek::fusion::DJC * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.DJC.ANDFromTerms
// Method mosek.fusion.DJC.ANDFromTerms @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:121:6-137:7
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::ANDFromTerms
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _54_tlist) { return mosek::fusion::p_DJC::ANDFromTerms(_54_tlist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::ANDFromTerms(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _54_tlist)
{
  int32_t _55_n = (int)0;
  int32_t _9204_ = (int)0;
  int32_t _9205_ = (int)((_54_tlist)->size(0));
  for (int32_t _56_i = _9204_; _56_i < _9205_; ++_56_i)
  {
    {
      _55_n += mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_56_i)).get())->num();
    }
  }
  std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > > _57_el = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Expression >,1 >(monty::shape(_55_n)));
  std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > > _58_dl = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::DJCDomain >,1 >(monty::shape(_55_n)));
  _55_n = (int)0;
  int32_t _9206_ = (int)0;
  int32_t _9207_ = (int)((_54_tlist)->size(0));
  for (int32_t _59_i = _9206_; _59_i < _9207_; ++_59_i)
  {
    {
      int32_t _9208_ = (int)0;
      int32_t _9209_ = mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)).get())->num();
      for (int32_t _60_j = _9208_; _60_j < _9209_; ++_60_j)
      {
        {
          ((*_57_el)(_55_n)) = ((*mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)))->exprs)(_60_j));
          ((*_58_dl)(_55_n)) = ((*mosek::fusion::p_Term::_get_impl(((*_54_tlist)(_59_i)))->domains)(_60_j));
          _55_n += (int)1;
        }
      }
    }
  }
  return ::mosek::fusion::p_Term::_new_Term(_57_el,_58_dl);
}
// End mosek.fusion.DJC.ANDFromTerms

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:116:6-136
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _61_elist) { return mosek::fusion::p_DJC::AND(_61_elist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _61_elist)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9210_;
  for (int32_t _62_i = (int)0, _9211_ = (int)((_61_elist)->size(0)); _62_i < _9211_ ; ++_62_i)
  {
    _9210_.push_back(((*_61_elist)(_62_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9212_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9210_.size()),_9210_.begin(),_9210_.end()));
  return ::mosek::fusion::p_DJC::ANDFromTerms(_9212_);
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:114:6-114
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _63_s1,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _64_s2,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _65_s3) { return mosek::fusion::p_DJC::AND(_63_s1,_64_s2,_65_s3); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _63_s1,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _64_s2,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _65_s3)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_63_s1.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_64_s2.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_65_s3.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:113:6-96
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _66_s1,
    monty::rc_ptr< ::mosek::fusion::SimpleTerm > _67_s2) { return mosek::fusion::p_DJC::AND(_66_s1,_67_s2); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _66_s1,monty::rc_ptr< ::mosek::fusion::SimpleTerm > _67_s2)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_66_s1.get()),::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_67_s2.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:112:6-78
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( monty::rc_ptr< ::mosek::fusion::SimpleTerm > _68_s1) { return mosek::fusion::p_DJC::AND(_68_s1); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(monty::rc_ptr< ::mosek::fusion::SimpleTerm > _68_s1)
{
  return ::mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 >{::monty::rc_ptr<::mosek::fusion::SimpleTerm>(_68_s1.get())}));
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.AND
// Method mosek.fusion.DJC.AND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:111:6-84
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::DJC::AND
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _69_slist) { return mosek::fusion::p_DJC::AND(_69_slist); }
monty::rc_ptr< ::mosek::fusion::Term > mosek::fusion::p_DJC::AND(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::SimpleTerm >,1 > > _69_slist)
{
  return ::mosek::fusion::p_Term::_new_Term(_69_slist);
}
// End mosek.fusion.DJC.AND

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:80:6-109:7
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Expression > _70_expr,
    monty::rc_ptr< ::mosek::fusion::RangeDomain > _71_dom) { return mosek::fusion::p_DJC::term(_70_expr,_71_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Expression > _70_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _71_dom)
{
  if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->sparsity.get()))
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Sparse domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  if (mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _72_domlb = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->lb;
  _checked_ptr_1<double> _ptr__72_domlb(_72_domlb ? _72_domlb->raw() : nullptr,_72_domlb ? _72_domlb->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _73_domub = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->ub;
  _checked_ptr_1<double> _ptr__73_domub(_73_domub ? _73_domub->raw() : nullptr,_73_domub ? _73_domub->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _74_eshape = _70_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__74_eshape(_74_eshape ? _74_eshape->raw() : nullptr,_74_eshape ? _74_eshape->size(0) : 0);
  int64_t _75_esize = _70_expr->getSize();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _76_shape = mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__76_shape(_76_shape ? _76_shape->raw() : nullptr,_76_shape ? _76_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _77_bl;
  _checked_ptr_1<double> _ptr__77_bl(_77_bl ? _77_bl->raw() : nullptr,_77_bl ? _77_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _78_bu;
  _checked_ptr_1<double> _ptr__78_bu(_78_bu ? _78_bu->raw() : nullptr,_78_bu ? _78_bu->size(0) : 0);
  if ((NULL == _76_shape.get()))
  {
    {
      _76_shape = _74_eshape;
      _ptr__76_shape.update(_76_shape ? _76_shape->raw() : nullptr, _76_shape ? _76_shape->size(0) : 0);
    }
  }
  else
  {
    {
      if (((int)((_76_shape)->size(0)) != (int)((_74_eshape)->size(0))))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
        }
      }
      else
      {
        {
          int32_t _9213_ = (int)0;
          int32_t _9214_ = (int)((_76_shape)->size(0));
          for (int32_t _79_i = _9213_; _79_i < _9214_; ++_79_i)
          {
            {
              if ((_ptr__76_shape[_79_i] != _ptr__74_eshape[_79_i]))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  if (mosek::fusion::p_RangeDomain::_get_impl(_71_dom)->scalable)
  {
    {
      std::vector<double> _9215_;
      for (int64_t _80_i = (int)0, _9216_ = _75_esize; _80_i < _9216_ ; ++_80_i)
      {
        _9215_.push_back(_ptr__72_domlb[(int)0]);
      }
      auto _9217_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9215_.size()),_9215_.begin(),_9215_.end()));
      _77_bl = _9217_;
      _ptr__77_bl.update(_77_bl ? _77_bl->raw() : nullptr, _77_bl ? _77_bl->size(0) : 0);
      std::vector<double> _9218_;
      for (int64_t _81_i = (int)0, _9219_ = _75_esize; _81_i < _9219_ ; ++_81_i)
      {
        _9218_.push_back(_ptr__73_domub[(int)0]);
      }
      auto _9220_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9218_.size()),_9218_.begin(),_9218_.end()));
      _78_bu = _9220_;
      _ptr__78_bu.update(_78_bu ? _78_bu->raw() : nullptr, _78_bu ? _78_bu->size(0) : 0);
    }
  }
  else
  {
    {
      _77_bl = _72_domlb;
      _ptr__77_bl.update(_77_bl ? _77_bl->raw() : nullptr, _77_bl ? _77_bl->size(0) : 0);
      _78_bu = _73_domub;
      _ptr__78_bu.update(_78_bu ? _78_bu->raw() : nullptr, _78_bu ? _78_bu->size(0) : 0);
    }
  }
  return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_70_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_77_bl,_78_bu,_76_shape,::mosek::fusion::DJCDomainType::InRange));
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:78:6-105
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Variable > _82_x,
    monty::rc_ptr< ::mosek::fusion::RangeDomain > _83_dom) { return mosek::fusion::p_DJC::term(_82_x,_83_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Variable > _82_x,monty::rc_ptr< ::mosek::fusion::RangeDomain > _83_dom)
{
  return ::mosek::fusion::p_DJC::term(_82_x->__mosek_2fusion_2Variable__asExpr(),_83_dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:76:6-110
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _84_exprdom) { return mosek::fusion::p_DJC::term(_84_exprdom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _84_exprdom)
{
  return ::mosek::fusion::p_DJC::term(mosek::fusion::p_ExprRangeDomain::_get_impl(_84_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_84_exprdom)->dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:41:6-73:7
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Expression > _85_expr,
    monty::rc_ptr< ::mosek::fusion::LinearDomain > _86_dom) { return mosek::fusion::p_DJC::term(_85_expr,_86_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Expression > _85_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _86_dom)
{
  if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->sparsity.get()))
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Sparse domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain is not acceptable in disjunctive constraints"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _87_dombnd = mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->bnd;
  _checked_ptr_1<double> _ptr__87_dombnd(_87_dombnd ? _87_dombnd->raw() : nullptr,_87_dombnd ? _87_dombnd->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _88_eshape = _85_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__88_eshape(_88_eshape ? _88_eshape->raw() : nullptr,_88_eshape ? _88_eshape->size(0) : 0);
  int64_t _89_esize = _85_expr->getSize();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _90_shape = mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__90_shape(_90_shape ? _90_shape->raw() : nullptr,_90_shape ? _90_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _91_b;
  _checked_ptr_1<double> _ptr__91_b(_91_b ? _91_b->raw() : nullptr,_91_b ? _91_b->size(0) : 0);
  if ((NULL == _90_shape.get()))
  {
    {
      _90_shape = _88_eshape;
      _ptr__90_shape.update(_90_shape ? _90_shape->raw() : nullptr, _90_shape ? _90_shape->size(0) : 0);
    }
  }
  else
  {
    {
      if (((int)((_90_shape)->size(0)) != (int)((_88_eshape)->size(0))))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
        }
      }
      else
      {
        {
          int32_t _9221_ = (int)0;
          int32_t _9222_ = (int)((_90_shape)->size(0));
          for (int32_t _92_i = _9221_; _92_i < _9222_; ++_92_i)
          {
            {
              if ((_ptr__90_shape[_92_i] != _ptr__88_eshape[_92_i]))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Mismatching shapes"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  if ((NULL == mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->bnd.get()))
  {
    {
      _91_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_89_esize)));
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else if(mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->scalable)
  {
    {
      std::vector<double> _9223_;
      for (int64_t _93_i = (int)0, _9224_ = _89_esize; _93_i < _9224_ ; ++_93_i)
      {
        _9223_.push_back(_ptr__87_dombnd[(int)0]);
      }
      auto _9225_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9223_.size()),_9223_.begin(),_9223_.end()));
      _91_b = _9225_;
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else if(mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->empty)
  {
    {
      _91_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_89_esize)));
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  else
  {
    {
      _91_b = _87_dombnd;
      _ptr__91_b.update(_91_b ? _91_b->raw() : nullptr, _91_b ? _91_b->size(0) : 0);
    }
  }
  if ((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::EqualTo));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::LessThan));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::GreaterThan));
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_86_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      return ::mosek::fusion::p_SimpleTerm::_new_SimpleTerm(_85_expr,::mosek::fusion::p_DJCDomain::_new_DJCDomain(_91_b,nullptr,_90_shape,::mosek::fusion::DJCDomainType::IsFree));
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Invalid linear domain"));
    }
  }
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:36:6-111
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _94_exprdom) { return mosek::fusion::p_DJC::term(_94_exprdom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _94_exprdom)
{
  return ::mosek::fusion::p_DJC::term(mosek::fusion::p_ExprLinearDomain::_get_impl(_94_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_94_exprdom)->dom);
}
// End mosek.fusion.DJC.term

// Begin mosek.fusion.DJC.term
// Method mosek.fusion.DJC.term @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Disjunction.mbi:33:6-105
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::DJC::term
  ( monty::rc_ptr< ::mosek::fusion::Variable > _95_x,
    monty::rc_ptr< ::mosek::fusion::LinearDomain > _96_dom) { return mosek::fusion::p_DJC::term(_95_x,_96_dom); }
monty::rc_ptr< ::mosek::fusion::SimpleTerm > mosek::fusion::p_DJC::term(monty::rc_ptr< ::mosek::fusion::Variable > _95_x,monty::rc_ptr< ::mosek::fusion::LinearDomain > _96_dom)
{
  return ::mosek::fusion::p_DJC::term(_95_x->__mosek_2fusion_2Variable__asExpr(),_96_dom);
}
// End mosek.fusion.DJC.term

void mosek::fusion::p_DJC::destroy()
{
}
void mosek::fusion::DJC::destroy() { mosek::fusion::p_DJC::_get_impl(this)->destroy(); }
// } class DJC
// class Sort { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:6:4-262:5
mosek::fusion::Sort::Sort(mosek::fusion::p_Sort *_impl) : _impl(_impl) { /*std::cout << "Sort() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Sort::~Sort() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Sort::p_Sort
  (::mosek::fusion::Sort * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.Sort.argTransposeSort
// Method mosek.fusion.Sort.argTransposeSort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:249:6-261:7
void mosek::fusion::Sort::argTransposeSort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _303_perm,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _304_ptrb,
    int32_t _305_m,
    int32_t _306_n,
    int32_t _307_p,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _308_val) { mosek::fusion::p_Sort::argTransposeSort(_303_perm,_304_ptrb,_305_m,_306_n,_307_p,_308_val); }
void mosek::fusion::p_Sort::argTransposeSort(std::shared_ptr< monty::ndarray< int64_t,1 > > _303_perm,std::shared_ptr< monty::ndarray< int64_t,1 > > _304_ptrb,int32_t _305_m,int32_t _306_n,int32_t _307_p,std::shared_ptr< monty::ndarray< int64_t,1 > > _308_val)
{
  _checked_ptr_1<int64_t> _ptr__303_perm(_303_perm ? _303_perm->raw() : nullptr, _303_perm ? _303_perm->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__304_ptrb(_304_ptrb ? _304_ptrb->raw() : nullptr, _304_ptrb ? _304_ptrb->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__308_val(_308_val ? _308_val->raw() : nullptr, _308_val ? _308_val->size(0) : 0);
  int32_t _9226_ = (int)0;
  int32_t _9227_ = safe_add( safe_mul( _305_m,_307_p ),(int)1 );
  for (int32_t _309_q = _9226_; _309_q < _9227_; ++_309_q)
  {
    {
      _ptr__304_ptrb[_309_q] = (int)0;
    }
  }
  int32_t _9228_ = (int)0;
  int32_t _9229_ = (int)((_308_val)->size(0));
  for (int32_t _310_q = _9228_; _310_q < _9229_; ++_310_q)
  {
    {
      ++ _ptr__304_ptrb[safe_add( safe_add( safe_mul( (_ptr__308_val[_ptr__303_perm[_310_q]] / safe_mul( _306_n,_307_p )),_307_p ),(_ptr__308_val[_ptr__303_perm[_310_q]] % _307_p) ),(int)1 )];
    }
  }
  int32_t _9230_ = (int)0;
  int32_t _9231_ = safe_mul( _305_m,_307_p );
  for (int32_t _311_i = _9230_; _311_i < _9231_; ++_311_i)
  {
    {
      _ptr__304_ptrb[safe_add( _311_i,(int)1 )] = safe_add( _ptr__304_ptrb[safe_add( _311_i,(int)1 )],_ptr__304_ptrb[_311_i] );
    }
  }
  std::vector<int64_t> _9232_;
  for (int32_t _313_i = (int)0, _9233_ = (int)((_303_perm)->size(0)); _313_i < _9233_ ; ++_313_i)
  {
    _9232_.push_back(_ptr__303_perm[_313_i]);
  }
  auto _9234_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9232_.size()),_9232_.begin(),_9232_.end()));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _312_per2 = _9234_;
  _checked_ptr_1<int64_t> _ptr__312_per2(_312_per2 ? _312_per2->raw() : nullptr,_312_per2 ? _312_per2->size(0) : 0);
  int64_t _9235_ = (int64_t)0;
  int64_t _9236_ = (int64_t)((_308_val)->size(0));
  for (int64_t _314_q = _9235_; _314_q < _9236_; ++_314_q)
  {
    {
      int64_t _315_k = safe_add( safe_mul( (_ptr__308_val[_ptr__312_per2[_314_q]] / safe_mul( _306_n,_307_p )),_307_p ),(_ptr__308_val[_ptr__312_per2[_314_q]] % _307_p) );
      _ptr__303_perm[_ptr__304_ptrb[_315_k]] = _ptr__312_per2[_314_q];
      ++ _ptr__304_ptrb[_315_k];
    }
  }
}
// End mosek.fusion.Sort.argTransposeSort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:207:6-210:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _316_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _317_vals1) { mosek::fusion::p_Sort::argsort(_316_idx,_317_vals1); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _316_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _317_vals1)
{
  _checked_ptr_1<int64_t> _ptr__316_idx(_316_idx ? _316_idx->raw() : nullptr, _316_idx ? _316_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__317_vals1(_317_vals1 ? _317_vals1->raw() : nullptr, _317_vals1 ? _317_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_316_idx,_317_vals1,(int64_t)((int)0),(int64_t)((int)((_316_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:202:6-205:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _318_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _319_vals1) { mosek::fusion::p_Sort::argsort(_318_idx,_319_vals1); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _318_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _319_vals1)
{
  _checked_ptr_1<int64_t> _ptr__318_idx(_318_idx ? _318_idx->raw() : nullptr, _318_idx ? _318_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__319_vals1(_319_vals1 ? _319_vals1->raw() : nullptr, _319_vals1 ? _319_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_318_idx,_319_vals1,(int64_t)((int)0),(int64_t)((int)((_318_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:197:6-200:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _320_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _321_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _322_vals2) { mosek::fusion::p_Sort::argsort(_320_idx,_321_vals1,_322_vals2); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _320_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _321_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _322_vals2)
{
  _checked_ptr_1<int64_t> _ptr__320_idx(_320_idx ? _320_idx->raw() : nullptr, _320_idx ? _320_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__321_vals1(_321_vals1 ? _321_vals1->raw() : nullptr, _321_vals1 ? _321_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__322_vals2(_322_vals2 ? _322_vals2->raw() : nullptr, _322_vals2 ? _322_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_320_idx,_321_vals1,_322_vals2,(int64_t)((int)0),(int64_t)((int)((_320_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:193:6-196:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _323_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _324_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _325_vals2) { mosek::fusion::p_Sort::argsort(_323_idx,_324_vals1,_325_vals2); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _323_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _324_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _325_vals2)
{
  _checked_ptr_1<int64_t> _ptr__323_idx(_323_idx ? _323_idx->raw() : nullptr, _323_idx ? _323_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__324_vals1(_324_vals1 ? _324_vals1->raw() : nullptr, _324_vals1 ? _324_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__325_vals2(_325_vals2 ? _325_vals2->raw() : nullptr, _325_vals2 ? _325_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_323_idx,_324_vals1,_325_vals2,(int64_t)((int)0),(int64_t)((int)((_323_idx)->size(0))),false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:188:6-191:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _326_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _327_vals1,
    int64_t _328_first,
    int64_t _329_last) { mosek::fusion::p_Sort::argsort(_326_idx,_327_vals1,_328_first,_329_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _326_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _327_vals1,int64_t _328_first,int64_t _329_last)
{
  _checked_ptr_1<int64_t> _ptr__326_idx(_326_idx ? _326_idx->raw() : nullptr, _326_idx ? _326_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__327_vals1(_327_vals1 ? _327_vals1->raw() : nullptr, _327_vals1 ? _327_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_326_idx,_327_vals1,_328_first,_329_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:183:6-186:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _330_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _331_vals1,
    int64_t _332_first,
    int64_t _333_last) { mosek::fusion::p_Sort::argsort(_330_idx,_331_vals1,_332_first,_333_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _330_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _331_vals1,int64_t _332_first,int64_t _333_last)
{
  _checked_ptr_1<int64_t> _ptr__330_idx(_330_idx ? _330_idx->raw() : nullptr, _330_idx ? _330_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__331_vals1(_331_vals1 ? _331_vals1->raw() : nullptr, _331_vals1 ? _331_vals1->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_330_idx,_331_vals1,_332_first,_333_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:178:6-181:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _334_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _335_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _336_vals2,
    int64_t _337_first,
    int64_t _338_last) { mosek::fusion::p_Sort::argsort(_334_idx,_335_vals1,_336_vals2,_337_first,_338_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _334_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _335_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _336_vals2,int64_t _337_first,int64_t _338_last)
{
  _checked_ptr_1<int64_t> _ptr__334_idx(_334_idx ? _334_idx->raw() : nullptr, _334_idx ? _334_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__335_vals1(_335_vals1 ? _335_vals1->raw() : nullptr, _335_vals1 ? _335_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__336_vals2(_336_vals2 ? _336_vals2->raw() : nullptr, _336_vals2 ? _336_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_334_idx,_335_vals1,_336_vals2,_337_first,_338_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:174:6-177:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _339_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _340_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _341_vals2,
    int64_t _342_first,
    int64_t _343_last) { mosek::fusion::p_Sort::argsort(_339_idx,_340_vals1,_341_vals2,_342_first,_343_last); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _339_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _340_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _341_vals2,int64_t _342_first,int64_t _343_last)
{
  _checked_ptr_1<int64_t> _ptr__339_idx(_339_idx ? _339_idx->raw() : nullptr, _339_idx ? _339_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__340_vals1(_340_vals1 ? _340_vals1->raw() : nullptr, _340_vals1 ? _340_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__341_vals2(_341_vals2 ? _341_vals2->raw() : nullptr, _341_vals2 ? _341_vals2->size(0) : 0);
  ::mosek::fusion::p_Sort::argsort(_339_idx,_340_vals1,_341_vals2,_342_first,_343_last,false);
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:156:6-171:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _344_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _345_vals1,
    int64_t _346_first,
    int64_t _347_last,
    bool _348_check) { mosek::fusion::p_Sort::argsort(_344_idx,_345_vals1,_346_first,_347_last,_348_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _344_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _345_vals1,int64_t _346_first,int64_t _347_last,bool _348_check)
{
  _checked_ptr_1<int64_t> _ptr__344_idx(_344_idx ? _344_idx->raw() : nullptr, _344_idx ? _344_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__345_vals1(_345_vals1 ? _345_vals1->raw() : nullptr, _345_vals1 ? _345_vals1->size(0) : 0);
  if (((!_348_check) || ((_347_last - _346_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_344_idx,_345_vals1,_346_first,_347_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_344_idx,_345_vals1,_346_first,_347_last,_348_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _349_minmax = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int64_t> _ptr__349_minmax(_349_minmax ? _349_minmax->raw() : nullptr,_349_minmax ? _349_minmax->size(0) : 0);
      int32_t _350_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_344_idx,_345_vals1,nullptr,_346_first,_347_last,_349_minmax);
      if (((_ptr__349_minmax[(int)1] - _ptr__349_minmax[(int)0]) < safe_mul( _350_scale,(_347_last - _346_first) )))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_344_idx,_345_vals1,_346_first,_347_last,_ptr__349_minmax[(int)0],_ptr__349_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_344_idx,_345_vals1,_346_first,_347_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:137:6-154:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _351_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _352_vals1,
    int64_t _353_first,
    int64_t _354_last,
    bool _355_check) { mosek::fusion::p_Sort::argsort(_351_idx,_352_vals1,_353_first,_354_last,_355_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _351_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _352_vals1,int64_t _353_first,int64_t _354_last,bool _355_check)
{
  _checked_ptr_1<int64_t> _ptr__351_idx(_351_idx ? _351_idx->raw() : nullptr, _351_idx ? _351_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__352_vals1(_352_vals1 ? _352_vals1->raw() : nullptr, _352_vals1 ? _352_vals1->size(0) : 0);
  if (((!_355_check) || ((_354_last - _353_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_351_idx,_352_vals1,_353_first,_354_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_351_idx,_352_vals1,_353_first,_354_last,_355_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _356_minmax = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int32_t> _ptr__356_minmax(_356_minmax ? _356_minmax->raw() : nullptr,_356_minmax ? _356_minmax->size(0) : 0);
      int32_t _357_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_351_idx,_352_vals1,nullptr,_353_first,_354_last,_356_minmax);
      if (((_ptr__356_minmax[(int)1] - _ptr__356_minmax[(int)0]) < safe_mul( _357_scale,(_354_last - _353_first) )))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_351_idx,_352_vals1,_353_first,_354_last,_ptr__356_minmax[(int)0],_ptr__356_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_351_idx,_352_vals1,_353_first,_354_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:116:6-135:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _358_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _359_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _360_vals2,
    int64_t _361_first,
    int64_t _362_last,
    bool _363_check) { mosek::fusion::p_Sort::argsort(_358_idx,_359_vals1,_360_vals2,_361_first,_362_last,_363_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _358_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _359_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _360_vals2,int64_t _361_first,int64_t _362_last,bool _363_check)
{
  _checked_ptr_1<int64_t> _ptr__358_idx(_358_idx ? _358_idx->raw() : nullptr, _358_idx ? _358_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__359_vals1(_359_vals1 ? _359_vals1->raw() : nullptr, _359_vals1 ? _359_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__360_vals2(_360_vals2 ? _360_vals2->raw() : nullptr, _360_vals2 ? _360_vals2->size(0) : 0);
  if (((!_363_check) || ((_362_last - _361_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_358_idx,_359_vals1,_360_vals2,_361_first,_362_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_358_idx,_359_vals1,_360_vals2,_361_first,_362_last,_363_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _364_minmax = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int64_t> _ptr__364_minmax(_364_minmax ? _364_minmax->raw() : nullptr,_364_minmax ? _364_minmax->size(0) : 0);
      int32_t _365_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_358_idx,_359_vals1,_360_vals2,_361_first,_362_last,_364_minmax);
      if ((((_ptr__364_minmax[(int)1] - _ptr__364_minmax[(int)0]) < safe_mul( _365_scale,(_362_last - _361_first) )) && ((_ptr__364_minmax[(int)3] - _ptr__364_minmax[(int)2]) < safe_mul( _365_scale,(_362_last - _361_first) ))))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_358_idx,_360_vals2,_361_first,_362_last,_ptr__364_minmax[(int)2],_ptr__364_minmax[(int)3]);
          ::mosek::fusion::p_Sort::argbucketsort(_358_idx,_359_vals1,_361_first,_362_last,_ptr__364_minmax[(int)0],_ptr__364_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_358_idx,_359_vals1,_360_vals2,_361_first,_362_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argsort
// Method mosek.fusion.Sort.argsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:95:6-114:7
void mosek::fusion::Sort::argsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _366_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _367_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _368_vals2,
    int64_t _369_first,
    int64_t _370_last,
    bool _371_check) { mosek::fusion::p_Sort::argsort(_366_idx,_367_vals1,_368_vals2,_369_first,_370_last,_371_check); }
void mosek::fusion::p_Sort::argsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _366_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _367_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _368_vals2,int64_t _369_first,int64_t _370_last,bool _371_check)
{
  _checked_ptr_1<int64_t> _ptr__366_idx(_366_idx ? _366_idx->raw() : nullptr, _366_idx ? _366_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__367_vals1(_367_vals1 ? _367_vals1->raw() : nullptr, _367_vals1 ? _367_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__368_vals2(_368_vals2 ? _368_vals2->raw() : nullptr, _368_vals2 ? _368_vals2->size(0) : 0);
  if (((!_371_check) || ((_370_last - _369_first) < (int)16)))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_366_idx,_367_vals1,_368_vals2,_369_first,_370_last);
    }
  }
  else if((!::mosek::fusion::p_Sort::issorted(_366_idx,_367_vals1,_368_vals2,_369_first,_370_last,_371_check)))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _372_minmax = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)4)));
      _checked_ptr_1<int32_t> _ptr__372_minmax(_372_minmax ? _372_minmax->raw() : nullptr,_372_minmax ? _372_minmax->size(0) : 0);
      int32_t _373_scale = (int)2;
      ::mosek::fusion::p_Sort::getminmax(_366_idx,_367_vals1,_368_vals2,_369_first,_370_last,_372_minmax);
      if ((((_ptr__372_minmax[(int)1] - _ptr__372_minmax[(int)0]) < safe_mul( _373_scale,(_370_last - _369_first) )) && ((_ptr__372_minmax[(int)3] - _ptr__372_minmax[(int)2]) < safe_mul( _373_scale,(_370_last - _369_first) ))))
      {
        {
          ::mosek::fusion::p_Sort::argbucketsort(_366_idx,_368_vals2,_369_first,_370_last,_ptr__372_minmax[(int)2],_ptr__372_minmax[(int)3]);
          ::mosek::fusion::p_Sort::argbucketsort(_366_idx,_367_vals1,_369_first,_370_last,_ptr__372_minmax[(int)0],_ptr__372_minmax[(int)1]);
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::argsort(_366_idx,_367_vals1,_368_vals2,_369_first,_370_last);
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.argsort

// Begin mosek.fusion.Sort.argbucketsort
// Method mosek.fusion.Sort.argbucketsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:90:6-93:7
void mosek::fusion::Sort::argbucketsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _374_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _375_vals,
    int64_t _376_first,
    int64_t _377_last,
    int64_t _378_minv,
    int64_t _379_maxv) { mosek::fusion::p_Sort::argbucketsort(_374_idx,_375_vals,_376_first,_377_last,_378_minv,_379_maxv); }
void mosek::fusion::p_Sort::argbucketsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _374_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _375_vals,int64_t _376_first,int64_t _377_last,int64_t _378_minv,int64_t _379_maxv)
{
  _checked_ptr_1<int64_t> _ptr__374_idx(_374_idx ? _374_idx->raw() : nullptr, _374_idx ? _374_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__375_vals(_375_vals ? _375_vals->raw() : nullptr, _375_vals ? _375_vals->size(0) : 0);
  ::mosek::fusion::Utils::Tools::bucketsort(_374_idx,_376_first,_377_last,_375_vals,_378_minv,_379_maxv);
}
// End mosek.fusion.Sort.argbucketsort

// Begin mosek.fusion.Sort.argbucketsort
// Method mosek.fusion.Sort.argbucketsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:85:6-88:7
void mosek::fusion::Sort::argbucketsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _380_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _381_vals,
    int64_t _382_first,
    int64_t _383_last,
    int32_t _384_minv,
    int32_t _385_maxv) { mosek::fusion::p_Sort::argbucketsort(_380_idx,_381_vals,_382_first,_383_last,_384_minv,_385_maxv); }
void mosek::fusion::p_Sort::argbucketsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _380_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _381_vals,int64_t _382_first,int64_t _383_last,int32_t _384_minv,int32_t _385_maxv)
{
  _checked_ptr_1<int64_t> _ptr__380_idx(_380_idx ? _380_idx->raw() : nullptr, _380_idx ? _380_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__381_vals(_381_vals ? _381_vals->raw() : nullptr, _381_vals ? _381_vals->size(0) : 0);
  ::mosek::fusion::Utils::Tools::bucketsort(_380_idx,_382_first,_383_last,_381_vals,_384_minv,_385_maxv);
}
// End mosek.fusion.Sort.argbucketsort

// Begin mosek.fusion.Sort.getminmax
// Method mosek.fusion.Sort.getminmax @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:61:6-80:7
void mosek::fusion::Sort::getminmax
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _386_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _387_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _388_vals2,
    int64_t _389_first,
    int64_t _390_last,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _391_res) { mosek::fusion::p_Sort::getminmax(_386_idx,_387_vals1,_388_vals2,_389_first,_390_last,_391_res); }
void mosek::fusion::p_Sort::getminmax(std::shared_ptr< monty::ndarray< int64_t,1 > > _386_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _387_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _388_vals2,int64_t _389_first,int64_t _390_last,std::shared_ptr< monty::ndarray< int64_t,1 > > _391_res)
{
  _checked_ptr_1<int64_t> _ptr__386_idx(_386_idx ? _386_idx->raw() : nullptr, _386_idx ? _386_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__387_vals1(_387_vals1 ? _387_vals1->raw() : nullptr, _387_vals1 ? _387_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__388_vals2(_388_vals2 ? _388_vals2->raw() : nullptr, _388_vals2 ? _388_vals2->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__391_res(_391_res ? _391_res->raw() : nullptr, _391_res ? _391_res->size(0) : 0);
  _ptr__391_res[(int)0] = _ptr__387_vals1[_ptr__386_idx[_389_first]];
  _ptr__391_res[(int)1] = _ptr__387_vals1[_ptr__386_idx[_389_first]];
  int64_t _9237_ = safe_add( _389_first,(int)1 );
  int64_t _9238_ = _390_last;
  for (int64_t _392_i = _9237_; _392_i < _9238_; ++_392_i)
  {
    {
      if ((_ptr__391_res[(int)0] > _ptr__387_vals1[_ptr__386_idx[_392_i]]))
      {
        {
          _ptr__391_res[(int)0] = _ptr__387_vals1[_ptr__386_idx[_392_i]];
        }
      }
      {}
      if ((_ptr__391_res[(int)1] < _ptr__387_vals1[_ptr__386_idx[_392_i]]))
      {
        {
          _ptr__391_res[(int)1] = _ptr__387_vals1[_ptr__386_idx[_392_i]];
        }
      }
      {}
    }
  }
  if ((NULL != _388_vals2.get()))
  {
    {
      _ptr__391_res[(int)2] = _ptr__388_vals2[_ptr__386_idx[_389_first]];
      _ptr__391_res[(int)3] = _ptr__388_vals2[_ptr__386_idx[_389_first]];
      int64_t _9239_ = safe_add( _389_first,(int)1 );
      int64_t _9240_ = _390_last;
      for (int64_t _393_i = _9239_; _393_i < _9240_; ++_393_i)
      {
        {
          if ((_ptr__391_res[(int)2] > _ptr__388_vals2[_ptr__386_idx[_393_i]]))
          {
            {
              _ptr__391_res[(int)2] = _ptr__388_vals2[_ptr__386_idx[_393_i]];
            }
          }
          {}
          if ((_ptr__391_res[(int)3] < _ptr__388_vals2[_ptr__386_idx[_393_i]]))
          {
            {
              _ptr__391_res[(int)3] = _ptr__388_vals2[_ptr__386_idx[_393_i]];
            }
          }
          {}
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.getminmax

// Begin mosek.fusion.Sort.getminmax
// Method mosek.fusion.Sort.getminmax @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:40:6-59:7
void mosek::fusion::Sort::getminmax
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _394_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _395_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _396_vals2,
    int64_t _397_first,
    int64_t _398_last,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _399_res) { mosek::fusion::p_Sort::getminmax(_394_idx,_395_vals1,_396_vals2,_397_first,_398_last,_399_res); }
void mosek::fusion::p_Sort::getminmax(std::shared_ptr< monty::ndarray< int64_t,1 > > _394_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _395_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _396_vals2,int64_t _397_first,int64_t _398_last,std::shared_ptr< monty::ndarray< int32_t,1 > > _399_res)
{
  _checked_ptr_1<int64_t> _ptr__394_idx(_394_idx ? _394_idx->raw() : nullptr, _394_idx ? _394_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__395_vals1(_395_vals1 ? _395_vals1->raw() : nullptr, _395_vals1 ? _395_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__396_vals2(_396_vals2 ? _396_vals2->raw() : nullptr, _396_vals2 ? _396_vals2->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__399_res(_399_res ? _399_res->raw() : nullptr, _399_res ? _399_res->size(0) : 0);
  _ptr__399_res[(int)0] = _ptr__395_vals1[_ptr__394_idx[_397_first]];
  _ptr__399_res[(int)1] = _ptr__395_vals1[_ptr__394_idx[_397_first]];
  int64_t _9241_ = safe_add( _397_first,(int)1 );
  int64_t _9242_ = _398_last;
  for (int64_t _400_i = _9241_; _400_i < _9242_; ++_400_i)
  {
    {
      if ((_ptr__399_res[(int)0] > _ptr__395_vals1[_ptr__394_idx[_400_i]]))
      {
        {
          _ptr__399_res[(int)0] = _ptr__395_vals1[_ptr__394_idx[_400_i]];
        }
      }
      {}
      if ((_ptr__399_res[(int)1] < _ptr__395_vals1[_ptr__394_idx[_400_i]]))
      {
        {
          _ptr__399_res[(int)1] = _ptr__395_vals1[_ptr__394_idx[_400_i]];
        }
      }
      {}
    }
  }
  if ((NULL != _396_vals2.get()))
  {
    {
      _ptr__399_res[(int)2] = _ptr__396_vals2[_ptr__394_idx[_397_first]];
      _ptr__399_res[(int)3] = _ptr__396_vals2[_ptr__394_idx[_397_first]];
      int64_t _9243_ = safe_add( _397_first,(int)1 );
      int64_t _9244_ = _398_last;
      for (int64_t _401_i = _9243_; _401_i < _9244_; ++_401_i)
      {
        {
          if ((_ptr__399_res[(int)2] > _ptr__396_vals2[_ptr__394_idx[_401_i]]))
          {
            {
              _ptr__399_res[(int)2] = _ptr__396_vals2[_ptr__394_idx[_401_i]];
            }
          }
          {}
          if ((_ptr__399_res[(int)3] < _ptr__396_vals2[_ptr__394_idx[_401_i]]))
          {
            {
              _ptr__399_res[(int)3] = _ptr__396_vals2[_ptr__394_idx[_401_i]];
            }
          }
          {}
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Sort.getminmax

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:32:6-38:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _402_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _403_vals1,
    int64_t _404_first,
    int64_t _405_last,
    bool _406_check) { return mosek::fusion::p_Sort::issorted(_402_idx,_403_vals1,_404_first,_405_last,_406_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _402_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _403_vals1,int64_t _404_first,int64_t _405_last,bool _406_check)
{
  _checked_ptr_1<int64_t> _ptr__402_idx(_402_idx ? _402_idx->raw() : nullptr, _402_idx ? _402_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__403_vals1(_403_vals1 ? _403_vals1->raw() : nullptr, _403_vals1 ? _403_vals1->size(0) : 0);
  int64_t _9245_ = _404_first;
  int64_t _9246_ = (_405_last - (int)1);
  for (int64_t _407_i = _9245_; _407_i < _9246_; ++_407_i)
  {
    {
      if ((_ptr__403_vals1[_ptr__402_idx[_407_i]] > _ptr__403_vals1[_ptr__402_idx[safe_add( _407_i,(int)1 )]]))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:24:6-30:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _408_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _409_vals1,
    int64_t _410_first,
    int64_t _411_last,
    bool _412_check) { return mosek::fusion::p_Sort::issorted(_408_idx,_409_vals1,_410_first,_411_last,_412_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _408_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _409_vals1,int64_t _410_first,int64_t _411_last,bool _412_check)
{
  _checked_ptr_1<int64_t> _ptr__408_idx(_408_idx ? _408_idx->raw() : nullptr, _408_idx ? _408_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__409_vals1(_409_vals1 ? _409_vals1->raw() : nullptr, _409_vals1 ? _409_vals1->size(0) : 0);
  int64_t _9247_ = _410_first;
  int64_t _9248_ = (_411_last - (int)1);
  for (int64_t _413_i = _9247_; _413_i < _9248_; ++_413_i)
  {
    {
      if ((_ptr__409_vals1[_ptr__408_idx[_413_i]] > _ptr__409_vals1[_ptr__408_idx[safe_add( _413_i,(int)1 )]]))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:16:6-22:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _414_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _415_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _416_vals2,
    int64_t _417_first,
    int64_t _418_last,
    bool _419_check) { return mosek::fusion::p_Sort::issorted(_414_idx,_415_vals1,_416_vals2,_417_first,_418_last,_419_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _414_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _415_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _416_vals2,int64_t _417_first,int64_t _418_last,bool _419_check)
{
  _checked_ptr_1<int64_t> _ptr__414_idx(_414_idx ? _414_idx->raw() : nullptr, _414_idx ? _414_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__415_vals1(_415_vals1 ? _415_vals1->raw() : nullptr, _415_vals1 ? _415_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__416_vals2(_416_vals2 ? _416_vals2->raw() : nullptr, _416_vals2 ? _416_vals2->size(0) : 0);
  int64_t _9249_ = _417_first;
  int64_t _9250_ = (_418_last - (int)1);
  for (int64_t _420_i = _9249_; _420_i < _9250_; ++_420_i)
  {
    {
      if (((_ptr__415_vals1[_ptr__414_idx[_420_i]] > _ptr__415_vals1[_ptr__414_idx[safe_add( _420_i,(int)1 )]]) || ((_ptr__415_vals1[_ptr__414_idx[safe_add( _420_i,(int)1 )]] == _ptr__415_vals1[_ptr__414_idx[_420_i]]) && (_ptr__416_vals2[_ptr__414_idx[_420_i]] > _ptr__416_vals2[_ptr__414_idx[safe_add( _420_i,(int)1 )]]))))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

// Begin mosek.fusion.Sort.issorted
// Method mosek.fusion.Sort.issorted @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Sort.mbi:8:6-14:7
bool mosek::fusion::Sort::issorted
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _421_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _422_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _423_vals2,
    int64_t _424_first,
    int64_t _425_last,
    bool _426_check) { return mosek::fusion::p_Sort::issorted(_421_idx,_422_vals1,_423_vals2,_424_first,_425_last,_426_check); }
bool mosek::fusion::p_Sort::issorted(std::shared_ptr< monty::ndarray< int64_t,1 > > _421_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _422_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _423_vals2,int64_t _424_first,int64_t _425_last,bool _426_check)
{
  _checked_ptr_1<int64_t> _ptr__421_idx(_421_idx ? _421_idx->raw() : nullptr, _421_idx ? _421_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__422_vals1(_422_vals1 ? _422_vals1->raw() : nullptr, _422_vals1 ? _422_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__423_vals2(_423_vals2 ? _423_vals2->raw() : nullptr, _423_vals2 ? _423_vals2->size(0) : 0);
  int64_t _9251_ = _424_first;
  int64_t _9252_ = (_425_last - (int)1);
  for (int64_t _427_i = _9251_; _427_i < _9252_; ++_427_i)
  {
    {
      if (((_ptr__422_vals1[_ptr__421_idx[_427_i]] > _ptr__422_vals1[_ptr__421_idx[safe_add( _427_i,(int)1 )]]) || ((_ptr__422_vals1[_ptr__421_idx[_427_i]] == _ptr__422_vals1[_ptr__421_idx[safe_add( _427_i,(int)1 )]]) && (_ptr__423_vals2[_ptr__421_idx[_427_i]] > _ptr__423_vals2[_ptr__421_idx[safe_add( _427_i,(int)1 )]]))))
      {
        {
          return false;
        }
      }
      {}
    }
  }
  return true;
}
// End mosek.fusion.Sort.issorted

void mosek::fusion::p_Sort::destroy()
{
}
void mosek::fusion::Sort::destroy() { mosek::fusion::p_Sort::_get_impl(this)->destroy(); }
// } class Sort
// class IndexCounter { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:448:4-525:5
mosek::fusion::IndexCounter::IndexCounter(mosek::fusion::p_IndexCounter *_impl) : _impl(_impl) { /*std::cout << "IndexCounter() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::IndexCounter::~IndexCounter() { delete _impl; _impl = nullptr; }
mosek::fusion::p_IndexCounter::p_IndexCounter
  (::mosek::fusion::IndexCounter * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.IndexCounter.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:488:6-498:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _428_shape)
{ return new mosek::fusion::IndexCounter(_428_shape); }
mosek::fusion::IndexCounter::IndexCounter
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _428_shape) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_428_shape); }
void mosek::fusion::p_IndexCounter::_initialize  (std::shared_ptr< monty::ndarray< int32_t,1 > > _428_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__428_shape(_428_shape ? _428_shape->raw() : nullptr, _428_shape ? _428_shape->size(0) : 0);
    n = (int)((_428_shape)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    start = (int64_t)0;
    strides = ::mosek::fusion::p_Set::strides(_428_shape);
    std::vector<int32_t> _9253_;
    for (int32_t _429_i = (int)0, _9254_ = (int)((_428_shape)->size(0)); _429_i < _9254_ ; ++_429_i)
    {
      _9253_.push_back(_ptr__428_shape[_429_i]);
    }
    auto _9255_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9253_.size()),_9253_.begin(),_9253_.end()));
    dims = _9255_;
  }
}
// mosek.fusion.IndexCounter.ctor

// mosek.fusion.IndexCounter.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:473:6-486:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (int64_t _430_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _431_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _432_shape)
{ return new mosek::fusion::IndexCounter(_430_start_,_431_dims_,_432_shape); }
mosek::fusion::IndexCounter::IndexCounter
  (int64_t _430_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _431_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _432_shape) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_430_start_,_431_dims_,_432_shape); }
void mosek::fusion::p_IndexCounter::_initialize  (int64_t _430_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _431_dims_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _432_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__431_dims_(_431_dims_ ? _431_dims_->raw() : nullptr, _431_dims_ ? _431_dims_->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__432_shape(_432_shape ? _432_shape->raw() : nullptr, _432_shape ? _432_shape->size(0) : 0);
    int32_t _433_nd = (int)((_432_shape)->size(0));
    n = (int)((_431_dims_)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    int32_t _9256_ = (int)0;
    int32_t _9257_ = n;
    for (int32_t _434_i = _9256_; _434_i < _9257_; ++_434_i)
    {
      {
        ((*st)(_434_i)) = _430_start_;
      }
    }
    strides = ::mosek::fusion::p_Set::strides(_432_shape);
    dims = _431_dims_;
    start = _430_start_;
  }
}
// mosek.fusion.IndexCounter.ctor

// mosek.fusion.IndexCounter.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:459:6-471:7
mosek::fusion::IndexCounter::t mosek::fusion::p_IndexCounter::_new_IndexCounter
  (int64_t _435_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _436_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _437_strides_)
{ return new mosek::fusion::IndexCounter(_435_start_,_436_dims_,_437_strides_); }
mosek::fusion::IndexCounter::IndexCounter
  (int64_t _435_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _436_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _437_strides_) : 
   IndexCounter(new mosek::fusion::p_IndexCounter(this))
{ mosek::fusion::p_IndexCounter::_get_impl(this)->_initialize(_435_start_,_436_dims_,_437_strides_); }
void mosek::fusion::p_IndexCounter::_initialize  (int64_t _435_start_,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _436_dims_,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _437_strides_)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__436_dims_(_436_dims_ ? _436_dims_->raw() : nullptr, _436_dims_ ? _436_dims_->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__437_strides_(_437_strides_ ? _437_strides_->raw() : nullptr, _437_strides_ ? _437_strides_->size(0) : 0);
    n = (int)((_436_dims_)->size(0));
    ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(n)));
    st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
    int32_t _9258_ = (int)0;
    int32_t _9259_ = n;
    for (int32_t _438_i = _9258_; _438_i < _9259_; ++_438_i)
    {
      {
        ((*st)(_438_i)) = _435_start_;
      }
    }
    strides = _437_strides_;
    dims = _436_dims_;
    start = _435_start_;
  }
}
// mosek.fusion.IndexCounter.ctor

// Begin mosek.fusion.IndexCounter.atEnd
// Method mosek.fusion.IndexCounter.atEnd @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:524:6-66
bool mosek::fusion::IndexCounter :: atEnd() { return mosek::fusion::p_IndexCounter::_get_impl(this)->atEnd(); }
bool mosek::fusion::p_IndexCounter::atEnd()
{
  return (((*ii)((n - (int)1))) >= ((*dims)((n - (int)1))));
}
// End mosek.fusion.IndexCounter.atEnd

// Begin mosek.fusion.IndexCounter.getIndex
// Method mosek.fusion.IndexCounter.getIndex @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:522:6-54
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::IndexCounter :: getIndex() { return mosek::fusion::p_IndexCounter::_get_impl(this)->getIndex(); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_IndexCounter::getIndex()
{
  return ii;
}
// End mosek.fusion.IndexCounter.getIndex

// Begin mosek.fusion.IndexCounter.next
// Method mosek.fusion.IndexCounter.next @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:521:6-68
int64_t mosek::fusion::IndexCounter :: next() { return mosek::fusion::p_IndexCounter::_get_impl(this)->next(); }
int64_t mosek::fusion::p_IndexCounter::next()
{
  int64_t _439_r = _pubthis->get();
  _pubthis->inc();
  return _439_r;
}
// End mosek.fusion.IndexCounter.next

// Begin mosek.fusion.IndexCounter.get
// Method mosek.fusion.IndexCounter.get @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:519:6-49
int64_t mosek::fusion::IndexCounter :: get() { return mosek::fusion::p_IndexCounter::_get_impl(this)->get(); }
int64_t mosek::fusion::p_IndexCounter::get()
{
  return ((*st)((int)0));
}
// End mosek.fusion.IndexCounter.get

// Begin mosek.fusion.IndexCounter.inc
// Method mosek.fusion.IndexCounter.inc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:505:6-517:7
void mosek::fusion::IndexCounter :: inc() { mosek::fusion::p_IndexCounter::_get_impl(this)->inc(); }
void mosek::fusion::p_IndexCounter::inc()
{
  ++ ((*ii)((int)0));
  ((*st)((int)0)) = safe_add( ((*st)((int)0)),((*strides)((n - (int)1))) );
  int32_t _440_i;
  {
    _440_i = (int)0;
    for(;((_440_i < (n - (int)1)) && (((*ii)(_440_i)) == ((*dims)(((n - _440_i) - (int)1)))));++ _440_i)
    {
      {
        ((*ii)(_440_i)) = (int)0;
        ++ ((*ii)(safe_add( _440_i,(int)1 )));
        ((*st)(safe_add( _440_i,(int)1 ))) = safe_add( ((*st)(safe_add( _440_i,(int)1 ))),((*strides)(((n - _440_i) - (int)2))) );
      }
    }
  }
  int32_t _9260_ = (int)0;
  int32_t _9261_ = _440_i;
  for (int32_t _441_j = _9260_; _441_j < _9261_; ++_441_j)
  {
    {
      ((*st)(_441_j)) = ((*st)(_440_i));
    }
  }
}
// End mosek.fusion.IndexCounter.inc

// Begin mosek.fusion.IndexCounter.reset
// Method mosek.fusion.IndexCounter.reset @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:500:6-503:7
void mosek::fusion::IndexCounter :: reset() { mosek::fusion::p_IndexCounter::_get_impl(this)->reset(); }
void mosek::fusion::p_IndexCounter::reset()
{
  st = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(n)));
  int32_t _9262_ = (int)0;
  int32_t _9263_ = n;
  for (int32_t _442_i = _9262_; _442_i < _9263_; ++_442_i)
  {
    {
      ((*st)(_442_i)) = start;
    }
  }
}
// End mosek.fusion.IndexCounter.reset

void mosek::fusion::p_IndexCounter::destroy()
{
  dims.reset();
  strides.reset();
  st.reset();
  ii.reset();
}
void mosek::fusion::IndexCounter::destroy() { mosek::fusion::p_IndexCounter::_get_impl(this)->destroy(); }
// } class IndexCounter
// class CommonTools { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:6:4-445:5
mosek::fusion::CommonTools::CommonTools(mosek::fusion::p_CommonTools *_impl) : _impl(_impl) { /*std::cout << "CommonTools() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::CommonTools::~CommonTools() { delete _impl; _impl = nullptr; }
mosek::fusion::p_CommonTools::p_CommonTools
  (::mosek::fusion::CommonTools * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:433:6-444:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _443_values,
    int32_t _444_newsize) { return mosek::fusion::p_CommonTools::resize(_443_values,_444_newsize); }
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< int64_t,1 > > _443_values,int32_t _444_newsize)
{
  _checked_ptr_1<int64_t> _ptr__443_values(_443_values ? _443_values->raw() : nullptr, _443_values ? _443_values->size(0) : 0);
  if ((_444_newsize == (int)((_443_values)->size(0))))
  {
    {
      return _443_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _445_newvals = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_444_newsize)));
      _checked_ptr_1<int64_t> _ptr__445_newvals(_445_newvals ? _445_newvals->raw() : nullptr,_445_newvals ? _445_newvals->size(0) : 0);
      if ((_444_newsize > (int)((_443_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_443_values,(int)0,_445_newvals,(int)0,(int)((_443_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_443_values,(int)0,_445_newvals,(int)0,_444_newsize);
        }
      }
      return _445_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:421:6-432:7
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _446_values,
    int32_t _447_newsize) { return mosek::fusion::p_CommonTools::resize(_446_values,_447_newsize); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< int32_t,1 > > _446_values,int32_t _447_newsize)
{
  _checked_ptr_1<int32_t> _ptr__446_values(_446_values ? _446_values->raw() : nullptr, _446_values ? _446_values->size(0) : 0);
  if ((_447_newsize == (int)((_446_values)->size(0))))
  {
    {
      return _446_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _448_newvals = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_447_newsize)));
      _checked_ptr_1<int32_t> _ptr__448_newvals(_448_newvals ? _448_newvals->raw() : nullptr,_448_newvals ? _448_newvals->size(0) : 0);
      if ((_447_newsize > (int)((_446_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_446_values,(int)0,_448_newvals,(int)0,(int)((_446_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_446_values,(int)0,_448_newvals,(int)0,_447_newsize);
        }
      }
      return _448_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.resize
// Method mosek.fusion.CommonTools.resize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:409:6-420:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::CommonTools::resize
  ( std::shared_ptr< monty::ndarray< double,1 > > _449_values,
    int32_t _450_newsize) { return mosek::fusion::p_CommonTools::resize(_449_values,_450_newsize); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_CommonTools::resize(std::shared_ptr< monty::ndarray< double,1 > > _449_values,int32_t _450_newsize)
{
  _checked_ptr_1<double> _ptr__449_values(_449_values ? _449_values->raw() : nullptr, _449_values ? _449_values->size(0) : 0);
  if ((_450_newsize == (int)((_449_values)->size(0))))
  {
    {
      return _449_values;
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _451_newvals = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_450_newsize)));
      _checked_ptr_1<double> _ptr__451_newvals(_451_newvals ? _451_newvals->raw() : nullptr,_451_newvals ? _451_newvals->size(0) : 0);
      if ((_450_newsize > (int)((_449_values)->size(0))))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_449_values,(int)0,_451_newvals,(int)0,(int)((_449_values)->size(0)));
        }
      }
      else
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(_449_values,(int)0,_451_newvals,(int)0,_450_newsize);
        }
      }
      return _451_newvals;
    }
  }
}
// End mosek.fusion.CommonTools.resize

// Begin mosek.fusion.CommonTools.binarySearch
// Method mosek.fusion.CommonTools.binarySearch @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:391:6-407:7
int32_t mosek::fusion::CommonTools::binarySearch
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _452_values,
    int32_t _453_target) { return mosek::fusion::p_CommonTools::binarySearch(_452_values,_453_target); }
int32_t mosek::fusion::p_CommonTools::binarySearch(std::shared_ptr< monty::ndarray< int32_t,1 > > _452_values,int32_t _453_target)
{
  _checked_ptr_1<int32_t> _ptr__452_values(_452_values ? _452_values->raw() : nullptr, _452_values ? _452_values->size(0) : 0);
  int32_t _454_i0 = (int)0;
  int32_t _455_i1 = (int)((_452_values)->size(0));
  if ((_ptr__452_values[_454_i0] == _453_target))
  {
    {
      return _454_i0;
    }
  }
  else if((_ptr__452_values[(_455_i1 - (int)1)] == _453_target))
  {
    {
      return (_455_i1 - (int)1);
    }
  }
  {}
  while ( (_454_i0 < (_455_i1 - (int)1)) )
  {
    {
      int32_t _456_p = ((_455_i1 - _454_i0) / (int)2);
      if ((_453_target < _ptr__452_values[_456_p]))
      {
        {
          _454_i0 = safe_add( _456_p,(int)1 );
        }
      }
      else if((_453_target > _ptr__452_values[_456_p]))
      {
        {
          _455_i1 = _456_p;
        }
      }
      else
      {
        {
          return _456_p;
        }
      }
    }
  }
  return (-(int)1);
}
// End mosek.fusion.CommonTools.binarySearch

// Begin mosek.fusion.CommonTools.binarySearch
// Method mosek.fusion.CommonTools.binarySearch @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:383:6-389:7
int32_t mosek::fusion::CommonTools::binarySearch
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _457_values,
    int64_t _458_target) { return mosek::fusion::p_CommonTools::binarySearch(_457_values,_458_target); }
int32_t mosek::fusion::p_CommonTools::binarySearch(std::shared_ptr< monty::ndarray< int64_t,1 > > _457_values,int64_t _458_target)
{
  _checked_ptr_1<int64_t> _ptr__457_values(_457_values ? _457_values->raw() : nullptr, _457_values ? _457_values->size(0) : 0);
  int32_t _459_i = ::mosek::fusion::p_CommonTools::binarySearchL(_457_values,_458_target);
  if (((_459_i < (int)((_457_values)->size(0))) && (_ptr__457_values[_459_i] == _458_target)))
  {
    {
      return _459_i;
    }
  }
  else
  {
    {
      return (-(int)1);
    }
  }
}
// End mosek.fusion.CommonTools.binarySearch

// Begin mosek.fusion.CommonTools.binarySearchR
// Method mosek.fusion.CommonTools.binarySearchR @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:367:6-381:7
int32_t mosek::fusion::CommonTools::binarySearchR
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _460_values,
    int64_t _461_target) { return mosek::fusion::p_CommonTools::binarySearchR(_460_values,_461_target); }
int32_t mosek::fusion::p_CommonTools::binarySearchR(std::shared_ptr< monty::ndarray< int64_t,1 > > _460_values,int64_t _461_target)
{
  _checked_ptr_1<int64_t> _ptr__460_values(_460_values ? _460_values->raw() : nullptr, _460_values ? _460_values->size(0) : 0);
  int32_t _462_i0 = (int)0;
  int32_t _463_i1 = (int)((_460_values)->size(0));
  if ((_ptr__460_values[_462_i0] > _461_target))
  {
    {
      return _462_i0;
    }
  }
  {}
  while ( (_462_i0 < (_463_i1 - (int)1)) )
  {
    {
      int32_t _464_p = (safe_add( _463_i1,_462_i0 ) / (int)2);
      if ((_461_target >= _ptr__460_values[_464_p]))
      {
        {
          _462_i0 = _464_p;
        }
      }
      else
      {
        {
          _463_i1 = _464_p;
        }
      }
    }
  }
  return _463_i1;
}
// End mosek.fusion.CommonTools.binarySearchR

// Begin mosek.fusion.CommonTools.binarySearchL
// Method mosek.fusion.CommonTools.binarySearchL @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:350:6-364:7
int32_t mosek::fusion::CommonTools::binarySearchL
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _465_values,
    int64_t _466_target) { return mosek::fusion::p_CommonTools::binarySearchL(_465_values,_466_target); }
int32_t mosek::fusion::p_CommonTools::binarySearchL(std::shared_ptr< monty::ndarray< int64_t,1 > > _465_values,int64_t _466_target)
{
  _checked_ptr_1<int64_t> _ptr__465_values(_465_values ? _465_values->raw() : nullptr, _465_values ? _465_values->size(0) : 0);
  int32_t _467_i0 = (int)0;
  int32_t _468_i1 = (int)((_465_values)->size(0));
  if ((_466_target <= _ptr__465_values[_467_i0]))
  {
    {
      return _467_i0;
    }
  }
  {}
  while ( (_467_i0 < (_468_i1 - (int)1)) )
  {
    {
      int32_t _469_p = (safe_add( _468_i1,_467_i0 ) / (int)2);
      if ((_ptr__465_values[_469_p] < _466_target))
      {
        {
          _467_i0 = _469_p;
        }
      }
      else
      {
        {
          _468_i1 = _469_p;
        }
      }
    }
  }
  return _468_i1;
}
// End mosek.fusion.CommonTools.binarySearchL

// Begin mosek.fusion.CommonTools.ndIncr
// Method mosek.fusion.CommonTools.ndIncr @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:337:6-346:7
void mosek::fusion::CommonTools::ndIncr
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _470_ndidx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _471_first,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _472_last) { mosek::fusion::p_CommonTools::ndIncr(_470_ndidx,_471_first,_472_last); }
void mosek::fusion::p_CommonTools::ndIncr(std::shared_ptr< monty::ndarray< int32_t,1 > > _470_ndidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _471_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _472_last)
{
  _checked_ptr_1<int32_t> _ptr__470_ndidx(_470_ndidx ? _470_ndidx->raw() : nullptr, _470_ndidx ? _470_ndidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__471_first(_471_first ? _471_first->raw() : nullptr, _471_first ? _471_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__472_last(_472_last ? _472_last->raw() : nullptr, _472_last ? _472_last->size(0) : 0);
  int32_t _473_i = ((int)((_470_ndidx)->size(0)) - (int)1);
  ++ _ptr__470_ndidx[_473_i];
  while ( ((_473_i > (int)0) && (_ptr__470_ndidx[_473_i] >= _ptr__472_last[_473_i])) )
  {
    {
      _ptr__470_ndidx[_473_i] = _ptr__471_first[_473_i];
      ++ _ptr__470_ndidx[(_473_i - (int)1)];
    }
  }
}
// End mosek.fusion.CommonTools.ndIncr

// Begin mosek.fusion.CommonTools.transposeTriplets
// Method mosek.fusion.CommonTools.transposeTriplets @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:290:6-335:7
void mosek::fusion::CommonTools::transposeTriplets
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _474_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _475_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _476_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _477_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _478_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _479_tval_,
    int64_t _480_nelm,
    int32_t _481_dimi,
    int32_t _482_dimj) { mosek::fusion::p_CommonTools::transposeTriplets(_474_subi,_475_subj,_476_val,_477_tsubi_,_478_tsubj_,_479_tval_,_480_nelm,_481_dimi,_482_dimj); }
void mosek::fusion::p_CommonTools::transposeTriplets(std::shared_ptr< monty::ndarray< int32_t,1 > > _474_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _475_subj,std::shared_ptr< monty::ndarray< double,1 > > _476_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _477_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int64_t,1 > >,1 > > _478_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _479_tval_,int64_t _480_nelm,int32_t _481_dimi,int32_t _482_dimj)
{
  _checked_ptr_1<int32_t> _ptr__474_subi(_474_subi ? _474_subi->raw() : nullptr, _474_subi ? _474_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__475_subj(_475_subj ? _475_subj->raw() : nullptr, _475_subj ? _475_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__476_val(_476_val ? _476_val->raw() : nullptr, _476_val ? _476_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _483_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _481_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__483_rptrb(_483_rptrb ? _483_rptrb->raw() : nullptr,_483_rptrb ? _483_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _484_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _482_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__484_cptrb(_484_cptrb ? _484_cptrb->raw() : nullptr,_484_cptrb ? _484_cptrb->size(0) : 0);
  {
    int64_t _485_i = (int64_t)0;
    for(;(_485_i < _480_nelm);++ _485_i)
    {
      {
        _ptr__484_cptrb[safe_add( _ptr__475_subj[_485_i],(int)2 )] = safe_add( _ptr__484_cptrb[safe_add( _ptr__475_subj[_485_i],(int)2 )],(int)1 );
        _ptr__483_rptrb[safe_add( _ptr__474_subi[_485_i],(int)2 )] = safe_add( _ptr__483_rptrb[safe_add( _ptr__474_subi[_485_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _486_i = (int)1;
    for(;(_486_i < _481_dimi);++ _486_i)
    {
      {
        _ptr__483_rptrb[safe_add( _486_i,(int)1 )] = safe_add( _ptr__483_rptrb[safe_add( _486_i,(int)1 )],_ptr__483_rptrb[_486_i] );
      }
    }
  }
  {
    int32_t _487_i = (int)1;
    for(;(_487_i < _482_dimj);++ _487_i)
    {
      {
        _ptr__484_cptrb[safe_add( _487_i,(int)1 )] = safe_add( _ptr__484_cptrb[safe_add( _487_i,(int)1 )],_ptr__484_cptrb[_487_i] );
      }
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _488_tsubi = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_480_nelm)));
  _checked_ptr_1<int64_t> _ptr__488_tsubi(_488_tsubi ? _488_tsubi->raw() : nullptr,_488_tsubi ? _488_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _489_tsubj = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_480_nelm)));
  _checked_ptr_1<int64_t> _ptr__489_tsubj(_489_tsubj ? _489_tsubj->raw() : nullptr,_489_tsubj ? _489_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _490_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_480_nelm)));
  _checked_ptr_1<double> _ptr__490_tval(_490_tval ? _490_tval->raw() : nullptr,_490_tval ? _490_tval->size(0) : 0);
  int32_t _9264_ = (int)0;
  int32_t _9265_ = _481_dimi;
  for (int32_t _491_i = _9264_; _491_i < _9265_; ++_491_i)
  {
    {
      {
        int64_t _492_k = _ptr__483_rptrb[_491_i];
        for(;(_492_k < _ptr__483_rptrb[safe_add( _491_i,(int)1 )]);++ _492_k)
        {
          {
            int32_t _493_j = _ptr__475_subj[_492_k];
            int64_t _494_cidx = _ptr__484_cptrb[_493_j];
            _ptr__488_tsubi[_494_cidx] = _491_i;
            _ptr__489_tsubj[_494_cidx] = _493_j;
            _ptr__490_tval[_494_cidx] = _ptr__476_val[_492_k];
            _ptr__484_cptrb[_493_j] = safe_add( _494_cidx,(int)1 );
          }
        }
      }
    }
  }
  ((*_477_tsubi_)((int)0)) = _489_tsubj;
  ((*_478_tsubj_)((int)0)) = _488_tsubi;
  ((*_479_tval_)((int)0)) = _490_tval;
}
// End mosek.fusion.CommonTools.transposeTriplets

// Begin mosek.fusion.CommonTools.transposeTriplets
// Method mosek.fusion.CommonTools.transposeTriplets @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:242:6-287:7
void mosek::fusion::CommonTools::transposeTriplets
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _495_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _496_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _497_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _498_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _499_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _500_tval_,
    int64_t _501_nelm,
    int32_t _502_dimi,
    int32_t _503_dimj) { mosek::fusion::p_CommonTools::transposeTriplets(_495_subi,_496_subj,_497_val,_498_tsubi_,_499_tsubj_,_500_tval_,_501_nelm,_502_dimi,_503_dimj); }
void mosek::fusion::p_CommonTools::transposeTriplets(std::shared_ptr< monty::ndarray< int32_t,1 > > _495_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _496_subj,std::shared_ptr< monty::ndarray< double,1 > > _497_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _498_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _499_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _500_tval_,int64_t _501_nelm,int32_t _502_dimi,int32_t _503_dimj)
{
  _checked_ptr_1<int32_t> _ptr__495_subi(_495_subi ? _495_subi->raw() : nullptr, _495_subi ? _495_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__496_subj(_496_subj ? _496_subj->raw() : nullptr, _496_subj ? _496_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__497_val(_497_val ? _497_val->raw() : nullptr, _497_val ? _497_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _504_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _502_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__504_rptrb(_504_rptrb ? _504_rptrb->raw() : nullptr,_504_rptrb ? _504_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _505_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _503_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__505_cptrb(_505_cptrb ? _505_cptrb->raw() : nullptr,_505_cptrb ? _505_cptrb->size(0) : 0);
  {
    int64_t _506_i = (int64_t)0;
    for(;(_506_i < _501_nelm);++ _506_i)
    {
      {
        _ptr__505_cptrb[safe_add( _ptr__496_subj[_506_i],(int)2 )] = safe_add( _ptr__505_cptrb[safe_add( _ptr__496_subj[_506_i],(int)2 )],(int)1 );
        _ptr__504_rptrb[safe_add( _ptr__495_subi[_506_i],(int)2 )] = safe_add( _ptr__504_rptrb[safe_add( _ptr__495_subi[_506_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _507_i = (int)1;
    for(;(_507_i < _502_dimi);++ _507_i)
    {
      {
        _ptr__504_rptrb[safe_add( _507_i,(int)1 )] = safe_add( _ptr__504_rptrb[safe_add( _507_i,(int)1 )],_ptr__504_rptrb[_507_i] );
      }
    }
  }
  {
    int32_t _508_i = (int)1;
    for(;(_508_i < _503_dimj);++ _508_i)
    {
      {
        _ptr__505_cptrb[safe_add( _508_i,(int)1 )] = safe_add( _ptr__505_cptrb[safe_add( _508_i,(int)1 )],_ptr__505_cptrb[_508_i] );
      }
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _509_tsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_501_nelm)));
  _checked_ptr_1<int32_t> _ptr__509_tsubi(_509_tsubi ? _509_tsubi->raw() : nullptr,_509_tsubi ? _509_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _510_tsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_501_nelm)));
  _checked_ptr_1<int32_t> _ptr__510_tsubj(_510_tsubj ? _510_tsubj->raw() : nullptr,_510_tsubj ? _510_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _511_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_501_nelm)));
  _checked_ptr_1<double> _ptr__511_tval(_511_tval ? _511_tval->raw() : nullptr,_511_tval ? _511_tval->size(0) : 0);
  {
    int32_t _512_i = (int)0;
    for(;(_512_i < _502_dimi);++ _512_i)
    {
      {
        int64_t _9266_ = _ptr__504_rptrb[_512_i];
        int64_t _9267_ = _ptr__504_rptrb[safe_add( _512_i,(int)1 )];
        for (int64_t _513_k = _9266_; _513_k < _9267_; ++_513_k)
        {
          {
            int32_t _514_j = _ptr__496_subj[_513_k];
            int64_t _515_cidx = _ptr__505_cptrb[_514_j];
            _ptr__509_tsubi[_515_cidx] = _512_i;
            _ptr__510_tsubj[_515_cidx] = _514_j;
            _ptr__511_tval[_515_cidx] = _ptr__497_val[_513_k];
            _ptr__505_cptrb[_514_j] = safe_add( _515_cidx,(int)1 );
          }
        }
      }
    }
  }
  ((*_498_tsubi_)((int)0)) = _510_tsubj;
  ((*_499_tsubj_)((int)0)) = _509_tsubi;
  ((*_500_tval_)((int)0)) = _511_tval;
}
// End mosek.fusion.CommonTools.transposeTriplets

// Begin mosek.fusion.CommonTools.tripletSort
// Method mosek.fusion.CommonTools.tripletSort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:115:6-214:7
void mosek::fusion::CommonTools::tripletSort
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _516_subi,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _517_subj,
    std::shared_ptr< monty::ndarray< double,1 > > _518_val,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _519_tsubi_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _520_tsubj_,
    std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _521_tval_,
    int64_t _522_nelm,
    int32_t _523_dimi,
    int32_t _524_dimj) { mosek::fusion::p_CommonTools::tripletSort(_516_subi,_517_subj,_518_val,_519_tsubi_,_520_tsubj_,_521_tval_,_522_nelm,_523_dimi,_524_dimj); }
void mosek::fusion::p_CommonTools::tripletSort(std::shared_ptr< monty::ndarray< int32_t,1 > > _516_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _517_subj,std::shared_ptr< monty::ndarray< double,1 > > _518_val,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _519_tsubi_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _520_tsubj_,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< double,1 > >,1 > > _521_tval_,int64_t _522_nelm,int32_t _523_dimi,int32_t _524_dimj)
{
  _checked_ptr_1<int32_t> _ptr__516_subi(_516_subi ? _516_subi->raw() : nullptr, _516_subi ? _516_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__517_subj(_517_subj ? _517_subj->raw() : nullptr, _517_subj ? _517_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__518_val(_518_val ? _518_val->raw() : nullptr, _518_val ? _518_val->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _525_cols = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_522_nelm)));
  _checked_ptr_1<int32_t> _ptr__525_cols(_525_cols ? _525_cols->raw() : nullptr,_525_cols ? _525_cols->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _526_cptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _524_dimj,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__526_cptrb(_526_cptrb ? _526_cptrb->raw() : nullptr,_526_cptrb ? _526_cptrb->size(0) : 0);
  int64_t _9268_ = (int64_t)0;
  int64_t _9269_ = _522_nelm;
  for (int64_t _527_i = _9268_; _527_i < _9269_; ++_527_i)
  {
    {
      _ptr__526_cptrb[safe_add( _ptr__517_subj[_527_i],(int)2 )] = safe_add( _ptr__526_cptrb[safe_add( _ptr__517_subj[_527_i],(int)2 )],(int)1 );
    }
  }
  int32_t _9270_ = (int)1;
  int32_t _9271_ = _524_dimj;
  for (int32_t _528_i = _9270_; _528_i < _9271_; ++_528_i)
  {
    {
      _ptr__526_cptrb[safe_add( _528_i,(int)1 )] = safe_add( _ptr__526_cptrb[safe_add( _528_i,(int)1 )],_ptr__526_cptrb[_528_i] );
    }
  }
  int32_t _9272_ = (int)0;
  int32_t _9273_ = (int32_t)_522_nelm;
  for (int32_t _529_i = _9272_; _529_i < _9273_; ++_529_i)
  {
    {
      int32_t _530_cidx = _ptr__517_subj[_529_i];
      _ptr__525_cols[_ptr__526_cptrb[safe_add( _530_cidx,(int)1 )]] = _529_i;
      _ptr__526_cptrb[safe_add( _530_cidx,(int)1 )] = safe_add( _ptr__526_cptrb[safe_add( _530_cidx,(int)1 )],(int)1 );
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _531_rptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _523_dimi,(int)2 ))));
  _checked_ptr_1<int64_t> _ptr__531_rptrb(_531_rptrb ? _531_rptrb->raw() : nullptr,_531_rptrb ? _531_rptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _532_rows = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_522_nelm)));
  _checked_ptr_1<int32_t> _ptr__532_rows(_532_rows ? _532_rows->raw() : nullptr,_532_rows ? _532_rows->size(0) : 0);
  {
    int32_t _533_i = (int)0;
    for(;(_533_i < _522_nelm);++ _533_i)
    {
      {
        _ptr__531_rptrb[safe_add( _ptr__516_subi[_533_i],(int)2 )] = safe_add( _ptr__531_rptrb[safe_add( _ptr__516_subi[_533_i],(int)2 )],(int)1 );
      }
    }
  }
  {
    int32_t _534_i = (int)1;
    for(;(_534_i < _523_dimi);++ _534_i)
    {
      {
        _ptr__531_rptrb[safe_add( _534_i,(int)1 )] = safe_add( _ptr__531_rptrb[safe_add( _534_i,(int)1 )],_ptr__531_rptrb[_534_i] );
      }
    }
  }
  {
    int32_t _535_j = (int)0;
    for(;(_535_j < _524_dimj);++ _535_j)
    {
      {
        {
          int64_t _536_i = _ptr__526_cptrb[_535_j];
          for(;(_536_i < _ptr__526_cptrb[safe_add( _535_j,(int)1 )]);++ _536_i)
          {
            {
              int32_t _537_ridx = _ptr__516_subi[_ptr__525_cols[_536_i]];
              _ptr__532_rows[_ptr__531_rptrb[safe_add( _537_ridx,(int)1 )]] = _ptr__525_cols[_536_i];
              _ptr__531_rptrb[safe_add( _537_ridx,(int)1 )] = safe_add( _ptr__531_rptrb[safe_add( _537_ridx,(int)1 )],(int)1 );
            }
          }
        }
      }
    }
  }
  int32_t _538_nunique = (int)1;
  {
    int32_t _539_si = _ptr__516_subi[_ptr__532_rows[(int)0]];
    int32_t _540_sj = _ptr__517_subj[_ptr__532_rows[(int)0]];
    {
      int32_t _541_i = (int)1;
      for(;(_541_i < _522_nelm);++ _541_i)
      {
        {
          int32_t _542_idx = _ptr__532_rows[_541_i];
          if (((_539_si != _ptr__516_subi[_542_idx]) || (_540_sj != _ptr__517_subj[_542_idx])))
          {
            {
              _539_si = _ptr__516_subi[_542_idx];
              _540_sj = _ptr__517_subj[_542_idx];
              ++ _538_nunique;
            }
          }
          {}
        }
      }
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _543_tsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_538_nunique)));
  _checked_ptr_1<int32_t> _ptr__543_tsubi(_543_tsubi ? _543_tsubi->raw() : nullptr,_543_tsubi ? _543_tsubi->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _544_tsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_538_nunique)));
  _checked_ptr_1<int32_t> _ptr__544_tsubj(_544_tsubj ? _544_tsubj->raw() : nullptr,_544_tsubj ? _544_tsubj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _545_tval = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_538_nunique)));
  _checked_ptr_1<double> _ptr__545_tval(_545_tval ? _545_tval->raw() : nullptr,_545_tval ? _545_tval->size(0) : 0);
  {
    int32_t _546_idx = _ptr__532_rows[(int)0];
    _ptr__543_tsubi[(int)0] = _ptr__516_subi[_546_idx];
    _ptr__544_tsubj[(int)0] = _ptr__517_subj[_546_idx];
    _ptr__545_tval[(int)0] = _ptr__518_val[_546_idx];
  }
  int32_t _547_di = (int)1;
  int32_t _9274_ = (int)1;
  int64_t _9275_ = _522_nelm;
  for (int64_t _548_i = _9274_; _548_i < _9275_; ++_548_i)
  {
    {
      int32_t _549_idx = _ptr__532_rows[_548_i];
      if (((_ptr__543_tsubi[(_547_di - (int)1)] == _ptr__516_subi[_549_idx]) && (_ptr__544_tsubj[(_547_di - (int)1)] == _ptr__517_subj[_549_idx])))
      {
        {
          _ptr__545_tval[(_547_di - (int)1)] = (_ptr__545_tval[(_547_di - (int)1)] + _ptr__518_val[_549_idx]);
        }
      }
      else
      {
        {
          _ptr__543_tsubi[_547_di] = _ptr__516_subi[_549_idx];
          _ptr__544_tsubj[_547_di] = _ptr__517_subj[_549_idx];
          _ptr__545_tval[_547_di] = _ptr__518_val[_549_idx];
          ++ _547_di;
        }
      }
    }
  }
  ((*_519_tsubi_)((int)0)) = _543_tsubi;
  ((*_520_tsubj_)((int)0)) = _544_tsubj;
  ((*_521_tval_)((int)0)) = _545_tval;
}
// End mosek.fusion.CommonTools.tripletSort

// Begin mosek.fusion.CommonTools.argMSort
// Method mosek.fusion.CommonTools.argMSort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:67:6-87:7
void mosek::fusion::CommonTools::argMSort
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _550_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _551_vals) { mosek::fusion::p_CommonTools::argMSort(_550_idx,_551_vals); }
void mosek::fusion::p_CommonTools::argMSort(std::shared_ptr< monty::ndarray< int32_t,1 > > _550_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _551_vals)
{
  _checked_ptr_1<int32_t> _ptr__550_idx(_550_idx ? _550_idx->raw() : nullptr, _550_idx ? _550_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__551_vals(_551_vals ? _551_vals->raw() : nullptr, _551_vals ? _551_vals->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _552_temp = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)((_550_idx)->size(0)))));
  _checked_ptr_1<int32_t> _ptr__552_temp(_552_temp ? _552_temp->raw() : nullptr,_552_temp ? _552_temp->size(0) : 0);
  bool _553_alt = false;
  {
    int32_t _554_intvlen = (int)1;
    for(;(_554_intvlen < (int)((_550_idx)->size(0)));_554_intvlen *= (int)2)
    {
      {
        _553_alt = (!_553_alt);
        int32_t _9276_ = (int)0;
        int32_t _9277_ = (int)((_550_idx)->size(0));
        int32_t _9278_ = safe_mul( _554_intvlen,(int)2 );
        for (int32_t _555_i = _9276_; _555_i < _9277_; _555_i += _9278_)
        {
          {
            if (_553_alt)
            {
              {
                ::mosek::fusion::p_CommonTools::mergeInto(_550_idx,_552_temp,_551_vals,_555_i,safe_add( _555_i,_554_intvlen ),safe_add( _555_i,safe_mul( (int)2,_554_intvlen ) ));
              }
            }
            else
            {
              {
                ::mosek::fusion::p_CommonTools::mergeInto(_552_temp,_550_idx,_551_vals,_555_i,safe_add( _555_i,_554_intvlen ),safe_add( _555_i,safe_mul( (int)2,_554_intvlen ) ));
              }
            }
          }
        }
      }
    }
  }
  if (_553_alt)
  {
    {
      ::mosek::fusion::Utils::Tools::arraycopy(_552_temp,(int)0,_550_idx,(int)0,(int)((_550_idx)->size(0)));
    }
  }
  {}
}
// End mosek.fusion.CommonTools.argMSort

// Begin mosek.fusion.CommonTools.mergeInto
// Method mosek.fusion.CommonTools.mergeInto @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:29:6-54:7
void mosek::fusion::p_CommonTools::mergeInto(std::shared_ptr< monty::ndarray< int32_t,1 > > _556_src,std::shared_ptr< monty::ndarray< int32_t,1 > > _557_tgt,std::shared_ptr< monty::ndarray< int32_t,1 > > _558_vals,int32_t _559_si0,int32_t _560_si1_,int32_t _561_si2_)
{
  _checked_ptr_1<int32_t> _ptr__556_src(_556_src ? _556_src->raw() : nullptr, _556_src ? _556_src->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__557_tgt(_557_tgt ? _557_tgt->raw() : nullptr, _557_tgt ? _557_tgt->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__558_vals(_558_vals ? _558_vals->raw() : nullptr, _558_vals ? _558_vals->size(0) : 0);
  int32_t _9279_;
  bool _9280_ = (_561_si2_ > (int)((_556_src)->size(0)));
  if (_9280_)
  {
    _9279_ = (int)((_556_src)->size(0));
  }
  else
  {
    _9279_ = _561_si2_;
  }
  int32_t _562_si2 = _9279_;
  int32_t _9281_;
  bool _9282_ = (_560_si1_ > _562_si2);
  if (_9282_)
  {
    _9281_ = _562_si2;
  }
  else
  {
    _9281_ = _560_si1_;
  }
  int32_t _563_si1 = _9281_;
  if ((_563_si1 == _562_si2))
  {
    {
      ::mosek::fusion::Utils::Tools::arraycopy(_556_src,_559_si0,_557_tgt,_559_si0,(_563_si1 - _559_si0));
    }
  }
  else
  {
    {
      int32_t _564_i0 = _559_si0;
      int32_t _565_i1 = _563_si1;
      int32_t _566_i;
      {
        _566_i = _559_si0;
        for(;((_564_i0 < _563_si1) && (_565_i1 < _562_si2));++ _566_i)
        {
          {
            if ((_ptr__558_vals[_ptr__556_src[_564_i0]] < _ptr__558_vals[_ptr__556_src[_565_i1]]))
            {
              {
                _ptr__557_tgt[_566_i] = _ptr__556_src[_564_i0];
                ++ _564_i0;
              }
            }
            else
            {
              {
                _ptr__557_tgt[_566_i] = _ptr__556_src[_565_i1];
                ++ _565_i1;
              }
            }
          }
        }
      }
      while ( (_564_i0 < _563_si1) )
      {
        {
          _ptr__557_tgt[_566_i] = _ptr__556_src[_564_i0];
          ++ _564_i0;
          ++ _566_i;
        }
      }
      while ( (_565_i1 < _562_si2) )
      {
        {
          _ptr__557_tgt[_566_i] = _ptr__556_src[_565_i1];
          ++ _565_i1;
          ++ _566_i;
        }
      }
    }
  }
}
// End mosek.fusion.CommonTools.mergeInto

// Begin mosek.fusion.CommonTools.argQsort
// Method mosek.fusion.CommonTools.argQsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:16:6-22:7
void mosek::fusion::CommonTools::argQsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _567_idx,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _568_vals1,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _569_vals2,
    int64_t _570_first,
    int64_t _571_last) { mosek::fusion::p_CommonTools::argQsort(_567_idx,_568_vals1,_569_vals2,_570_first,_571_last); }
void mosek::fusion::p_CommonTools::argQsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _567_idx,std::shared_ptr< monty::ndarray< int64_t,1 > > _568_vals1,std::shared_ptr< monty::ndarray< int64_t,1 > > _569_vals2,int64_t _570_first,int64_t _571_last)
{
  _checked_ptr_1<int64_t> _ptr__567_idx(_567_idx ? _567_idx->raw() : nullptr, _567_idx ? _567_idx->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__568_vals1(_568_vals1 ? _568_vals1->raw() : nullptr, _568_vals1 ? _568_vals1->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__569_vals2(_569_vals2 ? _569_vals2->raw() : nullptr, _569_vals2 ? _569_vals2->size(0) : 0);
  if ((NULL == _569_vals2.get()))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_567_idx,_568_vals1,_570_first,_571_last);
    }
  }
  else
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_567_idx,_568_vals1,_569_vals2,_570_first,_571_last);
    }
  }
}
// End mosek.fusion.CommonTools.argQsort

// Begin mosek.fusion.CommonTools.argQsort
// Method mosek.fusion.CommonTools.argQsort @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/CommonUtil.mbi:8:6-14:7
void mosek::fusion::CommonTools::argQsort
  ( std::shared_ptr< monty::ndarray< int64_t,1 > > _572_idx,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _573_vals1,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _574_vals2,
    int64_t _575_first,
    int64_t _576_last) { mosek::fusion::p_CommonTools::argQsort(_572_idx,_573_vals1,_574_vals2,_575_first,_576_last); }
void mosek::fusion::p_CommonTools::argQsort(std::shared_ptr< monty::ndarray< int64_t,1 > > _572_idx,std::shared_ptr< monty::ndarray< int32_t,1 > > _573_vals1,std::shared_ptr< monty::ndarray< int32_t,1 > > _574_vals2,int64_t _575_first,int64_t _576_last)
{
  _checked_ptr_1<int64_t> _ptr__572_idx(_572_idx ? _572_idx->raw() : nullptr, _572_idx ? _572_idx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__573_vals1(_573_vals1 ? _573_vals1->raw() : nullptr, _573_vals1 ? _573_vals1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__574_vals2(_574_vals2 ? _574_vals2->raw() : nullptr, _574_vals2 ? _574_vals2->size(0) : 0);
  if ((NULL == _574_vals2.get()))
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_572_idx,_573_vals1,_575_first,_576_last);
    }
  }
  else
  {
    {
      ::mosek::fusion::Utils::Tools::argsort(_572_idx,_573_vals1,_574_vals2,_575_first,_576_last);
    }
  }
}
// End mosek.fusion.CommonTools.argQsort

void mosek::fusion::p_CommonTools::destroy()
{
}
void mosek::fusion::CommonTools::destroy() { mosek::fusion::p_CommonTools::_get_impl(this)->destroy(); }
// } class CommonTools
// class SolutionStruct { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5522:4-5857:5
mosek::fusion::SolutionStruct::SolutionStruct(mosek::fusion::p_SolutionStruct *_impl) : _impl(_impl) { /*std::cout << "SolutionStruct() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::SolutionStruct::~SolutionStruct() { delete _impl; _impl = nullptr; }
mosek::fusion::p_SolutionStruct::p_SolutionStruct
  (::mosek::fusion::SolutionStruct * _pubthis) :     _pubthis(_pubthis)
{}
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_accy() { return _impl->accy; }
void mosek::fusion::SolutionStruct::set_accy(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->accy = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_accx() { return _impl->accx; }
void mosek::fusion::SolutionStruct::set_accx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->accx = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::SolutionStruct::get_accptr() { return _impl->accptr; }
void mosek::fusion::SolutionStruct::set_accptr(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->accptr = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_yx() { return _impl->yx; }
void mosek::fusion::SolutionStruct::set_yx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->yx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_sux() { return _impl->sux; }
void mosek::fusion::SolutionStruct::set_sux(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->sux = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_slx() { return _impl->slx; }
void mosek::fusion::SolutionStruct::set_slx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->slx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_bars() { return _impl->bars; }
void mosek::fusion::SolutionStruct::set_bars(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->bars = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_barx() { return _impl->barx; }
void mosek::fusion::SolutionStruct::set_barx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->barx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_y() { return _impl->y; }
void mosek::fusion::SolutionStruct::set_y(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->y = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_suc() { return _impl->suc; }
void mosek::fusion::SolutionStruct::set_suc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->suc = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_slc() { return _impl->slc; }
void mosek::fusion::SolutionStruct::set_slc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->slc = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_xx() { return _impl->xx; }
void mosek::fusion::SolutionStruct::set_xx(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->xx = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::SolutionStruct::get_xc() { return _impl->xc; }
void mosek::fusion::SolutionStruct::set_xc(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->xc = _val; }
double mosek::fusion::SolutionStruct::get_dobj() { return _impl->dobj; }
void mosek::fusion::SolutionStruct::set_dobj(double _val) { _impl->dobj = _val; }
double mosek::fusion::SolutionStruct::get_pobj() { return _impl->pobj; }
void mosek::fusion::SolutionStruct::set_pobj(double _val) { _impl->pobj = _val; }
mosek::fusion::ProblemStatus mosek::fusion::SolutionStruct::get_probstatus() { return _impl->probstatus; }
void mosek::fusion::SolutionStruct::set_probstatus(mosek::fusion::ProblemStatus _val) { _impl->probstatus = _val; }
mosek::fusion::SolutionStatus mosek::fusion::SolutionStruct::get_dstatus() { return _impl->dstatus; }
void mosek::fusion::SolutionStruct::set_dstatus(mosek::fusion::SolutionStatus _val) { _impl->dstatus = _val; }
mosek::fusion::SolutionStatus mosek::fusion::SolutionStruct::get_pstatus() { return _impl->pstatus; }
void mosek::fusion::SolutionStruct::set_pstatus(mosek::fusion::SolutionStatus _val) { _impl->pstatus = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numbarvar() { return _impl->sol_numbarvar; }
void mosek::fusion::SolutionStruct::set_sol_numbarvar(int32_t _val) { _impl->sol_numbarvar = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numaccelm() { return _impl->sol_numaccelm; }
void mosek::fusion::SolutionStruct::set_sol_numaccelm(int32_t _val) { _impl->sol_numaccelm = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numacc() { return _impl->sol_numacc; }
void mosek::fusion::SolutionStruct::set_sol_numacc(int32_t _val) { _impl->sol_numacc = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numvar() { return _impl->sol_numvar; }
void mosek::fusion::SolutionStruct::set_sol_numvar(int32_t _val) { _impl->sol_numvar = _val; }
int32_t mosek::fusion::SolutionStruct::get_sol_numcon() { return _impl->sol_numcon; }
void mosek::fusion::SolutionStruct::set_sol_numcon(int32_t _val) { _impl->sol_numcon = _val; }
// mosek.fusion.SolutionStruct.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5797:6-5856:7
mosek::fusion::SolutionStruct::t mosek::fusion::p_SolutionStruct::_new_SolutionStruct
  (int32_t _577_numvar,
    int32_t _578_numcon,
    int32_t _579_numbarvar,
    int32_t _580_numacc,
    int32_t _581_numaccelm)
{ return new mosek::fusion::SolutionStruct(_577_numvar,_578_numcon,_579_numbarvar,_580_numacc,_581_numaccelm); }
mosek::fusion::SolutionStruct::SolutionStruct
  (int32_t _577_numvar,
    int32_t _578_numcon,
    int32_t _579_numbarvar,
    int32_t _580_numacc,
    int32_t _581_numaccelm) : 
   SolutionStruct(new mosek::fusion::p_SolutionStruct(this))
{ mosek::fusion::p_SolutionStruct::_get_impl(this)->_initialize(_577_numvar,_578_numcon,_579_numbarvar,_580_numacc,_581_numaccelm); }
void mosek::fusion::p_SolutionStruct::_initialize  (int32_t _577_numvar,
    int32_t _578_numcon,
    int32_t _579_numbarvar,
    int32_t _580_numacc,
    int32_t _581_numaccelm)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    pobj = 0.0;
    dobj = 0.0;
    pstatus = ::mosek::fusion::SolutionStatus::Unknown;
    dstatus = ::mosek::fusion::SolutionStatus::Unknown;
    probstatus = ::mosek::fusion::ProblemStatus::Unknown;
    sol_numvar = _577_numvar;
    sol_numcon = _578_numcon;
    sol_numbarvar = _579_numbarvar;
    sol_numacc = _580_numacc;
    sol_numaccelm = _581_numaccelm;
    slx = nullptr;
    sux = nullptr;
    xx = nullptr;
    yx = nullptr;
    slc = nullptr;
    suc = nullptr;
    y = nullptr;
    xc = nullptr;
    barx = nullptr;
    bars = nullptr;
    accx = nullptr;
    accy = nullptr;
    accptr = nullptr;
    if ((_577_numvar > (int)0))
    {
      {
        slx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_577_numvar)));
        sux = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_577_numvar)));
        xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_577_numvar)));
        yx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_577_numvar)));
      }
    }
    {}
    if ((_579_numbarvar > (int)0))
    {
      {
        barx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_579_numbarvar)));
        bars = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_579_numbarvar)));
      }
    }
    {}
    if ((_578_numcon > (int)0))
    {
      {
        slc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_578_numcon)));
        suc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_578_numcon)));
        y = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_578_numcon)));
        xc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_578_numcon)));
      }
    }
    {}
    if ((_580_numacc > (int)0))
    {
      {
        accx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_581_numaccelm)));
        accy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_581_numaccelm)));
        accptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _580_numacc,(int)1 ))));
      }
    }
    {}
  }
}
// mosek.fusion.SolutionStruct.ctor

// mosek.fusion.SolutionStruct.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5764:6-5794:7
mosek::fusion::SolutionStruct::t mosek::fusion::p_SolutionStruct::_new_SolutionStruct
  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _582_that)
{ return new mosek::fusion::SolutionStruct(_582_that); }
mosek::fusion::SolutionStruct::SolutionStruct
  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _582_that) : 
   SolutionStruct(new mosek::fusion::p_SolutionStruct(this))
{ mosek::fusion::p_SolutionStruct::_get_impl(this)->_initialize(_582_that); }
void mosek::fusion::p_SolutionStruct::_initialize  (monty::rc_ptr< ::mosek::fusion::SolutionStruct > _582_that)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    sol_numcon = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sol_numcon;
    sol_numvar = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sol_numvar;
    sol_numbarvar = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sol_numbarvar;
    sol_numacc = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sol_numacc;
    sol_numaccelm = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sol_numaccelm;
    pstatus = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->pstatus;
    dstatus = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->dstatus;
    probstatus = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->probstatus;
    pobj = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->pobj;
    dobj = mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->dobj;
    std::shared_ptr< monty::ndarray< double,1 > > _9283_;
    bool _9284_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->xc.get());
    if (_9284_)
    {
      _9283_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->xc);
    }
    else
    {
      _9283_ = nullptr;
    }
    xc = _9283_;
    std::shared_ptr< monty::ndarray< double,1 > > _9285_;
    bool _9286_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->xx.get());
    if (_9286_)
    {
      _9285_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->xx);
    }
    else
    {
      _9285_ = nullptr;
    }
    xx = _9285_;
    std::shared_ptr< monty::ndarray< double,1 > > _9287_;
    bool _9288_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->yx.get());
    if (_9288_)
    {
      _9287_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->yx);
    }
    else
    {
      _9287_ = nullptr;
    }
    yx = _9287_;
    std::shared_ptr< monty::ndarray< double,1 > > _9289_;
    bool _9290_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->slc.get());
    if (_9290_)
    {
      _9289_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->slc);
    }
    else
    {
      _9289_ = nullptr;
    }
    slc = _9289_;
    std::shared_ptr< monty::ndarray< double,1 > > _9291_;
    bool _9292_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->suc.get());
    if (_9292_)
    {
      _9291_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->suc);
    }
    else
    {
      _9291_ = nullptr;
    }
    suc = _9291_;
    std::shared_ptr< monty::ndarray< double,1 > > _9293_;
    bool _9294_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->y.get());
    if (_9294_)
    {
      _9293_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->y);
    }
    else
    {
      _9293_ = nullptr;
    }
    y = _9293_;
    std::shared_ptr< monty::ndarray< double,1 > > _9295_;
    bool _9296_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->barx.get());
    if (_9296_)
    {
      _9295_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->barx);
    }
    else
    {
      _9295_ = nullptr;
    }
    barx = _9295_;
    std::shared_ptr< monty::ndarray< double,1 > > _9297_;
    bool _9298_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->bars.get());
    if (_9298_)
    {
      _9297_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->bars);
    }
    else
    {
      _9297_ = nullptr;
    }
    bars = _9297_;
    std::shared_ptr< monty::ndarray< double,1 > > _9299_;
    bool _9300_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->slx.get());
    if (_9300_)
    {
      _9299_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->slx);
    }
    else
    {
      _9299_ = nullptr;
    }
    slx = _9299_;
    std::shared_ptr< monty::ndarray< double,1 > > _9301_;
    bool _9302_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sux.get());
    if (_9302_)
    {
      _9301_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->sux);
    }
    else
    {
      _9301_ = nullptr;
    }
    sux = _9301_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9303_;
    bool _9304_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accptr.get());
    if (_9304_)
    {
      _9303_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accptr);
    }
    else
    {
      _9303_ = nullptr;
    }
    accptr = _9303_;
    std::shared_ptr< monty::ndarray< double,1 > > _9305_;
    bool _9306_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accx.get());
    if (_9306_)
    {
      _9305_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accx);
    }
    else
    {
      _9305_ = nullptr;
    }
    accx = _9305_;
    std::shared_ptr< monty::ndarray< double,1 > > _9307_;
    bool _9308_ = (NULL != mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accy.get());
    if (_9308_)
    {
      _9307_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_582_that)->accy);
    }
    else
    {
      _9307_ = nullptr;
    }
    accy = _9307_;
  }
}
// mosek.fusion.SolutionStruct.ctor

// Begin mosek.fusion.SolutionStruct.clone
// Method mosek.fusion.SolutionStruct.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5795:6-79
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::SolutionStruct :: __mosek_2fusion_2SolutionStruct__clone() { return mosek::fusion::p_SolutionStruct::_get_impl(this)->__mosek_2fusion_2SolutionStruct__clone(); }
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::SolutionStruct::clone() { return __mosek_2fusion_2SolutionStruct__clone(); }
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_SolutionStruct::__mosek_2fusion_2SolutionStruct__clone()
{
  return ::mosek::fusion::p_SolutionStruct::_new_SolutionStruct(::mosek::fusion::SolutionStruct::t(_pubthis));
}
// End mosek.fusion.SolutionStruct.clone

// Begin mosek.fusion.SolutionStruct.resize
// Method mosek.fusion.SolutionStruct.resize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5589:6-5759:7
void mosek::fusion::SolutionStruct :: resize(int32_t _583_numvar,int32_t _584_numcon,int32_t _585_numbarvar,int32_t _586_numacc,int32_t _587_numaccelm) { mosek::fusion::p_SolutionStruct::_get_impl(this)->resize(_583_numvar,_584_numcon,_585_numbarvar,_586_numacc,_587_numaccelm); }
void mosek::fusion::p_SolutionStruct::resize(int32_t _583_numvar,int32_t _584_numcon,int32_t _585_numbarvar,int32_t _586_numacc,int32_t _587_numaccelm)
{
  if ((_586_numacc <= (int)0))
  {
    {
      sol_numacc = (int)0;
    }
  }
  else if((_586_numacc <= ((int)((accptr)->size(0)) - (int)1)))
  {
    {
      sol_numacc = _586_numacc;
    }
  }
  else
  {
    {
      int32_t _588_newnum = safe_mul( ((int)((accptr)->size(0)) - (int)1),(int)2 );
      if ((_588_newnum < _586_numacc))
      {
        {
          _588_newnum = _586_numacc;
        }
      }
      {}
      std::shared_ptr< monty::ndarray< int32_t,1 > > _589_new_accptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _588_newnum,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__589_new_accptr(_589_new_accptr ? _589_new_accptr->raw() : nullptr,_589_new_accptr ? _589_new_accptr->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(accptr,(int)0,_589_new_accptr,(int)0,safe_add( _586_numacc,(int)1 ));
      accptr = _589_new_accptr;
      sol_numacc = _586_numacc;
    }
  }
  if ((_587_numaccelm <= (int)0))
  {
    {
      sol_numaccelm = (int)0;
    }
  }
  else if((_587_numaccelm > (int)((accx)->size(0))))
  {
    {
      sol_numaccelm = _587_numaccelm;
    }
  }
  else
  {
    {
      int32_t _590_newnum = safe_mul( (int)((accx)->size(0)),(int)2 );
      if ((_590_newnum < _587_numaccelm))
      {
        {
          _590_newnum = _587_numaccelm;
        }
      }
      {}
      std::shared_ptr< monty::ndarray< double,1 > > _591_new_accx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_590_newnum)));
      _checked_ptr_1<double> _ptr__591_new_accx(_591_new_accx ? _591_new_accx->raw() : nullptr,_591_new_accx ? _591_new_accx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _592_new_accy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_590_newnum)));
      _checked_ptr_1<double> _ptr__592_new_accy(_592_new_accy ? _592_new_accy->raw() : nullptr,_592_new_accy ? _592_new_accy->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(accx,(int)0,_591_new_accx,(int)0,_587_numaccelm);
      ::mosek::fusion::Utils::Tools::arraycopy(accy,(int)0,_592_new_accy,(int)0,_587_numaccelm);
      accx = _591_new_accx;
      accy = _592_new_accy;
      sol_numaccelm = _587_numaccelm;
    }
  }
  if ((_585_numbarvar < (int)0))
  {
    {
    }
  }
  else if((_585_numbarvar <= sol_numbarvar))
  {
    {
      sol_numbarvar = _585_numbarvar;
    }
  }
  else if((_585_numbarvar <= (int)((barx)->size(0))))
  {
    {
    }
  }
  else
  {
    {
      int32_t _593_newsize;
      if ((safe_mul( sol_numbarvar,(int)2 ) > safe_add( _585_numbarvar,(int)100 )))
      {
        {
          _593_newsize = safe_mul( sol_numbarvar,(int)2 );
        }
      }
      else
      {
        {
          _593_newsize = safe_add( _585_numbarvar,(int)100 );
        }
      }
      barx = ::mosek::fusion::p_CommonTools::resize(barx,_593_newsize);
      bars = ::mosek::fusion::p_CommonTools::resize(bars,_593_newsize);
      sol_numbarvar = _593_newsize;
    }
  }
  if ((_583_numvar < (int)0))
  {
    {
    }
  }
  else if((_583_numvar <= sol_numvar))
  {
    {
      sol_numvar = _583_numvar;
    }
  }
  else if((_583_numvar <= (int)((xx)->size(0))))
  {
    {
      int32_t _9309_ = sol_numvar;
      int32_t _9310_ = _583_numvar;
      for (int32_t _594_i = _9309_; _594_i < _9310_; ++_594_i)
      {
        {
          ((*slx)(_594_i)) = 0.0;
          ((*sux)(_594_i)) = 0.0;
          ((*xx)(_594_i)) = 0.0;
          ((*yx)(_594_i)) = 0.0;
        }
      }
      sol_numvar = _583_numvar;
    }
  }
  else
  {
    {
      int32_t _595_newsize;
      if ((safe_mul( sol_numvar,(int)2 ) > safe_add( _583_numvar,(int)100 )))
      {
        {
          _595_newsize = safe_mul( sol_numvar,(int)2 );
        }
      }
      else
      {
        {
          _595_newsize = safe_add( _583_numvar,(int)100 );
        }
      }
      std::shared_ptr< monty::ndarray< double,1 > > _596_newslx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_595_newsize)));
      _checked_ptr_1<double> _ptr__596_newslx(_596_newslx ? _596_newslx->raw() : nullptr,_596_newslx ? _596_newslx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(slx,(int)0,_596_newslx,(int)0,(int)((slx)->size(0)));
      slx = _596_newslx;
      std::shared_ptr< monty::ndarray< double,1 > > _597_newsux = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_595_newsize)));
      _checked_ptr_1<double> _ptr__597_newsux(_597_newsux ? _597_newsux->raw() : nullptr,_597_newsux ? _597_newsux->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(sux,(int)0,_597_newsux,(int)0,(int)((sux)->size(0)));
      sux = _597_newsux;
      std::shared_ptr< monty::ndarray< double,1 > > _598_newxx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_595_newsize)));
      _checked_ptr_1<double> _ptr__598_newxx(_598_newxx ? _598_newxx->raw() : nullptr,_598_newxx ? _598_newxx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(xx,(int)0,_598_newxx,(int)0,(int)((xx)->size(0)));
      xx = _598_newxx;
      std::shared_ptr< monty::ndarray< double,1 > > _599_newyx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_595_newsize)));
      _checked_ptr_1<double> _ptr__599_newyx(_599_newyx ? _599_newyx->raw() : nullptr,_599_newyx ? _599_newyx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(yx,(int)0,_599_newyx,(int)0,(int)((yx)->size(0)));
      yx = _599_newyx;
      {
        int32_t _600_i = sol_numvar;
        for(;(_600_i < _583_numvar);++ _600_i)
        {
          {
            ((*slx)(_600_i)) = 0.0;
            ((*sux)(_600_i)) = 0.0;
            ((*xx)(_600_i)) = 0.0;
          }
        }
      }
      sol_numvar = _583_numvar;
    }
  }
  if ((_584_numcon < (int)0))
  {
    {
    }
  }
  else if((_584_numcon <= sol_numcon))
  {
    {
      sol_numcon = _584_numcon;
    }
  }
  else if((_584_numcon <= (int)((xx)->size(0))))
  {
    {
      int32_t _9311_ = sol_numcon;
      int32_t _9312_ = _584_numcon;
      for (int32_t _601_i = _9311_; _601_i < _9312_; ++_601_i)
      {
        {
          ((*slc)(_601_i)) = 0.0;
          ((*suc)(_601_i)) = 0.0;
          ((*xc)(_601_i)) = 0.0;
          ((*y)(_601_i)) = 0.0;
        }
      }
      sol_numcon = _584_numcon;
    }
  }
  else
  {
    {
      int32_t _602_newsize;
      if ((safe_mul( sol_numcon,(int)2 ) > safe_add( _584_numcon,(int)100 )))
      {
        {
          _602_newsize = safe_mul( sol_numcon,(int)2 );
        }
      }
      else
      {
        {
          _602_newsize = safe_add( _584_numcon,(int)100 );
        }
      }
      std::shared_ptr< monty::ndarray< double,1 > > _603_newslc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_602_newsize)));
      _checked_ptr_1<double> _ptr__603_newslc(_603_newslc ? _603_newslc->raw() : nullptr,_603_newslc ? _603_newslc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(slc,(int)0,_603_newslc,(int)0,(int)((slc)->size(0)));
      slc = _603_newslc;
      std::shared_ptr< monty::ndarray< double,1 > > _604_newsuc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_602_newsize)));
      _checked_ptr_1<double> _ptr__604_newsuc(_604_newsuc ? _604_newsuc->raw() : nullptr,_604_newsuc ? _604_newsuc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(suc,(int)0,_604_newsuc,(int)0,(int)((suc)->size(0)));
      suc = _604_newsuc;
      std::shared_ptr< monty::ndarray< double,1 > > _605_newxc = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_602_newsize)));
      _checked_ptr_1<double> _ptr__605_newxc(_605_newxc ? _605_newxc->raw() : nullptr,_605_newxc ? _605_newxc->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(xc,(int)0,_605_newxc,(int)0,(int)((xc)->size(0)));
      xc = _605_newxc;
      std::shared_ptr< monty::ndarray< double,1 > > _606_newy = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_602_newsize)));
      _checked_ptr_1<double> _ptr__606_newy(_606_newy ? _606_newy->raw() : nullptr,_606_newy ? _606_newy->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(y,(int)0,_606_newy,(int)0,(int)((y)->size(0)));
      y = _606_newy;
      int32_t _9313_ = sol_numcon;
      int32_t _9314_ = _584_numcon;
      for (int32_t _607_i = _9313_; _607_i < _9314_; ++_607_i)
      {
        {
          ((*slc)(_607_i)) = 0.0;
          ((*suc)(_607_i)) = 0.0;
          ((*xc)(_607_i)) = 0.0;
          ((*y)(_607_i)) = 0.0;
        }
      }
      sol_numcon = _584_numcon;
    }
  }
}
// End mosek.fusion.SolutionStruct.resize

// Begin mosek.fusion.SolutionStruct.isDualAcceptable
// Method mosek.fusion.SolutionStruct.isDualAcceptable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5584:6-5587:7
bool mosek::fusion::SolutionStruct :: isDualAcceptable(mosek::fusion::AccSolutionStatus _608_acceptable_sol) { return mosek::fusion::p_SolutionStruct::_get_impl(this)->isDualAcceptable(_608_acceptable_sol); }
bool mosek::fusion::p_SolutionStruct::isDualAcceptable(mosek::fusion::AccSolutionStatus _608_acceptable_sol)
{
  return isAcceptable(dstatus,_608_acceptable_sol);
}
// End mosek.fusion.SolutionStruct.isDualAcceptable

// Begin mosek.fusion.SolutionStruct.isPrimalAcceptable
// Method mosek.fusion.SolutionStruct.isPrimalAcceptable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5579:6-5582:7
bool mosek::fusion::SolutionStruct :: isPrimalAcceptable(mosek::fusion::AccSolutionStatus _609_acceptable_sol) { return mosek::fusion::p_SolutionStruct::_get_impl(this)->isPrimalAcceptable(_609_acceptable_sol); }
bool mosek::fusion::p_SolutionStruct::isPrimalAcceptable(mosek::fusion::AccSolutionStatus _609_acceptable_sol)
{
  return isAcceptable(pstatus,_609_acceptable_sol);
}
// End mosek.fusion.SolutionStruct.isPrimalAcceptable

// Begin mosek.fusion.SolutionStruct.isAcceptable
// Method mosek.fusion.SolutionStruct.isAcceptable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5553:6-5577:7
bool mosek::fusion::p_SolutionStruct::isAcceptable(mosek::fusion::SolutionStatus _610_stat,mosek::fusion::AccSolutionStatus _611_accstat)
{
  if ((_611_accstat == ::mosek::fusion::AccSolutionStatus::Anything))
  {
    {
      return true;
    }
  }
  else if((_611_accstat == ::mosek::fusion::AccSolutionStatus::Optimal))
  {
    {
      return (_610_stat == ::mosek::fusion::SolutionStatus::Optimal);
    }
  }
  else if((_611_accstat == ::mosek::fusion::AccSolutionStatus::Feasible))
  {
    {
      return ((_610_stat == ::mosek::fusion::SolutionStatus::Optimal) || (_610_stat == ::mosek::fusion::SolutionStatus::Feasible));
    }
  }
  else if((_611_accstat == ::mosek::fusion::AccSolutionStatus::Certificate))
  {
    {
      return (_610_stat == ::mosek::fusion::SolutionStatus::Certificate);
    }
  }
  {}
  return false;
}
// End mosek.fusion.SolutionStruct.isAcceptable

void mosek::fusion::p_SolutionStruct::destroy()
{
  accy.reset();
  accx.reset();
  accptr.reset();
  yx.reset();
  sux.reset();
  slx.reset();
  bars.reset();
  barx.reset();
  y.reset();
  suc.reset();
  slc.reset();
  xx.reset();
  xc.reset();
}
void mosek::fusion::SolutionStruct::destroy() { mosek::fusion::p_SolutionStruct::_get_impl(this)->destroy(); }
// } class SolutionStruct
// class RowBlockManager { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5107:4-5517:5
mosek::fusion::RowBlockManager::RowBlockManager(mosek::fusion::p_RowBlockManager *_impl) : _impl(_impl) { /*std::cout << "RowBlockManager() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::RowBlockManager::~RowBlockManager() { delete _impl; _impl = nullptr; }
mosek::fusion::p_RowBlockManager::p_RowBlockManager
  (::mosek::fusion::RowBlockManager * _pubthis) :     _pubthis(_pubthis)
{}
int32_t mosek::fusion::RowBlockManager::get_varidx_used() { return _impl->varidx_used; }
void mosek::fusion::RowBlockManager::set_varidx_used(int32_t _val) { _impl->varidx_used = _val; }
int32_t mosek::fusion::RowBlockManager::get_code_used() { return _impl->code_used; }
void mosek::fusion::RowBlockManager::set_code_used(int32_t _val) { _impl->code_used = _val; }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::RowBlockManager::get_cconst() { return _impl->cconst; }
void mosek::fusion::RowBlockManager::set_cconst(std::shared_ptr< monty::ndarray< double,1 > > _val) { _impl->cconst = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_code() { return _impl->code; }
void mosek::fusion::RowBlockManager::set_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->code = _val; }
int32_t mosek::fusion::RowBlockManager::get_first_free_codeitem() { return _impl->first_free_codeitem; }
void mosek::fusion::RowBlockManager::set_first_free_codeitem(int32_t _val) { _impl->first_free_codeitem = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_param_code_sizes() { return _impl->param_code_sizes; }
void mosek::fusion::RowBlockManager::set_param_code_sizes(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->param_code_sizes = _val; }
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::RowBlockManager::get_param_varidx() { return _impl->param_varidx; }
void mosek::fusion::RowBlockManager::set_param_varidx(std::shared_ptr< monty::ndarray< int64_t,1 > > _val) { _impl->param_varidx = _val; }
int32_t mosek::fusion::RowBlockManager::get_first_free_entry() { return _impl->first_free_entry; }
void mosek::fusion::RowBlockManager::set_first_free_entry(int32_t _val) { _impl->first_free_entry = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_code_ptr() { return _impl->row_code_ptr; }
void mosek::fusion::RowBlockManager::set_row_code_ptr(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_code_ptr = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_param_ptre() { return _impl->row_param_ptre; }
void mosek::fusion::RowBlockManager::set_row_param_ptre(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_param_ptre = _val; }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::RowBlockManager::get_row_param_ptrb() { return _impl->row_param_ptrb; }
void mosek::fusion::RowBlockManager::set_row_param_ptrb(std::shared_ptr< monty::ndarray< int32_t,1 > > _val) { _impl->row_param_ptrb = _val; }
monty::rc_ptr< ::mosek::fusion::LinkedBlocks > mosek::fusion::RowBlockManager::get_blocks() { return _impl->blocks; }
void mosek::fusion::RowBlockManager::set_blocks(monty::rc_ptr< ::mosek::fusion::LinkedBlocks > _val) { _impl->blocks = _val; }
// mosek.fusion.RowBlockManager.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5170:6-5186:7
mosek::fusion::RowBlockManager::t mosek::fusion::p_RowBlockManager::_new_RowBlockManager
  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _612_that)
{ return new mosek::fusion::RowBlockManager(_612_that); }
mosek::fusion::RowBlockManager::RowBlockManager
  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _612_that) : 
   RowBlockManager(new mosek::fusion::p_RowBlockManager(this))
{ mosek::fusion::p_RowBlockManager::_get_impl(this)->_initialize(_612_that); }
void mosek::fusion::p_RowBlockManager::_initialize  (monty::rc_ptr< ::mosek::fusion::RowBlockManager > _612_that)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    blocks = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->blocks);
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9315_;
    bool _9316_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_param_ptrb.get());
    if (_9316_)
    {
      _9315_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_param_ptrb);
    }
    else
    {
      _9315_ = nullptr;
    }
    row_param_ptrb = _9315_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9317_;
    bool _9318_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_param_ptre.get());
    if (_9318_)
    {
      _9317_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_param_ptre);
    }
    else
    {
      _9317_ = nullptr;
    }
    row_param_ptre = _9317_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9319_;
    bool _9320_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_code_ptr.get());
    if (_9320_)
    {
      _9319_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->row_code_ptr);
    }
    else
    {
      _9319_ = nullptr;
    }
    row_code_ptr = _9319_;
    first_free_entry = mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->first_free_entry;
    std::shared_ptr< monty::ndarray< int64_t,1 > > _9321_;
    bool _9322_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->param_varidx.get());
    if (_9322_)
    {
      _9321_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->param_varidx);
    }
    else
    {
      _9321_ = nullptr;
    }
    param_varidx = _9321_;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9323_;
    bool _9324_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->param_code_sizes.get());
    if (_9324_)
    {
      _9323_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->param_code_sizes);
    }
    else
    {
      _9323_ = nullptr;
    }
    param_code_sizes = _9323_;
    first_free_codeitem = mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->first_free_codeitem;
    std::shared_ptr< monty::ndarray< int32_t,1 > > _9325_;
    bool _9326_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->code.get());
    if (_9326_)
    {
      _9325_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->code);
    }
    else
    {
      _9325_ = nullptr;
    }
    code = _9325_;
    std::shared_ptr< monty::ndarray< double,1 > > _9327_;
    bool _9328_ = (NULL != mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->cconst.get());
    if (_9328_)
    {
      _9327_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->cconst);
    }
    else
    {
      _9327_ = nullptr;
    }
    cconst = _9327_;
    code_used = mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->code_used;
    varidx_used = mosek::fusion::p_RowBlockManager::_get_impl(_612_that)->varidx_used;
  }
}
// mosek.fusion.RowBlockManager.ctor

// mosek.fusion.RowBlockManager.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5154:6-5168:7
mosek::fusion::RowBlockManager::t mosek::fusion::p_RowBlockManager::_new_RowBlockManager
  ()
{ return new mosek::fusion::RowBlockManager(); }
mosek::fusion::RowBlockManager::RowBlockManager
  () : 
   RowBlockManager(new mosek::fusion::p_RowBlockManager(this))
{ mosek::fusion::p_RowBlockManager::_get_impl(this)->_initialize(); }
void mosek::fusion::p_RowBlockManager::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    blocks = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    row_param_ptrb = nullptr;
    row_param_ptre = nullptr;
    row_code_ptr = nullptr;
    first_free_entry = (int)0;
    param_varidx = nullptr;
    param_code_sizes = nullptr;
    first_free_codeitem = (int)0;
    code = nullptr;
    cconst = nullptr;
    code_used = (int)0;
    varidx_used = (int)0;
  }
}
// mosek.fusion.RowBlockManager.ctor

// Begin mosek.fusion.RowBlockManager.num_parameterized
// Method mosek.fusion.RowBlockManager.num_parameterized @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5516:6-69
int32_t mosek::fusion::RowBlockManager :: num_parameterized() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->num_parameterized(); }
int32_t mosek::fusion::p_RowBlockManager::num_parameterized()
{
  return varidx_used;
}
// End mosek.fusion.RowBlockManager.num_parameterized

// Begin mosek.fusion.RowBlockManager.is_parameterized
// Method mosek.fusion.RowBlockManager.is_parameterized @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5515:6-69
bool mosek::fusion::RowBlockManager :: is_parameterized() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->is_parameterized(); }
bool mosek::fusion::p_RowBlockManager::is_parameterized()
{
  return (code_used > (int)0);
}
// End mosek.fusion.RowBlockManager.is_parameterized

// Begin mosek.fusion.RowBlockManager.blocksize
// Method mosek.fusion.RowBlockManager.blocksize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5514:6-79
int32_t mosek::fusion::RowBlockManager :: blocksize(int32_t _613_id) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->blocksize(_613_id); }
int32_t mosek::fusion::p_RowBlockManager::blocksize(int32_t _613_id)
{
  return blocks->blocksize(_613_id);
}
// End mosek.fusion.RowBlockManager.blocksize

// Begin mosek.fusion.RowBlockManager.block_capacity
// Method mosek.fusion.RowBlockManager.block_capacity @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5513:6-79
int32_t mosek::fusion::RowBlockManager :: block_capacity() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->block_capacity(); }
int32_t mosek::fusion::p_RowBlockManager::block_capacity()
{
  return blocks->block_capacity();
}
// End mosek.fusion.RowBlockManager.block_capacity

// Begin mosek.fusion.RowBlockManager.capacity
// Method mosek.fusion.RowBlockManager.capacity @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5512:6-67
int32_t mosek::fusion::RowBlockManager :: capacity() { return mosek::fusion::p_RowBlockManager::_get_impl(this)->capacity(); }
int32_t mosek::fusion::p_RowBlockManager::capacity()
{
  return blocks->capacity();
}
// End mosek.fusion.RowBlockManager.capacity

// Begin mosek.fusion.RowBlockManager.get
// Method mosek.fusion.RowBlockManager.get @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5511:6-102
void mosek::fusion::RowBlockManager :: get(int32_t _614_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _615_target,int32_t _616_offset) { mosek::fusion::p_RowBlockManager::_get_impl(this)->get(_614_id,_615_target,_616_offset); }
void mosek::fusion::p_RowBlockManager::get(int32_t _614_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _615_target,int32_t _616_offset)
{
  _checked_ptr_1<int32_t> _ptr__615_target(_615_target ? _615_target->raw() : nullptr, _615_target ? _615_target->size(0) : 0);
  blocks->get(_614_id,_615_target,_616_offset);
}
// End mosek.fusion.RowBlockManager.get

// Begin mosek.fusion.RowBlockManager.evaluate
// Method mosek.fusion.RowBlockManager.evaluate @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5443:6-5509:7
void mosek::fusion::RowBlockManager :: evaluate(monty::rc_ptr< ::mosek::fusion::WorkStack > _617_xs,std::shared_ptr< monty::ndarray< double,1 > > _618_param_value,std::shared_ptr< monty::ndarray< int32_t,1 > > _619_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _620_subj,std::shared_ptr< monty::ndarray< double,1 > > _621_val) { mosek::fusion::p_RowBlockManager::_get_impl(this)->evaluate(_617_xs,_618_param_value,_619_subi,_620_subj,_621_val); }
void mosek::fusion::p_RowBlockManager::evaluate(monty::rc_ptr< ::mosek::fusion::WorkStack > _617_xs,std::shared_ptr< monty::ndarray< double,1 > > _618_param_value,std::shared_ptr< monty::ndarray< int32_t,1 > > _619_subi,std::shared_ptr< monty::ndarray< int32_t,1 > > _620_subj,std::shared_ptr< monty::ndarray< double,1 > > _621_val)
{
  _checked_ptr_1<double> _ptr__618_param_value(_618_param_value ? _618_param_value->raw() : nullptr, _618_param_value ? _618_param_value->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__619_subi(_619_subi ? _619_subi->raw() : nullptr, _619_subi ? _619_subi->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__620_subj(_620_subj ? _620_subj->raw() : nullptr, _620_subj ? _620_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__621_val(_621_val ? _621_val->raw() : nullptr, _621_val ? _621_val->size(0) : 0);
  int32_t _622_stack = _617_xs->allocf64(code_used);
  int32_t _623_stackp = (int)0;
  std::shared_ptr< monty::ndarray< double,1 > > _624_xf64 = mosek::fusion::p_WorkStack::_get_impl(_617_xs)->f64;
  _checked_ptr_1<double> _ptr__624_xf64(_624_xf64 ? _624_xf64->raw() : nullptr,_624_xf64 ? _624_xf64->size(0) : 0);
  int32_t _625_idx = (int)0;
  int32_t _9329_ = (int)0;
  int32_t _9330_ = (int)((row_param_ptrb)->size(0));
  for (int32_t _626_i = _9329_; _626_i < _9330_; ++_626_i)
  {
    {
      if ((((*row_param_ptrb)(_626_i)) < ((*row_param_ptre)(_626_i))))
      {
        {
          int32_t _627_k = ((*row_code_ptr)(_626_i));
          int32_t _9331_ = ((*row_param_ptrb)(_626_i));
          int32_t _9332_ = ((*row_param_ptre)(_626_i));
          for (int32_t _628_j = _9331_; _628_j < _9332_; ++_628_j)
          {
            {
              int32_t _9333_ = (int)0;
              int32_t _9334_ = ((*param_code_sizes)(_628_j));
              for (int32_t _629_l = _9333_; _629_l < _9334_; ++_629_l)
              {
                {
                  if ((((*code)(_627_k)) == 0))
                  {
                    {
                    }
                  }
                  else
                  {
                    {
                      if ((((*code)(_627_k)) == 1))
                      {
                        {
                          _ptr__624_xf64[safe_add( _622_stack,_623_stackp )] = _ptr__618_param_value[(int32_t)((*cconst)(_627_k))];
                          ++ _623_stackp;
                        }
                      }
                      else
                      {
                        {
                          if ((((*code)(_627_k)) == 2))
                          {
                            {
                              _ptr__624_xf64[safe_add( _622_stack,_623_stackp )] = ((*cconst)(_627_k));
                              ++ _623_stackp;
                            }
                          }
                          else
                          {
                            {
                              if ((((*code)(_627_k)) == 3))
                              {
                                {
                                  _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)2)] += _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)];
                                  -- _623_stackp;
                                }
                              }
                              else
                              {
                                {
                                  if ((((*code)(_627_k)) == 4))
                                  {
                                    {
                                      _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)] = (-_ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)]);
                                    }
                                  }
                                  else
                                  {
                                    {
                                      if ((((*code)(_627_k)) == 5))
                                      {
                                        {
                                          _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)2)] *= _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)];
                                          -- _623_stackp;
                                        }
                                      }
                                      else
                                      {
                                        {
                                          if ((((*code)(_627_k)) == 6))
                                          {
                                            {
                                              _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)] = (1.0 / _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)]);
                                            }
                                          }
                                          else
                                          {
                                            {
                                              if ((((*code)(_627_k)) == 8))
                                              {
                                                {
                                                  double _630_v = 0.0;
                                                  int32_t _9335_ = (int)0;
                                                  int32_t _9336_ = (int32_t)((*cconst)(_627_k));
                                                  for (int32_t _631_i = _9335_; _631_i < _9336_; ++_631_i)
                                                  {
                                                    {
                                                      _630_v += _ptr__624_xf64[(safe_add( _622_stack,_623_stackp ) - (int)1)];
                                                      -- _623_stackp;
                                                    }
                                                  }
                                                  _ptr__624_xf64[safe_add( _622_stack,_623_stackp )] = _630_v;
                                                  ++ _623_stackp;
                                                }
                                              }
                                              else
                                              {
                                                {
                                                }
                                              }
                                            }
                                          }
                                        }
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                  ++ _627_k;
                }
              }
              _ptr__619_subi[_625_idx] = _626_i;
              _ptr__620_subj[_625_idx] = (int32_t)((*param_varidx)(_628_j));
              _ptr__621_val[_625_idx] = _ptr__624_xf64[_622_stack];
              _623_stackp = (int)0;
              ++ _625_idx;
            }
          }
        }
      }
      {}
    }
  }
  _617_xs->popf64(code_used);
}
// End mosek.fusion.RowBlockManager.evaluate

// Begin mosek.fusion.RowBlockManager.replace_row_code
// Method mosek.fusion.RowBlockManager.replace_row_code @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5330:6-5441:7
void mosek::fusion::RowBlockManager :: replace_row_code(monty::rc_ptr< ::mosek::fusion::WorkStack > _632_rs,std::shared_ptr< monty::ndarray< int32_t,1 > > _633_nativeidxs,int32_t _634_ptr,int32_t _635_nidxs,int32_t _636_codeptr,int32_t _637_code_p,int32_t _638_cconst_p) { mosek::fusion::p_RowBlockManager::_get_impl(this)->replace_row_code(_632_rs,_633_nativeidxs,_634_ptr,_635_nidxs,_636_codeptr,_637_code_p,_638_cconst_p); }
void mosek::fusion::p_RowBlockManager::replace_row_code(monty::rc_ptr< ::mosek::fusion::WorkStack > _632_rs,std::shared_ptr< monty::ndarray< int32_t,1 > > _633_nativeidxs,int32_t _634_ptr,int32_t _635_nidxs,int32_t _636_codeptr,int32_t _637_code_p,int32_t _638_cconst_p)
{
  _checked_ptr_1<int32_t> _ptr__633_nativeidxs(_633_nativeidxs ? _633_nativeidxs->raw() : nullptr, _633_nativeidxs ? _633_nativeidxs->size(0) : 0);
  int32_t _639_nelem = (int)((_633_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _640_ri32 = mosek::fusion::p_WorkStack::_get_impl(_632_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__640_ri32(_640_ri32 ? _640_ri32->raw() : nullptr,_640_ri32 ? _640_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _641_ri64 = mosek::fusion::p_WorkStack::_get_impl(_632_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__641_ri64(_641_ri64 ? _641_ri64->raw() : nullptr,_641_ri64 ? _641_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _642_rf64 = mosek::fusion::p_WorkStack::_get_impl(_632_rs)->f64;
  _checked_ptr_1<double> _ptr__642_rf64(_642_rf64 ? _642_rf64->raw() : nullptr,_642_rf64 ? _642_rf64->size(0) : 0);
  bool _643_ok = true;
  {
    int32_t _644_numop = (int)0;
    int32_t _645_nentry = (int)0;
    {
      int32_t _9337_ = (int)0;
      int32_t _9338_ = _639_nelem;
      for (int32_t _646_i = _9337_; _646_i < _9338_; ++_646_i)
      {
        {
          int32_t _9339_ = _ptr__640_ri32[safe_add( _634_ptr,_646_i )];
          int32_t _9340_ = _ptr__640_ri32[safe_add( safe_add( _634_ptr,_646_i ),(int)1 )];
          for (int32_t _647_j = _9339_; _647_j < _9340_; ++_647_j)
          {
            {
              if ((((_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_647_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_647_j )]) == (int)1) && ((_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_647_j )] )] == 10) || (_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_647_j )] )] == 2))))
              {
                {
                }
              }
              else
              {
                {
                  ++ _645_nentry;
                  _644_numop += (_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_647_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_647_j )]);
                }
              }
            }
          }
        }
      }
      _pubthis->clear_row_code(_633_nativeidxs);
    }
    ensure_code_cap(_645_nentry,_644_numop);
    code_used += _644_numop;
    varidx_used += _645_nentry;
  }
  if ((first_free_codeitem > safe_mul( code_used,(int)2 )))
  {
    {
      compress();
    }
  }
  {}
  {
    int32_t _9341_ = (int)0;
    int32_t _9342_ = _639_nelem;
    for (int32_t _648_i = _9341_; _648_i < _9342_; ++_648_i)
    {
      {
        int32_t _649_subi = _ptr__633_nativeidxs[_648_i];
        int32_t _650_cnnz = (int)0;
        int32_t _651_rncodeatom = (int)0;
        int32_t _9343_ = _ptr__640_ri32[safe_add( _634_ptr,_648_i )];
        int32_t _9344_ = _ptr__640_ri32[safe_add( safe_add( _634_ptr,_648_i ),(int)1 )];
        for (int32_t _652_j = _9343_; _652_j < _9344_; ++_652_j)
        {
          {
            if ((((_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_652_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_652_j )]) == (int)1) && ((_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_652_j )] )] == 10) || (_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_652_j )] )] == 2))))
            {
              {
              }
            }
            else
            {
              {
                ++ _650_cnnz;
                _651_rncodeatom += (_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_652_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_652_j )]);
              }
            }
          }
        }
        int32_t _653_ni = _ptr__633_nativeidxs[_648_i];
        int32_t _654_n = (((*row_param_ptre)(_653_ni)) - ((*row_param_ptrb)(_653_ni)));
        if ((_650_cnnz > (int)0))
        {
          {
            ((*row_param_ptrb)(_649_subi)) = first_free_entry;
            ((*row_param_ptre)(_649_subi)) = safe_add( first_free_entry,_650_cnnz );
            ((*row_code_ptr)(_649_subi)) = first_free_codeitem;
            first_free_entry += _650_cnnz;
            first_free_codeitem += _651_rncodeatom;
          }
        }
        {}
      }
    }
  }
  int32_t _9345_ = (int)0;
  int32_t _9346_ = _639_nelem;
  for (int32_t _655_i = _9345_; _655_i < _9346_; ++_655_i)
  {
    {
      int32_t _656_subi = _ptr__633_nativeidxs[_655_i];
      int32_t _657_codei = ((*row_code_ptr)(_656_subi));
      int32_t _658_elmi = ((*row_param_ptrb)(_656_subi));
      int32_t _9347_ = _ptr__640_ri32[safe_add( _634_ptr,_655_i )];
      int32_t _9348_ = _ptr__640_ri32[safe_add( safe_add( _634_ptr,_655_i ),(int)1 )];
      for (int32_t _659_j = _9347_; _659_j < _9348_; ++_659_j)
      {
        {
          if ((((_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_659_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_659_j )]) == (int)1) && ((_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_659_j )] )] == 10) || (_ptr__640_ri32[safe_add( _637_code_p,_ptr__640_ri32[safe_add( _636_codeptr,_659_j )] )] == 2))))
          {
            {
            }
          }
          else
          {
            {
              int32_t _9349_ = _ptr__640_ri32[safe_add( _636_codeptr,_659_j )];
              int32_t _9350_ = _ptr__640_ri32[safe_add( safe_add( _636_codeptr,_659_j ),(int)1 )];
              for (int32_t _660_k = _9349_; _660_k < _9350_; ++_660_k)
              {
                {
                  ((*code)(_657_codei)) = _ptr__640_ri32[safe_add( _637_code_p,_660_k )];
                  ((*cconst)(_657_codei)) = _ptr__642_rf64[safe_add( _638_cconst_p,_660_k )];
                  ++ _657_codei;
                }
              }
              ((*param_code_sizes)(_658_elmi)) = (_ptr__640_ri32[safe_add( safe_add( _636_codeptr,_659_j ),(int)1 )] - _ptr__640_ri32[safe_add( _636_codeptr,_659_j )]);
              ((*param_varidx)(_658_elmi)) = _ptr__641_ri64[safe_add( _635_nidxs,_659_j )];
              _643_ok = (_643_ok && (_ptr__641_ri64[safe_add( _635_nidxs,_659_j )] >= (int)0));
              ++ _658_elmi;
            }
          }
        }
      }
    }
  }
  if ((!_643_ok))
  {
    {
      throw ::mosek::fusion::ParameterError(std::string ("Expression contains parameterized SDP non-zero"));
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.replace_row_code

// Begin mosek.fusion.RowBlockManager.clear_row_code
// Method mosek.fusion.RowBlockManager.clear_row_code @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5303:6-5319:7
void mosek::fusion::RowBlockManager :: clear_row_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _661_nativeidxs) { mosek::fusion::p_RowBlockManager::_get_impl(this)->clear_row_code(_661_nativeidxs); }
void mosek::fusion::p_RowBlockManager::clear_row_code(std::shared_ptr< monty::ndarray< int32_t,1 > > _661_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__661_nativeidxs(_661_nativeidxs ? _661_nativeidxs->raw() : nullptr, _661_nativeidxs ? _661_nativeidxs->size(0) : 0);
  if ((NULL != row_param_ptrb.get()))
  {
    {
      int32_t _9351_ = (int)0;
      int32_t _9352_ = (int)((_661_nativeidxs)->size(0));
      for (int32_t _662_i = _9351_; _662_i < _9352_; ++_662_i)
      {
        {
          int32_t _9353_ = ((*row_param_ptrb)(_ptr__661_nativeidxs[_662_i]));
          int32_t _9354_ = ((*row_param_ptre)(_ptr__661_nativeidxs[_662_i]));
          for (int32_t _663_j = _9353_; _663_j < _9354_; ++_663_j)
          {
            {
              code_used -= ((*param_code_sizes)(_663_j));
            }
          }
          varidx_used -= (((*row_param_ptre)(_ptr__661_nativeidxs[_662_i])) - ((*row_param_ptrb)(_ptr__661_nativeidxs[_662_i])));
          ((*row_param_ptrb)(_ptr__661_nativeidxs[_662_i])) = (int)0;
          ((*row_param_ptre)(_ptr__661_nativeidxs[_662_i])) = (int)0;
        }
      }
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.clear_row_code

// Begin mosek.fusion.RowBlockManager.compress
// Method mosek.fusion.RowBlockManager.compress @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5259:6-5301:7
void mosek::fusion::p_RowBlockManager::compress()
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _664_newcode = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(code_used)));
  _checked_ptr_1<int32_t> _ptr__664_newcode(_664_newcode ? _664_newcode->raw() : nullptr,_664_newcode ? _664_newcode->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _665_newcconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(code_used)));
  _checked_ptr_1<double> _ptr__665_newcconst(_665_newcconst ? _665_newcconst->raw() : nullptr,_665_newcconst ? _665_newcconst->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _666_newvaridx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(first_free_entry)));
  _checked_ptr_1<int64_t> _ptr__666_newvaridx(_666_newvaridx ? _666_newvaridx->raw() : nullptr,_666_newvaridx ? _666_newvaridx->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _667_newcodesizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(first_free_entry)));
  _checked_ptr_1<int32_t> _ptr__667_newcodesizes(_667_newcodesizes ? _667_newcodesizes->raw() : nullptr,_667_newcodesizes ? _667_newcodesizes->size(0) : 0);
  int32_t _668_tgtptr = (int)0;
  int32_t _669_tgtcptr = (int)0;
  int32_t _9355_ = (int)0;
  int32_t _9356_ = (int)((row_param_ptrb)->size(0));
  for (int32_t _670_i = _9355_; _670_i < _9356_; ++_670_i)
  {
    {
      int32_t _671_codesize = (int)0;
      int32_t _672_n = (((*row_param_ptre)(_670_i)) - ((*row_param_ptrb)(_670_i)));
      ::mosek::fusion::Utils::Tools::arraycopy(param_varidx,((*row_param_ptrb)(_670_i)),_666_newvaridx,_668_tgtptr,_672_n);
      ::mosek::fusion::Utils::Tools::arraycopy(param_code_sizes,((*row_param_ptrb)(_670_i)),_667_newcodesizes,_668_tgtptr,_672_n);
      int32_t _9357_ = ((*row_param_ptrb)(_670_i));
      int32_t _9358_ = ((*row_param_ptre)(_670_i));
      for (int32_t _673_j = _9357_; _673_j < _9358_; ++_673_j)
      {
        {
          _671_codesize += ((*param_code_sizes)(_673_j));
        }
      }
      ::mosek::fusion::Utils::Tools::arraycopy(code,((*row_code_ptr)(_670_i)),_664_newcode,_669_tgtcptr,_671_codesize);
      ::mosek::fusion::Utils::Tools::arraycopy(cconst,((*row_code_ptr)(_670_i)),_665_newcconst,_669_tgtcptr,_671_codesize);
      ((*row_param_ptrb)(_670_i)) = _668_tgtptr;
      ((*row_param_ptre)(_670_i)) = safe_add( _668_tgtptr,_672_n );
      ((*row_code_ptr)(_670_i)) = _669_tgtcptr;
      _669_tgtcptr += _671_codesize;
      _668_tgtptr += _672_n;
    }
  }
  ::mosek::fusion::Utils::Tools::arraycopy(_666_newvaridx,(int)0,param_varidx,(int)0,_668_tgtptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_667_newcodesizes,(int)0,param_code_sizes,(int)0,_668_tgtptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_664_newcode,(int)0,code,(int)0,_669_tgtcptr);
  ::mosek::fusion::Utils::Tools::arraycopy(_665_newcconst,(int)0,cconst,(int)0,_669_tgtcptr);
  first_free_codeitem = _669_tgtcptr;
  first_free_entry = _668_tgtptr;
}
// End mosek.fusion.RowBlockManager.compress

// Begin mosek.fusion.RowBlockManager.ensure_code_cap
// Method mosek.fusion.RowBlockManager.ensure_code_cap @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5223:6-5257:7
void mosek::fusion::p_RowBlockManager::ensure_code_cap(int32_t _674_nentry,int32_t _675_codesize)
{
  if ((NULL == row_code_ptr.get()))
  {
    {
      int32_t _676_n = (int)128;
      while ( (_676_n < _674_nentry) )
      {
        {
          _676_n *= (int)2;
        }
      }
      int32_t _677_m = (int)128;
      while ( (_677_m < _675_codesize) )
      {
        {
          _677_m *= (int)2;
        }
      }
      row_param_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      row_param_ptre = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      row_code_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      param_varidx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_676_n)));
      param_code_sizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_676_n)));
      code = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_677_m)));
      cconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_677_m)));
    }
  }
  else
  {
    {
      if ((_674_nentry > ((int)((param_varidx)->size(0)) - first_free_entry)))
      {
        {
          int32_t _678_n = (int)((param_varidx)->size(0));
          while ( ((_678_n - first_free_entry) < _674_nentry) )
          {
            {
              _678_n *= (int)2;
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _679_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_678_n)));
          _checked_ptr_1<int32_t> _ptr__679_ptr(_679_ptr ? _679_ptr->raw() : nullptr,_679_ptr ? _679_ptr->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _680_varidx = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_678_n)));
          _checked_ptr_1<int64_t> _ptr__680_varidx(_680_varidx ? _680_varidx->raw() : nullptr,_680_varidx ? _680_varidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _681_code_sizes = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_678_n)));
          _checked_ptr_1<int32_t> _ptr__681_code_sizes(_681_code_sizes ? _681_code_sizes->raw() : nullptr,_681_code_sizes ? _681_code_sizes->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(param_varidx,(int)0,_680_varidx,(int)0,first_free_entry);
          ::mosek::fusion::Utils::Tools::arraycopy(param_code_sizes,(int)0,_681_code_sizes,(int)0,first_free_entry);
          param_varidx = _680_varidx;
          param_code_sizes = _681_code_sizes;
        }
      }
      {}
      if ((_675_codesize > ((int)((code)->size(0)) - first_free_codeitem)))
      {
        {
          int32_t _682_n = (int)((code)->size(0));
          while ( (_675_codesize > (_682_n - first_free_codeitem)) )
          {
            {
              _682_n *= (int)2;
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _683_tcode = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_682_n)));
          _checked_ptr_1<int32_t> _ptr__683_tcode(_683_tcode ? _683_tcode->raw() : nullptr,_683_tcode ? _683_tcode->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(code,(int)0,_683_tcode,(int)0,first_free_codeitem);
          std::shared_ptr< monty::ndarray< double,1 > > _684_tcconst = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_682_n)));
          _checked_ptr_1<double> _ptr__684_tcconst(_684_tcconst ? _684_tcconst->raw() : nullptr,_684_tcconst ? _684_tcconst->size(0) : 0);
          ::mosek::fusion::Utils::Tools::arraycopy(cconst,(int)0,_684_tcconst,(int)0,first_free_codeitem);
          code = _683_tcode;
          cconst = _684_tcconst;
        }
      }
      {}
    }
  }
}
// End mosek.fusion.RowBlockManager.ensure_code_cap

// Begin mosek.fusion.RowBlockManager.release
// Method mosek.fusion.RowBlockManager.release @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5210:6-5221:7
void mosek::fusion::RowBlockManager :: release(int32_t _685_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _686_nativeidxs) { mosek::fusion::p_RowBlockManager::_get_impl(this)->release(_685_id,_686_nativeidxs); }
void mosek::fusion::p_RowBlockManager::release(int32_t _685_id,std::shared_ptr< monty::ndarray< int32_t,1 > > _686_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__686_nativeidxs(_686_nativeidxs ? _686_nativeidxs->raw() : nullptr, _686_nativeidxs ? _686_nativeidxs->size(0) : 0);
  int32_t _687_num = blocks->blocksize(_685_id);
  blocks->get(_685_id,_686_nativeidxs,(int)0);
  blocks->free(_685_id);
  if ((NULL != row_code_ptr.get()))
  {
    {
      int32_t _9359_ = (int)0;
      int32_t _9360_ = _687_num;
      for (int32_t _688_i = _9359_; _688_i < _9360_; ++_688_i)
      {
        {
          int32_t _689_ni = _ptr__686_nativeidxs[_688_i];
          code_used -= (((*row_param_ptre)(_689_ni)) - ((*row_param_ptrb)(_689_ni)));
          ((*row_param_ptre)(_689_ni)) = (int)0;
          ((*row_param_ptrb)(_689_ni)) = (int)0;
        }
      }
    }
  }
  {}
}
// End mosek.fusion.RowBlockManager.release

// Begin mosek.fusion.RowBlockManager.allocate
// Method mosek.fusion.RowBlockManager.allocate @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5192:6-5208:7
int32_t mosek::fusion::RowBlockManager :: allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _690_nativeidxs) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->allocate(_690_nativeidxs); }
int32_t mosek::fusion::p_RowBlockManager::allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _690_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__690_nativeidxs(_690_nativeidxs ? _690_nativeidxs->raw() : nullptr, _690_nativeidxs ? _690_nativeidxs->size(0) : 0);
  int32_t _691_num = (int)((_690_nativeidxs)->size(0));
  int32_t _692_id = blocks->alloc(_691_num);
  blocks->get(_692_id,_690_nativeidxs,(int)0);
  if (((NULL != row_code_ptr.get()) && (blocks->capacity() > (int)((row_param_ptrb)->size(0)))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _693_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__693_ptrb(_693_ptrb ? _693_ptrb->raw() : nullptr,_693_ptrb ? _693_ptrb->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_param_ptrb,(int)0,_693_ptrb,(int)0,(int)((row_param_ptrb)->size(0)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _694_ptre = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__694_ptre(_694_ptre ? _694_ptre->raw() : nullptr,_694_ptre ? _694_ptre->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_param_ptre,(int)0,_694_ptre,(int)0,(int)((row_param_ptre)->size(0)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _695_cptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(blocks->capacity())));
      _checked_ptr_1<int32_t> _ptr__695_cptr(_695_cptr ? _695_cptr->raw() : nullptr,_695_cptr ? _695_cptr->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(row_code_ptr,(int)0,_695_cptr,(int)0,(int)((row_code_ptr)->size(0)));
      row_param_ptrb = _693_ptrb;
      row_param_ptre = _694_ptre;
      row_code_ptr = _695_cptr;
    }
  }
  {}
  return _692_id;
}
// End mosek.fusion.RowBlockManager.allocate

// Begin mosek.fusion.RowBlockManager.row_is_parameterized
// Method mosek.fusion.RowBlockManager.row_is_parameterized @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:5188:6-5190:7
bool mosek::fusion::RowBlockManager :: row_is_parameterized(int32_t _696_i) { return mosek::fusion::p_RowBlockManager::_get_impl(this)->row_is_parameterized(_696_i); }
bool mosek::fusion::p_RowBlockManager::row_is_parameterized(int32_t _696_i)
{
  return ((NULL != row_param_ptrb.get()) && (((*row_param_ptrb)(_696_i)) < ((*row_param_ptre)(_696_i))));
}
// End mosek.fusion.RowBlockManager.row_is_parameterized

void mosek::fusion::p_RowBlockManager::destroy()
{
  cconst.reset();
  code.reset();
  param_code_sizes.reset();
  param_varidx.reset();
  row_code_ptr.reset();
  row_param_ptre.reset();
  row_param_ptrb.reset();
  blocks.reset();
}
void mosek::fusion::RowBlockManager::destroy() { mosek::fusion::p_RowBlockManager::_get_impl(this)->destroy(); }
// } class RowBlockManager
// class Model { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:351:4-5104:5
mosek::fusion::Model::Model(mosek::fusion::p_Model *_impl) : mosek::fusion::BaseModel(_impl) { /*std::cout << "Model()" << this << std::endl; */ }
mosek::fusion::Model::~Model() {  /* std::cout << "~Model()" << this << std::endl << std::flush; */ }
mosek::fusion::p_Model::p_Model
  (::mosek::fusion::Model * _pubthis) :     mosek::fusion::p_BaseModel(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.Model.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:576:6-643:7
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (monty::rc_ptr< ::mosek::fusion::Model > _697_m)
{ return new mosek::fusion::Model(_697_m); }
mosek::fusion::Model::Model
  (monty::rc_ptr< ::mosek::fusion::Model > _697_m) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_697_m); }
void mosek::fusion::p_Model::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _697_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseModel::_initialize(_697_m);
    monty::rc_ptr< ::mosek::fusion::Model > _698_self = ::mosek::fusion::Model::t(_pubthis);
    model_name = mosek::fusion::p_Model::_get_impl(_697_m)->model_name;
    acceptable_sol = mosek::fusion::p_Model::_get_impl(_697_m)->acceptable_sol;
    solutionptr = mosek::fusion::p_Model::_get_impl(_697_m)->solutionptr;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9361_;
    bool _9362_ = (NULL != mosek::fusion::p_Model::_get_impl(_697_m)->sol_itr.get());
    if (_9362_)
    {
      _9361_ = mosek::fusion::p_Model::_get_impl(_697_m)->sol_itr->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9361_ = nullptr;
    }
    sol_itr = _9361_;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9363_;
    bool _9364_ = (NULL != mosek::fusion::p_Model::_get_impl(_697_m)->sol_itg.get());
    if (_9364_)
    {
      _9363_ = mosek::fusion::p_Model::_get_impl(_697_m)->sol_itg->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9363_ = nullptr;
    }
    sol_itg = _9363_;
    monty::rc_ptr< ::mosek::fusion::SolutionStruct > _9365_;
    bool _9366_ = (NULL != mosek::fusion::p_Model::_get_impl(_697_m)->sol_bas.get());
    if (_9366_)
    {
      _9365_ = mosek::fusion::p_Model::_get_impl(_697_m)->sol_bas->__mosek_2fusion_2SolutionStruct__clone();
    }
    else
    {
      _9365_ = nullptr;
    }
    sol_bas = _9365_;
    num_task_afe = mosek::fusion::p_Model::_get_impl(_697_m)->num_task_afe;
    num_task_con = mosek::fusion::p_Model::_get_impl(_697_m)->num_task_con;
    num_task_acc = mosek::fusion::p_Model::_get_impl(_697_m)->num_task_acc;
    con_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_697_m)->con_blocks);
    afe_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_697_m)->afe_blocks);
    obj_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager(mosek::fusion::p_Model::_get_impl(_697_m)->obj_blocks);
    acc_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_697_m)->acc_block_map);
    acon_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_697_m)->acon_block_map);
    acon_acc = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_acc);
    acon_afe = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_afe);
    acon_elm_accid = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_elm_accid);
    acon_elm_scale = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_elm_scale);
    acon_elm_ofs = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_elm_ofs);
    acon_elm_afe = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->acon_elm_afe);
    task_numaferow = mosek::fusion::p_Model::_get_impl(_697_m)->task_numaferow;
    var_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks(mosek::fusion::p_Model::_get_impl(_697_m)->var_block_map);
    var_block_acc_id = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->var_block_acc_id);
    var_elm_acc_idx = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->var_elm_acc_idx);
    var_elm_acc_ofs = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->var_elm_acc_ofs);
    barvar_num = (int)0;
    barvar_dim = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_dim);
    barvar_block_ptr = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_ptr);
    barvar_block_dim = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_dim);
    barvar_block_elm_ptr = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_elm_ptr);
    barvar_block_elm_barj = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_elm_barj);
    barvar_block_elm_i = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_elm_i);
    barvar_block_elm_j = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->barvar_block_elm_j);
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9367_;
    for (int32_t _699_i = (int)0, _9368_ = (int)((mosek::fusion::p_Model::_get_impl(_697_m)->vars)->size(0)); _699_i < _9368_ ; ++_699_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelVariable > _9369_;
      bool _9370_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_697_m)->vars)(_699_i)).get());
      if (_9370_)
      {
        _9369_ = mosek::fusion::p_ModelVariable::_get_impl(((*mosek::fusion::p_Model::_get_impl(_697_m)->vars)(_699_i)).get())->__mosek_2fusion_2ModelVariable__clone(_698_self);
      }
      else
      {
        _9369_ = nullptr;
      }
      _9367_.push_back(_9369_);
    }
    auto _9371_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9367_.size()),_9367_.begin(),_9367_.end()));
    vars = _9371_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9372_;
    for (int32_t _700_i = (int)0, _9373_ = (int)((mosek::fusion::p_Model::_get_impl(_697_m)->barvars)->size(0)); _700_i < _9373_ ; ++_700_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelVariable > _9374_;
      bool _9375_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_697_m)->barvars)(_700_i)).get());
      if (_9375_)
      {
        _9374_ = mosek::fusion::p_ModelVariable::_get_impl(((*mosek::fusion::p_Model::_get_impl(_697_m)->barvars)(_700_i)).get())->__mosek_2fusion_2ModelVariable__clone(_698_self);
      }
      else
      {
        _9374_ = nullptr;
      }
      _9372_.push_back(_9374_);
    }
    auto _9376_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9372_.size()),_9372_.begin(),_9372_.end()));
    barvars = _9376_;
    var_map = mosek::fusion::p_Model::_get_impl(_697_m)->var_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9377_;
    for (int32_t _701_i = (int)0, _9378_ = (int)((mosek::fusion::p_Model::_get_impl(_697_m)->cons)->size(0)); _701_i < _9378_ ; ++_701_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelConstraint > _9379_;
      bool _9380_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_697_m)->cons)(_701_i)).get());
      if (_9380_)
      {
        _9379_ = mosek::fusion::p_ModelConstraint::_get_impl(((*mosek::fusion::p_Model::_get_impl(_697_m)->cons)(_701_i)).get())->__mosek_2fusion_2ModelConstraint__clone(_698_self);
      }
      else
      {
        _9379_ = nullptr;
      }
      _9377_.push_back(_9379_);
    }
    auto _9381_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9377_.size()),_9377_.begin(),_9377_.end()));
    cons = _9381_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9382_;
    for (int32_t _702_i = (int)0, _9383_ = (int)((mosek::fusion::p_Model::_get_impl(_697_m)->acons)->size(0)); _702_i < _9383_ ; ++_702_i)
    {
      monty::rc_ptr< ::mosek::fusion::ModelConstraint > _9384_;
      bool _9385_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_697_m)->acons)(_702_i)).get());
      if (_9385_)
      {
        _9384_ = mosek::fusion::p_ModelConstraint::_get_impl(((*mosek::fusion::p_Model::_get_impl(_697_m)->acons)(_702_i)).get())->__mosek_2fusion_2ModelConstraint__clone(_698_self);
      }
      else
      {
        _9384_ = nullptr;
      }
      _9382_.push_back(_9384_);
    }
    auto _9386_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9382_.size()),_9382_.begin(),_9382_.end()));
    acons = _9386_;
    con_map = mosek::fusion::p_Model::_get_impl(_697_m)->con_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    numparameter = mosek::fusion::p_Model::_get_impl(_697_m)->numparameter;
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > > _9387_;
    bool _9388_ = (NULL == mosek::fusion::p_Model::_get_impl(_697_m)->parameters.get());
    if (_9388_)
    {
      _9387_ = nullptr;
    }
    else
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::Parameter >> _9389_;
      for (int32_t _703_i = (int)0, _9390_ = (int)((mosek::fusion::p_Model::_get_impl(_697_m)->parameters)->size(0)); _703_i < _9390_ ; ++_703_i)
      {
        monty::rc_ptr< ::mosek::fusion::Parameter > _9391_;
        bool _9392_ = (NULL != ((*mosek::fusion::p_Model::_get_impl(_697_m)->parameters)(_703_i)).get());
        if (_9392_)
        {
          _9391_ = ((*mosek::fusion::p_Model::_get_impl(_697_m)->parameters)(_703_i))->__mosek_2fusion_2Parameter__clone(_698_self);
        }
        else
        {
          _9391_ = nullptr;
        }
        _9389_.push_back(_9391_);
      }
      auto _9393_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 >(monty::shape(_9389_.size()),_9389_.begin(),_9389_.end()));
      _9387_ = _9393_;
    }
    parameters = _9387_;
    par_map = mosek::fusion::p_Model::_get_impl(_697_m)->par_map->__mosek_2fusion_2Utils_2StringIntMap__clone();
    param_num = mosek::fusion::p_Model::_get_impl(_697_m)->param_num;
    std::shared_ptr< monty::ndarray< double,1 > > _9394_;
    bool _9395_ = (NULL != mosek::fusion::p_Model::_get_impl(_697_m)->param_value.get());
    if (_9395_)
    {
      _9394_ = ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_Model::_get_impl(_697_m)->param_value);
    }
    else
    {
      _9394_ = nullptr;
    }
    param_value = _9394_;
    initsol_xx = nullptr;
    initsol_xx_flag = nullptr;
    bfixidx = mosek::fusion::p_Model::_get_impl(_697_m)->bfixidx;
    rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    xs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:498:6-573:7
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (const std::string &  _704_name,
    int32_t _705_basesize)
{ return new mosek::fusion::Model(_704_name,_705_basesize); }
mosek::fusion::Model::Model
  (const std::string &  _704_name,
    int32_t _705_basesize) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_704_name,_705_basesize); }
void mosek::fusion::p_Model::_initialize  (const std::string &  _704_name,
    int32_t _705_basesize)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseModel::_initialize(_704_name,std::string (""));
    model_name = _704_name;
    acceptable_sol = ::mosek::fusion::AccSolutionStatus::Optimal;
    solutionptr = ::mosek::fusion::SolutionType::Default;
    sol_itr = nullptr;
    sol_itg = nullptr;
    sol_bas = nullptr;
    num_task_afe = (int)0;
    num_task_con = (int)0;
    num_task_acc = (int)0;
    con_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    afe_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    obj_blocks = ::mosek::fusion::p_RowBlockManager::_new_RowBlockManager();
    acc_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    acon_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    task_numaferow = (int)0;
    acon_acc = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->block_capacity())));
    acon_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->block_capacity())));
    acon_elm_accid = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    std::vector<double> _9396_;
    for (int32_t _706_i = (int)0, _9397_ = acon_block_map->capacity(); _706_i < _9397_ ; ++_706_i)
    {
      _9396_.push_back(1.0);
    }
    auto _9398_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9396_.size()),_9396_.begin(),_9396_.end()));
    acon_elm_scale = _9398_;
    acon_elm_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    acon_elm_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(acon_block_map->capacity())));
    var_block_map = ::mosek::fusion::p_LinkedBlocks::_new_LinkedBlocks();
    var_block_acc_id = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->block_capacity())));
    var_elm_acc_idx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->capacity())));
    var_elm_acc_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(var_block_map->capacity())));
    barvar_num = (int)0;
    barvar_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_705_basesize)));
    barvar_block_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _705_basesize,(int)1 ))));
    barvar_block_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_705_basesize)));
    barvar_block_elm_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _705_basesize,(int)1 ))));
    barvar_block_elm_barj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_705_basesize)));
    barvar_block_elm_i = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_705_basesize)));
    barvar_block_elm_j = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_705_basesize)));
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9399_;
    for (int32_t _707_i = (int)0, _9400_ = safe_mul( _705_basesize,(int)8 ); _707_i < _9400_ ; ++_707_i)
    {
      _9399_.push_back(nullptr);
    }
    auto _9401_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9399_.size()),_9399_.begin(),_9399_.end()));
    vars = _9401_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9402_;
    for (int32_t _708_i = (int)0, _9403_ = _705_basesize; _708_i < _9403_ ; ++_708_i)
    {
      _9402_.push_back(nullptr);
    }
    auto _9404_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9402_.size()),_9402_.begin(),_9402_.end()));
    barvars = _9404_;
    var_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    initsol_xx = nullptr;
    initsol_xx_flag = nullptr;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9405_;
    for (int32_t _709_i = (int)0, _9406_ = safe_mul( _705_basesize,(int)8 ); _709_i < _9406_ ; ++_709_i)
    {
      _9405_.push_back(nullptr);
    }
    auto _9407_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9405_.size()),_9405_.begin(),_9405_.end()));
    cons = _9407_;
    std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9408_;
    for (int32_t _710_i = (int)0, _9409_ = safe_mul( _705_basesize,(int)8 ); _710_i < _9409_ ; ++_710_i)
    {
      _9408_.push_back(nullptr);
    }
    auto _9410_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9408_.size()),_9408_.begin(),_9408_.end()));
    acons = _9410_;
    con_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    numparameter = (int)0;
    parameters = nullptr;
    par_map = ::mosek::fusion::Utils::p_StringIntMap::_new_StringIntMap();
    param_num = (int)0;
    param_value = nullptr;
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _711_natvarbfixidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)1)));
      _checked_ptr_1<int32_t> _ptr__711_natvarbfixidx(_711_natvarbfixidx ? _711_natvarbfixidx->raw() : nullptr,_711_natvarbfixidx ? _711_natvarbfixidx->size(0) : 0);
      int32_t _712_bfixvarid = linearvar_alloc((int)1,_711_natvarbfixidx);
      bfixidx = _ptr__711_natvarbfixidx[(int)0];
    }
    task_var_putboundlist_fx(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(bfixidx)}),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >{(double)(1.0)}));
    task_var_name(bfixidx,std::string ("1.0"));
    task_append_domain_empty();
    rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
    xs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:490:6-58
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (int32_t _713_basesize)
{ return new mosek::fusion::Model(_713_basesize); }
mosek::fusion::Model::Model
  (int32_t _713_basesize) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_713_basesize); }
void mosek::fusion::p_Model::_initialize  (int32_t _713_basesize)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(std::string (""),_713_basesize);
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:484:6-53
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  (const std::string &  _714_name)
{ return new mosek::fusion::Model(_714_name); }
mosek::fusion::Model::Model
  (const std::string &  _714_name) : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(_714_name); }
void mosek::fusion::p_Model::_initialize  (const std::string &  _714_name)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(_714_name,(int)128);
  }
}
// mosek.fusion.Model.ctor

// mosek.fusion.Model.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:478:6-33
mosek::fusion::Model::t mosek::fusion::p_Model::_new_Model
  ()
{ return new mosek::fusion::Model(); }
mosek::fusion::Model::Model
  () : 
   Model(new mosek::fusion::p_Model(this))
{ mosek::fusion::p_Model::_get_impl(this)->_initialize(); }
void mosek::fusion::p_Model::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_Model::_initialize(std::string (""));
  }
}
// mosek.fusion.Model.ctor

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4985:6-5103:7
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _715_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _716_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_715_name,_716_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _715_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _716_terms) { return __mosek_2fusion_2Model__disjunction(_715_name,_716_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _715_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _716_terms)
{
  int32_t _717_n = (int)((_716_terms)->size(0));
  int32_t _718_nrows = (int)0;
  int32_t _719_nterms = (int)0;
  int32_t _720_naccterms = (int)0;
  int32_t _721_naccrows = (int)0;
  int32_t _9411_ = (int)0;
  int32_t _9412_ = _717_n;
  for (int32_t _722_i = _9411_; _722_i < _9412_; ++_722_i)
  {
    {
      _718_nrows += ((*_716_terms)(_722_i))->size();
      _719_nterms += mosek::fusion::p_Term::_get_impl(((*_716_terms)(_722_i)).get())->num();
      _721_naccrows += mosek::fusion::p_Term::_get_impl(((*_716_terms)(_722_i)).get())->numaccrows();
      _720_naccterms += mosek::fusion::p_Term::_get_impl(((*_716_terms)(_722_i)).get())->numaccterms();
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _723_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_718_nrows)));
  _checked_ptr_1<int32_t> _ptr__723_afeidxs(_723_afeidxs ? _723_afeidxs->raw() : nullptr,_723_afeidxs ? _723_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _724_accafeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_721_naccrows)));
  _checked_ptr_1<int32_t> _ptr__724_accafeidxs(_724_accafeidxs ? _724_accafeidxs->raw() : nullptr,_724_accafeidxs ? _724_accafeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _725_accb = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_721_naccrows)));
  _checked_ptr_1<double> _ptr__725_accb(_725_accb ? _725_accb->raw() : nullptr,_725_accb ? _725_accb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _726_domidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_720_naccterms)));
  _checked_ptr_1<int64_t> _ptr__726_domidxs(_726_domidxs ? _726_domidxs->raw() : nullptr,_726_domidxs ? _726_domidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _727_termsize = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_717_n)));
  _checked_ptr_1<int64_t> _ptr__727_termsize(_727_termsize ? _727_termsize->raw() : nullptr,_727_termsize ? _727_termsize->size(0) : 0);
  int32_t _728_afeblockid = afe_allocate(_723_afeidxs);
  int32_t _729_term_index = (int)0;
  int32_t _730_acctermptr = (int)0;
  int32_t _731_afetgtptr = (int)0;
  int32_t _732_afeptr = (int)0;
  int32_t _9413_ = (int)0;
  int32_t _9414_ = _717_n;
  for (int32_t _733_term_i = _9413_; _733_term_i < _9414_; ++_733_term_i)
  {
    {
      int32_t _9415_ = (int)0;
      int32_t _9416_ = mosek::fusion::p_Term::_get_impl(((*_716_terms)(_733_term_i)).get())->num();
      for (int32_t _734_simpterm_i = _9415_; _734_simpterm_i < _9416_; ++_734_simpterm_i)
      {
        {
          monty::rc_ptr< ::mosek::fusion::DJCDomain > _735_dom = ((*mosek::fusion::p_Term::_get_impl(((*_716_terms)(_733_term_i)))->domains)(_734_simpterm_i));
          monty::rc_ptr< ::mosek::fusion::ExprDense > _736_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(((*mosek::fusion::p_Term::_get_impl(((*_716_terms)(_733_term_i)))->exprs)(_734_simpterm_i)))).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
          _736_e->eval(rs,ws,xs);
          rs->pop_expr();
          std::shared_ptr< monty::ndarray< int32_t,1 > > _737_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
          _checked_ptr_1<int32_t> _ptr__737_ri32(_737_ri32 ? _737_ri32->raw() : nullptr,_737_ri32 ? _737_ri32->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _738_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
          _checked_ptr_1<int64_t> _ptr__738_ri64(_738_ri64 ? _738_ri64->raw() : nullptr,_738_ri64 ? _738_ri64->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _739_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
          _checked_ptr_1<double> _ptr__739_rf64(_739_rf64 ? _739_rf64->raw() : nullptr,_739_rf64 ? _739_rf64->size(0) : 0);
          int32_t _740_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
          int32_t _741_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
          int32_t _742_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
          int32_t _743_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
          int32_t _744_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
          bool _745_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
          int32_t _746_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
          int32_t _747_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
          int32_t _748_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
          int32_t _749_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
          int32_t _750_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
          int32_t _751_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
          std::vector<int32_t> _9417_;
          for (int32_t _753_i = (int)0, _9418_ = _740_nd; _753_i < _9418_ ; ++_753_i)
          {
            _9417_.push_back(_ptr__737_ri32[safe_add( _741_shape_base,_753_i )]);
          }
          auto _9419_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9417_.size()),_9417_.begin(),_9417_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _752_shape = _9419_;
          _checked_ptr_1<int32_t> _ptr__752_shape(_752_shape ? _752_shape->raw() : nullptr,_752_shape ? _752_shape->size(0) : 0);
          putfrows(_723_afeidxs,_732_afeptr,rs,_742_nelem,_743_nnz,_746_ptr_base,_747_nidxs_base,_748_cof_base);
          if ((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::InRange))
          {
            {
              std::shared_ptr< monty::ndarray< double,1 > > _754_lb = mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->b;
              _checked_ptr_1<double> _ptr__754_lb(_754_lb ? _754_lb->raw() : nullptr,_754_lb ? _754_lb->size(0) : 0);
              std::shared_ptr< monty::ndarray< double,1 > > _755_ub = mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->par;
              _checked_ptr_1<double> _ptr__755_ub(_755_ub ? _755_ub->raw() : nullptr,_755_ub ? _755_ub->size(0) : 0);
              int32_t _9420_ = (int)0;
              int32_t _9421_ = _742_nelem;
              for (int32_t _756_i = _9420_; _756_i < _9421_; ++_756_i)
              {
                {
                  _ptr__724_accafeidxs[safe_add( _731_afetgtptr,_756_i )] = _ptr__723_afeidxs[safe_add( _732_afeptr,_756_i )];
                  _ptr__724_accafeidxs[safe_add( safe_add( _731_afetgtptr,_742_nelem ),_756_i )] = _ptr__723_afeidxs[safe_add( _732_afeptr,_756_i )];
                  _ptr__725_accb[safe_add( _731_afetgtptr,_756_i )] = _ptr__754_lb[_756_i];
                  _ptr__725_accb[safe_add( safe_add( _731_afetgtptr,_742_nelem ),_756_i )] = _ptr__755_ub[_756_i];
                }
              }
              _ptr__726_domidxs[_730_acctermptr] = task_append_domain_rpos(_742_nelem);
              _ptr__726_domidxs[safe_add( _730_acctermptr,(int)1 )] = task_append_domain_rneg(_742_nelem);
              _ptr__727_termsize[_733_term_i] += (int)2;
              _730_acctermptr += (int)2;
              _731_afetgtptr += safe_mul( _742_nelem,(int)2 );
            }
          }
          else if(((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::EqualTo) || ((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::LessThan) || ((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::GreaterThan) || (mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::IsFree)))))
          {
            {
              std::shared_ptr< monty::ndarray< double,1 > > _757_b = mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->b;
              _checked_ptr_1<double> _ptr__757_b(_757_b ? _757_b->raw() : nullptr,_757_b ? _757_b->size(0) : 0);
              int32_t _9422_ = (int)0;
              int32_t _9423_ = _742_nelem;
              for (int32_t _758_i = _9422_; _758_i < _9423_; ++_758_i)
              {
                {
                  _ptr__724_accafeidxs[safe_add( _731_afetgtptr,_758_i )] = _ptr__723_afeidxs[safe_add( _732_afeptr,_758_i )];
                  _ptr__725_accb[safe_add( _731_afetgtptr,_758_i )] = _ptr__757_b[_758_i];
                }
              }
              if ((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::EqualTo))
              {
                {
                  _ptr__726_domidxs[_730_acctermptr] = task_append_domain_rzero(_742_nelem);
                }
              }
              else if((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::LessThan))
              {
                {
                  _ptr__726_domidxs[_730_acctermptr] = task_append_domain_rneg(_742_nelem);
                }
              }
              else if((mosek::fusion::p_DJCDomain::_get_impl(_735_dom)->dom == ::mosek::fusion::DJCDomainType::GreaterThan))
              {
                {
                  _ptr__726_domidxs[_730_acctermptr] = task_append_domain_rpos(_742_nelem);
                }
              }
              else
              {
                {
                  _ptr__726_domidxs[_730_acctermptr] = task_append_domain_r(_742_nelem);
                }
              }
              _ptr__727_termsize[_733_term_i] += (int)1;
              _730_acctermptr += (int)1;
              _731_afetgtptr += _742_nelem;
            }
          }
          else
          {
            {
              throw ::mosek::fusion::UnexpectedError(std::string ("Not implemented: Non-linear conic DJC constraints"));
            }
          }
          if ((_744_ncodeatom > (int)0))
          {
            {
              std::vector<int32_t> _9424_;
              for (int32_t _759_i = (int)0, _9425_ = _742_nelem; _759_i < _9425_ ; ++_759_i)
              {
                _9424_.push_back(_ptr__723_afeidxs[safe_add( _732_afeptr,_759_i )]);
              }
              auto _9426_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9424_.size()),_9424_.begin(),_9424_.end()));
              afe_blocks->replace_row_code(rs,_9426_,_746_ptr_base,_747_nidxs_base,_750_codeptr,_749_code,_751_cconst);
            }
          }
          {}
          _732_afeptr += _742_nelem;
          ++ _729_term_index;
        }
      }
    }
  }
  int64_t _760_djcidx = task_append_djc((int64_t)((int)1));
  std::vector<int64_t> _9427_;
  for (int32_t _761_i = (int)0, _9428_ = (int)((_724_accafeidxs)->size(0)); _761_i < _9428_ ; ++_761_i)
  {
    _9427_.push_back(_ptr__724_accafeidxs[_761_i]);
  }
  auto _9429_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9427_.size()),_9427_.begin(),_9427_.end()));
  task_putdjcslice(_760_djcidx,safe_add( _760_djcidx,(int)1 ),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >{(int64_t)((int64_t)_717_n)}),_727_termsize,_726_domidxs,_9429_,_725_accb);
  if (((int)((_715_name).size()) > (int)0))
  {
    {
      task_djc_name(_760_djcidx,::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(_715_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["))->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"))->toString());
    }
  }
  {}
  return ::mosek::fusion::p_Disjunction::_new_Disjunction(_760_djcidx);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4984:6-92
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _762_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_762_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _762_terms) { return __mosek_2fusion_2Model__disjunction(_762_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > > _762_terms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),_762_terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4982:6-111
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _763_djcterms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_763_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _763_djcterms) { return __mosek_2fusion_2Model__disjunction(_763_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _763_djcterms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(mosek::fusion::p_DisjunctionTerms::_get_impl(_763_djcterms)->terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4980:6-129
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _764_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _765_djcterms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_764_name,_765_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _764_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _765_djcterms) { return __mosek_2fusion_2Model__disjunction(_764_name,_765_djcterms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _764_name,monty::rc_ptr< ::mosek::fusion::DisjunctionTerms > _765_djcterms)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_764_name,mosek::fusion::p_DisjunctionTerms::_get_impl(_765_djcterms)->terms);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4978:6-157
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _766_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _767_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_766_name,_767_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _766_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _767_terms) { return __mosek_2fusion_2Model__disjunction(_766_name,_767_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _766_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _767_terms)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9430_;
  for (int32_t _768_i = (int)0, _9431_ = (int)((_767_terms)->size(0)); _768_i < _9431_ ; ++_768_i)
  {
    _9430_.push_back(((*_767_terms)(_768_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9432_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9430_.size()),_9430_.begin(),_9430_.end()));
  return _pubthis->__mosek_2fusion_2Model__disjunction(_766_name,_9432_);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4976:6-142
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _769_terms) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_769_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _769_terms) { return __mosek_2fusion_2Model__disjunction(_769_terms); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ExprDomain >,1 > > _769_terms)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Term >> _9433_;
  for (int32_t _770_i = (int)0, _9434_ = (int)((_769_terms)->size(0)); _770_i < _9434_ ; ++_770_i)
  {
    _9433_.push_back(((*_769_terms)(_770_i))->__mosek_2fusion_2ExprDomain__toDJCTerm());
  }
  auto _9435_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >(monty::shape(_9433_.size()),_9433_.begin(),_9433_.end()));
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),_9435_);
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4974:6-128
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _771_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _772_term) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_771_name,_772_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _771_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _772_term) { return __mosek_2fusion_2Model__disjunction(_771_name,_772_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _771_name,monty::rc_ptr< ::mosek::fusion::ExprDomain > _772_term)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_771_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_772_term->__mosek_2fusion_2ExprDomain__toDJCTerm().get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4972:6-113
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _773_term) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_773_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _773_term) { return __mosek_2fusion_2Model__disjunction(_773_term); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::ExprDomain > _773_term)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_773_term->__mosek_2fusion_2ExprDomain__toDJCTerm().get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4969:6-115
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _774_t1,monty::rc_ptr< ::mosek::fusion::Term > _775_t2,monty::rc_ptr< ::mosek::fusion::Term > _776_t3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_774_t1,_775_t2,_776_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _774_t1,monty::rc_ptr< ::mosek::fusion::Term > _775_t2,monty::rc_ptr< ::mosek::fusion::Term > _776_t3) { return __mosek_2fusion_2Model__disjunction(_774_t1,_775_t2,_776_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _774_t1,monty::rc_ptr< ::mosek::fusion::Term > _775_t2,monty::rc_ptr< ::mosek::fusion::Term > _776_t3)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_774_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_775_t2.get()),::monty::rc_ptr<::mosek::fusion::Term>(_776_t3.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4968:6-103
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _777_t1,monty::rc_ptr< ::mosek::fusion::Term > _778_t2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_777_t1,_778_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _777_t1,monty::rc_ptr< ::mosek::fusion::Term > _778_t2) { return __mosek_2fusion_2Model__disjunction(_777_t1,_778_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _777_t1,monty::rc_ptr< ::mosek::fusion::Term > _778_t2)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_777_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_778_t2.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4967:6-91
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _779_t1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_779_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(monty::rc_ptr< ::mosek::fusion::Term > _779_t1) { return __mosek_2fusion_2Model__disjunction(_779_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(monty::rc_ptr< ::mosek::fusion::Term > _779_t1)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(std::string (""),std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_779_t1.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4965:6-129
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _780_name,monty::rc_ptr< ::mosek::fusion::Term > _781_t1,monty::rc_ptr< ::mosek::fusion::Term > _782_t2,monty::rc_ptr< ::mosek::fusion::Term > _783_t3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_780_name,_781_t1,_782_t2,_783_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _780_name,monty::rc_ptr< ::mosek::fusion::Term > _781_t1,monty::rc_ptr< ::mosek::fusion::Term > _782_t2,monty::rc_ptr< ::mosek::fusion::Term > _783_t3) { return __mosek_2fusion_2Model__disjunction(_780_name,_781_t1,_782_t2,_783_t3); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _780_name,monty::rc_ptr< ::mosek::fusion::Term > _781_t1,monty::rc_ptr< ::mosek::fusion::Term > _782_t2,monty::rc_ptr< ::mosek::fusion::Term > _783_t3)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_780_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_781_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_782_t2.get()),::monty::rc_ptr<::mosek::fusion::Term>(_783_t3.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4964:6-117
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _784_name,monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_784_name,_785_t1,_786_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _784_name,monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2) { return __mosek_2fusion_2Model__disjunction(_784_name,_785_t1,_786_t2); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _784_name,monty::rc_ptr< ::mosek::fusion::Term > _785_t1,monty::rc_ptr< ::mosek::fusion::Term > _786_t2)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_784_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_785_t1.get()),::monty::rc_ptr<::mosek::fusion::Term>(_786_t2.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.disjunction
// Method mosek.fusion.Model.disjunction @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4963:6-105
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model :: __mosek_2fusion_2Model__disjunction(const std::string & _787_name,monty::rc_ptr< ::mosek::fusion::Term > _788_t1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__disjunction(_787_name,_788_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::Model::disjunction(const std::string &  _787_name,monty::rc_ptr< ::mosek::fusion::Term > _788_t1) { return __mosek_2fusion_2Model__disjunction(_787_name,_788_t1); }
monty::rc_ptr< ::mosek::fusion::Disjunction > mosek::fusion::p_Model::__mosek_2fusion_2Model__disjunction(const std::string & _787_name,monty::rc_ptr< ::mosek::fusion::Term > _788_t1)
{
  return _pubthis->__mosek_2fusion_2Model__disjunction(_787_name,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Term >,1 >{::monty::rc_ptr<::mosek::fusion::Term>(_788_t1.get())}));
}
// End mosek.fusion.Model.disjunction

// Begin mosek.fusion.Model.formstConstr
// Method mosek.fusion.Model.formstConstr @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4958:6-4961:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_Model::__mosek_2fusion_2Model__formstConstr(monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _789_sb,std::shared_ptr< monty::ndarray< int32_t,1 > > _790_shape,std::shared_ptr< monty::ndarray< int32_t,1 > > _791_idxs)
{
  _checked_ptr_1<int32_t> _ptr__790_shape(_790_shape ? _790_shape->raw() : nullptr, _790_shape ? _790_shape->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__791_idxs(_791_idxs ? _791_idxs->raw() : nullptr, _791_idxs ? _791_idxs->size(0) : 0);
  return _789_sb;
}
// End mosek.fusion.Model.formstConstr

// Begin mosek.fusion.Model.acon_release
// Method mosek.fusion.Model.acon_release @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4848:6-4867:7
void mosek::fusion::p_Model::acon_release(int32_t _792_id)
{
  int32_t _793_num = acon_block_map->blocksize(_792_id);
  if ((_793_num >= (int)0))
  {
    {
      int32_t _794_numcone = acc_block_map->blocksize(((*acon_acc)(_792_id)));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _795_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_794_numcone)));
      _checked_ptr_1<int32_t> _ptr__795_accidxs(_795_accidxs ? _795_accidxs->raw() : nullptr,_795_accidxs ? _795_accidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _796_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_793_num)));
      _checked_ptr_1<int32_t> _ptr__796_afeidxs(_796_afeidxs ? _796_afeidxs->raw() : nullptr,_796_afeidxs ? _796_afeidxs->size(0) : 0);
      acc_block_map->get(((*acon_acc)(_792_id)),_795_accidxs,(int)0);
      acc_block_map->free(((*acon_acc)(_792_id)));
      afe_blocks->get(((*acon_afe)(_792_id)),_796_afeidxs,(int)0);
      afe_release(((*acon_afe)(_792_id)));
      acon_block_map->free(_792_id);
      std::vector<int64_t> _9436_;
      for (int32_t _797_i = (int)0, _9437_ = _793_num; _797_i < _9437_ ; ++_797_i)
      {
        _9436_.push_back(_ptr__796_afeidxs[_797_i]);
      }
      auto _9438_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9436_.size()),_9436_.begin(),_9436_.end()));
      task_clearafelist(_9438_);
      std::vector<int64_t> _9439_;
      for (int32_t _798_i = (int)0, _9440_ = _794_numcone; _798_i < _9440_ ; ++_798_i)
      {
        _9439_.push_back(_ptr__795_accidxs[_798_i]);
      }
      auto _9441_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9439_.size()),_9439_.begin(),_9439_.end()));
      std::vector<int64_t> _9442_;
      for (int32_t _799_i = (int)0, _9443_ = _794_numcone; _799_i < _9443_ ; ++_799_i)
      {
        _9442_.push_back((int)0);
      }
      auto _9444_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9442_.size()),_9442_.begin(),_9442_.end()));
      task_putacclist(_9441_,_9444_,std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)0))));
    }
  }
  {}
}
// End mosek.fusion.Model.acon_release

// Begin mosek.fusion.Model.acon_allocate
// Method mosek.fusion.Model.acon_allocate @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4767:6-4846:7
int32_t mosek::fusion::p_Model::acon_allocate(int64_t _800_domidx,int32_t _801_conesize,int32_t _802_numcone,std::shared_ptr< monty::ndarray< double,1 > > _803_g,std::shared_ptr< monty::ndarray< int32_t,1 > > _804_afeidxs,std::shared_ptr< monty::ndarray< int32_t,1 > > _805_accidxs)
{
  _checked_ptr_1<double> _ptr__803_g(_803_g ? _803_g->raw() : nullptr, _803_g ? _803_g->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__804_afeidxs(_804_afeidxs ? _804_afeidxs->raw() : nullptr, _804_afeidxs ? _804_afeidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__805_accidxs(_805_accidxs ? _805_accidxs->raw() : nullptr, _805_accidxs ? _805_accidxs->size(0) : 0);
  int32_t _806_n = safe_mul( _802_numcone,_801_conesize );
  std::shared_ptr< monty::ndarray< int32_t,1 > > _807_aconidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_806_n)));
  _checked_ptr_1<int32_t> _ptr__807_aconidxs(_807_aconidxs ? _807_aconidxs->raw() : nullptr,_807_aconidxs ? _807_aconidxs->size(0) : 0);
  int32_t _808_aconid = acon_block_map->alloc(_806_n);
  int32_t _809_accid = acc_block_map->alloc(_802_numcone);
  int32_t _810_afeid = afe_allocate(_804_afeidxs);
  acon_block_map->get(_808_aconid,_807_aconidxs,(int)0);
  acc_block_map->get(_809_accid,_805_accidxs,(int)0);
  int32_t _811_acon_block_cap = acon_block_map->block_capacity();
  int32_t _812_acon_cap = acon_block_map->capacity();
  if ((_811_acon_block_cap > (int)((acon_acc)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _813_tmpacc = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_811_acon_block_cap)));
      _checked_ptr_1<int32_t> _ptr__813_tmpacc(_813_tmpacc ? _813_tmpacc->raw() : nullptr,_813_tmpacc ? _813_tmpacc->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _814_tmpafe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_811_acon_block_cap)));
      _checked_ptr_1<int32_t> _ptr__814_tmpafe(_814_tmpafe ? _814_tmpafe->raw() : nullptr,_814_tmpafe ? _814_tmpafe->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(acon_acc,(int)0,_813_tmpacc,(int)0,(int)((acon_acc)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_afe,(int)0,_814_tmpafe,(int)0,(int)((acon_afe)->size(0)));
      acon_acc = _813_tmpacc;
      acon_afe = _814_tmpafe;
    }
  }
  {}
  ((*acon_acc)(_808_aconid)) = _809_accid;
  ((*acon_afe)(_808_aconid)) = _810_afeid;
  if ((_808_aconid >= (int)((acons)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9445_;
      for (int32_t _816_i = (int)0, _9446_ = safe_mul( (int)((acons)->size(0)),(int)2 ); _816_i < _9446_ ; ++_816_i)
      {
        _9445_.push_back(nullptr);
      }
      auto _9447_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9445_.size()),_9445_.begin(),_9445_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > > _815_newacons = _9447_;
      int32_t _9448_ = (int)0;
      int32_t _9449_ = (int)((acons)->size(0));
      for (int32_t _817_i = _9448_; _817_i < _9449_; ++_817_i)
      {
        {
          ((*_815_newacons)(_817_i)) = ((*acons)(_817_i));
        }
      }
      acons = _815_newacons;
    }
  }
  {}
  if ((_812_acon_cap > (int)((acon_elm_accid)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _818_tmp_accid = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_812_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__818_tmp_accid(_818_tmp_accid ? _818_tmp_accid->raw() : nullptr,_818_tmp_accid ? _818_tmp_accid->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _819_tmp_scale = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_812_acon_cap)));
      _checked_ptr_1<double> _ptr__819_tmp_scale(_819_tmp_scale ? _819_tmp_scale->raw() : nullptr,_819_tmp_scale ? _819_tmp_scale->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _820_tmp_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_812_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__820_tmp_ofs(_820_tmp_ofs ? _820_tmp_ofs->raw() : nullptr,_820_tmp_ofs ? _820_tmp_ofs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _821_tmp_afe = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_812_acon_cap)));
      _checked_ptr_1<int32_t> _ptr__821_tmp_afe(_821_tmp_afe ? _821_tmp_afe->raw() : nullptr,_821_tmp_afe ? _821_tmp_afe->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_accid,(int)0,_818_tmp_accid,(int)0,(int)((acon_elm_accid)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_scale,(int)0,_819_tmp_scale,(int)0,(int)((acon_elm_scale)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_ofs,(int)0,_820_tmp_ofs,(int)0,(int)((acon_elm_ofs)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(acon_elm_afe,(int)0,_821_tmp_afe,(int)0,(int)((acon_elm_afe)->size(0)));
      acon_elm_accid = _818_tmp_accid;
      acon_elm_ofs = _820_tmp_ofs;
      acon_elm_afe = _821_tmp_afe;
      acon_elm_scale = _819_tmp_scale;
    }
  }
  {}
  int32_t _822_maxaccidx = (int)0;
  int32_t _9450_ = (int)0;
  int32_t _9451_ = (int)((_805_accidxs)->size(0));
  for (int32_t _823_i = _9450_; _823_i < _9451_; ++_823_i)
  {
    {
      if ((_ptr__805_accidxs[_823_i] > _822_maxaccidx))
      {
        {
          _822_maxaccidx = _ptr__805_accidxs[_823_i];
        }
      }
      {}
    }
  }
  _822_maxaccidx += (int)1;
  int32_t _824_nacc = task_numacc();
  if ((_824_nacc < _822_maxaccidx))
  {
    {
      task_append_accs((int64_t)0,(_822_maxaccidx - _824_nacc),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)0))));
    }
  }
  {}
  std::vector<int64_t> _9452_;
  for (int32_t _825_i = (int)0, _9453_ = (int)((_805_accidxs)->size(0)); _825_i < _9453_ ; ++_825_i)
  {
    _9452_.push_back(_ptr__805_accidxs[_825_i]);
  }
  auto _9454_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9452_.size()),_9452_.begin(),_9452_.end()));
  std::vector<int64_t> _9455_;
  for (int32_t _826_i = (int)0, _9456_ = (int)((_805_accidxs)->size(0)); _826_i < _9456_ ; ++_826_i)
  {
    _9455_.push_back(_800_domidx);
  }
  auto _9457_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9455_.size()),_9455_.begin(),_9455_.end()));
  std::vector<int64_t> _9458_;
  for (int32_t _827_i = (int)0, _9459_ = (int)((_804_afeidxs)->size(0)); _827_i < _9459_ ; ++_827_i)
  {
    _9458_.push_back(_ptr__804_afeidxs[_827_i]);
  }
  auto _9460_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9458_.size()),_9458_.begin(),_9458_.end()));
  task_putacclist(_9454_,_9457_,_9460_,_803_g);
  {
    int32_t _828_k = (int)0;
    int32_t _9461_ = (int)0;
    int32_t _9462_ = _802_numcone;
    for (int32_t _829_i = _9461_; _829_i < _9462_; ++_829_i)
    {
      {
        int32_t _9463_ = (int)0;
        int32_t _9464_ = _801_conesize;
        for (int32_t _830_j = _9463_; _830_j < _9464_; ++_830_j)
        {
          {
            ((*acon_elm_accid)(_ptr__807_aconidxs[_828_k])) = _ptr__805_accidxs[_829_i];
            ((*acon_elm_ofs)(_ptr__807_aconidxs[_828_k])) = _830_j;
            ((*acon_elm_afe)(_ptr__807_aconidxs[_828_k])) = _ptr__804_afeidxs[_828_k];
            ((*acon_elm_scale)(_ptr__807_aconidxs[_828_k])) = 1.0;
            ++ _828_k;
          }
        }
      }
    }
  }
  return _808_aconid;
}
// End mosek.fusion.Model.acon_allocate

// Begin mosek.fusion.Model.afe_release
// Method mosek.fusion.Model.afe_release @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4759:6-4765:7
void mosek::fusion::p_Model::afe_release(int32_t _831_id)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _832_nidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(afe_blocks->blocksize(_831_id))));
  _checked_ptr_1<int32_t> _ptr__832_nidxs(_832_nidxs ? _832_nidxs->raw() : nullptr,_832_nidxs ? _832_nidxs->size(0) : 0);
  afe_blocks->release(_831_id,_832_nidxs);
  std::vector<int64_t> _9465_;
  for (int32_t _833_i = (int)0, _9466_ = (int)((_832_nidxs)->size(0)); _833_i < _9466_ ; ++_833_i)
  {
    _9465_.push_back(_ptr__832_nidxs[_833_i]);
  }
  auto _9467_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9465_.size()),_9465_.begin(),_9465_.end()));
  task_clearafelist(_9467_);
}
// End mosek.fusion.Model.afe_release

// Begin mosek.fusion.Model.afe_allocate
// Method mosek.fusion.Model.afe_allocate @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4744:6-4757:7
int32_t mosek::fusion::p_Model::afe_allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _834_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__834_nativeidxs(_834_nativeidxs ? _834_nativeidxs->raw() : nullptr, _834_nativeidxs ? _834_nativeidxs->size(0) : 0);
  int32_t _835_afeid = afe_blocks->allocate(_834_nativeidxs);
  if (((int)((_834_nativeidxs)->size(0)) > (int)0))
  {
    {
      int32_t _836_tasknum = task_numafe();
      int32_t _837_maxidx = _ptr__834_nativeidxs[(int)0];
      int32_t _9468_ = (int)1;
      int32_t _9469_ = (int)((_834_nativeidxs)->size(0));
      for (int32_t _838_i = _9468_; _838_i < _9469_; ++_838_i)
      {
        {
          if ((_ptr__834_nativeidxs[_838_i] > _837_maxidx))
          {
            {
              _837_maxidx = _ptr__834_nativeidxs[_838_i];
            }
          }
          {}
        }
      }
      if ((safe_add( _837_maxidx,(int)1 ) > _836_tasknum))
      {
        {
          int32_t _839_n = (safe_add( _837_maxidx,(int)1 ) - _836_tasknum);
          task_append_afes((int64_t)(_839_n));
          task_numaferow += _839_n;
        }
      }
      {}
    }
  }
  {}
  return _835_afeid;
}
// End mosek.fusion.Model.afe_allocate

// Begin mosek.fusion.Model.con_release
// Method mosek.fusion.Model.con_release @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4729:6-4739:7
void mosek::fusion::p_Model::con_release(int32_t _840_id)
{
  int32_t _841_num = con_blocks->blocksize(_840_id);
  if ((_841_num >= (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _842_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_841_num)));
      _checked_ptr_1<int32_t> _ptr__842_nativeidxs(_842_nativeidxs ? _842_nativeidxs->raw() : nullptr,_842_nativeidxs ? _842_nativeidxs->size(0) : 0);
      con_blocks->release(_840_id,_842_nativeidxs);
      ((*cons)(_840_id)) = nullptr;
      task_con_putboundlist_fr(_842_nativeidxs);
    }
  }
  {}
}
// End mosek.fusion.Model.con_release

// Begin mosek.fusion.Model.con_allocate
// Method mosek.fusion.Model.con_allocate @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4709:6-4727:7
int32_t mosek::fusion::p_Model::con_allocate(std::shared_ptr< monty::ndarray< int32_t,1 > > _843_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__843_nativeidxs(_843_nativeidxs ? _843_nativeidxs->raw() : nullptr, _843_nativeidxs ? _843_nativeidxs->size(0) : 0);
  int32_t _844_conid = con_blocks->allocate(_843_nativeidxs);
  if (((int)((_843_nativeidxs)->size(0)) > (int)0))
  {
    {
      int32_t _845_maxidx = _ptr__843_nativeidxs[(int)0];
      int32_t _9470_ = (int)1;
      int32_t _9471_ = (int)((_843_nativeidxs)->size(0));
      for (int32_t _846_i = _9470_; _846_i < _9471_; ++_846_i)
      {
        {
          if ((_ptr__843_nativeidxs[_846_i] > _845_maxidx))
          {
            {
              _845_maxidx = _ptr__843_nativeidxs[_846_i];
            }
          }
          {}
        }
      }
      if ((safe_add( _845_maxidx,(int)1 ) > num_task_con))
      {
        {
          int32_t _847_n = (safe_add( _845_maxidx,(int)1 ) - num_task_con);
          task_append_con(_847_n);
          num_task_con += _847_n;
        }
      }
      {}
    }
  }
  {}
  if ((_844_conid >= (int)((cons)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelConstraint >> _9472_;
      for (int32_t _849_i = (int)0, _9473_ = safe_mul( (int)((cons)->size(0)),(int)2 ); _849_i < _9473_ ; ++_849_i)
      {
        _9472_.push_back(nullptr);
      }
      auto _9474_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 >(monty::shape(_9472_.size()),_9472_.begin(),_9472_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelConstraint >,1 > > _848_newcons = _9474_;
      int32_t _9475_ = (int)0;
      int32_t _9476_ = (int)((cons)->size(0));
      for (int32_t _850_i = _9475_; _850_i < _9476_; ++_850_i)
      {
        {
          ((*_848_newcons)(_850_i)) = ((*cons)(_850_i));
        }
      }
      cons = _848_newcons;
    }
  }
  {}
  return _844_conid;
}
// End mosek.fusion.Model.con_allocate

// Begin mosek.fusion.Model.barvar_alloc
// Method mosek.fusion.Model.barvar_alloc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4602:6-4703:7
int32_t mosek::fusion::p_Model::barvar_alloc(int32_t _851_conedim,int32_t _852_numcone,std::shared_ptr< monty::ndarray< int32_t,1 > > _853_barvaridxs,std::shared_ptr< monty::ndarray< int64_t,1 > > _854_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__853_barvaridxs(_853_barvaridxs ? _853_barvaridxs->raw() : nullptr, _853_barvaridxs ? _853_barvaridxs->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__854_nativeidxs(_854_nativeidxs ? _854_nativeidxs->raw() : nullptr, _854_nativeidxs ? _854_nativeidxs->size(0) : 0);
  int32_t _855_barvarid = barvar_num;
  ++ barvar_num;
  int32_t _856_conesize = (safe_mul( _851_conedim,safe_add( _851_conedim,(int)1 ) ) / (int)2);
  int32_t _857_domsize = safe_mul( _856_conesize,_852_numcone );
  int32_t _858_block_num = ((*barvar_block_ptr)(_855_barvarid));
  int32_t _859_block_cap = (int)((barvar_block_dim)->size(0));
  int32_t _860_block_elm_num = ((*barvar_block_elm_ptr)(_855_barvarid));
  int32_t _861_block_elm_cap = (int)((barvar_block_elm_barj)->size(0));
  if ((_859_block_cap < safe_add( _858_block_num,(int)1 )))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _862_new_barvar_block_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ))));
      _checked_ptr_1<int32_t> _ptr__862_new_barvar_block_dim(_862_new_barvar_block_dim ? _862_new_barvar_block_dim->raw() : nullptr,_862_new_barvar_block_dim ? _862_new_barvar_block_dim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _863_new_barvar_block_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__863_new_barvar_block_ptr(_863_new_barvar_block_ptr ? _863_new_barvar_block_ptr->raw() : nullptr,_863_new_barvar_block_ptr ? _863_new_barvar_block_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _864_new_barvar_block_elm_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__864_new_barvar_block_elm_ptr(_864_new_barvar_block_elm_ptr ? _864_new_barvar_block_elm_ptr->raw() : nullptr,_864_new_barvar_block_elm_ptr ? _864_new_barvar_block_elm_ptr->size(0) : 0);
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9477_;
      for (int32_t _866_i = (int)0, _9478_ = safe_add( safe_mul( (int)((barvar_block_dim)->size(0)),(int)2 ),(int)1 ); _866_i < _9478_ ; ++_866_i)
      {
        _9477_.push_back(nullptr);
      }
      auto _9479_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9477_.size()),_9477_.begin(),_9477_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > > _865_new_barvars = _9479_;
      int32_t _9480_ = (int)0;
      int32_t _9481_ = (int)((barvars)->size(0));
      for (int32_t _867_i = _9480_; _867_i < _9481_; ++_867_i)
      {
        {
          ((*_865_new_barvars)(_867_i)) = ((*barvars)(_867_i));
        }
      }
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_dim,(int)0,_862_new_barvar_block_dim,(int)0,(int)((barvar_block_dim)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_ptr,(int)0,_863_new_barvar_block_ptr,(int)0,(int)((barvar_block_ptr)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_ptr,(int)0,_864_new_barvar_block_elm_ptr,(int)0,(int)((barvar_block_elm_ptr)->size(0)));
      barvar_block_dim = _862_new_barvar_block_dim;
      barvar_block_ptr = _863_new_barvar_block_ptr;
      barvar_block_elm_ptr = _864_new_barvar_block_elm_ptr;
      barvars = _865_new_barvars;
    }
  }
  {}
  if ((_861_block_elm_cap < safe_add( _860_block_elm_num,_857_domsize )))
  {
    {
      int32_t _868_addnum = (safe_add( _861_block_elm_cap,_857_domsize ) - _860_block_elm_num);
      if ((_868_addnum < _861_block_elm_cap))
      {
        {
          _868_addnum = _861_block_elm_cap;
        }
      }
      {}
      int32_t _869_newlen = safe_add( _861_block_elm_cap,_868_addnum );
      std::shared_ptr< monty::ndarray< int32_t,1 > > _870_new_barvar_block_elm_barj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_869_newlen)));
      _checked_ptr_1<int32_t> _ptr__870_new_barvar_block_elm_barj(_870_new_barvar_block_elm_barj ? _870_new_barvar_block_elm_barj->raw() : nullptr,_870_new_barvar_block_elm_barj ? _870_new_barvar_block_elm_barj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _871_new_barvar_block_elm_i = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_869_newlen)));
      _checked_ptr_1<int32_t> _ptr__871_new_barvar_block_elm_i(_871_new_barvar_block_elm_i ? _871_new_barvar_block_elm_i->raw() : nullptr,_871_new_barvar_block_elm_i ? _871_new_barvar_block_elm_i->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _872_new_barvar_block_elm_j = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_869_newlen)));
      _checked_ptr_1<int32_t> _ptr__872_new_barvar_block_elm_j(_872_new_barvar_block_elm_j ? _872_new_barvar_block_elm_j->raw() : nullptr,_872_new_barvar_block_elm_j ? _872_new_barvar_block_elm_j->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_barj,(int)0,_870_new_barvar_block_elm_barj,(int)0,(int)((barvar_block_elm_barj)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_i,(int)0,_871_new_barvar_block_elm_i,(int)0,(int)((barvar_block_elm_i)->size(0)));
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_block_elm_j,(int)0,_872_new_barvar_block_elm_j,(int)0,(int)((barvar_block_elm_j)->size(0)));
      barvar_block_elm_barj = _870_new_barvar_block_elm_barj;
      barvar_block_elm_i = _871_new_barvar_block_elm_i;
      barvar_block_elm_j = _872_new_barvar_block_elm_j;
    }
  }
  {}
  int32_t _873_firstcone = task_numbarvar();
  int32_t _9482_ = (int)0;
  int32_t _9483_ = _852_numcone;
  for (int32_t _874_i = _9482_; _874_i < _9483_; ++_874_i)
  {
    {
      _ptr__853_barvaridxs[_874_i] = safe_add( _873_firstcone,_874_i );
    }
  }
  if ((safe_add( _873_firstcone,_852_numcone ) >= (int)((barvar_dim)->size(0))))
  {
    {
      int32_t _875_addnum = (safe_add( _873_firstcone,_852_numcone ) - (int)((barvar_dim)->size(0)));
      if ((_875_addnum < (int)((barvar_dim)->size(0))))
      {
        {
          _875_addnum = (int)((barvar_dim)->size(0));
        }
      }
      {}
      int32_t _876_newnum = safe_add( _875_addnum,(int)((barvar_dim)->size(0)) );
      std::shared_ptr< monty::ndarray< int32_t,1 > > _877_new_barvar_dim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_876_newnum)));
      _checked_ptr_1<int32_t> _ptr__877_new_barvar_dim(_877_new_barvar_dim ? _877_new_barvar_dim->raw() : nullptr,_877_new_barvar_dim ? _877_new_barvar_dim->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(barvar_dim,(int)0,_877_new_barvar_dim,(int)0,(int)((barvar_dim)->size(0)));
      barvar_dim = _877_new_barvar_dim;
    }
  }
  {}
  ((*barvar_block_dim)(_855_barvarid)) = _851_conedim;
  ((*barvar_block_ptr)(safe_add( _855_barvarid,(int)1 ))) = safe_add( ((*barvar_block_ptr)(_855_barvarid)),_852_numcone );
  ((*barvar_block_elm_ptr)(safe_add( _855_barvarid,(int)1 ))) = safe_add( ((*barvar_block_elm_ptr)(_855_barvarid)),_857_domsize );
  {
    int32_t _878_k = (int)0;
    int32_t _879_b = ((*barvar_block_elm_ptr)(_855_barvarid));
    int32_t _9484_ = (int)0;
    int32_t _9485_ = _852_numcone;
    for (int32_t _880_i = _9484_; _880_i < _9485_; ++_880_i)
    {
      {
        int32_t _9486_ = (int)0;
        int32_t _9487_ = _851_conedim;
        for (int32_t _881_j1 = _9486_; _881_j1 < _9487_; ++_881_j1)
        {
          {
            int32_t _9488_ = _881_j1;
            int32_t _9489_ = _851_conedim;
            for (int32_t _882_j0 = _9488_; _882_j0 < _9489_; ++_882_j0)
            {
              {
                ((*barvar_block_elm_barj)(safe_add( _879_b,_878_k ))) = safe_add( _873_firstcone,_880_i );
                ((*barvar_block_elm_i)(safe_add( _879_b,_878_k ))) = _882_j0;
                ((*barvar_block_elm_j)(safe_add( _879_b,_878_k ))) = _881_j1;
                _ptr__854_nativeidxs[_878_k] = safe_add( _879_b,_878_k );
                ++ _878_k;
              }
            }
          }
        }
      }
    }
    int32_t _9490_ = (int)0;
    int32_t _9491_ = _852_numcone;
    for (int32_t _883_i = _9490_; _883_i < _9491_; ++_883_i)
    {
      {
        ((*barvar_dim)(safe_add( _873_firstcone,_883_i ))) = _851_conedim;
      }
    }
  }
  int32_t _884_barvaridx = task_append_barvar(_851_conedim,_852_numcone);
  return _855_barvarid;
}
// End mosek.fusion.Model.barvar_alloc

// Begin mosek.fusion.Model.conicvar_alloc
// Method mosek.fusion.Model.conicvar_alloc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4569:6-4600:7
int32_t mosek::fusion::p_Model::conicvar_alloc(int64_t _885_domidx,int32_t _886_conesize,int32_t _887_numcone,std::shared_ptr< monty::ndarray< int32_t,1 > > _888_accidxs,std::shared_ptr< monty::ndarray< int32_t,1 > > _889_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__888_accidxs(_888_accidxs ? _888_accidxs->raw() : nullptr, _888_accidxs ? _888_accidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__889_nativeidxs(_889_nativeidxs ? _889_nativeidxs->raw() : nullptr, _889_nativeidxs ? _889_nativeidxs->size(0) : 0);
  int32_t _890_n = safe_mul( _886_conesize,_887_numcone );
  int32_t _891_varid = linearvar_alloc(_890_n,_889_nativeidxs);
  std::shared_ptr< monty::ndarray< double,1 > > _892_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_890_n)));
  _checked_ptr_1<double> _ptr__892_g(_892_g ? _892_g->raw() : nullptr,_892_g ? _892_g->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _893_conelmidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_890_n)));
  _checked_ptr_1<int32_t> _ptr__893_conelmidx(_893_conelmidx ? _893_conelmidx->raw() : nullptr,_893_conelmidx ? _893_conelmidx->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _894_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_890_n)));
  _checked_ptr_1<int32_t> _ptr__894_afeidxs(_894_afeidxs ? _894_afeidxs->raw() : nullptr,_894_afeidxs ? _894_afeidxs->size(0) : 0);
  int32_t _895_conid = acon_allocate(_885_domidx,_886_conesize,_887_numcone,_892_g,_894_afeidxs,_888_accidxs);
  ((*var_block_acc_id)(_891_varid)) = _895_conid;
  int32_t _9492_ = (int)0;
  int32_t _9493_ = _890_n;
  for (int32_t _896_i = _9492_; _896_i < _9493_; ++_896_i)
  {
    {
      ((*var_elm_acc_idx)(_ptr__889_nativeidxs[_896_i])) = _ptr__888_accidxs[(_896_i / _886_conesize)];
      ((*var_elm_acc_ofs)(_ptr__889_nativeidxs[_896_i])) = (_896_i % _886_conesize);
    }
  }
  std::vector<int64_t> _9494_;
  for (int32_t _897_i = (int)0, _9495_ = (int)((_894_afeidxs)->size(0)); _897_i < _9495_ ; ++_897_i)
  {
    _9494_.push_back(_ptr__894_afeidxs[_897_i]);
  }
  auto _9496_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9494_.size()),_9494_.begin(),_9494_.end()));
  std::vector<int32_t> _9497_;
  for (int32_t _898_i = (int)0, _9498_ = safe_add( _890_n,(int)1 ); _898_i < _9498_ ; ++_898_i)
  {
    _9497_.push_back(_898_i);
  }
  auto _9499_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9497_.size()),_9497_.begin(),_9497_.end()));
  std::vector<double> _9500_;
  for (int32_t _899_i = (int)0, _9501_ = _890_n; _899_i < _9501_ ; ++_899_i)
  {
    _9500_.push_back(1.0);
  }
  auto _9502_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9500_.size()),_9500_.begin(),_9500_.end()));
  std::vector<double> _9503_;
  for (int32_t _900_i = (int)0, _9504_ = _890_n; _900_i < _9504_ ; ++_900_i)
  {
    _9503_.push_back(0.0);
  }
  auto _9505_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9503_.size()),_9503_.begin(),_9503_.end()));
  task_putafeflist(_9496_,_9499_,_889_nativeidxs,_9502_,_9505_);
  return _891_varid;
}
// End mosek.fusion.Model.conicvar_alloc

// Begin mosek.fusion.Model.linearvar_alloc
// Method mosek.fusion.Model.linearvar_alloc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4531:6-4567:7
int32_t mosek::fusion::p_Model::linearvar_alloc(int32_t _901_n,std::shared_ptr< monty::ndarray< int32_t,1 > > _902_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__902_nativeidxs(_902_nativeidxs ? _902_nativeidxs->raw() : nullptr, _902_nativeidxs ? _902_nativeidxs->size(0) : 0);
  int32_t _903_varid = var_block_map->alloc(_901_n);
  int32_t _904_maxvaridx = safe_add( var_block_map->maxidx(_903_varid),(int)1 );
  int32_t _905_cap = var_block_map->capacity();
  int32_t _906_numblocks = var_block_map->block_capacity();
  var_block_map->get(_903_varid,_902_nativeidxs,(int)0);
  if ((task_numvar() < _904_maxvaridx))
  {
    {
      task_append_var((_904_maxvaridx - task_numvar()));
    }
  }
  {}
  if ((_903_varid >= (int)((vars)->size(0))))
  {
    {
      std::vector<monty::rc_ptr< ::mosek::fusion::ModelVariable >> _9506_;
      for (int32_t _908_i = (int)0, _9507_ = safe_mul( (int)((vars)->size(0)),(int)2 ); _908_i < _9507_ ; ++_908_i)
      {
        _9506_.push_back(nullptr);
      }
      auto _9508_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 >(monty::shape(_9506_.size()),_9506_.begin(),_9506_.end()));
      std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::ModelVariable >,1 > > _907_newvars = _9508_;
      int32_t _9509_ = (int)0;
      int32_t _9510_ = (int)((vars)->size(0));
      for (int32_t _909_i = _9509_; _909_i < _9510_; ++_909_i)
      {
        {
          ((*_907_newvars)(_909_i)) = ((*vars)(_909_i));
        }
      }
      vars = _907_newvars;
    }
  }
  {}
  if ((_906_numblocks > (int)((var_block_acc_id)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _910_new_var_block_acc_id = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_906_numblocks)));
      _checked_ptr_1<int32_t> _ptr__910_new_var_block_acc_id(_910_new_var_block_acc_id ? _910_new_var_block_acc_id->raw() : nullptr,_910_new_var_block_acc_id ? _910_new_var_block_acc_id->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_block_acc_id,(int)0,_910_new_var_block_acc_id,(int)0,(int)((var_block_acc_id)->size(0)));
      var_block_acc_id = _910_new_var_block_acc_id;
    }
  }
  {}
  if ((_905_cap > (int)((var_elm_acc_idx)->size(0))))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _911_new_var_elm_acc_idx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_905_cap)));
      _checked_ptr_1<int32_t> _ptr__911_new_var_elm_acc_idx(_911_new_var_elm_acc_idx ? _911_new_var_elm_acc_idx->raw() : nullptr,_911_new_var_elm_acc_idx ? _911_new_var_elm_acc_idx->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_elm_acc_idx,(int)0,_911_new_var_elm_acc_idx,(int)0,(int)((var_elm_acc_idx)->size(0)));
      var_elm_acc_idx = _911_new_var_elm_acc_idx;
      std::shared_ptr< monty::ndarray< int32_t,1 > > _912_new_var_elm_acc_ofs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_905_cap)));
      _checked_ptr_1<int32_t> _ptr__912_new_var_elm_acc_ofs(_912_new_var_elm_acc_ofs ? _912_new_var_elm_acc_ofs->raw() : nullptr,_912_new_var_elm_acc_ofs ? _912_new_var_elm_acc_ofs->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(var_elm_acc_ofs,(int)0,_912_new_var_elm_acc_ofs,(int)0,(int)((var_elm_acc_ofs)->size(0)));
      var_elm_acc_ofs = _912_new_var_elm_acc_ofs;
    }
  }
  {}
  ((*var_block_acc_id)(_903_varid)) = (-(int)1);
  int32_t _9511_ = (int)0;
  int32_t _9512_ = _901_n;
  for (int32_t _913_i = _9511_; _913_i < _9512_; ++_913_i)
  {
    {
      ((*var_elm_acc_idx)(_ptr__902_nativeidxs[_913_i])) = (-(int)1);
    }
  }
  return _903_varid;
}
// End mosek.fusion.Model.linearvar_alloc

// Begin mosek.fusion.Model.make_continuous
// Method mosek.fusion.Model.make_continuous @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4517:6-4528:7
void mosek::fusion::p_Model::make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _914_idxs)
{
  _checked_ptr_1<int64_t> _ptr__914_idxs(_914_idxs ? _914_idxs->raw() : nullptr, _914_idxs ? _914_idxs->size(0) : 0);
  int32_t _915_count = (int)0;
  int32_t _9513_ = (int)0;
  int32_t _9514_ = (int)((_914_idxs)->size(0));
  for (int32_t _916_i = _9513_; _916_i < _9514_; ++_916_i)
  {
    {
      if ((_ptr__914_idxs[_916_i] > (int)0))
      {
        {
          ++ _915_count;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _917_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_915_count)));
  _checked_ptr_1<int32_t> _ptr__917_nativeidxs(_917_nativeidxs ? _917_nativeidxs->raw() : nullptr,_917_nativeidxs ? _917_nativeidxs->size(0) : 0);
  int32_t _918_k = (int)0;
  int32_t _9515_ = (int)0;
  int32_t _9516_ = (int)((_914_idxs)->size(0));
  for (int32_t _919_i = _9515_; _919_i < _9516_; ++_919_i)
  {
    {
      if ((_ptr__914_idxs[_919_i] > (int)0))
      {
        {
          _ptr__917_nativeidxs[_918_k] = (int32_t)_ptr__914_idxs[_919_i];
          ++ _918_k;
        }
      }
      {}
    }
  }
  task_var_putcontlist(_917_nativeidxs);
}
// End mosek.fusion.Model.make_continuous

// Begin mosek.fusion.Model.make_integer
// Method mosek.fusion.Model.make_integer @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4505:6-4515:7
void mosek::fusion::p_Model::make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _920_idxs)
{
  _checked_ptr_1<int64_t> _ptr__920_idxs(_920_idxs ? _920_idxs->raw() : nullptr, _920_idxs ? _920_idxs->size(0) : 0);
  int32_t _921_count = (int)0;
  int32_t _9517_ = (int)0;
  int32_t _9518_ = (int)((_920_idxs)->size(0));
  for (int32_t _922_i = _9517_; _922_i < _9518_; ++_922_i)
  {
    {
      if ((_ptr__920_idxs[_922_i] > (int)0))
      {
        {
          ++ _921_count;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _923_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_921_count)));
  _checked_ptr_1<int32_t> _ptr__923_nativeidxs(_923_nativeidxs ? _923_nativeidxs->raw() : nullptr,_923_nativeidxs ? _923_nativeidxs->size(0) : 0);
  int32_t _924_k = (int)0;
  int32_t _9519_ = (int)0;
  int32_t _9520_ = (int)((_920_idxs)->size(0));
  for (int32_t _925_i = _9519_; _925_i < _9520_; ++_925_i)
  {
    {
      if ((_ptr__920_idxs[_925_i] > (int)0))
      {
        {
          _ptr__923_nativeidxs[_924_k] = (int32_t)_ptr__920_idxs[_925_i];
          ++ _924_k;
        }
      }
      {}
    }
  }
  task_var_putintlist(_923_nativeidxs);
}
// End mosek.fusion.Model.make_integer

// Begin mosek.fusion.Model.putlicensewait
// Method mosek.fusion.Model.putlicensewait @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4503:6-87
void mosek::fusion::Model::putlicensewait
  ( bool _926_wait) { mosek::fusion::p_Model::putlicensewait(_926_wait); }
void mosek::fusion::p_Model::putlicensewait(bool _926_wait)
{
  ::mosek::fusion::p_BaseModel::env_putlicensewait(_926_wait);
}
// End mosek.fusion.Model.putlicensewait

// Begin mosek.fusion.Model.putlicensepath
// Method mosek.fusion.Model.putlicensepath @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4500:6-90
void mosek::fusion::Model::putlicensepath
  ( const std::string & _927_licfile) { mosek::fusion::p_Model::putlicensepath(_927_licfile); }
void mosek::fusion::p_Model::putlicensepath(const std::string & _927_licfile)
{
  ::mosek::fusion::p_BaseModel::env_putlicensepath(_927_licfile);
}
// End mosek.fusion.Model.putlicensepath

// Begin mosek.fusion.Model.putlicensecode
// Method mosek.fusion.Model.putlicensecode @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4496:6-87
void mosek::fusion::Model::putlicensecode
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _928_code) { mosek::fusion::p_Model::putlicensecode(_928_code); }
void mosek::fusion::p_Model::putlicensecode(std::shared_ptr< monty::ndarray< int32_t,1 > > _928_code)
{
  _checked_ptr_1<int32_t> _ptr__928_code(_928_code ? _928_code->raw() : nullptr, _928_code ? _928_code->size(0) : 0);
  ::mosek::fusion::p_BaseModel::env_putlicensecode(_928_code);
}
// End mosek.fusion.Model.putlicensecode

// Begin mosek.fusion.Model.dispose
// Method mosek.fusion.Model.dispose @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4481:6-4490:7
void mosek::fusion::Model :: dispose() { mosek::fusion::p_Model::_get_impl(this)->dispose(); }
void mosek::fusion::p_Model::dispose()
{
  int32_t _9521_ = (int)0;
  int32_t _9522_ = (int)((vars)->size(0));
  for (int32_t _929_i = _9521_; _929_i < _9522_; ++_929_i)
  {
    {
      ((*vars)(_929_i)) = nullptr;
    }
  }
  int32_t _9523_ = (int)0;
  int32_t _9524_ = (int)((cons)->size(0));
  for (int32_t _930_i = _9523_; _930_i < _9524_; ++_930_i)
  {
    {
      ((*cons)(_930_i)) = nullptr;
    }
  }
  int32_t _9525_ = (int)0;
  int32_t _9526_ = (int)((acons)->size(0));
  for (int32_t _931_i = _9525_; _931_i < _9526_; ++_931_i)
  {
    {
      ((*acons)(_931_i)) = nullptr;
    }
  }
  int32_t _9527_ = (int)0;
  int32_t _9528_ = (int)((barvars)->size(0));
  for (int32_t _932_i = _9527_; _932_i < _9528_; ++_932_i)
  {
    {
      ((*barvars)(_932_i)) = nullptr;
    }
  }
  if ((NULL != parameters.get()))
  {
    {
      int32_t _9529_ = (int)0;
      int32_t _9530_ = (int)((parameters)->size(0));
      for (int32_t _933_i = _9529_; _933_i < _9530_; ++_933_i)
      {
        {
          ((*parameters)(_933_i)) = nullptr;
        }
      }
    }
  }
  {}
  ::mosek::fusion::p_BaseModel::dispose();
}
// End mosek.fusion.Model.dispose

// Begin mosek.fusion.Model.getTask
// Method mosek.fusion.Model.getTask @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4461:6-64
MSKtask_t mosek::fusion::Model :: __mosek_2fusion_2Model__getTask() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getTask(); }
MSKtask_t mosek::fusion::Model::getTask() { return __mosek_2fusion_2Model__getTask(); }
MSKtask_t mosek::fusion::p_Model::__mosek_2fusion_2Model__getTask()
{
  return __mosek_2fusion_2BaseModel__task_get();
}
// End mosek.fusion.Model.getTask

// Begin mosek.fusion.Model.getConstraintDuals
// Method mosek.fusion.Model.getConstraintDuals @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4435:6-4450:7
void mosek::fusion::Model :: getConstraintDuals(bool _934_lower,std::shared_ptr< monty::ndarray< int32_t,1 > > _935_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _936_res,int32_t _937_offset) { mosek::fusion::p_Model::_get_impl(this)->getConstraintDuals(_934_lower,_935_nativeidxs,_936_res,_937_offset); }
void mosek::fusion::p_Model::getConstraintDuals(bool _934_lower,std::shared_ptr< monty::ndarray< int32_t,1 > > _935_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _936_res,int32_t _937_offset)
{
  _checked_ptr_1<int32_t> _ptr__935_nativeidxs(_935_nativeidxs ? _935_nativeidxs->raw() : nullptr, _935_nativeidxs ? _935_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__936_res(_936_res ? _936_res->raw() : nullptr, _936_res ? _936_res->size(0) : 0);
  int32_t _938_n = (int)((_935_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< double,1 > > _939_slc = getSolution_slc(::mosek::fusion::SolutionType::Default);
  _checked_ptr_1<double> _ptr__939_slc(_939_slc ? _939_slc->raw() : nullptr,_939_slc ? _939_slc->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _940_suc = getSolution_suc(::mosek::fusion::SolutionType::Default);
  _checked_ptr_1<double> _ptr__940_suc(_940_suc ? _940_suc->raw() : nullptr,_940_suc ? _940_suc->size(0) : 0);
  int32_t _9531_ = (int)0;
  int32_t _9532_ = _938_n;
  for (int32_t _941_p = _9531_; _941_p < _9532_; ++_941_p)
  {
    {
      if ((_ptr__935_nativeidxs[_941_p] >= (int)0))
      {
        {
          if (_934_lower)
          {
            {
              _ptr__936_res[safe_add( _937_offset,_941_p )] = _ptr__939_slc[_ptr__935_nativeidxs[_941_p]];
            }
          }
          else
          {
            {
              _ptr__936_res[safe_add( _937_offset,_941_p )] = (-_ptr__940_suc[_ptr__935_nativeidxs[_941_p]]);
            }
          }
        }
      }
      else
      {
        {
          throw ::mosek::fusion::SolutionError(std::string ("Requested invalid solution item"));
        }
      }
    }
  }
}
// End mosek.fusion.Model.getConstraintDuals

// Begin mosek.fusion.Model.getConstraintValues
// Method mosek.fusion.Model.getConstraintValues @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4401:6-4432:7
void mosek::fusion::Model :: getConstraintValues(bool _942_primal,std::shared_ptr< monty::ndarray< int32_t,1 > > _943_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _944_res,int32_t _945_offset) { mosek::fusion::p_Model::_get_impl(this)->getConstraintValues(_942_primal,_943_nativeidxs,_944_res,_945_offset); }
void mosek::fusion::p_Model::getConstraintValues(bool _942_primal,std::shared_ptr< monty::ndarray< int32_t,1 > > _943_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _944_res,int32_t _945_offset)
{
  _checked_ptr_1<int32_t> _ptr__943_nativeidxs(_943_nativeidxs ? _943_nativeidxs->raw() : nullptr, _943_nativeidxs ? _943_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__944_res(_944_res ? _944_res->raw() : nullptr, _944_res ? _944_res->size(0) : 0);
  int32_t _946_n = (int)((_943_nativeidxs)->size(0));
  if (_942_primal)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _947_xc = getSolution_xc(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__947_xc(_947_xc ? _947_xc->raw() : nullptr,_947_xc ? _947_xc->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _948_accx = getSolution_accx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__948_accx(_948_accx ? _948_accx->raw() : nullptr,_948_accx ? _948_accx->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _949_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__949_accptr(_949_accptr ? _949_accptr->raw() : nullptr,_949_accptr ? _949_accptr->size(0) : 0);
      int32_t _9533_ = (int)0;
      int32_t _9534_ = _946_n;
      for (int32_t _950_p = _9533_; _950_p < _9534_; ++_950_p)
      {
        {
          if ((_ptr__943_nativeidxs[_950_p] >= (int)0))
          {
            {
              _ptr__944_res[safe_add( _945_offset,_950_p )] = _ptr__947_xc[_ptr__943_nativeidxs[_950_p]];
            }
          }
          else
          {
            {
              int32_t _951_i = (-safe_add( (int)1,_ptr__943_nativeidxs[_950_p] ));
              _ptr__944_res[safe_add( _945_offset,_950_p )] = (_ptr__948_accx[safe_add( _ptr__949_accptr[((*acon_elm_accid)(_951_i))],((*acon_elm_ofs)(_951_i)) )] * ((*acon_elm_scale)(_951_i)));
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _952_y = getSolution_y(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__952_y(_952_y ? _952_y->raw() : nullptr,_952_y ? _952_y->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _953_accy = getSolution_accy(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__953_accy(_953_accy ? _953_accy->raw() : nullptr,_953_accy ? _953_accy->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _954_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__954_accptr(_954_accptr ? _954_accptr->raw() : nullptr,_954_accptr ? _954_accptr->size(0) : 0);
      int32_t _9535_ = (int)0;
      int32_t _9536_ = _946_n;
      for (int32_t _955_p = _9535_; _955_p < _9536_; ++_955_p)
      {
        {
          if ((_ptr__943_nativeidxs[_955_p] >= (int)0))
          {
            {
              _ptr__944_res[safe_add( _945_offset,_955_p )] = _ptr__952_y[_ptr__943_nativeidxs[_955_p]];
            }
          }
          else
          {
            {
              int32_t _956_i = (-safe_add( (int)1,_ptr__943_nativeidxs[_955_p] ));
              _ptr__944_res[safe_add( _945_offset,_955_p )] = (_ptr__953_accy[safe_add( _ptr__954_accptr[((*acon_elm_accid)(_956_i))],((*acon_elm_ofs)(_956_i)) )] * ((*acon_elm_scale)(_956_i)));
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getConstraintValues

// Begin mosek.fusion.Model.getVariableDuals
// Method mosek.fusion.Model.getVariableDuals @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4371:6-4398:7
void mosek::fusion::Model :: getVariableDuals(bool _957_lower,std::shared_ptr< monty::ndarray< int64_t,1 > > _958_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _959_res,int32_t _960_offset) { mosek::fusion::p_Model::_get_impl(this)->getVariableDuals(_957_lower,_958_nativeidxs,_959_res,_960_offset); }
void mosek::fusion::p_Model::getVariableDuals(bool _957_lower,std::shared_ptr< monty::ndarray< int64_t,1 > > _958_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _959_res,int32_t _960_offset)
{
  _checked_ptr_1<int64_t> _ptr__958_nativeidxs(_958_nativeidxs ? _958_nativeidxs->raw() : nullptr, _958_nativeidxs ? _958_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__959_res(_959_res ? _959_res->raw() : nullptr, _959_res ? _959_res->size(0) : 0);
  int32_t _961_n = (int)((_958_nativeidxs)->size(0));
  if (_957_lower)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _962_xx = getSolution_slx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__962_xx(_962_xx ? _962_xx->raw() : nullptr,_962_xx ? _962_xx->size(0) : 0);
      int32_t _9537_ = (int)0;
      int32_t _9538_ = _961_n;
      for (int32_t _963_p = _9537_; _963_p < _9538_; ++_963_p)
      {
        {
          if ((_ptr__958_nativeidxs[_963_p] > (int)0))
          {
            {
              _ptr__959_res[safe_add( _960_offset,_963_p )] = _ptr__962_xx[_ptr__958_nativeidxs[_963_p]];
            }
          }
          else
          {
            {
              _ptr__959_res[safe_add( _960_offset,_963_p )] = (int)0;
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _964_xx = getSolution_sux(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__964_xx(_964_xx ? _964_xx->raw() : nullptr,_964_xx ? _964_xx->size(0) : 0);
      int32_t _9539_ = (int)0;
      int32_t _9540_ = _961_n;
      for (int32_t _965_p = _9539_; _965_p < _9540_; ++_965_p)
      {
        {
          if ((_ptr__958_nativeidxs[_965_p] > (int)0))
          {
            {
              _ptr__959_res[safe_add( _960_offset,_965_p )] = (-_ptr__964_xx[_ptr__958_nativeidxs[_965_p]]);
            }
          }
          else
          {
            {
              _ptr__959_res[safe_add( _960_offset,_965_p )] = (int)0;
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getVariableDuals

// Begin mosek.fusion.Model.getVariableValues
// Method mosek.fusion.Model.getVariableValues @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4330:6-4367:7
void mosek::fusion::Model :: getVariableValues(bool _966_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _967_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _968_res,int32_t _969_offset) { mosek::fusion::p_Model::_get_impl(this)->getVariableValues(_966_primal,_967_nativeidxs,_968_res,_969_offset); }
void mosek::fusion::p_Model::getVariableValues(bool _966_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _967_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _968_res,int32_t _969_offset)
{
  _checked_ptr_1<int64_t> _ptr__967_nativeidxs(_967_nativeidxs ? _967_nativeidxs->raw() : nullptr, _967_nativeidxs ? _967_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__968_res(_968_res ? _968_res->raw() : nullptr, _968_res ? _968_res->size(0) : 0);
  int32_t _970_n = (int)((_967_nativeidxs)->size(0));
  barvar_num = task_numbarvar();
  if (_966_primal)
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _971_xx = getSolution_xx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__971_xx(_971_xx ? _971_xx->raw() : nullptr,_971_xx ? _971_xx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _9541_;
      bool _9542_ = (barvar_num > (int)0);
      if (_9542_)
      {
        _9541_ = getSolution_barx(::mosek::fusion::SolutionType::Default);
      }
      else
      {
        _9541_ = nullptr;
      }
      std::shared_ptr< monty::ndarray< double,1 > > _972_barx = _9541_;
      _checked_ptr_1<double> _ptr__972_barx(_972_barx ? _972_barx->raw() : nullptr,_972_barx ? _972_barx->size(0) : 0);
      int32_t _9543_ = (int)0;
      int32_t _9544_ = _970_n;
      for (int32_t _973_p = _9543_; _973_p < _9544_; ++_973_p)
      {
        {
          if ((_ptr__967_nativeidxs[_973_p] > (int)0))
          {
            {
              _ptr__968_res[safe_add( _969_offset,_973_p )] = _ptr__971_xx[_ptr__967_nativeidxs[_973_p]];
            }
          }
          else if((_ptr__967_nativeidxs[_973_p] < (int)0))
          {
            {
              int64_t _974_idx = (-safe_add( _ptr__967_nativeidxs[_973_p],(int)1 ));
              _ptr__968_res[safe_add( _969_offset,_973_p )] = _ptr__972_barx[_974_idx];
            }
          }
          else
          {
            {
              _ptr__968_res[safe_add( _969_offset,_973_p )] = (int)0;
            }
          }
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _975_yx = getSolution_yx(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__975_yx(_975_yx ? _975_yx->raw() : nullptr,_975_yx ? _975_yx->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _9545_;
      bool _9546_ = (barvar_num > (int)0);
      if (_9546_)
      {
        _9545_ = getSolution_bars(::mosek::fusion::SolutionType::Default);
      }
      else
      {
        _9545_ = nullptr;
      }
      std::shared_ptr< monty::ndarray< double,1 > > _976_bars = _9545_;
      _checked_ptr_1<double> _ptr__976_bars(_976_bars ? _976_bars->raw() : nullptr,_976_bars ? _976_bars->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _977_accy = getSolution_accy(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<double> _ptr__977_accy(_977_accy ? _977_accy->raw() : nullptr,_977_accy ? _977_accy->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _978_accptr = getSolution_accptr(::mosek::fusion::SolutionType::Default);
      _checked_ptr_1<int32_t> _ptr__978_accptr(_978_accptr ? _978_accptr->raw() : nullptr,_978_accptr ? _978_accptr->size(0) : 0);
      int32_t _9547_ = (int)0;
      int32_t _9548_ = _970_n;
      for (int32_t _979_p = _9547_; _979_p < _9548_; ++_979_p)
      {
        {
          if ((_ptr__967_nativeidxs[_979_p] < (int)0))
          {
            {
              int64_t _980_idx = (-safe_add( _ptr__967_nativeidxs[_979_p],(int)1 ));
              _ptr__968_res[safe_add( _969_offset,_979_p )] = _ptr__976_bars[_980_idx];
            }
          }
          else if((_ptr__967_nativeidxs[_979_p] == (int)0))
          {
            {
              _ptr__968_res[safe_add( _969_offset,_979_p )] = (int)0;
            }
          }
          else if((((*var_elm_acc_idx)(_ptr__967_nativeidxs[_979_p])) >= (int)0))
          {
            {
              _ptr__968_res[safe_add( _969_offset,_979_p )] = _ptr__977_accy[safe_add( _ptr__978_accptr[((*var_elm_acc_idx)(_ptr__967_nativeidxs[_979_p]))],((*var_elm_acc_ofs)(_ptr__967_nativeidxs[_979_p])) )];
            }
          }
          else
          {
            {
              _ptr__968_res[safe_add( _969_offset,_979_p )] = _ptr__975_yx[_ptr__967_nativeidxs[_979_p]];
            }
          }
        }
      }
    }
  }
}
// End mosek.fusion.Model.getVariableValues

// Begin mosek.fusion.Model.setVariableValues
// Method mosek.fusion.Model.setVariableValues @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4307:6-4327:7
void mosek::fusion::Model :: setVariableValues(bool _981_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _982_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _983_values) { mosek::fusion::p_Model::_get_impl(this)->setVariableValues(_981_primal,_982_nativeidxs,_983_values); }
void mosek::fusion::p_Model::setVariableValues(bool _981_primal,std::shared_ptr< monty::ndarray< int64_t,1 > > _982_nativeidxs,std::shared_ptr< monty::ndarray< double,1 > > _983_values)
{
  _checked_ptr_1<int64_t> _ptr__982_nativeidxs(_982_nativeidxs ? _982_nativeidxs->raw() : nullptr, _982_nativeidxs ? _982_nativeidxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__983_values(_983_values ? _983_values->raw() : nullptr, _983_values ? _983_values->size(0) : 0);
  if (_981_primal)
  {
    {
      int32_t _984_numnz = (int)0;
      int32_t _9549_ = (int)0;
      int32_t _9550_ = (int)((_982_nativeidxs)->size(0));
      for (int32_t _985_i = _9549_; _985_i < _9550_; ++_985_i)
      {
        {
          if ((_ptr__982_nativeidxs[_985_i] >= (int)0))
          {
            {
              ++ _984_numnz;
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _986_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_984_numnz)));
      _checked_ptr_1<int32_t> _ptr__986_subj(_986_subj ? _986_subj->raw() : nullptr,_986_subj ? _986_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _987_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_984_numnz)));
      _checked_ptr_1<double> _ptr__987_val(_987_val ? _987_val->raw() : nullptr,_987_val ? _987_val->size(0) : 0);
      int32_t _988_k = (int)0;
      int32_t _9551_ = (int)0;
      int32_t _9552_ = (int)((_982_nativeidxs)->size(0));
      for (int32_t _989_i = _9551_; _989_i < _9552_; ++_989_i)
      {
        {
          if ((_ptr__982_nativeidxs[_989_i] >= (int)0))
          {
            {
              _ptr__986_subj[_988_k] = (int32_t)_ptr__982_nativeidxs[_989_i];
              _ptr__987_val[_988_k] = _ptr__983_values[_989_i];
              ++ _988_k;
            }
          }
          {}
        }
      }
      setSolution_xx(_986_subj,_987_val);
    }
  }
  else
  {
    {
    }
  }
}
// End mosek.fusion.Model.setVariableValues

// Begin mosek.fusion.Model.flushNames
// Method mosek.fusion.Model.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4285:6-4299:7
void mosek::fusion::Model :: flushNames() { mosek::fusion::p_Model::_get_impl(this)->flushNames(); }
void mosek::fusion::p_Model::flushNames()
{
  int32_t _9553_ = (int)0;
  int32_t _9554_ = (int)((vars)->size(0));
  for (int32_t _990_i = _9553_; _990_i < _9554_; ++_990_i)
  {
    {
      if ((NULL != ((*vars)(_990_i)).get()))
      {
        {
          mosek::fusion::p_ModelVariable::_get_impl(((*vars)(_990_i)).get())->flushNames();
        }
      }
      {}
    }
  }
  int32_t _9555_ = (int)0;
  int32_t _9556_ = (int)((barvars)->size(0));
  for (int32_t _991_i = _9555_; _991_i < _9556_; ++_991_i)
  {
    {
      if ((NULL != ((*barvars)(_991_i)).get()))
      {
        {
          mosek::fusion::p_ModelVariable::_get_impl(((*barvars)(_991_i)).get())->flushNames();
        }
      }
      {}
    }
  }
  int32_t _9557_ = (int)0;
  int32_t _9558_ = (int)((cons)->size(0));
  for (int32_t _992_i = _9557_; _992_i < _9558_; ++_992_i)
  {
    {
      if ((NULL != ((*cons)(_992_i)).get()))
      {
        {
          mosek::fusion::p_ModelConstraint::_get_impl(((*cons)(_992_i)).get())->flushNames();
        }
      }
      {}
    }
  }
  int32_t _9559_ = (int)0;
  int32_t _9560_ = (int)((acons)->size(0));
  for (int32_t _993_i = _9559_; _993_i < _9560_; ++_993_i)
  {
    {
      if ((NULL != ((*acons)(_993_i)).get()))
      {
        {
          mosek::fusion::p_ModelConstraint::_get_impl(((*acons)(_993_i)).get())->flushNames();
        }
      }
      {}
    }
  }
}
// End mosek.fusion.Model.flushNames

// Begin mosek.fusion.Model.writeTaskNoFlush
// Method mosek.fusion.Model.writeTaskNoFlush @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4276:6-4280:7
void mosek::fusion::Model :: writeTaskNoFlush(const std::string & _994_filename) { mosek::fusion::p_Model::_get_impl(this)->writeTaskNoFlush(_994_filename); }
void mosek::fusion::p_Model::writeTaskNoFlush(const std::string & _994_filename)
{
  _pubthis->flushNames();
  task_write(_994_filename);
}
// End mosek.fusion.Model.writeTaskNoFlush

// Begin mosek.fusion.Model.writeTaskStream
// Method mosek.fusion.Model.writeTaskStream @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4268:6-4273:7
void mosek::fusion::Model :: writeTaskStream(const std::string & _995_ext,std::ostream& _996_stream) { mosek::fusion::p_Model::_get_impl(this)->writeTaskStream(_995_ext,_996_stream); }
void mosek::fusion::p_Model::writeTaskStream(const std::string & _995_ext,std::ostream& _996_stream)
{
  _pubthis->flushNames();
  flush_parameters();
  task_write_stream(_995_ext,_996_stream);
}
// End mosek.fusion.Model.writeTaskStream

// Begin mosek.fusion.Model.dataReport
// Method mosek.fusion.Model.dataReport @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4239:6-58
void mosek::fusion::Model :: dataReport() { mosek::fusion::p_Model::_get_impl(this)->dataReport(); }
void mosek::fusion::p_Model::dataReport()
{
  _pubthis->dataReport((int)10);
}
// End mosek.fusion.Model.dataReport

// Begin mosek.fusion.Model.dataReport
// Method mosek.fusion.Model.dataReport @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4234:6-4238:7
void mosek::fusion::Model :: dataReport(int32_t _997_detail) { mosek::fusion::p_Model::_get_impl(this)->dataReport(_997_detail); }
void mosek::fusion::p_Model::dataReport(int32_t _997_detail)
{
  flush_parameters();
  task_analyze_problem(_997_detail);
}
// End mosek.fusion.Model.dataReport

// Begin mosek.fusion.Model.writeTask
// Method mosek.fusion.Model.writeTask @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4227:6-4232:7
void mosek::fusion::Model :: writeTask(const std::string & _998_filename) { mosek::fusion::p_Model::_get_impl(this)->writeTask(_998_filename); }
void mosek::fusion::p_Model::writeTask(const std::string & _998_filename)
{
  _pubthis->flushNames();
  flush_parameters();
  task_write(_998_filename);
}
// End mosek.fusion.Model.writeTask

// Begin mosek.fusion.Model.getSolverLIntInfo
// Method mosek.fusion.Model.getSolverLIntInfo @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4219:6-93
int64_t mosek::fusion::Model :: getSolverLIntInfo(const std::string & _999_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverLIntInfo(_999_name); }
int64_t mosek::fusion::p_Model::getSolverLIntInfo(const std::string & _999_name)
{
  return task_get_liinf(_999_name);
}
// End mosek.fusion.Model.getSolverLIntInfo

// Begin mosek.fusion.Model.getSolverIntInfo
// Method mosek.fusion.Model.getSolverIntInfo @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4209:6-92
int32_t mosek::fusion::Model :: getSolverIntInfo(const std::string & _1000_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverIntInfo(_1000_name); }
int32_t mosek::fusion::p_Model::getSolverIntInfo(const std::string & _1000_name)
{
  return task_get_iinf(_1000_name);
}
// End mosek.fusion.Model.getSolverIntInfo

// Begin mosek.fusion.Model.getSolverDoubleInfo
// Method mosek.fusion.Model.getSolverDoubleInfo @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4199:6-92
double mosek::fusion::Model :: getSolverDoubleInfo(const std::string & _1001_name) { return mosek::fusion::p_Model::_get_impl(this)->getSolverDoubleInfo(_1001_name); }
double mosek::fusion::p_Model::getSolverDoubleInfo(const std::string & _1001_name)
{
  return task_get_dinf(_1001_name);
}
// End mosek.fusion.Model.getSolverDoubleInfo

// Begin mosek.fusion.Model.setCallbackHandler
// Method mosek.fusion.Model.setCallbackHandler @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4189:6-110
void mosek::fusion::Model :: setCallbackHandler(mosek::cbhandler_t _1002_h) { mosek::fusion::p_Model::_get_impl(this)->setCallbackHandler(_1002_h); }
void mosek::fusion::p_Model::setCallbackHandler(mosek::cbhandler_t _1002_h)
{
  task_setCallbackHandler(_1002_h);
}
// End mosek.fusion.Model.setCallbackHandler

// Begin mosek.fusion.Model.setDataCallbackHandler
// Method mosek.fusion.Model.setDataCallbackHandler @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4168:6-114
void mosek::fusion::Model :: setDataCallbackHandler(mosek::datacbhandler_t _1003_h) { mosek::fusion::p_Model::_get_impl(this)->setDataCallbackHandler(_1003_h); }
void mosek::fusion::p_Model::setDataCallbackHandler(mosek::datacbhandler_t _1003_h)
{
  task_setDataCallbackHandler(_1003_h);
}
// End mosek.fusion.Model.setDataCallbackHandler

// Begin mosek.fusion.Model.setLogHandler
// Method mosek.fusion.Model.setLogHandler @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4146:6-89
void mosek::fusion::Model :: setLogHandler(mosek::msghandler_t _1004_h) { mosek::fusion::p_Model::_get_impl(this)->setLogHandler(_1004_h); }
void mosek::fusion::p_Model::setLogHandler(mosek::msghandler_t _1004_h)
{
  task_setLogHandler(_1004_h);
}
// End mosek.fusion.Model.setLogHandler

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4106:6-132
void mosek::fusion::Model :: setSolverParam(const std::string & _1005_name,double _1006_floatval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1005_name,_1006_floatval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1005_name,double _1006_floatval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1005_name,_1006_floatval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4094:6-128
void mosek::fusion::Model :: setSolverParam(const std::string & _1007_name,int32_t _1008_intval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1007_name,_1008_intval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1007_name,int32_t _1008_intval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1007_name,_1008_intval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.setSolverParam
// Method mosek.fusion.Model.setSolverParam @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4082:6-128
void mosek::fusion::Model :: setSolverParam(const std::string & _1009_name,const std::string & _1010_strval) { mosek::fusion::p_Model::_get_impl(this)->setSolverParam(_1009_name,_1010_strval); }
void mosek::fusion::p_Model::setSolverParam(const std::string & _1009_name,const std::string & _1010_strval)
{
  ::mosek::fusion::p_Parameters::setParameter(::mosek::fusion::Model::t(_pubthis),_1009_name,_1010_strval);
}
// End mosek.fusion.Model.setSolverParam

// Begin mosek.fusion.Model.breakSolver
// Method mosek.fusion.Model.breakSolver @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4066:6-4069:7
void mosek::fusion::Model :: breakSolver() { mosek::fusion::p_Model::_get_impl(this)->breakSolver(); }
void mosek::fusion::p_Model::breakSolver()
{
  task_break_solve();
}
// End mosek.fusion.Model.breakSolver

// Begin mosek.fusion.Model.optserverHost
// Method mosek.fusion.Model.optserverHost @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:4052:6-4055:7
void mosek::fusion::Model :: optserverHost(const std::string & _1011_addr) { mosek::fusion::p_Model::_get_impl(this)->optserverHost(_1011_addr); }
void mosek::fusion::p_Model::optserverHost(const std::string & _1011_addr)
{
  task_putoptserver_host(_1011_addr);
}
// End mosek.fusion.Model.optserverHost

// Begin mosek.fusion.Model.report_solution
// Method mosek.fusion.Model.report_solution @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3971:6-4047:7
void mosek::fusion::p_Model::report_solution(mosek::fusion::SolutionType _1012_soltype,mosek::fusion::ProblemStatus _1013_prosta,mosek::fusion::SolutionStatus _1014_psolsta,mosek::fusion::SolutionStatus _1015_dsolsta,double _1016_pobj,double _1017_dobj,int32_t _1018_numvar,int32_t _1019_numcon,int32_t _1020_numbarelm,int32_t _1021_numacc,int32_t _1022_numaccelm,bool _1023_hasprimal,bool _1024_hasdual)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1025_sol;
  if ((_1012_soltype == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1025_sol = sol_itr;
    }
  }
  else if((_1012_soltype == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1025_sol = sol_bas;
    }
  }
  else if((_1012_soltype == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1025_sol = sol_itg;
    }
  }
  else
  {
    {
      _1025_sol = nullptr;
    }
  }
  if ((NULL == _1025_sol.get()))
  {
    {
      _1025_sol = ::mosek::fusion::p_SolutionStruct::_new_SolutionStruct(_1018_numvar,_1019_numcon,_1020_numbarelm,_1021_numacc,_1022_numaccelm);
    }
  }
  else
  {
    {
      _1025_sol->resize(_1018_numvar,_1019_numcon,_1020_numbarelm,_1021_numacc,_1022_numaccelm);
    }
  }
  mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->pobj = _1016_pobj;
  mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->dobj = _1017_dobj;
  mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->pstatus = _1014_psolsta;
  mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->dstatus = _1015_dsolsta;
  mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->probstatus = _1013_prosta;
  if ((_1018_numvar > (int)0))
  {
    {
      if (_1023_hasprimal)
      {
        {
          report_solution_get_xx(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->xx);
        }
      }
      {}
      if (_1024_hasdual)
      {
        {
          report_solution_get_slx(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->slx);
          report_solution_get_sux(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->sux);
          std::shared_ptr< monty::ndarray< double,1 > > _1026_yx = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->yx;
          _checked_ptr_1<double> _ptr__1026_yx(_1026_yx ? _1026_yx->raw() : nullptr,_1026_yx ? _1026_yx->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1027_slx = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->slx;
          _checked_ptr_1<double> _ptr__1027_slx(_1027_slx ? _1027_slx->raw() : nullptr,_1027_slx ? _1027_slx->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1028_sux = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->sux;
          _checked_ptr_1<double> _ptr__1028_sux(_1028_sux ? _1028_sux->raw() : nullptr,_1028_sux ? _1028_sux->size(0) : 0);
          int32_t _9561_ = (int)0;
          int32_t _9562_ = _1018_numvar;
          for (int32_t _1029_i = _9561_; _1029_i < _9562_; ++_1029_i)
          {
            {
              _ptr__1026_yx[_1029_i] = (_ptr__1027_slx[_1029_i] - _ptr__1028_sux[_1029_i]);
            }
          }
        }
      }
      {}
    }
  }
  {}
  if ((_1019_numcon > (int)0))
  {
    {
      if (_1023_hasprimal)
      {
        {
          report_solution_get_xc(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->xc);
        }
      }
      {}
      if (_1024_hasdual)
      {
        {
          report_solution_get_slc(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->slc);
          report_solution_get_suc(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->suc);
          std::shared_ptr< monty::ndarray< double,1 > > _1030_y = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->y;
          _checked_ptr_1<double> _ptr__1030_y(_1030_y ? _1030_y->raw() : nullptr,_1030_y ? _1030_y->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1031_slc = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->slc;
          _checked_ptr_1<double> _ptr__1031_slc(_1031_slc ? _1031_slc->raw() : nullptr,_1031_slc ? _1031_slc->size(0) : 0);
          std::shared_ptr< monty::ndarray< double,1 > > _1032_suc = mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->suc;
          _checked_ptr_1<double> _ptr__1032_suc(_1032_suc ? _1032_suc->raw() : nullptr,_1032_suc ? _1032_suc->size(0) : 0);
          int32_t _9563_ = (int)0;
          int32_t _9564_ = _1019_numcon;
          for (int32_t _1033_i = _9563_; _1033_i < _9564_; ++_1033_i)
          {
            {
              _ptr__1030_y[_1033_i] = (_ptr__1031_slc[_1033_i] - _ptr__1032_suc[_1033_i]);
            }
          }
        }
      }
      {}
    }
  }
  {}
  if ((_1020_numbarelm > (int)0))
  {
    {
      if (_1023_hasprimal)
      {
        {
          report_solution_get_barx(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->barx);
        }
      }
      {}
      if (_1024_hasdual)
      {
        {
          report_solution_get_bars(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->bars);
        }
      }
      {}
    }
  }
  {}
  if ((task_numacc() > (int)0))
  {
    {
      report_solution_get_accptr(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->accptr);
      if (_1023_hasprimal)
      {
        {
          report_solution_get_accx(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->accx);
        }
      }
      {}
      if (_1024_hasdual)
      {
        {
          report_solution_get_accy(mosek::fusion::p_SolutionStruct::_get_impl(_1025_sol)->accy);
        }
      }
      {}
    }
  }
  {}
  if ((_1012_soltype == ::mosek::fusion::SolutionType::Interior))
  {
    {
      sol_itr = _1025_sol;
    }
  }
  else if((_1012_soltype == ::mosek::fusion::SolutionType::Basic))
  {
    {
      sol_bas = _1025_sol;
    }
  }
  else if((_1012_soltype == ::mosek::fusion::SolutionType::Integer))
  {
    {
      sol_itg = _1025_sol;
    }
  }
  {}
}
// End mosek.fusion.Model.report_solution

// Begin mosek.fusion.Model.clear_solutions
// Method mosek.fusion.Model.clear_solutions @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3965:6-3969:7
void mosek::fusion::p_Model::clear_solutions()
{
  sol_itr = nullptr;
  sol_itg = nullptr;
  sol_bas = nullptr;
}
// End mosek.fusion.Model.clear_solutions

// Begin mosek.fusion.Model.solveBatch
// Method mosek.fusion.Model.solveBatch @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3940:6-3963:7
std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > mosek::fusion::Model::solveBatch
  ( bool _1034_israce,
    double _1035_maxtime,
    int32_t _1036_numthreads,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Model >,1 > > _1037_models) { return mosek::fusion::p_Model::solveBatch(_1034_israce,_1035_maxtime,_1036_numthreads,_1037_models); }
std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > mosek::fusion::p_Model::solveBatch(bool _1034_israce,double _1035_maxtime,int32_t _1036_numthreads,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Model >,1 > > _1037_models)
{
  int32_t _1038_n = (int)((_1037_models)->size(0));
  int32_t _9565_ = (int)0;
  int32_t _9566_ = _1038_n;
  for (int32_t _1039_i = _9565_; _1039_i < _9566_; ++_1039_i)
  {
    {
      if ((NULL == ((*_1037_models)(_1039_i)).get()))
      {
        {
          throw ::mosek::fusion::OptimizeError(std::string ("One of the models is a null object"));
        }
      }
      {}
    }
  }
  int32_t _9567_ = (int)0;
  int32_t _9568_ = _1038_n;
  for (int32_t _1040_i = _9567_; _1040_i < _9568_; ++_1040_i)
  {
    {
      ((*_1037_models)(_1040_i))->flushSolutions();
      mosek::fusion::p_Model::_get_impl(((*_1037_models)(_1040_i)).get())->flush_parameters();
    }
  }
  std::shared_ptr< monty::ndarray< mosek::fusion::SolverStatus,1 > > _1041_wasOK = ::mosek::fusion::p_BaseModel::env_solve_batch(_1034_israce,_1035_maxtime,_1036_numthreads,_1037_models);
  int32_t _9569_ = (int)0;
  int32_t _9570_ = _1038_n;
  for (int32_t _1042_i = _9569_; _1042_i < _9570_; ++_1042_i)
  {
    {
      mosek::fusion::p_Model::_get_impl(((*_1037_models)(_1042_i)).get())->task_post_solve();
    }
  }
  return _1041_wasOK;
}
// End mosek.fusion.Model.solveBatch

// Begin mosek.fusion.Model.solve
// Method mosek.fusion.Model.solve @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3931:6-3938:7
void mosek::fusion::Model :: solve(const std::string & _1043_addr,const std::string & _1044_accesstoken) { mosek::fusion::p_Model::_get_impl(this)->solve(_1043_addr,_1044_accesstoken); }
void mosek::fusion::p_Model::solve(const std::string & _1043_addr,const std::string & _1044_accesstoken)
{
  _pubthis->flushSolutions();
  flush_parameters();
  task_solve(true,_1043_addr,_1044_accesstoken);
  task_post_solve();
}
// End mosek.fusion.Model.solve

// Begin mosek.fusion.Model.solve
// Method mosek.fusion.Model.solve @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3922:6-3929:7
void mosek::fusion::Model :: solve() { mosek::fusion::p_Model::_get_impl(this)->solve(); }
void mosek::fusion::p_Model::solve()
{
  _pubthis->flushSolutions();
  flush_parameters();
  task_solve(false,std::string (""),std::string (""));
  task_post_solve();
}
// End mosek.fusion.Model.solve

// Begin mosek.fusion.Model.flush_parameters
// Method mosek.fusion.Model.flush_parameters @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3843:6-3890:7
void mosek::fusion::p_Model::flush_parameters()
{
  int32_t _1045_num_con_param_nz = con_blocks->num_parameterized();
  int32_t _1046_num_afe_param_nz = afe_blocks->num_parameterized();
  int32_t _1047_num_obj_param_nz = obj_blocks->num_parameterized();
  if ((_1045_num_con_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1048_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1045_num_con_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1048_subi(_1048_subi ? _1048_subi->raw() : nullptr,_1048_subi ? _1048_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1049_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1045_num_con_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1049_subj(_1049_subj ? _1049_subj->raw() : nullptr,_1049_subj ? _1049_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1050_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1045_num_con_param_nz)));
      _checked_ptr_1<double> _ptr__1050_val(_1050_val ? _1050_val->raw() : nullptr,_1050_val ? _1050_val->size(0) : 0);
      con_blocks->evaluate(xs,param_value,_1048_subi,_1049_subj,_1050_val);
      task_putaijlist(_1048_subi,_1049_subj,_1050_val,(int64_t)(_1045_num_con_param_nz));
    }
  }
  {}
  if ((_1046_num_afe_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1051_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1046_num_afe_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1051_subi(_1051_subi ? _1051_subi->raw() : nullptr,_1051_subi ? _1051_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1052_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1046_num_afe_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1052_subj(_1052_subj ? _1052_subj->raw() : nullptr,_1052_subj ? _1052_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1053_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1046_num_afe_param_nz)));
      _checked_ptr_1<double> _ptr__1053_val(_1053_val ? _1053_val->raw() : nullptr,_1053_val ? _1053_val->size(0) : 0);
      afe_blocks->evaluate(xs,param_value,_1051_subi,_1052_subj,_1053_val);
      task_putafefijlist(_1051_subi,_1052_subj,_1053_val);
    }
  }
  {}
  if ((_1047_num_obj_param_nz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1054_subi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1047_num_obj_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1054_subi(_1054_subi ? _1054_subi->raw() : nullptr,_1054_subi ? _1054_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1055_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1047_num_obj_param_nz)));
      _checked_ptr_1<int32_t> _ptr__1055_subj(_1055_subj ? _1055_subj->raw() : nullptr,_1055_subj ? _1055_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1056_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1047_num_obj_param_nz)));
      _checked_ptr_1<double> _ptr__1056_val(_1056_val ? _1056_val->raw() : nullptr,_1056_val ? _1056_val->size(0) : 0);
      obj_blocks->evaluate(xs,param_value,_1054_subi,_1055_subj,_1056_val);
      task_putclist(_1055_subj,_1056_val);
    }
  }
  {}
}
// End mosek.fusion.Model.flush_parameters

// Begin mosek.fusion.Model.flushParameters
// Method mosek.fusion.Model.flushParameters @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3841:6-61
void mosek::fusion::Model :: flushParameters() { mosek::fusion::p_Model::_get_impl(this)->flushParameters(); }
void mosek::fusion::p_Model::flushParameters()
{
  flush_parameters();
}
// End mosek.fusion.Model.flushParameters

// Begin mosek.fusion.Model.evaluate_parameterized
// Method mosek.fusion.Model.evaluate_parameterized @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3772:6-3835:7
void mosek::fusion::p_Model::evaluate_parameterized(monty::rc_ptr< ::mosek::fusion::WorkStack > _1057_xs,int32_t _1058_numrow,std::shared_ptr< monty::ndarray< int32_t,1 > > _1059_rowptrb,std::shared_ptr< monty::ndarray< int32_t,1 > > _1060_rowptre,std::shared_ptr< monty::ndarray< int64_t,1 > > _1061_codenidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _1062_codeptr,std::shared_ptr< monty::ndarray< int32_t,1 > > _1063_codesizes,std::shared_ptr< monty::ndarray< int32_t,1 > > _1064_code,std::shared_ptr< monty::ndarray< double,1 > > _1065_cconst,std::shared_ptr< monty::ndarray< int32_t,1 > > _1066_subj,std::shared_ptr< monty::ndarray< double,1 > > _1067_val)
{
  _checked_ptr_1<int32_t> _ptr__1059_rowptrb(_1059_rowptrb ? _1059_rowptrb->raw() : nullptr, _1059_rowptrb ? _1059_rowptrb->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1060_rowptre(_1060_rowptre ? _1060_rowptre->raw() : nullptr, _1060_rowptre ? _1060_rowptre->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1061_codenidx(_1061_codenidx ? _1061_codenidx->raw() : nullptr, _1061_codenidx ? _1061_codenidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1062_codeptr(_1062_codeptr ? _1062_codeptr->raw() : nullptr, _1062_codeptr ? _1062_codeptr->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1063_codesizes(_1063_codesizes ? _1063_codesizes->raw() : nullptr, _1063_codesizes ? _1063_codesizes->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1064_code(_1064_code ? _1064_code->raw() : nullptr, _1064_code ? _1064_code->size(0) : 0);
  _checked_ptr_1<double> _ptr__1065_cconst(_1065_cconst ? _1065_cconst->raw() : nullptr, _1065_cconst ? _1065_cconst->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1066_subj(_1066_subj ? _1066_subj->raw() : nullptr, _1066_subj ? _1066_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__1067_val(_1067_val ? _1067_val->raw() : nullptr, _1067_val ? _1067_val->size(0) : 0);
  int32_t _1068_stack = _1057_xs->allocf64((int)((_1064_code)->size(0)));
  int32_t _1069_stackp = (int)0;
  std::shared_ptr< monty::ndarray< double,1 > > _1070_xf64 = mosek::fusion::p_WorkStack::_get_impl(_1057_xs)->f64;
  _checked_ptr_1<double> _ptr__1070_xf64(_1070_xf64 ? _1070_xf64->raw() : nullptr,_1070_xf64 ? _1070_xf64->size(0) : 0);
  int32_t _1071_idx = (int)0;
  int32_t _9571_ = (int)0;
  int32_t _9572_ = _1058_numrow;
  for (int32_t _1072_i = _9571_; _1072_i < _9572_; ++_1072_i)
  {
    {
      int32_t _1073_k = _ptr__1062_codeptr[_1072_i];
      int32_t _9573_ = _ptr__1059_rowptrb[_1072_i];
      int32_t _9574_ = _ptr__1060_rowptre[_1072_i];
      for (int32_t _1074_j = _9573_; _1074_j < _9574_; ++_1074_j)
      {
        {
          int32_t _9575_ = (int)0;
          int32_t _9576_ = _ptr__1063_codesizes[_1074_j];
          for (int32_t _1075_l = _9575_; _1075_l < _9576_; ++_1075_l)
          {
            {
              if ((_ptr__1064_code[_1073_k] == 0))
              {
                {
                }
              }
              else
              {
                {
                  if ((_ptr__1064_code[_1073_k] == 1))
                  {
                    {
                      _ptr__1070_xf64[safe_add( _1068_stack,_1069_stackp )] = ((*param_value)((int32_t)_ptr__1065_cconst[_1073_k]));
                      ++ _1069_stackp;
                    }
                  }
                  else
                  {
                    {
                      if ((_ptr__1064_code[_1073_k] == 2))
                      {
                        {
                          _ptr__1070_xf64[safe_add( _1068_stack,_1069_stackp )] = _ptr__1065_cconst[_1073_k];
                          ++ _1069_stackp;
                        }
                      }
                      else
                      {
                        {
                          if ((_ptr__1064_code[_1073_k] == 3))
                          {
                            {
                              _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)2)] += _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)];
                              -- _1069_stackp;
                            }
                          }
                          else
                          {
                            {
                              if ((_ptr__1064_code[_1073_k] == 4))
                              {
                                {
                                  _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)] = (-_ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)]);
                                }
                              }
                              else
                              {
                                {
                                  if ((_ptr__1064_code[_1073_k] == 5))
                                  {
                                    {
                                      _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)2)] *= _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)];
                                      -- _1069_stackp;
                                    }
                                  }
                                  else
                                  {
                                    {
                                      if ((_ptr__1064_code[_1073_k] == 6))
                                      {
                                        {
                                          _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)] = (1.0 / _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)]);
                                        }
                                      }
                                      else
                                      {
                                        {
                                          if ((_ptr__1064_code[_1073_k] == 8))
                                          {
                                            {
                                              double _1076_v = 0.0;
                                              int32_t _9577_ = (int)0;
                                              int32_t _9578_ = (int32_t)_ptr__1065_cconst[_1073_k];
                                              for (int32_t _1077_i = _9577_; _1077_i < _9578_; ++_1077_i)
                                              {
                                                {
                                                  _1076_v += _ptr__1070_xf64[(safe_add( _1068_stack,_1069_stackp ) - (int)1)];
                                                  -- _1069_stackp;
                                                }
                                              }
                                              _ptr__1070_xf64[safe_add( _1068_stack,_1069_stackp )] = _1076_v;
                                              ++ _1069_stackp;
                                            }
                                          }
                                          else
                                          {
                                            {
                                            }
                                          }
                                        }
                                      }
                                    }
                                  }
                                }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
              ++ _1073_k;
            }
          }
          _ptr__1066_subj[_1071_idx] = (int32_t)_ptr__1061_codenidx[_1074_j];
          _ptr__1067_val[_1071_idx] = _ptr__1070_xf64[_1068_stack];
          _1069_stackp = (int)0;
          ++ _1071_idx;
        }
      }
    }
  }
  _1057_xs->popf64((int)((_1064_code)->size(0)));
}
// End mosek.fusion.Model.evaluate_parameterized

// Begin mosek.fusion.Model.flushSolutions
// Method mosek.fusion.Model.flushSolutions @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3765:6-3770:7
void mosek::fusion::Model :: flushSolutions() { mosek::fusion::p_Model::_get_impl(this)->flushSolutions(); }
void mosek::fusion::p_Model::flushSolutions()
{
  flush_initsol(::mosek::fusion::SolutionType::Interior);
  flush_initsol(::mosek::fusion::SolutionType::Integer);
  flush_initsol(::mosek::fusion::SolutionType::Basic);
}
// End mosek.fusion.Model.flushSolutions

// Begin mosek.fusion.Model.flush_initsol
// Method mosek.fusion.Model.flush_initsol @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3739:6-3761:7
void mosek::fusion::p_Model::flush_initsol(mosek::fusion::SolutionType _1078_which)
{
  if ((NULL != initsol_xx.get()))
  {
    {
      int32_t _1079_numvar = task_numvar();
      std::shared_ptr< monty::ndarray< double,1 > > _1080_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1079_numvar)));
      _checked_ptr_1<double> _ptr__1080_xx(_1080_xx ? _1080_xx->raw() : nullptr,_1080_xx ? _1080_xx->size(0) : 0);
      /* Try @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3744:10-3750:11 */
      try
      {
        {
          monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1082_sol = __mosek_2fusion_2Model__get_sol_cache(_1078_which,true);
          if ((NULL != mosek::fusion::p_SolutionStruct::_get_impl(_1082_sol)->xx.get()))
          {
            {
              int32_t _9579_;
              bool _9580_ = ((int)((mosek::fusion::p_SolutionStruct::_get_impl(_1082_sol)->xx)->size(0)) <= _1079_numvar);
              if (_9580_)
              {
                _9579_ = (int)((mosek::fusion::p_SolutionStruct::_get_impl(_1082_sol)->xx)->size(0));
              }
              else
              {
                _9579_ = _1079_numvar;
              }
              ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_SolutionStruct::_get_impl(_1082_sol)->xx,(int)0,_1080_xx,(int)0,_9579_);
            }
          }
          {}
        }
      }
      catch (mosek::fusion::SolutionError _1081_e)
      {
        {
        }
      }
      int32_t _9581_;
      bool _9582_ = ((int)((initsol_xx)->size(0)) <= _1079_numvar);
      if (_9582_)
      {
        _9581_ = (int)((initsol_xx)->size(0));
      }
      else
      {
        _9581_ = _1079_numvar;
      }
      int32_t _1083_n = _9581_;
      int32_t _9583_ = (int)0;
      int32_t _9584_ = _1083_n;
      for (int32_t _1084_i = _9583_; _1084_i < _9584_; ++_1084_i)
      {
        {
          if (((*initsol_xx_flag)(_1084_i)))
          {
            {
              _ptr__1080_xx[_1084_i] = ((*initsol_xx)(_1084_i));
            }
          }
          {}
        }
      }
      _ptr__1080_xx[bfixidx] = 1.0;
      task_putxx_slice(_1078_which,(int)0,_1079_numvar,_1080_xx);
    }
  }
  {}
}
// End mosek.fusion.Model.flush_initsol

// Begin mosek.fusion.Model.getDualSolutionStatus
// Method mosek.fusion.Model.getDualSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3734:6-3737:7
mosek::fusion::SolutionStatus mosek::fusion::Model :: getDualSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getDualSolutionStatus(); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getDualSolutionStatus()
{
  return _pubthis->getDualSolutionStatus(solutionptr);
}
// End mosek.fusion.Model.getDualSolutionStatus

// Begin mosek.fusion.Model.getProblemStatus
// Method mosek.fusion.Model.getProblemStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3728:6-3731:7
mosek::fusion::ProblemStatus mosek::fusion::Model :: getProblemStatus() { return mosek::fusion::p_Model::_get_impl(this)->getProblemStatus(); }
mosek::fusion::ProblemStatus mosek::fusion::p_Model::getProblemStatus()
{
  return _pubthis->getProblemStatus(solutionptr);
}
// End mosek.fusion.Model.getProblemStatus

// Begin mosek.fusion.Model.getPrimalSolutionStatus
// Method mosek.fusion.Model.getPrimalSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3721:6-3724:7
mosek::fusion::SolutionStatus mosek::fusion::Model :: getPrimalSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getPrimalSolutionStatus(); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getPrimalSolutionStatus()
{
  return getPrimalSolutionStatus(solutionptr);
}
// End mosek.fusion.Model.getPrimalSolutionStatus

// Begin mosek.fusion.Model.dualObjValue
// Method mosek.fusion.Model.dualObjValue @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3714:6-3717:7
double mosek::fusion::Model :: dualObjValue() { return mosek::fusion::p_Model::_get_impl(this)->dualObjValue(); }
double mosek::fusion::p_Model::dualObjValue()
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(::mosek::fusion::SolutionType::Default,false))->dobj;
}
// End mosek.fusion.Model.dualObjValue

// Begin mosek.fusion.Model.primalObjValue
// Method mosek.fusion.Model.primalObjValue @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3708:6-3711:7
double mosek::fusion::Model :: primalObjValue() { return mosek::fusion::p_Model::_get_impl(this)->primalObjValue(); }
double mosek::fusion::p_Model::primalObjValue()
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(::mosek::fusion::SolutionType::Default,true))->pobj;
}
// End mosek.fusion.Model.primalObjValue

// Begin mosek.fusion.Model.get_sol_cache
// Method mosek.fusion.Model.get_sol_cache @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3634:6-3705:7
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_Model::__mosek_2fusion_2Model__get_sol_cache(mosek::fusion::SolutionType _1085_which_,bool _1086_primal,bool _1087_nothrow)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1088_res = nullptr;
  mosek::fusion::SolutionType _1089_which = _1085_which_;
  if ((_1089_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      _1089_which = solutionptr;
    }
  }
  {}
  if ((_1089_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      if ((NULL != sol_itg.get()))
      {
        {
          _1088_res = sol_itg;
        }
      }
      else if((NULL != sol_bas.get()))
      {
        {
          _1088_res = sol_bas;
        }
      }
      else
      {
        {
          _1088_res = sol_itr;
        }
      }
    }
  }
  else if((_1089_which == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1088_res = sol_itr;
    }
  }
  else if((_1089_which == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1088_res = sol_bas;
    }
  }
  else if((_1089_which == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1088_res = sol_itg;
    }
  }
  {}
  if ((NULL == _1088_res.get()))
  {
    {
      throw ::mosek::fusion::SolutionError(std::string ("Solution not available"));
    }
  }
  else
  {
    {
      if ((_1087_nothrow || ((_1086_primal && _1088_res->isPrimalAcceptable(acceptable_sol)) || ((!_1086_primal) && _1088_res->isDualAcceptable(acceptable_sol)))))
      {
        {
          return _1088_res;
        }
      }
      else if(((_1088_res == sol_itg) && (!_1086_primal)))
      {
        {
          throw ::mosek::fusion::SolutionError(std::string ("Dual solution does not exists for mixed-integer problems."));
        }
      }
      else
      {
        {
          monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1090_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
          mosek::fusion::SolutionStatus _1091_solsta = mosek::fusion::p_SolutionStruct::_get_impl(_1088_res)->pstatus;
          mosek::fusion::ProblemStatus _1092_probsta = mosek::fusion::p_SolutionStruct::_get_impl(_1088_res)->probstatus;
          if ((!_1086_primal))
          {
            {
              _1091_solsta = mosek::fusion::p_SolutionStruct::_get_impl(_1088_res)->dstatus;
            }
          }
          {}
          _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Solution status is "));
          if ((_1091_solsta == ::mosek::fusion::SolutionStatus::Undefined))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Undefined"));
            }
          }
          else if((_1091_solsta == ::mosek::fusion::SolutionStatus::Unknown))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Unknown"));
            }
          }
          else if((_1091_solsta == ::mosek::fusion::SolutionStatus::Optimal))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Optimal"));
            }
          }
          else if((_1091_solsta == ::mosek::fusion::SolutionStatus::Feasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Feasible"));
            }
          }
          else if((_1091_solsta == ::mosek::fusion::SolutionStatus::Certificate))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Certificate"));
            }
          }
          else if((_1091_solsta == ::mosek::fusion::SolutionStatus::IllposedCert))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("IllposedCert"));
            }
          }
          {}
          _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" but "));
          if ((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Anything))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Anything"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Optimal))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Optimal"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Feasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("at least Feasible"));
            }
          }
          else if((acceptable_sol == ::mosek::fusion::AccSolutionStatus::Certificate))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Certificate"));
            }
          }
          {}
          _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" is expected. Reason: Accessing "));
          if ((_1088_res == sol_itr))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("interior-point solution"));
            }
          }
          else if((_1088_res == sol_bas))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("basic solution"));
            }
          }
          else if((_1088_res == sol_itg))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("integer solution"));
            }
          }
          {}
          _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" whose problem status is "));
          if ((_1092_probsta == ::mosek::fusion::ProblemStatus::Unknown))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Unknown"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::PrimalAndDualFeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalAndDualFeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::PrimalFeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalFeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::DualFeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("DualFeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::PrimalInfeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalInfeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::DualInfeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("DualInfeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::PrimalAndDualInfeasible))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalAndDualInfeasible"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::IllPosed))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("IllPosed"));
            }
          }
          else if((_1092_probsta == ::mosek::fusion::ProblemStatus::PrimalInfeasibleOrUnbounded))
          {
            {
              _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PrimalInfeasibleOrUnbounded"));
            }
          }
          {}
          _1090_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("."));
          throw ::mosek::fusion::SolutionError(_1090_sb->toString());
        }
      }
    }
  }
}
// End mosek.fusion.Model.get_sol_cache

// Begin mosek.fusion.Model.get_sol_cache
// Method mosek.fusion.Model.get_sol_cache @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3627:6-3632:7
monty::rc_ptr< ::mosek::fusion::SolutionStruct > mosek::fusion::p_Model::__mosek_2fusion_2Model__get_sol_cache(mosek::fusion::SolutionType _1093_which_,bool _1094_primal)
{
  return __mosek_2fusion_2Model__get_sol_cache(_1093_which_,_1094_primal,false);
}
// End mosek.fusion.Model.get_sol_cache

// Begin mosek.fusion.Model.setSolution_xx
// Method mosek.fusion.Model.setSolution_xx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3612:6-3620:7
void mosek::fusion::p_Model::setSolution_xx(std::shared_ptr< monty::ndarray< int32_t,1 > > _1095_subj,std::shared_ptr< monty::ndarray< double,1 > > _1096_val)
{
  _checked_ptr_1<int32_t> _ptr__1095_subj(_1095_subj ? _1095_subj->raw() : nullptr, _1095_subj ? _1095_subj->size(0) : 0);
  _checked_ptr_1<double> _ptr__1096_val(_1096_val ? _1096_val->raw() : nullptr, _1096_val ? _1096_val->size(0) : 0);
  ensure_initsol_xx();
  int32_t _9585_ = (int)0;
  int32_t _9586_ = (int)((_1095_subj)->size(0));
  for (int32_t _1097_i = _9585_; _1097_i < _9586_; ++_1097_i)
  {
    {
      ((*initsol_xx)(_ptr__1095_subj[_1097_i])) = _ptr__1096_val[_1097_i];
      ((*initsol_xx_flag)(_ptr__1095_subj[_1097_i])) = true;
    }
  }
}
// End mosek.fusion.Model.setSolution_xx

// Begin mosek.fusion.Model.ensure_initsol_xx
// Method mosek.fusion.Model.ensure_initsol_xx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3595:6-3610:7
void mosek::fusion::p_Model::ensure_initsol_xx()
{
  int32_t _1098_natvarmap_num = task_numvar();
  if ((NULL == initsol_xx.get()))
  {
    {
      initsol_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1098_natvarmap_num)));
      std::vector<bool> _9587_;
      for (int32_t _1099_i = (int)0, _9588_ = _1098_natvarmap_num; _1099_i < _9588_ ; ++_1099_i)
      {
        _9587_.push_back(false);
      }
      auto _9589_ = std::shared_ptr< monty::ndarray< bool,1 > >(new monty::ndarray< bool,1 >(monty::shape(_9587_.size()),_9587_.begin(),_9587_.end()));
      initsol_xx_flag = _9589_;
    }
  }
  else if(((int)((initsol_xx)->size(0)) < _1098_natvarmap_num))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _1100_tmp = initsol_xx;
      _checked_ptr_1<double> _ptr__1100_tmp(_1100_tmp ? _1100_tmp->raw() : nullptr,_1100_tmp ? _1100_tmp->size(0) : 0);
      std::shared_ptr< monty::ndarray< bool,1 > > _1101_tmp_flag = initsol_xx_flag;
      initsol_xx = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1098_natvarmap_num)));
      initsol_xx_flag = std::shared_ptr< monty::ndarray< bool,1 > >(new monty::ndarray< bool,1 >(monty::shape(_1098_natvarmap_num)));
      ::mosek::fusion::Utils::Tools::arraycopy(_1100_tmp,(int)0,initsol_xx,(int)0,(int)((_1100_tmp)->size(0)));
      int32_t _9590_ = (int)0;
      int32_t _9591_ = (int)((_1100_tmp)->size(0));
      for (int32_t _1102_i = _9590_; _1102_i < _9591_; ++_1102_i)
      {
        {
          ((*initsol_xx_flag)(_1102_i)) = ((*_1101_tmp_flag)(_1102_i));
        }
      }
    }
  }
  {}
}
// End mosek.fusion.Model.ensure_initsol_xx

// Begin mosek.fusion.Model.getSolution_accptr
// Method mosek.fusion.Model.getSolution_accptr @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3588:6-3592:7
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_Model::getSolution_accptr(mosek::fusion::SolutionType _1103_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1103_which,true,true))->accptr;
}
// End mosek.fusion.Model.getSolution_accptr

// Begin mosek.fusion.Model.getSolution_accy
// Method mosek.fusion.Model.getSolution_accy @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3582:6-3586:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_accy(mosek::fusion::SolutionType _1104_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1104_which,false))->accy;
}
// End mosek.fusion.Model.getSolution_accy

// Begin mosek.fusion.Model.getSolution_accx
// Method mosek.fusion.Model.getSolution_accx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3576:6-3580:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_accx(mosek::fusion::SolutionType _1105_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1105_which,true))->accx;
}
// End mosek.fusion.Model.getSolution_accx

// Begin mosek.fusion.Model.getSolution_bars
// Method mosek.fusion.Model.getSolution_bars @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3569:6-3573:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_bars(mosek::fusion::SolutionType _1106_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1106_which,false))->bars;
}
// End mosek.fusion.Model.getSolution_bars

// Begin mosek.fusion.Model.getSolution_barx
// Method mosek.fusion.Model.getSolution_barx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3563:6-3567:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_barx(mosek::fusion::SolutionType _1107_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1107_which,true))->barx;
}
// End mosek.fusion.Model.getSolution_barx

// Begin mosek.fusion.Model.getSolution_y
// Method mosek.fusion.Model.getSolution_y @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3557:6-3561:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_y(mosek::fusion::SolutionType _1108_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1108_which,false))->y;
}
// End mosek.fusion.Model.getSolution_y

// Begin mosek.fusion.Model.getSolution_xc
// Method mosek.fusion.Model.getSolution_xc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3551:6-3555:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_xc(mosek::fusion::SolutionType _1109_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1109_which,true))->xc;
}
// End mosek.fusion.Model.getSolution_xc

// Begin mosek.fusion.Model.getSolution_suc
// Method mosek.fusion.Model.getSolution_suc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3545:6-3549:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_suc(mosek::fusion::SolutionType _1110_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1110_which,false))->suc;
}
// End mosek.fusion.Model.getSolution_suc

// Begin mosek.fusion.Model.getSolution_slc
// Method mosek.fusion.Model.getSolution_slc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3539:6-3543:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_slc(mosek::fusion::SolutionType _1111_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1111_which,false))->slc;
}
// End mosek.fusion.Model.getSolution_slc

// Begin mosek.fusion.Model.getSolution_sux
// Method mosek.fusion.Model.getSolution_sux @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3533:6-3537:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_sux(mosek::fusion::SolutionType _1112_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1112_which,false))->sux;
}
// End mosek.fusion.Model.getSolution_sux

// Begin mosek.fusion.Model.getSolution_slx
// Method mosek.fusion.Model.getSolution_slx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3527:6-3531:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_slx(mosek::fusion::SolutionType _1113_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1113_which,false))->slx;
}
// End mosek.fusion.Model.getSolution_slx

// Begin mosek.fusion.Model.getSolution_yx
// Method mosek.fusion.Model.getSolution_yx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3521:6-3525:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_yx(mosek::fusion::SolutionType _1114_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1114_which,false))->yx;
}
// End mosek.fusion.Model.getSolution_yx

// Begin mosek.fusion.Model.getSolution_xx
// Method mosek.fusion.Model.getSolution_xx @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3516:6-3520:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getSolution_xx(mosek::fusion::SolutionType _1115_which)
{
  return mosek::fusion::p_SolutionStruct::_get_impl(__mosek_2fusion_2Model__get_sol_cache(_1115_which,true))->xx;
}
// End mosek.fusion.Model.getSolution_xx

// Begin mosek.fusion.Model.selectedSolution
// Method mosek.fusion.Model.selectedSolution @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3511:6-3514:7
void mosek::fusion::Model :: selectedSolution(mosek::fusion::SolutionType _1116_soltype) { mosek::fusion::p_Model::_get_impl(this)->selectedSolution(_1116_soltype); }
void mosek::fusion::p_Model::selectedSolution(mosek::fusion::SolutionType _1116_soltype)
{
  solutionptr = _1116_soltype;
}
// End mosek.fusion.Model.selectedSolution

// Begin mosek.fusion.Model.getAcceptedSolutionStatus
// Method mosek.fusion.Model.getAcceptedSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3502:6-3505:7
mosek::fusion::AccSolutionStatus mosek::fusion::Model :: getAcceptedSolutionStatus() { return mosek::fusion::p_Model::_get_impl(this)->getAcceptedSolutionStatus(); }
mosek::fusion::AccSolutionStatus mosek::fusion::p_Model::getAcceptedSolutionStatus()
{
  return acceptable_sol;
}
// End mosek.fusion.Model.getAcceptedSolutionStatus

// Begin mosek.fusion.Model.acceptedSolutionStatus
// Method mosek.fusion.Model.acceptedSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3496:6-3499:7
void mosek::fusion::Model :: acceptedSolutionStatus(mosek::fusion::AccSolutionStatus _1117_what) { mosek::fusion::p_Model::_get_impl(this)->acceptedSolutionStatus(_1117_what); }
void mosek::fusion::p_Model::acceptedSolutionStatus(mosek::fusion::AccSolutionStatus _1117_what)
{
  acceptable_sol = _1117_what;
}
// End mosek.fusion.Model.acceptedSolutionStatus

// Begin mosek.fusion.Model.getProblemStatus
// Method mosek.fusion.Model.getProblemStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3465:6-3483:7
mosek::fusion::ProblemStatus mosek::fusion::Model :: getProblemStatus(mosek::fusion::SolutionType _1118_which) { return mosek::fusion::p_Model::_get_impl(this)->getProblemStatus(_1118_which); }
mosek::fusion::ProblemStatus mosek::fusion::p_Model::getProblemStatus(mosek::fusion::SolutionType _1118_which)
{
  monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1119_res = nullptr;
  if ((_1118_which == ::mosek::fusion::SolutionType::Default))
  {
    {
      if ((NULL != sol_itg.get()))
      {
        {
          _1119_res = sol_itg;
        }
      }
      else if((NULL != sol_bas.get()))
      {
        {
          _1119_res = sol_bas;
        }
      }
      else
      {
        {
          _1119_res = sol_itr;
        }
      }
    }
  }
  else if((_1118_which == ::mosek::fusion::SolutionType::Interior))
  {
    {
      _1119_res = sol_itr;
    }
  }
  else if((_1118_which == ::mosek::fusion::SolutionType::Basic))
  {
    {
      _1119_res = sol_bas;
    }
  }
  else if((_1118_which == ::mosek::fusion::SolutionType::Integer))
  {
    {
      _1119_res = sol_itg;
    }
  }
  {}
  if ((NULL == _1119_res.get()))
  {
    {
      return ::mosek::fusion::ProblemStatus::Unknown;
    }
  }
  else
  {
    {
      return mosek::fusion::p_SolutionStruct::_get_impl(_1119_res)->probstatus;
    }
  }
}
// End mosek.fusion.Model.getProblemStatus

// Begin mosek.fusion.Model.getDualSolutionStatus
// Method mosek.fusion.Model.getDualSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3459:6-122
mosek::fusion::SolutionStatus mosek::fusion::Model :: getDualSolutionStatus(mosek::fusion::SolutionType _1120_which) { return mosek::fusion::p_Model::_get_impl(this)->getDualSolutionStatus(_1120_which); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getDualSolutionStatus(mosek::fusion::SolutionType _1120_which)
{
  return getSolutionStatus(_1120_which,false);
}
// End mosek.fusion.Model.getDualSolutionStatus

// Begin mosek.fusion.Model.getPrimalSolutionStatus
// Method mosek.fusion.Model.getPrimalSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3453:6-128
mosek::fusion::SolutionStatus mosek::fusion::Model :: getPrimalSolutionStatus(mosek::fusion::SolutionType _1121_which) { return mosek::fusion::p_Model::_get_impl(this)->getPrimalSolutionStatus(_1121_which); }
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getPrimalSolutionStatus(mosek::fusion::SolutionType _1121_which)
{
  return getSolutionStatus(_1121_which,true);
}
// End mosek.fusion.Model.getPrimalSolutionStatus

// Begin mosek.fusion.Model.getSolutionStatus
// Method mosek.fusion.Model.getSolutionStatus @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3436:6-3447:7
mosek::fusion::SolutionStatus mosek::fusion::p_Model::getSolutionStatus(mosek::fusion::SolutionType _1122_which,bool _1123_primal)
{
  /* Try @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3437:8-3446:9 */
  try
  {
    {
      monty::rc_ptr< ::mosek::fusion::SolutionStruct > _1125_sol = __mosek_2fusion_2Model__get_sol_cache(_1122_which,_1123_primal,true);
      if (_1123_primal)
      {
        {
          return mosek::fusion::p_SolutionStruct::_get_impl(_1125_sol)->pstatus;
        }
      }
      else
      {
        {
          return mosek::fusion::p_SolutionStruct::_get_impl(_1125_sol)->dstatus;
        }
      }
    }
  }
  catch (mosek::fusion::SolutionError _1124_e)
  {
    {
      return ::mosek::fusion::SolutionStatus::Undefined;
    }
  }
}
// End mosek.fusion.Model.getSolutionStatus

// Begin mosek.fusion.Model.update
// Method mosek.fusion.Model.update @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3298:6-3432:7
void mosek::fusion::p_Model::update(std::shared_ptr< monty::ndarray< int32_t,1 > > _1126_conidxs,monty::rc_ptr< ::mosek::fusion::Expression > _1127_expr)
{
  _checked_ptr_1<int32_t> _ptr__1126_conidxs(_1126_conidxs ? _1126_conidxs->raw() : nullptr, _1126_conidxs ? _1126_conidxs->size(0) : 0);
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1127_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>())->eval(ws,rs,xs);
  ws->pop_expr();
  int32_t _1128_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1129_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1130_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1131_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1132_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1133_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1134_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1135_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1136_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1137_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1138_code = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1139_codeptr = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1140_cconst = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1141_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1141_wi32(_1141_wi32 ? _1141_wi32->raw() : nullptr,_1141_wi32 ? _1141_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1142_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1142_wi64(_1142_wi64 ? _1142_wi64->raw() : nullptr,_1142_wi64 ? _1142_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1143_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1143_wf64(_1143_wf64 ? _1143_wf64->raw() : nullptr,_1143_wf64 ? _1143_wf64->size(0) : 0);
  int32_t _1144_num_lincon = (int)0;
  int32_t _9592_ = (int)0;
  int32_t _9593_ = _1130_nelem;
  for (int32_t _1145_i = _9592_; _1145_i < _9593_; ++_1145_i)
  {
    {
      if ((_ptr__1126_conidxs[_1145_i] >= (int)0))
      {
        {
          ++ _1144_num_lincon;
        }
      }
      {}
    }
  }
  int32_t _1146_num_concon = (_1130_nelem - _1144_num_lincon);
  if ((_1144_num_lincon == _1130_nelem))
  {
    {
      putarows(_1126_conidxs,ws,_1130_nelem,_1131_nnz,_1134_ptr_base,_1136_nidxs_base,_1137_cof_base);
      if ((_1133_ncodeatom > (int)0))
      {
        {
          con_blocks->replace_row_code(rs,_1126_conidxs,_1134_ptr_base,_1136_nidxs_base,_1139_codeptr,_1138_code,_1140_cconst);
        }
      }
      else
      {
        {
          con_blocks->clear_row_code(_1126_conidxs);
        }
      }
    }
  }
  else if((_1146_num_concon == _1130_nelem))
  {
    {
      std::vector<int32_t> _9594_;
      for (int32_t _1147_i = (int)0, _9595_ = _1130_nelem; _1147_i < _9595_ ; ++_1147_i)
      {
        _9594_.push_back((-safe_add( (int)1,_ptr__1126_conidxs[_1147_i] )));
      }
      auto _9596_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9594_.size()),_9594_.begin(),_9594_.end()));
      putfrows(_9596_,(int)0,ws,_1130_nelem,_1131_nnz,_1134_ptr_base,_1136_nidxs_base,_1137_cof_base);
      if ((_1133_ncodeatom > (int)0))
      {
        {
          std::vector<int32_t> _9597_;
          for (int32_t _1148_i = (int)0, _9598_ = _1130_nelem; _1148_i < _9598_ ; ++_1148_i)
          {
            _9597_.push_back((-safe_add( (int)1,_ptr__1126_conidxs[_1148_i] )));
          }
          auto _9599_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9597_.size()),_9597_.begin(),_9597_.end()));
          afe_blocks->replace_row_code(rs,_9599_,_1134_ptr_base,_1136_nidxs_base,_1139_codeptr,_1138_code,_1140_cconst);
        }
      }
      else
      {
        {
          std::vector<int32_t> _9600_;
          for (int32_t _1149_i = (int)0, _9601_ = _1130_nelem; _1149_i < _9601_ ; ++_1149_i)
          {
            _9600_.push_back((-safe_add( (int)1,_ptr__1126_conidxs[_1149_i] )));
          }
          auto _9602_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9600_.size()),_9600_.begin(),_9600_.end()));
          afe_blocks->clear_row_code(_9602_);
        }
      }
    }
  }
  else
  {
    {
      int32_t _1150_cncodeelm = (int)0;
      int32_t _1151_lncodeelm = (int)0;
      int32_t _1152_cnnz = (int)0;
      int32_t _1153_lnnz = (int)0;
      int32_t _9603_ = (int)0;
      int32_t _9604_ = _1130_nelem;
      for (int32_t _1154_i = _9603_; _1154_i < _9604_; ++_1154_i)
      {
        {
          if ((_ptr__1126_conidxs[_1154_i] >= (int)0))
          {
            {
              _1153_lnnz += (_ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1154_i ),(int)1 )] - _ptr__1141_wi32[safe_add( _1134_ptr_base,_1154_i )]);
            }
          }
          else
          {
            {
              _1152_cnnz += (_ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1154_i ),(int)1 )] - _ptr__1141_wi32[safe_add( _1134_ptr_base,_1154_i )]);
            }
          }
        }
      }
      int32_t _1155_lrowidx = xs->alloci32(_1144_num_lincon);
      int32_t _1156_crowidx = xs->alloci32(_1146_num_concon);
      int32_t _1157_lptr = xs->alloci32(safe_add( _1144_num_lincon,(int)1 ));
      int32_t _1158_cptr = xs->alloci32(safe_add( _1146_num_concon,(int)1 ));
      int32_t _1159_lnidxs = xs->alloci64(_1153_lnnz);
      int32_t _1160_cnidxs = xs->alloci64(_1152_cnnz);
      int32_t _1161_lcof = xs->allocf64(_1153_lnnz);
      int32_t _1162_ccof = xs->allocf64(_1152_cnnz);
      std::shared_ptr< monty::ndarray< double,1 > > _1163_xf64 = mosek::fusion::p_WorkStack::_get_impl(xs)->f64;
      _checked_ptr_1<double> _ptr__1163_xf64(_1163_xf64 ? _1163_xf64->raw() : nullptr,_1163_xf64 ? _1163_xf64->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1164_xi64 = mosek::fusion::p_WorkStack::_get_impl(xs)->i64;
      _checked_ptr_1<int64_t> _ptr__1164_xi64(_1164_xi64 ? _1164_xi64->raw() : nullptr,_1164_xi64 ? _1164_xi64->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1165_xi32 = mosek::fusion::p_WorkStack::_get_impl(xs)->i32;
      _checked_ptr_1<int32_t> _ptr__1165_xi32(_1165_xi32 ? _1165_xi32->raw() : nullptr,_1165_xi32 ? _1165_xi32->size(0) : 0);
      _ptr__1165_xi32[_1157_lptr] = (int)0;
      _ptr__1165_xi32[_1158_cptr] = (int)0;
      int32_t _1166_ci = (int)0;
      int32_t _1167_li = (int)0;
      int32_t _1168_cnzi = (int)0;
      int32_t _1169_lnzi = (int)0;
      int32_t _9605_ = (int)0;
      int32_t _9606_ = _1130_nelem;
      for (int32_t _1170_i = _9605_; _1170_i < _9606_; ++_1170_i)
      {
        {
          int32_t _1171_pb = _ptr__1141_wi32[safe_add( _1134_ptr_base,_1170_i )];
          int32_t _1172_pe = _ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1170_i ),(int)1 )];
          if ((_ptr__1126_conidxs[_1170_i] >= (int)0))
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1142_wi64,safe_add( _1136_nidxs_base,_1171_pb ),_1164_xi64,safe_add( _1159_lnidxs,_1169_lnzi ),(_1172_pe - _1171_pb));
              ::mosek::fusion::Utils::Tools::arraycopy(_1143_wf64,safe_add( _1137_cof_base,_1171_pb ),_1163_xf64,safe_add( _1161_lcof,_1169_lnzi ),(_1172_pe - _1171_pb));
              _1169_lnzi += (_1172_pe - _1171_pb);
              _ptr__1165_xi32[safe_add( _1155_lrowidx,_1167_li )] = _ptr__1126_conidxs[_1170_i];
              _ptr__1165_xi32[safe_add( safe_add( _1157_lptr,_1167_li ),(int)1 )] = _1169_lnzi;
              ++ _1167_li;
            }
          }
          else
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1142_wi64,safe_add( _1136_nidxs_base,_1171_pb ),_1164_xi64,safe_add( _1160_cnidxs,_1168_cnzi ),(_1172_pe - _1171_pb));
              ::mosek::fusion::Utils::Tools::arraycopy(_1143_wf64,safe_add( _1137_cof_base,_1171_pb ),_1163_xf64,safe_add( _1162_ccof,_1168_cnzi ),(_1172_pe - _1171_pb));
              _1168_cnzi += (_1172_pe - _1171_pb);
              _ptr__1165_xi32[safe_add( _1156_crowidx,_1166_ci )] = (-safe_add( (int)1,_ptr__1126_conidxs[_1170_i] ));
              _ptr__1165_xi32[safe_add( safe_add( _1158_cptr,_1166_ci ),(int)1 )] = _1168_cnzi;
              ++ _1166_ci;
            }
          }
        }
      }
      std::vector<int32_t> _9607_;
      for (int32_t _1173_i = (int)0, _9608_ = _1144_num_lincon; _1173_i < _9608_ ; ++_1173_i)
      {
        _9607_.push_back(_ptr__1165_xi32[safe_add( _1155_lrowidx,_1173_i )]);
      }
      auto _9609_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9607_.size()),_9607_.begin(),_9607_.end()));
      putarows(_9609_,xs,_1144_num_lincon,_1153_lnnz,_1157_lptr,_1159_lnidxs,_1161_lcof);
      putfrows(_1165_xi32,_1156_crowidx,xs,_1146_num_concon,_1152_cnnz,_1158_cptr,_1160_cnidxs,_1162_ccof);
      if ((_1133_ncodeatom > (int)0))
      {
        {
          int32_t _9610_ = (int)0;
          int32_t _9611_ = _1130_nelem;
          for (int32_t _1174_i = _9610_; _1174_i < _9611_; ++_1174_i)
          {
            {
              if ((_ptr__1126_conidxs[_1174_i] >= (int)0))
              {
                {
                  _1151_lncodeelm += (_ptr__1141_wi32[safe_add( _1139_codeptr,_ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1174_i ),(int)1 )] )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_ptr__1141_wi32[safe_add( _1134_ptr_base,_1174_i )] )]);
                }
              }
              else
              {
                {
                  _1150_cncodeelm += (_ptr__1141_wi32[safe_add( _1139_codeptr,_ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1174_i ),(int)1 )] )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_ptr__1141_wi32[safe_add( _1134_ptr_base,_1174_i )] )]);
                }
              }
            }
          }
          int32_t _1175_lcodeptr = xs->alloci32(safe_add( _1153_lnnz,(int)1 ));
          int32_t _1176_ccodeptr = xs->alloci32(safe_add( _1152_cnnz,(int)1 ));
          int32_t _1177_lcode = xs->alloci32(_1151_lncodeelm);
          int32_t _1178_ccode = xs->alloci32(_1150_cncodeelm);
          int32_t _1179_lcconst = xs->allocf64(_1151_lncodeelm);
          int32_t _1180_ccconst = xs->allocf64(_1150_cncodeelm);
          _ptr__1165_xi32[_1175_lcodeptr] = (int)0;
          _ptr__1165_xi32[_1176_ccodeptr] = (int)0;
          int32_t _1181_ccodei = (int)0;
          int32_t _1182_lcodei = (int)0;
          int32_t _1183_rnzi = (int)0;
          int32_t _1184_lnzi = (int)0;
          int32_t _9612_ = (int)0;
          int32_t _9613_ = _1130_nelem;
          for (int32_t _1185_i = _9612_; _1185_i < _9613_; ++_1185_i)
          {
            {
              int32_t _1186_pb = _ptr__1141_wi32[safe_add( _1134_ptr_base,_1185_i )];
              int32_t _1187_pe = _ptr__1141_wi32[safe_add( safe_add( _1134_ptr_base,_1185_i ),(int)1 )];
              if ((_ptr__1126_conidxs[_1185_i] >= (int)0))
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1141_wi32,safe_add( _1138_code,_ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )] ),_1165_xi32,safe_add( _1177_lcode,_1182_lcodei ),(_ptr__1141_wi32[safe_add( _1139_codeptr,_1187_pe )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1143_wf64,safe_add( _1140_cconst,_ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )] ),_1163_xf64,safe_add( _1179_lcconst,_1182_lcodei ),(_ptr__1141_wi32[safe_add( _1139_codeptr,_1187_pe )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )]));
                  int32_t _9614_ = _1186_pb;
                  int32_t _9615_ = _1187_pe;
                  for (int32_t _1188_j = _9614_; _1188_j < _9615_; ++_1188_j)
                  {
                    {
                      _1182_lcodei += (_ptr__1141_wi32[safe_add( safe_add( _1139_codeptr,_1188_j ),(int)1 )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1188_j )]);
                      _ptr__1165_xi32[safe_add( safe_add( _1175_lcodeptr,_1184_lnzi ),(int)1 )] = _1182_lcodei;
                      ++ _1184_lnzi;
                    }
                  }
                }
              }
              else
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1141_wi32,safe_add( _1138_code,_ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )] ),_1165_xi32,safe_add( _1178_ccode,_1181_ccodei ),(_ptr__1141_wi32[safe_add( _1139_codeptr,_1187_pe )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1143_wf64,safe_add( _1140_cconst,_ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )] ),_1163_xf64,safe_add( _1180_ccconst,_1181_ccodei ),(_ptr__1141_wi32[safe_add( _1139_codeptr,_1187_pe )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1186_pb )]));
                  int32_t _9616_ = _1186_pb;
                  int32_t _9617_ = _1187_pe;
                  for (int32_t _1189_j = _9616_; _1189_j < _9617_; ++_1189_j)
                  {
                    {
                      _1181_ccodei += (_ptr__1141_wi32[safe_add( safe_add( _1139_codeptr,_1189_j ),(int)1 )] - _ptr__1141_wi32[safe_add( _1139_codeptr,_1189_j )]);
                      _ptr__1165_xi32[safe_add( safe_add( _1176_ccodeptr,_1168_cnzi ),(int)1 )] = _1181_ccodei;
                      ++ _1168_cnzi;
                    }
                  }
                }
              }
            }
          }
          std::vector<int32_t> _9618_;
          for (int32_t _1190_i = (int)0, _9619_ = _1144_num_lincon; _1190_i < _9619_ ; ++_1190_i)
          {
            _9618_.push_back(_ptr__1165_xi32[safe_add( _1155_lrowidx,_1190_i )]);
          }
          auto _9620_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9618_.size()),_9618_.begin(),_9618_.end()));
          con_blocks->replace_row_code(xs,_9620_,_1157_lptr,_1159_lnidxs,_1175_lcodeptr,_1177_lcode,_1179_lcconst);
          std::vector<int32_t> _9621_;
          for (int32_t _1191_i = (int)0, _9622_ = _1146_num_concon; _1191_i < _9622_ ; ++_1191_i)
          {
            _9621_.push_back((int32_t)_ptr__1164_xi64[safe_add( _1156_crowidx,_1191_i )]);
          }
          auto _9623_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9621_.size()),_9621_.begin(),_9621_.end()));
          afe_blocks->replace_row_code(xs,_9623_,_1158_cptr,_1160_cnidxs,_1176_ccodeptr,_1178_ccode,_1180_ccconst);
        }
      }
      else
      {
        {
          std::vector<int32_t> _9624_;
          for (int32_t _1192_i = (int)0, _9625_ = _1144_num_lincon; _1192_i < _9625_ ; ++_1192_i)
          {
            _9624_.push_back(_ptr__1165_xi32[safe_add( _1155_lrowidx,_1192_i )]);
          }
          auto _9626_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9624_.size()),_9624_.begin(),_9624_.end()));
          con_blocks->clear_row_code(_9626_);
          std::vector<int32_t> _9627_;
          for (int32_t _1193_i = (int)0, _9628_ = _1146_num_concon; _1193_i < _9628_ ; ++_1193_i)
          {
            _9627_.push_back((int32_t)_ptr__1164_xi64[safe_add( _1156_crowidx,_1193_i )]);
          }
          auto _9629_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9627_.size()),_9627_.begin(),_9627_.end()));
          afe_blocks->clear_row_code(_9629_);
        }
      }
    }
  }
  xs->clear();
}
// End mosek.fusion.Model.update

// Begin mosek.fusion.Model.update
// Method mosek.fusion.Model.update @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:3083:6-3294:7
void mosek::fusion::p_Model::update(std::shared_ptr< monty::ndarray< int32_t,1 > > _1194_conidxs,monty::rc_ptr< ::mosek::fusion::Expression > _1195_expr,std::shared_ptr< monty::ndarray< int32_t,1 > > _1196_varidxs)
{
  _checked_ptr_1<int32_t> _ptr__1194_conidxs(_1194_conidxs ? _1194_conidxs->raw() : nullptr, _1194_conidxs ? _1194_conidxs->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__1196_varidxs(_1196_varidxs ? _1196_varidxs->raw() : nullptr, _1196_varidxs ? _1196_varidxs->size(0) : 0);
  int32_t _9630_ = (int)0;
  int32_t _9631_ = (int)((_1194_conidxs)->size(0));
  for (int32_t _1197_i = _9630_; _1197_i < _9631_; ++_1197_i)
  {
    {
      if (((_ptr__1194_conidxs[_1197_i] >= (int)0) && con_blocks->row_is_parameterized(_1197_i)))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
        }
      }
      else if(((_ptr__1194_conidxs[_1197_i] < (int)0) && con_blocks->row_is_parameterized((-safe_add( _1197_i,(int)1 )))))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
        }
      }
      {}
    }
  }
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1195_expr)).as<::mosek::fusion::Expression>())->eval(ws,rs,xs);
  ws->pop_expr();
  int32_t _1198_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1199_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1200_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1201_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1202_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1203_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1204_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1205_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1206_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1207_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1208_code = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1209_codeptr = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1210_cconst = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1211_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1211_wi32(_1211_wi32 ? _1211_wi32->raw() : nullptr,_1211_wi32 ? _1211_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1212_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1212_wi64(_1212_wi64 ? _1212_wi64->raw() : nullptr,_1212_wi64 ? _1212_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1213_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1213_wf64(_1213_wf64 ? _1213_wf64->raw() : nullptr,_1213_wf64 ? _1213_wf64->size(0) : 0);
  int32_t _1214_num_lincon = (int)0;
  int32_t _9632_ = (int)0;
  int32_t _9633_ = _1199_nelem;
  for (int32_t _1215_i = _9632_; _1215_i < _9633_; ++_1215_i)
  {
    {
      if ((_ptr__1194_conidxs[_1215_i] >= (int)0))
      {
        {
          ++ _1214_num_lincon;
        }
      }
      {}
    }
  }
  int32_t _1216_num_concon = (_1199_nelem - _1214_num_lincon);
  if (((_1214_num_lincon == _1199_nelem) || (_1216_num_concon == _1199_nelem)))
  {
    {
      int32_t _1217_N = safe_mul( (int)((_1194_conidxs)->size(0)),(int)((_1196_varidxs)->size(0)) );
      std::shared_ptr< monty::ndarray< double,1 > > _1218_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1217_N)));
      _checked_ptr_1<double> _ptr__1218_cof(_1218_cof ? _1218_cof->raw() : nullptr,_1218_cof ? _1218_cof->size(0) : 0);
      int32_t _1219_nzi = (int)0;
      int32_t _9634_ = (int)0;
      int32_t _9635_ = _1199_nelem;
      for (int32_t _1220_i = _9634_; _1220_i < _9635_; ++_1220_i)
      {
        {
          int32_t _1221_p0 = _ptr__1211_wi32[safe_add( _1204_ptr_base,_1220_i )];
          int32_t _1222_p1 = _ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1220_i ),(int)1 )];
          int32_t _1223_p = _1221_p0;
          int32_t _1224_l = (int)0;
          while ( ((_1224_l < (int)((_1196_varidxs)->size(0))) && (_1223_p < _1222_p1)) )
          {
            {
              if ((_ptr__1196_varidxs[_1224_l] < _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1223_p )]))
              {
                {
                  ++ _1224_l;
                }
              }
              else if((_ptr__1196_varidxs[_1224_l] > _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1223_p )]))
              {
                {
                  ++ _1223_p;
                }
              }
              else
              {
                {
                  _ptr__1218_cof[safe_add( _1219_nzi,_1224_l )] = _ptr__1213_wf64[safe_add( _1207_cof_base,_1223_p )];
                  ++ _1224_l;
                  ++ _1223_p;
                }
              }
            }
          }
          _1219_nzi += (int)((_1196_varidxs)->size(0));
        }
      }
      std::vector<int32_t> _9636_;
      for (int32_t _1226_i = (int)0, _9637_ = _1199_nelem; _1226_i < _9637_ ; ++_1226_i)
      {
        for (int32_t _1227_j = (int)0, _9638_ = (int)((_1196_varidxs)->size(0)); _1227_j < _9638_ ; ++_1227_j)
        {
          _9636_.push_back(_ptr__1196_varidxs[_1227_j]);
        }
      }
      auto _9639_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9636_.size()),_9636_.begin(),_9636_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1225_subj = _9639_;
      _checked_ptr_1<int32_t> _ptr__1225_subj(_1225_subj ? _1225_subj->raw() : nullptr,_1225_subj ? _1225_subj->size(0) : 0);
      if ((_1214_num_lincon == _1199_nelem))
      {
        {
          std::vector<int32_t> _9640_;
          for (int32_t _1229_i = (int)0, _9641_ = _1199_nelem; _1229_i < _9641_ ; ++_1229_i)
          {
            for (int32_t _1230_j = (int)0, _9642_ = (int)((_1196_varidxs)->size(0)); _1230_j < _9642_ ; ++_1230_j)
            {
              _9640_.push_back(_ptr__1194_conidxs[_1229_i]);
            }
          }
          auto _9643_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9640_.size()),_9640_.begin(),_9640_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1228_subi = _9643_;
          _checked_ptr_1<int32_t> _ptr__1228_subi(_1228_subi ? _1228_subi->raw() : nullptr,_1228_subi ? _1228_subi->size(0) : 0);
          task_putaijlist(_1228_subi,_1225_subj,_1218_cof,(int64_t)(_1219_nzi));
          if ((_1202_ncodeatom > (int)0))
          {
            {
              con_blocks->replace_row_code(rs,_1194_conidxs,_1204_ptr_base,_1206_nidxs_base,_1209_codeptr,_1208_code,_1210_cconst);
            }
          }
          {}
        }
      }
      else
      {
        {
          std::vector<int32_t> _9644_;
          for (int32_t _1232_i = (int)0, _9645_ = _1199_nelem; _1232_i < _9645_ ; ++_1232_i)
          {
            for (int32_t _1233_j = (int)0, _9646_ = (int)((_1196_varidxs)->size(0)); _1233_j < _9646_ ; ++_1233_j)
            {
              _9644_.push_back((-safe_add( _ptr__1194_conidxs[_1232_i],(int)1 )));
            }
          }
          auto _9647_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9644_.size()),_9644_.begin(),_9644_.end()));
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1231_subi = _9647_;
          _checked_ptr_1<int32_t> _ptr__1231_subi(_1231_subi ? _1231_subi->raw() : nullptr,_1231_subi ? _1231_subi->size(0) : 0);
          task_putafefijlist(_1231_subi,_1225_subj,_1218_cof);
          if ((_1202_ncodeatom > (int)0))
          {
            {
              std::vector<int32_t> _9648_;
              for (int32_t _1234_i = (int)0, _9649_ = _1199_nelem; _1234_i < _9649_ ; ++_1234_i)
              {
                _9648_.push_back((-safe_add( (int)1,_ptr__1194_conidxs[_1234_i] )));
              }
              auto _9650_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9648_.size()),_9648_.begin(),_9648_.end()));
              afe_blocks->replace_row_code(rs,_9650_,_1204_ptr_base,_1206_nidxs_base,_1209_codeptr,_1208_code,_1210_cconst);
            }
          }
          {}
        }
      }
    }
  }
  else
  {
    {
      int32_t _1235_cN = safe_mul( _1216_num_concon,(int)((_1196_varidxs)->size(0)) );
      int32_t _1236_lN = safe_mul( _1214_num_lincon,(int)((_1196_varidxs)->size(0)) );
      std::shared_ptr< monty::ndarray< double,1 > > _1237_lcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1236_lN)));
      _checked_ptr_1<double> _ptr__1237_lcof(_1237_lcof ? _1237_lcof->raw() : nullptr,_1237_lcof ? _1237_lcof->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1238_ccof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1235_cN)));
      _checked_ptr_1<double> _ptr__1238_ccof(_1238_ccof ? _1238_ccof->raw() : nullptr,_1238_ccof ? _1238_ccof->size(0) : 0);
      std::vector<int32_t> _9651_;
      for (int32_t _1240_i = (int)0, _9652_ = _1214_num_lincon; _1240_i < _9652_ ; ++_1240_i)
      {
        for (int32_t _1241_j = (int)0, _9653_ = (int)((_1196_varidxs)->size(0)); _1241_j < _9653_ ; ++_1241_j)
        {
          _9651_.push_back(_ptr__1196_varidxs[_1241_j]);
        }
      }
      auto _9654_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9651_.size()),_9651_.begin(),_9651_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1239_lsubj = _9654_;
      _checked_ptr_1<int32_t> _ptr__1239_lsubj(_1239_lsubj ? _1239_lsubj->raw() : nullptr,_1239_lsubj ? _1239_lsubj->size(0) : 0);
      std::vector<int32_t> _9655_;
      for (int32_t _1243_i = (int)0, _9656_ = _1216_num_concon; _1243_i < _9656_ ; ++_1243_i)
      {
        for (int32_t _1244_j = (int)0, _9657_ = (int)((_1196_varidxs)->size(0)); _1244_j < _9657_ ; ++_1244_j)
        {
          _9655_.push_back(_ptr__1196_varidxs[_1244_j]);
        }
      }
      auto _9658_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9655_.size()),_9655_.begin(),_9655_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1242_csubj = _9658_;
      _checked_ptr_1<int32_t> _ptr__1242_csubj(_1242_csubj ? _1242_csubj->raw() : nullptr,_1242_csubj ? _1242_csubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1245_lsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1214_num_lincon)));
      _checked_ptr_1<int32_t> _ptr__1245_lsubi(_1245_lsubi ? _1245_lsubi->raw() : nullptr,_1245_lsubi ? _1245_lsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1246_csubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1216_num_concon)));
      _checked_ptr_1<int32_t> _ptr__1246_csubi(_1246_csubi ? _1246_csubi->raw() : nullptr,_1246_csubi ? _1246_csubi->size(0) : 0);
      int32_t _1247_crowi = (int)0;
      int32_t _1248_lrowi = (int)0;
      int32_t _1249_cncodeelm = (int)0;
      int32_t _1250_lncodeelm = (int)0;
      int32_t _1251_cnnz = (int)0;
      int32_t _1252_lnnz = (int)0;
      int32_t _9659_ = (int)0;
      int32_t _9660_ = _1199_nelem;
      for (int32_t _1253_i = _9659_; _1253_i < _9660_; ++_1253_i)
      {
        {
          int32_t _1254_p0 = _ptr__1211_wi32[safe_add( _1204_ptr_base,_1253_i )];
          int32_t _1255_p1 = _ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1253_i ),(int)1 )];
          int32_t _1256_p = _1254_p0;
          int32_t _1257_l = (int)0;
          if ((_ptr__1194_conidxs[_1253_i] >= (int)0))
          {
            {
              while ( ((_1257_l < (int)((_1196_varidxs)->size(0))) && (_1256_p < _1255_p1)) )
              {
                {
                  if ((_ptr__1196_varidxs[_1257_l] < _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1256_p )]))
                  {
                    {
                      ++ _1257_l;
                    }
                  }
                  else if((_ptr__1196_varidxs[_1257_l] > _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1256_p )]))
                  {
                    {
                      ++ _1256_p;
                    }
                  }
                  else
                  {
                    {
                      _ptr__1237_lcof[safe_add( _1252_lnnz,_1257_l )] = _ptr__1213_wf64[safe_add( _1207_cof_base,_1256_p )];
                      ++ _1257_l;
                      ++ _1256_p;
                    }
                  }
                }
              }
              int32_t _9661_ = (int)0;
              int32_t _9662_ = (int)((_1196_varidxs)->size(0));
              for (int32_t _1258_j = _9661_; _1258_j < _9662_; ++_1258_j)
              {
                {
                  _ptr__1245_lsubi[safe_add( _1252_lnnz,_1258_j )] = _ptr__1194_conidxs[_1253_i];
                }
              }
              _1252_lnnz += (int)((_1196_varidxs)->size(0));
            }
          }
          else
          {
            {
              while ( ((_1257_l < (int)((_1196_varidxs)->size(0))) && (_1256_p < _1255_p1)) )
              {
                {
                  if ((_ptr__1196_varidxs[_1257_l] < _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1256_p )]))
                  {
                    {
                      ++ _1257_l;
                    }
                  }
                  else if((_ptr__1196_varidxs[_1257_l] > _ptr__1212_wi64[safe_add( _1206_nidxs_base,_1256_p )]))
                  {
                    {
                      ++ _1256_p;
                    }
                  }
                  else
                  {
                    {
                      _ptr__1238_ccof[safe_add( _1251_cnnz,_1257_l )] = _ptr__1213_wf64[safe_add( _1207_cof_base,_1256_p )];
                      ++ _1257_l;
                      ++ _1256_p;
                    }
                  }
                }
              }
              int32_t _9663_ = (int)0;
              int32_t _9664_ = (int)((_1196_varidxs)->size(0));
              for (int32_t _1259_j = _9663_; _1259_j < _9664_; ++_1259_j)
              {
                {
                  _ptr__1246_csubi[safe_add( _1251_cnnz,_1259_j )] = (-safe_add( (int)1,_ptr__1194_conidxs[_1253_i] ));
                }
              }
              _1251_cnnz += (int)((_1196_varidxs)->size(0));
            }
          }
        }
      }
      task_putaijlist(_1245_lsubi,_1239_lsubj,_1237_lcof,(int64_t)(_1252_lnnz));
      task_putafefijlist(_1246_csubi,_1242_csubj,_1238_ccof);
      if ((_1202_ncodeatom > (int)0))
      {
        {
          int32_t _9665_ = (int)0;
          int32_t _9666_ = _1199_nelem;
          for (int32_t _1260_i = _9665_; _1260_i < _9666_; ++_1260_i)
          {
            {
              if ((_ptr__1194_conidxs[_1260_i] >= (int)0))
              {
                {
                  _1252_lnnz += (_ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1260_i ),(int)1 )] - _ptr__1211_wi32[safe_add( _1204_ptr_base,_1260_i )]);
                }
              }
              else
              {
                {
                  _1251_cnnz += (_ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1260_i ),(int)1 )] - _ptr__1211_wi32[safe_add( _1204_ptr_base,_1260_i )]);
                }
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1261_lrowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1214_num_lincon)));
          _checked_ptr_1<int32_t> _ptr__1261_lrowidx(_1261_lrowidx ? _1261_lrowidx->raw() : nullptr,_1261_lrowidx ? _1261_lrowidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1262_crowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1216_num_concon)));
          _checked_ptr_1<int32_t> _ptr__1262_crowidx(_1262_crowidx ? _1262_crowidx->raw() : nullptr,_1262_crowidx ? _1262_crowidx->size(0) : 0);
          int32_t _1263_lptr = xs->alloci32(safe_add( _1214_num_lincon,(int)1 ));
          int32_t _1264_cptr = xs->alloci32(safe_add( _1216_num_concon,(int)1 ));
          int32_t _1265_lnidxs = xs->alloci64(_1252_lnnz);
          int32_t _1266_cnidxs = xs->alloci64(_1251_cnnz);
          std::shared_ptr< monty::ndarray< double,1 > > _1267_xf64 = mosek::fusion::p_WorkStack::_get_impl(xs)->f64;
          _checked_ptr_1<double> _ptr__1267_xf64(_1267_xf64 ? _1267_xf64->raw() : nullptr,_1267_xf64 ? _1267_xf64->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _1268_xi64 = mosek::fusion::p_WorkStack::_get_impl(xs)->i64;
          _checked_ptr_1<int64_t> _ptr__1268_xi64(_1268_xi64 ? _1268_xi64->raw() : nullptr,_1268_xi64 ? _1268_xi64->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1269_xi32 = mosek::fusion::p_WorkStack::_get_impl(xs)->i32;
          _checked_ptr_1<int32_t> _ptr__1269_xi32(_1269_xi32 ? _1269_xi32->raw() : nullptr,_1269_xi32 ? _1269_xi32->size(0) : 0);
          _ptr__1269_xi32[_1263_lptr] = (int)0;
          _ptr__1269_xi32[_1264_cptr] = (int)0;
          int32_t _1270_ci = (int)0;
          int32_t _1271_li = (int)0;
          int32_t _1272_cnzi = (int)0;
          int32_t _1273_lnzi = (int)0;
          int32_t _9667_ = (int)0;
          int32_t _9668_ = _1199_nelem;
          for (int32_t _1274_i = _9667_; _1274_i < _9668_; ++_1274_i)
          {
            {
              int32_t _1275_pb = _ptr__1211_wi32[safe_add( _1204_ptr_base,_1274_i )];
              int32_t _1276_pe = _ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1274_i ),(int)1 )];
              if ((_ptr__1194_conidxs[_1274_i] >= (int)0))
              {
                {
                  _1273_lnzi += (_1276_pe - _1275_pb);
                  _ptr__1261_lrowidx[_1271_li] = _ptr__1194_conidxs[_1274_i];
                  _ptr__1269_xi32[safe_add( safe_add( _1263_lptr,_1271_li ),(int)1 )] = _1273_lnzi;
                  ++ _1271_li;
                }
              }
              else
              {
                {
                  _1272_cnzi += (_1276_pe - _1275_pb);
                  _ptr__1262_crowidx[_1270_ci] = (-safe_add( (int)1,_ptr__1194_conidxs[_1274_i] ));
                  _ptr__1269_xi32[safe_add( safe_add( _1264_cptr,_1270_ci ),(int)1 )] = _1272_cnzi;
                  ++ _1270_ci;
                }
              }
            }
          }
          int32_t _9669_ = (int)0;
          int32_t _9670_ = _1199_nelem;
          for (int32_t _1277_i = _9669_; _1277_i < _9670_; ++_1277_i)
          {
            {
              if ((_ptr__1194_conidxs[_1277_i] >= (int)0))
              {
                {
                  _1250_lncodeelm += (_ptr__1211_wi32[safe_add( _1209_codeptr,_ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1277_i ),(int)1 )] )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_ptr__1211_wi32[safe_add( _1204_ptr_base,_1277_i )] )]);
                }
              }
              else
              {
                {
                  _1249_cncodeelm += (_ptr__1211_wi32[safe_add( _1209_codeptr,_ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1277_i ),(int)1 )] )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_ptr__1211_wi32[safe_add( _1204_ptr_base,_1277_i )] )]);
                }
              }
            }
          }
          int32_t _1278_lcodeptr = xs->alloci32(safe_add( _1252_lnnz,(int)1 ));
          int32_t _1279_ccodeptr = xs->alloci32(safe_add( _1251_cnnz,(int)1 ));
          int32_t _1280_lcode = xs->alloci32(_1250_lncodeelm);
          int32_t _1281_ccode = xs->alloci32(_1249_cncodeelm);
          int32_t _1282_lcconst = xs->allocf64(_1250_lncodeelm);
          int32_t _1283_ccconst = xs->allocf64(_1249_cncodeelm);
          _ptr__1269_xi32[_1278_lcodeptr] = (int)0;
          _ptr__1269_xi32[_1279_ccodeptr] = (int)0;
          int32_t _1284_ccodei = (int)0;
          int32_t _1285_lcodei = (int)0;
          int32_t _1286_rnzi = (int)0;
          int32_t _1287_lnzi = (int)0;
          int32_t _9671_ = (int)0;
          int32_t _9672_ = _1199_nelem;
          for (int32_t _1288_i = _9671_; _1288_i < _9672_; ++_1288_i)
          {
            {
              int32_t _1289_pb = _ptr__1211_wi32[safe_add( _1204_ptr_base,_1288_i )];
              int32_t _1290_pe = _ptr__1211_wi32[safe_add( safe_add( _1204_ptr_base,_1288_i ),(int)1 )];
              if ((_ptr__1194_conidxs[_1288_i] >= (int)0))
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1211_wi32,safe_add( _1208_code,_ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )] ),_1269_xi32,safe_add( _1280_lcode,_1285_lcodei ),(_ptr__1211_wi32[safe_add( _1209_codeptr,_1290_pe )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1213_wf64,safe_add( _1210_cconst,_ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )] ),_1267_xf64,safe_add( _1282_lcconst,_1285_lcodei ),(_ptr__1211_wi32[safe_add( _1209_codeptr,_1290_pe )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )]));
                  int32_t _9673_ = _1289_pb;
                  int32_t _9674_ = _1290_pe;
                  for (int32_t _1291_j = _9673_; _1291_j < _9674_; ++_1291_j)
                  {
                    {
                      _1285_lcodei += (_ptr__1211_wi32[safe_add( safe_add( _1209_codeptr,_1291_j ),(int)1 )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1291_j )]);
                      _ptr__1269_xi32[safe_add( safe_add( _1278_lcodeptr,_1287_lnzi ),(int)1 )] = _1285_lcodei;
                      ++ _1287_lnzi;
                    }
                  }
                }
              }
              else
              {
                {
                  ::mosek::fusion::Utils::Tools::arraycopy(_1211_wi32,safe_add( _1208_code,_ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )] ),_1269_xi32,safe_add( _1281_ccode,_1284_ccodei ),(_ptr__1211_wi32[safe_add( _1209_codeptr,_1290_pe )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )]));
                  ::mosek::fusion::Utils::Tools::arraycopy(_1213_wf64,safe_add( _1210_cconst,_ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )] ),_1267_xf64,safe_add( _1283_ccconst,_1284_ccodei ),(_ptr__1211_wi32[safe_add( _1209_codeptr,_1290_pe )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1289_pb )]));
                  int32_t _9675_ = _1289_pb;
                  int32_t _9676_ = _1290_pe;
                  for (int32_t _1292_j = _9675_; _1292_j < _9676_; ++_1292_j)
                  {
                    {
                      _1284_ccodei += (_ptr__1211_wi32[safe_add( safe_add( _1209_codeptr,_1292_j ),(int)1 )] - _ptr__1211_wi32[safe_add( _1209_codeptr,_1292_j )]);
                      _ptr__1269_xi32[safe_add( safe_add( _1279_ccodeptr,_1272_cnzi ),(int)1 )] = _1284_ccodei;
                      ++ _1272_cnzi;
                    }
                  }
                }
              }
            }
          }
          con_blocks->replace_row_code(xs,_1261_lrowidx,_1263_lptr,_1265_lnidxs,_1278_lcodeptr,_1280_lcode,_1282_lcconst);
          afe_blocks->replace_row_code(xs,_1262_crowidx,_1264_cptr,_1266_cnidxs,_1279_ccodeptr,_1281_ccode,_1283_ccconst);
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1293_lrowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1214_num_lincon)));
          _checked_ptr_1<int32_t> _ptr__1293_lrowidx(_1293_lrowidx ? _1293_lrowidx->raw() : nullptr,_1293_lrowidx ? _1293_lrowidx->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1294_crowidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1216_num_concon)));
          _checked_ptr_1<int32_t> _ptr__1294_crowidx(_1294_crowidx ? _1294_crowidx->raw() : nullptr,_1294_crowidx ? _1294_crowidx->size(0) : 0);
          int32_t _1295_ci = (int)0;
          int32_t _1296_li = (int)0;
          int32_t _9677_ = (int)0;
          int32_t _9678_ = _1199_nelem;
          for (int32_t _1297_i = _9677_; _1297_i < _9678_; ++_1297_i)
          {
            {
              if ((_ptr__1194_conidxs[_1297_i] >= (int)0))
              {
                {
                  _ptr__1293_lrowidx[_1296_li] = _ptr__1194_conidxs[_1297_i];
                  ++ _1296_li;
                }
              }
              else
              {
                {
                  _ptr__1294_crowidx[_1295_ci] = (-safe_add( (int)1,_ptr__1194_conidxs[_1297_i] ));
                  ++ _1295_ci;
                }
              }
            }
          }
          con_blocks->clear_row_code(_1293_lrowidx);
          afe_blocks->clear_row_code(_1294_crowidx);
        }
      }
    }
  }
  xs->clear();
}
// End mosek.fusion.Model.update

// Begin mosek.fusion.Model.updateObjective
// Method mosek.fusion.Model.updateObjective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2995:6-3078:7
void mosek::fusion::Model :: updateObjective(monty::rc_ptr< ::mosek::fusion::Expression > _1298_expr,monty::rc_ptr< ::mosek::fusion::Variable > _1299_x) { mosek::fusion::p_Model::_get_impl(this)->updateObjective(_1298_expr,_1299_x); }
void mosek::fusion::p_Model::updateObjective(monty::rc_ptr< ::mosek::fusion::Expression > _1298_expr,monty::rc_ptr< ::mosek::fusion::Variable > _1299_x)
{
  if ((_1298_expr->getSize() != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid expression size for objective"));
    }
  }
  {}
  if ((mosek::fusion::p_RowBlockManager::_get_impl(obj_blocks)->code_used > (int)0))
  {
    {
      throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized constraint"));
    }
  }
  {}
  int32_t _1300_ni = _1299_x->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1301_varidxs_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1300_ni)));
  _checked_ptr_1<int64_t> _ptr__1301_varidxs_(_1301_varidxs_ ? _1301_varidxs_->raw() : nullptr,_1301_varidxs_ ? _1301_varidxs_->size(0) : 0);
  if ((_1300_ni < _1299_x->getSize()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1302_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1300_ni)));
      _checked_ptr_1<int64_t> _ptr__1302_sp(_1302_sp ? _1302_sp->raw() : nullptr,_1302_sp ? _1302_sp->size(0) : 0);
      _1299_x->inst((int)0,_1302_sp,(int)0,_1301_varidxs_);
    }
  }
  else
  {
    {
      _1299_x->inst((int)0,_1301_varidxs_);
    }
  }
  int32_t _9679_ = (int)0;
  int32_t _9680_ = _1300_ni;
  for (int32_t _1303_i = _9679_; _1303_i < _9680_; ++_1303_i)
  {
    {
      if ((_ptr__1301_varidxs_[_1303_i] < (int)0))
      {
        {
          throw ::mosek::fusion::UpdateError(std::string ("Updating semidefinite terms is currently not possible"));
        }
      }
      {}
    }
  }
  bool _1304_varidxs_issorted = true;
  int32_t _9681_ = (int)0;
  int32_t _9682_ = (_1300_ni - (int)1);
  for (int32_t _1305_i = _9681_; _1305_i < _9682_; ++_1305_i)
  {
    {
      _1304_varidxs_issorted = (_1304_varidxs_issorted && (_ptr__1301_varidxs_[_1305_i] < _ptr__1301_varidxs_[(_1305_i - (int)1)]));
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1306_varidxs = nullptr;
  _checked_ptr_1<int32_t> _ptr__1306_varidxs(_1306_varidxs ? _1306_varidxs->raw() : nullptr,_1306_varidxs ? _1306_varidxs->size(0) : 0);
  if (_1304_varidxs_issorted)
  {
    {
      std::vector<int32_t> _9683_;
      for (int32_t _1307_i = (int)0, _9684_ = _1300_ni; _1307_i < _9684_ ; ++_1307_i)
      {
        _9683_.push_back((int32_t)_ptr__1301_varidxs_[_1307_i]);
      }
      auto _9685_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9683_.size()),_9683_.begin(),_9683_.end()));
      _1306_varidxs = _9685_;
      _ptr__1306_varidxs.update(_1306_varidxs ? _1306_varidxs->raw() : nullptr, _1306_varidxs ? _1306_varidxs->size(0) : 0);
    }
  }
  else
  {
    {
      std::vector<int64_t> _9686_;
      for (int32_t _1309_i = (-(int)1), _9687_ = _1300_ni; _1309_i < _9687_ ; ++_1309_i)
      {
        _9686_.push_back(_1309_i);
      }
      auto _9688_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9686_.size()),_9686_.begin(),_9686_.end()));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1308_perm = _9688_;
      _checked_ptr_1<int64_t> _ptr__1308_perm(_1308_perm ? _1308_perm->raw() : nullptr,_1308_perm ? _1308_perm->size(0) : 0);
      ::mosek::fusion::p_CommonTools::argQsort(_1308_perm,_1301_varidxs_,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_1300_ni));
      std::vector<int32_t> _9689_;
      for (int32_t _1310_i = (int)0, _9690_ = _1300_ni; _1310_i < _9690_ ; ++_1310_i)
      {
        _9689_.push_back((int32_t)_ptr__1301_varidxs_[_ptr__1308_perm[_1310_i]]);
      }
      auto _9691_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9689_.size()),_9689_.begin(),_9689_.end()));
      _1306_varidxs = _9691_;
      _ptr__1306_varidxs.update(_1306_varidxs ? _1306_varidxs->raw() : nullptr, _1306_varidxs ? _1306_varidxs->size(0) : 0);
    }
  }
  int32_t _1311_N = (int)((_1306_varidxs)->size(0));
  std::shared_ptr< monty::ndarray< double,1 > > _1312_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1311_N)));
  _checked_ptr_1<double> _ptr__1312_cof(_1312_cof ? _1312_cof->raw() : nullptr,_1312_cof ? _1312_cof->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1313_subj = _1306_varidxs;
  _checked_ptr_1<int32_t> _ptr__1313_subj(_1313_subj ? _1313_subj->raw() : nullptr,_1313_subj ? _1313_subj->size(0) : 0);
  ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1298_expr)).as<::mosek::fusion::Expression>())->eval(ws,rs,xs);
  ws->pop_expr();
  {
  }
  int32_t _1314_nd = mosek::fusion::p_WorkStack::_get_impl(ws)->nd;
  int32_t _1315_nelem = mosek::fusion::p_WorkStack::_get_impl(ws)->nelem;
  int32_t _1316_nnz = mosek::fusion::p_WorkStack::_get_impl(ws)->nnz;
  bool _1317_hassp = mosek::fusion::p_WorkStack::_get_impl(ws)->hassp;
  int32_t _1318_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(ws)->ncodeatom;
  int32_t _1319_shape_base = mosek::fusion::p_WorkStack::_get_impl(ws)->shape_base;
  int32_t _1320_ptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->ptr_base;
  int32_t _1321_sp_base = mosek::fusion::p_WorkStack::_get_impl(ws)->sp_base;
  int32_t _1322_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(ws)->nidxs_base;
  int32_t _1323_cof_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cof_base;
  int32_t _1324_code_base = mosek::fusion::p_WorkStack::_get_impl(ws)->code_base;
  int32_t _1325_codeptr_base = mosek::fusion::p_WorkStack::_get_impl(ws)->codeptr_base;
  int32_t _1326_cconst_base = mosek::fusion::p_WorkStack::_get_impl(ws)->cconst_base;
  {
  }
  if ((_1318_ncodeatom > (int)0))
  {
    {
      throw ::mosek::fusion::UpdateError(std::string ("Cannot update parameterized objective"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1327_wi32 = mosek::fusion::p_WorkStack::_get_impl(ws)->i32;
  _checked_ptr_1<int32_t> _ptr__1327_wi32(_1327_wi32 ? _1327_wi32->raw() : nullptr,_1327_wi32 ? _1327_wi32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1328_wi64 = mosek::fusion::p_WorkStack::_get_impl(ws)->i64;
  _checked_ptr_1<int64_t> _ptr__1328_wi64(_1328_wi64 ? _1328_wi64->raw() : nullptr,_1328_wi64 ? _1328_wi64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1329_wf64 = mosek::fusion::p_WorkStack::_get_impl(ws)->f64;
  _checked_ptr_1<double> _ptr__1329_wf64(_1329_wf64 ? _1329_wf64->raw() : nullptr,_1329_wf64 ? _1329_wf64->size(0) : 0);
  int32_t _1330_k = (int)0;
  int32_t _9692_ = (int)0;
  int32_t _9693_ = _1315_nelem;
  for (int32_t _1331_i = _9692_; _1331_i < _9693_; ++_1331_i)
  {
    {
      int32_t _1332_p0 = _ptr__1327_wi32[safe_add( _1320_ptr_base,_1331_i )];
      int32_t _1333_p1 = _ptr__1327_wi32[safe_add( safe_add( _1320_ptr_base,_1331_i ),(int)1 )];
      int32_t _1334_p = _1332_p0;
      int32_t _1335_l = (int)0;
      while ( ((_1335_l < (int)((_1306_varidxs)->size(0))) && (_1334_p < _1333_p1)) )
      {
        {
          if ((_ptr__1306_varidxs[_1335_l] < _ptr__1328_wi64[safe_add( _1322_nidxs_base,_1334_p )]))
          {
            {
              ++ _1335_l;
            }
          }
          else if((_ptr__1306_varidxs[_1335_l] > _ptr__1328_wi64[safe_add( _1322_nidxs_base,_1334_p )]))
          {
            {
              ++ _1334_p;
            }
          }
          else
          {
            {
              _ptr__1312_cof[safe_add( _1330_k,_1335_l )] = _ptr__1329_wf64[safe_add( _1323_cof_base,_1334_p )];
              ++ _1335_l;
              ++ _1334_p;
            }
          }
        }
      }
      _1330_k += (int)((_1306_varidxs)->size(0));
    }
  }
  task_putclist(_1313_subj,_1312_cof);
  if ((_1318_ncodeatom > (int)0))
  {
    {
      obj_blocks->replace_row_code(rs,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0)}),_1320_ptr_base,_1322_nidxs_base,_1325_codeptr_base,_1324_code_base,_1326_cconst_base);
    }
  }
  {}
}
// End mosek.fusion.Model.updateObjective

// Begin mosek.fusion.Model.parameter_unchecked
// Method mosek.fusion.Model.parameter_unchecked @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2947:6-2980:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_unchecked(const std::string & _1336_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1337_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1338_sp)
{
  _checked_ptr_1<int32_t> _ptr__1337_shape(_1337_shape ? _1337_shape->raw() : nullptr, _1337_shape ? _1337_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1338_sp(_1338_sp ? _1338_sp->raw() : nullptr, _1338_sp ? _1338_sp->size(0) : 0);
  if ((((int)((_1336_name).size()) > (int)0) && par_map->hasItem(_1336_name)))
  {
    {
      throw ::mosek::fusion::NameError(std::string ("Duplicate parameter name"));
    }
  }
  {}
  int64_t _1339_sz = ::mosek::fusion::p_Set::size(_1337_shape);
  int32_t _9694_;
  bool _9695_ = (NULL != _1338_sp.get());
  if (_9695_)
  {
    _9694_ = (int)((_1338_sp)->size(0));
  }
  else
  {
    _9694_ = (int32_t)_1339_sz;
  }
  int32_t _1340_nelem = _9694_;
  if ((NULL == param_value.get()))
  {
    {
      param_value = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1340_nelem)));
    }
  }
  else
  {
    {
      if ((safe_add( param_num,_1340_nelem ) > (int)((param_value)->size(0))))
      {
        {
          std::shared_ptr< monty::ndarray< double,1 > > _1341_old_param_value = param_value;
          _checked_ptr_1<double> _ptr__1341_old_param_value(_1341_old_param_value ? _1341_old_param_value->raw() : nullptr,_1341_old_param_value ? _1341_old_param_value->size(0) : 0);
          int32_t _9696_;
          bool _9697_ = (safe_mul( (int)2,(int)((_1341_old_param_value)->size(0)) ) > safe_add( (int)((_1341_old_param_value)->size(0)),_1340_nelem ));
          if (_9697_)
          {
            _9696_ = safe_mul( (int)2,(int)((_1341_old_param_value)->size(0)) );
          }
          else
          {
            _9696_ = safe_add( (int)((_1341_old_param_value)->size(0)),_1340_nelem );
          }
          int32_t _1342_new_len = _9696_;
          param_value = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1342_new_len)));
          if ((param_num > (int)0))
          {
            {
              ::mosek::fusion::Utils::Tools::arraycopy(_1341_old_param_value,(int)0,param_value,(int)0,param_num);
            }
          }
          {}
        }
      }
      {}
    }
  }
  std::vector<int32_t> _9698_;
  for (int32_t _1344_i = (int)0, _9699_ = (int32_t)_1340_nelem; _1344_i < _9699_ ; ++_1344_i)
  {
    _9698_.push_back(safe_add( param_num,_1344_i ));
  }
  auto _9700_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9698_.size()),_9698_.begin(),_9698_.end()));
  monty::rc_ptr< ::mosek::fusion::ParameterImpl > _1343_r = ::mosek::fusion::p_ParameterImpl::_new_ParameterImpl(::mosek::fusion::Model::t(_pubthis),_1337_shape,_1338_sp,_9700_);
  param_num += _1340_nelem;
  if (((int)((_1336_name).size()) > (int)0))
  {
    {
      if (((NULL == parameters.get()) || (numparameter == (int)((parameters)->size(0)))))
      {
        {
          std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > > _1345_tmppar = parameters;
          std::vector<monty::rc_ptr< ::mosek::fusion::Parameter >> _9701_;
          for (int32_t _1346_i = (int)0, _9702_ = safe_add( safe_mul( numparameter,(int)2 ),(int)64 ); _1346_i < _9702_ ; ++_1346_i)
          {
            _9701_.push_back(nullptr);
          }
          auto _9703_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Parameter >,1 >(monty::shape(_9701_.size()),_9701_.begin(),_9701_.end()));
          parameters = _9703_;
          int32_t _9704_ = (int)0;
          int32_t _9705_ = numparameter;
          for (int32_t _1347_i = _9704_; _1347_i < _9705_; ++_1347_i)
          {
            {
              ((*parameters)(_1347_i)) = ((*_1345_tmppar)(_1347_i));
            }
          }
        }
      }
      {}
      ((*parameters)(numparameter)) = (_1343_r).as<::mosek::fusion::Parameter>();
      par_map->setItem(_1336_name,numparameter);
      numparameter += (int)1;
    }
  }
  {}
  return (_1343_r).as<::mosek::fusion::Parameter>();
}
// End mosek.fusion.Model.parameter_unchecked

// Begin mosek.fusion.Model.parameter_
// Method mosek.fusion.Model.parameter_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2931:6-2945:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_(const std::string & _1348_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1349_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1350_sp)
{
  _checked_ptr_1<int32_t> _ptr__1349_shape(_1349_shape ? _1349_shape->raw() : nullptr, _1349_shape ? _1349_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1350_sp(_1350_sp ? _1350_sp->raw() : nullptr, _1350_sp ? _1350_sp->size(0) : 0);
  bool _1351_sorted = true;
  int32_t _1352_nd = (int)((_1349_shape)->size(0));
  if (((NULL != _1350_sp.get()) && ((int)((_1350_sp)->size(0)) > (int)0)))
  {
    {
      bool _1353_ok = (_ptr__1350_sp[((int)((_1350_sp)->size(0)) - (int)1)] < ::mosek::fusion::p_Set::size(_1349_shape));
      int32_t _9706_ = (int)1;
      int32_t _9707_ = (int)((_1350_sp)->size(0));
      for (int32_t _1354_i = _9706_; _1354_i < _9707_; ++_1354_i)
      {
        {
          _1353_ok = (_1353_ok && (_ptr__1350_sp[(_1354_i - (int)1)] < _ptr__1350_sp[_1354_i]));
        }
      }
      if ((!_1353_ok))
      {
        {
          throw ::mosek::fusion::IndexError(std::string ("Unordered sparsity pattern"));
        }
      }
      {}
    }
  }
  {}
  return __mosek_2fusion_2Model__parameter_unchecked(_1348_name,_1349_shape,_1350_sp);
}
// End mosek.fusion.Model.parameter_

// Begin mosek.fusion.Model.parameter_
// Method mosek.fusion.Model.parameter_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2908:6-2929:7
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter_(const std::string & _1355_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1356_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1357_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1356_shape(_1356_shape ? _1356_shape->raw() : nullptr, _1356_shape ? _1356_shape->size(0) : 0);
  int32_t _1358_n = (int)((_1357_sparsity)->size(0));
  int32_t _1359_m = (int)((_1357_sparsity)->size(1));
  if (((int)((_1356_shape)->size(0)) != _1359_m))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid sparsity"));
    }
  }
  {}
  if ((NULL == _1357_sparsity.get()))
  {
    {
      return __mosek_2fusion_2Model__parameter_unchecked(_1355_name,_1356_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr));
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1360_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1358_n)));
      _checked_ptr_1<int64_t> _ptr__1360_sp(_1360_sp ? _1360_sp->raw() : nullptr,_1360_sp ? _1360_sp->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1361_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1359_m)));
      _checked_ptr_1<int64_t> _ptr__1361_strides(_1361_strides ? _1361_strides->raw() : nullptr,_1361_strides ? _1361_strides->size(0) : 0);
      _ptr__1361_strides[(_1359_m - (int)1)] = (int)1;
      int32_t _9708_ = (int)1;
      int32_t _9709_ = _1359_m;
      for (int32_t _1362_i = _9708_; _1362_i < _9709_; ++_1362_i)
      {
        {
          _ptr__1361_strides[((_1359_m - (int)1) - _1362_i)] = safe_mul( _ptr__1361_strides[(_1359_m - _1362_i)],_ptr__1356_shape[(_1359_m - _1362_i)] );
        }
      }
      int32_t _9710_ = (int)0;
      int32_t _9711_ = _1358_n;
      for (int32_t _1363_i = _9710_; _1363_i < _9711_; ++_1363_i)
      {
        {
          _ptr__1360_sp[_1363_i] = (int)0;
          int32_t _9712_ = (int)0;
          int32_t _9713_ = _1359_m;
          for (int32_t _1364_j = _9712_; _1364_j < _9713_; ++_1364_j)
          {
            {
              _ptr__1360_sp[_1363_i] += safe_mul( ((*_1357_sparsity)(_1363_i,_1364_j)),_ptr__1361_strides[_1364_j] );
            }
          }
        }
      }
      return __mosek_2fusion_2Model__parameter_(_1355_name,_1356_shape,_1360_sp);
    }
  }
}
// End mosek.fusion.Model.parameter_

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2906:6-139
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1365_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1365_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1365_name) { return __mosek_2fusion_2Model__parameter(_1365_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1365_name)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1366_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1366_sp(_1366_sp ? _1366_sp->raw() : nullptr,_1366_sp ? _1366_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1365_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),_1366_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2905:6-159
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1367_name,int32_t _1368_d1,int32_t _1369_d2,int32_t _1370_d3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1367_name,_1368_d1,_1369_d2,_1370_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1367_name,int32_t _1368_d1,int32_t _1369_d2,int32_t _1370_d3) { return __mosek_2fusion_2Model__parameter(_1367_name,_1368_d1,_1369_d2,_1370_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1367_name,int32_t _1368_d1,int32_t _1369_d2,int32_t _1370_d3)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1371_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1371_sp(_1371_sp ? _1371_sp->raw() : nullptr,_1371_sp ? _1371_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1367_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1368_d1),(int32_t)(_1369_d2),(int32_t)(_1370_d3)}),_1371_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2904:6-146
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1372_name,int32_t _1373_d1,int32_t _1374_d2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1372_name,_1373_d1,_1374_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1372_name,int32_t _1373_d1,int32_t _1374_d2) { return __mosek_2fusion_2Model__parameter(_1372_name,_1373_d1,_1374_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1372_name,int32_t _1373_d1,int32_t _1374_d2)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1375_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1375_sp(_1375_sp ? _1375_sp->raw() : nullptr,_1375_sp ? _1375_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1372_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1373_d1),(int32_t)(_1374_d2)}),_1375_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2903:6-143
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1376_name,int32_t _1377_d1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1376_name,_1377_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1376_name,int32_t _1377_d1) { return __mosek_2fusion_2Model__parameter(_1376_name,_1377_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1376_name,int32_t _1377_d1)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1378_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1378_sp(_1378_sp ? _1378_sp->raw() : nullptr,_1378_sp ? _1378_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1376_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1377_d1)}),_1378_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2902:6-134
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1379_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1380_shape) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1379_name,_1380_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1379_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1380_shape) { return __mosek_2fusion_2Model__parameter(_1379_name,_1380_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1379_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1380_shape)
{
  _checked_ptr_1<int32_t> _ptr__1380_shape(_1380_shape ? _1380_shape->raw() : nullptr, _1380_shape ? _1380_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1381_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1381_sp(_1381_sp ? _1381_sp->raw() : nullptr,_1381_sp ? _1381_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1379_name,_1380_shape,_1381_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2901:6-123
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1382_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1383_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1384_sp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1382_name,_1383_shape,_1384_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1382_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1383_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1384_sp) { return __mosek_2fusion_2Model__parameter(_1382_name,_1383_shape,_1384_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1382_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1383_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1384_sp)
{
  _checked_ptr_1<int32_t> _ptr__1383_shape(_1383_shape ? _1383_shape->raw() : nullptr, _1383_shape ? _1383_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1384_sp(_1384_sp ? _1384_sp->raw() : nullptr, _1384_sp ? _1384_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1382_name,_1383_shape,_1384_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2900:6-136
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(const std::string & _1385_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1386_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1387_sparsity) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1385_name,_1386_shape,_1387_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(const std::string &  _1385_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1386_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1387_sparsity) { return __mosek_2fusion_2Model__parameter(_1385_name,_1386_shape,_1387_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(const std::string & _1385_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1386_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1387_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1386_shape(_1386_shape ? _1386_shape->raw() : nullptr, _1386_shape ? _1386_shape->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(_1385_name,_1386_shape,_1387_sparsity);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2898:6-126
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter() { return __mosek_2fusion_2Model__parameter(); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter()
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1388_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1388_sp(_1388_sp ? _1388_sp->raw() : nullptr,_1388_sp ? _1388_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),_1388_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2897:6-145
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1389_d1,int32_t _1390_d2,int32_t _1391_d3) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1389_d1,_1390_d2,_1391_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1389_d1,int32_t _1390_d2,int32_t _1391_d3) { return __mosek_2fusion_2Model__parameter(_1389_d1,_1390_d2,_1391_d3); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1389_d1,int32_t _1390_d2,int32_t _1391_d3)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1392_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1392_sp(_1392_sp ? _1392_sp->raw() : nullptr,_1392_sp ? _1392_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1389_d1),(int32_t)(_1390_d2),(int32_t)(_1391_d3)}),_1392_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2896:6-132
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1393_d1,int32_t _1394_d2) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1393_d1,_1394_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1393_d1,int32_t _1394_d2) { return __mosek_2fusion_2Model__parameter(_1393_d1,_1394_d2); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1393_d1,int32_t _1394_d2)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1395_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1395_sp(_1395_sp ? _1395_sp->raw() : nullptr,_1395_sp ? _1395_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1393_d1),(int32_t)(_1394_d2)}),_1395_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2895:6-129
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(int32_t _1396_d1) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1396_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(int32_t _1396_d1) { return __mosek_2fusion_2Model__parameter(_1396_d1); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(int32_t _1396_d1)
{
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1397_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1397_sp(_1397_sp ? _1397_sp->raw() : nullptr,_1397_sp ? _1397_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1396_d1)}),_1397_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2894:6-120
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1398_shape) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1398_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1398_shape) { return __mosek_2fusion_2Model__parameter(_1398_shape); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1398_shape)
{
  _checked_ptr_1<int32_t> _ptr__1398_shape(_1398_shape ? _1398_shape->raw() : nullptr, _1398_shape ? _1398_shape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1399_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__1399_sp(_1399_sp ? _1399_sp->raw() : nullptr,_1399_sp ? _1399_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1398_shape,_1399_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2893:6-109
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1400_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1401_sp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1400_shape,_1401_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1400_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1401_sp) { return __mosek_2fusion_2Model__parameter(_1400_shape,_1401_sp); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1400_shape,std::shared_ptr< monty::ndarray< int64_t,1 > > _1401_sp)
{
  _checked_ptr_1<int32_t> _ptr__1400_shape(_1400_shape ? _1400_shape->raw() : nullptr, _1400_shape ? _1400_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__1401_sp(_1401_sp ? _1401_sp->raw() : nullptr, _1401_sp ? _1401_sp->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1400_shape,_1401_sp);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.parameter
// Method mosek.fusion.Model.parameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2892:6-122
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1403_sparsity) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__parameter(_1402_shape,_1403_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1403_sparsity) { return __mosek_2fusion_2Model__parameter(_1402_shape,_1403_sparsity); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__parameter(std::shared_ptr< monty::ndarray< int32_t,1 > > _1402_shape,std::shared_ptr< monty::ndarray< int32_t,2 > > _1403_sparsity)
{
  _checked_ptr_1<int32_t> _ptr__1402_shape(_1402_shape ? _1402_shape->raw() : nullptr, _1402_shape ? _1402_shape->size(0) : 0);
  return __mosek_2fusion_2Model__parameter_(std::string (""),_1402_shape,_1403_sparsity);
}
// End mosek.fusion.Model.parameter

// Begin mosek.fusion.Model.objective_
// Method mosek.fusion.Model.objective_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2795:6-2884:7
void mosek::fusion::p_Model::objective_(const std::string & _1404_name,mosek::fusion::ObjectiveSense _1405_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1406_expr)
{
  task_putobjectivename(_1404_name);
  monty::rc_ptr< ::mosek::fusion::WorkStack > _1407_rs = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  monty::rc_ptr< ::mosek::fusion::WorkStack > _1408_ws = ::mosek::fusion::p_WorkStack::_new_WorkStack();
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1409_obje = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1406_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1409_obje->eval(_1407_rs,_1408_ws,xs);
  _1407_rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1410_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1410_ri32(_1410_ri32 ? _1410_ri32->raw() : nullptr,_1410_ri32 ? _1410_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1411_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1411_ri64(_1411_ri64 ? _1411_ri64->raw() : nullptr,_1411_ri64 ? _1411_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1412_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->f64;
  _checked_ptr_1<double> _ptr__1412_rf64(_1412_rf64 ? _1412_rf64->raw() : nullptr,_1412_rf64 ? _1412_rf64->size(0) : 0);
  int32_t _1413_nd = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->nd;
  int32_t _1414_nelem = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->nelem;
  int32_t _1415_nnz = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->nnz;
  bool _1416_hassp = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->hassp;
  int32_t _1417_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->ncodeatom;
  int32_t _1418_ptr_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->ptr_base;
  int32_t _1419_shape_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->shape_base;
  int32_t _1420_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->nidxs_base;
  int32_t _1421_cof_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->cof_base;
  int32_t _1422_code_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->code_base;
  int32_t _1423_codeptr_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->codeptr_base;
  int32_t _1424_cconst_base = mosek::fusion::p_WorkStack::_get_impl(_1407_rs)->cconst_base;
  int64_t _1425_totalsize = (int64_t)1;
  int32_t _9714_ = (int)0;
  int32_t _9715_ = _1413_nd;
  for (int32_t _1426_i = _9714_; _1426_i < _9715_; ++_1426_i)
  {
    {
      _1413_nd *= _ptr__1410_ri32[safe_add( _1419_shape_base,_1426_i )];
    }
  }
  if ((_1425_totalsize != (int)1))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid expression size for objective"));
    }
  }
  {}
  if ((_1414_nelem != (int)1))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Invalid expression for objective"));
    }
  }
  {}
  int32_t _1427_linnz = (int)0;
  int32_t _1428_psdnz = (int)0;
  int32_t _9716_ = (int)0;
  int32_t _9717_ = _1415_nnz;
  for (int32_t _1429_i = _9716_; _1429_i < _9717_; ++_1429_i)
  {
    {
      if ((_ptr__1411_ri64[safe_add( _1420_nidxs_base,_1429_i )] < (int)0))
      {
        {
          ++ _1428_psdnz;
        }
      }
      else if((_ptr__1411_ri64[safe_add( _1420_nidxs_base,_1429_i )] >= (int)0))
      {
        {
          ++ _1427_linnz;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1430_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1427_linnz)));
  _checked_ptr_1<int32_t> _ptr__1430_subj(_1430_subj ? _1430_subj->raw() : nullptr,_1430_subj ? _1430_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1431_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1427_linnz)));
  _checked_ptr_1<double> _ptr__1431_cof(_1431_cof ? _1431_cof->raw() : nullptr,_1431_cof ? _1431_cof->size(0) : 0);
  {
    int32_t _1432_k = (int)0;
    int32_t _9718_ = (int)0;
    int32_t _9719_ = _1415_nnz;
    for (int32_t _1433_j = _9718_; _1433_j < _9719_; ++_1433_j)
    {
      {
        if ((_ptr__1411_ri64[safe_add( _1420_nidxs_base,_1433_j )] >= (int)0))
        {
          {
            _ptr__1430_subj[_1432_k] = (int32_t)_ptr__1411_ri64[safe_add( _1420_nidxs_base,_1433_j )];
            _ptr__1431_cof[_1432_k] = _ptr__1412_rf64[safe_add( _1421_cof_base,_1433_j )];
            ++ _1432_k;
          }
        }
        {}
      }
    }
  }
  task_putobjective((_1405_sense == ::mosek::fusion::ObjectiveSense::Maximize),_1430_subj,_1431_cof,0.0);
  if ((_1428_psdnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1434_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1428_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1434_subj(_1434_subj ? _1434_subj->raw() : nullptr,_1434_subj ? _1434_subj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1435_subk = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1428_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1435_subk(_1435_subk ? _1435_subk->raw() : nullptr,_1435_subk ? _1435_subk->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1436_subl = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1428_psdnz)));
      _checked_ptr_1<int32_t> _ptr__1436_subl(_1436_subl ? _1436_subl->raw() : nullptr,_1436_subl ? _1436_subl->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1437_val = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1428_psdnz)));
      _checked_ptr_1<double> _ptr__1437_val(_1437_val ? _1437_val->raw() : nullptr,_1437_val ? _1437_val->size(0) : 0);
      int32_t _9720_ = (int)0;
      int32_t _9721_ = _1428_psdnz;
      for (int32_t _1438_i = _9720_; _1438_i < _9721_; ++_1438_i)
      {
        {
          int64_t _1439_varidx = (-safe_add( _ptr__1411_ri64[safe_add( _1420_nidxs_base,_1438_i )],(int)1 ));
          _ptr__1434_subj[_1438_i] = ((*barvar_block_elm_barj)(_1439_varidx));
          _ptr__1435_subk[_1438_i] = ((*barvar_block_elm_i)(_1439_varidx));
          _ptr__1436_subl[_1438_i] = ((*barvar_block_elm_j)(_1439_varidx));
          double _9722_;
          bool _9723_ = (_ptr__1435_subk[_1438_i] == _ptr__1436_subl[_1438_i]);
          if (_9723_)
          {
            _9722_ = _ptr__1412_rf64[safe_add( _1421_cof_base,_1438_i )];
          }
          else
          {
            _9722_ = (0.5 * _ptr__1412_rf64[safe_add( _1421_cof_base,_1438_i )]);
          }
          _ptr__1437_val[_1438_i] = _9722_;
        }
      }
      task_putbarc(_1434_subj,_1435_subk,_1436_subl,_1437_val);
    }
  }
  {}
  if ((_1417_ncodeatom > (int)0))
  {
    {
      obj_blocks->replace_row_code(_1407_rs,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0)}),_1418_ptr_base,_1420_nidxs_base,_1423_codeptr_base,_1422_code_base,_1424_cconst_base);
    }
  }
  {}
}
// End mosek.fusion.Model.objective_

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2792:6-163
void mosek::fusion::Model :: objective(double _1440_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1440_c); }
void mosek::fusion::p_Model::objective(double _1440_c)
{
  objective_(std::string (""),::mosek::fusion::ObjectiveSense::Minimize,::mosek::fusion::p_Expr::constTerm(_1440_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2791:6-145
void mosek::fusion::Model :: objective(mosek::fusion::ObjectiveSense _1441_sense,double _1442_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1441_sense,_1442_c); }
void mosek::fusion::p_Model::objective(mosek::fusion::ObjectiveSense _1441_sense,double _1442_c)
{
  objective_(std::string (""),_1441_sense,::mosek::fusion::p_Expr::constTerm(_1442_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2790:6-132
void mosek::fusion::Model :: objective(mosek::fusion::ObjectiveSense _1443_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1444_expr) { mosek::fusion::p_Model::_get_impl(this)->objective(_1443_sense,_1444_expr); }
void mosek::fusion::p_Model::objective(mosek::fusion::ObjectiveSense _1443_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1444_expr)
{
  objective_(std::string (""),_1443_sense,_1444_expr);
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2789:6-165
void mosek::fusion::Model :: objective(const std::string & _1445_name,double _1446_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1445_name,_1446_c); }
void mosek::fusion::p_Model::objective(const std::string & _1445_name,double _1446_c)
{
  objective_(_1445_name,::mosek::fusion::ObjectiveSense::Minimize,::mosek::fusion::p_Expr::constTerm(_1446_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2788:6-147
void mosek::fusion::Model :: objective(const std::string & _1447_name,mosek::fusion::ObjectiveSense _1448_sense,double _1449_c) { mosek::fusion::p_Model::_get_impl(this)->objective(_1447_name,_1448_sense,_1449_c); }
void mosek::fusion::p_Model::objective(const std::string & _1447_name,mosek::fusion::ObjectiveSense _1448_sense,double _1449_c)
{
  objective_(_1447_name,_1448_sense,::mosek::fusion::p_Expr::constTerm(_1449_c));
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.objective
// Method mosek.fusion.Model.objective @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2787:6-134
void mosek::fusion::Model :: objective(const std::string & _1450_name,mosek::fusion::ObjectiveSense _1451_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1452_expr) { mosek::fusion::p_Model::_get_impl(this)->objective(_1450_name,_1451_sense,_1452_expr); }
void mosek::fusion::p_Model::objective(const std::string & _1450_name,mosek::fusion::ObjectiveSense _1451_sense,monty::rc_ptr< ::mosek::fusion::Expression > _1452_expr)
{
  objective_(_1450_name,_1451_sense,_1452_expr);
}
// End mosek.fusion.Model.objective

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2785:6-159
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1453_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1453_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1453_exprdom) { return __mosek_2fusion_2Model__constraint(_1453_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1453_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprRangeDomain::_get_impl(_1453_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_1453_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2783:6-159
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1454_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1455_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1454_name,_1455_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(const std::string &  _1454_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1455_exprdom) { return __mosek_2fusion_2Model__constraint(_1454_name,_1455_exprdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1454_name,monty::rc_ptr< ::mosek::fusion::ExprRangeDomain > _1455_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1454_name,mosek::fusion::p_ExprRangeDomain::_get_impl(_1455_exprdom)->expr,mosek::fusion::p_ExprRangeDomain::_get_impl(_1455_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2781:6-144
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1456_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1457_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1456_expr,_1457_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1456_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1457_rdom) { return __mosek_2fusion_2Model__constraint(_1456_expr,_1457_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1456_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1457_rdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1456_expr,_1457_rdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2780:6-144
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1458_name,monty::rc_ptr< ::mosek::fusion::Expression > _1459_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1460_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1458_name,_1459_expr,_1460_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::Model::constraint(const std::string &  _1458_name,monty::rc_ptr< ::mosek::fusion::Expression > _1459_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1460_rdom) { return __mosek_2fusion_2Model__constraint(_1458_name,_1459_expr,_1460_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1458_name,monty::rc_ptr< ::mosek::fusion::Expression > _1459_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1460_rdom)
{
  return __mosek_2fusion_2Model__constraint_(_1458_name,_1459_expr,_1460_rdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2778:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1461_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1461_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1461_exprdom) { return __mosek_2fusion_2Model__constraint(_1461_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1461_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprConicDomain::_get_impl(_1461_exprdom)->expr,mosek::fusion::p_ExprConicDomain::_get_impl(_1461_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2776:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1462_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1463_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1462_name,_1463_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1462_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1463_exprdom) { return __mosek_2fusion_2Model__constraint(_1462_name,_1463_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1462_name,monty::rc_ptr< ::mosek::fusion::ExprConicDomain > _1463_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1462_name,mosek::fusion::p_ExprConicDomain::_get_impl(_1463_exprdom)->expr,mosek::fusion::p_ExprConicDomain::_get_impl(_1463_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2774:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1464_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1465_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1464_expr,_1465_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1464_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1465_qdom) { return __mosek_2fusion_2Model__constraint(_1464_expr,_1465_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1464_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1465_qdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1464_expr,_1465_qdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2773:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1466_name,monty::rc_ptr< ::mosek::fusion::Expression > _1467_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1468_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1466_name,_1467_expr,_1468_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1466_name,monty::rc_ptr< ::mosek::fusion::Expression > _1467_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1468_qdom) { return __mosek_2fusion_2Model__constraint(_1466_name,_1467_expr,_1468_qdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1466_name,monty::rc_ptr< ::mosek::fusion::Expression > _1467_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1468_qdom)
{
  return __mosek_2fusion_2Model__constraint_(_1466_name,_1467_expr,_1468_qdom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2771:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1469_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1469_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1469_exprdom) { return __mosek_2fusion_2Model__constraint(_1469_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1469_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprLinearDomain::_get_impl(_1469_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_1469_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2769:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1470_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1471_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1470_name,_1471_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1470_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1471_exprdom) { return __mosek_2fusion_2Model__constraint(_1470_name,_1471_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1470_name,monty::rc_ptr< ::mosek::fusion::ExprLinearDomain > _1471_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1470_name,mosek::fusion::p_ExprLinearDomain::_get_impl(_1471_exprdom)->expr,mosek::fusion::p_ExprLinearDomain::_get_impl(_1471_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2767:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1472_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1473_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1472_expr,_1473_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1472_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1473_ldom) { return __mosek_2fusion_2Model__constraint(_1472_expr,_1473_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1472_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1473_ldom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1472_expr,_1473_ldom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2766:6-139
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1474_name,monty::rc_ptr< ::mosek::fusion::Expression > _1475_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1476_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1474_name,_1475_expr,_1476_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1474_name,monty::rc_ptr< ::mosek::fusion::Expression > _1475_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1476_ldom) { return __mosek_2fusion_2Model__constraint(_1474_name,_1475_expr,_1476_ldom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1474_name,monty::rc_ptr< ::mosek::fusion::Expression > _1475_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1476_ldom)
{
  return __mosek_2fusion_2Model__constraint_(_1474_name,_1475_expr,_1476_ldom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2764:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1477_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1477_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1477_exprdom) { return __mosek_2fusion_2Model__constraint(_1477_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1477_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),mosek::fusion::p_ExprPSDDomain::_get_impl(_1477_exprdom)->expr,mosek::fusion::p_ExprPSDDomain::_get_impl(_1477_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2762:6-154
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1478_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1479_exprdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1478_name,_1479_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1478_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1479_exprdom) { return __mosek_2fusion_2Model__constraint(_1478_name,_1479_exprdom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1478_name,monty::rc_ptr< ::mosek::fusion::ExprPSDDomain > _1479_exprdom)
{
  return __mosek_2fusion_2Model__constraint_(_1478_name,mosek::fusion::p_ExprPSDDomain::_get_impl(_1479_exprdom)->expr,mosek::fusion::p_ExprPSDDomain::_get_impl(_1479_exprdom)->dom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2760:6-141
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1480_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1481_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1480_expr,_1481_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1480_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1481_psddom) { return __mosek_2fusion_2Model__constraint(_1480_expr,_1481_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(monty::rc_ptr< ::mosek::fusion::Expression > _1480_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1481_psddom)
{
  return __mosek_2fusion_2Model__constraint_(std::string (""),_1480_expr,_1481_psddom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.constraint
// Method mosek.fusion.Model.constraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2759:6-141
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__constraint(const std::string & _1482_name,monty::rc_ptr< ::mosek::fusion::Expression > _1483_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1484_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__constraint(_1482_name,_1483_expr,_1484_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::constraint(const std::string &  _1482_name,monty::rc_ptr< ::mosek::fusion::Expression > _1483_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1484_psddom) { return __mosek_2fusion_2Model__constraint(_1482_name,_1483_expr,_1484_psddom); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint(const std::string & _1482_name,monty::rc_ptr< ::mosek::fusion::Expression > _1483_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1484_psddom)
{
  return __mosek_2fusion_2Model__constraint_(_1482_name,_1483_expr,_1484_psddom);
}
// End mosek.fusion.Model.constraint

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2748:6-170
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1485_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom) { return __mosek_2fusion_2Model__variable(_1485_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::PSDDomain > _1485_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),mosek::fusion::p_PSDDomain::_get_impl(_1485_psddom)->shape,_1485_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2747:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1486_n,int32_t _1487_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1486_n,_1487_m,_1488_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1486_n,int32_t _1487_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom) { return __mosek_2fusion_2Model__variable(_1486_n,_1487_m,_1488_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1486_n,int32_t _1487_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1488_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1487_m),(int32_t)(_1486_n),(int32_t)(_1486_n)}),_1488_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2746:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1489_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1490_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1489_n,_1490_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1489_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1490_psddom) { return __mosek_2fusion_2Model__variable(_1489_n,_1490_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1489_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1490_psddom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1489_n),(int32_t)(_1489_n)}),_1490_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2745:6-170
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1491_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1491_name,_1492_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1491_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom) { return __mosek_2fusion_2Model__variable(_1491_name,_1492_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1491_name,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1492_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1491_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1492_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2744:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1493_name,int32_t _1494_n,int32_t _1495_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1493_name,_1494_n,_1495_m,_1496_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1493_name,int32_t _1494_n,int32_t _1495_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom) { return __mosek_2fusion_2Model__variable(_1493_name,_1494_n,_1495_m,_1496_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1493_name,int32_t _1494_n,int32_t _1495_m,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1496_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1493_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1495_m),(int32_t)(_1494_n),(int32_t)(_1494_n)}),_1496_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2743:6-172
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1497_name,int32_t _1498_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1499_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1497_name,_1498_n,_1499_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1497_name,int32_t _1498_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1499_psddom) { return __mosek_2fusion_2Model__variable(_1497_name,_1498_n,_1499_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1497_name,int32_t _1498_n,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1499_psddom)
{
  return __mosek_2fusion_2Model__variable_(_1497_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1498_n),(int32_t)(_1498_n)}),_1499_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2742:6-157
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1500_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1501_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1502_psddom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1500_name,_1501_shp,_1502_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1500_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1501_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1502_psddom) { return __mosek_2fusion_2Model__variable(_1500_name,_1501_shp,_1502_psddom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1500_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1501_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1502_psddom)
{
  _checked_ptr_1<int32_t> _ptr__1501_shp(_1501_shp ? _1501_shp->raw() : nullptr, _1501_shp ? _1501_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1500_name,_1501_shp,_1502_psddom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2736:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1503_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1503_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1503_qdom) { return __mosek_2fusion_2Model__variable(_1503_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::ConeDomain > _1503_qdom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1503_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2735:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1504_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1504_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1504_rdom) { return __mosek_2fusion_2Model__variable(_1504_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::RangeDomain > _1504_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1504_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2734:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1505_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1505_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1505_ldom) { return __mosek_2fusion_2Model__variable(_1505_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(monty::rc_ptr< ::mosek::fusion::LinearDomain > _1505_ldom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1505_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2733:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1506_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1506_shp,_1507_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1506_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom) { return __mosek_2fusion_2Model__variable(_1506_shp,_1507_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1506_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1507_qdom)
{
  _checked_ptr_1<int32_t> _ptr__1506_shp(_1506_shp ? _1506_shp->raw() : nullptr, _1506_shp ? _1506_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1506_shp,_1507_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2732:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1508_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1509_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1508_shp,_1509_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1508_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1509_rdom) { return __mosek_2fusion_2Model__variable(_1508_shp,_1509_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1508_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1509_rdom)
{
  _checked_ptr_1<int32_t> _ptr__1508_shp(_1508_shp ? _1508_shp->raw() : nullptr, _1508_shp ? _1508_shp->size(0) : 0);
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),_1508_shp,_1509_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2731:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1511_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1510_shp,_1511_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1511_ldom) { return __mosek_2fusion_2Model__variable(_1510_shp,_1511_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1510_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1511_ldom)
{
  _checked_ptr_1<int32_t> _ptr__1510_shp(_1510_shp ? _1510_shp->raw() : nullptr, _1510_shp ? _1510_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1510_shp,_1511_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2730:6-162
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1512_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp) { return __mosek_2fusion_2Model__variable(_1512_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(std::shared_ptr< monty::ndarray< int32_t,1 > > _1512_shp)
{
  _checked_ptr_1<int32_t> _ptr__1512_shp(_1512_shp ? _1512_shp->raw() : nullptr, _1512_shp ? _1512_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(std::string (""),_1512_shp,::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2729:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1513_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1514_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1513_size,_1514_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1513_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1514_qdom) { return __mosek_2fusion_2Model__variable(_1513_size,_1514_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1513_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1514_qdom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1513_size)}),_1514_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2728:6-161
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1515_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1516_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1515_size,_1516_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(int32_t _1515_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1516_rdom) { return __mosek_2fusion_2Model__variable(_1515_size,_1516_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1515_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1516_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1515_size)}),_1516_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2727:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1518_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1517_size,_1518_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1518_ldom) { return __mosek_2fusion_2Model__variable(_1517_size,_1518_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1517_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1518_ldom)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1517_size)}),_1518_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2726:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(int32_t _1519_size) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1519_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(int32_t _1519_size) { return __mosek_2fusion_2Model__variable(_1519_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(int32_t _1519_size)
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1519_size)}),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2725:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable() { return __mosek_2fusion_2Model__variable(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable()
{
  return __mosek_2fusion_2Model__variable_(std::string (""),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2723:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1520_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1521_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1520_name,_1521_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1520_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1521_qdom) { return __mosek_2fusion_2Model__variable(_1520_name,_1521_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1520_name,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1521_qdom)
{
  return __mosek_2fusion_2Model__variable_(_1520_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1521_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2722:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1522_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1523_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1522_name,_1523_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1522_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1523_rdom) { return __mosek_2fusion_2Model__variable(_1522_name,_1523_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1522_name,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1523_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(_1522_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1523_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2721:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1524_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1525_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1524_name,_1525_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1524_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1525_ldom) { return __mosek_2fusion_2Model__variable(_1524_name,_1525_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1524_name,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1525_ldom)
{
  return __mosek_2fusion_2Model__variable_(_1524_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(nullptr),_1525_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2720:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1526_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1527_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1528_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1526_name,_1527_shp,_1528_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1526_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1527_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1528_qdom) { return __mosek_2fusion_2Model__variable(_1526_name,_1527_shp,_1528_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1526_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1527_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1528_qdom)
{
  _checked_ptr_1<int32_t> _ptr__1527_shp(_1527_shp ? _1527_shp->raw() : nullptr, _1527_shp ? _1527_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1526_name,_1527_shp,_1528_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2719:6-159
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1529_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1530_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1531_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1529_name,_1530_shp,_1531_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1529_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1530_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1531_rdom) { return __mosek_2fusion_2Model__variable(_1529_name,_1530_shp,_1531_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1529_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1530_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1531_rdom)
{
  _checked_ptr_1<int32_t> _ptr__1530_shp(_1530_shp ? _1530_shp->raw() : nullptr, _1530_shp ? _1530_shp->size(0) : 0);
  return __mosek_2fusion_2Model__ranged_variable(_1529_name,_1530_shp,_1531_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2718:6-159
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1532_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1533_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1534_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1532_name,_1533_shp,_1534_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1532_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1533_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1534_ldom) { return __mosek_2fusion_2Model__variable(_1532_name,_1533_shp,_1534_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1532_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1533_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1534_ldom)
{
  _checked_ptr_1<int32_t> _ptr__1533_shp(_1533_shp ? _1533_shp->raw() : nullptr, _1533_shp ? _1533_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1532_name,_1533_shp,_1534_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2717:6-162
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1535_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1536_shp) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1535_name,_1536_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1535_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1536_shp) { return __mosek_2fusion_2Model__variable(_1535_name,_1536_shp); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1535_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1536_shp)
{
  _checked_ptr_1<int32_t> _ptr__1536_shp(_1536_shp ? _1536_shp->raw() : nullptr, _1536_shp ? _1536_shp->size(0) : 0);
  return __mosek_2fusion_2Model__variable_(_1535_name,_1536_shp,::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2716:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1537_name,int32_t _1538_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1539_qdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1537_name,_1538_size,_1539_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1537_name,int32_t _1538_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1539_qdom) { return __mosek_2fusion_2Model__variable(_1537_name,_1538_size,_1539_qdom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1537_name,int32_t _1538_size,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1539_qdom)
{
  return __mosek_2fusion_2Model__variable_(_1537_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1538_size)}),_1539_qdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2715:6-161
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1540_name,int32_t _1541_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1542_rdom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1540_name,_1541_size,_1542_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::Model::variable(const std::string &  _1540_name,int32_t _1541_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1542_rdom) { return __mosek_2fusion_2Model__variable(_1540_name,_1541_size,_1542_rdom); }
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1540_name,int32_t _1541_size,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1542_rdom)
{
  return __mosek_2fusion_2Model__ranged_variable(_1540_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1541_size)}),_1542_rdom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2714:6-161
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1543_name,int32_t _1544_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1545_ldom) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1543_name,_1544_size,_1545_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1543_name,int32_t _1544_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1545_ldom) { return __mosek_2fusion_2Model__variable(_1543_name,_1544_size,_1545_ldom); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1543_name,int32_t _1544_size,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1545_ldom)
{
  return __mosek_2fusion_2Model__variable_(_1543_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1544_size)}),_1545_ldom);
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2713:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1546_name,int32_t _1547_size) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1546_name,_1547_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1546_name,int32_t _1547_size) { return __mosek_2fusion_2Model__variable(_1546_name,_1547_size); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1546_name,int32_t _1547_size)
{
  return __mosek_2fusion_2Model__variable_(_1546_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_1547_size)}),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.variable
// Method mosek.fusion.Model.variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2712:6-175
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__variable(const std::string & _1548_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__variable(_1548_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::variable(const std::string &  _1548_name) { return __mosek_2fusion_2Model__variable(_1548_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable(const std::string & _1548_name)
{
  return __mosek_2fusion_2Model__variable_(_1548_name,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))),::mosek::fusion::p_Domain::unbounded());
}
// End mosek.fusion.Model.variable

// Begin mosek.fusion.Model.removeConstraintBlock
// Method mosek.fusion.Model.removeConstraintBlock @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2703:6-2710:7
void mosek::fusion::p_Model::removeConstraintBlock(int32_t _1549_conid)
{
  if ((_1549_conid >= (int)0))
  {
    {
      con_release(_1549_conid);
    }
  }
  else
  {
    {
      acon_release((-safe_add( (int)1,_1549_conid )));
    }
  }
}
// End mosek.fusion.Model.removeConstraintBlock

// Begin mosek.fusion.Model.removeVariableBlock
// Method mosek.fusion.Model.removeVariableBlock @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2675:6-2700:7
void mosek::fusion::p_Model::removeVariableBlock(int64_t _1550_varid64)
{
  if ((con_blocks->is_parameterized() || afe_blocks->is_parameterized()))
  {
    {
      throw ::mosek::fusion::DeletionError(std::string ("Cannot remove variables in parameterized model"));
    }
  }
  {}
  if ((_1550_varid64 < (int)0))
  {
    {
      throw ::mosek::fusion::DeletionError(std::string ("PSD variables cannot be removed"));
    }
  }
  else
  {
    {
      int32_t _1551_varid = (int32_t)_1550_varid64;
      int32_t _1552_sz = var_block_map->blocksize(_1551_varid);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1553_nidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1552_sz)));
      _checked_ptr_1<int32_t> _ptr__1553_nidxs(_1553_nidxs ? _1553_nidxs->raw() : nullptr,_1553_nidxs ? _1553_nidxs->size(0) : 0);
      var_block_map->get(_1551_varid,_1553_nidxs,(int)0);
      var_block_map->free(_1551_varid);
      int32_t _1554_conid = ((*var_block_acc_id)(_1551_varid));
      if ((_1554_conid >= (int)0))
      {
        {
          removeConstraintBlock(_1554_conid);
        }
      }
      {}
      task_var_putboundlist_fr(_1553_nidxs);
      task_clearacollist(_1553_nidxs);
    }
  }
}
// End mosek.fusion.Model.removeVariableBlock

// Begin mosek.fusion.Model.ranged_variable
// Method mosek.fusion.Model.ranged_variable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2446:6-2672:7
monty::rc_ptr< ::mosek::fusion::RangedVariable > mosek::fusion::p_Model::__mosek_2fusion_2Model__ranged_variable(const std::string & _1555_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1556_shp,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1557_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1556_shp(_1556_shp ? _1556_shp->raw() : nullptr, _1556_shp ? _1556_shp->size(0) : 0);
  if ((((int)((_1555_name).size()) > (int)0) && hasVariable(_1555_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1555_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::RangeDomain > _1558_dom = mosek::fusion::p_RangeDomain::_get_impl(_1557_dom_.get())->__mosek_2fusion_2RangeDomain__finalize_and_validate(_1556_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1559_shape = mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1559_shape(_1559_shape ? _1559_shape->raw() : nullptr,_1559_shape ? _1559_shape->size(0) : 0);
  int32_t _1560_nd = (int)((_1559_shape)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1561_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1560_nd)));
  _checked_ptr_1<int32_t> _ptr__1561_strides(_1561_strides ? _1561_strides->raw() : nullptr,_1561_strides ? _1561_strides->size(0) : 0);
  if ((_1560_nd > (int)0))
  {
    {
      _ptr__1561_strides[(_1560_nd - (int)1)] = (int)1;
      int32_t _9724_ = (int)1;
      int32_t _9725_ = _1560_nd;
      for (int32_t _1562_i = _9724_; _1562_i < _9725_; ++_1562_i)
      {
        {
          _ptr__1561_strides[((_1560_nd - _1562_i) - (int)1)] = safe_mul( _ptr__1561_strides[(_1560_nd - _1562_i)],_ptr__1559_shape[(_1560_nd - _1562_i)] );
        }
      }
    }
  }
  {}
  int32_t _1563_domsize = (int)1;
  int32_t _9726_ = (int)0;
  int32_t _9727_ = (int)((_1559_shape)->size(0));
  for (int32_t _1564_i = _9726_; _1564_i < _9727_; ++_1564_i)
  {
    {
      _1563_domsize *= _ptr__1559_shape[_1564_i];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1565_axisptr = nullptr;
  _checked_ptr_1<int32_t> _ptr__1565_axisptr(_1565_axisptr ? _1565_axisptr->raw() : nullptr,_1565_axisptr ? _1565_axisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1566_nameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1566_nameptr(_1566_nameptr ? _1566_nameptr->raw() : nullptr,_1566_nameptr ? _1566_nameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1567_axisnames = nullptr;
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1568_accaxisptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1568_accaxisptr(_1568_accaxisptr ? _1568_accaxisptr->raw() : nullptr,_1568_accaxisptr ? _1568_accaxisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1569_accnameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1569_accnameptr(_1569_accnameptr ? _1569_accnameptr->raw() : nullptr,_1569_accnameptr ? _1569_accnameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1570_accaxisnames = nullptr;
  if ((((int)((_1555_name).size()) > (int)0) && (NULL != mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames.get())))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1571_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1572_accsb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      int32_t _1573_axisp = (int)0;
      int64_t _1574_namep = (int)0;
      int64_t _1575_numnames = (int)0;
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)->size(0)) != _1560_nd))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
        }
      }
      {}
      int32_t _9728_ = (int)0;
      int32_t _9729_ = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)->size(0));
      for (int32_t _1576_i = _9728_; _1576_i < _9729_; ++_1576_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)(_1576_i)).get()))
          {
            {
              if ((_ptr__1559_shape[_1576_i] != (int)((((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)(_1576_i)))->size(0))))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
                }
              }
              {}
              _1575_numnames += _ptr__1559_shape[_1576_i];
            }
          }
          {}
        }
      }
      _1566_nameptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1575_numnames,(int)1 ))));
      _ptr__1566_nameptr.update(_1566_nameptr ? _1566_nameptr->raw() : nullptr, _1566_nameptr ? _1566_nameptr->size(0) : 0);
      std::vector<int32_t> _9730_;
      for (int32_t _1577_i = (int)0, _9731_ = _1560_nd; _1577_i < _9731_ ; ++_1577_i)
      {
        _9730_.push_back((-(int)1));
      }
      auto _9732_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9730_.size()),_9730_.begin(),_9730_.end()));
      _1565_axisptr = _9732_;
      _ptr__1565_axisptr.update(_1565_axisptr ? _1565_axisptr->raw() : nullptr, _1565_axisptr ? _1565_axisptr->size(0) : 0);
      int32_t _9733_ = (int)0;
      int32_t _9734_ = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)->size(0));
      for (int32_t _1578_i = _9733_; _1578_i < _9734_; ++_1578_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)(_1578_i)).get()))
          {
            {
              _ptr__1565_axisptr[_1578_i] = _1573_axisp;
              int32_t _9735_ = (int)0;
              int32_t _9736_ = _ptr__1559_shape[_1578_i];
              for (int32_t _1579_j = _9735_; _1579_j < _9736_; ++_1579_j)
              {
                {
                  _1571_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)(_1578_i)))(_1579_j)));
                  ++ _1573_axisp;
                  _1574_namep += (int)((((*((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames)(_1578_i)))(_1579_j))).size());
                  _ptr__1566_nameptr[_1573_axisp] = _1574_namep;
                }
              }
            }
          }
          {}
        }
      }
      _1567_axisnames = _1571_sb;
      _1570_accaxisnames = _1572_accsb;
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1580_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__1580_sparsity(_1580_sparsity ? _1580_sparsity->raw() : nullptr,_1580_sparsity ? _1580_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1581_perm = nullptr;
  _checked_ptr_1<int32_t> _ptr__1581_perm(_1581_perm ? _1581_perm->raw() : nullptr,_1581_perm ? _1581_perm->size(0) : 0);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->empty)
  {
    {
      _1580_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
      _ptr__1580_sparsity.update(_1580_sparsity ? _1580_sparsity->raw() : nullptr, _1580_sparsity ? _1580_sparsity->size(0) : 0);
    }
  }
  else if((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->sparsity.get()))
  {
    {
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->sparsity)->size(1)) != _1560_nd))
      {
        {
          throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is not correct"));
        }
      }
      {}
      int32_t _1582_n = (int)((mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->sparsity)->size(0));
      _1580_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1582_n)));
      _ptr__1580_sparsity.update(_1580_sparsity ? _1580_sparsity->raw() : nullptr, _1580_sparsity ? _1580_sparsity->size(0) : 0);
      {
        bool _1583_err = false;
        std::shared_ptr< monty::ndarray< int32_t,2 > > _1584_sp = mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->sparsity;
        int32_t _9737_ = (int)0;
        int32_t _9738_ = _1582_n;
        for (int32_t _1585_i = _9737_; _1585_i < _9738_; ++_1585_i)
        {
          {
            int32_t _9739_ = (int)0;
            int32_t _9740_ = (int)((_1584_sp)->size(1));
            for (int32_t _1586_j = _9739_; _1586_j < _9740_; ++_1586_j)
            {
              {
                _1583_err = (_1583_err || ((((*_1584_sp)(_1585_i,_1586_j)) < (int)0) || (((*_1584_sp)(_1585_i,_1586_j)) >= _ptr__1559_shape[_1586_j])));
              }
            }
            if ((!_1583_err))
            {
              {
                int64_t _1587_linidx = (int)0;
                int32_t _9741_ = (int)0;
                int32_t _9742_ = (int)((_1584_sp)->size(1));
                for (int32_t _1588_j = _9741_; _1588_j < _9742_; ++_1588_j)
                {
                  {
                    _1587_linidx += safe_mul( _ptr__1561_strides[_1588_j],((*_1584_sp)(_1585_i,_1588_j)) );
                  }
                }
                _ptr__1580_sparsity[_1585_i] = _1587_linidx;
              }
            }
            {}
          }
        }
        if (_1583_err)
        {
          {
            throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is out of bounds"));
          }
        }
        {}
      }
      std::shared_ptr< monty::ndarray< int32_t,2 > > _1589_midxs = mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->sparsity;
      int32_t _1590_maxdim = (int)0;
      int32_t _9743_ = (int)0;
      int32_t _9744_ = _1560_nd;
      for (int32_t _1591_i = _9743_; _1591_i < _9744_; ++_1591_i)
      {
        {
          if ((_1590_maxdim < _ptr__1559_shape[_1591_i]))
          {
            {
              _1590_maxdim = _ptr__1559_shape[_1591_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1592_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1582_n)));
      _checked_ptr_1<int32_t> _ptr__1592_curperm(_1592_curperm ? _1592_curperm->raw() : nullptr,_1592_curperm ? _1592_curperm->size(0) : 0);
      int32_t _9745_ = (int)0;
      int32_t _9746_ = _1582_n;
      for (int32_t _1593_i = _9745_; _1593_i < _9746_; ++_1593_i)
      {
        {
          _ptr__1592_curperm[_1593_i] = _1593_i;
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1594_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1582_n)));
      _checked_ptr_1<int32_t> _ptr__1594_tgtperm(_1594_tgtperm ? _1594_tgtperm->raw() : nullptr,_1594_tgtperm ? _1594_tgtperm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1595_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1590_maxdim,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1595_ptr(_1595_ptr ? _1595_ptr->raw() : nullptr,_1595_ptr ? _1595_ptr->size(0) : 0);
      int32_t _9747_ = (int)0;
      int32_t _9748_ = _1560_nd;
      for (int32_t _1596_dd = _9747_; _1596_dd < _9748_; ++_1596_dd)
      {
        {
          {
            int32_t _1597_d = ((_1560_nd - _1596_dd) - (int)1);
            int32_t _9749_ = (int)0;
            int32_t _9750_ = safe_add( _ptr__1559_shape[_1597_d],(int)1 );
            for (int32_t _1598_i = _9749_; _1598_i < _9750_; ++_1598_i)
            {
              {
                _ptr__1595_ptr[_1598_i] = (int)0;
              }
            }
            int32_t _9751_ = (int)0;
            int32_t _9752_ = _1582_n;
            for (int32_t _1599_i = _9751_; _1599_i < _9752_; ++_1599_i)
            {
              {
                ++ _ptr__1595_ptr[safe_add( ((*_1589_midxs)(_1599_i,_1597_d)),(int)1 )];
              }
            }
            int32_t _9753_ = (int)0;
            int32_t _9754_ = _ptr__1559_shape[_1597_d];
            for (int32_t _1600_i = _9753_; _1600_i < _9754_; ++_1600_i)
            {
              {
                _ptr__1595_ptr[safe_add( _1600_i,(int)1 )] += _ptr__1595_ptr[_1600_i];
              }
            }
            int32_t _9755_ = (int)0;
            int32_t _9756_ = _1582_n;
            for (int32_t _1601_ii = _9755_; _1601_ii < _9756_; ++_1601_ii)
            {
              {
                int32_t _1602_i = _ptr__1592_curperm[_1601_ii];
                _ptr__1594_tgtperm[_ptr__1595_ptr[((*_1589_midxs)(_1602_i,_1597_d))]] = _1602_i;
                ++ _ptr__1595_ptr[((*_1589_midxs)(_1602_i,_1597_d))];
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1603_tmp = _1592_curperm;
          _checked_ptr_1<int32_t> _ptr__1603_tmp(_1603_tmp ? _1603_tmp->raw() : nullptr,_1603_tmp ? _1603_tmp->size(0) : 0);
          _1592_curperm = _1594_tgtperm;
          _ptr__1592_curperm.update(_1592_curperm ? _1592_curperm->raw() : nullptr, _1592_curperm ? _1592_curperm->size(0) : 0);
          _1594_tgtperm = _1603_tmp;
          _ptr__1594_tgtperm.update(_1594_tgtperm ? _1594_tgtperm->raw() : nullptr, _1594_tgtperm ? _1594_tgtperm->size(0) : 0);
        }
      }
      int32_t _9757_ = (int)0;
      int32_t _9758_ = _1582_n;
      for (int32_t _1604_ii = _9757_; _1604_ii < _9758_; ++_1604_ii)
      {
        {
          int32_t _1605_i = _ptr__1592_curperm[_1604_ii];
          int64_t _1606_k = (int)0;
          int32_t _9759_ = (int)0;
          int32_t _9760_ = _1560_nd;
          for (int32_t _1607_j = _9759_; _1607_j < _9760_; ++_1607_j)
          {
            {
              _1606_k += safe_mul( _ptr__1561_strides[_1607_j],((*_1589_midxs)(_1605_i,_1607_j)) );
            }
          }
          _ptr__1580_sparsity[_1604_ii] = _1606_k;
        }
      }
    }
  }
  {}
  int32_t _9761_;
  bool _9762_ = (NULL != _1580_sparsity.get());
  if (_9762_)
  {
    _9761_ = (int)((_1580_sparsity)->size(0));
  }
  else
  {
    _9761_ = _1563_domsize;
  }
  int32_t _1608_nvaritems = _9761_;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1609_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1608_nvaritems)));
  _checked_ptr_1<int32_t> _ptr__1609_nativeidxs(_1609_nativeidxs ? _1609_nativeidxs->raw() : nullptr,_1609_nativeidxs ? _1609_nativeidxs->size(0) : 0);
  int32_t _1610_varblockid = linearvar_alloc(_1608_nvaritems,_1609_nativeidxs);
  if (((int)((_1555_name).size()) > (int)0))
  {
    {
      var_map->setItem(_1555_name,_1610_varblockid);
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1611_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1611_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1555_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1560_nd > (int)0))
      {
        {
          _1611_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9763_ = (int)1;
          int32_t _9764_ = _1560_nd;
          for (int32_t _1612_i = _9763_; _1612_i < _9764_; ++_1612_i)
          {
            {
              _1611_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1612_i);
            }
          }
        }
      }
      {}
      _1611_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames.get()))
      {
        {
          task_format_var_names(_1609_nativeidxs,_1611_sb->toString(),_1559_shape,_1580_sparsity,mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_var_names(_1609_nativeidxs,_1611_sb->toString(),_1559_shape,_1580_sparsity,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  std::shared_ptr< monty::ndarray< double,1 > > _1613_bl;
  _checked_ptr_1<double> _ptr__1613_bl(_1613_bl ? _1613_bl->raw() : nullptr,_1613_bl ? _1613_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1614_bu;
  _checked_ptr_1<double> _ptr__1614_bu(_1614_bu ? _1614_bu->raw() : nullptr,_1614_bu ? _1614_bu->size(0) : 0);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->scalable)
  {
    {
      std::vector<double> _9765_;
      for (int32_t _1615_i = (int)0, _9766_ = _1608_nvaritems; _1615_i < _9766_ ; ++_1615_i)
      {
        _9765_.push_back(((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->lb)((int)0)));
      }
      auto _9767_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9765_.size()),_9765_.begin(),_9765_.end()));
      _1613_bl = _9767_;
      _ptr__1613_bl.update(_1613_bl ? _1613_bl->raw() : nullptr, _1613_bl ? _1613_bl->size(0) : 0);
      std::vector<double> _9768_;
      for (int32_t _1616_i = (int)0, _9769_ = _1608_nvaritems; _1616_i < _9769_ ; ++_1616_i)
      {
        _9768_.push_back(((*mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->ub)((int)0)));
      }
      auto _9770_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_9768_.size()),_9768_.begin(),_9768_.end()));
      _1614_bu = _9770_;
      _ptr__1614_bu.update(_1614_bu ? _1614_bu->raw() : nullptr, _1614_bu ? _1614_bu->size(0) : 0);
    }
  }
  else
  {
    {
      _1613_bl = mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->lb;
      _ptr__1613_bl.update(_1613_bl ? _1613_bl->raw() : nullptr, _1613_bl ? _1613_bl->size(0) : 0);
      _1614_bu = mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->ub;
      _ptr__1614_bu.update(_1614_bu ? _1614_bu->raw() : nullptr, _1614_bu ? _1614_bu->size(0) : 0);
    }
  }
  task_var_putboundlist_ra(_1609_nativeidxs,_1613_bl,_1614_bu);
  monty::rc_ptr< ::mosek::fusion::RangedVariable > _1617_res = ::mosek::fusion::p_RangedVariable::_new_RangedVariable(::mosek::fusion::Model::t(_pubthis),_1555_name,(int64_t)(_1610_varblockid),_1559_shape,_1580_sparsity,_1609_nativeidxs);
  if (mosek::fusion::p_RangeDomain::_get_impl(_1558_dom)->cardinal_flag)
  {
    {
      task_var_putintlist(_1609_nativeidxs);
    }
  }
  {}
  ((*vars)(_1610_varblockid)) = (_1617_res).as<::mosek::fusion::ModelVariable>();
  return _1617_res;
}
// End mosek.fusion.Model.ranged_variable

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:2190:6-2440:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1618_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1619_shp,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1620_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1619_shp(_1619_shp ? _1619_shp->raw() : nullptr, _1619_shp ? _1619_shp->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::ConeDomain > _1621_dom = mosek::fusion::p_ConeDomain::_get_impl(_1620_dom_.get())->__mosek_2fusion_2ConeDomain__finalize_and_validate(_1619_shp);
  if ((((int)((_1618_name).size()) > (int)0) && hasVariable(_1618_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1618_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1622_shape = mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1622_shape(_1622_shape ? _1622_shape->raw() : nullptr,_1622_shape ? _1622_shape->size(0) : 0);
  int32_t _1623_nd = (int)((_1622_shape)->size(0));
  int32_t _1624_coneaxis = _1621_dom->getAxis();
  int32_t _1625_conesize = _ptr__1622_shape[_1624_coneaxis];
  int32_t _1626_d1 = (int)1;
  int32_t _1627_d0 = (int)1;
  int32_t _1628_domsize = (int)1;
  int32_t _9771_ = (int)0;
  int32_t _9772_ = _1623_nd;
  for (int32_t _1629_i = _9771_; _1629_i < _9772_; ++_1629_i)
  {
    {
      _1628_domsize *= _ptr__1622_shape[_1629_i];
    }
  }
  int32_t _9773_ = (int)0;
  int32_t _9774_ = _1624_coneaxis;
  for (int32_t _1630_i = _9773_; _1630_i < _9774_; ++_1630_i)
  {
    {
      _1627_d0 *= _ptr__1622_shape[_1630_i];
    }
  }
  int32_t _9775_ = safe_add( _1624_coneaxis,(int)1 );
  int32_t _9776_ = _1623_nd;
  for (int32_t _1631_i = _9775_; _1631_i < _9776_; ++_1631_i)
  {
    {
      _1626_d1 *= _ptr__1622_shape[_1631_i];
    }
  }
  int32_t _1632_numcone = safe_mul( _1627_d0,_1626_d1 );
  int64_t _1633_domainidx;
  if ((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InQCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for quadratic cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_quad(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InRotatedQCone))
  {
    {
      if ((_1625_conesize < (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for rotated quadratic cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_rquad(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InPExpCone))
  {
    {
      if ((_1625_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for exponential cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_pexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InPPowCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_ppow(_1625_conesize,mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InDExpCone))
  {
    {
      if ((_1625_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual exponential cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_dexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InDPowCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual power cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_dpow(_1625_conesize,mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InPGeoMeanCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for primal geometric mean cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_pgeomean(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InDGeoMeanCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual geometric mean cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_dgeomean(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::Positive))
  {
    {
      _1633_domainidx = task_append_domain_rpos(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InSVecPSDCone))
  {
    {
      int32_t _1634_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_1625_conesize ) ))) - (int)1) / (int)2);
      if (((safe_mul( _1634_d,safe_add( _1634_d,(int)1 ) ) / (int)2) != _1625_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for vector positive semidefinite cone"))->toString());
        }
      }
      {}
      _1633_domainidx = task_append_domain_svec_psd(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::Negative))
  {
    {
      _1633_domainidx = task_append_domain_rneg(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::Unbounded))
  {
    {
      _1633_domainidx = task_append_domain_r(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::Zero))
  {
    {
      _1633_domainidx = task_append_domain_rzero(_1625_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->key == ::mosek::fusion::QConeKey::InPSDCone))
  {
    {
      if ((_1625_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
        }
      }
      else
      {
        {
          int32_t _1635_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_1625_conesize ) ))) - (int)1) / (int)2);
          if (((safe_mul( _1635_d,safe_add( _1635_d,(int)1 ) ) / (int)2) != _1625_conesize))
          {
            {
              throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_1625_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
            }
          }
          {}
        }
      }
      _1633_domainidx = task_append_domain_svec_psd(_1625_conesize);
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal error: Invalid cone type"));
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1636_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( _1632_numcone,_1625_conesize ))));
  _checked_ptr_1<int32_t> _ptr__1636_nativeidxs(_1636_nativeidxs ? _1636_nativeidxs->raw() : nullptr,_1636_nativeidxs ? _1636_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1637_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1632_numcone)));
  _checked_ptr_1<int32_t> _ptr__1637_accidxs(_1637_accidxs ? _1637_accidxs->raw() : nullptr,_1637_accidxs ? _1637_accidxs->size(0) : 0);
  int32_t _1638_varid = conicvar_alloc(_1633_domainidx,_1625_conesize,_1632_numcone,_1637_accidxs,_1636_nativeidxs);
  task_var_putboundlist_fr(_1636_nativeidxs);
  if (mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->int_flag)
  {
    {
      task_var_putintlist(_1636_nativeidxs);
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1639_permidxs = _1636_nativeidxs;
  _checked_ptr_1<int32_t> _ptr__1639_permidxs(_1639_permidxs ? _1639_permidxs->raw() : nullptr,_1639_permidxs ? _1639_permidxs->size(0) : 0);
  if ((_1624_coneaxis < ((int)((_1622_shape)->size(0)) - (int)1)))
  {
    {
      _1639_permidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)((_1636_nativeidxs)->size(0)))));
      _ptr__1639_permidxs.update(_1639_permidxs ? _1639_permidxs->raw() : nullptr, _1639_permidxs ? _1639_permidxs->size(0) : 0);
      int32_t _1640_p = (int)0;
      int32_t _9777_ = (int)0;
      int32_t _9778_ = _1627_d0;
      for (int32_t _1641_i0 = _9777_; _1641_i0 < _9778_; ++_1641_i0)
      {
        {
          int32_t _9779_ = (int)0;
          int32_t _9780_ = _1626_d1;
          for (int32_t _1642_i1 = _9779_; _1642_i1 < _9780_; ++_1642_i1)
          {
            {
              int32_t _9781_ = (int)0;
              int32_t _9782_ = _1625_conesize;
              for (int32_t _1643_k = _9781_; _1643_k < _9782_; ++_1643_k)
              {
                {
                  _ptr__1639_permidxs[_1640_p] = _ptr__1636_nativeidxs[safe_add( safe_mul( safe_add( safe_mul( _1641_i0,_1625_conesize ),_1643_k ),_1626_d1 ),_1642_i1 )];
                  ++ _1640_p;
                }
              }
            }
          }
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicVariable > _1644_res = ::mosek::fusion::p_ConicVariable::_new_ConicVariable(::mosek::fusion::Model::t(_pubthis),_1618_name,_1638_varid,_1622_shape,_1636_nativeidxs);
  ((*vars)(_1638_varid)) = (_1644_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1618_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1645_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      {
        _1645_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1618_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
        if ((_1623_nd > (int)0))
        {
          {
            _1645_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
            int32_t _9783_ = (int)1;
            int32_t _9784_ = _1623_nd;
            for (int32_t _1646_i = _9783_; _1646_i < _9784_; ++_1646_i)
            {
              {
                _1645_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1646_i);
              }
            }
          }
        }
        {}
        _1645_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
        if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->indexnames.get()))
        {
          {
            task_format_var_names(_1639_permidxs,_1645_sb->toString(),_1622_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->indexnames);
          }
        }
        else
        {
          {
            task_format_var_names(_1639_permidxs,_1645_sb->toString(),_1622_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
          }
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1647_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1632_numcone)));
      _checked_ptr_1<int32_t> _ptr__1647_accidxs(_1647_accidxs ? _1647_accidxs->raw() : nullptr,_1647_accidxs ? _1647_accidxs->size(0) : 0);
      acc_block_map->get(((*var_block_acc_id)(_1638_varid)),_1647_accidxs,(int)0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _1648_accshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_1622_shape)->size(0)) - (int)1))));
        _checked_ptr_1<int32_t> _ptr__1648_accshape(_1648_accshape ? _1648_accshape->raw() : nullptr,_1648_accshape ? _1648_accshape->size(0) : 0);
        int32_t _9785_ = (int)0;
        int32_t _9786_ = _1624_coneaxis;
        for (int32_t _1649_i = _9785_; _1649_i < _9786_; ++_1649_i)
        {
          {
            _ptr__1648_accshape[_1649_i] = _ptr__1622_shape[_1649_i];
          }
        }
        int32_t _9787_ = safe_add( _1624_coneaxis,(int)1 );
        int32_t _9788_ = _1623_nd;
        for (int32_t _1650_i = _9787_; _1650_i < _9788_; ++_1650_i)
        {
          {
            _ptr__1648_accshape[(_1650_i - (int)1)] = _ptr__1622_shape[_1650_i];
          }
        }
        monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1651_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
        _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1618_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
        if (((_1623_nd - (int)1) > (int)0))
        {
          {
            _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
            int32_t _9789_ = (int)1;
            int32_t _9790_ = _1623_nd;
            for (int32_t _1652_i = _9789_; _1652_i < _9790_; ++_1652_i)
            {
              {
                _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1652_i);
              }
            }
          }
        }
        {}
        _1651_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
        if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->indexnames.get()))
        {
          {
            std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1653_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1623_nd - (int)1))));
            int32_t _9791_ = (int)0;
            int32_t _9792_ = _1624_coneaxis;
            for (int32_t _1654_i = _9791_; _1654_i < _9792_; ++_1654_i)
            {
              {
                ((*_1653_indexnames)(_1654_i)) = ((*mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->indexnames)(_1654_i));
              }
            }
            int32_t _9793_ = safe_add( _1624_coneaxis,(int)1 );
            int32_t _9794_ = _1623_nd;
            for (int32_t _1655_i = _9793_; _1655_i < _9794_; ++_1655_i)
            {
              {
                ((*_1653_indexnames)((_1655_i - (int)1))) = ((*mosek::fusion::p_ConeDomain::_get_impl(_1621_dom)->indexnames)(_1655_i));
              }
            }
            std::vector<int64_t> _9795_;
            for (int32_t _1656_i = (int)0, _9796_ = _1632_numcone; _1656_i < _9796_ ; ++_1656_i)
            {
              _9795_.push_back((int64_t)_ptr__1647_accidxs[_1656_i]);
            }
            auto _9797_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9795_.size()),_9795_.begin(),_9795_.end()));
            task_format_acc_names(_9797_,_1651_sb->toString(),_1648_accshape,_1653_indexnames);
          }
        }
        else
        {
          {
            std::vector<int64_t> _9798_;
            for (int32_t _1657_i = (int)0, _9799_ = _1632_numcone; _1657_i < _9799_ ; ++_1657_i)
            {
              _9798_.push_back((int64_t)_ptr__1647_accidxs[_1657_i]);
            }
            auto _9800_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9798_.size()),_9798_.begin(),_9798_.end()));
            task_format_acc_names(_9800_,_1651_sb->toString(),_1648_accshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
          }
        }
      }
      var_map->setItem(_1618_name,_1638_varid);
    }
  }
  {}
  return (_1644_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1999:6-2185:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1658_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1659_shp,monty::rc_ptr< ::mosek::fusion::LinearDomain > _1660_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1659_shp(_1659_shp ? _1659_shp->raw() : nullptr, _1659_shp ? _1659_shp->size(0) : 0);
  if ((((int)((_1658_name).size()) > (int)0) && hasVariable(_1658_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1658_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearDomain > _1661_dom = mosek::fusion::p_LinearDomain::_get_impl(_1660_dom_.get())->__mosek_2fusion_2LinearDomain__finalize_and_validate(_1659_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1662_shape = mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1662_shape(_1662_shape ? _1662_shape->raw() : nullptr,_1662_shape ? _1662_shape->size(0) : 0);
  int32_t _1663_nd = (int)((_1662_shape)->size(0));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1664_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1663_nd)));
  _checked_ptr_1<int32_t> _ptr__1664_strides(_1664_strides ? _1664_strides->raw() : nullptr,_1664_strides ? _1664_strides->size(0) : 0);
  if ((_1663_nd > (int)0))
  {
    {
      _ptr__1664_strides[(_1663_nd - (int)1)] = (int)1;
      int32_t _9801_ = (int)1;
      int32_t _9802_ = _1663_nd;
      for (int32_t _1665_i = _9801_; _1665_i < _9802_; ++_1665_i)
      {
        {
          _ptr__1664_strides[((_1663_nd - _1665_i) - (int)1)] = safe_mul( _ptr__1664_strides[(_1663_nd - _1665_i)],_ptr__1662_shape[(_1663_nd - _1665_i)] );
        }
      }
    }
  }
  {}
  int32_t _1666_domsize = (int)1;
  int32_t _9803_ = (int)0;
  int32_t _9804_ = (int)((_1662_shape)->size(0));
  for (int32_t _1667_i = _9803_; _1667_i < _9804_; ++_1667_i)
  {
    {
      _1666_domsize *= _ptr__1662_shape[_1667_i];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1668_axisptr = nullptr;
  _checked_ptr_1<int32_t> _ptr__1668_axisptr(_1668_axisptr ? _1668_axisptr->raw() : nullptr,_1668_axisptr ? _1668_axisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1669_nameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1669_nameptr(_1669_nameptr ? _1669_nameptr->raw() : nullptr,_1669_nameptr ? _1669_nameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1670_axisnames = nullptr;
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1671_accaxisptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1671_accaxisptr(_1671_accaxisptr ? _1671_accaxisptr->raw() : nullptr,_1671_accaxisptr ? _1671_accaxisptr->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1672_accnameptr = nullptr;
  _checked_ptr_1<int64_t> _ptr__1672_accnameptr(_1672_accnameptr ? _1672_accnameptr->raw() : nullptr,_1672_accnameptr ? _1672_accnameptr->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1673_accaxisnames = nullptr;
  if ((((int)((_1658_name).size()) > (int)0) && (NULL != mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames.get())))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1674_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1675_accsb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      int32_t _1676_axisp = (int)0;
      int64_t _1677_namep = (int)0;
      int64_t _1678_numnames = (int)0;
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)->size(0)) != _1663_nd))
      {
        {
          throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
        }
      }
      {}
      int32_t _9805_ = (int)0;
      int32_t _9806_ = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)->size(0));
      for (int32_t _1679_i = _9805_; _1679_i < _9806_; ++_1679_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)(_1679_i)).get()))
          {
            {
              if ((_ptr__1662_shape[_1679_i] != (int)((((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)(_1679_i)))->size(0))))
              {
                {
                  throw ::mosek::fusion::DomainError(std::string ("Invalid index name definition"));
                }
              }
              {}
              _1678_numnames += _ptr__1662_shape[_1679_i];
            }
          }
          {}
        }
      }
      _1669_nameptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1678_numnames,(int)1 ))));
      _ptr__1669_nameptr.update(_1669_nameptr ? _1669_nameptr->raw() : nullptr, _1669_nameptr ? _1669_nameptr->size(0) : 0);
      std::vector<int32_t> _9807_;
      for (int32_t _1680_i = (int)0, _9808_ = _1663_nd; _1680_i < _9808_ ; ++_1680_i)
      {
        _9807_.push_back((-(int)1));
      }
      auto _9809_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9807_.size()),_9807_.begin(),_9807_.end()));
      _1668_axisptr = _9809_;
      _ptr__1668_axisptr.update(_1668_axisptr ? _1668_axisptr->raw() : nullptr, _1668_axisptr ? _1668_axisptr->size(0) : 0);
      int32_t _9810_ = (int)0;
      int32_t _9811_ = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)->size(0));
      for (int32_t _1681_i = _9810_; _1681_i < _9811_; ++_1681_i)
      {
        {
          if ((NULL != ((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)(_1681_i)).get()))
          {
            {
              _ptr__1668_axisptr[_1681_i] = _1676_axisp;
              int32_t _9812_ = (int)0;
              int32_t _9813_ = _ptr__1662_shape[_1681_i];
              for (int32_t _1682_j = _9812_; _1682_j < _9813_; ++_1682_j)
              {
                {
                  _1674_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)(_1681_i)))(_1682_j)));
                  ++ _1676_axisp;
                  _1677_namep += (int)((((*((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames)(_1681_i)))(_1682_j))).size());
                  _ptr__1669_nameptr[_1676_axisp] = _1677_namep;
                }
              }
            }
          }
          {}
        }
      }
      _1670_axisnames = _1674_sb;
      _1673_accaxisnames = _1675_accsb;
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1683_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__1683_sparsity(_1683_sparsity ? _1683_sparsity->raw() : nullptr,_1683_sparsity ? _1683_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1684_perm = nullptr;
  _checked_ptr_1<int32_t> _ptr__1684_perm(_1684_perm ? _1684_perm->raw() : nullptr,_1684_perm ? _1684_perm->size(0) : 0);
  if (mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->empty)
  {
    {
      _1683_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0)));
      _ptr__1683_sparsity.update(_1683_sparsity ? _1683_sparsity->raw() : nullptr, _1683_sparsity ? _1683_sparsity->size(0) : 0);
    }
  }
  else if((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->sparsity.get()))
  {
    {
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->sparsity)->size(1)) != _1663_nd))
      {
        {
          throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is not correct"));
        }
      }
      {}
      int32_t _1685_n = (int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->sparsity)->size(0));
      _1683_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1685_n)));
      _ptr__1683_sparsity.update(_1683_sparsity ? _1683_sparsity->raw() : nullptr, _1683_sparsity ? _1683_sparsity->size(0) : 0);
      {
        bool _1686_err = false;
        std::shared_ptr< monty::ndarray< int32_t,2 > > _1687_sp = mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->sparsity;
        int32_t _9814_ = (int)0;
        int32_t _9815_ = _1685_n;
        for (int32_t _1688_i = _9814_; _1688_i < _9815_; ++_1688_i)
        {
          {
            int32_t _9816_ = (int)0;
            int32_t _9817_ = (int)((_1687_sp)->size(1));
            for (int32_t _1689_j = _9816_; _1689_j < _9817_; ++_1689_j)
            {
              {
                _1686_err = (_1686_err || ((((*_1687_sp)(_1688_i,_1689_j)) < (int)0) || (((*_1687_sp)(_1688_i,_1689_j)) >= _ptr__1662_shape[_1689_j])));
              }
            }
            if ((!_1686_err))
            {
              {
                int64_t _1690_linidx = (int)0;
                int32_t _9818_ = (int)0;
                int32_t _9819_ = (int)((_1687_sp)->size(1));
                for (int32_t _1691_j = _9818_; _1691_j < _9819_; ++_1691_j)
                {
                  {
                    _1690_linidx += safe_mul( _ptr__1664_strides[_1691_j],((*_1687_sp)(_1688_i,_1691_j)) );
                  }
                }
                _ptr__1683_sparsity[_1688_i] = _1690_linidx;
              }
            }
            {}
          }
        }
        if (_1686_err)
        {
          {
            throw ::mosek::fusion::SparseFormatError(std::string ("Sparsity pattern is out of bounds"));
          }
        }
        {}
      }
      std::shared_ptr< monty::ndarray< int32_t,2 > > _1692_midxs = mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->sparsity;
      int32_t _1693_maxdim = (int)0;
      int32_t _9820_ = (int)0;
      int32_t _9821_ = _1663_nd;
      for (int32_t _1694_i = _9820_; _1694_i < _9821_; ++_1694_i)
      {
        {
          if ((_1693_maxdim < _ptr__1662_shape[_1694_i]))
          {
            {
              _1693_maxdim = _ptr__1662_shape[_1694_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1695_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1685_n)));
      _checked_ptr_1<int32_t> _ptr__1695_curperm(_1695_curperm ? _1695_curperm->raw() : nullptr,_1695_curperm ? _1695_curperm->size(0) : 0);
      int32_t _9822_ = (int)0;
      int32_t _9823_ = _1685_n;
      for (int32_t _1696_i = _9822_; _1696_i < _9823_; ++_1696_i)
      {
        {
          _ptr__1695_curperm[_1696_i] = _1696_i;
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1697_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1685_n)));
      _checked_ptr_1<int32_t> _ptr__1697_tgtperm(_1697_tgtperm ? _1697_tgtperm->raw() : nullptr,_1697_tgtperm ? _1697_tgtperm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1698_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1693_maxdim,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1698_ptr(_1698_ptr ? _1698_ptr->raw() : nullptr,_1698_ptr ? _1698_ptr->size(0) : 0);
      int32_t _9824_ = (int)0;
      int32_t _9825_ = _1663_nd;
      for (int32_t _1699_dd = _9824_; _1699_dd < _9825_; ++_1699_dd)
      {
        {
          {
            int32_t _1700_d = ((_1663_nd - _1699_dd) - (int)1);
            int32_t _9826_ = (int)0;
            int32_t _9827_ = safe_add( _ptr__1662_shape[_1700_d],(int)1 );
            for (int32_t _1701_i = _9826_; _1701_i < _9827_; ++_1701_i)
            {
              {
                _ptr__1698_ptr[_1701_i] = (int)0;
              }
            }
            int32_t _9828_ = (int)0;
            int32_t _9829_ = _1685_n;
            for (int32_t _1702_i = _9828_; _1702_i < _9829_; ++_1702_i)
            {
              {
                ++ _ptr__1698_ptr[safe_add( ((*_1692_midxs)(_1702_i,_1700_d)),(int)1 )];
              }
            }
            int32_t _9830_ = (int)0;
            int32_t _9831_ = _ptr__1662_shape[_1700_d];
            for (int32_t _1703_i = _9830_; _1703_i < _9831_; ++_1703_i)
            {
              {
                _ptr__1698_ptr[safe_add( _1703_i,(int)1 )] += _ptr__1698_ptr[_1703_i];
              }
            }
            int32_t _9832_ = (int)0;
            int32_t _9833_ = _1685_n;
            for (int32_t _1704_ii = _9832_; _1704_ii < _9833_; ++_1704_ii)
            {
              {
                int32_t _1705_i = _ptr__1695_curperm[_1704_ii];
                _ptr__1697_tgtperm[_ptr__1698_ptr[((*_1692_midxs)(_1705_i,_1700_d))]] = _1705_i;
                ++ _ptr__1698_ptr[((*_1692_midxs)(_1705_i,_1700_d))];
              }
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _1706_tmp = _1695_curperm;
          _checked_ptr_1<int32_t> _ptr__1706_tmp(_1706_tmp ? _1706_tmp->raw() : nullptr,_1706_tmp ? _1706_tmp->size(0) : 0);
          _1695_curperm = _1697_tgtperm;
          _ptr__1695_curperm.update(_1695_curperm ? _1695_curperm->raw() : nullptr, _1695_curperm ? _1695_curperm->size(0) : 0);
          _1697_tgtperm = _1706_tmp;
          _ptr__1697_tgtperm.update(_1697_tgtperm ? _1697_tgtperm->raw() : nullptr, _1697_tgtperm ? _1697_tgtperm->size(0) : 0);
        }
      }
      int32_t _9834_ = (int)0;
      int32_t _9835_ = _1685_n;
      for (int32_t _1707_ii = _9834_; _1707_ii < _9835_; ++_1707_ii)
      {
        {
          int32_t _1708_i = _ptr__1695_curperm[_1707_ii];
          int64_t _1709_k = (int)0;
          int32_t _9836_ = (int)0;
          int32_t _9837_ = _1663_nd;
          for (int32_t _1710_j = _9836_; _1710_j < _9837_; ++_1710_j)
          {
            {
              _1709_k += safe_mul( _ptr__1664_strides[_1710_j],((*_1692_midxs)(_1708_i,_1710_j)) );
            }
          }
          _ptr__1683_sparsity[_1707_ii] = _1709_k;
        }
      }
    }
  }
  {}
  int32_t _9838_;
  bool _9839_ = (NULL != _1683_sparsity.get());
  if (_9839_)
  {
    _9838_ = (int)((_1683_sparsity)->size(0));
  }
  else
  {
    _9838_ = _1666_domsize;
  }
  int32_t _1711_nvaritems = _9838_;
  std::shared_ptr< monty::ndarray< double,1 > > _1712_bnds = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1711_nvaritems)));
  _checked_ptr_1<double> _ptr__1712_bnds(_1712_bnds ? _1712_bnds->raw() : nullptr,_1712_bnds ? _1712_bnds->size(0) : 0);
  if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->bnd.get()))
  {
    {
      if (((int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->bnd)->size(0)) == (int)1))
      {
        {
          int32_t _9840_ = (int)0;
          int32_t _9841_ = _1711_nvaritems;
          for (int32_t _1713_i = _9840_; _1713_i < _9841_; ++_1713_i)
          {
            {
              _ptr__1712_bnds[_1713_i] = ((*mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->bnd)((int)0));
            }
          }
        }
      }
      else if(((int)((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->bnd)->size(0)) == _1711_nvaritems))
      {
        {
          ::mosek::fusion::Utils::Tools::arraycopy(mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->bnd,(int)0,_1712_bnds,(int)0,_1711_nvaritems);
        }
      }
      else
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid right hand side shape in domain"));
        }
      }
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1714_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1711_nvaritems)));
  _checked_ptr_1<int32_t> _ptr__1714_nativeidxs(_1714_nativeidxs ? _1714_nativeidxs->raw() : nullptr,_1714_nativeidxs ? _1714_nativeidxs->size(0) : 0);
  int32_t _1715_varblockid = linearvar_alloc(_1711_nvaritems,_1714_nativeidxs);
  if ((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      task_var_putboundlist_fr(_1714_nativeidxs);
    }
  }
  {}
  if ((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      task_var_putboundlist_up(_1714_nativeidxs,_1712_bnds);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      task_var_putboundlist_lo(_1714_nativeidxs,_1712_bnds);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      task_var_putboundlist_fx(_1714_nativeidxs,_1712_bnds);
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearVariable > _1716_res = ::mosek::fusion::p_LinearVariable::_new_LinearVariable(::mosek::fusion::Model::t(_pubthis),_1658_name,(int64_t)(_1715_varblockid),_1662_shape,_1683_sparsity,_1714_nativeidxs);
  ((*vars)(_1715_varblockid)) = (_1716_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1658_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1717_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1717_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1658_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1663_nd > (int)0))
      {
        {
          _1717_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9842_ = (int)1;
          int32_t _9843_ = _1663_nd;
          for (int32_t _1718_i = _9842_; _1718_i < _9843_; ++_1718_i)
          {
            {
              _1717_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1718_i);
            }
          }
        }
      }
      {}
      _1717_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames.get()))
      {
        {
          task_format_var_names(_1714_nativeidxs,_1717_sb->toString(),_1662_shape,_1683_sparsity,mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_var_names(_1714_nativeidxs,_1717_sb->toString(),_1662_shape,_1683_sparsity,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
      var_map->setItem(_1658_name,_1715_varblockid);
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_1661_dom)->cardinal_flag)
  {
    {
      task_var_putintlist(_1714_nativeidxs);
    }
  }
  {}
  return (_1716_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.variable_
// Method mosek.fusion.Model.variable_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1815:6-1937:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__variable_(const std::string & _1719_name,std::shared_ptr< monty::ndarray< int32_t,1 > > _1720_shp,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1721_dom_)
{
  _checked_ptr_1<int32_t> _ptr__1720_shp(_1720_shp ? _1720_shp->raw() : nullptr, _1720_shp ? _1720_shp->size(0) : 0);
  if ((((int)((_1719_name).size()) > (int)0) && hasVariable(_1719_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate variable name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1719_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::PSDDomain > _1722_dom = mosek::fusion::p_PSDDomain::_get_impl(_1721_dom_.get())->__mosek_2fusion_2PSDDomain__finalize_and_validate(_1720_shp);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1723_shape = mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->shape;
  _checked_ptr_1<int32_t> _ptr__1723_shape(_1723_shape ? _1723_shape->raw() : nullptr,_1723_shape ? _1723_shape->size(0) : 0);
  int32_t _1724_conedim1 = mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->conedim1;
  int32_t _1725_conedim2 = mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->conedim2;
  int32_t _1726_nd = (int)((_1723_shape)->size(0));
  int32_t _1727_d = _ptr__1723_shape[_1724_conedim1];
  if ((_1724_conedim1 > _1725_conedim2))
  {
    {
      int32_t _1728_tmp = _1724_conedim1;
      _1724_conedim1 = _1725_conedim2;
      _1725_conedim2 = _1728_tmp;
    }
  }
  {}
  if (((_1724_conedim1 == _1725_conedim2) || ((_1724_conedim1 < (int)0) || (_1725_conedim2 >= _1726_nd))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid cone order dimensions for semidefinite variables"));
    }
  }
  {}
  if ((_ptr__1723_shape[_1724_conedim1] != _ptr__1723_shape[_1725_conedim2]))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching shape in cone order dimensions"));
    }
  }
  {}
  int32_t _1729_numcone = (int)1;
  int32_t _9844_ = (int)0;
  int32_t _9845_ = _1724_conedim1;
  for (int32_t _1730_i = _9844_; _1730_i < _9845_; ++_1730_i)
  {
    {
      _1729_numcone *= _ptr__1723_shape[_1730_i];
    }
  }
  int32_t _9846_ = safe_add( _1724_conedim1,(int)1 );
  int32_t _9847_ = _1725_conedim2;
  for (int32_t _1731_i = _9846_; _1731_i < _9847_; ++_1731_i)
  {
    {
      _1729_numcone *= _ptr__1723_shape[_1731_i];
    }
  }
  int32_t _9848_ = safe_add( _1725_conedim2,(int)1 );
  int32_t _9849_ = _1726_nd;
  for (int32_t _1732_i = _9848_; _1732_i < _9849_; ++_1732_i)
  {
    {
      _1729_numcone *= _ptr__1723_shape[_1732_i];
    }
  }
  int32_t _1733_numelm = (safe_mul( safe_mul( _1729_numcone,_1727_d ),safe_add( _1727_d,(int)1 ) ) / (int)2);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1734_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1733_numelm)));
  _checked_ptr_1<int64_t> _ptr__1734_nativeidxs(_1734_nativeidxs ? _1734_nativeidxs->raw() : nullptr,_1734_nativeidxs ? _1734_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1735_barvaridxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1729_numcone)));
  _checked_ptr_1<int32_t> _ptr__1735_barvaridxs(_1735_barvaridxs ? _1735_barvaridxs->raw() : nullptr,_1735_barvaridxs ? _1735_barvaridxs->size(0) : 0);
  int32_t _1736_varid = barvar_alloc(_1727_d,_1729_numcone,_1735_barvaridxs,_1734_nativeidxs);
  monty::rc_ptr< ::mosek::fusion::PSDVariable > _1737_res = ::mosek::fusion::p_PSDVariable::_new_PSDVariable(::mosek::fusion::Model::t(_pubthis),_1719_name,_1736_varid,_1723_shape,mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->conedim1,mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->conedim2,_1735_barvaridxs,_1734_nativeidxs);
  if (((int)((_1719_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1738_coneshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((_1726_nd - (int)2))));
      _checked_ptr_1<int32_t> _ptr__1738_coneshape(_1738_coneshape ? _1738_coneshape->raw() : nullptr,_1738_coneshape ? _1738_coneshape->size(0) : 0);
      int32_t _9850_;
      bool _9851_ = (_1724_conedim1 < _1725_conedim2);
      if (_9851_)
      {
        _9850_ = _1724_conedim1;
      }
      else
      {
        _9850_ = _1725_conedim2;
      }
      int32_t _1739_cdim1 = _9850_;
      int32_t _9852_;
      bool _9853_ = (_1724_conedim1 < _1725_conedim2);
      if (_9853_)
      {
        _9852_ = _1725_conedim2;
      }
      else
      {
        _9852_ = _1724_conedim1;
      }
      int32_t _1740_cdim2 = _9852_;
      int32_t _9854_ = (int)0;
      int32_t _9855_ = _1739_cdim1;
      for (int32_t _1741_i = _9854_; _1741_i < _9855_; ++_1741_i)
      {
        {
          _ptr__1738_coneshape[_1741_i] = _ptr__1723_shape[_1741_i];
        }
      }
      int32_t _9856_ = safe_add( _1739_cdim1,(int)1 );
      int32_t _9857_ = _1740_cdim2;
      for (int32_t _1742_i = _9856_; _1742_i < _9857_; ++_1742_i)
      {
        {
          _ptr__1738_coneshape[(_1742_i - (int)1)] = _ptr__1723_shape[_1742_i];
        }
      }
      int32_t _9858_ = safe_add( _1740_cdim2,(int)1 );
      int32_t _9859_ = _1726_nd;
      for (int32_t _1743_i = _9858_; _1743_i < _9859_; ++_1743_i)
      {
        {
          _ptr__1738_coneshape[(_1743_i - (int)2)] = _ptr__1723_shape[_1743_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1744_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1744_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1719_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((_1726_nd - (int)2) > (int)0))
      {
        {
          _1744_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9860_ = (int)1;
          int32_t _9861_ = (_1726_nd - (int)2);
          for (int32_t _1745_i = _9860_; _1745_i < _9861_; ++_1745_i)
          {
            {
              _1744_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1745_i);
            }
          }
        }
      }
      {}
      _1744_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1746_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1726_nd - (int)2))));
          int32_t _9862_ = (int)0;
          int32_t _9863_ = _1739_cdim1;
          for (int32_t _1747_i = _9862_; _1747_i < _9863_; ++_1747_i)
          {
            {
              ((*_1746_indexnames)(_1747_i)) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->indexnames)(_1747_i));
            }
          }
          int32_t _9864_ = safe_add( _1739_cdim1,(int)1 );
          int32_t _9865_ = _1740_cdim2;
          for (int32_t _1748_i = _9864_; _1748_i < _9865_; ++_1748_i)
          {
            {
              ((*_1746_indexnames)((_1748_i - (int)1))) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->indexnames)(_1748_i));
            }
          }
          int32_t _9866_ = safe_add( _1740_cdim2,(int)1 );
          int32_t _9867_ = _1726_nd;
          for (int32_t _1749_i = _9866_; _1749_i < _9867_; ++_1749_i)
          {
            {
              ((*_1746_indexnames)((_1749_i - (int)2))) = ((*mosek::fusion::p_PSDDomain::_get_impl(_1722_dom)->indexnames)(_1749_i));
            }
          }
          task_format_barvar_names(_1735_barvaridxs,_1744_sb->toString(),_1738_coneshape,_1746_indexnames);
        }
      }
      else
      {
        {
          task_format_barvar_names(_1735_barvaridxs,_1744_sb->toString(),_1738_coneshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  ((*barvars)(_1736_varid)) = (_1737_res).as<::mosek::fusion::ModelVariable>();
  if (((int)((_1719_name).size()) > (int)0))
  {
    {
      var_map->setItem(_1719_name,(-safe_add( _1736_varid,(int)1 )));
    }
  }
  {}
  return (_1737_res).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.variable_

// Begin mosek.fusion.Model.putfrows
// Method mosek.fusion.Model.putfrows @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1684:6-1808:7
void mosek::fusion::p_Model::putfrows(std::shared_ptr< monty::ndarray< int32_t,1 > > _1750_nativeidxs,int32_t _1751_nativebaseptr,monty::rc_ptr< ::mosek::fusion::WorkStack > _1752_rs,int32_t _1753_nelem,int32_t _1754_nnz,int32_t _1755_ptr_base,int32_t _1756_nidxs_base,int32_t _1757_cof_base)
{
  _checked_ptr_1<int32_t> _ptr__1750_nativeidxs(_1750_nativeidxs ? _1750_nativeidxs->raw() : nullptr, _1750_nativeidxs ? _1750_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1758_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1752_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1758_ri32(_1758_ri32 ? _1758_ri32->raw() : nullptr,_1758_ri32 ? _1758_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1759_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1752_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1759_ri64(_1759_ri64 ? _1759_ri64->raw() : nullptr,_1759_ri64 ? _1759_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1760_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1752_rs)->f64;
  _checked_ptr_1<double> _ptr__1760_rf64(_1760_rf64 ? _1760_rf64->raw() : nullptr,_1760_rf64 ? _1760_rf64->size(0) : 0);
  int32_t _1761_linnz = (int)0;
  int32_t _1762_barnnz = (int)0;
  int32_t _1763_totalpsdelm = (int)0;
  int32_t _9868_ = (int)0;
  int32_t _9869_ = _1753_nelem;
  for (int32_t _1764_i = _9868_; _1764_i < _9869_; ++_1764_i)
  {
    {
      int32_t _1765_ps = _ptr__1758_ri32[safe_add( _1755_ptr_base,_1764_i )];
      int32_t _1766_pe = _ptr__1758_ri32[safe_add( safe_add( _1755_ptr_base,_1764_i ),(int)1 )];
      int32_t _1767_p = _1765_ps;
      while ( ((_1767_p < _1766_pe) && (_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1767_p )] < (int)0)) )
      {
        {
          int32_t _1768_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1767_p )],(int)1 ))));
          int32_t _1769_p0 = _1767_p;
          ++ _1767_p;
          while ( ((_1767_p < _1766_pe) && ((_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1767_p )] < (int)0) && (_1768_barj == ((*barvar_block_elm_barj)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1767_p )],(int)1 ))))))) )
          {
            {
              ++ _1767_p;
            }
          }
          ++ _1762_barnnz;
        }
      }
      _1763_totalpsdelm += (_1767_p - _1765_ps);
      while ( ((_1767_p < _1766_pe) && (_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1767_p )] == (int)0)) )
      {
        {
          ++ _1767_p;
        }
      }
      _1761_linnz += (_1766_pe - _1767_p);
    }
  }
  if ((_1753_nelem > ((int)((_1750_nativeidxs)->size(0)) - _1751_nativebaseptr)))
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal: Mismatching sizes"));
    }
  }
  {}
  int32_t _1770_numcon = _1753_nelem;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1771_ptrb = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1770_numcon,(int)1 ))));
  _checked_ptr_1<int32_t> _ptr__1771_ptrb(_1771_ptrb ? _1771_ptrb->raw() : nullptr,_1771_ptrb ? _1771_ptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1772_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1761_linnz)));
  _checked_ptr_1<int32_t> _ptr__1772_subj(_1772_subj ? _1772_subj->raw() : nullptr,_1772_subj ? _1772_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1773_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1761_linnz)));
  _checked_ptr_1<double> _ptr__1773_cof(_1773_cof ? _1773_cof->raw() : nullptr,_1773_cof ? _1773_cof->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1774_bfix = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1753_nelem)));
  _checked_ptr_1<double> _ptr__1774_bfix(_1774_bfix ? _1774_bfix->raw() : nullptr,_1774_bfix ? _1774_bfix->size(0) : 0);
  {
    int32_t _1775_k = (int)0;
    int32_t _9870_ = (int)0;
    int32_t _9871_ = _1753_nelem;
    for (int32_t _1776_i = _9870_; _1776_i < _9871_; ++_1776_i)
    {
      {
        int32_t _1777_knz = (int)0;
        int32_t _9872_ = _ptr__1758_ri32[safe_add( _1755_ptr_base,_1776_i )];
        int32_t _9873_ = _ptr__1758_ri32[safe_add( safe_add( _1755_ptr_base,_1776_i ),(int)1 )];
        for (int32_t _1778_j = _9872_; _1778_j < _9873_; ++_1778_j)
        {
          {
            if ((_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1778_j )] == (int)0))
            {
              {
                _ptr__1774_bfix[_1776_i] += _ptr__1760_rf64[safe_add( _1757_cof_base,_1778_j )];
              }
            }
            else if((_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1778_j )] > (int)0))
            {
              {
                _ptr__1772_subj[safe_add( _1775_k,_1777_knz )] = (int32_t)_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1778_j )];
                _ptr__1773_cof[safe_add( _1775_k,_1777_knz )] = _ptr__1760_rf64[safe_add( _1757_cof_base,_1778_j )];
                ++ _1777_knz;
              }
            }
            {}
          }
        }
        _ptr__1771_ptrb[safe_add( _1776_i,(int)1 )] = safe_add( _ptr__1771_ptrb[_1776_i],_1777_knz );
        _1775_k += _1777_knz;
      }
    }
  }
  std::vector<int64_t> _9874_;
  for (int32_t _1779_i = (int)0, _9875_ = _1753_nelem; _1779_i < _9875_ ; ++_1779_i)
  {
    _9874_.push_back(_ptr__1750_nativeidxs[safe_add( _1779_i,_1751_nativebaseptr )]);
  }
  auto _9876_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9874_.size()),_9874_.begin(),_9874_.end()));
  task_putafeflist(_9876_,_1771_ptrb,_1772_subj,_1773_cof,_1774_bfix);
  if ((_1762_barnnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1780_subi = _1750_nativeidxs;
      _checked_ptr_1<int32_t> _ptr__1780_subi(_1780_subi ? _1780_subi->raw() : nullptr,_1780_subi ? _1780_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1781_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _1770_numcon,(int)1 ))));
      _checked_ptr_1<int32_t> _ptr__1781_ptr(_1781_ptr ? _1781_ptr->raw() : nullptr,_1781_ptr ? _1781_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1782_barsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1762_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1782_barsubj(_1782_barsubj ? _1782_barsubj->raw() : nullptr,_1782_barsubj ? _1782_barsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1783_matdim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1762_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1783_matdim(_1783_matdim ? _1783_matdim->raw() : nullptr,_1783_matdim ? _1783_matdim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1784_matnnz = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1762_barnnz)));
      _checked_ptr_1<int64_t> _ptr__1784_matnnz(_1784_matnnz ? _1784_matnnz->raw() : nullptr,_1784_matnnz ? _1784_matnnz->size(0) : 0);
      std::vector<int32_t> _9877_;
      for (int32_t _1786_i = (int)0, _9878_ = _1763_totalpsdelm; _1786_i < _9878_ ; ++_1786_i)
      {
        _9877_.push_back((-(int)1));
      }
      auto _9879_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9877_.size()),_9877_.begin(),_9877_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1785_matsubi = _9879_;
      _checked_ptr_1<int32_t> _ptr__1785_matsubi(_1785_matsubi ? _1785_matsubi->raw() : nullptr,_1785_matsubi ? _1785_matsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1787_matsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1763_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1787_matsubj(_1787_matsubj ? _1787_matsubj->raw() : nullptr,_1787_matsubj ? _1787_matsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1788_matcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1763_totalpsdelm)));
      _checked_ptr_1<double> _ptr__1788_matcof(_1788_matcof ? _1788_matcof->raw() : nullptr,_1788_matcof ? _1788_matcof->size(0) : 0);
      int32_t _1789_barnzi = (int)0;
      int32_t _1790_barmatnzi = (int)0;
      int32_t _9880_ = (int)0;
      int32_t _9881_ = _1770_numcon;
      for (int32_t _1791_i = _9880_; _1791_i < _9881_; ++_1791_i)
      {
        {
          int32_t _1792_p = _ptr__1758_ri32[safe_add( _1755_ptr_base,_1791_i )];
          int32_t _1793_pe = _ptr__1758_ri32[safe_add( safe_add( _1755_ptr_base,_1791_i ),(int)1 )];
          while ( ((_1792_p < _1793_pe) && (_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )] < (int)0)) )
          {
            {
              int32_t _1794_p0 = _1792_p;
              int32_t _1795_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )],(int)1 ))));
              while ( ((_1792_p < _1793_pe) && ((_ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )] < (int)0) && (((*barvar_block_elm_barj)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )],(int)1 )))) == _1795_barj))) )
              {
                {
                  _ptr__1785_matsubi[_1790_barmatnzi] = ((*barvar_block_elm_i)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )],(int)1 ))));
                  _ptr__1787_matsubj[_1790_barmatnzi] = ((*barvar_block_elm_j)((-safe_add( _ptr__1759_ri64[safe_add( _1756_nidxs_base,_1792_p )],(int)1 ))));
                  if ((_ptr__1785_matsubi[_1790_barmatnzi] == _ptr__1787_matsubj[_1790_barmatnzi]))
                  {
                    {
                      _ptr__1788_matcof[_1790_barmatnzi] = _ptr__1760_rf64[safe_add( _1757_cof_base,_1792_p )];
                    }
                  }
                  else
                  {
                    {
                      _ptr__1788_matcof[_1790_barmatnzi] = (0.5 * _ptr__1760_rf64[safe_add( _1757_cof_base,_1792_p )]);
                    }
                  }
                  ++ _1790_barmatnzi;
                  ++ _1792_p;
                }
              }
              _ptr__1782_barsubj[_1789_barnzi] = _1795_barj;
              _ptr__1783_matdim[_1789_barnzi] = ((*barvar_dim)(_1795_barj));
              _ptr__1784_matnnz[_1789_barnzi] = (_1792_p - _1794_p0);
              ++ _1789_barnzi;
            }
          }
          _1792_p = _1793_pe;
          _ptr__1781_ptr[safe_add( _1791_i,(int)1 )] = _1789_barnzi;
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1796_symmatidx = task_appendsymmatlist(_1783_matdim,_1784_matnnz,_1785_matsubi,_1787_matsubj,_1788_matcof);
      _checked_ptr_1<int64_t> _ptr__1796_symmatidx(_1796_symmatidx ? _1796_symmatidx->raw() : nullptr,_1796_symmatidx ? _1796_symmatidx->size(0) : 0);
      task_putafebarfrowlist(_1780_subi,_1781_ptr,_1782_barsubj,_1796_symmatidx);
    }
  }
  {}
}
// End mosek.fusion.Model.putfrows

// Begin mosek.fusion.Model.putarows
// Method mosek.fusion.Model.putarows @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1581:6-1681:7
void mosek::fusion::p_Model::putarows(std::shared_ptr< monty::ndarray< int32_t,1 > > _1797_nativeidxs,monty::rc_ptr< ::mosek::fusion::WorkStack > _1798_rs,int32_t _1799_nelem,int32_t _1800_nnz,int32_t _1801_ptr_base,int32_t _1802_nidxs_base,int32_t _1803_cof_base)
{
  _checked_ptr_1<int32_t> _ptr__1797_nativeidxs(_1797_nativeidxs ? _1797_nativeidxs->raw() : nullptr, _1797_nativeidxs ? _1797_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1804_ri32 = mosek::fusion::p_WorkStack::_get_impl(_1798_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1804_ri32(_1804_ri32 ? _1804_ri32->raw() : nullptr,_1804_ri32 ? _1804_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1805_ri64 = mosek::fusion::p_WorkStack::_get_impl(_1798_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1805_ri64(_1805_ri64 ? _1805_ri64->raw() : nullptr,_1805_ri64 ? _1805_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1806_rf64 = mosek::fusion::p_WorkStack::_get_impl(_1798_rs)->f64;
  _checked_ptr_1<double> _ptr__1806_rf64(_1806_rf64 ? _1806_rf64->raw() : nullptr,_1806_rf64 ? _1806_rf64->size(0) : 0);
  int32_t _1807_linnz = (int)0;
  int32_t _1808_barnnz = (int)0;
  int32_t _1809_totalpsdelm = (int)0;
  int32_t _9882_ = (int)0;
  int32_t _9883_ = _1799_nelem;
  for (int32_t _1810_i = _9882_; _1810_i < _9883_; ++_1810_i)
  {
    {
      int32_t _1811_ps = _ptr__1804_ri32[safe_add( _1801_ptr_base,_1810_i )];
      int32_t _1812_pe = _ptr__1804_ri32[safe_add( safe_add( _1801_ptr_base,_1810_i ),(int)1 )];
      int32_t _1813_p = _1811_ps;
      while ( ((_1813_p < _1812_pe) && (_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1813_p )] < (int)0)) )
      {
        {
          int32_t _1814_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1813_p )],(int)1 ))));
          int32_t _1815_p0 = _1813_p;
          ++ _1813_p;
          while ( ((_1813_p < _1812_pe) && ((_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1813_p )] < (int)0) && (_1814_barj == ((*barvar_block_elm_barj)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1813_p )],(int)1 ))))))) )
          {
            {
              ++ _1813_p;
            }
          }
          ++ _1808_barnnz;
        }
      }
      _1809_totalpsdelm += (_1813_p - _1811_ps);
      _1807_linnz += (_1812_pe - _1813_p);
    }
  }
  _1807_linnz += _1799_nelem;
  int32_t _1816_numcon = (int)((_1797_nativeidxs)->size(0));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1817_ptrb = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1816_numcon,(int)1 ))));
  _checked_ptr_1<int64_t> _ptr__1817_ptrb(_1817_ptrb ? _1817_ptrb->raw() : nullptr,_1817_ptrb ? _1817_ptrb->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1818_subj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1807_linnz)));
  _checked_ptr_1<int32_t> _ptr__1818_subj(_1818_subj ? _1818_subj->raw() : nullptr,_1818_subj ? _1818_subj->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1819_cof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1807_linnz)));
  _checked_ptr_1<double> _ptr__1819_cof(_1819_cof ? _1819_cof->raw() : nullptr,_1819_cof ? _1819_cof->size(0) : 0);
  {
    int32_t _1820_k = (int)0;
    int32_t _9884_ = (int)0;
    int32_t _9885_ = _1799_nelem;
    for (int32_t _1821_i = _9884_; _1821_i < _9885_; ++_1821_i)
    {
      {
        int32_t _1822_knz = (int)0;
        int32_t _9886_ = _ptr__1804_ri32[safe_add( _1801_ptr_base,_1821_i )];
        int32_t _9887_ = _ptr__1804_ri32[safe_add( safe_add( _1801_ptr_base,_1821_i ),(int)1 )];
        for (int32_t _1823_j = _9886_; _1823_j < _9887_; ++_1823_j)
        {
          {
            if ((_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1823_j )] >= (int)0))
            {
              {
                _ptr__1818_subj[safe_add( _1820_k,_1822_knz )] = (int32_t)_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1823_j )];
                _ptr__1819_cof[safe_add( _1820_k,_1822_knz )] = _ptr__1806_rf64[safe_add( _1803_cof_base,_1823_j )];
                ++ _1822_knz;
              }
            }
            {}
          }
        }
        _ptr__1817_ptrb[safe_add( _1821_i,(int)1 )] = safe_add( _ptr__1817_ptrb[_1821_i],_1822_knz );
        _1820_k += _1822_knz;
      }
    }
  }
  task_putarowlist(_1797_nativeidxs,_1817_ptrb,_1818_subj,_1819_cof);
  if ((_1808_barnnz > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1824_subi = _1797_nativeidxs;
      _checked_ptr_1<int32_t> _ptr__1824_subi(_1824_subi ? _1824_subi->raw() : nullptr,_1824_subi ? _1824_subi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1825_ptr = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_add( _1816_numcon,(int)1 ))));
      _checked_ptr_1<int64_t> _ptr__1825_ptr(_1825_ptr ? _1825_ptr->raw() : nullptr,_1825_ptr ? _1825_ptr->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1826_barsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1808_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1826_barsubj(_1826_barsubj ? _1826_barsubj->raw() : nullptr,_1826_barsubj ? _1826_barsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1827_matdim = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1808_barnnz)));
      _checked_ptr_1<int32_t> _ptr__1827_matdim(_1827_matdim ? _1827_matdim->raw() : nullptr,_1827_matdim ? _1827_matdim->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1828_matnnz = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_1808_barnnz)));
      _checked_ptr_1<int64_t> _ptr__1828_matnnz(_1828_matnnz ? _1828_matnnz->raw() : nullptr,_1828_matnnz ? _1828_matnnz->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1829_matsubi = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1809_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1829_matsubi(_1829_matsubi ? _1829_matsubi->raw() : nullptr,_1829_matsubi ? _1829_matsubi->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1830_matsubj = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1809_totalpsdelm)));
      _checked_ptr_1<int32_t> _ptr__1830_matsubj(_1830_matsubj ? _1830_matsubj->raw() : nullptr,_1830_matsubj ? _1830_matsubj->size(0) : 0);
      std::shared_ptr< monty::ndarray< double,1 > > _1831_matcof = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1809_totalpsdelm)));
      _checked_ptr_1<double> _ptr__1831_matcof(_1831_matcof ? _1831_matcof->raw() : nullptr,_1831_matcof ? _1831_matcof->size(0) : 0);
      int32_t _1832_barnzi = (int)0;
      int32_t _1833_barmatnzi = (int)0;
      int32_t _9888_ = (int)0;
      int32_t _9889_ = _1816_numcon;
      for (int32_t _1834_i = _9888_; _1834_i < _9889_; ++_1834_i)
      {
        {
          int32_t _1835_p = _ptr__1804_ri32[safe_add( _1801_ptr_base,_1834_i )];
          int32_t _1836_pe = _ptr__1804_ri32[safe_add( safe_add( _1801_ptr_base,_1834_i ),(int)1 )];
          while ( ((_1835_p < _1836_pe) && (_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )] < (int)0)) )
          {
            {
              int32_t _1837_p0 = _1835_p;
              int32_t _1838_barj = ((*barvar_block_elm_barj)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )],(int)1 ))));
              while ( ((_1835_p < _1836_pe) && ((_ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )] < (int)0) && (((*barvar_block_elm_barj)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )],(int)1 )))) == _1838_barj))) )
              {
                {
                  _ptr__1829_matsubi[_1833_barmatnzi] = ((*barvar_block_elm_i)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )],(int)1 ))));
                  _ptr__1830_matsubj[_1833_barmatnzi] = ((*barvar_block_elm_j)((-safe_add( _ptr__1805_ri64[safe_add( _1802_nidxs_base,_1835_p )],(int)1 ))));
                  if ((_ptr__1829_matsubi[_1833_barmatnzi] == _ptr__1830_matsubj[_1833_barmatnzi]))
                  {
                    {
                      _ptr__1831_matcof[_1833_barmatnzi] = _ptr__1806_rf64[safe_add( _1803_cof_base,_1835_p )];
                    }
                  }
                  else
                  {
                    {
                      _ptr__1831_matcof[_1833_barmatnzi] = (0.5 * _ptr__1806_rf64[safe_add( _1803_cof_base,_1835_p )]);
                    }
                  }
                  ++ _1833_barmatnzi;
                  ++ _1835_p;
                }
              }
              _ptr__1826_barsubj[_1832_barnzi] = _1838_barj;
              _ptr__1827_matdim[_1832_barnzi] = ((*barvar_dim)(_1838_barj));
              _ptr__1828_matnnz[_1832_barnzi] = (_1835_p - _1837_p0);
              ++ _1832_barnzi;
            }
          }
          _1835_p = _1836_pe;
          _ptr__1825_ptr[safe_add( _1834_i,(int)1 )] = _1832_barnzi;
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _1839_symmatidx = task_appendsymmatlist(_1827_matdim,_1828_matnnz,_1829_matsubi,_1830_matsubj,_1831_matcof);
      _checked_ptr_1<int64_t> _ptr__1839_symmatidx(_1839_symmatidx ? _1839_symmatidx->raw() : nullptr,_1839_symmatidx ? _1839_symmatidx->size(0) : 0);
      task_putbararowlist(_1824_subi,_1825_ptr,_1826_barsubj,_1839_symmatidx);
    }
  }
  {}
}
// End mosek.fusion.Model.putarows

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1453:6-1565:7
monty::rc_ptr< ::mosek::fusion::RangedConstraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1840_name,monty::rc_ptr< ::mosek::fusion::Expression > _1841_expr,monty::rc_ptr< ::mosek::fusion::RangeDomain > _1842_dom_)
{
  if ((((int)((_1840_name).size()) > (int)0) && hasConstraint(_1840_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1840_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_RangeDomain::_get_impl(_1842_dom_)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1843_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1841_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1843_e->eval(rs,ws,xs);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1844_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1844_ri32(_1844_ri32 ? _1844_ri32->raw() : nullptr,_1844_ri32 ? _1844_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1845_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1845_ri64(_1845_ri64 ? _1845_ri64->raw() : nullptr,_1845_ri64 ? _1845_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1846_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1846_rf64(_1846_rf64 ? _1846_rf64->raw() : nullptr,_1846_rf64 ? _1846_rf64->size(0) : 0);
  int32_t _1847_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1848_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1849_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1850_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  int32_t _1851_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  bool _1852_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _1853_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _1854_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _1855_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _1856_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _1857_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _1858_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _9890_;
  for (int32_t _1860_i = (int)0, _9891_ = _1847_nd; _1860_i < _9891_ ; ++_1860_i)
  {
    _9890_.push_back(_ptr__1844_ri32[safe_add( _1848_shape_base,_1860_i )]);
  }
  auto _9892_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9890_.size()),_9890_.begin(),_9890_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1859_shape = _9892_;
  _checked_ptr_1<int32_t> _ptr__1859_shape(_1859_shape ? _1859_shape->raw() : nullptr,_1859_shape ? _1859_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::RangeDomain > _1861_dom = mosek::fusion::p_RangeDomain::_get_impl(_1842_dom_.get())->__mosek_2fusion_2RangeDomain__finalize_and_validate(_1859_shape);
  if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->shape.get()))
  {
    {
      if (((int)((mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->shape)->size(0)) != _1847_nd))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Mismatching expression and domain"));
        }
      }
      else
      {
        {
          int32_t _9893_ = (int)0;
          int32_t _9894_ = _1847_nd;
          for (int32_t _1862_i = _9893_; _1862_i < _9894_; ++_1862_i)
          {
            {
              if ((((*mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->shape)(_1862_i)) != _ptr__1844_ri32[safe_add( _1848_shape_base,_1862_i )]))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Mismatching expression and domain"));
                }
              }
              {}
            }
          }
        }
      }
    }
  }
  {}
  int32_t _1863_numcon = (int)1;
  int32_t _9895_ = (int)0;
  int32_t _9896_ = _1847_nd;
  for (int32_t _1864_i = _9895_; _1864_i < _9896_; ++_1864_i)
  {
    {
      _1863_numcon *= _ptr__1844_ri32[safe_add( _1848_shape_base,_1864_i )];
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1865_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1863_numcon)));
  _checked_ptr_1<int32_t> _ptr__1865_nativeidxs(_1865_nativeidxs ? _1865_nativeidxs->raw() : nullptr,_1865_nativeidxs ? _1865_nativeidxs->size(0) : 0);
  int32_t _1866_conid = con_allocate(_1865_nativeidxs);
  putarows(_1865_nativeidxs,rs,_1849_nelem,_1850_nnz,_1853_ptr_base,_1854_nidxs_base,_1855_cof_base);
  xs->clear();
  std::shared_ptr< monty::ndarray< double,1 > > _1867_bl = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1863_numcon)));
  _checked_ptr_1<double> _ptr__1867_bl(_1867_bl ? _1867_bl->raw() : nullptr,_1867_bl ? _1867_bl->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1868_bu = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1863_numcon)));
  _checked_ptr_1<double> _ptr__1868_bu(_1868_bu ? _1868_bu->raw() : nullptr,_1868_bu ? _1868_bu->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1869_domlb = mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->lb;
  _checked_ptr_1<double> _ptr__1869_domlb(_1869_domlb ? _1869_domlb->raw() : nullptr,_1869_domlb ? _1869_domlb->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1870_domub = mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->ub;
  _checked_ptr_1<double> _ptr__1870_domub(_1870_domub ? _1870_domub->raw() : nullptr,_1870_domub ? _1870_domub->size(0) : 0);
  bool _1871_domempty = mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->empty;
  bool _1872_domsc = mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->scalable;
  std::shared_ptr< monty::ndarray< int32_t,2 > > _1873_domsp = mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->sparsity;
  {
    if (_1872_domsc)
    {
      {
        int32_t _9897_ = (int)0;
        int32_t _9898_ = _1863_numcon;
        for (int32_t _1874_i = _9897_; _1874_i < _9898_; ++_1874_i)
        {
          {
            _ptr__1867_bl[_1874_i] = _ptr__1869_domlb[(int)0];
            _ptr__1868_bu[_1874_i] = _ptr__1870_domub[(int)0];
          }
        }
      }
    }
    else if(_1871_domempty)
    {
      {
      }
    }
    else if((NULL != _1873_domsp.get()))
    {
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _1875_stride = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1847_nd)));
        _checked_ptr_1<int32_t> _ptr__1875_stride(_1875_stride ? _1875_stride->raw() : nullptr,_1875_stride ? _1875_stride->size(0) : 0);
        _ptr__1875_stride[(_1847_nd - (int)1)] = (int)1;
        int32_t _9899_ = (int)1;
        int32_t _9900_ = _1847_nd;
        for (int32_t _1876_i = _9899_; _1876_i < _9900_; ++_1876_i)
        {
          {
            _ptr__1875_stride[((_1847_nd - _1876_i) - (int)1)] = safe_mul( _ptr__1875_stride[(_1847_nd - _1876_i)],_ptr__1844_ri32[(safe_add( _1848_shape_base,_1847_nd ) - _1876_i)] );
          }
        }
        int32_t _9901_ = (int)0;
        int32_t _9902_ = (int)((_1869_domlb)->size(0));
        for (int32_t _1877_i = _9901_; _1877_i < _9902_; ++_1877_i)
        {
          {
            int32_t _1878_idx = (int)0;
            int32_t _9903_ = (int)0;
            int32_t _9904_ = _1847_nd;
            for (int32_t _1879_j = _9903_; _1879_j < _9904_; ++_1879_j)
            {
              {
                _1878_idx += safe_mul( _ptr__1875_stride[_1879_j],((*_1873_domsp)(_1877_i,_1879_j)) );
              }
            }
            _ptr__1867_bl[_1878_idx] = _ptr__1869_domlb[_1877_i];
            _ptr__1868_bu[_1878_idx] = _ptr__1870_domub[_1877_i];
          }
        }
      }
    }
    else
    {
      {
        int32_t _9905_ = (int)0;
        int32_t _9906_ = _1863_numcon;
        for (int32_t _1880_i = _9905_; _1880_i < _9906_; ++_1880_i)
        {
          {
            _ptr__1867_bl[_1880_i] = _ptr__1869_domlb[_1880_i];
            _ptr__1868_bu[_1880_i] = _ptr__1870_domub[_1880_i];
          }
        }
      }
    }
  }
  task_con_putboundlist_ra(_1865_nativeidxs,_1867_bl,_1868_bu);
  if ((_1851_ncodeatom > (int)0))
  {
    {
      con_blocks->replace_row_code(rs,_1865_nativeidxs,_1853_ptr_base,_1854_nidxs_base,_1856_codeptr,_1857_code,_1858_cconst);
    }
  }
  {}
  if (((int)((_1840_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1881_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1881_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1840_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((int)((_1859_shape)->size(0)) > (int)0))
      {
        {
          _1881_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
        }
      }
      {}
      int32_t _9907_ = (int)1;
      int32_t _9908_ = (int)((_1859_shape)->size(0));
      for (int32_t _1882_i = _9907_; _1882_i < _9908_; ++_1882_i)
      {
        {
          _1881_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1882_i);
        }
      }
      _1881_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->indexnames.get()))
      {
        {
          task_format_con_names(_1865_nativeidxs,_1881_sb->toString(),_1859_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_RangeDomain::_get_impl(_1861_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_con_names(_1865_nativeidxs,_1881_sb->toString(),_1859_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  std::vector<int32_t> _9909_;
  for (int32_t _1884_i = (int)0, _9910_ = _1847_nd; _1884_i < _9910_ ; ++_1884_i)
  {
    _9909_.push_back(_ptr__1844_ri32[safe_add( _1848_shape_base,_1884_i )]);
  }
  auto _9911_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9909_.size()),_9909_.begin(),_9909_.end()));
  monty::rc_ptr< ::mosek::fusion::RangedConstraint > _1883_res = ::mosek::fusion::p_RangedConstraint::_new_RangedConstraint(::mosek::fusion::Model::t(_pubthis),_1840_name,_9911_,_1865_nativeidxs,_1866_conid);
  ((*cons)(_1866_conid)) = (_1883_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1840_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1840_name,_1866_conid);
    }
  }
  {}
  return _1883_res;
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:1256:6-1433:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1885_name,monty::rc_ptr< ::mosek::fusion::Expression > _1886_expr,monty::rc_ptr< ::mosek::fusion::PSDDomain > _1887_dom_)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1888_shape = _1886_expr->getShape();
  _checked_ptr_1<int32_t> _ptr__1888_shape(_1888_shape ? _1888_shape->raw() : nullptr,_1888_shape ? _1888_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::PSDDomain > _1889_dom = mosek::fusion::p_PSDDomain::_get_impl(_1887_dom_.get())->__mosek_2fusion_2PSDDomain__finalize_and_validate(_1888_shape);
  bool _1890_tril = (mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim2);
  int32_t _9912_;
  bool _9913_ = (mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim2);
  if (_9913_)
  {
    _9912_ = mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim2;
  }
  else
  {
    _9912_ = mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim1;
  }
  int32_t _1891_conedim1 = _9912_;
  int32_t _9914_;
  bool _9915_ = (mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim1 < mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim2);
  if (_9915_)
  {
    _9914_ = mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim1;
  }
  else
  {
    _9914_ = mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->conedim2;
  }
  int32_t _1892_conedim0 = _9914_;
  monty::rc_ptr< ::mosek::fusion::Expression > _9916_;
  bool _9917_ = (mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->key == ::mosek::fusion::PSDKey::IsSymPSD);
  if (_9917_)
  {
    _9916_ = ::mosek::fusion::p_ExprSymmetrize::_new_ExprSymmetrize(_1892_conedim0,_1891_conedim1,_1886_expr).as<::mosek::fusion::Expression>();
  }
  else
  {
    _9916_ = _1886_expr;
  }
  monty::rc_ptr< ::mosek::fusion::ExprDenseTril > _1893_e = ::mosek::fusion::p_ExprDenseTril::_new_ExprDenseTril(_1892_conedim0,_1891_conedim1,(::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress((::mosek::fusion::p_ExprScaleVecPSD::_new_ExprScaleVecPSD(_1892_conedim0,_1891_conedim1,(::mosek::fusion::p_ExprDenseTril::_new_ExprDenseTril(_1892_conedim0,_1891_conedim1,_9916_)).as<::mosek::fusion::BaseExpression>())).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1893_e->eval(rs,ws,xs);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1894_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1894_ri32(_1894_ri32 ? _1894_ri32->raw() : nullptr,_1894_ri32 ? _1894_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1895_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1895_ri64(_1895_ri64 ? _1895_ri64->raw() : nullptr,_1895_ri64 ? _1895_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1896_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1896_rf64(_1896_rf64 ? _1896_rf64->raw() : nullptr,_1896_rf64 ? _1896_rf64->size(0) : 0);
  int32_t _1897_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1898_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1899_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  bool _1900_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _1901_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  int32_t _1902_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1903_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _1904_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _1905_sp_base = mosek::fusion::p_WorkStack::_get_impl(rs)->sp_base;
  int32_t _1906_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _1907_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _1908_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _1909_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  int32_t _1910_conedim = _ptr__1888_shape[_1891_conedim1];
  int32_t _1911_conesize = (safe_mul( _1910_conedim,safe_add( _1910_conedim,(int)1 ) ) / (int)2);
  int32_t _1912_numcone = (int)1;
  int32_t _9918_ = (int)0;
  int32_t _9919_ = _1897_nd;
  for (int32_t _1913_i = _9918_; _1913_i < _9919_; ++_1913_i)
  {
    {
      if (((_1913_i != _1892_conedim0) && (_1913_i != _1891_conedim1)))
      {
        {
          _1912_numcone *= _ptr__1888_shape[_1913_i];
        }
      }
      {}
    }
  }
  int64_t _1914_domainidx = task_append_domain_svec_psd(_1911_conesize);
  int32_t _1915_domsize = safe_mul( _1911_conesize,_1912_numcone );
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1916_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1915_domsize)));
  _checked_ptr_1<int32_t> _ptr__1916_afeidxs(_1916_afeidxs ? _1916_afeidxs->raw() : nullptr,_1916_afeidxs ? _1916_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1917_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1912_numcone)));
  _checked_ptr_1<int32_t> _ptr__1917_accidxs(_1917_accidxs ? _1917_accidxs->raw() : nullptr,_1917_accidxs ? _1917_accidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1918_conelmidx = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1915_domsize)));
  _checked_ptr_1<int32_t> _ptr__1918_conelmidx(_1918_conelmidx ? _1918_conelmidx->raw() : nullptr,_1918_conelmidx ? _1918_conelmidx->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1919_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_1915_domsize)));
  _checked_ptr_1<double> _ptr__1919_g(_1919_g ? _1919_g->raw() : nullptr,_1919_g ? _1919_g->size(0) : 0);
  int32_t _1920_conid = acon_allocate(_1914_domainidx,_1911_conesize,_1912_numcone,_1919_g,_1916_afeidxs,_1917_accidxs);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1921_aconidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1915_domsize)));
  _checked_ptr_1<int32_t> _ptr__1921_aconidxs(_1921_aconidxs ? _1921_aconidxs->raw() : nullptr,_1921_aconidxs ? _1921_aconidxs->size(0) : 0);
  acon_block_map->get(_1920_conid,_1921_aconidxs,(int)0);
  int64_t _1922_d5 = (int)1;
  int64_t _1923_d4 = _ptr__1888_shape[_1891_conedim1];
  int64_t _1924_d3 = (int)1;
  int64_t _1925_d2 = _ptr__1888_shape[_1892_conedim0];
  int64_t _1926_d1 = (int)1;
  int32_t _9920_ = (int)0;
  int32_t _9921_ = _1892_conedim0;
  for (int32_t _1927_i = _9920_; _1927_i < _9921_; ++_1927_i)
  {
    {
      _1926_d1 *= _ptr__1888_shape[_1927_i];
    }
  }
  int32_t _9922_ = safe_add( _1892_conedim0,(int)1 );
  int32_t _9923_ = _1891_conedim1;
  for (int32_t _1928_i = _9922_; _1928_i < _9923_; ++_1928_i)
  {
    {
      _1924_d3 *= _ptr__1888_shape[_1928_i];
    }
  }
  int32_t _9924_ = safe_add( _1891_conedim1,(int)1 );
  int32_t _9925_ = _1897_nd;
  for (int32_t _1929_i = _9924_; _1929_i < _9925_; ++_1929_i)
  {
    {
      _1922_d5 *= _ptr__1888_shape[_1929_i];
    }
  }
  {
    int32_t _1930_k = (int)0;
    double _1931_s = (1.0 / ::mosek::fusion::Utils::Tools::sqrt((double)((int)2)));
    int32_t _9926_ = (int)0;
    int64_t _9927_ = safe_mul( safe_mul( _1926_d1,_1924_d3 ),_1922_d5 );
    for (int64_t _1932_i1 = _9926_; _1932_i1 < _9927_; ++_1932_i1)
    {
      {
        int32_t _9928_ = (int)0;
        int64_t _9929_ = _1925_d2;
        for (int64_t _1933_i2 = _9928_; _1933_i2 < _9929_; ++_1933_i2)
        {
          {
            ++ _1930_k;
            int64_t _9930_ = safe_add( _1933_i2,(int)1 );
            int64_t _9931_ = _1923_d4;
            for (int64_t _1934_i4 = _9930_; _1934_i4 < _9931_; ++_1934_i4)
            {
              {
                ((*acon_elm_scale)(_ptr__1921_aconidxs[_1930_k])) = _1931_s;
                ++ _1930_k;
              }
            }
          }
        }
      }
    }
  }
  {
    std::shared_ptr< monty::ndarray< int32_t,1 > > _1935_afetrilperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_1898_nelem)));
    _checked_ptr_1<int32_t> _ptr__1935_afetrilperm(_1935_afetrilperm ? _1935_afetrilperm->raw() : nullptr,_1935_afetrilperm ? _1935_afetrilperm->size(0) : 0);
    int32_t _1936_k = (int)0;
    if (_1890_tril)
    {
      {
        int32_t _9932_ = (int)0;
        int64_t _9933_ = _1926_d1;
        for (int64_t _1937_i1 = _9932_; _1937_i1 < _9933_; ++_1937_i1)
        {
          {
            int32_t _9934_ = (int)0;
            int64_t _9935_ = _1923_d4;
            for (int64_t _1938_i4 = _9934_; _1938_i4 < _9935_; ++_1938_i4)
            {
              {
                int32_t _9936_ = (int)0;
                int64_t _9937_ = _1924_d3;
                for (int64_t _1939_i3 = _9936_; _1939_i3 < _9937_; ++_1939_i3)
                {
                  {
                    int64_t _9938_ = _1938_i4;
                    int64_t _9939_ = _1925_d2;
                    for (int64_t _1940_i2 = _9938_; _1940_i2 < _9939_; ++_1940_i2)
                    {
                      {
                        int32_t _9940_ = (int)0;
                        int64_t _9941_ = _1922_d5;
                        for (int64_t _1941_i5 = _9940_; _1941_i5 < _9941_; ++_1941_i5)
                        {
                          {
                            int64_t _1942_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1937_i1,_1924_d3 ),_1922_d5 ),safe_mul( _1939_i3,_1922_d5 ) ),_1941_i5 );
                            int64_t _1943_coneofs = safe_add( (safe_mul( _1940_i2,safe_add( _1940_i2,(int)1 ) ) / (int)2),_1938_i4 );
                            _ptr__1935_afetrilperm[safe_add( safe_mul( _1942_coneidx,_1911_conesize ),_1943_coneofs )] = _ptr__1916_afeidxs[_1936_k];
                            ++ _1936_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    else
    {
      {
        int32_t _9942_ = (int)0;
        int64_t _9943_ = _1926_d1;
        for (int64_t _1944_i1 = _9942_; _1944_i1 < _9943_; ++_1944_i1)
        {
          {
            int32_t _9944_ = (int)0;
            int64_t _9945_ = _1923_d4;
            for (int64_t _1945_i4 = _9944_; _1945_i4 < _9945_; ++_1945_i4)
            {
              {
                int32_t _9946_ = (int)0;
                int64_t _9947_ = _1924_d3;
                for (int64_t _1946_i3 = _9946_; _1946_i3 < _9947_; ++_1946_i3)
                {
                  {
                    int32_t _9948_ = (int)0;
                    int64_t _9949_ = safe_add( _1945_i4,(int)1 );
                    for (int64_t _1947_i2 = _9948_; _1947_i2 < _9949_; ++_1947_i2)
                    {
                      {
                        int32_t _9950_ = (int)0;
                        int64_t _9951_ = _1922_d5;
                        for (int64_t _1948_i5 = _9950_; _1948_i5 < _9951_; ++_1948_i5)
                        {
                          {
                            int64_t _1949_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1944_i1,_1924_d3 ),_1922_d5 ),safe_mul( _1946_i3,_1922_d5 ) ),_1948_i5 );
                            int64_t _1950_coneofs = safe_add( (safe_mul( _1947_i2,((safe_mul( (int)2,_1925_d2 ) - _1947_i2) - (int)1) ) / (int)2),_1945_i4 );
                            _ptr__1935_afetrilperm[safe_add( safe_mul( _1949_coneidx,_1911_conesize ),_1950_coneofs )] = _ptr__1916_afeidxs[_1936_k];
                            ++ _1936_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    putfrows(_1935_afetrilperm,(int)0,rs,_1898_nelem,_1899_nnz,_1903_ptr_base,_1904_nidxs_base,_1906_cof_base);
    if ((_1901_ncodeatom > (int)0))
    {
      {
        afe_blocks->replace_row_code(rs,_1935_afetrilperm,_1903_ptr_base,_1904_nidxs_base,_1907_codeptr,_1908_code,_1909_cconst);
      }
    }
    {}
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1951_pafeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_mul( safe_mul( _1912_numcone,_1925_d2 ),_1925_d2 ))));
  _checked_ptr_1<int32_t> _ptr__1951_pafeidxs(_1951_pafeidxs ? _1951_pafeidxs->raw() : nullptr,_1951_pafeidxs ? _1951_pafeidxs->size(0) : 0);
  {
    int32_t _1952_k = (int)0;
    if (_1890_tril)
    {
      {
        int32_t _9952_ = (int)0;
        int64_t _9953_ = _1926_d1;
        for (int64_t _1953_i1 = _9952_; _1953_i1 < _9953_; ++_1953_i1)
        {
          {
            int32_t _9954_ = (int)0;
            int64_t _9955_ = _1925_d2;
            for (int64_t _1954_i2 = _9954_; _1954_i2 < _9955_; ++_1954_i2)
            {
              {
                int32_t _9956_ = (int)0;
                int64_t _9957_ = _1924_d3;
                for (int64_t _1955_i3 = _9956_; _1955_i3 < _9957_; ++_1955_i3)
                {
                  {
                    int32_t _9958_ = (int)0;
                    int64_t _9959_ = _1923_d4;
                    for (int64_t _1956_i4 = _9958_; _1956_i4 < _9959_; ++_1956_i4)
                    {
                      {
                        int32_t _9960_ = (int)0;
                        int64_t _9961_ = _1922_d5;
                        for (int64_t _1957_i5 = _9960_; _1957_i5 < _9961_; ++_1957_i5)
                        {
                          {
                            int64_t _1958_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1953_i1,_1924_d3 ),_1922_d5 ),safe_mul( _1955_i3,_1922_d5 ) ),_1957_i5 );
                            int64_t _9962_;
                            bool _9963_ = (_1954_i2 >= _1956_i4);
                            if (_9963_)
                            {
                              _9962_ = safe_add( (safe_mul( _1956_i4,((safe_mul( (int)2,_1925_d2 ) - _1956_i4) - (int)1) ) / (int)2),_1954_i2 );
                            }
                            else
                            {
                              _9962_ = safe_add( (safe_mul( _1954_i2,((safe_mul( (int)2,_1925_d2 ) - _1954_i2) - (int)1) ) / (int)2),_1956_i4 );
                            }
                            int64_t _1959_coneofs = _9962_;
                            _ptr__1951_pafeidxs[_1952_k] = _ptr__1916_afeidxs[safe_add( safe_mul( _1958_coneidx,_1911_conesize ),_1959_coneofs )];
                            ++ _1952_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    else
    {
      {
        int32_t _9964_ = (int)0;
        int64_t _9965_ = _1926_d1;
        for (int64_t _1960_i1 = _9964_; _1960_i1 < _9965_; ++_1960_i1)
        {
          {
            int32_t _9966_ = (int)0;
            int64_t _9967_ = _1925_d2;
            for (int64_t _1961_i2 = _9966_; _1961_i2 < _9967_; ++_1961_i2)
            {
              {
                int32_t _9968_ = (int)0;
                int64_t _9969_ = _1924_d3;
                for (int64_t _1962_i3 = _9968_; _1962_i3 < _9969_; ++_1962_i3)
                {
                  {
                    int32_t _9970_ = (int)0;
                    int64_t _9971_ = _1923_d4;
                    for (int64_t _1963_i4 = _9970_; _1963_i4 < _9971_; ++_1963_i4)
                    {
                      {
                        int32_t _9972_ = (int)0;
                        int64_t _9973_ = _1922_d5;
                        for (int64_t _1964_i5 = _9972_; _1964_i5 < _9973_; ++_1964_i5)
                        {
                          {
                            int64_t _1965_coneidx = safe_add( safe_add( safe_mul( safe_mul( _1960_i1,_1924_d3 ),_1922_d5 ),safe_mul( _1962_i3,_1922_d5 ) ),_1964_i5 );
                            int64_t _9974_;
                            bool _9975_ = (_1961_i2 >= _1963_i4);
                            if (_9975_)
                            {
                              _9974_ = safe_add( (safe_mul( _1963_i4,((safe_mul( (int)2,_1925_d2 ) - _1963_i4) - (int)1) ) / (int)2),_1961_i2 );
                            }
                            else
                            {
                              _9974_ = safe_add( (safe_mul( _1961_i2,((safe_mul( (int)2,_1925_d2 ) - _1961_i2) - (int)1) ) / (int)2),_1963_i4 );
                            }
                            int64_t _1966_coneofs = _9974_;
                            _ptr__1951_pafeidxs[_1952_k] = _ptr__1916_afeidxs[safe_add( safe_mul( _1965_coneidx,_1911_conesize ),_1966_coneofs )];
                            ++ _1952_k;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  if (((int)((_1885_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _1967_coneshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_1888_shape)->size(0)) - (int)2))));
      _checked_ptr_1<int32_t> _ptr__1967_coneshape(_1967_coneshape ? _1967_coneshape->raw() : nullptr,_1967_coneshape ? _1967_coneshape->size(0) : 0);
      int32_t _9976_ = (int)0;
      int32_t _9977_ = _1892_conedim0;
      for (int32_t _1968_i = _9976_; _1968_i < _9977_; ++_1968_i)
      {
        {
          _ptr__1967_coneshape[_1968_i] = _ptr__1888_shape[_1968_i];
        }
      }
      int32_t _9978_ = safe_add( _1892_conedim0,(int)1 );
      int32_t _9979_ = _1891_conedim1;
      for (int32_t _1969_i = _9978_; _1969_i < _9979_; ++_1969_i)
      {
        {
          _ptr__1967_coneshape[(_1969_i - (int)1)] = _ptr__1888_shape[_1969_i];
        }
      }
      int32_t _9980_ = safe_add( _1891_conedim1,(int)1 );
      int32_t _9981_ = _1897_nd;
      for (int32_t _1970_i = _9980_; _1970_i < _9981_; ++_1970_i)
      {
        {
          _ptr__1967_coneshape[(_1970_i - (int)2)] = _ptr__1888_shape[_1970_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _1971_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _1971_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1885_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1897_nd > (int)2))
      {
        {
          _1971_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _9982_ = (int)1;
          int32_t _9983_ = (_1897_nd - (int)2);
          for (int32_t _1972_i = _9982_; _1972_i < _9983_; ++_1972_i)
          {
            {
              _1971_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1972_i);
            }
          }
        }
      }
      {}
      _1971_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1973_dindexnames = mosek::fusion::p_PSDDomain::_get_impl(_1889_dom)->indexnames;
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _1974_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1897_nd - (int)2))));
          int32_t _9984_ = (int)0;
          int32_t _9985_ = _1892_conedim0;
          for (int32_t _1975_i = _9984_; _1975_i < _9985_; ++_1975_i)
          {
            {
              ((*_1974_indexnames)(_1975_i)) = ((*_1973_dindexnames)(_1975_i));
            }
          }
          int32_t _9986_ = safe_add( _1892_conedim0,(int)1 );
          int32_t _9987_ = _1891_conedim1;
          for (int32_t _1976_i = _9986_; _1976_i < _9987_; ++_1976_i)
          {
            {
              ((*_1974_indexnames)((_1976_i - (int)1))) = ((*_1973_dindexnames)(_1976_i));
            }
          }
          int32_t _9988_ = safe_add( _1891_conedim1,(int)1 );
          int32_t _9989_ = _1897_nd;
          for (int32_t _1977_i = _9988_; _1977_i < _9989_; ++_1977_i)
          {
            {
              ((*_1974_indexnames)((_1977_i - (int)2))) = ((*_1973_dindexnames)(_1977_i));
            }
          }
          std::vector<int64_t> _9990_;
          for (int32_t _1978_i = (int)0, _9991_ = (int)((_1917_accidxs)->size(0)); _1978_i < _9991_ ; ++_1978_i)
          {
            _9990_.push_back(_ptr__1917_accidxs[_1978_i]);
          }
          auto _9992_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9990_.size()),_9990_.begin(),_9990_.end()));
          task_format_acc_names(_9992_,_1971_sb->toString(),_1967_coneshape,_1974_indexnames);
        }
      }
      else
      {
        {
          std::vector<int64_t> _9993_;
          for (int32_t _1979_i = (int)0, _9994_ = (int)((_1917_accidxs)->size(0)); _1979_i < _9994_ ; ++_1979_i)
          {
            _9993_.push_back(_ptr__1917_accidxs[_1979_i]);
          }
          auto _9995_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_9993_.size()),_9993_.begin(),_9993_.end()));
          task_format_acc_names(_9995_,_1971_sb->toString(),_1967_coneshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicConstraint > _1980_res = ::mosek::fusion::p_ConicConstraint::_new_ConicConstraint(::mosek::fusion::Model::t(_pubthis),_1885_name,nullptr,_1888_shape,_1920_conid,_1951_pafeidxs,nullptr);
  ((*acons)(_1920_conid)) = (_1980_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1885_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1885_name,(-safe_add( (int)1,_1920_conid )));
    }
  }
  {}
  return (_1980_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:831:6-1031:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _1981_name,monty::rc_ptr< ::mosek::fusion::Expression > _1982_expr,monty::rc_ptr< ::mosek::fusion::ConeDomain > _1983_dom_)
{
  if ((((int)((_1981_name).size()) > (int)0) && hasConstraint(_1981_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1981_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if ((((int)((_1981_name).size()) > (int)0) && hasConstraint(_1981_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_1981_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_ConeDomain::_get_impl(_1983_dom_)->int_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _1984_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_1982_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _1984_e->eval(rs,ws,xs);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _1985_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__1985_ri32(_1985_ri32 ? _1985_ri32->raw() : nullptr,_1985_ri32 ? _1985_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _1986_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__1986_ri64(_1986_ri64 ? _1986_ri64->raw() : nullptr,_1986_ri64 ? _1986_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _1987_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__1987_rf64(_1987_rf64 ? _1987_rf64->raw() : nullptr,_1987_rf64 ? _1987_rf64->size(0) : 0);
  int32_t _1988_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _1989_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _1990_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _1991_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  int32_t _1992_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  bool _1993_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _1994_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _1995_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _1996_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _1997_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _1998_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _1999_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _9996_;
  for (int32_t _2001_i = (int)0, _9997_ = _1988_nd; _2001_i < _9997_ ; ++_2001_i)
  {
    _9996_.push_back(_ptr__1985_ri32[safe_add( _1989_shape_base,_2001_i )]);
  }
  auto _9998_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_9996_.size()),_9996_.begin(),_9996_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2000_shape = _9998_;
  _checked_ptr_1<int32_t> _ptr__2000_shape(_2000_shape ? _2000_shape->raw() : nullptr,_2000_shape ? _2000_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::ConeDomain > _2002_dom = mosek::fusion::p_ConeDomain::_get_impl(_1983_dom_.get())->__mosek_2fusion_2ConeDomain__finalize_and_validate(_2000_shape);
  int32_t _9999_;
  bool _10000_ = _2002_dom->axisIsSet();
  if (_10000_)
  {
    _9999_ = _2002_dom->getAxis();
  }
  else
  {
    _9999_ = (_1988_nd - (int)1);
  }
  int32_t _2003_coneaxis = _9999_;
  int32_t _2004_conesize = _ptr__2000_shape[_2003_coneaxis];
  int32_t _2005_d1 = (int)1;
  int32_t _2006_d0 = (int)1;
  int32_t _2007_domsize = (int)1;
  int32_t _10001_ = (int)0;
  int32_t _10002_ = _1988_nd;
  for (int32_t _2008_i = _10001_; _2008_i < _10002_; ++_2008_i)
  {
    {
      _2007_domsize *= _ptr__2000_shape[_2008_i];
    }
  }
  int32_t _10003_ = (int)0;
  int32_t _10004_ = _2003_coneaxis;
  for (int32_t _2009_i = _10003_; _2009_i < _10004_; ++_2009_i)
  {
    {
      _2006_d0 *= _ptr__2000_shape[_2009_i];
    }
  }
  int32_t _10005_ = safe_add( _2003_coneaxis,(int)1 );
  int32_t _10006_ = _1988_nd;
  for (int32_t _2010_i = _10005_; _2010_i < _10006_; ++_2010_i)
  {
    {
      _2005_d1 *= _ptr__2000_shape[_2010_i];
    }
  }
  int32_t _2011_numcone = safe_mul( _2006_d0,_2005_d1 );
  int64_t _2012_domainidx;
  if ((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InQCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for quadratic cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_quad(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InRotatedQCone))
  {
    {
      if ((_2004_conesize < (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for rotated quadratic cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_rquad(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InPExpCone))
  {
    {
      if ((_2004_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for exponential cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_pexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InPPowCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for power cone"))->toString());
        }
      }
      {}
      if (((int)((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->alpha)->size(0)) < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Power cone requires at least two parameters"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_ppow(_2004_conesize,mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InDExpCone))
  {
    {
      if ((_2004_conesize != (int)3))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual exponential cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_dexp();
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InDPowCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual power cone"))->toString());
        }
      }
      {}
      if (((int)((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->alpha)->size(0)) < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Power cone requires at least two parameters"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_dpow(_2004_conesize,mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->alpha);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InPGeoMeanCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for primal geometric mean cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_pgeomean(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InDGeoMeanCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for dual geometric mean cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_dgeomean(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InSVecPSDCone))
  {
    {
      int32_t _2013_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_2004_conesize ) ))) - (int)1) / (int)2);
      if (((safe_mul( _2013_d,safe_add( _2013_d,(int)1 ) ) / (int)2) != _2004_conesize))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for vector positive semidefinite cone"))->toString());
        }
      }
      {}
      _2012_domainidx = task_append_domain_svec_psd(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::Positive))
  {
    {
      _2012_domainidx = task_append_domain_rpos(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::Negative))
  {
    {
      _2012_domainidx = task_append_domain_rneg(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::Unbounded))
  {
    {
      _2012_domainidx = task_append_domain_r(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::Zero))
  {
    {
      _2012_domainidx = task_append_domain_rzero(_2004_conesize);
    }
  }
  else if((mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->key == ::mosek::fusion::QConeKey::InPSDCone))
  {
    {
      if ((_2004_conesize < (int)2))
      {
        {
          throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
        }
      }
      else
      {
        {
          int32_t _2014_d = (int32_t)((::mosek::fusion::Utils::Tools::sqrt((double)(safe_add( (int)1,safe_mul( (int)8,_2004_conesize ) ))) - (int)1) / (int)2);
          if (((safe_mul( _2014_d,safe_add( _2014_d,(int)1 ) ) / (int)2) != _2004_conesize))
          {
            {
              throw ::mosek::fusion::DimensionError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Invalid size ("))->__mosek_2fusion_2Utils_2StringBuffer__a(_2004_conesize)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") for positive semidefinite cone"))->toString());
            }
          }
          {}
        }
      }
      _2012_domainidx = task_append_domain_svec_psd(_2004_conesize);
    }
  }
  else
  {
    {
      throw ::mosek::fusion::UnexpectedError(std::string ("Internal error: Invalid cone type"));
    }
  }
  std::shared_ptr< monty::ndarray< double,1 > > _2015_g = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_2007_domsize)));
  _checked_ptr_1<double> _ptr__2015_g(_2015_g ? _2015_g->raw() : nullptr,_2015_g ? _2015_g->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2016_afeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2007_domsize)));
  _checked_ptr_1<int32_t> _ptr__2016_afeidxs(_2016_afeidxs ? _2016_afeidxs->raw() : nullptr,_2016_afeidxs ? _2016_afeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2017_accidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2011_numcone)));
  _checked_ptr_1<int32_t> _ptr__2017_accidxs(_2017_accidxs ? _2017_accidxs->raw() : nullptr,_2017_accidxs ? _2017_accidxs->size(0) : 0);
  int32_t _2018_conid = acon_allocate(_2012_domainidx,_2004_conesize,_2011_numcone,_2015_g,_2016_afeidxs,_2017_accidxs);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2019_perm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2007_domsize)));
  _checked_ptr_1<int32_t> _ptr__2019_perm(_2019_perm ? _2019_perm->raw() : nullptr,_2019_perm ? _2019_perm->size(0) : 0);
  {
    int32_t _2020_idx = (int)0;
    std::shared_ptr< monty::ndarray< double,1 > > _2021_ofs = mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->domofs;
    _checked_ptr_1<double> _ptr__2021_ofs(_2021_ofs ? _2021_ofs->raw() : nullptr,_2021_ofs ? _2021_ofs->size(0) : 0);
    {
      int32_t _10007_ = (int)0;
      int32_t _10008_ = _2006_d0;
      for (int32_t _2022_i0 = _10007_; _2022_i0 < _10008_; ++_2022_i0)
      {
        {
          int32_t _10009_ = (int)0;
          int32_t _10010_ = _2005_d1;
          for (int32_t _2023_i1 = _10009_; _2023_i1 < _10010_; ++_2023_i1)
          {
            {
              int32_t _10011_ = (int)0;
              int32_t _10012_ = _2004_conesize;
              for (int32_t _2024_ic = _10011_; _2024_ic < _10012_; ++_2024_ic)
              {
                {
                  _ptr__2019_perm[safe_add( safe_add( safe_mul( safe_mul( _2022_i0,_2005_d1 ),_2004_conesize ),safe_mul( _2024_ic,_2005_d1 ) ),_2023_i1 )] = _2020_idx;
                  ++ _2020_idx;
                }
              }
            }
          }
        }
      }
      if ((NULL != _2021_ofs.get()))
      {
        {
          int32_t _10013_ = (int)0;
          int32_t _10014_ = _2007_domsize;
          for (int32_t _2025_i = _10013_; _2025_i < _10014_; ++_2025_i)
          {
            {
              _ptr__2015_g[_ptr__2019_perm[_2025_i]] = _ptr__2021_ofs[_2025_i];
            }
          }
        }
      }
      {}
    }
  }
  std::vector<int32_t> _10015_;
  for (int32_t _2027_i = (int)0, _10016_ = _2007_domsize; _2027_i < _10016_ ; ++_2027_i)
  {
    _10015_.push_back(_ptr__2016_afeidxs[_ptr__2019_perm[_2027_i]]);
  }
  auto _10017_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10015_.size()),_10015_.begin(),_10015_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2026_pafeidxs = _10017_;
  _checked_ptr_1<int32_t> _ptr__2026_pafeidxs(_2026_pafeidxs ? _2026_pafeidxs->raw() : nullptr,_2026_pafeidxs ? _2026_pafeidxs->size(0) : 0);
  putfrows(_2026_pafeidxs,(int)0,rs,_1990_nelem,_1991_nnz,_1994_ptr_base,_1995_nidxs_base,_1996_cof_base);
  if ((_1992_ncodeatom > (int)0))
  {
    {
      afe_blocks->replace_row_code(rs,_2026_pafeidxs,_1994_ptr_base,_1995_nidxs_base,_1998_codeptr,_1997_code,_1999_cconst);
    }
  }
  {}
  if (((int)((_1981_name).size()) > (int)0))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2028_accshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((_2000_shape)->size(0)) - (int)1))));
      _checked_ptr_1<int32_t> _ptr__2028_accshape(_2028_accshape ? _2028_accshape->raw() : nullptr,_2028_accshape ? _2028_accshape->size(0) : 0);
      int32_t _10018_ = (int)0;
      int32_t _10019_ = _2003_coneaxis;
      for (int32_t _2029_i = _10018_; _2029_i < _10019_; ++_2029_i)
      {
        {
          _ptr__2028_accshape[_2029_i] = _ptr__2000_shape[_2029_i];
        }
      }
      int32_t _10020_ = safe_add( _2003_coneaxis,(int)1 );
      int32_t _10021_ = _1988_nd;
      for (int32_t _2030_i = _10020_; _2030_i < _10021_; ++_2030_i)
      {
        {
          _ptr__2028_accshape[(_2030_i - (int)1)] = _ptr__2000_shape[_2030_i];
        }
      }
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2031_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _2031_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_1981_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if ((_1988_nd > (int)1))
      {
        {
          _2031_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
          int32_t _10022_ = (int)1;
          int32_t _10023_ = (_1988_nd - (int)1);
          for (int32_t _2032_i = _10022_; _2032_i < _10023_; ++_2032_i)
          {
            {
              _2031_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2032_i);
            }
          }
        }
      }
      {}
      _2031_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->indexnames.get()))
      {
        {
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2033_dindexnames = mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->indexnames;
          std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2034_indexnames = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 >(monty::shape((_1988_nd - (int)1))));
          int32_t _10024_ = (int)0;
          int32_t _10025_ = _2003_coneaxis;
          for (int32_t _2035_i = _10024_; _2035_i < _10025_; ++_2035_i)
          {
            {
              ((*_2034_indexnames)(_2035_i)) = ((*_2033_dindexnames)(_2035_i));
            }
          }
          int32_t _10026_ = safe_add( _2003_coneaxis,(int)1 );
          int32_t _10027_ = _1988_nd;
          for (int32_t _2036_i = _10026_; _2036_i < _10027_; ++_2036_i)
          {
            {
              ((*_2034_indexnames)((_2036_i - (int)1))) = ((*_2033_dindexnames)(_2036_i));
            }
          }
          std::vector<int64_t> _10028_;
          for (int32_t _2037_i = (int)0, _10029_ = (int)((_2017_accidxs)->size(0)); _2037_i < _10029_ ; ++_2037_i)
          {
            _10028_.push_back(_ptr__2017_accidxs[_2037_i]);
          }
          auto _10030_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10028_.size()),_10028_.begin(),_10028_.end()));
          task_format_acc_names(_10030_,_2031_sb->toString(),_2028_accshape,_2034_indexnames);
        }
      }
      else
      {
        {
          std::vector<int64_t> _10031_;
          for (int32_t _2038_i = (int)0, _10032_ = (int)((_2017_accidxs)->size(0)); _2038_i < _10032_ ; ++_2038_i)
          {
            _10031_.push_back(_ptr__2017_accidxs[_2038_i]);
          }
          auto _10033_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10031_.size()),_10031_.begin(),_10031_.end()));
          task_format_acc_names(_10033_,_2031_sb->toString(),_2028_accshape,std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ConicConstraint > _2039_res = ::mosek::fusion::p_ConicConstraint::_new_ConicConstraint(::mosek::fusion::Model::t(_pubthis),_1981_name,_2002_dom,_2000_shape,_2018_conid,_2016_afeidxs,mosek::fusion::p_ConeDomain::_get_impl(_2002_dom)->indexnames);
  ((*acons)(_2018_conid)) = (_2039_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_1981_name).size()) > (int)0))
  {
    {
      con_map->setItem(_1981_name,(-safe_add( _2018_conid,(int)1 )));
    }
  }
  {}
  return (_2039_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.constraint_
// Method mosek.fusion.Model.constraint_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:704:6-826:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__constraint_(const std::string & _2040_name,monty::rc_ptr< ::mosek::fusion::Expression > _2041_expr,monty::rc_ptr< ::mosek::fusion::LinearDomain > _2042_dom_)
{
  if ((((int)((_2040_name).size()) > (int)0) && hasConstraint(_2040_name)))
  {
    {
      throw ::mosek::fusion::NameError(::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer()->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Duplicate constraint name '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2040_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->toString());
    }
  }
  {}
  if (mosek::fusion::p_LinearDomain::_get_impl(_2042_dom_)->cardinal_flag)
  {
    {
      throw ::mosek::fusion::DomainError(std::string ("Integer domain not allowed in a constraint"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::ExprDense > _2043_e = ::mosek::fusion::p_ExprDense::_new_ExprDense((::mosek::fusion::p_ExprOptimizeCode::_new_ExprOptimizeCode((::mosek::fusion::p_ExprCompress::_new_ExprCompress(_2041_expr)).as<::mosek::fusion::Expression>())).as<::mosek::fusion::Expression>());
  _2043_e->eval(rs,ws,xs);
  rs->pop_expr();
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2044_ri32 = mosek::fusion::p_WorkStack::_get_impl(rs)->i32;
  _checked_ptr_1<int32_t> _ptr__2044_ri32(_2044_ri32 ? _2044_ri32->raw() : nullptr,_2044_ri32 ? _2044_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2045_ri64 = mosek::fusion::p_WorkStack::_get_impl(rs)->i64;
  _checked_ptr_1<int64_t> _ptr__2045_ri64(_2045_ri64 ? _2045_ri64->raw() : nullptr,_2045_ri64 ? _2045_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2046_rf64 = mosek::fusion::p_WorkStack::_get_impl(rs)->f64;
  _checked_ptr_1<double> _ptr__2046_rf64(_2046_rf64 ? _2046_rf64->raw() : nullptr,_2046_rf64 ? _2046_rf64->size(0) : 0);
  int32_t _2047_nd = mosek::fusion::p_WorkStack::_get_impl(rs)->nd;
  int32_t _2048_nelem = mosek::fusion::p_WorkStack::_get_impl(rs)->nelem;
  int32_t _2049_nnz = mosek::fusion::p_WorkStack::_get_impl(rs)->nnz;
  bool _2050_hassp = mosek::fusion::p_WorkStack::_get_impl(rs)->hassp;
  int32_t _2051_ncodeatom = mosek::fusion::p_WorkStack::_get_impl(rs)->ncodeatom;
  int32_t _2052_shape_base = mosek::fusion::p_WorkStack::_get_impl(rs)->shape_base;
  int32_t _2053_ptr_base = mosek::fusion::p_WorkStack::_get_impl(rs)->ptr_base;
  int32_t _2054_nidxs_base = mosek::fusion::p_WorkStack::_get_impl(rs)->nidxs_base;
  int32_t _2055_cof_base = mosek::fusion::p_WorkStack::_get_impl(rs)->cof_base;
  int32_t _2056_codeptr = mosek::fusion::p_WorkStack::_get_impl(rs)->codeptr_base;
  int32_t _2057_code = mosek::fusion::p_WorkStack::_get_impl(rs)->code_base;
  int32_t _2058_cconst = mosek::fusion::p_WorkStack::_get_impl(rs)->cconst_base;
  std::vector<int32_t> _10034_;
  for (int32_t _2060_i = (int)0, _10035_ = _2047_nd; _2060_i < _10035_ ; ++_2060_i)
  {
    _10034_.push_back(_ptr__2044_ri32[safe_add( _2052_shape_base,_2060_i )]);
  }
  auto _10036_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10034_.size()),_10034_.begin(),_10034_.end()));
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2059_shape = _10036_;
  _checked_ptr_1<int32_t> _ptr__2059_shape(_2059_shape ? _2059_shape->raw() : nullptr,_2059_shape ? _2059_shape->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::LinearDomain > _2061_dom = mosek::fusion::p_LinearDomain::_get_impl(_2042_dom_.get())->__mosek_2fusion_2LinearDomain__finalize_and_validate(_2059_shape);
  int64_t _2062_domsize = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->domsize;
  std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > > _2063_indexnames = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->indexnames;
  std::shared_ptr< monty::ndarray< double,1 > > _2064_b;
  _checked_ptr_1<double> _ptr__2064_b(_2064_b ? _2064_b->raw() : nullptr,_2064_b ? _2064_b->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2065_nativeidxs = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2062_domsize)));
  _checked_ptr_1<int32_t> _ptr__2065_nativeidxs(_2065_nativeidxs ? _2065_nativeidxs->raw() : nullptr,_2065_nativeidxs ? _2065_nativeidxs->size(0) : 0);
  int32_t _2066_conid = con_allocate(_2065_nativeidxs);
  putarows(_2065_nativeidxs,rs,_2048_nelem,_2049_nnz,_2053_ptr_base,_2054_nidxs_base,_2055_cof_base);
  {
    std::shared_ptr< monty::ndarray< int32_t,2 > > _2067_domsp = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->sparsity;
    bool _2068_domempty = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->empty;
    std::shared_ptr< monty::ndarray< double,1 > > _2069_dombnd = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->bnd;
    _checked_ptr_1<double> _ptr__2069_dombnd(_2069_dombnd ? _2069_dombnd->raw() : nullptr,_2069_dombnd ? _2069_dombnd->size(0) : 0);
    bool _2070_domscalable = mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->scalable;
    {
      if ((NULL == _2069_dombnd.get()))
      {
        {
          _2064_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2065_nativeidxs)->size(0)))));
          _ptr__2064_b.update(_2064_b ? _2064_b->raw() : nullptr, _2064_b ? _2064_b->size(0) : 0);
        }
      }
      else if(_2070_domscalable)
      {
        {
          std::vector<double> _10037_;
          for (int32_t _2071_i = (int)0, _10038_ = (int)((_2065_nativeidxs)->size(0)); _2071_i < _10038_ ; ++_2071_i)
          {
            _10037_.push_back(_ptr__2069_dombnd[(int)0]);
          }
          auto _10039_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10037_.size()),_10037_.begin(),_10037_.end()));
          _2064_b = _10039_;
          _ptr__2064_b.update(_2064_b ? _2064_b->raw() : nullptr, _2064_b ? _2064_b->size(0) : 0);
        }
      }
      else if((NULL != _2067_domsp.get()))
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2072_stride = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2047_nd)));
          _checked_ptr_1<int32_t> _ptr__2072_stride(_2072_stride ? _2072_stride->raw() : nullptr,_2072_stride ? _2072_stride->size(0) : 0);
          _ptr__2072_stride[(_2047_nd - (int)1)] = (int)1;
          int32_t _10040_ = (int)1;
          int32_t _10041_ = _2047_nd;
          for (int32_t _2073_i = _10040_; _2073_i < _10041_; ++_2073_i)
          {
            {
              _ptr__2072_stride[((_2047_nd - _2073_i) - (int)1)] = safe_mul( _ptr__2072_stride[(_2047_nd - _2073_i)],_ptr__2059_shape[(_2047_nd - _2073_i)] );
            }
          }
          _2064_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2065_nativeidxs)->size(0)))));
          _ptr__2064_b.update(_2064_b ? _2064_b->raw() : nullptr, _2064_b ? _2064_b->size(0) : 0);
          int32_t _10042_ = (int)0;
          int32_t _10043_ = (int)((_2067_domsp)->size(0));
          for (int32_t _2074_i = _10042_; _2074_i < _10043_; ++_2074_i)
          {
            {
              int32_t _2075_idx = (int)0;
              int32_t _10044_ = (int)0;
              int32_t _10045_ = _2047_nd;
              for (int32_t _2076_j = _10044_; _2076_j < _10045_; ++_2076_j)
              {
                {
                  _2075_idx += safe_mul( _ptr__2072_stride[_2076_j],((*_2067_domsp)(_2074_i,_2076_j)) );
                }
              }
              _ptr__2064_b[_2075_idx] = _ptr__2069_dombnd[_2074_i];
            }
          }
        }
      }
      else if(_2068_domempty)
      {
        {
          _2064_b = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2065_nativeidxs)->size(0)))));
          _ptr__2064_b.update(_2064_b ? _2064_b->raw() : nullptr, _2064_b ? _2064_b->size(0) : 0);
        }
      }
      else
      {
        {
          std::vector<double> _10046_;
          for (int32_t _2077_i = (int)0, _10047_ = (int)((_2065_nativeidxs)->size(0)); _2077_i < _10047_ ; ++_2077_i)
          {
            _10046_.push_back(_ptr__2069_dombnd[_2077_i]);
          }
          auto _10048_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10046_.size()),_10046_.begin(),_10046_.end()));
          _2064_b = _10048_;
          _ptr__2064_b.update(_2064_b ? _2064_b->raw() : nullptr, _2064_b ? _2064_b->size(0) : 0);
        }
      }
    }
  }
  if ((mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->key == ::mosek::fusion::RelationKey::IsFree))
  {
    {
      task_con_putboundlist_fr(_2065_nativeidxs);
    }
  }
  {}
  if ((mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->key == ::mosek::fusion::RelationKey::LessThan))
  {
    {
      task_con_putboundlist_up(_2065_nativeidxs,_2064_b);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->key == ::mosek::fusion::RelationKey::GreaterThan))
  {
    {
      task_con_putboundlist_lo(_2065_nativeidxs,_2064_b);
    }
  }
  else if((mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->key == ::mosek::fusion::RelationKey::EqualsTo))
  {
    {
      task_con_putboundlist_fx(_2065_nativeidxs,_2064_b);
    }
  }
  {}
  if ((_2051_ncodeatom > (int)0))
  {
    {
      con_blocks->replace_row_code(rs,_2065_nativeidxs,_2053_ptr_base,_2054_nidxs_base,_2056_codeptr,_2057_code,_2058_cconst);
    }
  }
  {}
  if (((int)((_2040_name).size()) > (int)0))
  {
    {
      monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2078_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
      _2078_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_2040_name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["));
      if (((int)((_2059_shape)->size(0)) > (int)0))
      {
        {
          _2078_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("%0"));
        }
      }
      {}
      int32_t _10049_ = (int)1;
      int32_t _10050_ = (int)((_2059_shape)->size(0));
      for (int32_t _2079_i = _10049_; _2079_i < _10050_; ++_2079_i)
      {
        {
          _2078_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",%"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2079_i);
        }
      }
      _2078_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
      if ((NULL != mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->indexnames.get()))
      {
        {
          task_format_con_names(_2065_nativeidxs,_2078_sb->toString(),_2059_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),mosek::fusion::p_LinearDomain::_get_impl(_2061_dom)->indexnames);
        }
      }
      else
      {
        {
          task_format_con_names(_2065_nativeidxs,_2078_sb->toString(),_2059_shape,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< std::string,1 > >,1 > >(nullptr));
        }
      }
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::LinearConstraint > _2080_res = ::mosek::fusion::p_LinearConstraint::_new_LinearConstraint(::mosek::fusion::Model::t(_pubthis),_2040_name,_2066_conid,_2059_shape,_2065_nativeidxs,_2063_indexnames);
  ((*cons)(_2066_conid)) = (_2080_res).as<::mosek::fusion::ModelConstraint>();
  if (((int)((_2040_name).size()) > (int)0))
  {
    {
      con_map->setItem(_2040_name,_2066_conid);
    }
  }
  {}
  return (_2080_res).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.Model.constraint_

// Begin mosek.fusion.Model.getVersion
// Method mosek.fusion.Model.getVersion @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:698:6-700:7
std::string mosek::fusion::Model::getVersion
  ( ) { return mosek::fusion::p_Model::getVersion(); }
std::string mosek::fusion::p_Model::getVersion()
{
  return ::mosek::fusion::p_BaseModel::env_getversion();
}
// End mosek.fusion.Model.getVersion

// Begin mosek.fusion.Model.hasParameter
// Method mosek.fusion.Model.hasParameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:693:6-93
bool mosek::fusion::Model :: hasParameter(const std::string & _2081_name) { return mosek::fusion::p_Model::_get_impl(this)->hasParameter(_2081_name); }
bool mosek::fusion::p_Model::hasParameter(const std::string & _2081_name)
{
  return par_map->hasItem(_2081_name);
}
// End mosek.fusion.Model.hasParameter

// Begin mosek.fusion.Model.hasConstraint
// Method mosek.fusion.Model.hasConstraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:692:6-93
bool mosek::fusion::Model :: hasConstraint(const std::string & _2082_name) { return mosek::fusion::p_Model::_get_impl(this)->hasConstraint(_2082_name); }
bool mosek::fusion::p_Model::hasConstraint(const std::string & _2082_name)
{
  return con_map->hasItem(_2082_name);
}
// End mosek.fusion.Model.hasConstraint

// Begin mosek.fusion.Model.hasVariable
// Method mosek.fusion.Model.hasVariable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:691:6-93
bool mosek::fusion::Model :: hasVariable(const std::string & _2083_name) { return mosek::fusion::p_Model::_get_impl(this)->hasVariable(_2083_name); }
bool mosek::fusion::p_Model::hasVariable(const std::string & _2083_name)
{
  return var_map->hasItem(_2083_name);
}
// End mosek.fusion.Model.hasVariable

// Begin mosek.fusion.Model.getParameter
// Method mosek.fusion.Model.getParameter @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:689:6-149
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model :: __mosek_2fusion_2Model__getParameter(const std::string & _2084_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getParameter(_2084_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::Model::getParameter(const std::string &  _2084_name) { return __mosek_2fusion_2Model__getParameter(_2084_name); }
monty::rc_ptr< ::mosek::fusion::Parameter > mosek::fusion::p_Model::__mosek_2fusion_2Model__getParameter(const std::string & _2084_name)
{
  monty::rc_ptr< ::mosek::fusion::Parameter > _10051_;
  bool _10052_ = par_map->hasItem(_2084_name);
  if (_10052_)
  {
    _10051_ = ((*parameters)(par_map->getItem(_2084_name)));
  }
  else
  {
    _10051_ = nullptr;
  }
  return _10051_;
}
// End mosek.fusion.Model.getParameter

// Begin mosek.fusion.Model.getConstraint
// Method mosek.fusion.Model.getConstraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:688:6-167
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__getConstraint(int32_t _2085_index) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getConstraint(_2085_index); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::getConstraint(int32_t _2085_index) { return __mosek_2fusion_2Model__getConstraint(_2085_index); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__getConstraint(int32_t _2085_index)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2086_res;
  if ((_2085_index >= (int)0))
  {
    {
      _2086_res = (((*cons)(_2085_index))).as<::mosek::fusion::Constraint>();
    }
  }
  else
  {
    {
      _2086_res = (((*acons)((-safe_add( (int)1,_2085_index ))))).as<::mosek::fusion::Constraint>();
    }
  }
  return _2086_res;
}
// End mosek.fusion.Model.getConstraint

// Begin mosek.fusion.Model.getConstraint
// Method mosek.fusion.Model.getConstraint @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:678:6-687:7
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model :: __mosek_2fusion_2Model__getConstraint(const std::string & _2087_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getConstraint(_2087_name); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::Model::getConstraint(const std::string &  _2087_name) { return __mosek_2fusion_2Model__getConstraint(_2087_name); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_Model::__mosek_2fusion_2Model__getConstraint(const std::string & _2087_name)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2088_retval = nullptr;
  if (con_map->hasItem(_2087_name))
  {
    {
      int32_t _2089_idx = con_map->getItem(_2087_name);
      if ((_2089_idx >= (int)0))
      {
        {
          _2088_retval = (((*cons)(_2089_idx))).as<::mosek::fusion::Constraint>();
        }
      }
      else
      {
        {
          _2088_retval = (((*acons)((-safe_add( (int)1,_2089_idx ))))).as<::mosek::fusion::Constraint>();
        }
      }
    }
  }
  {}
  return _2088_retval;
}
// End mosek.fusion.Model.getConstraint

// Begin mosek.fusion.Model.getVariable
// Method mosek.fusion.Model.getVariable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:676:6-86
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__getVariable(int32_t _2090_index) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getVariable(_2090_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::getVariable(int32_t _2090_index) { return __mosek_2fusion_2Model__getVariable(_2090_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__getVariable(int32_t _2090_index)
{
  return (((*vars)(_2090_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Model.getVariable

// Begin mosek.fusion.Model.getVariable
// Method mosek.fusion.Model.getVariable @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:665:6-675:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model :: __mosek_2fusion_2Model__getVariable(const std::string & _2091_name) { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__getVariable(_2091_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Model::getVariable(const std::string &  _2091_name) { return __mosek_2fusion_2Model__getVariable(_2091_name); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Model::__mosek_2fusion_2Model__getVariable(const std::string & _2091_name)
{
  if (var_map->hasItem(_2091_name))
  {
    {
      int32_t _2092_varid = var_map->getItem(_2091_name);
      if ((_2092_varid < (int)0))
      {
        {
          return (((*barvars)((-safe_add( _2092_varid,(int)1 ))))).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          return (((*vars)(_2092_varid))).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
  else
  {
    {
      return nullptr;
    }
  }
}
// End mosek.fusion.Model.getVariable

// Begin mosek.fusion.Model.getName
// Method mosek.fusion.Model.getName @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:663:6-67
std::string mosek::fusion::Model :: getName() { return mosek::fusion::p_Model::_get_impl(this)->getName(); }
std::string mosek::fusion::p_Model::getName()
{
  return model_name;
}
// End mosek.fusion.Model.getName

// Begin mosek.fusion.Model.getParameterValue
// Method mosek.fusion.Model.getParameterValue @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:655:6-660:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_Model::getParameterValue(std::shared_ptr< monty::ndarray< int32_t,1 > > _2093_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2093_idxs(_2093_idxs ? _2093_idxs->raw() : nullptr, _2093_idxs ? _2093_idxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2094_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((_2093_idxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2094_res(_2094_res ? _2094_res->raw() : nullptr,_2094_res ? _2094_res->size(0) : 0);
  int32_t _10053_ = (int)0;
  int32_t _10054_ = (int)((_2093_idxs)->size(0));
  for (int32_t _2095_i = _10053_; _2095_i < _10054_; ++_2095_i)
  {
    {
      _ptr__2094_res[_2095_i] = ((*param_value)(_ptr__2093_idxs[_2095_i]));
    }
  }
  return _2094_res;
}
// End mosek.fusion.Model.getParameterValue

// Begin mosek.fusion.Model.setParameterValue
// Method mosek.fusion.Model.setParameterValue @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:650:6-653:7
void mosek::fusion::p_Model::setParameterValue(std::shared_ptr< monty::ndarray< int32_t,1 > > _2096_idxs,std::shared_ptr< monty::ndarray< double,1 > > _2097_vals)
{
  _checked_ptr_1<int32_t> _ptr__2096_idxs(_2096_idxs ? _2096_idxs->raw() : nullptr, _2096_idxs ? _2096_idxs->size(0) : 0);
  _checked_ptr_1<double> _ptr__2097_vals(_2097_vals ? _2097_vals->raw() : nullptr, _2097_vals ? _2097_vals->size(0) : 0);
  int32_t _10055_ = (int)0;
  int32_t _10056_ = (int)((_2096_idxs)->size(0));
  for (int32_t _2098_i = _10055_; _2098_i < _10056_; ++_2098_i)
  {
    {
      ((*param_value)(_ptr__2096_idxs[_2098_i])) = _ptr__2097_vals[_2098_i];
    }
  }
}
// End mosek.fusion.Model.setParameterValue

// Begin mosek.fusion.Model.clone
// Method mosek.fusion.Model.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Model.mbi:645:6-648:7
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::Model :: __mosek_2fusion_2Model__clone() { return mosek::fusion::p_Model::_get_impl(this)->__mosek_2fusion_2Model__clone(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::Model::clone() { return __mosek_2fusion_2Model__clone(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::p_Model::__mosek_2fusion_2Model__clone()
{
  return ::mosek::fusion::p_Model::_new_Model(::mosek::fusion::Model::t(_pubthis));
}
// End mosek.fusion.Model.clone

void mosek::fusion::p_Model::destroy()
{
  xs.reset();
  ws.reset();
  rs.reset();
  sol_itg.reset();
  sol_bas.reset();
  sol_itr.reset();
  con_map.reset();
  acons.reset();
  cons.reset();
  param_value.reset();
  par_map.reset();
  parameters.reset();
  initsol_xx_flag.reset();
  initsol_xx.reset();
  var_map.reset();
  barvars.reset();
  vars.reset();
  barvar_block_elm_j.reset();
  barvar_block_elm_i.reset();
  barvar_block_elm_barj.reset();
  barvar_block_elm_ptr.reset();
  barvar_block_dim.reset();
  barvar_block_ptr.reset();
  barvar_dim.reset();
  var_elm_acc_ofs.reset();
  var_elm_acc_idx.reset();
  var_block_acc_id.reset();
  var_block_map.reset();
  acon_elm_afe.reset();
  acon_elm_ofs.reset();
  acon_elm_scale.reset();
  acon_elm_accid.reset();
  acon_afe.reset();
  acon_acc.reset();
  acon_block_map.reset();
  acc_block_map.reset();
  obj_blocks.reset();
  afe_blocks.reset();
  con_blocks.reset();
}
void mosek::fusion::Model::destroy() { mosek::fusion::p_Model::_get_impl(this)->destroy(); }
// } class Model
// class BoundInterfaceVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:29:4-85:5
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable(mosek::fusion::p_BoundInterfaceVariable *_impl) : mosek::fusion::SliceVariable(_impl) { /*std::cout << "BoundInterfaceVariable()" << this << std::endl; */ }
mosek::fusion::BoundInterfaceVariable::~BoundInterfaceVariable() {  /* std::cout << "~BoundInterfaceVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_BoundInterfaceVariable::p_BoundInterfaceVariable
  (::mosek::fusion::BoundInterfaceVariable * _pubthis) :     mosek::fusion::p_SliceVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.BoundInterfaceVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:41:6-50:7
mosek::fusion::BoundInterfaceVariable::t mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2099_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2100_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2101_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2102_nativeidxs,
    bool _2103_islower)
{ return new mosek::fusion::BoundInterfaceVariable(_2099_m,_2100_shape,_2101_sparsity,_2102_nativeidxs,_2103_islower); }
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2099_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2100_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2101_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2102_nativeidxs,
    bool _2103_islower) : 
   BoundInterfaceVariable(new mosek::fusion::p_BoundInterfaceVariable(this))
{ mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->_initialize(_2099_m,_2100_shape,_2101_sparsity,_2102_nativeidxs,_2103_islower); }
void mosek::fusion::p_BoundInterfaceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2099_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2100_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2101_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2102_nativeidxs,
    bool _2103_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2100_shape(_2100_shape ? _2100_shape->raw() : nullptr, _2100_shape ? _2100_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2101_sparsity(_2101_sparsity ? _2101_sparsity->raw() : nullptr, _2101_sparsity ? _2101_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2102_nativeidxs(_2102_nativeidxs ? _2102_nativeidxs->raw() : nullptr, _2102_nativeidxs ? _2102_nativeidxs->size(0) : 0);
    mosek::fusion::p_SliceVariable::_initialize(_2099_m,_2100_shape,_2101_sparsity,_2102_nativeidxs);
    mosek::fusion::p_BoundInterfaceVariable::_get_impl(::mosek::fusion::BoundInterfaceVariable::t(_pubthis))->islower = _2103_islower;
  }
}
// mosek.fusion.BoundInterfaceVariable.ctor

// mosek.fusion.BoundInterfaceVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:33:6-39:7
mosek::fusion::BoundInterfaceVariable::t mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2104_v,
    bool _2105_islower)
{ return new mosek::fusion::BoundInterfaceVariable(_2104_v,_2105_islower); }
mosek::fusion::BoundInterfaceVariable::BoundInterfaceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2104_v,
    bool _2105_islower) : 
   BoundInterfaceVariable(new mosek::fusion::p_BoundInterfaceVariable(this))
{ mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->_initialize(_2104_v,_2105_islower); }
void mosek::fusion::p_BoundInterfaceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2104_v,
    bool _2105_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_SliceVariable::_initialize(_2104_v);
    mosek::fusion::p_BoundInterfaceVariable::_get_impl(::mosek::fusion::BoundInterfaceVariable::t(_pubthis))->islower = _2105_islower;
  }
}
// mosek.fusion.BoundInterfaceVariable.ctor

// Begin mosek.fusion.BoundInterfaceVariable.dual
// Method mosek.fusion.BoundInterfaceVariable.dual @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:80:6-84:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BoundInterfaceVariable :: dual() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->dual(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BoundInterfaceVariable::dual()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2106_target = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
  _checked_ptr_1<double> _ptr__2106_target(_2106_target ? _2106_target->raw() : nullptr,_2106_target ? _2106_target->size(0) : 0);
  dual_lu((int)0,_2106_target,islower);
  return _2106_target;
}
// End mosek.fusion.BoundInterfaceVariable.dual

// Begin mosek.fusion.BoundInterfaceVariable.transpose
// Method mosek.fusion.BoundInterfaceVariable.transpose @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:78:6-78
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__transpose() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::transpose() { return __mosek_2fusion_2BoundInterfaceVariable__transpose(); }
/* override: mosek.fusion.BaseVariable.transpose*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__transpose() { return __mosek_2fusion_2BoundInterfaceVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__transpose()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__transpose())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.transpose

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:77:6-112
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2107_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2108_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2109_i2) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2107_i0,_2108_i1,_2109_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2107_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2108_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2109_i2) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2107_i0,_2108_i1,_2109_i2); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2107_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2108_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2109_i2) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2107_i0,_2108_i1,_2109_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2107_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2108_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2109_i2)
{
  _checked_ptr_1<int32_t> _ptr__2107_i0(_2107_i0 ? _2107_i0->raw() : nullptr, _2107_i0 ? _2107_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2108_i1(_2108_i1 ? _2108_i1->raw() : nullptr, _2108_i1 ? _2108_i1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2109_i2(_2109_i2 ? _2109_i2->raw() : nullptr, _2109_i2 ? _2109_i2->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2107_i0,_2108_i1,_2109_i2))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:76:6-98
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2110_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_i1) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2110_i0,_2111_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2110_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_i1) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2110_i0,_2111_i1); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2110_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_i1) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2110_i0,_2111_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2110_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2111_i1)
{
  _checked_ptr_1<int32_t> _ptr__2110_i0(_2110_i0 ? _2110_i0->raw() : nullptr, _2110_i0 ? _2110_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2111_i1(_2111_i1 ? _2111_i1->raw() : nullptr, _2111_i1 ? _2111_i1->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2110_i0,_2111_i1))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:75:6-91
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2112_midxs) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2112_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2112_midxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2112_midxs); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2112_midxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2112_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2112_midxs)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2112_midxs))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.pick
// Method mosek.fusion.BoundInterfaceVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:74:6-88
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2113_idxs) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__pick(_2113_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2113_idxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2113_idxs); }
/* override: mosek.fusion.BaseVariable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2113_idxs) { return __mosek_2fusion_2BoundInterfaceVariable__pick(_2113_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2113_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2113_idxs(_2113_idxs ? _2113_idxs->raw() : nullptr, _2113_idxs ? _2113_idxs->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(_2113_idxs))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.pick

// Begin mosek.fusion.BoundInterfaceVariable.antidiag
// Method mosek.fusion.BoundInterfaceVariable.antidiag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:73:6-93
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__antidiag(int32_t _2114_index) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__antidiag(_2114_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::antidiag(int32_t _2114_index) { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(_2114_index); }
/* override: mosek.fusion.BaseVariable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__antidiag(int32_t _2114_index) { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(_2114_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__antidiag(int32_t _2114_index)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag(_2114_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.antidiag

// Begin mosek.fusion.BoundInterfaceVariable.antidiag
// Method mosek.fusion.BoundInterfaceVariable.antidiag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:72:6-76
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__antidiag() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::antidiag() { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
/* override: mosek.fusion.BaseVariable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__antidiag() { return __mosek_2fusion_2BoundInterfaceVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__antidiag()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.antidiag

// Begin mosek.fusion.BoundInterfaceVariable.diag
// Method mosek.fusion.BoundInterfaceVariable.diag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:71:6-88
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__diag(int32_t _2115_index) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__diag(_2115_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::diag(int32_t _2115_index) { return __mosek_2fusion_2BoundInterfaceVariable__diag(_2115_index); }
/* override: mosek.fusion.BaseVariable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__diag(int32_t _2115_index) { return __mosek_2fusion_2BoundInterfaceVariable__diag(_2115_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__diag(int32_t _2115_index)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag(_2115_index))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.diag

// Begin mosek.fusion.BoundInterfaceVariable.diag
// Method mosek.fusion.BoundInterfaceVariable.diag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:70:6-72
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__diag() { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::diag() { return __mosek_2fusion_2BoundInterfaceVariable__diag(); }
/* override: mosek.fusion.BaseVariable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__diag() { return __mosek_2fusion_2BoundInterfaceVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__diag()
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag())).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.diag

// Begin mosek.fusion.BoundInterfaceVariable.slice
// Method mosek.fusion.BoundInterfaceVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:69:6-113
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2116_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2117_lasta) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__slice(_2116_firsta,_2117_lasta); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2116_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2117_lasta) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2116_firsta,_2117_lasta); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2116_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2117_lasta) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2116_firsta,_2117_lasta); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2116_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2117_lasta)
{
  _checked_ptr_1<int32_t> _ptr__2116_firsta(_2116_firsta ? _2116_firsta->raw() : nullptr, _2116_firsta ? _2116_firsta->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2117_lasta(_2117_lasta ? _2117_lasta->raw() : nullptr, _2117_lasta ? _2117_lasta->size(0) : 0);
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(_2116_firsta,_2117_lasta))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.slice

// Begin mosek.fusion.BoundInterfaceVariable.slice
// Method mosek.fusion.BoundInterfaceVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:68:6-105
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable :: __mosek_2fusion_2BoundInterfaceVariable__slice(int32_t _2118_first,int32_t _2119_last) { return mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->__mosek_2fusion_2BoundInterfaceVariable__slice(_2118_first,_2119_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::slice(int32_t _2118_first,int32_t _2119_last) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2118_first,_2119_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BoundInterfaceVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2118_first,int32_t _2119_last) { return __mosek_2fusion_2BoundInterfaceVariable__slice(_2118_first,_2119_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__slice(int32_t _2118_first,int32_t _2119_last)
{
  return (__mosek_2fusion_2BoundInterfaceVariable__from_(::mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(_2118_first,_2119_last))).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BoundInterfaceVariable.slice

// Begin mosek.fusion.BoundInterfaceVariable.from_
// Method mosek.fusion.BoundInterfaceVariable.from_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceVariable.mbi:52:6-66:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_BoundInterfaceVariable::__mosek_2fusion_2BoundInterfaceVariable__from_(monty::rc_ptr< ::mosek::fusion::Variable > _2120_v)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2121_shape = _pubthis->getShape();
  _checked_ptr_1<int32_t> _ptr__2121_shape(_2121_shape ? _2121_shape->raw() : nullptr,_2121_shape ? _2121_shape->size(0) : 0);
  int32_t _2122_nnz = _2120_v->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2123_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2122_nnz)));
  _checked_ptr_1<int64_t> _ptr__2123_nativeidxs(_2123_nativeidxs ? _2123_nativeidxs->raw() : nullptr,_2123_nativeidxs ? _2123_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2124_sparsity = nullptr;
  _checked_ptr_1<int64_t> _ptr__2124_sparsity(_2124_sparsity ? _2124_sparsity->raw() : nullptr,_2124_sparsity ? _2124_sparsity->size(0) : 0);
  if ((_2120_v->getSize() == _2122_nnz))
  {
    {
      _2120_v->inst((int)0,_2123_nativeidxs);
    }
  }
  else
  {
    {
      _2124_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2122_nnz)));
      _ptr__2124_sparsity.update(_2124_sparsity ? _2124_sparsity->raw() : nullptr, _2124_sparsity ? _2124_sparsity->size(0) : 0);
      _2120_v->inst((int)0,_2124_sparsity,(int)0,_2123_nativeidxs);
    }
  }
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_2120_v->__mosek_2fusion_2Variable__getModel(),_2121_shape,_2124_sparsity,_2123_nativeidxs,islower);
}
// End mosek.fusion.BoundInterfaceVariable.from_

void mosek::fusion::p_BoundInterfaceVariable::destroy()
{
}
void mosek::fusion::BoundInterfaceVariable::destroy() { mosek::fusion::p_BoundInterfaceVariable::_get_impl(this)->destroy(); }
// } class BoundInterfaceVariable
// class SliceVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SliceVariable.mbi:18:4-66:5
mosek::fusion::SliceVariable::SliceVariable(mosek::fusion::p_SliceVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "SliceVariable()" << this << std::endl; */ }
mosek::fusion::SliceVariable::~SliceVariable() {  /* std::cout << "~SliceVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_SliceVariable::p_SliceVariable
  (::mosek::fusion::SliceVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.SliceVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SliceVariable.mbi:54:6-65:7
mosek::fusion::SliceVariable::t mosek::fusion::p_SliceVariable::_new_SliceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2125_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2126_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2127_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2128_nativeidxs)
{ return new mosek::fusion::SliceVariable(_2125_m,_2126_shape,_2127_sparsity,_2128_nativeidxs); }
mosek::fusion::SliceVariable::SliceVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2125_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2126_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2127_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2128_nativeidxs) : 
   SliceVariable(new mosek::fusion::p_SliceVariable(this))
{ mosek::fusion::p_SliceVariable::_get_impl(this)->_initialize(_2125_m,_2126_shape,_2127_sparsity,_2128_nativeidxs); }
void mosek::fusion::p_SliceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2125_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2126_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2127_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2128_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2126_shape(_2126_shape ? _2126_shape->raw() : nullptr, _2126_shape ? _2126_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2127_sparsity(_2127_sparsity ? _2127_sparsity->raw() : nullptr, _2127_sparsity ? _2127_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2128_nativeidxs(_2128_nativeidxs ? _2128_nativeidxs->raw() : nullptr, _2128_nativeidxs ? _2128_nativeidxs->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(_2125_m,_2126_shape,_2127_sparsity,_2128_nativeidxs);
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->nativeidxs = _2128_nativeidxs;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->sparsity = _2127_sparsity;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->shape = _2126_shape;
  }
}
// mosek.fusion.SliceVariable.ctor

// mosek.fusion.SliceVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SliceVariable.mbi:47:6-52:7
mosek::fusion::SliceVariable::t mosek::fusion::p_SliceVariable::_new_SliceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2129_v)
{ return new mosek::fusion::SliceVariable(_2129_v); }
mosek::fusion::SliceVariable::SliceVariable
  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2129_v) : 
   SliceVariable(new mosek::fusion::p_SliceVariable(this))
{ mosek::fusion::p_SliceVariable::_get_impl(this)->_initialize(_2129_v); }
void mosek::fusion::p_SliceVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::SliceVariable > _2129_v)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseVariable::_initialize(_2129_v,_2129_v->__mosek_2fusion_2BaseVariable__getModel());
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->nativeidxs = mosek::fusion::p_SliceVariable::_get_impl(_2129_v)->nativeidxs;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->sparsity = mosek::fusion::p_SliceVariable::_get_impl(_2129_v)->sparsity;
    mosek::fusion::p_SliceVariable::_get_impl(::mosek::fusion::SliceVariable::t(_pubthis))->shape = mosek::fusion::p_SliceVariable::_get_impl(_2129_v)->shape;
  }
}
// mosek.fusion.SliceVariable.ctor

void mosek::fusion::p_SliceVariable::destroy()
{
  shape.reset();
  sparsity.reset();
  nativeidxs.reset();
}
void mosek::fusion::SliceVariable::destroy() { mosek::fusion::p_SliceVariable::_get_impl(this)->destroy(); }
// } class SliceVariable
// class RangedVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:15:4-125:5
mosek::fusion::RangedVariable::RangedVariable(mosek::fusion::p_RangedVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "RangedVariable()" << this << std::endl; */ }
mosek::fusion::RangedVariable::~RangedVariable() {  /* std::cout << "~RangedVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_RangedVariable::p_RangedVariable
  (::mosek::fusion::RangedVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.RangedVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:55:6-64:7
mosek::fusion::RangedVariable::t mosek::fusion::p_RangedVariable::_new_RangedVariable
  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2130_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2131_m)
{ return new mosek::fusion::RangedVariable(_2130_v,_2131_m); }
mosek::fusion::RangedVariable::RangedVariable
  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2130_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2131_m) : 
   RangedVariable(new mosek::fusion::p_RangedVariable(this))
{ mosek::fusion::p_RangedVariable::_get_impl(this)->_initialize(_2130_v,_2131_m); }
void mosek::fusion::p_RangedVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::RangedVariable > _2130_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2131_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2130_v,_2131_m);
    name = mosek::fusion::p_RangedVariable::_get_impl(_2130_v)->name;
    nativeidxs = mosek::fusion::p_RangedVariable::_get_impl(_2130_v)->nativeidxs;
    sparsity = mosek::fusion::p_RangedVariable::_get_impl(_2130_v)->sparsity;
    names_flushed = mosek::fusion::p_RangedVariable::_get_impl(_2130_v)->names_flushed;
    shape = mosek::fusion::p_RangedVariable::_get_impl(_2130_v)->shape;
  }
}
// mosek.fusion.RangedVariable.ctor

// mosek.fusion.RangedVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:39:6-53:7
mosek::fusion::RangedVariable::t mosek::fusion::p_RangedVariable::_new_RangedVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2132_model,
    const std::string &  _2133_name,
    int64_t _2134_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2135_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2136_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_nativeidxs)
{ return new mosek::fusion::RangedVariable(_2132_model,_2133_name,_2134_varid,_2135_shape,_2136_sparsity,_2137_nativeidxs); }
mosek::fusion::RangedVariable::RangedVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2132_model,
    const std::string &  _2133_name,
    int64_t _2134_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2135_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2136_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_nativeidxs) : 
   RangedVariable(new mosek::fusion::p_RangedVariable(this))
{ mosek::fusion::p_RangedVariable::_get_impl(this)->_initialize(_2132_model,_2133_name,_2134_varid,_2135_shape,_2136_sparsity,_2137_nativeidxs); }
void mosek::fusion::p_RangedVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2132_model,
    const std::string &  _2133_name,
    int64_t _2134_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2135_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2136_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2137_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2135_shape(_2135_shape ? _2135_shape->raw() : nullptr, _2135_shape ? _2135_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2136_sparsity(_2136_sparsity ? _2136_sparsity->raw() : nullptr, _2136_sparsity ? _2136_sparsity->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2137_nativeidxs(_2137_nativeidxs ? _2137_nativeidxs->raw() : nullptr, _2137_nativeidxs ? _2137_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2132_model,_2133_name,_2135_shape,_2134_varid,_2136_sparsity,::mosek::fusion::p_RangedVariable::globalNativeIndexes(_2137_nativeidxs));
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->nativeidxs = _2137_nativeidxs;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->sparsity = _2136_sparsity;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->name = _2133_name;
    mosek::fusion::p_RangedVariable::_get_impl(::mosek::fusion::RangedVariable::t(_pubthis))->shape = _2135_shape;
    names_flushed = false;
  }
}
// mosek.fusion.RangedVariable.ctor

// Begin mosek.fusion.RangedVariable.elementDesc
// Method mosek.fusion.RangedVariable.elementDesc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:119:6-124:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__elementDesc(int64_t _2138_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2139_sb) { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__elementDesc(_2138_index,_2139_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::RangedVariable::elementDesc(int64_t _2138_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2139_sb) { return __mosek_2fusion_2RangedVariable__elementDesc(_2138_index,_2139_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__elementDesc(int64_t _2138_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2139_sb)
{
  _pubthis->elementName(_2138_index,_2139_sb);
  _2139_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" : in range"));
  return _2139_sb;
}
// End mosek.fusion.RangedVariable.elementDesc

// Begin mosek.fusion.RangedVariable.flushNames
// Method mosek.fusion.RangedVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:98:6-116:7
void mosek::fusion::p_RangedVariable::flushNames()
{
}
// End mosek.fusion.RangedVariable.flushNames

// Begin mosek.fusion.RangedVariable.dual_u
// Method mosek.fusion.RangedVariable.dual_u @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:91:6-96:7
void mosek::fusion::p_RangedVariable::dual_u(int32_t _2140_offset,std::shared_ptr< monty::ndarray< double,1 > > _2141_target)
{
  _checked_ptr_1<double> _ptr__2141_target(_2141_target ? _2141_target->raw() : nullptr, _2141_target ? _2141_target->size(0) : 0);
  dual_lu(_2140_offset,_2141_target,false);
}
// End mosek.fusion.RangedVariable.dual_u

// Begin mosek.fusion.RangedVariable.dual_l
// Method mosek.fusion.RangedVariable.dual_l @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:83:6-88:7
void mosek::fusion::p_RangedVariable::dual_l(int32_t _2142_offset,std::shared_ptr< monty::ndarray< double,1 > > _2143_target)
{
  _checked_ptr_1<double> _ptr__2143_target(_2143_target ? _2143_target->raw() : nullptr, _2143_target ? _2143_target->size(0) : 0);
  dual_lu(_2142_offset,_2143_target,true);
}
// End mosek.fusion.RangedVariable.dual_l

// Begin mosek.fusion.RangedVariable.upperBoundVar
// Method mosek.fusion.RangedVariable.upperBoundVar @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:79:6-81:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__upperBoundVar() { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__upperBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable::upperBoundVar() { return __mosek_2fusion_2RangedVariable__upperBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__upperBoundVar()
{
  std::vector<int64_t> _10057_;
  for (int32_t _2144_i = (int)0, _10058_ = (int)((nativeidxs)->size(0)); _2144_i < _10058_ ; ++_2144_i)
  {
    _10057_.push_back(((*nativeidxs)(_2144_i)));
  }
  auto _10059_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10057_.size()),_10057_.begin(),_10057_.end()));
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),shape,sparsity,_10059_,false);
}
// End mosek.fusion.RangedVariable.upperBoundVar

// Begin mosek.fusion.RangedVariable.lowerBoundVar
// Method mosek.fusion.RangedVariable.lowerBoundVar @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:73:6-75:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable :: __mosek_2fusion_2RangedVariable__lowerBoundVar() { return mosek::fusion::p_RangedVariable::_get_impl(this)->__mosek_2fusion_2RangedVariable__lowerBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::RangedVariable::lowerBoundVar() { return __mosek_2fusion_2RangedVariable__lowerBoundVar(); }
monty::rc_ptr< ::mosek::fusion::BoundInterfaceVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__lowerBoundVar()
{
  std::vector<int64_t> _10060_;
  for (int32_t _2145_i = (int)0, _10061_ = (int)((nativeidxs)->size(0)); _2145_i < _10061_ ; ++_2145_i)
  {
    _10060_.push_back(((*nativeidxs)(_2145_i)));
  }
  auto _10062_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10060_.size()),_10060_.begin(),_10060_.end()));
  return ::mosek::fusion::p_BoundInterfaceVariable::_new_BoundInterfaceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),shape,sparsity,_10062_,true);
}
// End mosek.fusion.RangedVariable.lowerBoundVar

// Begin mosek.fusion.RangedVariable.clone
// Method mosek.fusion.RangedVariable.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:66:6-68:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_RangedVariable::__mosek_2fusion_2RangedVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2146_m)
{
  return (::mosek::fusion::p_RangedVariable::_new_RangedVariable(::mosek::fusion::RangedVariable::t(_pubthis),_2146_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.RangedVariable.clone

// Begin mosek.fusion.RangedVariable.globalNativeIndexes
// Method mosek.fusion.RangedVariable.globalNativeIndexes @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/RangedVariable.mbi:25:6-28:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_RangedVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2147_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2147_nativeidxs(_2147_nativeidxs ? _2147_nativeidxs->raw() : nullptr, _2147_nativeidxs ? _2147_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2148_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2147_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2148_r(_2148_r ? _2148_r->raw() : nullptr,_2148_r ? _2148_r->size(0) : 0);
  int32_t _10063_ = (int)0;
  int32_t _10064_ = (int)((_2147_nativeidxs)->size(0));
  for (int32_t _2149_i = _10063_; _2149_i < _10064_; ++_2149_i)
  {
    {
      _ptr__2148_r[_2149_i] = _ptr__2147_nativeidxs[_2149_i];
    }
  }
  return _2148_r;
}
// End mosek.fusion.RangedVariable.globalNativeIndexes

void mosek::fusion::p_RangedVariable::destroy()
{
  shape.reset();
  nativeidxs.reset();
  sparsity.reset();
}
void mosek::fusion::RangedVariable::destroy() { mosek::fusion::p_RangedVariable::_get_impl(this)->destroy(); }
// } class RangedVariable
// class LinearPSDVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:8:4-73:5
mosek::fusion::LinearPSDVariable::LinearPSDVariable(mosek::fusion::p_LinearPSDVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "LinearPSDVariable()" << this << std::endl; */ }
mosek::fusion::LinearPSDVariable::~LinearPSDVariable() {  /* std::cout << "~LinearPSDVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_LinearPSDVariable::p_LinearPSDVariable
  (::mosek::fusion::LinearPSDVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.LinearPSDVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:35:6-43:7
mosek::fusion::LinearPSDVariable::t mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2150_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2151_m)
{ return new mosek::fusion::LinearPSDVariable(_2150_v,_2151_m); }
mosek::fusion::LinearPSDVariable::LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2150_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2151_m) : 
   LinearPSDVariable(new mosek::fusion::p_LinearPSDVariable(this))
{ mosek::fusion::p_LinearPSDVariable::_get_impl(this)->_initialize(_2150_v,_2151_m); }
void mosek::fusion::p_LinearPSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::LinearPSDVariable > _2150_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2151_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2150_v,_2151_m);
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->conedim = mosek::fusion::p_LinearPSDVariable::_get_impl(_2150_v)->conedim;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->varid = mosek::fusion::p_LinearPSDVariable::_get_impl(_2150_v)->varid;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->name = mosek::fusion::p_LinearPSDVariable::_get_impl(_2150_v)->name;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->shape = mosek::fusion::p_LinearPSDVariable::_get_impl(_2150_v)->shape;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->nativeidxs = mosek::fusion::p_LinearPSDVariable::_get_impl(_2150_v)->nativeidxs;
  }
}
// mosek.fusion.LinearPSDVariable.ctor

// mosek.fusion.LinearPSDVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:19:6-33:7
mosek::fusion::LinearPSDVariable::t mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2152_model,
    const std::string &  _2153_name,
    int32_t _2154_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2155_shape,
    int32_t _2156_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2157_nativeidxs)
{ return new mosek::fusion::LinearPSDVariable(_2152_model,_2153_name,_2154_varid,_2155_shape,_2156_conedim,_2157_nativeidxs); }
mosek::fusion::LinearPSDVariable::LinearPSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2152_model,
    const std::string &  _2153_name,
    int32_t _2154_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2155_shape,
    int32_t _2156_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2157_nativeidxs) : 
   LinearPSDVariable(new mosek::fusion::p_LinearPSDVariable(this))
{ mosek::fusion::p_LinearPSDVariable::_get_impl(this)->_initialize(_2152_model,_2153_name,_2154_varid,_2155_shape,_2156_conedim,_2157_nativeidxs); }
void mosek::fusion::p_LinearPSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2152_model,
    const std::string &  _2153_name,
    int32_t _2154_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2155_shape,
    int32_t _2156_conedim,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2157_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2155_shape(_2155_shape ? _2155_shape->raw() : nullptr, _2155_shape ? _2155_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2157_nativeidxs(_2157_nativeidxs ? _2157_nativeidxs->raw() : nullptr, _2157_nativeidxs ? _2157_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2152_model,_2153_name,_2155_shape,_2154_varid,nullptr,::mosek::fusion::p_LinearPSDVariable::globalNativeIndexes(_2157_nativeidxs));
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->name = _2153_name;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->conedim = _2156_conedim;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->varid = _2154_varid;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->shape = _2155_shape;
    mosek::fusion::p_LinearPSDVariable::_get_impl(::mosek::fusion::LinearPSDVariable::t(_pubthis))->nativeidxs = _2157_nativeidxs;
  }
}
// mosek.fusion.LinearPSDVariable.ctor

// Begin mosek.fusion.LinearPSDVariable.flushNames
// Method mosek.fusion.LinearPSDVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:71:6-38
void mosek::fusion::p_LinearPSDVariable::flushNames()
{
}
// End mosek.fusion.LinearPSDVariable.flushNames

// Begin mosek.fusion.LinearPSDVariable.toString
// Method mosek.fusion.LinearPSDVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:60:6-69:7
std::string mosek::fusion::LinearPSDVariable :: toString() { return mosek::fusion::p_LinearPSDVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_LinearPSDVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2158_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2158_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PSDVariable("));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2158_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2158_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10065_ = (int)1;
  int32_t _10066_ = (int)((shape)->size(0));
  for (int32_t _2159_i = _10065_; _2159_i < _10066_; ++_2159_i)
  {
    {
      _2158_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2159_i)));
    }
  }
  _2158_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2158_sb->toString();
}
// End mosek.fusion.LinearPSDVariable.toString

// Begin mosek.fusion.LinearPSDVariable.make_continuous
// Method mosek.fusion.LinearPSDVariable.make_continuous @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:56:6-58:7
void mosek::fusion::LinearPSDVariable :: make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _2160_idxs) { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->make_continuous(_2160_idxs); }
void mosek::fusion::p_LinearPSDVariable::make_continuous(std::shared_ptr< monty::ndarray< int64_t,1 > > _2160_idxs)
{
  _checked_ptr_1<int64_t> _ptr__2160_idxs(_2160_idxs ? _2160_idxs->raw() : nullptr, _2160_idxs ? _2160_idxs->size(0) : 0);
}
// End mosek.fusion.LinearPSDVariable.make_continuous

// Begin mosek.fusion.LinearPSDVariable.make_integer
// Method mosek.fusion.LinearPSDVariable.make_integer @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:51:6-54:7
void mosek::fusion::LinearPSDVariable :: make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _2161_idxs) { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->make_integer(_2161_idxs); }
void mosek::fusion::p_LinearPSDVariable::make_integer(std::shared_ptr< monty::ndarray< int64_t,1 > > _2161_idxs)
{
  _checked_ptr_1<int64_t> _ptr__2161_idxs(_2161_idxs ? _2161_idxs->raw() : nullptr, _2161_idxs ? _2161_idxs->size(0) : 0);
  throw ::mosek::fusion::UnimplementedError(std::string ("Cannot make PSDVariable elements integer"));
}
// End mosek.fusion.LinearPSDVariable.make_integer

// Begin mosek.fusion.LinearPSDVariable.clone
// Method mosek.fusion.LinearPSDVariable.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:45:6-48:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_LinearPSDVariable::__mosek_2fusion_2LinearPSDVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2162_m)
{
  return (::mosek::fusion::p_LinearPSDVariable::_new_LinearPSDVariable(::mosek::fusion::LinearPSDVariable::t(_pubthis),_2162_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.LinearPSDVariable.clone

// Begin mosek.fusion.LinearPSDVariable.globalNativeIndexes
// Method mosek.fusion.LinearPSDVariable.globalNativeIndexes @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDVariable.monty:17:6-147
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_LinearPSDVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int64_t,1 > > _2163_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2163_nativeidxs(_2163_nativeidxs ? _2163_nativeidxs->raw() : nullptr, _2163_nativeidxs ? _2163_nativeidxs->size(0) : 0);
  std::vector<int64_t> _10067_;
  for (int32_t _2164_i = (int)0, _10068_ = (int)((_2163_nativeidxs)->size(0)); _2164_i < _10068_ ; ++_2164_i)
  {
    _10067_.push_back((-safe_add( _ptr__2163_nativeidxs[_2164_i],(int)1 )));
  }
  auto _10069_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10067_.size()),_10067_.begin(),_10067_.end()));
  return _10069_;
}
// End mosek.fusion.LinearPSDVariable.globalNativeIndexes

void mosek::fusion::p_LinearPSDVariable::destroy()
{
  shape.reset();
  nativeidxs.reset();
}
void mosek::fusion::LinearPSDVariable::destroy() { mosek::fusion::p_LinearPSDVariable::_get_impl(this)->destroy(); }
// } class LinearPSDVariable
// class PSDVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:8:4-139:5
mosek::fusion::PSDVariable::PSDVariable(mosek::fusion::p_PSDVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "PSDVariable()" << this << std::endl; */ }
mosek::fusion::PSDVariable::~PSDVariable() {  /* std::cout << "~PSDVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_PSDVariable::p_PSDVariable
  (::mosek::fusion::PSDVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.PSDVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:89:6-102:7
mosek::fusion::PSDVariable::t mosek::fusion::p_PSDVariable::_new_PSDVariable
  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2165_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2166_m)
{ return new mosek::fusion::PSDVariable(_2165_v,_2166_m); }
mosek::fusion::PSDVariable::PSDVariable
  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2165_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2166_m) : 
   PSDVariable(new mosek::fusion::p_PSDVariable(this))
{ mosek::fusion::p_PSDVariable::_get_impl(this)->_initialize(_2165_v,_2166_m); }
void mosek::fusion::p_PSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::PSDVariable > _2165_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2166_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2165_v,_2166_m);
    varid = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->varid;
    nativeidxs = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->nativeidxs;
    name = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->name;
    shape = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->shape;
    conedim1 = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->conedim1;
    conedim2 = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->conedim2;
    names_flushed = false;
    model = _2166_m;
    barvaridxs = mosek::fusion::p_PSDVariable::_get_impl(_2165_v)->barvaridxs;
  }
}
// mosek.fusion.PSDVariable.ctor

// mosek.fusion.PSDVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:66:6-87:7
mosek::fusion::PSDVariable::t mosek::fusion::p_PSDVariable::_new_PSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2167_model,
    const std::string &  _2168_name,
    int32_t _2169_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2170_shape,
    int32_t _2171_conedim1,
    int32_t _2172_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2173_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2174_nativeidxs)
{ return new mosek::fusion::PSDVariable(_2167_model,_2168_name,_2169_varid,_2170_shape,_2171_conedim1,_2172_conedim2,_2173_barvaridxs,_2174_nativeidxs); }
mosek::fusion::PSDVariable::PSDVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2167_model,
    const std::string &  _2168_name,
    int32_t _2169_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2170_shape,
    int32_t _2171_conedim1,
    int32_t _2172_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2173_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2174_nativeidxs) : 
   PSDVariable(new mosek::fusion::p_PSDVariable(this))
{ mosek::fusion::p_PSDVariable::_get_impl(this)->_initialize(_2167_model,_2168_name,_2169_varid,_2170_shape,_2171_conedim1,_2172_conedim2,_2173_barvaridxs,_2174_nativeidxs); }
void mosek::fusion::p_PSDVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2167_model,
    const std::string &  _2168_name,
    int32_t _2169_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2170_shape,
    int32_t _2171_conedim1,
    int32_t _2172_conedim2,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2173_barvaridxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2174_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2170_shape(_2170_shape ? _2170_shape->raw() : nullptr, _2170_shape ? _2170_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2173_barvaridxs(_2173_barvaridxs ? _2173_barvaridxs->raw() : nullptr, _2173_barvaridxs ? _2173_barvaridxs->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2174_nativeidxs(_2174_nativeidxs ? _2174_nativeidxs->raw() : nullptr, _2174_nativeidxs ? _2174_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2167_model,_2168_name,_2170_shape,_2169_varid,nullptr,::mosek::fusion::p_PSDVariable::fullnativeidxs(_2170_shape,_2171_conedim1,_2172_conedim2,_2174_nativeidxs));
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->varid = _2169_varid;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->nativeidxs = _2174_nativeidxs;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->name = _2168_name;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->shape = _2170_shape;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->conedim1 = _2171_conedim1;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->conedim2 = _2172_conedim2;
    names_flushed = false;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->model = _2167_model;
    mosek::fusion::p_PSDVariable::_get_impl(::mosek::fusion::PSDVariable::t(_pubthis))->barvaridxs = _2173_barvaridxs;
  }
}
// mosek.fusion.PSDVariable.ctor

// Begin mosek.fusion.PSDVariable.flushNames
// Method mosek.fusion.PSDVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:130:6-137:7
void mosek::fusion::p_PSDVariable::flushNames()
{
}
// End mosek.fusion.PSDVariable.flushNames

// Begin mosek.fusion.PSDVariable.toString
// Method mosek.fusion.PSDVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:119:6-128:7
std::string mosek::fusion::PSDVariable :: toString() { return mosek::fusion::p_PSDVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_PSDVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2175_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2175_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("PSDVariable("));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2175_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2175_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10070_ = (int)1;
  int32_t _10071_ = (int)((shape)->size(0));
  for (int32_t _2176_i = _10070_; _2176_i < _10071_; ++_2176_i)
  {
    {
      _2175_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2176_i)));
    }
  }
  _2175_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2175_sb->toString();
}
// End mosek.fusion.PSDVariable.toString

// Begin mosek.fusion.PSDVariable.elementDesc
// Method mosek.fusion.PSDVariable.elementDesc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:111:6-116:7
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::PSDVariable :: __mosek_2fusion_2PSDVariable__elementDesc(int64_t _2177_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2178_sb) { return mosek::fusion::p_PSDVariable::_get_impl(this)->__mosek_2fusion_2PSDVariable__elementDesc(_2177_index,_2178_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::PSDVariable::elementDesc(int64_t _2177_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2178_sb) { return __mosek_2fusion_2PSDVariable__elementDesc(_2177_index,_2178_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_PSDVariable::__mosek_2fusion_2PSDVariable__elementDesc(int64_t _2177_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2178_sb)
{
  _pubthis->elementName(_2177_index,_2178_sb);
  _2178_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (" : element in positive semidefinite cone"));
  return _2178_sb;
}
// End mosek.fusion.PSDVariable.elementDesc

// Begin mosek.fusion.PSDVariable.clone
// Method mosek.fusion.PSDVariable.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:104:6-107:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_PSDVariable::__mosek_2fusion_2PSDVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2179_m)
{
  return (::mosek::fusion::p_PSDVariable::_new_PSDVariable(::mosek::fusion::PSDVariable::t(_pubthis),_2179_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.PSDVariable.clone

// Begin mosek.fusion.PSDVariable.fullnativeidxs
// Method mosek.fusion.PSDVariable.fullnativeidxs @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/SemidefiniteVariable.mbi:23:6-59:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_PSDVariable::fullnativeidxs(std::shared_ptr< monty::ndarray< int32_t,1 > > _2180_shape,int32_t _2181_conedim1,int32_t _2182_conedim2,std::shared_ptr< monty::ndarray< int64_t,1 > > _2183_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2180_shape(_2180_shape ? _2180_shape->raw() : nullptr, _2180_shape ? _2180_shape->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2183_nativeidxs(_2183_nativeidxs ? _2183_nativeidxs->raw() : nullptr, _2183_nativeidxs ? _2183_nativeidxs->size(0) : 0);
  int32_t _2184_nd = (int)((_2180_shape)->size(0));
  bool _2185_tril = (_2181_conedim1 < _2182_conedim2);
  int32_t _10072_;
  bool _10073_ = _2185_tril;
  if (_10073_)
  {
    _10072_ = _2181_conedim1;
  }
  else
  {
    _10072_ = _2182_conedim2;
  }
  int32_t _2186_cd1 = _10072_;
  int32_t _10074_;
  bool _10075_ = (!_2185_tril);
  if (_10075_)
  {
    _10074_ = _2181_conedim1;
  }
  else
  {
    _10074_ = _2182_conedim2;
  }
  int32_t _2187_cd2 = _10074_;
  int32_t _2188_d0 = (int)1;
  int32_t _10076_ = (int)0;
  int32_t _10077_ = _2186_cd1;
  for (int32_t _2189_i = _10076_; _2189_i < _10077_; ++_2189_i)
  {
    {
      _2188_d0 *= _ptr__2180_shape[_2189_i];
    }
  }
  int32_t _2190_d1 = _ptr__2180_shape[_2186_cd1];
  int32_t _2191_d2 = (int)1;
  int32_t _10078_ = safe_add( _2186_cd1,(int)1 );
  int32_t _10079_ = _2187_cd2;
  for (int32_t _2192_i = _10078_; _2192_i < _10079_; ++_2192_i)
  {
    {
      _2191_d2 *= _ptr__2180_shape[_2192_i];
    }
  }
  int32_t _2193_d3 = _ptr__2180_shape[_2187_cd2];
  int32_t _2194_d4 = (int)1;
  int32_t _10080_ = safe_add( _2187_cd2,(int)1 );
  int32_t _10081_ = _2184_nd;
  for (int32_t _2195_i = _10080_; _2195_i < _10081_; ++_2195_i)
  {
    {
      _2194_d4 *= _ptr__2180_shape[_2195_i];
    }
  }
  int32_t _2196_size = safe_mul( safe_mul( safe_mul( safe_mul( _2188_d0,_2190_d1 ),_2191_d2 ),_2193_d3 ),_2194_d4 );
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2197_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2196_size)));
  _checked_ptr_1<int64_t> _ptr__2197_residxs(_2197_residxs ? _2197_residxs->raw() : nullptr,_2197_residxs ? _2197_residxs->size(0) : 0);
  int32_t _2198_s3 = _2194_d4;
  int32_t _2199_s2 = safe_mul( _2198_s3,_2193_d3 );
  int32_t _2200_s1 = safe_mul( _2199_s2,_2191_d2 );
  int32_t _2201_s0 = safe_mul( _2200_s1,_2200_s1 );
  {
    int32_t _2202_k = (int)0;
    int32_t _10082_ = (int)0;
    int32_t _10083_ = _2188_d0;
    for (int32_t _2203_i0 = _10082_; _2203_i0 < _10083_; ++_2203_i0)
    {
      {
        int32_t _10084_ = (int)0;
        int32_t _10085_ = _2193_d3;
        for (int32_t _2204_i3 = _10084_; _2204_i3 < _10085_; ++_2204_i3)
        {
          {
            int32_t _10086_ = (int)0;
            int32_t _10087_ = _2191_d2;
            for (int32_t _2205_i2 = _10086_; _2205_i2 < _10087_; ++_2205_i2)
            {
              {
                int32_t _10088_ = _2204_i3;
                int32_t _10089_ = _2190_d1;
                for (int32_t _2206_i1 = _10088_; _2206_i1 < _10089_; ++_2206_i1)
                {
                  {
                    int32_t _10090_ = (int)0;
                    int32_t _10091_ = _2194_d4;
                    for (int32_t _2207_i4 = _10090_; _2207_i4 < _10091_; ++_2207_i4)
                    {
                      {
                        _ptr__2197_residxs[safe_add( safe_add( safe_add( safe_add( safe_mul( _2203_i0,_2201_s0 ),safe_mul( _2206_i1,_2200_s1 ) ),safe_mul( _2205_i2,_2199_s2 ) ),safe_mul( _2204_i3,_2198_s3 ) ),_2207_i4 )] = (-safe_add( _ptr__2183_nativeidxs[_2202_k],(int)1 ));
                        _ptr__2197_residxs[safe_add( safe_add( safe_add( safe_add( safe_mul( _2203_i0,_2201_s0 ),safe_mul( _2204_i3,_2200_s1 ) ),safe_mul( _2205_i2,_2199_s2 ) ),safe_mul( _2206_i1,_2198_s3 ) ),_2207_i4 )] = (-safe_add( _ptr__2183_nativeidxs[_2202_k],(int)1 ));
                        ++ _2202_k;
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  return _2197_residxs;
}
// End mosek.fusion.PSDVariable.fullnativeidxs

void mosek::fusion::p_PSDVariable::destroy()
{
  model.reset();
  barvaridxs.reset();
  shape.reset();
  nativeidxs.reset();
}
void mosek::fusion::PSDVariable::destroy() { mosek::fusion::p_PSDVariable::_get_impl(this)->destroy(); }
// } class PSDVariable
// class LinearVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:21:4-111:5
mosek::fusion::LinearVariable::LinearVariable(mosek::fusion::p_LinearVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "LinearVariable()" << this << std::endl; */ }
mosek::fusion::LinearVariable::~LinearVariable() {  /* std::cout << "~LinearVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_LinearVariable::p_LinearVariable
  (::mosek::fusion::LinearVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.LinearVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:53:6-62:7
mosek::fusion::LinearVariable::t mosek::fusion::p_LinearVariable::_new_LinearVariable
  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2208_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2209_m)
{ return new mosek::fusion::LinearVariable(_2208_v,_2209_m); }
mosek::fusion::LinearVariable::LinearVariable
  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2208_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2209_m) : 
   LinearVariable(new mosek::fusion::p_LinearVariable(this))
{ mosek::fusion::p_LinearVariable::_get_impl(this)->_initialize(_2208_v,_2209_m); }
void mosek::fusion::p_LinearVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::LinearVariable > _2208_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2209_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2208_v,_2209_m);
    names_flushed = mosek::fusion::p_LinearVariable::_get_impl(_2208_v)->names_flushed;
    nativeidxs = mosek::fusion::p_LinearVariable::_get_impl(_2208_v)->nativeidxs;
    sparsity = mosek::fusion::p_LinearVariable::_get_impl(_2208_v)->sparsity;
    shape = mosek::fusion::p_LinearVariable::_get_impl(_2208_v)->shape;
    name = mosek::fusion::p_LinearVariable::_get_impl(_2208_v)->name;
  }
}
// mosek.fusion.LinearVariable.ctor

// mosek.fusion.LinearVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:36:6-51:7
mosek::fusion::LinearVariable::t mosek::fusion::p_LinearVariable::_new_LinearVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2210_model,
    const std::string &  _2211_name,
    int64_t _2212_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2213_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2214_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2215_nativeidxs)
{ return new mosek::fusion::LinearVariable(_2210_model,_2211_name,_2212_varid,_2213_shape,_2214_sparsity,_2215_nativeidxs); }
mosek::fusion::LinearVariable::LinearVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2210_model,
    const std::string &  _2211_name,
    int64_t _2212_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2213_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2214_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2215_nativeidxs) : 
   LinearVariable(new mosek::fusion::p_LinearVariable(this))
{ mosek::fusion::p_LinearVariable::_get_impl(this)->_initialize(_2210_model,_2211_name,_2212_varid,_2213_shape,_2214_sparsity,_2215_nativeidxs); }
void mosek::fusion::p_LinearVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2210_model,
    const std::string &  _2211_name,
    int64_t _2212_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2213_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2214_sparsity,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2215_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2213_shape(_2213_shape ? _2213_shape->raw() : nullptr, _2213_shape ? _2213_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2214_sparsity(_2214_sparsity ? _2214_sparsity->raw() : nullptr, _2214_sparsity ? _2214_sparsity->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2215_nativeidxs(_2215_nativeidxs ? _2215_nativeidxs->raw() : nullptr, _2215_nativeidxs ? _2215_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2210_model,_2211_name,_2213_shape,_2212_varid,_2214_sparsity,::mosek::fusion::p_LinearVariable::globalNativeIndexes(_2215_nativeidxs));
    names_flushed = false;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->name = _2211_name;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->nativeidxs = _2215_nativeidxs;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->sparsity = _2214_sparsity;
    mosek::fusion::p_LinearVariable::_get_impl(::mosek::fusion::LinearVariable::t(_pubthis))->shape = _2213_shape;
  }
}
// mosek.fusion.LinearVariable.ctor

// Begin mosek.fusion.LinearVariable.toString
// Method mosek.fusion.LinearVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:95:6-109:7
std::string mosek::fusion::LinearVariable :: toString() { return mosek::fusion::p_LinearVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_LinearVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2216_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("LinearVariable(("));
  if (((int)((shape)->size(0)) > (int)0))
  {
    {
      _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
      int32_t _10092_ = (int)1;
      int32_t _10093_ = (int)((shape)->size(0));
      for (int32_t _2217_i = _10092_; _2217_i < _10093_; ++_2217_i)
      {
        {
          _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2217_i)));
        }
      }
    }
  }
  {}
  _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  if ((NULL != sparsity.get()))
  {
    {
      _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (",sparse"));
    }
  }
  {}
  _2216_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (")"));
  return _2216_sb->toString();
}
// End mosek.fusion.LinearVariable.toString

// Begin mosek.fusion.LinearVariable.flushNames
// Method mosek.fusion.LinearVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:68:6-93:7
void mosek::fusion::p_LinearVariable::flushNames()
{
}
// End mosek.fusion.LinearVariable.flushNames

// Begin mosek.fusion.LinearVariable.clone
// Method mosek.fusion.LinearVariable.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:64:6-66:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_LinearVariable::__mosek_2fusion_2LinearVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2218_m)
{
  return (::mosek::fusion::p_LinearVariable::_new_LinearVariable(::mosek::fusion::LinearVariable::t(_pubthis),_2218_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.LinearVariable.clone

// Begin mosek.fusion.LinearVariable.globalNativeIndexes
// Method mosek.fusion.LinearVariable.globalNativeIndexes @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearVariable.mbi:31:6-34:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_LinearVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2219_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2219_nativeidxs(_2219_nativeidxs ? _2219_nativeidxs->raw() : nullptr, _2219_nativeidxs ? _2219_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2220_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2219_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2220_r(_2220_r ? _2220_r->raw() : nullptr,_2220_r ? _2220_r->size(0) : 0);
  int32_t _10094_ = (int)0;
  int32_t _10095_ = (int)((_2219_nativeidxs)->size(0));
  for (int32_t _2221_i = _10094_; _2221_i < _10095_; ++_2221_i)
  {
    {
      _ptr__2220_r[_2221_i] = _ptr__2219_nativeidxs[_2221_i];
    }
  }
  return _2220_r;
}
// End mosek.fusion.LinearVariable.globalNativeIndexes

void mosek::fusion::p_LinearVariable::destroy()
{
  shape.reset();
  sparsity.reset();
  nativeidxs.reset();
}
void mosek::fusion::LinearVariable::destroy() { mosek::fusion::p_LinearVariable::_get_impl(this)->destroy(); }
// } class LinearVariable
// class ConicVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:23:4-107:5
mosek::fusion::ConicVariable::ConicVariable(mosek::fusion::p_ConicVariable *_impl) : mosek::fusion::ModelVariable(_impl) { /*std::cout << "ConicVariable()" << this << std::endl; */ }
mosek::fusion::ConicVariable::~ConicVariable() {  /* std::cout << "~ConicVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_ConicVariable::p_ConicVariable
  (::mosek::fusion::ConicVariable * _pubthis) :     mosek::fusion::p_ModelVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.ConicVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:55:6-64:7
mosek::fusion::ConicVariable::t mosek::fusion::p_ConicVariable::_new_ConicVariable
  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2222_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2223_m)
{ return new mosek::fusion::ConicVariable(_2222_v,_2223_m); }
mosek::fusion::ConicVariable::ConicVariable
  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2222_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2223_m) : 
   ConicVariable(new mosek::fusion::p_ConicVariable(this))
{ mosek::fusion::p_ConicVariable::_get_impl(this)->_initialize(_2222_v,_2223_m); }
void mosek::fusion::p_ConicVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::ConicVariable > _2222_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2223_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelVariable::_initialize(_2222_v,_2223_m);
    names_flushed = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->names_flushed;
    shape = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->shape;
    nativeidxs = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->nativeidxs;
    shape = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->shape;
    varid = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->varid;
    name = mosek::fusion::p_ConicVariable::_get_impl(_2222_v)->name;
  }
}
// mosek.fusion.ConicVariable.ctor

// mosek.fusion.ConicVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:40:6-53:7
mosek::fusion::ConicVariable::t mosek::fusion::p_ConicVariable::_new_ConicVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2224_model,
    const std::string &  _2225_name,
    int32_t _2226_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2227_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2228_nativeidxs)
{ return new mosek::fusion::ConicVariable(_2224_model,_2225_name,_2226_varid,_2227_shape,_2228_nativeidxs); }
mosek::fusion::ConicVariable::ConicVariable
  (monty::rc_ptr< ::mosek::fusion::Model > _2224_model,
    const std::string &  _2225_name,
    int32_t _2226_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2227_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2228_nativeidxs) : 
   ConicVariable(new mosek::fusion::p_ConicVariable(this))
{ mosek::fusion::p_ConicVariable::_get_impl(this)->_initialize(_2224_model,_2225_name,_2226_varid,_2227_shape,_2228_nativeidxs); }
void mosek::fusion::p_ConicVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2224_model,
    const std::string &  _2225_name,
    int32_t _2226_varid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2227_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2228_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2227_shape(_2227_shape ? _2227_shape->raw() : nullptr, _2227_shape ? _2227_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2228_nativeidxs(_2228_nativeidxs ? _2228_nativeidxs->raw() : nullptr, _2228_nativeidxs ? _2228_nativeidxs->size(0) : 0);
    mosek::fusion::p_ModelVariable::_initialize(_2224_model,_2225_name,_2227_shape,_2226_varid,nullptr,::mosek::fusion::p_ConicVariable::globalNativeIndexes(_2228_nativeidxs));
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->name = _2225_name;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->names_flushed = false;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->varid = _2226_varid;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->shape = _2227_shape;
    mosek::fusion::p_ConicVariable::_get_impl(::mosek::fusion::ConicVariable::t(_pubthis))->nativeidxs = _2228_nativeidxs;
  }
}
// mosek.fusion.ConicVariable.ctor

// Begin mosek.fusion.ConicVariable.toString
// Method mosek.fusion.ConicVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:96:6-106:7
std::string mosek::fusion::ConicVariable :: toString() { return mosek::fusion::p_ConicVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_ConicVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2229_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("ConicVariable( "));
  if (((int)((name).size()) > (int)0))
  {
    {
      _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"))->__mosek_2fusion_2Utils_2StringBuffer__a(name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("',"));
    }
  }
  {}
  _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("("));
  _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
  int32_t _10096_ = (int)1;
  int32_t _10097_ = (int)((shape)->size(0));
  for (int32_t _2230_i = _10096_; _2230_i < _10097_; ++_2230_i)
  {
    {
      _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2230_i)));
    }
  }
  _2229_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (") )"));
  return _2229_sb->toString();
}
// End mosek.fusion.ConicVariable.toString

// Begin mosek.fusion.ConicVariable.flushNames
// Method mosek.fusion.ConicVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:71:6-95:7
void mosek::fusion::p_ConicVariable::flushNames()
{
}
// End mosek.fusion.ConicVariable.flushNames

// Begin mosek.fusion.ConicVariable.clone
// Method mosek.fusion.ConicVariable.clone @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:66:6-69:7
monty::rc_ptr< ::mosek::fusion::ModelVariable > mosek::fusion::p_ConicVariable::__mosek_2fusion_2ConicVariable__clone(monty::rc_ptr< ::mosek::fusion::Model > _2231_m)
{
  return (::mosek::fusion::p_ConicVariable::_new_ConicVariable(::mosek::fusion::ConicVariable::t(_pubthis),_2231_m)).as<::mosek::fusion::ModelVariable>();
}
// End mosek.fusion.ConicVariable.clone

// Begin mosek.fusion.ConicVariable.globalNativeIndexes
// Method mosek.fusion.ConicVariable.globalNativeIndexes @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ConicVariable.mbi:35:6-38:7
std::shared_ptr< monty::ndarray< int64_t,1 > > mosek::fusion::p_ConicVariable::globalNativeIndexes(std::shared_ptr< monty::ndarray< int32_t,1 > > _2232_nativeidxs)
{
  _checked_ptr_1<int32_t> _ptr__2232_nativeidxs(_2232_nativeidxs ? _2232_nativeidxs->raw() : nullptr, _2232_nativeidxs ? _2232_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2233_r = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2232_nativeidxs)->size(0)))));
  _checked_ptr_1<int64_t> _ptr__2233_r(_2233_r ? _2233_r->raw() : nullptr,_2233_r ? _2233_r->size(0) : 0);
  int32_t _10098_ = (int)0;
  int32_t _10099_ = (int)((_2232_nativeidxs)->size(0));
  for (int32_t _2234_i = _10098_; _2234_i < _10099_; ++_2234_i)
  {
    {
      _ptr__2233_r[_2234_i] = _ptr__2232_nativeidxs[_2234_i];
    }
  }
  return _2233_r;
}
// End mosek.fusion.ConicVariable.globalNativeIndexes

void mosek::fusion::p_ConicVariable::destroy()
{
  nativeidxs.reset();
  shape.reset();
}
void mosek::fusion::ConicVariable::destroy() { mosek::fusion::p_ConicVariable::_get_impl(this)->destroy(); }
// } class ConicVariable
// class ModelVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ModelVariable.mbi:14:4-92:5
mosek::fusion::ModelVariable::ModelVariable(mosek::fusion::p_ModelVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "ModelVariable()" << this << std::endl; */ }
mosek::fusion::ModelVariable::~ModelVariable() {  /* std::cout << "~ModelVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_ModelVariable::p_ModelVariable
  (::mosek::fusion::ModelVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.ModelVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ModelVariable.mbi:62:6-72:7
void mosek::fusion::p_ModelVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::ModelVariable > _2235_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2236_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_BaseVariable::_initialize(_2235_v,_2236_m);
    name = mosek::fusion::p_ModelVariable::_get_impl(_2235_v)->name;
    varid = mosek::fusion::p_ModelVariable::_get_impl(_2235_v)->varid;
    modelvar_nativeidxs = mosek::fusion::p_ModelVariable::_get_impl(_2235_v)->modelvar_nativeidxs;
    shape = mosek::fusion::p_ModelVariable::_get_impl(_2235_v)->shape;
    sparsity = mosek::fusion::p_ModelVariable::_get_impl(_2235_v)->sparsity;
  }
}
// mosek.fusion.ModelVariable.ctor

// mosek.fusion.ModelVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ModelVariable.mbi:44:6-58:7
void mosek::fusion::p_ModelVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2237_model,
    const std::string &  _2238_name,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2239_shape,
    int64_t _2240_varid,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2241_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2242_modelvar_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2239_shape(_2239_shape ? _2239_shape->raw() : nullptr, _2239_shape ? _2239_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2241_sparsity(_2241_sparsity ? _2241_sparsity->raw() : nullptr, _2241_sparsity ? _2241_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2242_modelvar_nativeidxs(_2242_modelvar_nativeidxs ? _2242_modelvar_nativeidxs->raw() : nullptr, _2242_modelvar_nativeidxs ? _2242_modelvar_nativeidxs->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(_2237_model,_2239_shape,_2241_sparsity,_2242_modelvar_nativeidxs);
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->name = _2238_name;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->varid = _2240_varid;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->shape = _2239_shape;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->sparsity = _2241_sparsity;
    mosek::fusion::p_ModelVariable::_get_impl(::mosek::fusion::ModelVariable::t(_pubthis))->modelvar_nativeidxs = _2242_modelvar_nativeidxs;
  }
}
// mosek.fusion.ModelVariable.ctor

// Begin mosek.fusion.ModelVariable.flushNames
// End mosek.fusion.ModelVariable.flushNames

// Begin mosek.fusion.ModelVariable.elementName
// Method mosek.fusion.ModelVariable.elementName @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ModelVariable.mbi:84:6-88:7
void mosek::fusion::ModelVariable :: elementName(int64_t _2243_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2244_sb) { mosek::fusion::p_ModelVariable::_get_impl(this)->elementName(_2243_index,_2244_sb); }
void mosek::fusion::p_ModelVariable::elementName(int64_t _2243_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2244_sb)
{
  std::string _10100_;
  bool _10101_ = ((int)((name).size()) == (int)0);
  if (_10101_)
  {
    _10100_ = std::string ("_");
  }
  else
  {
    _10100_ = name;
  }
  _2244_sb->__mosek_2fusion_2Utils_2StringBuffer__a(_10100_)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("["))->__mosek_2fusion_2Utils_2StringBuffer__a(::mosek::fusion::p_Set::indexToString(shape,_2243_index))->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("]"));
}
// End mosek.fusion.ModelVariable.elementName

// Begin mosek.fusion.ModelVariable.remove
// Method mosek.fusion.ModelVariable.remove @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/ModelVariable.mbi:78:6-80:7
void mosek::fusion::ModelVariable :: remove() { mosek::fusion::p_ModelVariable::_get_impl(this)->remove(); }
void mosek::fusion::p_ModelVariable::remove()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->removeVariableBlock(varid);
}
// End mosek.fusion.ModelVariable.remove

// Begin mosek.fusion.ModelVariable.clone
// End mosek.fusion.ModelVariable.clone

void mosek::fusion::p_ModelVariable::destroy()
{
  sparsity.reset();
  shape.reset();
  modelvar_nativeidxs.reset();
}
void mosek::fusion::ModelVariable::destroy() { mosek::fusion::p_ModelVariable::_get_impl(this)->destroy(); }
// } class ModelVariable
// class NilVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1000:4-1088:5
mosek::fusion::NilVariable::NilVariable(mosek::fusion::p_NilVariable *_impl) : mosek::fusion::BaseVariable(_impl) { /*std::cout << "NilVariable()" << this << std::endl; */ }
mosek::fusion::NilVariable::~NilVariable() {  /* std::cout << "~NilVariable()" << this << std::endl << std::flush; */ }
mosek::fusion::p_NilVariable::p_NilVariable
  (::mosek::fusion::NilVariable * _pubthis) :     mosek::fusion::p_BaseVariable(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.NilVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1008:6-1012:7
mosek::fusion::NilVariable::t mosek::fusion::p_NilVariable::_new_NilVariable
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2246_shape)
{ return new mosek::fusion::NilVariable(_2246_shape); }
mosek::fusion::NilVariable::NilVariable
  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2246_shape) : 
   NilVariable(new mosek::fusion::p_NilVariable(this))
{ mosek::fusion::p_NilVariable::_get_impl(this)->_initialize(_2246_shape); }
void mosek::fusion::p_NilVariable::_initialize  (std::shared_ptr< monty::ndarray< int32_t,1 > > _2246_shape)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2246_shape(_2246_shape ? _2246_shape->raw() : nullptr, _2246_shape ? _2246_shape->size(0) : 0);
    mosek::fusion::p_BaseVariable::_initialize(nullptr,::mosek::fusion::Utils::Tools::arraycopy(_2246_shape),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))),std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)0))));
    std::vector<int32_t> _10102_;
    for (int32_t _2247_i = (int)0, _10103_ = (int)((_2246_shape)->size(0)); _2247_i < _10103_ ; ++_2247_i)
    {
      _10102_.push_back(_ptr__2246_shape[_2247_i]);
    }
    auto _10104_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10102_.size()),_10102_.begin(),_10102_.end()));
    mosek::fusion::p_NilVariable::_get_impl(::mosek::fusion::NilVariable::t(_pubthis))->shape = _10104_;
  }
}
// mosek.fusion.NilVariable.ctor

// mosek.fusion.NilVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1006:6-45
mosek::fusion::NilVariable::t mosek::fusion::p_NilVariable::_new_NilVariable
  ()
{ return new mosek::fusion::NilVariable(); }
mosek::fusion::NilVariable::NilVariable
  () : 
   NilVariable(new mosek::fusion::p_NilVariable(this))
{ mosek::fusion::p_NilVariable::_get_impl(this)->_initialize(); }
void mosek::fusion::p_NilVariable::_initialize  ()
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_NilVariable::_initialize(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
  }
}
// mosek.fusion.NilVariable.ctor

// Begin mosek.fusion.NilVariable.flushNames
// Method mosek.fusion.NilVariable.flushNames @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1085:6-44
void mosek::fusion::p_NilVariable::flushNames()
{
}
// End mosek.fusion.NilVariable.flushNames

// Begin mosek.fusion.NilVariable.elementDesc
// Method mosek.fusion.NilVariable.elementDesc @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1084:6-102
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__elementDesc(int64_t _2248_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2249_sb) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__elementDesc(_2248_index,_2249_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::NilVariable::elementDesc(int64_t _2248_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2249_sb) { return __mosek_2fusion_2NilVariable__elementDesc(_2248_index,_2249_sb); }
monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__elementDesc(int64_t _2248_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2249_sb)
{
  return _2249_sb;
}
// End mosek.fusion.NilVariable.elementDesc

// Begin mosek.fusion.NilVariable.elementName
// Method mosek.fusion.NilVariable.elementName @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1083:6-76
void mosek::fusion::NilVariable :: elementName(int64_t _2250_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2251_sb) { mosek::fusion::p_NilVariable::_get_impl(this)->elementName(_2250_index,_2251_sb); }
void mosek::fusion::p_NilVariable::elementName(int64_t _2250_index,monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2251_sb)
{
}
// End mosek.fusion.NilVariable.elementName

// Begin mosek.fusion.NilVariable.numInst
// Method mosek.fusion.NilVariable.numInst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1079:6-1081:7
int32_t mosek::fusion::NilVariable :: numInst() { return mosek::fusion::p_NilVariable::_get_impl(this)->numInst(); }
int32_t mosek::fusion::p_NilVariable::numInst()
{
  return (int)0;
}
// End mosek.fusion.NilVariable.numInst

// Begin mosek.fusion.NilVariable.inst
// Method mosek.fusion.NilVariable.inst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1071:6-1077:7
int32_t mosek::fusion::NilVariable :: inst(int32_t _2252_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2253_sparsity,std::shared_ptr< monty::ndarray< int64_t,1 > > _2254_basevar_nativeidxs) { return mosek::fusion::p_NilVariable::_get_impl(this)->inst(_2252_offset,_2253_sparsity,_2254_basevar_nativeidxs); }
int32_t mosek::fusion::p_NilVariable::inst(int32_t _2252_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2253_sparsity,std::shared_ptr< monty::ndarray< int64_t,1 > > _2254_basevar_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2253_sparsity(_2253_sparsity ? _2253_sparsity->raw() : nullptr, _2253_sparsity ? _2253_sparsity->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2254_basevar_nativeidxs(_2254_basevar_nativeidxs ? _2254_basevar_nativeidxs->raw() : nullptr, _2254_basevar_nativeidxs ? _2254_basevar_nativeidxs->size(0) : 0);
  return (int)0;
}
// End mosek.fusion.NilVariable.inst

// Begin mosek.fusion.NilVariable.inst
// Method mosek.fusion.NilVariable.inst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1066:6-1069:8
void mosek::fusion::NilVariable :: inst(int32_t _2255_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2256_nindex) { mosek::fusion::p_NilVariable::_get_impl(this)->inst(_2255_offset,_2256_nindex); }
void mosek::fusion::p_NilVariable::inst(int32_t _2255_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2256_nindex)
{
  _checked_ptr_1<int64_t> _ptr__2256_nindex(_2256_nindex ? _2256_nindex->raw() : nullptr, _2256_nindex ? _2256_nindex->size(0) : 0);
}
// End mosek.fusion.NilVariable.inst

// Begin mosek.fusion.NilVariable.set_values
// Method mosek.fusion.NilVariable.set_values @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1061:6-1064:8
void mosek::fusion::NilVariable :: set_values(std::shared_ptr< monty::ndarray< double,1 > > _2257_target,bool _2258_primal) { mosek::fusion::p_NilVariable::_get_impl(this)->set_values(_2257_target,_2258_primal); }
void mosek::fusion::p_NilVariable::set_values(std::shared_ptr< monty::ndarray< double,1 > > _2257_target,bool _2258_primal)
{
  _checked_ptr_1<double> _ptr__2257_target(_2257_target ? _2257_target->raw() : nullptr, _2257_target ? _2257_target->size(0) : 0);
}
// End mosek.fusion.NilVariable.set_values

// Begin mosek.fusion.NilVariable.values
// Method mosek.fusion.NilVariable.values @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1055:6-1059:8
void mosek::fusion::NilVariable :: values(int32_t _2259_offset,std::shared_ptr< monty::ndarray< double,1 > > _2260_target,bool _2261_primal) { mosek::fusion::p_NilVariable::_get_impl(this)->values(_2259_offset,_2260_target,_2261_primal); }
void mosek::fusion::p_NilVariable::values(int32_t _2259_offset,std::shared_ptr< monty::ndarray< double,1 > > _2260_target,bool _2261_primal)
{
  _checked_ptr_1<double> _ptr__2260_target(_2260_target ? _2260_target->raw() : nullptr, _2260_target ? _2260_target->size(0) : 0);
}
// End mosek.fusion.NilVariable.values

// Begin mosek.fusion.NilVariable.make_continuous
// Method mosek.fusion.NilVariable.make_continuous @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1053:6-40
void mosek::fusion::NilVariable :: make_continuous() { mosek::fusion::p_NilVariable::_get_impl(this)->make_continuous(); }
void mosek::fusion::p_NilVariable::make_continuous()
{
}
// End mosek.fusion.NilVariable.make_continuous

// Begin mosek.fusion.NilVariable.make_integer
// Method mosek.fusion.NilVariable.make_integer @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1052:6-37
void mosek::fusion::NilVariable :: make_integer() { mosek::fusion::p_NilVariable::_get_impl(this)->make_integer(); }
void mosek::fusion::p_NilVariable::make_integer()
{
}
// End mosek.fusion.NilVariable.make_integer

// Begin mosek.fusion.NilVariable.toString
// Method mosek.fusion.NilVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1050:6-63
std::string mosek::fusion::NilVariable :: toString() { return mosek::fusion::p_NilVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_NilVariable::toString()
{
  return std::string ("NilVariable");
}
// End mosek.fusion.NilVariable.toString

// Begin mosek.fusion.NilVariable.index
// Method mosek.fusion.NilVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1044:6-1048:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2262_first) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__index(_2262_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2262_first) { return __mosek_2fusion_2NilVariable__index(_2262_first); }
/* override: mosek.fusion.BaseVariable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2262_first) { return __mosek_2fusion_2NilVariable__index(_2262_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2262_first)
{
  _checked_ptr_1<int32_t> _ptr__2262_first(_2262_first ? _2262_first->raw() : nullptr, _2262_first ? _2262_first->size(0) : 0);
  std::vector<int32_t> _10105_;
  for (int32_t _2263_i = (int)0, _10106_ = (int)((shape)->size(0)); _2263_i < _10106_ ; ++_2263_i)
  {
    _10105_.push_back((int)1);
  }
  auto _10107_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10105_.size()),_10105_.begin(),_10105_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10107_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.index

// Begin mosek.fusion.NilVariable.index
// Method mosek.fusion.NilVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1038:6-1042:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__index(int32_t _2264_first) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__index(_2264_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::index(int32_t _2264_first) { return __mosek_2fusion_2NilVariable__index(_2264_first); }
/* override: mosek.fusion.BaseVariable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2264_first) { return __mosek_2fusion_2NilVariable__index(_2264_first); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__index(int32_t _2264_first)
{
  std::vector<int32_t> _10108_;
  for (int32_t _2265_i = (int)0, _10109_ = (int)((shape)->size(0)); _2265_i < _10109_ ; ++_2265_i)
  {
    _10108_.push_back((int)1);
  }
  auto _10110_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10108_.size()),_10108_.begin(),_10108_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10110_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.index

// Begin mosek.fusion.NilVariable.slice
// Method mosek.fusion.NilVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1026:6-1036:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2266_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2267_last) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__slice(_2266_first,_2267_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2266_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2267_last) { return __mosek_2fusion_2NilVariable__slice(_2266_first,_2267_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2266_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2267_last) { return __mosek_2fusion_2NilVariable__slice(_2266_first,_2267_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2266_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2267_last)
{
  _checked_ptr_1<int32_t> _ptr__2266_first(_2266_first ? _2266_first->raw() : nullptr, _2266_first ? _2266_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2267_last(_2267_last ? _2267_last->raw() : nullptr, _2267_last ? _2267_last->size(0) : 0);
  if ((((int)((shape)->size(0)) != (int)((_2266_first)->size(0))) || ((int)((shape)->size(0)) != (int)((_2267_last)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching slice shape"));
    }
  }
  {}
  int32_t _10111_ = (int)0;
  int32_t _10112_ = (int)((_2266_first)->size(0));
  for (int32_t _2268_i = _10111_; _2268_i < _10112_; ++_2268_i)
  {
    {
      if (((_ptr__2267_last[_2268_i] > _ptr__2266_first[_2268_i]) || ((_ptr__2266_first[_2268_i] < (int)0) || (_ptr__2267_last[_2268_i] >= ((*shape)(_2268_i))))))
      {
        {
          throw ::mosek::fusion::SliceError(std::string ("Slice is out of bounds"));
        }
      }
      {}
    }
  }
  std::vector<int32_t> _10113_;
  for (int32_t _2269_i = (int)0, _10114_ = (int)((shape)->size(0)); _2269_i < _10114_ ; ++_2269_i)
  {
    _10113_.push_back((_ptr__2267_last[_2269_i] - _ptr__2266_first[_2269_i]));
  }
  auto _10115_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10113_.size()),_10113_.begin(),_10113_.end()));
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_10115_)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.NilVariable.slice

// Begin mosek.fusion.NilVariable.slice
// Method mosek.fusion.NilVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:1014:6-1024:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable :: __mosek_2fusion_2NilVariable__slice(int32_t _2270_first,int32_t _2271_last) { return mosek::fusion::p_NilVariable::_get_impl(this)->__mosek_2fusion_2NilVariable__slice(_2270_first,_2271_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::slice(int32_t _2270_first,int32_t _2271_last) { return __mosek_2fusion_2NilVariable__slice(_2270_first,_2271_last); }
/* override: mosek.fusion.BaseVariable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::NilVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2270_first,int32_t _2271_last) { return __mosek_2fusion_2NilVariable__slice(_2270_first,_2271_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_NilVariable::__mosek_2fusion_2NilVariable__slice(int32_t _2270_first,int32_t _2271_last)
{
  if (((_2271_last > _2270_first) || ((_2270_first < (int)0) || (_2271_last > ((*shape)((int)0))))))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Slice is out of bounds"));
    }
  }
  else if(((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Mismatching slice shape"));
    }
  }
  else
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((_2271_last - _2270_first))}))).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.NilVariable.slice

void mosek::fusion::p_NilVariable::destroy()
{
  shape.reset();
}
void mosek::fusion::NilVariable::destroy() { mosek::fusion::p_NilVariable::_get_impl(this)->destroy(); }
// } class NilVariable
// class BaseVariable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:53:4-969:5
mosek::fusion::BaseVariable::BaseVariable(mosek::fusion::p_BaseVariable *_impl) : _impl(_impl) { /*std::cout << "BaseVariable() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::BaseVariable::~BaseVariable() { delete _impl; _impl = nullptr; }
mosek::fusion::p_BaseVariable::p_BaseVariable
  (::mosek::fusion::BaseVariable * _pubthis) :     _pubthis(_pubthis)
{}
// mosek.fusion.BaseVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:99:6-108:7
void mosek::fusion::p_BaseVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::BaseVariable > _2272_v,
    monty::rc_ptr< ::mosek::fusion::Model > _2273_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    model = _2273_m;
    shape = mosek::fusion::p_BaseVariable::_get_impl(_2272_v)->shape;
    sparsity = mosek::fusion::p_BaseVariable::_get_impl(_2272_v)->sparsity;
    basevar_nativeidxs = mosek::fusion::p_BaseVariable::_get_impl(_2272_v)->basevar_nativeidxs;
  }
}
// mosek.fusion.BaseVariable.ctor

// mosek.fusion.BaseVariable.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:83:6-94:7
void mosek::fusion::p_BaseVariable::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2274_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2275_shape,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2276_sparsity,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2277_basevar_nativeidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2275_shape(_2275_shape ? _2275_shape->raw() : nullptr, _2275_shape ? _2275_shape->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2276_sparsity(_2276_sparsity ? _2276_sparsity->raw() : nullptr, _2276_sparsity ? _2276_sparsity->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2277_basevar_nativeidxs(_2277_basevar_nativeidxs ? _2277_basevar_nativeidxs->raw() : nullptr, _2277_basevar_nativeidxs ? _2277_basevar_nativeidxs->size(0) : 0);
    model = _2274_m;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->shape = _2275_shape;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity = _2276_sparsity;
    mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs = _2277_basevar_nativeidxs;
  }
}
// mosek.fusion.BaseVariable.ctor

// Begin mosek.fusion.BaseVariable.toString
// Method mosek.fusion.BaseVariable.toString @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:957:6-968:7
std::string mosek::fusion::BaseVariable :: toString() { return mosek::fusion::p_BaseVariable::_get_impl(this)->toString(); }
std::string mosek::fusion::p_BaseVariable::toString()
{
  monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2278_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
  _2278_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Variable(("));
  if (((int)((shape)->size(0)) > (int)0))
  {
    {
      _2278_sb->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)((int)0)));
      int32_t _10116_ = (int)1;
      int32_t _10117_ = (int)((shape)->size(0));
      for (int32_t _2279_i = _10116_; _2279_i < _10117_; ++_2279_i)
      {
        {
          _2278_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string (","))->__mosek_2fusion_2Utils_2StringBuffer__a(((*shape)(_2279_i)));
        }
      }
    }
  }
  {}
  _2278_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("))"));
  return _2278_sb->toString();
}
// End mosek.fusion.BaseVariable.toString

// Begin mosek.fusion.BaseVariable.eval
// Method mosek.fusion.BaseVariable.eval @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:926:6-954:7
void mosek::fusion::BaseVariable :: eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2280_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2281_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2282_xs) { mosek::fusion::p_BaseVariable::_get_impl(this)->eval(_2280_rs,_2281_ws,_2282_xs); }
void mosek::fusion::p_BaseVariable::eval(monty::rc_ptr< ::mosek::fusion::WorkStack > _2280_rs,monty::rc_ptr< ::mosek::fusion::WorkStack > _2281_ws,monty::rc_ptr< ::mosek::fusion::WorkStack > _2282_xs)
{
  int32_t _2283_nnz = (int)((basevar_nativeidxs)->size(0));
  int32_t _2284_nelem = _2283_nnz;
  int32_t _2285_nd = (int)((shape)->size(0));
  bool _2286_hassp = (NULL != sparsity.get());
  _2280_rs->alloc_expr(_2285_nd,_2284_nelem,_2283_nnz,_2286_hassp);
  int32_t _2287_ptr = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->ptr_base;
  int32_t _2288_nidxs = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->nidxs_base;
  int32_t _2289_sp = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->sp_base;
  int32_t _2290_cof = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->cof_base;
  int32_t _2291_shape_base = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->shape_base;
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2292_ri32 = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->i32;
  _checked_ptr_1<int32_t> _ptr__2292_ri32(_2292_ri32 ? _2292_ri32->raw() : nullptr,_2292_ri32 ? _2292_ri32->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2293_ri64 = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->i64;
  _checked_ptr_1<int64_t> _ptr__2293_ri64(_2293_ri64 ? _2293_ri64->raw() : nullptr,_2293_ri64 ? _2293_ri64->size(0) : 0);
  std::shared_ptr< monty::ndarray< double,1 > > _2294_rf64 = mosek::fusion::p_WorkStack::_get_impl(_2280_rs)->f64;
  _checked_ptr_1<double> _ptr__2294_rf64(_2294_rf64 ? _2294_rf64->raw() : nullptr,_2294_rf64 ? _2294_rf64->size(0) : 0);
  int32_t _10118_ = (int)0;
  int32_t _10119_ = _2285_nd;
  for (int32_t _2295_i = _10118_; _2295_i < _10119_; ++_2295_i)
  {
    {
      _ptr__2292_ri32[safe_add( _2291_shape_base,_2295_i )] = ((*shape)(_2295_i));
    }
  }
  {
    int32_t _10120_ = (int)0;
    int32_t _10121_ = safe_add( _2283_nnz,(int)1 );
    for (int32_t _2296_i = _10120_; _2296_i < _10121_; ++_2296_i)
    {
      {
        _ptr__2292_ri32[safe_add( _2287_ptr,_2296_i )] = _2296_i;
      }
    }
    int32_t _10122_ = (int)0;
    int32_t _10123_ = _2283_nnz;
    for (int32_t _2297_i = _10122_; _2297_i < _10123_; ++_2297_i)
    {
      {
        _ptr__2293_ri64[safe_add( _2288_nidxs,_2297_i )] = ((*basevar_nativeidxs)(_2297_i));
      }
    }
    if (_2286_hassp)
    {
      {
        int32_t _10124_ = (int)0;
        int32_t _10125_ = _2283_nnz;
        for (int32_t _2298_i = _10124_; _2298_i < _10125_; ++_2298_i)
        {
          {
            _ptr__2293_ri64[safe_add( _2289_sp,_2298_i )] = ((*sparsity)(_2298_i));
          }
        }
      }
    }
    {}
    int32_t _10126_ = (int)0;
    int32_t _10127_ = _2283_nnz;
    for (int32_t _2299_i = _10126_; _2299_i < _10127_; ++_2299_i)
    {
      {
        _ptr__2294_rf64[safe_add( _2290_cof,_2299_i )] = 1.0;
      }
    }
  }
}
// End mosek.fusion.BaseVariable.eval

// Begin mosek.fusion.BaseVariable.remove
// Method mosek.fusion.BaseVariable.remove @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:920:6-922:7
void mosek::fusion::BaseVariable :: remove() { mosek::fusion::p_BaseVariable::_get_impl(this)->remove(); }
void mosek::fusion::p_BaseVariable::remove()
{
  throw ::mosek::fusion::DeletionError(std::string ("Only original variables can be removed"));
}
// End mosek.fusion.BaseVariable.remove

// Begin mosek.fusion.BaseVariable.fromTril
// Method mosek.fusion.BaseVariable.fromTril @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:844:6-918:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__fromTril(int32_t _2300_dim0,int32_t _2301_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__fromTril(_2300_dim0,_2301_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::fromTril(int32_t _2300_dim0,int32_t _2301_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2300_dim0,_2301_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__fromTril(int32_t _2300_dim0,int32_t _2301_d)
{
  if (((_2300_dim0 < (int)0) || (_2300_dim0 >= (int)((shape)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid dimension"));
    }
  }
  else if((((*shape)(_2300_dim0)) != (safe_mul( _2301_d,safe_add( _2301_d,(int)1 ) ) / (int)2)))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid shape for operation"));
    }
  }
  {}
  int32_t _2302_d0 = (int)1;
  int32_t _10128_ = (int)0;
  int32_t _10129_ = _2300_dim0;
  for (int32_t _2303_i = _10128_; _2303_i < _10129_; ++_2303_i)
  {
    {
      _2302_d0 *= ((*shape)(_2303_i));
    }
  }
  int32_t _2304_d1 = ((*shape)(_2300_dim0));
  int32_t _2305_d2 = (int)1;
  int32_t _10130_ = safe_add( _2300_dim0,(int)1 );
  int32_t _10131_ = (int)((shape)->size(0));
  for (int32_t _2306_i = _10130_; _2306_i < _10131_; ++_2306_i)
  {
    {
      _2305_d2 *= ((*shape)(_2306_i));
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2307_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( (int)((shape)->size(0)),(int)1 ))));
  _checked_ptr_1<int32_t> _ptr__2307_rshape(_2307_rshape ? _2307_rshape->raw() : nullptr,_2307_rshape ? _2307_rshape->size(0) : 0);
  int32_t _10132_ = (int)0;
  int32_t _10133_ = _2300_dim0;
  for (int32_t _2308_i = _10132_; _2308_i < _10133_; ++_2308_i)
  {
    {
      _ptr__2307_rshape[_2308_i] = ((*shape)(_2308_i));
    }
  }
  _ptr__2307_rshape[_2300_dim0] = _2301_d;
  _ptr__2307_rshape[safe_add( _2300_dim0,(int)1 )] = _2301_d;
  int32_t _10134_ = safe_add( _2300_dim0,(int)1 );
  int32_t _10135_ = (int)((shape)->size(0));
  for (int32_t _2309_i = _10134_; _2309_i < _10135_; ++_2309_i)
  {
    {
      _ptr__2307_rshape[safe_add( _2309_i,(int)1 )] = ((*shape)(_2309_i));
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2310_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_mul( safe_mul( safe_mul( _2302_d0,_2301_d ),_2301_d ),_2305_d2 ))));
      _checked_ptr_1<int64_t> _ptr__2310_nidxs(_2310_nidxs ? _2310_nidxs->raw() : nullptr,_2310_nidxs ? _2310_nidxs->size(0) : 0);
      int32_t _2311_ofs = (int)0;
      int32_t _10136_ = (int)0;
      int32_t _10137_ = _2302_d0;
      for (int32_t _2312_i0 = _10136_; _2312_i0 < _10137_; ++_2312_i0)
      {
        {
          int32_t _10138_ = (int)0;
          int32_t _10139_ = _2301_d;
          for (int32_t _2313_i10 = _10138_; _2313_i10 < _10139_; ++_2313_i10)
          {
            {
              int32_t _10140_ = (int)0;
              int32_t _10141_ = _2301_d;
              for (int32_t _2314_i11 = _10140_; _2314_i11 < _10141_; ++_2314_i11)
              {
                {
                  int32_t _10142_ = (int)0;
                  int32_t _10143_ = _2305_d2;
                  for (int32_t _2315_i2 = _10142_; _2315_i2 < _10143_; ++_2315_i2)
                  {
                    {
                      if ((_2313_i10 >= _2314_i11))
                      {
                        {
                          _ptr__2310_nidxs[_2311_ofs] = ((*basevar_nativeidxs)(safe_add( safe_add( safe_mul( safe_mul( _2312_i0,_2304_d1 ),_2305_d2 ),safe_mul( safe_add( (safe_mul( _2313_i10,safe_add( _2313_i10,(int)1 ) ) / (int)2),_2314_i11 ),_2305_d2 ) ),_2315_i2 )));
                        }
                      }
                      else
                      {
                        {
                          _ptr__2310_nidxs[_2311_ofs] = ((*basevar_nativeidxs)(safe_add( safe_add( safe_mul( safe_mul( _2312_i0,_2304_d1 ),_2305_d2 ),safe_mul( safe_add( (safe_mul( _2314_i11,safe_add( _2314_i11,(int)1 ) ) / (int)2),_2313_i10 ),_2305_d2 ) ),_2315_i2 )));
                        }
                      }
                      ++ _2311_ofs;
                    }
                  }
                }
              }
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2307_rshape,nullptr,_2310_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2316_nelm = (int)0;
      int32_t _10144_ = (int)0;
      int32_t _10145_ = (int)((sparsity)->size(0));
      for (int32_t _2317_i = _10144_; _2317_i < _10145_; ++_2317_i)
      {
        {
          int64_t _2318_i1 = ((((*sparsity)(_2317_i)) / _2305_d2) % _2304_d1);
          int32_t _2319_ii = (int32_t)(((-(int)1) + ::mosek::fusion::Utils::Tools::sqrt((double)(safe_mul( safe_add( (int)1,safe_mul( (int)2,_2318_i1 ) ),safe_add( (int)1,safe_mul( (int)2,_2318_i1 ) ) )))) / (int)4);
          int64_t _2320_jj = (((*sparsity)(_2317_i)) - (safe_mul( _2319_ii,safe_add( _2319_ii,(int)1 ) ) / (int)2));
          if ((_2319_ii == _2320_jj))
          {
            {
              _2316_nelm += (int)1;
            }
          }
          else
          {
            {
              _2316_nelm += (int)2;
            }
          }
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2321_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2316_nelm)));
      _checked_ptr_1<int64_t> _ptr__2321_nidxs(_2321_nidxs ? _2321_nidxs->raw() : nullptr,_2321_nidxs ? _2321_nidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2322_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2316_nelm)));
      _checked_ptr_1<int64_t> _ptr__2322_sp(_2322_sp ? _2322_sp->raw() : nullptr,_2322_sp ? _2322_sp->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2323_perm = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2316_nelm)));
      _checked_ptr_1<int64_t> _ptr__2323_perm(_2323_perm ? _2323_perm->raw() : nullptr,_2323_perm ? _2323_perm->size(0) : 0);
      int32_t _10146_ = (int)0;
      int32_t _10147_ = _2316_nelm;
      for (int32_t _2324_i = _10146_; _2324_i < _10147_; ++_2324_i)
      {
        {
          _ptr__2323_perm[_2324_i] = _2324_i;
        }
      }
      int32_t _2325_elmi = (int)0;
      int32_t _10148_ = (int)0;
      int32_t _10149_ = (int)((sparsity)->size(0));
      for (int32_t _2326_i = _10148_; _2326_i < _10149_; ++_2326_i)
      {
        {
          int64_t _2327_i0 = (((*sparsity)(_2326_i)) / safe_mul( _2304_d1,_2305_d2 ));
          int64_t _2328_i1 = ((((*sparsity)(_2326_i)) / _2305_d2) % _2304_d1);
          int64_t _2329_i2 = (((*sparsity)(_2326_i)) % _2305_d2);
          int32_t _2330_ii1 = (int32_t)(((-(int)1) + ::mosek::fusion::Utils::Tools::sqrt((double)(safe_mul( safe_add( (int)1,safe_mul( (int)2,_2328_i1 ) ),safe_add( (int)1,safe_mul( (int)2,_2328_i1 ) ) )))) / (int)4);
          int64_t _2331_jj1 = (((*sparsity)(_2326_i)) - (safe_mul( _2330_ii1,safe_add( _2330_ii1,(int)1 ) ) / (int)2));
          if ((_2330_ii1 == _2331_jj1))
          {
            {
              _ptr__2321_nidxs[_2325_elmi] = ((*basevar_nativeidxs)(_2326_i));
              _ptr__2322_sp[_2325_elmi] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2327_i0,_2301_d ),_2301_d ),_2305_d2 ),safe_mul( safe_add( safe_mul( _2330_ii1,_2301_d ),_2331_jj1 ),_2305_d2 ) ),_2329_i2 );
              _2325_elmi += (int)1;
            }
          }
          else
          {
            {
              _ptr__2321_nidxs[_2325_elmi] = ((*basevar_nativeidxs)(_2326_i));
              _ptr__2321_nidxs[safe_add( _2325_elmi,(int)1 )] = ((*basevar_nativeidxs)(_2326_i));
              _ptr__2322_sp[_2325_elmi] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2327_i0,_2301_d ),_2301_d ),_2305_d2 ),safe_mul( safe_add( safe_mul( _2330_ii1,_2301_d ),_2331_jj1 ),_2305_d2 ) ),_2329_i2 );
              _ptr__2322_sp[safe_add( _2325_elmi,(int)1 )] = safe_add( safe_add( safe_mul( safe_mul( safe_mul( _2327_i0,_2301_d ),_2301_d ),_2305_d2 ),safe_mul( safe_add( safe_mul( _2331_jj1,_2301_d ),_2330_ii1 ),_2305_d2 ) ),_2329_i2 );
              _2325_elmi += (int)2;
            }
          }
        }
      }
      ::mosek::fusion::p_CommonTools::argQsort(_2323_perm,_2321_nidxs,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2316_nelm));
      std::vector<int64_t> _10150_;
      for (int32_t _2332_i = (int)0, _10151_ = _2316_nelm; _2332_i < _10151_ ; ++_2332_i)
      {
        _10150_.push_back(_ptr__2322_sp[_ptr__2323_perm[_2332_i]]);
      }
      auto _10152_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10150_.size()),_10150_.begin(),_10150_.end()));
      std::vector<int64_t> _10153_;
      for (int32_t _2333_i = (int)0, _10154_ = _2316_nelm; _2333_i < _10154_ ; ++_2333_i)
      {
        _10153_.push_back(_ptr__2321_nidxs[_ptr__2323_perm[_2333_i]]);
      }
      auto _10155_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10153_.size()),_10153_.begin(),_10153_.end()));
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2307_rshape,_10152_,_10155_)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.fromTril

// Begin mosek.fusion.BaseVariable.fromTril
// Method mosek.fusion.BaseVariable.fromTril @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:827:6-842:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__fromTril(int32_t _2334_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__fromTril(_2334_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::fromTril(int32_t _2334_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2334_d); }
/* override: mosek.fusion.Variable.fromTril*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__fromTril(int32_t _2334_d) { return __mosek_2fusion_2BaseVariable__fromTril(_2334_d); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__fromTril(int32_t _2334_d)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__fromTril((int)0,_2334_d);
}
// End mosek.fusion.BaseVariable.fromTril

// Begin mosek.fusion.BaseVariable.tril
// Method mosek.fusion.BaseVariable.tril @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:738:6-825:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__tril(int32_t _2335_dim1,int32_t _2336_dim2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__tril(_2335_dim1,_2336_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::tril(int32_t _2335_dim1,int32_t _2336_dim2) { return __mosek_2fusion_2BaseVariable__tril(_2335_dim1,_2336_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__tril(int32_t _2335_dim1,int32_t _2336_dim2)
{
  if (((_2335_dim1 < (int)0) || ((_2335_dim1 >= _2336_dim2) || (_2336_dim2 >= (int)((shape)->size(0))))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid dimension specification"));
    }
  }
  else
  {
    {
      if ((((*shape)(_2335_dim1)) != ((*shape)(_2336_dim2))))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Operation only valid for square variables"));
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2337_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(((int)((shape)->size(0)) - (int)1))));
  _checked_ptr_1<int32_t> _ptr__2337_rshape(_2337_rshape ? _2337_rshape->raw() : nullptr,_2337_rshape ? _2337_rshape->size(0) : 0);
  int32_t _10156_ = (int)0;
  int32_t _10157_ = _2335_dim1;
  for (int32_t _2338_i = _10156_; _2338_i < _10157_; ++_2338_i)
  {
    {
      _ptr__2337_rshape[_2338_i] = ((*shape)(_2338_i));
    }
  }
  _ptr__2337_rshape[_2335_dim1] = (safe_mul( ((*shape)(_2335_dim1)),safe_add( ((*shape)(_2335_dim1)),(int)1 ) ) / (int)2);
  int32_t _10158_ = safe_add( _2335_dim1,(int)1 );
  int32_t _10159_ = _2336_dim2;
  for (int32_t _2339_i = _10158_; _2339_i < _10159_; ++_2339_i)
  {
    {
      _ptr__2337_rshape[_2339_i] = ((*shape)(_2339_i));
    }
  }
  int32_t _10160_ = safe_add( _2336_dim2,(int)1 );
  int32_t _10161_ = (int)((shape)->size(0));
  for (int32_t _2340_i = _10160_; _2340_i < _10161_; ++_2340_i)
  {
    {
      _ptr__2337_rshape[(_2340_i - (int)1)] = ((*shape)(_2340_i));
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      int32_t _2341_d0 = (int)1;
      int32_t _10162_ = (int)0;
      int32_t _10163_ = _2335_dim1;
      for (int32_t _2342_i = _10162_; _2342_i < _10163_; ++_2342_i)
      {
        {
          _2341_d0 *= ((*shape)(_2342_i));
        }
      }
      int32_t _2343_d1 = ((*shape)(_2335_dim1));
      int32_t _2344_d2 = (int)1;
      int32_t _10164_ = safe_add( _2335_dim1,(int)1 );
      int32_t _10165_ = _2336_dim2;
      for (int32_t _2345_i = _10164_; _2345_i < _10165_; ++_2345_i)
      {
        {
          _2344_d2 *= ((*shape)(_2345_i));
        }
      }
      int32_t _2346_d3 = ((*shape)(_2336_dim2));
      int32_t _2347_d4 = (int)1;
      int32_t _10166_ = safe_add( _2336_dim2,(int)1 );
      int32_t _10167_ = (int)((shape)->size(0));
      for (int32_t _2348_i = _10166_; _2348_i < _10167_; ++_2348_i)
      {
        {
          _2347_d4 *= ((*shape)(_2348_i));
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2349_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((safe_mul( safe_mul( safe_mul( safe_mul( _2341_d0,_2344_d2 ),_2347_d4 ),_2343_d1 ),safe_add( _2343_d1,(int)1 ) ) / (int)2))));
      _checked_ptr_1<int64_t> _ptr__2349_nidxs(_2349_nidxs ? _2349_nidxs->raw() : nullptr,_2349_nidxs ? _2349_nidxs->size(0) : 0);
      int32_t _2350_i = (int)0;
      int32_t _10168_ = (int)0;
      int32_t _10169_ = _2341_d0;
      for (int32_t _2351_i0 = _10168_; _2351_i0 < _10169_; ++_2351_i0)
      {
        {
          int32_t _10170_ = (int)0;
          int32_t _10171_ = _2343_d1;
          for (int32_t _2352_i1 = _10170_; _2352_i1 < _10171_; ++_2352_i1)
          {
            {
              int32_t _10172_ = (int)0;
              int32_t _10173_ = safe_add( _2352_i1,(int)1 );
              for (int32_t _2353_i3 = _10172_; _2353_i3 < _10173_; ++_2353_i3)
              {
                {
                  int32_t _10174_ = (int)0;
                  int32_t _10175_ = _2344_d2;
                  for (int32_t _2354_i2 = _10174_; _2354_i2 < _10175_; ++_2354_i2)
                  {
                    {
                      int32_t _10176_ = (int)0;
                      int32_t _10177_ = _2347_d4;
                      for (int32_t _2355_i4 = _10176_; _2355_i4 < _10177_; ++_2355_i4)
                      {
                        {
                          _ptr__2349_nidxs[_2350_i] = ((*basevar_nativeidxs)(safe_add( _2355_i4,safe_mul( _2347_d4,safe_add( _2353_i3,safe_mul( _2346_d3,safe_add( _2354_i2,safe_mul( _2344_d2,safe_add( _2352_i1,safe_mul( _2343_d1,_2351_i0 ) ) ) ) ) ) ) )));
                          _2350_i += (int)1;
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2337_rshape,nullptr,_2349_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2356_d0 = (int)1;
      int32_t _10178_ = (int)0;
      int32_t _10179_ = _2335_dim1;
      for (int32_t _2357_i = _10178_; _2357_i < _10179_; ++_2357_i)
      {
        {
          _2356_d0 *= ((*shape)(_2357_i));
        }
      }
      int32_t _2358_d1 = ((*shape)(_2335_dim1));
      int32_t _2359_d2 = (int)1;
      int32_t _10180_ = safe_add( _2335_dim1,(int)1 );
      int32_t _10181_ = _2336_dim2;
      for (int32_t _2360_i = _10180_; _2360_i < _10181_; ++_2360_i)
      {
        {
          _2359_d2 *= ((*shape)(_2360_i));
        }
      }
      int32_t _2361_d3 = ((*shape)(_2336_dim2));
      int32_t _2362_d4 = (int)1;
      int32_t _10182_ = safe_add( _2336_dim2,(int)1 );
      int32_t _10183_ = (int)((shape)->size(0));
      for (int32_t _2363_i = _10182_; _2363_i < _10183_; ++_2363_i)
      {
        {
          _2362_d4 *= ((*shape)(_2363_i));
        }
      }
      int32_t _2364_s4 = (int)1;
      int32_t _2365_s3 = _2362_d4;
      int32_t _2366_s2 = safe_mul( _2365_s3,_2361_d3 );
      int32_t _2367_s1 = safe_mul( _2366_s2,_2359_d2 );
      int32_t _2368_s0 = safe_mul( _2367_s1,_2358_d1 );
      int32_t _2369_nelm = (int)0;
      int32_t _10184_ = (int)0;
      int32_t _10185_ = (int)((sparsity)->size(0));
      for (int32_t _2370_i = _10184_; _2370_i < _10185_; ++_2370_i)
      {
        {
          if ((((((*sparsity)(_2370_i)) / _2367_s1) % _2358_d1) >= ((((*sparsity)(_2370_i)) / _2365_s3) % _2361_d3)))
          {
            {
              ++ _2369_nelm;
            }
          }
          {}
        }
      }
      int32_t _2371_rs3 = (int)1;
      int32_t _2372_rs2 = _2362_d4;
      int32_t _2373_rs1 = safe_mul( _2372_rs2,_2359_d2 );
      int32_t _2374_rs0 = (safe_mul( safe_mul( _2373_rs1,_2358_d1 ),safe_add( _2358_d1,(int)1 ) ) / (int)2);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2375_perm = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2369_nelm)));
      _checked_ptr_1<int64_t> _ptr__2375_perm(_2375_perm ? _2375_perm->raw() : nullptr,_2375_perm ? _2375_perm->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2376_rnidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2369_nelm)));
      _checked_ptr_1<int64_t> _ptr__2376_rnidxs(_2376_rnidxs ? _2376_rnidxs->raw() : nullptr,_2376_rnidxs ? _2376_rnidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2377_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2369_nelm)));
      _checked_ptr_1<int64_t> _ptr__2377_rsp(_2377_rsp ? _2377_rsp->raw() : nullptr,_2377_rsp ? _2377_rsp->size(0) : 0);
      int32_t _2378_elmi = (int)0;
      {
        int32_t _10186_ = (int)0;
        int32_t _10187_ = (int)((sparsity)->size(0));
        for (int32_t _2379_i = _10186_; _2379_i < _10187_; ++_2379_i)
        {
          {
            if ((((((*sparsity)(_2379_i)) / _2367_s1) % _2358_d1) >= ((((*sparsity)(_2379_i)) / _2365_s3) % _2361_d3)))
            {
              {
                int64_t _2380_i0 = (((*sparsity)(_2379_i)) / _2368_s0);
                int64_t _2381_i1 = ((((*sparsity)(_2379_i)) / _2367_s1) % _2358_d1);
                int64_t _2382_i2 = ((((*sparsity)(_2379_i)) / _2366_s2) % _2359_d2);
                int64_t _2383_i3 = ((((*sparsity)(_2379_i)) / _2365_s3) % _2361_d3);
                int64_t _2384_i4 = ((((*sparsity)(_2379_i)) / _2364_s4) % _2362_d4);
                _ptr__2376_rnidxs[_2378_elmi] = ((*basevar_nativeidxs)(_2379_i));
                _ptr__2377_rsp[_2378_elmi] = safe_add( safe_add( safe_add( safe_mul( _2374_rs0,_2380_i0 ),safe_mul( _2373_rs1,safe_add( (safe_mul( _2381_i1,safe_add( _2381_i1,(int)1 ) ) / (int)2),_2383_i3 ) ) ),safe_mul( _2372_rs2,_2382_i2 ) ),safe_mul( _2371_rs3,_2384_i4 ) );
                ++ _2378_elmi;
              }
            }
            {}
          }
        }
        int32_t _10188_ = (int)0;
        int32_t _10189_ = _2369_nelm;
        for (int32_t _2385_i = _10188_; _2385_i < _10189_; ++_2385_i)
        {
          {
            _ptr__2375_perm[_2385_i] = _2385_i;
          }
        }
      }
      ::mosek::fusion::p_CommonTools::argQsort(_2375_perm,_2376_rnidxs,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2369_nelm));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2386_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2369_nelm)));
      _checked_ptr_1<int64_t> _ptr__2386_nidxs(_2386_nidxs ? _2386_nidxs->raw() : nullptr,_2386_nidxs ? _2386_nidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2387_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2369_nelm)));
      _checked_ptr_1<int64_t> _ptr__2387_sp(_2387_sp ? _2387_sp->raw() : nullptr,_2387_sp ? _2387_sp->size(0) : 0);
      int32_t _10190_ = (int)0;
      int32_t _10191_ = _2369_nelm;
      for (int32_t _2388_i = _10190_; _2388_i < _10191_; ++_2388_i)
      {
        {
          _ptr__2386_nidxs[_2388_i] = _ptr__2376_rnidxs[_ptr__2375_perm[_2388_i]];
        }
      }
      int32_t _10192_ = (int)0;
      int32_t _10193_ = _2369_nelm;
      for (int32_t _2389_i = _10192_; _2389_i < _10193_; ++_2389_i)
      {
        {
          _ptr__2387_sp[_2389_i] = _ptr__2377_rsp[_ptr__2375_perm[_2389_i]];
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2337_rshape,_2387_sp,_2386_nidxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.tril

// Begin mosek.fusion.BaseVariable.tril
// Method mosek.fusion.BaseVariable.tril @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:736:6-57
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__tril() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__tril(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::tril() { return __mosek_2fusion_2BaseVariable__tril(); }
/* override: mosek.fusion.Variable.tril*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__tril() { return __mosek_2fusion_2BaseVariable__tril(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__tril()
{
  return _pubthis->__mosek_2fusion_2BaseVariable__tril((int)0,(int)1);
}
// End mosek.fusion.BaseVariable.tril

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:734:6-114
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2390_dim0,int32_t _2391_dim1,int32_t _2392_dim2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2390_dim0,_2391_dim1,_2392_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2390_dim0,int32_t _2391_dim1,int32_t _2392_dim2) { return __mosek_2fusion_2BaseVariable__reshape(_2390_dim0,_2391_dim1,_2392_dim2); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2390_dim0,int32_t _2391_dim1,int32_t _2392_dim2) { return __mosek_2fusion_2BaseVariable__reshape(_2390_dim0,_2391_dim1,_2392_dim2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2390_dim0,int32_t _2391_dim1,int32_t _2392_dim2)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2390_dim0),(int32_t)(_2391_dim1),(int32_t)(_2392_dim2)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:733:6-98
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2393_dim0,int32_t _2394_dim1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2393_dim0,_2394_dim1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2393_dim0,int32_t _2394_dim1) { return __mosek_2fusion_2BaseVariable__reshape(_2393_dim0,_2394_dim1); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2393_dim0,int32_t _2394_dim1) { return __mosek_2fusion_2BaseVariable__reshape(_2393_dim0,_2394_dim1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2393_dim0,int32_t _2394_dim1)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2393_dim0),(int32_t)(_2394_dim1)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:732:6-82
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(int32_t _2395_dim0) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2395_dim0); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(int32_t _2395_dim0) { return __mosek_2fusion_2BaseVariable__reshape(_2395_dim0); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(int32_t _2395_dim0) { return __mosek_2fusion_2BaseVariable__reshape(_2395_dim0); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(int32_t _2395_dim0)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2395_dim0)}));
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.reshape
// Method mosek.fusion.BaseVariable.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:719:6-730:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2396_shape) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__reshape(_2396_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2396_shape) { return __mosek_2fusion_2BaseVariable__reshape(_2396_shape); }
/* override: mosek.fusion.Variable.reshape*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2396_shape) { return __mosek_2fusion_2BaseVariable__reshape(_2396_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2396_shape)
{
  _checked_ptr_1<int32_t> _ptr__2396_shape(_2396_shape ? _2396_shape->raw() : nullptr, _2396_shape ? _2396_shape->size(0) : 0);
  int64_t _2397_shapesize = (int)1;
  int32_t _10194_ = (int)0;
  int32_t _10195_ = (int)((_2396_shape)->size(0));
  for (int32_t _2398_i = _10194_; _2398_i < _10195_; ++_2398_i)
  {
    {
      if ((_ptr__2396_shape[_2398_i] < (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid shape"));
        }
      }
      {}
      _2397_shapesize *= _ptr__2396_shape[_2398_i];
    }
  }
  if ((_2397_shapesize != _pubthis->getSize()))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Shape size does not match variable size"));
    }
  }
  {}
  std::vector<int32_t> _10196_;
  for (int32_t _2399_i = (int)0, _10197_ = (int)((_2396_shape)->size(0)); _2399_i < _10197_ ; ++_2399_i)
  {
    _10196_.push_back(_ptr__2396_shape[_2399_i]);
  }
  auto _10198_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10196_.size()),_10196_.begin(),_10196_.end()));
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_10198_,sparsity,basevar_nativeidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.reshape

// Begin mosek.fusion.BaseVariable.setLevel
// Method mosek.fusion.BaseVariable.setLevel @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:705:6-717:7
void mosek::fusion::BaseVariable :: setLevel(std::shared_ptr< monty::ndarray< double,1 > > _2400_v) { mosek::fusion::p_BaseVariable::_get_impl(this)->setLevel(_2400_v); }
void mosek::fusion::p_BaseVariable::setLevel(std::shared_ptr< monty::ndarray< double,1 > > _2400_v)
{
  _checked_ptr_1<double> _ptr__2400_v(_2400_v ? _2400_v->raw() : nullptr, _2400_v ? _2400_v->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      if (((int)((_2400_v)->size(0)) != (int)((basevar_nativeidxs)->size(0))))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Invalid length of solution values vector"));
        }
      }
      {}
      model->setVariableValues(true,basevar_nativeidxs,_2400_v);
    }
  }
  else
  {
    {
      if (((int)((_2400_v)->size(0)) != _pubthis->getSize()))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Invalid length of solution values vector"));
        }
      }
      {}
      std::vector<double> _10199_;
      for (int32_t _2402_i = (int)0, _10200_ = (int)((basevar_nativeidxs)->size(0)); _2402_i < _10200_ ; ++_2402_i)
      {
        _10199_.push_back(_ptr__2400_v[((*sparsity)(_2402_i))]);
      }
      auto _10201_ = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_10199_.size()),_10199_.begin(),_10199_.end()));
      std::shared_ptr< monty::ndarray< double,1 > > _2401_v = _10201_;
      _checked_ptr_1<double> _ptr__2401_v(_2401_v ? _2401_v->raw() : nullptr,_2401_v ? _2401_v->size(0) : 0);
      model->setVariableValues(true,basevar_nativeidxs,_2401_v);
    }
  }
}
// End mosek.fusion.BaseVariable.setLevel

// Begin mosek.fusion.BaseVariable.getModel
// Method mosek.fusion.BaseVariable.getModel @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:703:6-55
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__getModel() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__getModel(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable::getModel() { return __mosek_2fusion_2BaseVariable__getModel(); }
/* override: mosek.fusion.Variable.getModel*/
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__getModel() { return __mosek_2fusion_2BaseVariable__getModel(); }
monty::rc_ptr< ::mosek::fusion::Model > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__getModel()
{
  return model;
}
// End mosek.fusion.BaseVariable.getModel

// Begin mosek.fusion.BaseVariable.getND
// Method mosek.fusion.BaseVariable.getND @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:702:6-56
int32_t mosek::fusion::BaseVariable :: getND() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getND(); }
int32_t mosek::fusion::p_BaseVariable::getND()
{
  return (int)((shape)->size(0));
}
// End mosek.fusion.BaseVariable.getND

// Begin mosek.fusion.BaseVariable.getDim
// Method mosek.fusion.BaseVariable.getDim @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:701:6-62
int32_t mosek::fusion::BaseVariable :: getDim(int32_t _2403_d) { return mosek::fusion::p_BaseVariable::_get_impl(this)->getDim(_2403_d); }
int32_t mosek::fusion::p_BaseVariable::getDim(int32_t _2403_d)
{
  return ((*shape)(_2403_d));
}
// End mosek.fusion.BaseVariable.getDim

// Begin mosek.fusion.BaseVariable.getShape
// Method mosek.fusion.BaseVariable.getShape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:700:6-79
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::BaseVariable :: getShape() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getShape(); }
std::shared_ptr< monty::ndarray< int32_t,1 > > mosek::fusion::p_BaseVariable::getShape()
{
  return ::mosek::fusion::Utils::Tools::arraycopy(shape);
}
// End mosek.fusion.BaseVariable.getShape

// Begin mosek.fusion.BaseVariable.getSize
// Method mosek.fusion.BaseVariable.getSize @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:699:6-107
int64_t mosek::fusion::BaseVariable :: getSize() { return mosek::fusion::p_BaseVariable::_get_impl(this)->getSize(); }
int64_t mosek::fusion::p_BaseVariable::getSize()
{
  int64_t _2404_sz = (int64_t)1;
  int32_t _10202_ = (int)0;
  int32_t _10203_ = (int)((shape)->size(0));
  for (int32_t _2405_i = _10202_; _2405_i < _10203_; ++_2405_i)
  {
    {
      _2404_sz *= ((*shape)(_2405_i));
    }
  }
  return _2404_sz;
}
// End mosek.fusion.BaseVariable.getSize

// Begin mosek.fusion.BaseVariable.dual
// Method mosek.fusion.BaseVariable.dual @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:686:6-696:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BaseVariable :: dual() { return mosek::fusion::p_BaseVariable::_get_impl(this)->dual(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BaseVariable::dual()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2406_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2406_res(_2406_res ? _2406_res->raw() : nullptr,_2406_res ? _2406_res->size(0) : 0);
  model->getVariableValues(false,basevar_nativeidxs,_2406_res,(int)0);
  if ((NULL != sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _2407_spread_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<double> _ptr__2407_spread_res(_2407_spread_res ? _2407_spread_res->raw() : nullptr,_2407_spread_res ? _2407_spread_res->size(0) : 0);
      int32_t _10204_ = (int)0;
      int32_t _10205_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2408_i = _10204_; _2408_i < _10205_; ++_2408_i)
      {
        {
          _ptr__2407_spread_res[((*sparsity)(_2408_i))] = _ptr__2406_res[_2408_i];
        }
      }
      _2406_res = _2407_spread_res;
      _ptr__2406_res.update(_2406_res ? _2406_res->raw() : nullptr, _2406_res ? _2406_res->size(0) : 0);
    }
  }
  {}
  return _2406_res;
}
// End mosek.fusion.BaseVariable.dual

// Begin mosek.fusion.BaseVariable.level
// Method mosek.fusion.BaseVariable.level @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:674:6-684:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BaseVariable :: level() { return mosek::fusion::p_BaseVariable::_get_impl(this)->level(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BaseVariable::level()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2409_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
  _checked_ptr_1<double> _ptr__2409_res(_2409_res ? _2409_res->raw() : nullptr,_2409_res ? _2409_res->size(0) : 0);
  model->getVariableValues(true,basevar_nativeidxs,_2409_res,(int)0);
  if ((NULL != sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< double,1 > > _2410_spread_res = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<double> _ptr__2410_spread_res(_2410_spread_res ? _2410_spread_res->raw() : nullptr,_2410_spread_res ? _2410_spread_res->size(0) : 0);
      int32_t _10206_ = (int)0;
      int32_t _10207_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2411_i = _10206_; _2411_i < _10207_; ++_2411_i)
      {
        {
          _ptr__2410_spread_res[((*sparsity)(_2411_i))] = _ptr__2409_res[_2411_i];
        }
      }
      _2409_res = _2410_spread_res;
      _ptr__2409_res.update(_2409_res ? _2409_res->raw() : nullptr, _2409_res ? _2409_res->size(0) : 0);
    }
  }
  {}
  return _2409_res;
}
// End mosek.fusion.BaseVariable.level

// Begin mosek.fusion.BaseVariable.makeContinuous
// Method mosek.fusion.BaseVariable.makeContinuous @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:670:6-59
void mosek::fusion::BaseVariable :: makeContinuous() { mosek::fusion::p_BaseVariable::_get_impl(this)->makeContinuous(); }
void mosek::fusion::p_BaseVariable::makeContinuous()
{
  _pubthis->make_continuous();
}
// End mosek.fusion.BaseVariable.makeContinuous

// Begin mosek.fusion.BaseVariable.makeInteger
// Method mosek.fusion.BaseVariable.makeInteger @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:669:6-53
void mosek::fusion::BaseVariable :: makeInteger() { mosek::fusion::p_BaseVariable::_get_impl(this)->makeInteger(); }
void mosek::fusion::p_BaseVariable::makeInteger()
{
  _pubthis->make_integer();
}
// End mosek.fusion.BaseVariable.makeInteger

// Begin mosek.fusion.BaseVariable.transpose
// Method mosek.fusion.BaseVariable.transpose @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:630:6-665:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__transpose() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::transpose() { return __mosek_2fusion_2BaseVariable__transpose(); }
/* override: mosek.fusion.Variable.transpose*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__transpose() { return __mosek_2fusion_2BaseVariable__transpose(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__transpose()
{
  if (((int)((shape)->size(0)) == (int)1))
  {
    {
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)(((*shape)((int)0)))}),sparsity,basevar_nativeidxs)).as<::mosek::fusion::Variable>();
    }
  }
  else if(((int)((shape)->size(0)) > (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid shape for operation"));
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2412_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
      _checked_ptr_1<int64_t> _ptr__2412_idxs(_2412_idxs ? _2412_idxs->raw() : nullptr,_2412_idxs ? _2412_idxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _10208_;
      bool _10209_ = (NULL == sparsity.get());
      if (_10209_)
      {
        _10208_ = nullptr;
      }
      else
      {
        _10208_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((basevar_nativeidxs)->size(0)))));
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2413_sp = _10208_;
      _checked_ptr_1<int64_t> _ptr__2413_sp(_2413_sp ? _2413_sp->raw() : nullptr,_2413_sp ? _2413_sp->size(0) : 0);
      if ((NULL == sparsity.get()))
      {
        {
          int32_t _2414_k = (int)0;
          int32_t _10210_ = (int)0;
          int32_t _10211_ = ((*shape)((int)1));
          for (int32_t _2415_j = _10210_; _2415_j < _10211_; ++_2415_j)
          {
            {
              int32_t _10212_ = (int)0;
              int32_t _10213_ = ((*shape)((int)0));
              for (int32_t _2416_i = _10212_; _2416_i < _10213_; ++_2416_i)
              {
                {
                  _ptr__2412_idxs[_2414_k] = ((*basevar_nativeidxs)(safe_add( safe_mul( _2416_i,((*shape)((int)1)) ),_2415_j )));
                  ++ _2414_k;
                }
              }
            }
          }
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2417_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( ((*shape)((int)1)),(int)1 ))));
          _checked_ptr_1<int32_t> _ptr__2417_ptr(_2417_ptr ? _2417_ptr->raw() : nullptr,_2417_ptr ? _2417_ptr->size(0) : 0);
          int32_t _2418_dim0 = ((*shape)((int)0));
          int32_t _2419_dim1 = ((*shape)((int)1));
          int32_t _10214_ = (int)0;
          int32_t _10215_ = (int)((sparsity)->size(0));
          for (int32_t _2420_i = _10214_; _2420_i < _10215_; ++_2420_i)
          {
            {
              ++ _ptr__2417_ptr[safe_add( (((*sparsity)(_2420_i)) % _2419_dim1),(int)1 )];
            }
          }
          int32_t _10216_ = (int)0;
          int32_t _10217_ = _2419_dim1;
          for (int32_t _2421_i = _10216_; _2421_i < _10217_; ++_2421_i)
          {
            {
              _ptr__2417_ptr[safe_add( _2421_i,(int)1 )] += _ptr__2417_ptr[_2421_i];
            }
          }
          int32_t _10218_ = (int)0;
          int32_t _10219_ = (int)((sparsity)->size(0));
          for (int32_t _2422_i = _10218_; _2422_i < _10219_; ++_2422_i)
          {
            {
              int64_t _2423_ii = (((*sparsity)(_2422_i)) % _2419_dim1);
              _ptr__2412_idxs[_ptr__2417_ptr[_2423_ii]] = ((*basevar_nativeidxs)(_2422_i));
              _ptr__2413_sp[_ptr__2417_ptr[_2423_ii]] = safe_add( (((*sparsity)(_2422_i)) / _2419_dim1),safe_mul( (((*sparsity)(_2422_i)) % _2419_dim1),_2418_dim0 ) );
              ++ _ptr__2417_ptr[_2423_ii];
            }
          }
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(((*shape)((int)1))),(int32_t)(((*shape)((int)0)))}),_2413_sp,_2412_idxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.transpose

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:619:6-622:40
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2424_i0,int32_t _2425_i1,int32_t _2426_i2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2424_i0,_2425_i1,_2426_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2424_i0,int32_t _2425_i1,int32_t _2426_i2) { return __mosek_2fusion_2BaseVariable__index(_2424_i0,_2425_i1,_2426_i2); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2424_i0,int32_t _2425_i1,int32_t _2426_i2) { return __mosek_2fusion_2BaseVariable__index(_2424_i0,_2425_i1,_2426_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2424_i0,int32_t _2425_i1,int32_t _2426_i2)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2424_i0),(int32_t)(_2425_i1),(int32_t)(_2426_i2)}));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:614:6-616:37
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2427_i0,int32_t _2428_i1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2427_i0,_2428_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2427_i0,int32_t _2428_i1) { return __mosek_2fusion_2BaseVariable__index(_2427_i0,_2428_i1); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2427_i0,int32_t _2428_i1) { return __mosek_2fusion_2BaseVariable__index(_2427_i0,_2428_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2427_i0,int32_t _2428_i1)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2427_i0),(int32_t)(_2428_i1)}));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:606:6-611:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2429_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2429_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2429_index) { return __mosek_2fusion_2BaseVariable__index(_2429_index); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2429_index) { return __mosek_2fusion_2BaseVariable__index(_2429_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2429_index)
{
  _checked_ptr_1<int32_t> _ptr__2429_index(_2429_index ? _2429_index->raw() : nullptr, _2429_index ? _2429_index->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2430_midx = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)1,(int)((_2429_index)->size(0)))));
  int32_t _10220_ = (int)0;
  int32_t _10221_ = (int)((_2429_index)->size(0));
  for (int32_t _2431_i = _10220_; _2431_i < _10221_; ++_2431_i)
  {
    {
      ((*_2430_midx)((int)0,_2431_i)) = _ptr__2429_index[_2431_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2430_midx)->__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.index
// Method mosek.fusion.BaseVariable.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:600:6-603:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__index(int32_t _2432_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__index(_2432_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::index(int32_t _2432_index) { return __mosek_2fusion_2BaseVariable__index(_2432_index); }
/* override: mosek.fusion.Variable.index*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__index(int32_t _2432_index) { return __mosek_2fusion_2BaseVariable__index(_2432_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__index(int32_t _2432_index)
{
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2432_index)}))->__mosek_2fusion_2Variable__reshape(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape((int)0))));
}
// End mosek.fusion.BaseVariable.index

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:579:6-597:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2433_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2434_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2435_i2) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2433_i0,_2434_i1,_2435_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2433_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2434_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2435_i2) { return __mosek_2fusion_2BaseVariable__pick(_2433_i0,_2434_i1,_2435_i2); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2433_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2434_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2435_i2) { return __mosek_2fusion_2BaseVariable__pick(_2433_i0,_2434_i1,_2435_i2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2433_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2434_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2435_i2)
{
  _checked_ptr_1<int32_t> _ptr__2433_i0(_2433_i0 ? _2433_i0->raw() : nullptr, _2433_i0 ? _2433_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2434_i1(_2434_i1 ? _2434_i1->raw() : nullptr, _2434_i1 ? _2434_i1->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2435_i2(_2435_i2 ? _2435_i2->raw() : nullptr, _2435_i2 ? _2435_i2->size(0) : 0);
  if (((int)((shape)->size(0)) != (int)3))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  if ((((int)((_2433_i0)->size(0)) != (int)((_2434_i1)->size(0))) || ((int)((_2433_i0)->size(0)) != (int)((_2435_i2)->size(0)))))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Mismatching argument lengths"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2436_midxs = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)((_2433_i0)->size(0)),(int)3)));
  int32_t _10222_ = (int)0;
  int32_t _10223_ = (int)((_2433_i0)->size(0));
  for (int32_t _2437_i = _10222_; _2437_i < _10223_; ++_2437_i)
  {
    {
      ((*_2436_midxs)(_2437_i,(int)0)) = _ptr__2433_i0[_2437_i];
      ((*_2436_midxs)(_2437_i,(int)1)) = _ptr__2434_i1[_2437_i];
      ((*_2436_midxs)(_2437_i,(int)2)) = _ptr__2435_i2[_2437_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2436_midxs);
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:563:6-577:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2438_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2439_i1) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2438_i0,_2439_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2438_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2439_i1) { return __mosek_2fusion_2BaseVariable__pick(_2438_i0,_2439_i1); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2438_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2439_i1) { return __mosek_2fusion_2BaseVariable__pick(_2438_i0,_2439_i1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2438_i0,std::shared_ptr< monty::ndarray< int32_t,1 > > _2439_i1)
{
  _checked_ptr_1<int32_t> _ptr__2438_i0(_2438_i0 ? _2438_i0->raw() : nullptr, _2438_i0 ? _2438_i0->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2439_i1(_2439_i1 ? _2439_i1->raw() : nullptr, _2439_i1 ? _2439_i1->size(0) : 0);
  if ((((int)((shape)->size(0)) != (int)2) || ((int)((_2438_i0)->size(0)) != (int)((_2439_i1)->size(0)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int32_t,2 > > _2440_midxs = std::shared_ptr< monty::ndarray< int32_t,2 > >(new monty::ndarray< int32_t,2 >(monty::shape((int)((_2438_i0)->size(0)),(int)2)));
  int32_t _10224_ = (int)0;
  int32_t _10225_ = (int)((_2438_i0)->size(0));
  for (int32_t _2441_i = _10224_; _2441_i < _10225_; ++_2441_i)
  {
    {
      ((*_2440_midxs)(_2441_i,(int)0)) = _ptr__2438_i0[_2441_i];
      ((*_2440_midxs)(_2441_i,(int)1)) = _ptr__2439_i1[_2441_i];
    }
  }
  return _pubthis->__mosek_2fusion_2BaseVariable__pick(_2440_midxs);
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:502:6-561:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2442_midxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2442_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2442_midxs) { return __mosek_2fusion_2BaseVariable__pick(_2442_midxs); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2442_midxs) { return __mosek_2fusion_2BaseVariable__pick(_2442_midxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2442_midxs)
{
  if (((int)((shape)->size(0)) != (int)((_2442_midxs)->size(1))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  bool _2443_invalidindex = false;
  int32_t _10226_ = (int)0;
  int32_t _10227_ = (int)((_2442_midxs)->size(0));
  for (int32_t _2444_i = _10226_; _2444_i < _10227_; ++_2444_i)
  {
    {
      int32_t _10228_ = (int)0;
      int32_t _10229_ = (int)((_2442_midxs)->size(1));
      for (int32_t _2445_j = _10228_; _2445_j < _10229_; ++_2445_j)
      {
        {
          if (((((*_2442_midxs)(_2444_i,_2445_j)) < (int)0) || (((*_2442_midxs)(_2444_i,_2445_j)) >= ((*shape)(_2445_j)))))
          {
            {
              _2443_invalidindex = true;
            }
          }
          {}
        }
      }
    }
  }
  if (_2443_invalidindex)
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  int32_t _2446_nd = (int)((shape)->size(0));
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2447_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2446_nd)));
  _checked_ptr_1<int64_t> _ptr__2447_strides(_2447_strides ? _2447_strides->raw() : nullptr,_2447_strides ? _2447_strides->size(0) : 0);
  _ptr__2447_strides[(_2446_nd - (int)1)] = (int)1;
  int32_t _10230_ = (int)1;
  int32_t _10231_ = _2446_nd;
  for (int32_t _2448_i = _10230_; _2448_i < _10231_; ++_2448_i)
  {
    {
      _ptr__2447_strides[((_2446_nd - _2448_i) - (int)1)] = safe_mul( _ptr__2447_strides[(_2446_nd - _2448_i)],((*shape)((_2446_nd - _2448_i))) );
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2449_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)((_2442_midxs)->size(0)))});
  _checked_ptr_1<int32_t> _ptr__2449_rshape(_2449_rshape ? _2449_rshape->raw() : nullptr,_2449_rshape ? _2449_rshape->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2450_idxs;
  _checked_ptr_1<int64_t> _ptr__2450_idxs(_2450_idxs ? _2450_idxs->raw() : nullptr,_2450_idxs ? _2450_idxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2451_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__2451_sp(_2451_sp ? _2451_sp->raw() : nullptr,_2451_sp ? _2451_sp->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      _2450_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape((int)((_2442_midxs)->size(0)))));
      _ptr__2450_idxs.update(_2450_idxs ? _2450_idxs->raw() : nullptr, _2450_idxs ? _2450_idxs->size(0) : 0);
      {
        int32_t _10232_ = (int)0;
        int32_t _10233_ = (int)((_2442_midxs)->size(0));
        for (int32_t _2452_i = _10232_; _2452_i < _10233_; ++_2452_i)
        {
          {
            int64_t _2453_ii = (int64_t)0;
            int32_t _10234_ = (int)0;
            int32_t _10235_ = _2446_nd;
            for (int32_t _2454_j = _10234_; _2454_j < _10235_; ++_2454_j)
            {
              {
                _2453_ii += safe_mul( _ptr__2447_strides[_2454_j],((*_2442_midxs)(_2452_i,_2454_j)) );
              }
            }
            _ptr__2450_idxs[_2452_i] = ((*basevar_nativeidxs)(_2453_ii));
          }
        }
      }
    }
  }
  else
  {
    {
      int32_t _2455_nnz = (int)0;
      int32_t _10236_ = (int)0;
      int32_t _10237_ = (int)((_2442_midxs)->size(0));
      for (int32_t _2456_i = _10236_; _2456_i < _10237_; ++_2456_i)
      {
        {
          int64_t _2457_ii = (int64_t)0;
          int32_t _10238_ = (int)0;
          int32_t _10239_ = _2446_nd;
          for (int32_t _2458_j = _10238_; _2458_j < _10239_; ++_2458_j)
          {
            {
              _2457_ii += safe_mul( _ptr__2447_strides[_2458_j],((*_2442_midxs)(_2456_i,_2458_j)) );
            }
          }
          if ((::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2457_ii) >= (int)0))
          {
            {
              ++ _2455_nnz;
            }
          }
          {}
        }
      }
      _2450_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2455_nnz)));
      _ptr__2450_idxs.update(_2450_idxs ? _2450_idxs->raw() : nullptr, _2450_idxs ? _2450_idxs->size(0) : 0);
      _2451_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2455_nnz)));
      _ptr__2451_sp.update(_2451_sp ? _2451_sp->raw() : nullptr, _2451_sp ? _2451_sp->size(0) : 0);
      {
        int32_t _2459_k = (int)0;
        int32_t _10240_ = (int)0;
        int32_t _10241_ = (int)((_2442_midxs)->size(0));
        for (int32_t _2460_i = _10240_; _2460_i < _10241_; ++_2460_i)
        {
          {
            int64_t _2461_ii = (int64_t)0;
            int32_t _10242_ = (int)0;
            int32_t _10243_ = _2446_nd;
            for (int32_t _2462_j = _10242_; _2462_j < _10243_; ++_2462_j)
            {
              {
                _2461_ii += safe_mul( _ptr__2447_strides[_2462_j],((*_2442_midxs)(_2460_i,_2462_j)) );
              }
            }
            int32_t _2463_iidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2461_ii);
            if ((_2463_iidx >= (int)0))
            {
              {
                _ptr__2450_idxs[_2459_k] = ((*basevar_nativeidxs)(_2463_iidx));
                _ptr__2451_sp[_2459_k] = _2460_i;
                ++ _2459_k;
              }
            }
            {}
          }
        }
      }
      if ((_2455_nnz == (int)((_2442_midxs)->size(0))))
      {
        {
          _2451_sp = nullptr;
          _ptr__2451_sp.update(_2451_sp ? _2451_sp->raw() : nullptr, _2451_sp ? _2451_sp->size(0) : 0);
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2449_rshape,_2451_sp,_2450_idxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.pick
// Method mosek.fusion.BaseVariable.pick @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:460:6-500:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2464_idxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__pick(_2464_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2464_idxs) { return __mosek_2fusion_2BaseVariable__pick(_2464_idxs); }
/* override: mosek.fusion.Variable.pick*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2464_idxs) { return __mosek_2fusion_2BaseVariable__pick(_2464_idxs); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2464_idxs)
{
  _checked_ptr_1<int32_t> _ptr__2464_idxs(_2464_idxs ? _2464_idxs->raw() : nullptr, _2464_idxs ? _2464_idxs->size(0) : 0);
  if (((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  bool _2465_invalidindex = false;
  int32_t _10244_ = (int)0;
  int32_t _10245_ = (int)((_2464_idxs)->size(0));
  for (int32_t _2466_i = _10244_; _2466_i < _10245_; ++_2466_i)
  {
    {
      if (((_ptr__2464_idxs[_2466_i] < (int)0) || (_ptr__2464_idxs[_2466_i] >= ((*shape)((int)0)))))
      {
        {
          _2465_invalidindex = true;
        }
      }
      {}
    }
  }
  if (_2465_invalidindex)
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Invalid index"));
    }
  }
  {}
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2467_nidxs;
  _checked_ptr_1<int64_t> _ptr__2467_nidxs(_2467_nidxs ? _2467_nidxs->raw() : nullptr,_2467_nidxs ? _2467_nidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2468_sp = nullptr;
  _checked_ptr_1<int64_t> _ptr__2468_sp(_2468_sp ? _2468_sp->raw() : nullptr,_2468_sp ? _2468_sp->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      std::vector<int64_t> _10246_;
      for (int32_t _2469_i = (int)0, _10247_ = (int)((_2464_idxs)->size(0)); _2469_i < _10247_ ; ++_2469_i)
      {
        _10246_.push_back(((*basevar_nativeidxs)(_ptr__2464_idxs[_2469_i])));
      }
      auto _10248_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10246_.size()),_10246_.begin(),_10246_.end()));
      _2467_nidxs = _10248_;
      _ptr__2467_nidxs.update(_2467_nidxs ? _2467_nidxs->raw() : nullptr, _2467_nidxs ? _2467_nidxs->size(0) : 0);
    }
  }
  else
  {
    {
      int32_t _2470_nnz = (int)0;
      int32_t _10249_ = (int)0;
      int32_t _10250_ = (int)((_2464_idxs)->size(0));
      for (int32_t _2471_i = _10249_; _2471_i < _10250_; ++_2471_i)
      {
        {
          if ((::mosek::fusion::p_CommonTools::binarySearch(sparsity,(int64_t)(_ptr__2464_idxs[_2471_i])) >= (int)0))
          {
            {
              ++ _2470_nnz;
            }
          }
          {}
        }
      }
      _2467_nidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2470_nnz)));
      _ptr__2467_nidxs.update(_2467_nidxs ? _2467_nidxs->raw() : nullptr, _2467_nidxs ? _2467_nidxs->size(0) : 0);
      _2468_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2470_nnz)));
      _ptr__2468_sp.update(_2468_sp ? _2468_sp->raw() : nullptr, _2468_sp ? _2468_sp->size(0) : 0);
      int32_t _2472_k = (int)0;
      int32_t _10251_ = (int)0;
      int32_t _10252_ = (int)((_2464_idxs)->size(0));
      for (int32_t _2473_i = _10251_; _2473_i < _10252_; ++_2473_i)
      {
        {
          int32_t _2474_j = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,(int64_t)(_ptr__2464_idxs[_2473_i]));
          if ((_2474_j >= (int)0))
          {
            {
              _ptr__2468_sp[_2472_k] = _2473_i;
              _ptr__2467_nidxs[_2472_k] = ((*basevar_nativeidxs)(_2474_j));
              ++ _2472_k;
            }
          }
          {}
        }
      }
      if ((_2470_nnz == (int)((_2464_idxs)->size(0))))
      {
        {
          _2468_sp = nullptr;
          _ptr__2468_sp.update(_2468_sp ? _2468_sp->raw() : nullptr, _2468_sp ? _2468_sp->size(0) : 0);
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)((_2464_idxs)->size(0)))}),_2468_sp,_2467_nidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.BaseVariable.pick

// Begin mosek.fusion.BaseVariable.antidiag
// Method mosek.fusion.BaseVariable.antidiag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:446:6-457:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__antidiag(int32_t _2475_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__antidiag(_2475_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::antidiag(int32_t _2475_index) { return __mosek_2fusion_2BaseVariable__antidiag(_2475_index); }
/* override: mosek.fusion.Variable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__antidiag(int32_t _2475_index) { return __mosek_2fusion_2BaseVariable__antidiag(_2475_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag(int32_t _2475_index)
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("antidiag() only works on two-dimensional objects"));
    }
  }
  else if(((_2475_index >= ((*shape)((int)1))) || ((-_2475_index) >= ((*shape)((int)2)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Diagonal index out of bounds"));
    }
  }
  {}
  if ((_2475_index >= (int)0))
  {
    {
      int32_t _10253_;
      bool _10254_ = (((*shape)((int)0)) < (((*shape)((int)1)) - _2475_index));
      if (_10254_)
      {
        _10253_ = ((*shape)((int)0));
      }
      else
      {
        _10253_ = (((*shape)((int)1)) - _2475_index);
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)(((((*shape)((int)1)) - (int)1) - _2475_index))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10253_);
    }
  }
  else
  {
    {
      int32_t _10255_;
      bool _10256_ = (((*shape)((int)1)) < safe_add( ((*shape)((int)0)),_2475_index ));
      if (_10256_)
      {
        _10255_ = ((*shape)((int)1));
      }
      else
      {
        _10255_ = safe_add( ((*shape)((int)0)),_2475_index );
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((-_2475_index)),(int32_t)((((*shape)((int)1)) - (int)1))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10255_);
    }
  }
}
// End mosek.fusion.BaseVariable.antidiag

// Begin mosek.fusion.BaseVariable.antidiag
// Method mosek.fusion.BaseVariable.antidiag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:438:6-444:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__antidiag() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::antidiag() { return __mosek_2fusion_2BaseVariable__antidiag(); }
/* override: mosek.fusion.Variable.antidiag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__antidiag() { return __mosek_2fusion_2BaseVariable__antidiag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__antidiag()
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("antidiag() only works on two-dimensional objects"));
    }
  }
  {}
  int32_t _10257_;
  bool _10258_ = (((*shape)((int)0)) < ((*shape)((int)1)));
  if (_10258_)
  {
    _10257_ = ((*shape)((int)0));
  }
  else
  {
    _10257_ = ((*shape)((int)1));
  }
  return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)((((*shape)((int)1)) - (int)1))}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((-(int)1))}),_10257_);
}
// End mosek.fusion.BaseVariable.antidiag

// Begin mosek.fusion.BaseVariable.diag
// Method mosek.fusion.BaseVariable.diag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:425:6-436:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__diag(int32_t _2476_index) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__diag(_2476_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::diag(int32_t _2476_index) { return __mosek_2fusion_2BaseVariable__diag(_2476_index); }
/* override: mosek.fusion.Variable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__diag(int32_t _2476_index) { return __mosek_2fusion_2BaseVariable__diag(_2476_index); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag(int32_t _2476_index)
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("diag() only works on two-dimensional objects"));
    }
  }
  else if(((_2476_index >= ((*shape)((int)1))) || ((-_2476_index) >= ((*shape)((int)2)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Diagonal index out of bounds"));
    }
  }
  {}
  if ((_2476_index >= (int)0))
  {
    {
      int32_t _10259_;
      bool _10260_ = (((*shape)((int)0)) < (((*shape)((int)1)) - _2476_index));
      if (_10260_)
      {
        _10259_ = ((*shape)((int)0));
      }
      else
      {
        _10259_ = (((*shape)((int)1)) - _2476_index);
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)(_2476_index)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10259_);
    }
  }
  else
  {
    {
      int32_t _10261_;
      bool _10262_ = (((*shape)((int)1)) < safe_add( ((*shape)((int)0)),_2476_index ));
      if (_10262_)
      {
        _10261_ = ((*shape)((int)1));
      }
      else
      {
        _10261_ = safe_add( ((*shape)((int)0)),_2476_index );
      }
      return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((-_2476_index)),(int32_t)((int)0)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10261_);
    }
  }
}
// End mosek.fusion.BaseVariable.diag

// Begin mosek.fusion.BaseVariable.diag
// Method mosek.fusion.BaseVariable.diag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:418:6-423:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__diag() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::diag() { return __mosek_2fusion_2BaseVariable__diag(); }
/* override: mosek.fusion.Variable.diag*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__diag() { return __mosek_2fusion_2BaseVariable__diag(); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__diag()
{
  if (((int)((shape)->size(0)) != (int)2))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("diag() only works on two-dimensional objects"));
    }
  }
  {}
  int32_t _10263_;
  bool _10264_ = (((*shape)((int)0)) < ((*shape)((int)1)));
  if (_10264_)
  {
    _10263_ = ((*shape)((int)0));
  }
  else
  {
    _10263_ = ((*shape)((int)1));
  }
  return __mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)0),(int32_t)((int)0)}),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int)1),(int32_t)((int)1)}),_10263_);
}
// End mosek.fusion.BaseVariable.diag

// Begin mosek.fusion.BaseVariable.general_diag
// Method mosek.fusion.BaseVariable.general_diag @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:368:6-415:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__general_diag(std::shared_ptr< monty::ndarray< int32_t,1 > > _2477_firstidx,std::shared_ptr< monty::ndarray< int32_t,1 > > _2478_step,int32_t _2479_num)
{
  _checked_ptr_1<int32_t> _ptr__2477_firstidx(_2477_firstidx ? _2477_firstidx->raw() : nullptr, _2477_firstidx ? _2477_firstidx->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2478_step(_2478_step ? _2478_step->raw() : nullptr, _2478_step ? _2478_step->size(0) : 0);
  int32_t _2480_nd = (int)((shape)->size(0));
  if (((_2480_nd != (int)((_2477_firstidx)->size(0))) || (_2480_nd != (int)((_2478_step)->size(0)))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid start or step definition"));
    }
  }
  {}
  int32_t _10265_ = (int)0;
  int32_t _10266_ = _2480_nd;
  for (int32_t _2481_i = _10265_; _2481_i < _10266_; ++_2481_i)
  {
    {
      if (((_ptr__2477_firstidx[_2481_i] < (int)0) || (_ptr__2477_firstidx[_2481_i] >= ((*shape)(_2481_i)))))
      {
        {
          throw ::mosek::fusion::LengthError(std::string ("Start element is out of bounds"));
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2482_strides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2480_nd)));
  _checked_ptr_1<int64_t> _ptr__2482_strides(_2482_strides ? _2482_strides->raw() : nullptr,_2482_strides ? _2482_strides->size(0) : 0);
  _ptr__2482_strides[(_2480_nd - (int)1)] = (int)1;
  int32_t _10267_ = (int)1;
  int32_t _10268_ = _2480_nd;
  for (int32_t _2483_i = _10267_; _2483_i < _10268_; ++_2483_i)
  {
    {
      _ptr__2482_strides[((_2480_nd - _2483_i) - (int)1)] = safe_mul( _ptr__2482_strides[(_2480_nd - _2483_i)],((*shape)((_2480_nd - _2483_i))) );
    }
  }
  int64_t _2484_startidx = (int64_t)0;
  int32_t _10269_ = (int)0;
  int32_t _10270_ = (int)((_2478_step)->size(0));
  for (int32_t _2485_i = _10269_; _2485_i < _10270_; ++_2485_i)
  {
    {
      _2484_startidx += safe_mul( _ptr__2477_firstidx[_2485_i],_ptr__2482_strides[_2485_i] );
    }
  }
  int64_t _2486_stepstride = (int64_t)0;
  int32_t _10271_ = (int)0;
  int32_t _10272_ = (int)((_2478_step)->size(0));
  for (int32_t _2487_i = _10271_; _2487_i < _10272_; ++_2487_i)
  {
    {
      _2486_stepstride += safe_mul( _ptr__2478_step[_2487_i],_ptr__2482_strides[_2487_i] );
    }
  }
  if ((NULL == sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2488_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2479_num)));
      _checked_ptr_1<int64_t> _ptr__2488_residxs(_2488_residxs ? _2488_residxs->raw() : nullptr,_2488_residxs ? _2488_residxs->size(0) : 0);
      int32_t _10273_ = (int)0;
      int32_t _10274_ = _2479_num;
      for (int32_t _2489_i = _10273_; _2489_i < _10274_; ++_2489_i)
      {
        {
          _ptr__2488_residxs[_2489_i] = ((*basevar_nativeidxs)(safe_add( _2484_startidx,safe_mul( _2486_stepstride,_2489_i ) )));
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2479_num)}),nullptr,_2488_residxs)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2490_numnz = (int)0;
      int32_t _10275_ = (int)0;
      int32_t _10276_ = _2479_num;
      for (int32_t _2491_i = _10275_; _2491_i < _10276_; ++_2491_i)
      {
        {
          int64_t _2492_residx = safe_add( _2484_startidx,safe_mul( _2486_stepstride,_2491_i ) );
          int32_t _2493_spidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2492_residx);
          if ((_2493_spidx >= (int)0))
          {
            {
              ++ _2490_numnz;
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2494_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2490_numnz)));
      _checked_ptr_1<int64_t> _ptr__2494_residxs(_2494_residxs ? _2494_residxs->raw() : nullptr,_2494_residxs ? _2494_residxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2495_ressp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2490_numnz)));
      _checked_ptr_1<int64_t> _ptr__2495_ressp(_2495_ressp ? _2495_ressp->raw() : nullptr,_2495_ressp ? _2495_ressp->size(0) : 0);
      int32_t _2496_k = (int)0;
      int32_t _10277_ = (int)0;
      int32_t _10278_ = _2479_num;
      for (int32_t _2497_i = _10277_; _2497_i < _10278_; ++_2497_i)
      {
        {
          _ptr__2494_residxs[_2497_i] = ((*basevar_nativeidxs)(safe_add( _2484_startidx,safe_mul( _2486_stepstride,_2497_i ) )));
          int64_t _2498_residx = safe_add( _2484_startidx,safe_mul( _2486_stepstride,_2497_i ) );
          int32_t _2499_spidx = ::mosek::fusion::p_CommonTools::binarySearch(sparsity,_2498_residx);
          if ((_2499_spidx >= (int)0))
          {
            {
              _ptr__2495_ressp[_2496_k] = _2497_i;
              _ptr__2494_residxs[_2496_k] = ((*basevar_nativeidxs)(_2499_spidx));
              ++ _2496_k;
            }
          }
          {}
        }
      }
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2479_num)}),_2495_ressp,_2494_residxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.general_diag

// Begin mosek.fusion.BaseVariable.slice
// Method mosek.fusion.BaseVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:260:6-357:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2500_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2501_last) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__slice(_2500_first,_2501_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2500_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2501_last) { return __mosek_2fusion_2BaseVariable__slice(_2500_first,_2501_last); }
/* override: mosek.fusion.Variable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2500_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2501_last) { return __mosek_2fusion_2BaseVariable__slice(_2500_first,_2501_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2500_first,std::shared_ptr< monty::ndarray< int32_t,1 > > _2501_last)
{
  _checked_ptr_1<int32_t> _ptr__2500_first(_2500_first ? _2500_first->raw() : nullptr, _2500_first ? _2500_first->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2501_last(_2501_last ? _2501_last->raw() : nullptr, _2501_last ? _2501_last->size(0) : 0);
  if (((int)((_2500_first)->size(0)) != (int)((_2501_last)->size(0))))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
    }
  }
  else if(((int)((_2500_first)->size(0)) != (int)((shape)->size(0))))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Variable shape and slice do not match"));
    }
  }
  else
  {
    {
      int32_t _10279_ = (int)0;
      int32_t _10280_ = (int)((shape)->size(0));
      for (int32_t _2502_i = _10279_; _2502_i < _10280_; ++_2502_i)
      {
        {
          if (((_ptr__2500_first[_2502_i] < (int)0) || (_ptr__2501_last[_2502_i] > ((*shape)(_2502_i)))))
          {
            {
              throw ::mosek::fusion::IndexError(std::string ("Slice is out of bounds"));
            }
          }
          else if((_ptr__2500_first[_2502_i] > _ptr__2501_last[_2502_i]))
          {
            {
              throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
            }
          }
          {}
        }
      }
      int32_t _2503_nd = (int)((shape)->size(0));
      std::vector<int32_t> _10281_;
      for (int32_t _2505_i = (int)0, _10282_ = _2503_nd; _2505_i < _10282_ ; ++_2505_i)
      {
        _10281_.push_back((_ptr__2501_last[_2505_i] - _ptr__2500_first[_2505_i]));
      }
      auto _10283_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10281_.size()),_10281_.begin(),_10281_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2504_sliceshape = _10283_;
      _checked_ptr_1<int32_t> _ptr__2504_sliceshape(_2504_sliceshape ? _2504_sliceshape->raw() : nullptr,_2504_sliceshape ? _2504_sliceshape->size(0) : 0);
      int32_t _2506_slicesize = (int)1;
      int32_t _10284_ = (int)0;
      int32_t _10285_ = _2503_nd;
      for (int32_t _2507_i = _10284_; _2507_i < _10285_; ++_2507_i)
      {
        {
          _2506_slicesize *= (_ptr__2501_last[_2507_i] - _ptr__2500_first[_2507_i]);
        }
      }
      if ((_2506_slicesize == (int)0))
      {
        {
          return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
        }
      }
      else if((NULL != sparsity.get()))
      {
        {
          int32_t _2508_nnz = (int)0;
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2509_key = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2503_nd)));
          _checked_ptr_1<int32_t> _ptr__2509_key(_2509_key ? _2509_key->raw() : nullptr,_2509_key ? _2509_key->size(0) : 0);
          int32_t _10286_ = (int)0;
          int32_t _10287_ = (int)((sparsity)->size(0));
          for (int32_t _2510_i = _10286_; _2510_i < _10287_; ++_2510_i)
          {
            {
              int64_t _2511_rem = ((*sparsity)(_2510_i));
              int32_t _10288_ = (int)1;
              int32_t _10289_ = _2503_nd;
              for (int32_t _2512_k = _10288_; _2512_k < _10289_; ++_2512_k)
              {
                {
                  _ptr__2509_key[(_2503_nd - _2512_k)] = (int32_t)(_2511_rem % ((*shape)((_2503_nd - _2512_k))));
                  _2511_rem /= ((*shape)((_2503_nd - _2512_k)));
                }
              }
              _ptr__2509_key[(int)0] = (int32_t)_2511_rem;
              int32_t _2513_k = (int)0;
              while ( ((_ptr__2509_key[_2513_k] >= _ptr__2500_first[_2513_k]) && (_ptr__2509_key[_2513_k] < _ptr__2501_last[_2513_k])) )
              {
                {
                  ++ _2513_k;
                }
              }
              if ((_2513_k == _2503_nd))
              {
                {
                  ++ _2508_nnz;
                }
              }
              {}
            }
          }
          int32_t _2514_idx = (int)0;
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2515_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2508_nnz)));
          _checked_ptr_1<int64_t> _ptr__2515_idxs(_2515_idxs ? _2515_idxs->raw() : nullptr,_2515_idxs ? _2515_idxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2516_sp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2508_nnz)));
          _checked_ptr_1<int64_t> _ptr__2516_sp(_2516_sp ? _2516_sp->raw() : nullptr,_2516_sp ? _2516_sp->size(0) : 0);
          int32_t _10290_ = (int)0;
          int32_t _10291_ = (int)((sparsity)->size(0));
          for (int32_t _2517_i = _10290_; _2517_i < _10291_; ++_2517_i)
          {
            {
              int64_t _2518_rem = ((*sparsity)(_2517_i));
              int32_t _10292_ = (int)1;
              int32_t _10293_ = _2503_nd;
              for (int32_t _2519_k = _10292_; _2519_k < _10293_; ++_2519_k)
              {
                {
                  _ptr__2509_key[(_2503_nd - _2519_k)] = (int32_t)(_2518_rem % ((*shape)((_2503_nd - _2519_k))));
                  _2518_rem /= ((*shape)((_2503_nd - _2519_k)));
                }
              }
              _ptr__2509_key[(int)0] = (int32_t)_2518_rem;
              int32_t _2520_k = (int)0;
              while ( ((_ptr__2509_key[_2520_k] >= _ptr__2500_first[_2520_k]) && (_ptr__2509_key[_2520_k] < _ptr__2501_last[_2520_k])) )
              {
                {
                  ++ _2520_k;
                }
              }
              if ((_2520_k == _2503_nd))
              {
                {
                  _ptr__2515_idxs[_2514_idx] = ((*basevar_nativeidxs)(_2517_i));
                  int32_t _2521_spidx = (int)0;
                  int32_t _10294_ = (int)0;
                  int32_t _10295_ = _2503_nd;
                  for (int32_t _2522_i = _10294_; _2522_i < _10295_; ++_2522_i)
                  {
                    {
                      _2521_spidx = safe_add( safe_mul( _2521_spidx,_ptr__2504_sliceshape[_2522_i] ),(_ptr__2509_key[_2522_i] - _ptr__2500_first[_2522_i]) );
                    }
                  }
                  _ptr__2516_sp[_2514_idx] = _2521_spidx;
                  ++ _2514_idx;
                }
              }
              {}
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2504_sliceshape,_2516_sp,_2515_idxs)).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2523_sliceidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2506_slicesize)));
          _checked_ptr_1<int64_t> _ptr__2523_sliceidxs(_2523_sliceidxs ? _2523_sliceidxs->raw() : nullptr,_2523_sliceidxs ? _2523_sliceidxs->size(0) : 0);
          {
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2524_strides = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2503_nd)));
            _checked_ptr_1<int32_t> _ptr__2524_strides(_2524_strides ? _2524_strides->raw() : nullptr,_2524_strides ? _2524_strides->size(0) : 0);
            _ptr__2524_strides[(_2503_nd - (int)1)] = (int)1;
            int32_t _10296_ = (int)1;
            int32_t _10297_ = _2503_nd;
            for (int32_t _2525_i = _10296_; _2525_i < _10297_; ++_2525_i)
            {
              {
                _ptr__2524_strides[((_2503_nd - _2525_i) - (int)1)] = safe_mul( _ptr__2524_strides[(_2503_nd - _2525_i)],((*shape)((_2503_nd - _2525_i))) );
              }
            }
            int32_t _2526_start = (int)0;
            int32_t _10298_ = (int)0;
            int32_t _10299_ = _2503_nd;
            for (int32_t _2527_i = _10298_; _2527_i < _10299_; ++_2527_i)
            {
              {
                _2526_start += safe_mul( _ptr__2500_first[_2527_i],_ptr__2524_strides[_2527_i] );
              }
            }
            std::vector<int32_t> _10300_;
            for (int32_t _2529_i = (int)0, _10301_ = _2503_nd; _2529_i < _10301_ ; ++_2529_i)
            {
              _10300_.push_back(_2526_start);
            }
            auto _10302_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10300_.size()),_10300_.begin(),_10300_.end()));
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2528_pp = _10302_;
            _checked_ptr_1<int32_t> _ptr__2528_pp(_2528_pp ? _2528_pp->raw() : nullptr,_2528_pp ? _2528_pp->size(0) : 0);
            std::vector<int32_t> _10303_;
            for (int32_t _2531_i = (int)0, _10304_ = _2503_nd; _2531_i < _10304_ ; ++_2531_i)
            {
              _10303_.push_back(_ptr__2500_first[_2531_i]);
            }
            auto _10305_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10303_.size()),_10303_.begin(),_10303_.end()));
            std::shared_ptr< monty::ndarray< int32_t,1 > > _2530_ii = _10305_;
            _checked_ptr_1<int32_t> _ptr__2530_ii(_2530_ii ? _2530_ii->raw() : nullptr,_2530_ii ? _2530_ii->size(0) : 0);
            int32_t _10306_ = (int)0;
            int32_t _10307_ = _2506_slicesize;
            for (int32_t _2532_i = _10306_; _2532_i < _10307_; ++_2532_i)
            {
              {
                _ptr__2523_sliceidxs[_2532_i] = ((*basevar_nativeidxs)(_ptr__2528_pp[(_2503_nd - (int)1)]));
                _ptr__2530_ii[(_2503_nd - (int)1)] += (int)1;
                _ptr__2528_pp[(_2503_nd - (int)1)] += _ptr__2524_strides[(_2503_nd - (int)1)];
                if ((_ptr__2530_ii[(_2503_nd - (int)1)] >= _ptr__2501_last[(_2503_nd - (int)1)]))
                {
                  {
                    int32_t _2533_k = (_2503_nd - (int)1);
                    while ( ((_2533_k > (int)0) && (_ptr__2530_ii[_2533_k] >= _ptr__2501_last[_2533_k])) )
                    {
                      {
                        _ptr__2530_ii[_2533_k] = _ptr__2500_first[_2533_k];
                        _ptr__2530_ii[(_2533_k - (int)1)] += (int)1;
                        _ptr__2528_pp[(_2533_k - (int)1)] += _ptr__2524_strides[(_2533_k - (int)1)];
                        -- _2533_k;
                      }
                    }
                    int32_t _10308_ = safe_add( _2533_k,(int)1 );
                    int32_t _10309_ = _2503_nd;
                    for (int32_t _2534_k2 = _10308_; _2534_k2 < _10309_; ++_2534_k2)
                    {
                      {
                        _ptr__2528_pp[_2534_k2] = _ptr__2528_pp[_2533_k];
                      }
                    }
                  }
                }
                {}
              }
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),_2504_sliceshape,nullptr,_2523_sliceidxs)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.BaseVariable.slice

// Begin mosek.fusion.BaseVariable.slice
// Method mosek.fusion.BaseVariable.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:228:6-257:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__slice(int32_t _2535_first,int32_t _2536_last) { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__slice(_2535_first,_2536_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::slice(int32_t _2535_first,int32_t _2536_last) { return __mosek_2fusion_2BaseVariable__slice(_2535_first,_2536_last); }
/* override: mosek.fusion.Variable.slice*/
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__slice(int32_t _2535_first,int32_t _2536_last) { return __mosek_2fusion_2BaseVariable__slice(_2535_first,_2536_last); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__slice(int32_t _2535_first,int32_t _2536_last)
{
  if (((int)((shape)->size(0)) != (int)1))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Variable shape and slice do not match"));
    }
  }
  {}
  if (((_2535_first == (int)0) && (_2536_last == ((*shape)((int)0)))))
  {
    {
      return (::mosek::fusion::BaseVariable::t(_pubthis)).as<::mosek::fusion::Variable>();
    }
  }
  else if(((_2535_first < (int)0) || (_2536_last > ((*shape)((int)0)))))
  {
    {
      throw ::mosek::fusion::IndexError(std::string ("Slice is out of bounds"));
    }
  }
  else if((_2535_first > _2536_last))
  {
    {
      throw ::mosek::fusion::SliceError(std::string ("Invalid slice"));
    }
  }
  else if((_2535_first == _2536_last))
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
    }
  }
  else if((NULL != sparsity.get()))
  {
    {
      int32_t _2537_slicesize = (_2536_last - _2535_first);
      int32_t _2538_nnz = (int)0;
      int32_t _2539_i0 = ::mosek::fusion::p_CommonTools::binarySearchL(sparsity,(int64_t)(_2535_first));
      int32_t _2540_i1 = ::mosek::fusion::p_CommonTools::binarySearchL(sparsity,(int64_t)(_2536_last));
      std::vector<int64_t> _10310_;
      for (int32_t _2541_i = _2539_i0, _10311_ = _2540_i1; _2541_i < _10311_ ; ++_2541_i)
      {
        _10310_.push_back((((*sparsity)(_2541_i)) - _2535_first));
      }
      auto _10312_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10310_.size()),_10310_.begin(),_10310_.end()));
      std::vector<int64_t> _10313_;
      for (int32_t _2542_i = _2539_i0, _10314_ = _2540_i1; _2542_i < _10314_ ; ++_2542_i)
      {
        _10313_.push_back(((*basevar_nativeidxs)(_2542_i)));
      }
      auto _10315_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10313_.size()),_10313_.begin(),_10313_.end()));
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2537_slicesize)}),_10312_,_10315_)).as<::mosek::fusion::Variable>();
    }
  }
  else
  {
    {
      int32_t _2543_slicesize = (_2536_last - _2535_first);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2544_sliceidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2543_slicesize)));
      _checked_ptr_1<int64_t> _ptr__2544_sliceidxs(_2544_sliceidxs ? _2544_sliceidxs->raw() : nullptr,_2544_sliceidxs ? _2544_sliceidxs->size(0) : 0);
      ::mosek::fusion::Utils::Tools::arraycopy(basevar_nativeidxs,_2535_first,_2544_sliceidxs,(int)0,_2543_slicesize);
      return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_pubthis->__mosek_2fusion_2BaseVariable__getModel(),std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2543_slicesize)}),nullptr,_2544_sliceidxs)).as<::mosek::fusion::Variable>();
    }
  }
}
// End mosek.fusion.BaseVariable.slice

// Begin mosek.fusion.BaseVariable.asExpr
// Method mosek.fusion.BaseVariable.asExpr @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:226:6-74
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable :: __mosek_2fusion_2BaseVariable__asExpr() { return mosek::fusion::p_BaseVariable::_get_impl(this)->__mosek_2fusion_2BaseVariable__asExpr(); }
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable::asExpr() { return __mosek_2fusion_2BaseVariable__asExpr(); }
/* override: mosek.fusion.Variable.asExpr*/
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::BaseVariable::__mosek_2fusion_2Variable__asExpr() { return __mosek_2fusion_2BaseVariable__asExpr(); }
monty::rc_ptr< ::mosek::fusion::Expression > mosek::fusion::p_BaseVariable::__mosek_2fusion_2BaseVariable__asExpr()
{
  return (::mosek::fusion::p_ExprFromVar::_new_ExprFromVar((::mosek::fusion::BaseVariable::t(_pubthis)).as<::mosek::fusion::Variable>())).as<::mosek::fusion::Expression>();
}
// End mosek.fusion.BaseVariable.asExpr

// Begin mosek.fusion.BaseVariable.inst
// Method mosek.fusion.BaseVariable.inst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:185:6-207:7
int32_t mosek::fusion::BaseVariable :: inst(int32_t _2545_spoffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2546_sparsity,int32_t _2547_nioffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2548_basevar_nativeidxs) { return mosek::fusion::p_BaseVariable::_get_impl(this)->inst(_2545_spoffset,_2546_sparsity,_2547_nioffset,_2548_basevar_nativeidxs); }
int32_t mosek::fusion::p_BaseVariable::inst(int32_t _2545_spoffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2546_sparsity,int32_t _2547_nioffset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2548_basevar_nativeidxs)
{
  _checked_ptr_1<int64_t> _ptr__2546_sparsity(_2546_sparsity ? _2546_sparsity->raw() : nullptr, _2546_sparsity ? _2546_sparsity->size(0) : 0);
  _checked_ptr_1<int64_t> _ptr__2548_basevar_nativeidxs(_2548_basevar_nativeidxs ? _2548_basevar_nativeidxs->raw() : nullptr, _2548_basevar_nativeidxs ? _2548_basevar_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2549_thisnidxs = mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs;
  _checked_ptr_1<int64_t> _ptr__2549_thisnidxs(_2549_thisnidxs ? _2549_thisnidxs->raw() : nullptr,_2549_thisnidxs ? _2549_thisnidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2550_thissp = mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity;
  _checked_ptr_1<int64_t> _ptr__2550_thissp(_2550_thissp ? _2550_thissp->raw() : nullptr,_2550_thissp ? _2550_thissp->size(0) : 0);
  if ((NULL == _2550_thissp.get()))
  {
    {
      int32_t _10316_ = (int)0;
      int32_t _10317_ = (int)((_2549_thisnidxs)->size(0));
      for (int32_t _2551_i = _10316_; _2551_i < _10317_; ++_2551_i)
      {
        {
          _ptr__2546_sparsity[safe_add( _2551_i,_2545_spoffset )] = _2551_i;
          _ptr__2548_basevar_nativeidxs[safe_add( _2551_i,_2547_nioffset )] = _ptr__2549_thisnidxs[_2551_i];
        }
      }
    }
  }
  else
  {
    {
      int32_t _10318_ = (int)0;
      int32_t _10319_ = (int)((mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->sparsity)->size(0));
      for (int32_t _2552_i = _10318_; _2552_i < _10319_; ++_2552_i)
      {
        {
          _ptr__2546_sparsity[safe_add( _2545_spoffset,_2552_i )] = _ptr__2550_thissp[_2552_i];
          _ptr__2548_basevar_nativeidxs[safe_add( _2552_i,_2547_nioffset )] = _ptr__2549_thisnidxs[_2552_i];
        }
      }
    }
  }
  return (int)((mosek::fusion::p_BaseVariable::_get_impl(::mosek::fusion::BaseVariable::t(_pubthis))->basevar_nativeidxs)->size(0));
}
// End mosek.fusion.BaseVariable.inst

// Begin mosek.fusion.BaseVariable.numInst
// Method mosek.fusion.BaseVariable.numInst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:181:6-183:7
int32_t mosek::fusion::BaseVariable :: numInst() { return mosek::fusion::p_BaseVariable::_get_impl(this)->numInst(); }
int32_t mosek::fusion::p_BaseVariable::numInst()
{
  return (int)((basevar_nativeidxs)->size(0));
}
// End mosek.fusion.BaseVariable.numInst

// Begin mosek.fusion.BaseVariable.inst
// Method mosek.fusion.BaseVariable.inst @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:165:6-179:7
void mosek::fusion::BaseVariable :: inst(int32_t _2553_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2554_nindex) { mosek::fusion::p_BaseVariable::_get_impl(this)->inst(_2553_offset,_2554_nindex); }
void mosek::fusion::p_BaseVariable::inst(int32_t _2553_offset,std::shared_ptr< monty::ndarray< int64_t,1 > > _2554_nindex)
{
  _checked_ptr_1<int64_t> _ptr__2554_nindex(_2554_nindex ? _2554_nindex->raw() : nullptr, _2554_nindex ? _2554_nindex->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      int32_t _10320_ = (int)0;
      int32_t _10321_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2555_i = _10320_; _2555_i < _10321_; ++_2555_i)
      {
        {
          _ptr__2554_nindex[safe_add( _2553_offset,_2555_i )] = ((*basevar_nativeidxs)(_2555_i));
        }
      }
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2556_shape = _pubthis->getShape();
      _checked_ptr_1<int32_t> _ptr__2556_shape(_2556_shape ? _2556_shape->raw() : nullptr,_2556_shape ? _2556_shape->size(0) : 0);
      int32_t _2557_domsize = (int)1;
      int32_t _10322_ = (int)0;
      int32_t _10323_ = (int)((_2556_shape)->size(0));
      for (int32_t _2558_i = _10322_; _2558_i < _10323_; ++_2558_i)
      {
        {
          _2557_domsize *= _ptr__2556_shape[_2558_i];
        }
      }
      int32_t _10324_ = (int)0;
      int32_t _10325_ = _2557_domsize;
      for (int32_t _2559_i = _10324_; _2559_i < _10325_; ++_2559_i)
      {
        {
          _ptr__2554_nindex[safe_add( _2553_offset,_2559_i )] = (int)0;
        }
      }
      int32_t _10326_ = (int)0;
      int32_t _10327_ = (int)((sparsity)->size(0));
      for (int32_t _2560_i = _10326_; _2560_i < _10327_; ++_2560_i)
      {
        {
          _ptr__2554_nindex[safe_add( _2553_offset,((*sparsity)(_2560_i)) )] = ((*basevar_nativeidxs)(_2560_i));
        }
      }
    }
  }
}
// End mosek.fusion.BaseVariable.inst

// Begin mosek.fusion.BaseVariable.set_values
// Method mosek.fusion.BaseVariable.set_values @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:147:6-163:7
void mosek::fusion::BaseVariable :: set_values(std::shared_ptr< monty::ndarray< double,1 > > _2561_values,bool _2562_primal) { mosek::fusion::p_BaseVariable::_get_impl(this)->set_values(_2561_values,_2562_primal); }
void mosek::fusion::p_BaseVariable::set_values(std::shared_ptr< monty::ndarray< double,1 > > _2561_values,bool _2562_primal)
{
  _checked_ptr_1<double> _ptr__2561_values(_2561_values ? _2561_values->raw() : nullptr, _2561_values ? _2561_values->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      _pubthis->__mosek_2fusion_2BaseVariable__getModel()->setVariableValues(_2562_primal,basevar_nativeidxs,_2561_values);
    }
  }
  else
  {
    {
      int32_t _2563_varsize = (int)1;
      int32_t _10328_ = (int)0;
      int32_t _10329_ = (int)((shape)->size(0));
      for (int32_t _2564_i = _10328_; _2564_i < _10329_; ++_2564_i)
      {
        {
          _2563_varsize *= ((*shape)(_2564_i));
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2565_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2563_varsize)));
      _checked_ptr_1<int64_t> _ptr__2565_idxs(_2565_idxs ? _2565_idxs->raw() : nullptr,_2565_idxs ? _2565_idxs->size(0) : 0);
      int32_t _10330_ = (int)0;
      int32_t _10331_ = (int)((basevar_nativeidxs)->size(0));
      for (int32_t _2566_i = _10330_; _2566_i < _10331_; ++_2566_i)
      {
        {
          _ptr__2565_idxs[((*sparsity)(_2566_i))] = ((*basevar_nativeidxs)(_2566_i));
        }
      }
      _pubthis->__mosek_2fusion_2BaseVariable__getModel()->setVariableValues(_2562_primal,_2565_idxs,_2561_values);
    }
  }
}
// End mosek.fusion.BaseVariable.set_values

// Begin mosek.fusion.BaseVariable.dual_lu
// Method mosek.fusion.BaseVariable.dual_lu @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:131:6-144:7
void mosek::fusion::p_BaseVariable::dual_lu(int32_t _2567_offset,std::shared_ptr< monty::ndarray< double,1 > > _2568_target,bool _2569_lower)
{
  _checked_ptr_1<double> _ptr__2568_target(_2568_target ? _2568_target->raw() : nullptr, _2568_target ? _2568_target->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      model->getVariableDuals(_2569_lower,basevar_nativeidxs,_2568_target,_2567_offset);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2570_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<int64_t> _ptr__2570_idxs(_2570_idxs ? _2570_idxs->raw() : nullptr,_2570_idxs ? _2570_idxs->size(0) : 0);
      int32_t _10332_ = (int)0;
      int32_t _10333_ = (int)((sparsity)->size(0));
      for (int32_t _2571_i = _10332_; _2571_i < _10333_; ++_2571_i)
      {
        {
          _ptr__2570_idxs[((*sparsity)(_2571_i))] = ((*basevar_nativeidxs)(_2571_i));
        }
      }
      model->getVariableDuals(_2569_lower,_2570_idxs,_2568_target,_2567_offset);
    }
  }
}
// End mosek.fusion.BaseVariable.dual_lu

// Begin mosek.fusion.BaseVariable.values
// Method mosek.fusion.BaseVariable.values @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:116:6-129:7
void mosek::fusion::BaseVariable :: values(int32_t _2572_offset,std::shared_ptr< monty::ndarray< double,1 > > _2573_target,bool _2574_primal) { mosek::fusion::p_BaseVariable::_get_impl(this)->values(_2572_offset,_2573_target,_2574_primal); }
void mosek::fusion::p_BaseVariable::values(int32_t _2572_offset,std::shared_ptr< monty::ndarray< double,1 > > _2573_target,bool _2574_primal)
{
  _checked_ptr_1<double> _ptr__2573_target(_2573_target ? _2573_target->raw() : nullptr, _2573_target ? _2573_target->size(0) : 0);
  if ((NULL == sparsity.get()))
  {
    {
      model->getVariableValues(_2574_primal,basevar_nativeidxs,_2573_target,_2572_offset);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2575_idxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_pubthis->getSize())));
      _checked_ptr_1<int64_t> _ptr__2575_idxs(_2575_idxs ? _2575_idxs->raw() : nullptr,_2575_idxs ? _2575_idxs->size(0) : 0);
      int32_t _10334_ = (int)0;
      int32_t _10335_ = (int)((sparsity)->size(0));
      for (int32_t _2576_i = _10334_; _2576_i < _10335_; ++_2576_i)
      {
        {
          _ptr__2575_idxs[((*sparsity)(_2576_i))] = ((*basevar_nativeidxs)(_2576_i));
        }
      }
      model->getVariableValues(_2574_primal,_2575_idxs,_2573_target,_2572_offset);
    }
  }
}
// End mosek.fusion.BaseVariable.values

// Begin mosek.fusion.BaseVariable.make_continuous
// Method mosek.fusion.BaseVariable.make_continuous @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:114:6-89
void mosek::fusion::BaseVariable :: make_continuous() { mosek::fusion::p_BaseVariable::_get_impl(this)->make_continuous(); }
void mosek::fusion::p_BaseVariable::make_continuous()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->make_continuous(basevar_nativeidxs);
}
// End mosek.fusion.BaseVariable.make_continuous

// Begin mosek.fusion.BaseVariable.make_integer
// Method mosek.fusion.BaseVariable.make_integer @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BaseVariable.monty:113:6-83
void mosek::fusion::BaseVariable :: make_integer() { mosek::fusion::p_BaseVariable::_get_impl(this)->make_integer(); }
void mosek::fusion::p_BaseVariable::make_integer()
{
  mosek::fusion::p_Model::_get_impl(_pubthis->__mosek_2fusion_2BaseVariable__getModel().get())->make_integer(basevar_nativeidxs);
}
// End mosek.fusion.BaseVariable.make_integer

void mosek::fusion::p_BaseVariable::destroy()
{
  sparsity.reset();
  basevar_nativeidxs.reset();
  model.reset();
  shape.reset();
}
void mosek::fusion::BaseVariable::destroy() { mosek::fusion::p_BaseVariable::_get_impl(this)->destroy(); }
// } class BaseVariable
// class Variable { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Variable.mbi:187:4-419:5
// Begin mosek.fusion.Variable.toString
// End mosek.fusion.Variable.toString

// Begin mosek.fusion.Variable.numInst
// End mosek.fusion.Variable.numInst

// Begin mosek.fusion.Variable.inst
// End mosek.fusion.Variable.inst

// Begin mosek.fusion.Variable.inst
// End mosek.fusion.Variable.inst

// Begin mosek.fusion.Variable.remove
// End mosek.fusion.Variable.remove

// Begin mosek.fusion.Variable.getND
// End mosek.fusion.Variable.getND

// Begin mosek.fusion.Variable.getShape
// End mosek.fusion.Variable.getShape

// Begin mosek.fusion.Variable.getModel
monty::rc_ptr< mosek::fusion::Model > mosek::fusion::Variable::getModel() { return __mosek_2fusion_2Variable__getModel(); }
// End mosek.fusion.Variable.getModel

// Begin mosek.fusion.Variable.getSize
// End mosek.fusion.Variable.getSize

// Begin mosek.fusion.Variable.setLevel
// End mosek.fusion.Variable.setLevel

// Begin mosek.fusion.Variable.fromTril
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::fromTril(int32_t _2584_dim) { return __mosek_2fusion_2Variable__fromTril(_2584_dim); }
// End mosek.fusion.Variable.fromTril

// Begin mosek.fusion.Variable.tril
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::tril() { return __mosek_2fusion_2Variable__tril(); }
// End mosek.fusion.Variable.tril

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2585_dim0,int32_t _2586_dim1,int32_t _2587_dim2) { return __mosek_2fusion_2Variable__reshape(_2585_dim0,_2586_dim1,_2587_dim2); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2588_dim0,int32_t _2589_dim1) { return __mosek_2fusion_2Variable__reshape(_2588_dim0,_2589_dim1); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(int32_t _2590_dim0) { return __mosek_2fusion_2Variable__reshape(_2590_dim0); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.reshape
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::reshape(std::shared_ptr< monty::ndarray< int32_t,1 > > _2591_shape) { return __mosek_2fusion_2Variable__reshape(_2591_shape); }
// End mosek.fusion.Variable.reshape

// Begin mosek.fusion.Variable.set_values
// End mosek.fusion.Variable.set_values

// Begin mosek.fusion.Variable.dual
// End mosek.fusion.Variable.dual

// Begin mosek.fusion.Variable.level
// End mosek.fusion.Variable.level

// Begin mosek.fusion.Variable.values
// End mosek.fusion.Variable.values

// Begin mosek.fusion.Variable.make_continuous
// End mosek.fusion.Variable.make_continuous

// Begin mosek.fusion.Variable.make_integer
// End mosek.fusion.Variable.make_integer

// Begin mosek.fusion.Variable.makeContinuous
// End mosek.fusion.Variable.makeContinuous

// Begin mosek.fusion.Variable.makeInteger
// End mosek.fusion.Variable.makeInteger

// Begin mosek.fusion.Variable.transpose
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::transpose() { return __mosek_2fusion_2Variable__transpose(); }
// End mosek.fusion.Variable.transpose

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2597_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2598_i2,std::shared_ptr< monty::ndarray< int32_t,1 > > _2599_i3) { return __mosek_2fusion_2Variable__pick(_2597_i1,_2598_i2,_2599_i3); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2600_i1,std::shared_ptr< monty::ndarray< int32_t,1 > > _2601_i2) { return __mosek_2fusion_2Variable__pick(_2600_i1,_2601_i2); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2602_midxs) { return __mosek_2fusion_2Variable__pick(_2602_midxs); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,2 > > _2602_midxs) { return __mosek_2fusion_2Variable__pick(_2602_midxs); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.pick
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2603_idxs) { return __mosek_2fusion_2Variable__pick(_2603_idxs); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::pick(std::shared_ptr< monty::ndarray< int32_t,1 > > _2603_idxs) { return __mosek_2fusion_2Variable__pick(_2603_idxs); }
// End mosek.fusion.Variable.pick

// Begin mosek.fusion.Variable.antidiag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::antidiag() { return __mosek_2fusion_2Variable__antidiag(); }
// End mosek.fusion.Variable.antidiag

// Begin mosek.fusion.Variable.antidiag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::antidiag(int32_t _2604_index) { return __mosek_2fusion_2Variable__antidiag(_2604_index); }
// End mosek.fusion.Variable.antidiag

// Begin mosek.fusion.Variable.diag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::diag() { return __mosek_2fusion_2Variable__diag(); }
// End mosek.fusion.Variable.diag

// Begin mosek.fusion.Variable.diag
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::diag(int32_t _2605_index) { return __mosek_2fusion_2Variable__diag(_2605_index); }
// End mosek.fusion.Variable.diag

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2606_idx) { return __mosek_2fusion_2Variable__index(_2606_idx); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2606_idx) { return __mosek_2fusion_2Variable__index(_2606_idx); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2607_i1,int32_t _2608_i2,int32_t _2609_i3) { return __mosek_2fusion_2Variable__index(_2607_i1,_2608_i2,_2609_i3); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2610_i1,int32_t _2611_i2) { return __mosek_2fusion_2Variable__index(_2610_i1,_2611_i2); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.index
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__index(int32_t _2612_i1) { return __mosek_2fusion_2Variable__index(_2612_i1); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::index(int32_t _2612_i1) { return __mosek_2fusion_2Variable__index(_2612_i1); }
// End mosek.fusion.Variable.index

// Begin mosek.fusion.Variable.slice
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2613_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2614_lasta) { return __mosek_2fusion_2Variable__slice(_2613_firsta,_2614_lasta); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2613_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2614_lasta) { return __mosek_2fusion_2Variable__slice(_2613_firsta,_2614_lasta); }
// End mosek.fusion.Variable.slice

// Begin mosek.fusion.Variable.slice
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::__mosek_2fusion_2Expression__slice(int32_t _2615_first,int32_t _2616_last) { return __mosek_2fusion_2Variable__slice(_2615_first,_2616_last); }
monty::rc_ptr< mosek::fusion::Variable > mosek::fusion::Variable::slice(int32_t _2615_first,int32_t _2616_last) { return __mosek_2fusion_2Variable__slice(_2615_first,_2616_last); }
// End mosek.fusion.Variable.slice

// Begin mosek.fusion.Variable.asExpr
monty::rc_ptr< mosek::fusion::Expression > mosek::fusion::Variable::asExpr() { return __mosek_2fusion_2Variable__asExpr(); }
// End mosek.fusion.Variable.asExpr

// } class Variable
// class Var { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:144:4-814:5
mosek::fusion::Var::Var(mosek::fusion::p_Var *_impl) : _impl(_impl) { /*std::cout << "Var() " << this << std::endl;*/ _impl->_pubthis = this; }
mosek::fusion::Var::~Var() { delete _impl; _impl = nullptr; }
mosek::fusion::p_Var::p_Var
  (::mosek::fusion::Var * _pubthis) :     _pubthis(_pubthis)
{}
// Begin mosek.fusion.Var.empty
// Method mosek.fusion.Var.empty @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:808:6-813:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::empty
  ( std::shared_ptr< monty::ndarray< int32_t,1 > > _2617_shape) { return mosek::fusion::p_Var::empty(_2617_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::empty(std::shared_ptr< monty::ndarray< int32_t,1 > > _2617_shape)
{
  _checked_ptr_1<int32_t> _ptr__2617_shape(_2617_shape ? _2617_shape->raw() : nullptr, _2617_shape ? _2617_shape->size(0) : 0);
  int32_t _10336_ = (int)0;
  int32_t _10337_ = (int)((_2617_shape)->size(0));
  for (int32_t _2618_i = _10336_; _2618_i < _10337_; ++_2618_i)
  {
    {
      if ((_ptr__2617_shape[_2618_i] < (int)0))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid shape"));
        }
      }
      {}
    }
  }
  return (::mosek::fusion::p_NilVariable::_new_NilVariable(_2617_shape)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Var.empty

// Begin mosek.fusion.Var.compress
// Method mosek.fusion.Var.compress @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:782:6-800:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::compress
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2619_v) { return mosek::fusion::p_Var::compress(_2619_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::compress(monty::rc_ptr< ::mosek::fusion::Variable > _2619_v)
{
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2620_shape = _2619_v->getShape();
  _checked_ptr_1<int32_t> _ptr__2620_shape(_2620_shape ? _2620_shape->raw() : nullptr,_2620_shape ? _2620_shape->size(0) : 0);
  int32_t _2621_nd = (int)((_2620_shape)->size(0));
  int32_t _2622_realnd = (int)0;
  int32_t _10338_ = (int)0;
  int32_t _10339_ = _2621_nd;
  for (int32_t _2623_i = _10338_; _2623_i < _10339_; ++_2623_i)
  {
    {
      if ((_ptr__2620_shape[_2623_i] != (int)1))
      {
        {
          ++ _2622_realnd;
        }
      }
      {}
    }
  }
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2624_realshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2622_realnd)));
  _checked_ptr_1<int32_t> _ptr__2624_realshape(_2624_realshape ? _2624_realshape->raw() : nullptr,_2624_realshape ? _2624_realshape->size(0) : 0);
  {
    int32_t _2625_k = (int)0;
    int32_t _10340_ = (int)0;
    int32_t _10341_ = _2621_nd;
    for (int32_t _2626_i = _10340_; _2626_i < _10341_; ++_2626_i)
    {
      {
        if ((_ptr__2620_shape[_2626_i] != (int)1))
        {
          {
            _ptr__2624_realshape[_2625_k] = _ptr__2620_shape[_2626_i];
            ++ _2625_k;
          }
        }
        {}
      }
    }
  }
  return ::mosek::fusion::p_Var::reshape(_2619_v,_2624_realshape);
}
// End mosek.fusion.Var.compress

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:765:6-768:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2627_v,
    int32_t _2628_d1) { return mosek::fusion::p_Var::reshape(_2627_v,_2628_d1); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2627_v,int32_t _2628_d1)
{
  return ::mosek::fusion::p_Var::reshape(_2627_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2628_d1)}));
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:751:6-754:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2629_v,
    int32_t _2630_d1,
    int32_t _2631_d2) { return mosek::fusion::p_Var::reshape(_2629_v,_2630_d1,_2631_d2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2629_v,int32_t _2630_d1,int32_t _2631_d2)
{
  return ::mosek::fusion::p_Var::reshape(_2629_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)(_2630_d1),(int32_t)(_2631_d2)}));
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.flatten
// Method mosek.fusion.Var.flatten @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:738:6-741:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::flatten
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2632_v) { return mosek::fusion::p_Var::flatten(_2632_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::flatten(monty::rc_ptr< ::mosek::fusion::Variable > _2632_v)
{
  return ::mosek::fusion::p_Var::reshape(_2632_v,std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >{(int32_t)((int32_t)::mosek::fusion::p_Set::size(_2632_v->getShape()))}));
}
// End mosek.fusion.Var.flatten

// Begin mosek.fusion.Var.reshape
// Method mosek.fusion.Var.reshape @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:707:6-709:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::reshape
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2633_v,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2634_shape) { return mosek::fusion::p_Var::reshape(_2633_v,_2634_shape); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::reshape(monty::rc_ptr< ::mosek::fusion::Variable > _2633_v,std::shared_ptr< monty::ndarray< int32_t,1 > > _2634_shape)
{
  _checked_ptr_1<int32_t> _ptr__2634_shape(_2634_shape ? _2634_shape->raw() : nullptr, _2634_shape ? _2634_shape->size(0) : 0);
  return _2633_v->__mosek_2fusion_2Variable__reshape(_2634_shape);
}
// End mosek.fusion.Var.reshape

// Begin mosek.fusion.Var.index_permute_
// Method mosek.fusion.Var.index_permute_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:619:6-672:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::index_permute_(monty::rc_ptr< ::mosek::fusion::Variable > _2635_v,std::shared_ptr< monty::ndarray< int32_t,1 > > _2636_perm)
{
  _checked_ptr_1<int32_t> _ptr__2636_perm(_2636_perm ? _2636_perm->raw() : nullptr, _2636_perm ? _2636_perm->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2637_sparsity;
  _checked_ptr_1<int64_t> _ptr__2637_sparsity(_2637_sparsity ? _2637_sparsity->raw() : nullptr,_2637_sparsity ? _2637_sparsity->size(0) : 0);
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2638_nativeidxs;
  _checked_ptr_1<int64_t> _ptr__2638_nativeidxs(_2638_nativeidxs ? _2638_nativeidxs->raw() : nullptr,_2638_nativeidxs ? _2638_nativeidxs->size(0) : 0);
  std::shared_ptr< monty::ndarray< int32_t,1 > > _2639_shape = _2635_v->getShape();
  _checked_ptr_1<int32_t> _ptr__2639_shape(_2639_shape ? _2639_shape->raw() : nullptr,_2639_shape ? _2639_shape->size(0) : 0);
  int32_t _2640_nd = (int)((_2639_shape)->size(0));
  int32_t _2641_shapesize = (int)1;
  int32_t _10342_ = (int)0;
  int32_t _10343_ = _2640_nd;
  for (int32_t _2642_i = _10342_; _2642_i < _10343_; ++_2642_i)
  {
    {
      _2641_shapesize *= _ptr__2639_shape[_2642_i];
    }
  }
  int32_t _2643_nnz = _2635_v->numInst();
  std::shared_ptr< monty::ndarray< int64_t,1 > > _2644_pstrides = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2640_nd)));
  _checked_ptr_1<int64_t> _ptr__2644_pstrides(_2644_pstrides ? _2644_pstrides->raw() : nullptr,_2644_pstrides ? _2644_pstrides->size(0) : 0);
  _ptr__2644_pstrides[(_2640_nd - (int)1)] = (int)1;
  int32_t _10344_ = (int)1;
  int32_t _10345_ = _2640_nd;
  for (int32_t _2645_i = _10344_; _2645_i < _10345_; ++_2645_i)
  {
    {
      _ptr__2644_pstrides[((_2640_nd - _2645_i) - (int)1)] = safe_mul( _ptr__2644_pstrides[(_2640_nd - _2645_i)],_ptr__2639_shape[_ptr__2636_perm[(_2640_nd - _2645_i)]] );
    }
  }
  if ((_2643_nnz < _2641_shapesize))
  {
    {
      _2637_sparsity = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2643_nnz)));
      _ptr__2637_sparsity.update(_2637_sparsity ? _2637_sparsity->raw() : nullptr, _2637_sparsity ? _2637_sparsity->size(0) : 0);
      _2638_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2643_nnz)));
      _ptr__2638_nativeidxs.update(_2638_nativeidxs ? _2638_nativeidxs->raw() : nullptr, _2638_nativeidxs ? _2638_nativeidxs->size(0) : 0);
      _2635_v->inst((int)0,_2637_sparsity,(int)0,_2638_nativeidxs);
    }
  }
  else
  {
    {
      _2637_sparsity = nullptr;
      _ptr__2637_sparsity.update(_2637_sparsity ? _2637_sparsity->raw() : nullptr, _2637_sparsity ? _2637_sparsity->size(0) : 0);
      _2638_nativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2643_nnz)));
      _ptr__2638_nativeidxs.update(_2638_nativeidxs ? _2638_nativeidxs->raw() : nullptr, _2638_nativeidxs ? _2638_nativeidxs->size(0) : 0);
      _2635_v->inst((int)0,_2638_nativeidxs);
    }
  }
  if ((NULL == _2637_sparsity.get()))
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2646_residxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2643_nnz)));
      _checked_ptr_1<int64_t> _ptr__2646_residxs(_2646_residxs ? _2646_residxs->raw() : nullptr,_2646_residxs ? _2646_residxs->size(0) : 0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _2647_ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2640_nd)));
        _checked_ptr_1<int32_t> _ptr__2647_ii(_2647_ii ? _2647_ii->raw() : nullptr,_2647_ii ? _2647_ii->size(0) : 0);
        int32_t _10346_ = (int)0;
        int32_t _10347_ = _2643_nnz;
        for (int32_t _2648_i = _10346_; _2648_i < _10347_; ++_2648_i)
        {
          {
            int32_t _2649_rem = _2648_i;
            int32_t _10348_ = (int)0;
            int32_t _10349_ = _2640_nd;
            for (int32_t _2650_j = _10348_; _2650_j < _10349_; ++_2650_j)
            {
              {
                _ptr__2647_ii[((_2640_nd - _2648_i) - (int)1)] = (_2649_rem % _ptr__2639_shape[((_2640_nd - _2648_i) - (int)1)]);
                _2649_rem /= _ptr__2639_shape[((_2640_nd - _2648_i) - (int)1)];
              }
            }
            int32_t _2651_newi = (int)0;
            int32_t _10350_ = (int)0;
            int32_t _10351_ = _2640_nd;
            for (int32_t _2652_j = _10350_; _2652_j < _10351_; ++_2652_j)
            {
              {
                _2651_newi += (int32_t)safe_mul( _ptr__2647_ii[_ptr__2636_perm[_2652_j]],_ptr__2644_pstrides[_2652_j] );
              }
            }
            _ptr__2646_residxs[_2651_newi] = _ptr__2638_nativeidxs[_2648_i];
          }
        }
      }
      _2638_nativeidxs = _2646_residxs;
      _ptr__2638_nativeidxs.update(_2638_nativeidxs ? _2638_nativeidxs->raw() : nullptr, _2638_nativeidxs ? _2638_nativeidxs->size(0) : 0);
    }
  }
  else
  {
    {
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2653_ressp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2643_nnz)));
      _checked_ptr_1<int64_t> _ptr__2653_ressp(_2653_ressp ? _2653_ressp->raw() : nullptr,_2653_ressp ? _2653_ressp->size(0) : 0);
      {
        std::shared_ptr< monty::ndarray< int32_t,1 > > _2654_ii = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2640_nd)));
        _checked_ptr_1<int32_t> _ptr__2654_ii(_2654_ii ? _2654_ii->raw() : nullptr,_2654_ii ? _2654_ii->size(0) : 0);
        int32_t _10352_ = (int)0;
        int32_t _10353_ = _2643_nnz;
        for (int32_t _2655_i = _10352_; _2655_i < _10353_; ++_2655_i)
        {
          {
            int32_t _2656_rem = _2655_i;
            int32_t _10354_ = (int)0;
            int32_t _10355_ = _2640_nd;
            for (int32_t _2657_j = _10354_; _2657_j < _10355_; ++_2657_j)
            {
              {
                _ptr__2654_ii[((_2640_nd - _2655_i) - (int)1)] = (_2656_rem % _ptr__2639_shape[((_2640_nd - _2655_i) - (int)1)]);
                _2656_rem /= _ptr__2639_shape[((_2640_nd - _2655_i) - (int)1)];
              }
            }
            int32_t _2658_newi = (int)0;
            int32_t _10356_ = (int)0;
            int32_t _10357_ = _2640_nd;
            for (int32_t _2659_j = _10356_; _2659_j < _10357_; ++_2659_j)
            {
              {
                _2658_newi += (int32_t)safe_mul( _ptr__2654_ii[_ptr__2636_perm[_2659_j]],_ptr__2644_pstrides[_2659_j] );
              }
            }
            _ptr__2653_ressp[_2655_i] = _2658_newi;
          }
        }
      }
      std::vector<int64_t> _10358_;
      for (int32_t _2661_i = (int)0, _10359_ = _2643_nnz; _2661_i < _10359_ ; ++_2661_i)
      {
        _10358_.push_back(_2661_i);
      }
      auto _10360_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10358_.size()),_10358_.begin(),_10358_.end()));
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2660_spperm = _10360_;
      _checked_ptr_1<int64_t> _ptr__2660_spperm(_2660_spperm ? _2660_spperm->raw() : nullptr,_2660_spperm ? _2660_spperm->size(0) : 0);
      ::mosek::fusion::p_CommonTools::argQsort(_2660_spperm,_2653_ressp,std::shared_ptr< monty::ndarray< int64_t,1 > >(nullptr),(int64_t)((int)0),(int64_t)(_2643_nnz));
      std::vector<int64_t> _10361_;
      for (int32_t _2662_i = (int)0, _10362_ = _2643_nnz; _2662_i < _10362_ ; ++_2662_i)
      {
        _10361_.push_back(_ptr__2637_sparsity[_ptr__2660_spperm[_2662_i]]);
      }
      auto _10363_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10361_.size()),_10361_.begin(),_10361_.end()));
      _2637_sparsity = _10363_;
      _ptr__2637_sparsity.update(_2637_sparsity ? _2637_sparsity->raw() : nullptr, _2637_sparsity ? _2637_sparsity->size(0) : 0);
      std::vector<int64_t> _10364_;
      for (int32_t _2663_i = (int)0, _10365_ = _2643_nnz; _2663_i < _10365_ ; ++_2663_i)
      {
        _10364_.push_back(_ptr__2638_nativeidxs[_ptr__2660_spperm[_2663_i]]);
      }
      auto _10366_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10364_.size()),_10364_.begin(),_10364_.end()));
      _2638_nativeidxs = _10366_;
      _ptr__2638_nativeidxs.update(_2638_nativeidxs ? _2638_nativeidxs->raw() : nullptr, _2638_nativeidxs ? _2638_nativeidxs->size(0) : 0);
    }
  }
  std::vector<int32_t> _10367_;
  for (int32_t _2664_i = (int)0, _10368_ = _2640_nd; _2664_i < _10368_ ; ++_2664_i)
  {
    _10367_.push_back(_ptr__2639_shape[_ptr__2636_perm[_2664_i]]);
  }
  auto _10369_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10367_.size()),_10367_.begin(),_10367_.end()));
  return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2635_v->__mosek_2fusion_2Variable__getModel(),_10369_,_2637_sparsity,_2638_nativeidxs)).as<::mosek::fusion::Variable>();
}
// End mosek.fusion.Var.index_permute_

// Begin mosek.fusion.Var.hrepeat
// Method mosek.fusion.Var.hrepeat @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:611:6-108
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hrepeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2665_v,
    int32_t _2666_n) { return mosek::fusion::p_Var::hrepeat(_2665_v,_2666_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hrepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2665_v,int32_t _2666_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2665_v,(int)1,_2666_n);
}
// End mosek.fusion.Var.hrepeat

// Begin mosek.fusion.Var.vrepeat
// Method mosek.fusion.Var.vrepeat @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:610:6-108
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vrepeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2667_v,
    int32_t _2668_n) { return mosek::fusion::p_Var::vrepeat(_2667_v,_2668_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vrepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2667_v,int32_t _2668_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2667_v,(int)0,_2668_n);
}
// End mosek.fusion.Var.vrepeat

// Begin mosek.fusion.Var.repeat
// Method mosek.fusion.Var.repeat @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:609:6-108
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::repeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2669_v,
    int32_t _2670_n) { return mosek::fusion::p_Var::repeat(_2669_v,_2670_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::repeat(monty::rc_ptr< ::mosek::fusion::Variable > _2669_v,int32_t _2670_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2669_v,(int)0,_2670_n);
}
// End mosek.fusion.Var.repeat

// Begin mosek.fusion.Var.repeat
// Method mosek.fusion.Var.repeat @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:608:6-108
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::repeat
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2671_v,
    int32_t _2672_dim,
    int32_t _2673_n) { return mosek::fusion::p_Var::repeat(_2671_v,_2672_dim,_2673_n); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::repeat(monty::rc_ptr< ::mosek::fusion::Variable > _2671_v,int32_t _2672_dim,int32_t _2673_n)
{
  return ::mosek::fusion::p_Var::drepeat(_2671_v,_2672_dim,_2673_n);
}
// End mosek.fusion.Var.repeat

// Begin mosek.fusion.Var.drepeat
// Method mosek.fusion.Var.drepeat @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:472:6-606:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::drepeat(monty::rc_ptr< ::mosek::fusion::Variable > _2674_v,int32_t _2675_dim,int32_t _2676_n)
{
  if ((_2676_n < (int)0))
  {
    {
      throw ::mosek::fusion::LengthError(std::string ("Cannot repeat less than 0 times"));
    }
  }
  else if((_2676_n == (int)0))
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
    }
  }
  else if((_2676_n == (int)1))
  {
    {
      return _2674_v;
    }
  }
  else
  {
    {
      monty::rc_ptr< ::mosek::fusion::Model > _2677_model = _2674_v->__mosek_2fusion_2Variable__getModel();
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2678_shape = _2674_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2678_shape(_2678_shape ? _2678_shape->raw() : nullptr,_2678_shape ? _2678_shape->size(0) : 0);
      int32_t _2679_nd = (int)((_2678_shape)->size(0));
      if (((_2675_dim < (int)0) || (_2675_dim > _2679_nd)))
      {
        {
          throw ::mosek::fusion::DimensionError(std::string ("Invalid stacking dimension"));
        }
      }
      {}
      int64_t _2680_domsize = ::mosek::fusion::p_Set::size(_2678_shape);
      int32_t _2681_nnz = _2674_v->numInst();
      int32_t _2682_rnnz = safe_mul( _2681_nnz,_2676_n );
      int32_t _2683_d0 = (int)1;
      int32_t _10370_ = (int)0;
      int32_t _10371_ = _2675_dim;
      for (int32_t _2684_i = _10370_; _2684_i < _10371_; ++_2684_i)
      {
        {
          _2683_d0 *= _ptr__2678_shape[_2684_i];
        }
      }
      int32_t _10372_;
      bool _10373_ = (_2675_dim < _2679_nd);
      if (_10373_)
      {
        _10372_ = _ptr__2678_shape[_2675_dim];
      }
      else
      {
        _10372_ = (int)1;
      }
      int32_t _2685_d1 = _10372_;
      int32_t _2686_nd1 = safe_mul( _2685_d1,_2676_n );
      int32_t _2687_d2 = (int)1;
      int32_t _10374_ = safe_add( _2675_dim,(int)1 );
      int32_t _10375_ = _2679_nd;
      for (int32_t _2688_i = _10374_; _2688_i < _10375_; ++_2688_i)
      {
        {
          _2687_d2 *= _ptr__2678_shape[_2688_i];
        }
      }
      int32_t _10376_;
      bool _10377_ = (_2675_dim < _2679_nd);
      if (_10377_)
      {
        _10376_ = _2679_nd;
      }
      else
      {
        _10376_ = safe_add( _2675_dim,(int)1 );
      }
      int32_t _2689_rnd = _10376_;
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2690_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2689_rnd)));
      _checked_ptr_1<int32_t> _ptr__2690_rshape(_2690_rshape ? _2690_rshape->raw() : nullptr,_2690_rshape ? _2690_rshape->size(0) : 0);
      int32_t _10378_ = (int)0;
      int32_t _10379_ = _2679_nd;
      for (int32_t _2691_i = _10378_; _2691_i < _10379_; ++_2691_i)
      {
        {
          _ptr__2690_rshape[_2691_i] = _ptr__2678_shape[_2691_i];
        }
      }
      int32_t _10380_ = _2679_nd;
      int32_t _10381_ = _2689_rnd;
      for (int32_t _2692_i = _10380_; _2692_i < _10381_; ++_2692_i)
      {
        {
          _ptr__2690_rshape[_2692_i] = (int)1;
        }
      }
      _ptr__2690_rshape[_2675_dim] = _2686_nd1;
      if ((_2681_nnz < _2680_domsize))
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2693_tnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2681_nnz)));
          _checked_ptr_1<int64_t> _ptr__2693_tnativeidxs(_2693_tnativeidxs ? _2693_tnativeidxs->raw() : nullptr,_2693_tnativeidxs ? _2693_tnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2694_tsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2681_nnz)));
          _checked_ptr_1<int64_t> _ptr__2694_tsp(_2694_tsp ? _2694_tsp->raw() : nullptr,_2694_tsp ? _2694_tsp->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2695_rnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2682_rnnz)));
          _checked_ptr_1<int64_t> _ptr__2695_rnativeidxs(_2695_rnativeidxs ? _2695_rnativeidxs->raw() : nullptr,_2695_rnativeidxs ? _2695_rnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2696_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2682_rnnz)));
          _checked_ptr_1<int64_t> _ptr__2696_rsp(_2696_rsp ? _2696_rsp->raw() : nullptr,_2696_rsp ? _2696_rsp->size(0) : 0);
          _2674_v->inst((int)0,_2694_tsp,(int)0,_2693_tnativeidxs);
          {
            int32_t _2697_ns0 = safe_mul( _2686_nd1,_2687_d2 );
            int32_t _2698_s0 = safe_mul( _2685_d1,_2687_d2 );
            int32_t _2699_s1 = _2687_d2;
            if ((_2675_dim == (int)0))
            {
              {
                int32_t _2700_k = (int)0;
                int32_t _10382_ = (int)0;
                int32_t _10383_ = _2676_n;
                for (int32_t _2701_i = _10382_; _2701_i < _10383_; ++_2701_i)
                {
                  {
                    int32_t _10384_ = (int)0;
                    int32_t _10385_ = _2681_nnz;
                    for (int32_t _2702_j = _10384_; _2702_j < _10385_; ++_2702_j)
                    {
                      {
                        _ptr__2696_rsp[_2700_k] = safe_add( _ptr__2694_tsp[_2702_j],safe_mul( _2701_i,_2680_domsize ) );
                        _ptr__2695_rnativeidxs[_2700_k] = _ptr__2693_tnativeidxs[_2702_j];
                        ++ _2700_k;
                      }
                    }
                  }
                }
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2677_model,_2690_rshape,_2696_rsp,_2695_rnativeidxs)).as<::mosek::fusion::Variable>();
              }
            }
            else if((_2675_dim >= _2679_nd))
            {
              {
                int32_t _2703_k = (int)0;
                int32_t _10386_ = (int)0;
                int32_t _10387_ = _2681_nnz;
                for (int32_t _2704_j = _10386_; _2704_j < _10387_; ++_2704_j)
                {
                  {
                    int32_t _10388_ = (int)0;
                    int32_t _10389_ = _2676_n;
                    for (int32_t _2705_i = _10388_; _2705_i < _10389_; ++_2705_i)
                    {
                      {
                        _ptr__2696_rsp[_2703_k] = safe_add( safe_mul( _ptr__2694_tsp[_2704_j],_2676_n ),_2705_i );
                        _ptr__2695_rnativeidxs[_2703_k] = _ptr__2693_tnativeidxs[_2704_j];
                        ++ _2703_k;
                      }
                    }
                  }
                }
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2677_model,_2690_rshape,_2696_rsp,_2695_rnativeidxs)).as<::mosek::fusion::Variable>();
              }
            }
            else
            {
              {
                int32_t _2706_k = (int)0;
                int32_t _10390_ = (int)0;
                int32_t _10391_ = _2681_nnz;
                for (int32_t _2707_i = _10390_; _2707_i < _10391_; ++_2707_i)
                {
                  {
                    int64_t _2708_i0 = (_ptr__2694_tsp[_2707_i] / _2698_s0);
                    int64_t _2709_i1 = ((_ptr__2694_tsp[_2707_i] / _2699_s1) % _2685_d1);
                    int64_t _2710_i2 = (_ptr__2694_tsp[_2707_i] % _2687_d2);
                    int32_t _10392_ = (int)0;
                    int32_t _10393_ = _2676_n;
                    for (int32_t _2711_j = _10392_; _2711_j < _10393_; ++_2711_j)
                    {
                      {
                        _ptr__2696_rsp[_2706_k] = safe_add( safe_add( safe_mul( _2708_i0,_2697_ns0 ),safe_mul( safe_add( _2709_i1,safe_mul( _2685_d1,_2711_j ) ),_2687_d2 ) ),_2710_i2 );
                        _ptr__2695_rnativeidxs[_2706_k] = _ptr__2693_tnativeidxs[_2707_i];
                        ++ _2706_k;
                      }
                    }
                  }
                }
                int32_t _2712_maxdim = (int)1;
                int32_t _10394_ = (int)0;
                int32_t _10395_ = _2689_rnd;
                for (int32_t _2713_i = _10394_; _2713_i < _10395_; ++_2713_i)
                {
                  {
                    if ((_2712_maxdim < _ptr__2690_rshape[_2713_i]))
                    {
                      {
                        _2712_maxdim = _ptr__2690_rshape[_2713_i];
                      }
                    }
                    {}
                  }
                }
                std::shared_ptr< monty::ndarray< int32_t,1 > > _2714_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2682_rnnz)));
                _checked_ptr_1<int32_t> _ptr__2714_curperm(_2714_curperm ? _2714_curperm->raw() : nullptr,_2714_curperm ? _2714_curperm->size(0) : 0);
                std::shared_ptr< monty::ndarray< int32_t,1 > > _2715_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2682_rnnz)));
                _checked_ptr_1<int32_t> _ptr__2715_tgtperm(_2715_tgtperm ? _2715_tgtperm->raw() : nullptr,_2715_tgtperm ? _2715_tgtperm->size(0) : 0);
                {
                  std::shared_ptr< monty::ndarray< int32_t,1 > > _2716_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2712_maxdim,(int)1 ))));
                  _checked_ptr_1<int32_t> _ptr__2716_ptr(_2716_ptr ? _2716_ptr->raw() : nullptr,_2716_ptr ? _2716_ptr->size(0) : 0);
                  std::shared_ptr< monty::ndarray< int64_t,1 > > _2717_s = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2689_rnd)));
                  _checked_ptr_1<int64_t> _ptr__2717_s(_2717_s ? _2717_s->raw() : nullptr,_2717_s ? _2717_s->size(0) : 0);
                  _ptr__2717_s[(_2689_rnd - (int)1)] = (int)1;
                  int32_t _10396_ = (int)1;
                  int32_t _10397_ = _2689_rnd;
                  for (int32_t _2718_i = _10396_; _2718_i < _10397_; ++_2718_i)
                  {
                    {
                      _ptr__2717_s[((_2689_rnd - _2718_i) - (int)1)] = safe_mul( _ptr__2717_s[(_2689_rnd - _2718_i)],_ptr__2690_rshape[(_2689_rnd - _2718_i)] );
                    }
                  }
                  int32_t _10398_ = (int)0;
                  int32_t _10399_ = _2682_rnnz;
                  for (int32_t _2719_i = _10398_; _2719_i < _10399_; ++_2719_i)
                  {
                    {
                      _ptr__2714_curperm[_2719_i] = _2719_i;
                    }
                  }
                  int32_t _10400_ = (int)0;
                  int32_t _10401_ = _2689_rnd;
                  for (int32_t _2720_ii = _10400_; _2720_ii < _10401_; ++_2720_ii)
                  {
                    {
                      int32_t _2721_i = ((_2689_rnd - _2720_ii) - (int)1);
                      int32_t _10402_ = (int)0;
                      int32_t _10403_ = safe_add( _ptr__2690_rshape[_2721_i],(int)1 );
                      for (int32_t _2722_j = _10402_; _2722_j < _10403_; ++_2722_j)
                      {
                        {
                          _ptr__2716_ptr[_2722_j] = (int)0;
                        }
                      }
                      int32_t _10404_ = (int)0;
                      int32_t _10405_ = _2682_rnnz;
                      for (int32_t _2723_j = _10404_; _2723_j < _10405_; ++_2723_j)
                      {
                        {
                          ++ _ptr__2716_ptr[safe_add( ((_ptr__2696_rsp[_2723_j] / _ptr__2717_s[_2721_i]) % _ptr__2690_rshape[_2721_i]),(int)1 )];
                        }
                      }
                      int32_t _10406_ = (int)0;
                      int32_t _10407_ = _ptr__2690_rshape[_2721_i];
                      for (int32_t _2724_j = _10406_; _2724_j < _10407_; ++_2724_j)
                      {
                        {
                          _ptr__2716_ptr[safe_add( _2724_j,(int)1 )] += _ptr__2716_ptr[_2724_j];
                        }
                      }
                      int32_t _10408_ = (int)0;
                      int32_t _10409_ = _2682_rnnz;
                      for (int32_t _2725_jj = _10408_; _2725_jj < _10409_; ++_2725_jj)
                      {
                        {
                          int32_t _2726_j = _ptr__2714_curperm[_2725_jj];
                          int64_t _2727_idx = ((_ptr__2696_rsp[_2726_j] / _ptr__2717_s[_2721_i]) % _ptr__2690_rshape[_2721_i]);
                          _ptr__2715_tgtperm[_ptr__2716_ptr[_2727_idx]] = _2726_j;
                          ++ _ptr__2716_ptr[_2727_idx];
                        }
                      }
                      std::shared_ptr< monty::ndarray< int32_t,1 > > _2728_tmp = _2714_curperm;
                      _checked_ptr_1<int32_t> _ptr__2728_tmp(_2728_tmp ? _2728_tmp->raw() : nullptr,_2728_tmp ? _2728_tmp->size(0) : 0);
                      _2714_curperm = _2715_tgtperm;
                      _ptr__2714_curperm.update(_2714_curperm ? _2714_curperm->raw() : nullptr, _2714_curperm ? _2714_curperm->size(0) : 0);
                      _2715_tgtperm = _2728_tmp;
                      _ptr__2715_tgtperm.update(_2715_tgtperm ? _2715_tgtperm->raw() : nullptr, _2715_tgtperm ? _2715_tgtperm->size(0) : 0);
                    }
                  }
                }
                std::vector<int64_t> _10410_;
                for (int32_t _2729_i = (int)0, _10411_ = _2682_rnnz; _2729_i < _10411_ ; ++_2729_i)
                {
                  _10410_.push_back(_ptr__2696_rsp[_ptr__2714_curperm[_2729_i]]);
                }
                auto _10412_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10410_.size()),_10410_.begin(),_10410_.end()));
                std::vector<int64_t> _10413_;
                for (int32_t _2730_i = (int)0, _10414_ = _2682_rnnz; _2730_i < _10414_ ; ++_2730_i)
                {
                  _10413_.push_back(_ptr__2695_rnativeidxs[_ptr__2714_curperm[_2730_i]]);
                }
                auto _10415_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10413_.size()),_10413_.begin(),_10413_.end()));
                return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2677_model,_2690_rshape,_10412_,_10415_)).as<::mosek::fusion::Variable>();
              }
            }
          }
        }
      }
      else
      {
        {
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2731_tnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2681_nnz)));
          _checked_ptr_1<int64_t> _ptr__2731_tnativeidxs(_2731_tnativeidxs ? _2731_tnativeidxs->raw() : nullptr,_2731_tnativeidxs ? _2731_tnativeidxs->size(0) : 0);
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2732_rnativeidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(safe_mul( _2681_nnz,_2676_n ))));
          _checked_ptr_1<int64_t> _ptr__2732_rnativeidxs(_2732_rnativeidxs ? _2732_rnativeidxs->raw() : nullptr,_2732_rnativeidxs ? _2732_rnativeidxs->size(0) : 0);
          _2674_v->inst((int)0,_2731_tnativeidxs);
          int32_t _2733_s0 = safe_mul( _2686_nd1,_2687_d2 );
          int32_t _2734_s1 = _2687_d2;
          int32_t _2735_k = (int)0;
          int32_t _10416_ = (int)0;
          int32_t _10417_ = _2683_d0;
          for (int32_t _2736_i0 = _10416_; _2736_i0 < _10417_; ++_2736_i0)
          {
            {
              int32_t _10418_ = (int)0;
              int32_t _10419_ = _2685_d1;
              for (int32_t _2737_i1 = _10418_; _2737_i1 < _10419_; ++_2737_i1)
              {
                {
                  int32_t _10420_ = (int)0;
                  int32_t _10421_ = _2687_d2;
                  for (int32_t _2738_i2 = _10420_; _2738_i2 < _10421_; ++_2738_i2)
                  {
                    {
                      int32_t _10422_ = (int)0;
                      int32_t _10423_ = _2676_n;
                      for (int32_t _2739_j = _10422_; _2739_j < _10423_; ++_2739_j)
                      {
                        {
                          _ptr__2732_rnativeidxs[safe_add( safe_add( safe_mul( _2736_i0,_2733_s0 ),safe_mul( safe_add( _2737_i1,safe_mul( _2739_j,_2685_d1 ) ),_2734_s1 ) ),_2738_i2 )] = _ptr__2731_tnativeidxs[_2735_k];
                        }
                      }
                      ++ _2735_k;
                    }
                  }
                }
              }
            }
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2677_model,_2690_rshape,nullptr,_2732_rnativeidxs)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.Var.drepeat

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:450:6-454:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >,1 > > _2740_vlist) { return mosek::fusion::p_Var::stack(_2740_vlist); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >,1 > > _2740_vlist)
{
  std::vector<monty::rc_ptr< ::mosek::fusion::Variable >> _10424_;
  for (int32_t _2741_i = (int)0, _10425_ = (int)((_2740_vlist)->size(0)); _2741_i < _10425_ ; ++_2741_i)
  {
    _10424_.push_back(::mosek::fusion::p_Var::dstack(((*_2740_vlist)(_2741_i)),(int)1));
  }
  auto _10426_ = std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >(monty::shape(_10424_.size()),_10424_.begin(),_10424_.end()));
  return ::mosek::fusion::p_Var::dstack(_10426_,(int)0);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:414:6-420:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2742_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2743_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2744_v3) { return mosek::fusion::p_Var::vstack(_2742_v1,_2743_v2,_2744_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(monty::rc_ptr< ::mosek::fusion::Variable > _2742_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2743_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2744_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2742_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2743_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2744_v3.get())}),(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:396:6-401:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2745_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2746_v2) { return mosek::fusion::p_Var::vstack(_2745_v1,_2746_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(monty::rc_ptr< ::mosek::fusion::Variable > _2745_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2746_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2745_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2746_v2.get())}),(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.vstack
// Method mosek.fusion.Var.vstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:377:6-381:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::vstack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2747_v) { return mosek::fusion::p_Var::vstack(_2747_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::vstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2747_v)
{
  return ::mosek::fusion::p_Var::dstack(_2747_v,(int)0);
}
// End mosek.fusion.Var.vstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:357:6-363:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2748_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2749_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2750_v3) { return mosek::fusion::p_Var::hstack(_2748_v1,_2749_v2,_2750_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(monty::rc_ptr< ::mosek::fusion::Variable > _2748_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2749_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2750_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2748_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2749_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2750_v3.get())}),(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:336:6-341:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2751_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2752_v2) { return mosek::fusion::p_Var::hstack(_2751_v1,_2752_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(monty::rc_ptr< ::mosek::fusion::Variable > _2751_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2752_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2751_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2752_v2.get())}),(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.hstack
// Method mosek.fusion.Var.hstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:315:6-319:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::hstack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2753_v) { return mosek::fusion::p_Var::hstack(_2753_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::hstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2753_v)
{
  return ::mosek::fusion::p_Var::dstack(_2753_v,(int)1);
}
// End mosek.fusion.Var.hstack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:298:6-91
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2754_v,
    int32_t _2755_dim) { return mosek::fusion::p_Var::stack(_2754_v,_2755_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2754_v,int32_t _2755_dim)
{
  return ::mosek::fusion::p_Var::stack(_2755_dim,_2754_v);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:297:6-130
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2756_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2757_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2758_v3,
    int32_t _2759_dim) { return mosek::fusion::p_Var::stack(_2756_v1,_2757_v2,_2758_v3,_2759_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(monty::rc_ptr< ::mosek::fusion::Variable > _2756_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2757_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2758_v3,int32_t _2759_dim)
{
  return ::mosek::fusion::p_Var::stack(_2759_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2756_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2757_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2758_v3.get())}));
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:296:6-114
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2760_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2761_v2,
    int32_t _2762_dim) { return mosek::fusion::p_Var::stack(_2760_v1,_2761_v2,_2762_dim); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(monty::rc_ptr< ::mosek::fusion::Variable > _2760_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2761_v2,int32_t _2762_dim)
{
  return ::mosek::fusion::p_Var::stack(_2762_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2760_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2761_v2.get())}));
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:290:6-293:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2763_dim,
    std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2764_v) { return mosek::fusion::p_Var::stack(_2763_dim,_2764_v); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2763_dim,std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2764_v)
{
  int32_t _2765_maxnd = _2763_dim;
  int32_t _10427_ = (int)0;
  int32_t _10428_ = (int)((_2764_v)->size(0));
  for (int32_t _2766_i = _10427_; _2766_i < _10428_; ++_2766_i)
  {
    {
      if ((_2765_maxnd < ((*_2764_v)(_2766_i))->getND()))
      {
        {
          _2765_maxnd = ((*_2764_v)(_2766_i))->getND();
        }
      }
      {}
    }
  }
  return ::mosek::fusion::p_Var::dstack(_2764_v,_2763_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:289:6-131
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2767_dim,
    monty::rc_ptr< ::mosek::fusion::Variable > _2768_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2769_v2,
    monty::rc_ptr< ::mosek::fusion::Variable > _2770_v3) { return mosek::fusion::p_Var::stack(_2767_dim,_2768_v1,_2769_v2,_2770_v3); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2767_dim,monty::rc_ptr< ::mosek::fusion::Variable > _2768_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2769_v2,monty::rc_ptr< ::mosek::fusion::Variable > _2770_v3)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2768_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2769_v2.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2770_v3.get())}),_2767_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.stack
// Method mosek.fusion.Var.stack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:288:6-115
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::stack
  ( int32_t _2771_dim,
    monty::rc_ptr< ::mosek::fusion::Variable > _2772_v1,
    monty::rc_ptr< ::mosek::fusion::Variable > _2773_v2) { return mosek::fusion::p_Var::stack(_2771_dim,_2772_v1,_2773_v2); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::stack(int32_t _2771_dim,monty::rc_ptr< ::mosek::fusion::Variable > _2772_v1,monty::rc_ptr< ::mosek::fusion::Variable > _2773_v2)
{
  return ::mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > >(new monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 >{::monty::rc_ptr<::mosek::fusion::Variable>(_2772_v1.get()),::monty::rc_ptr<::mosek::fusion::Variable>(_2773_v2.get())}),_2771_dim);
}
// End mosek.fusion.Var.stack

// Begin mosek.fusion.Var.promote
// Method mosek.fusion.Var.promote @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:275:6-285:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::Var::promote
  ( monty::rc_ptr< ::mosek::fusion::Variable > _2774_v,
    int32_t _2775_nd) { return mosek::fusion::p_Var::promote(_2774_v,_2775_nd); }
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::promote(monty::rc_ptr< ::mosek::fusion::Variable > _2774_v,int32_t _2775_nd)
{
  if ((_2775_nd > _2774_v->getND()))
  {
    {
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2776_shape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2775_nd)));
      _checked_ptr_1<int32_t> _ptr__2776_shape(_2776_shape ? _2776_shape->raw() : nullptr,_2776_shape ? _2776_shape->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2777_sh = _2774_v->getShape();
      _checked_ptr_1<int32_t> _ptr__2777_sh(_2777_sh ? _2777_sh->raw() : nullptr,_2777_sh ? _2777_sh->size(0) : 0);
      int32_t _10429_ = (int)0;
      int32_t _10430_ = (int)((_2777_sh)->size(0));
      for (int32_t _2778_i = _10429_; _2778_i < _10430_; ++_2778_i)
      {
        {
          _ptr__2776_shape[_2778_i] = _ptr__2777_sh[_2778_i];
        }
      }
      int32_t _10431_ = (int)((_2777_sh)->size(0));
      int32_t _10432_ = _2775_nd;
      for (int32_t _2779_i = _10431_; _2779_i < _10432_; ++_2779_i)
      {
        {
          _ptr__2776_shape[_2779_i] = (int)1;
        }
      }
      return _2774_v->__mosek_2fusion_2Variable__reshape(_2776_shape);
    }
  }
  else
  {
    {
      return _2774_v;
    }
  }
}
// End mosek.fusion.Var.promote

// Begin mosek.fusion.Var.dstack
// Method mosek.fusion.Var.dstack @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/Var.monty:151:6-267:7
monty::rc_ptr< ::mosek::fusion::Variable > mosek::fusion::p_Var::dstack(std::shared_ptr< monty::ndarray< monty::rc_ptr< ::mosek::fusion::Variable >,1 > > _2780_v,int32_t _2781_dim)
{
  if ((_2781_dim < (int)0))
  {
    {
      throw ::mosek::fusion::DimensionError(std::string ("Invalid stacking dimension"));
    }
  }
  {}
  monty::rc_ptr< ::mosek::fusion::Model > _2782_m0 = nullptr;
  {
    int32_t _10433_ = (int)0;
    int32_t _10434_ = (int)((_2780_v)->size(0));
    for (int32_t _2783_i = _10433_; _2783_i < _10434_; ++_2783_i)
    {
      {
        if ((NULL == _2782_m0.get()))
        {
          {
            _2782_m0 = ((*_2780_v)(_2783_i))->__mosek_2fusion_2Variable__getModel();
          }
        }
        else if(((NULL != ((*_2780_v)(_2783_i))->__mosek_2fusion_2Variable__getModel().get()) && (((*_2780_v)(_2783_i))->__mosek_2fusion_2Variable__getModel() != _2782_m0)))
        {
          {
            monty::rc_ptr< ::mosek::fusion::Utils::StringBuffer > _2784_sb = ::mosek::fusion::Utils::p_StringBuffer::_new_StringBuffer();
            std::string _10435_;
            bool _10436_ = ((int)((mosek::fusion::p_Model::_get_impl(_2782_m0.get())->getName()).size()) > (int)0);
            if (_10436_)
            {
              _10435_ = mosek::fusion::p_Model::_get_impl(_2782_m0.get())->getName();
            }
            else
            {
              _10435_ = std::string ("?");
            }
            std::string _2785_m1name = _10435_;
            std::string _10437_;
            bool _10438_ = ((int)((mosek::fusion::p_Model::_get_impl(((*_2780_v)(_2783_i))->__mosek_2fusion_2Variable__getModel().get())->getName()).size()) > (int)0);
            if (_10438_)
            {
              _10437_ = mosek::fusion::p_Model::_get_impl(((*_2780_v)(_2783_i))->__mosek_2fusion_2Variable__getModel().get())->getName();
            }
            else
            {
              _10437_ = std::string ("?");
            }
            std::string _2786_m2name = _10437_;
            _2784_sb->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("Variables belong to different models: '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2785_m1name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("' and '"))->__mosek_2fusion_2Utils_2StringBuffer__a(_2786_m2name)->__mosek_2fusion_2Utils_2StringBuffer__a(std::string ("'"));
            throw ::mosek::fusion::ModelError(_2784_sb->toString());
          }
        }
        {}
      }
    }
  }
  if (((int)((_2780_v)->size(0)) == (int)0))
  {
    {
      return (::mosek::fusion::p_NilVariable::_new_NilVariable()).as<::mosek::fusion::Variable>();
    }
  }
  else if(((int)((_2780_v)->size(0)) == (int)1))
  {
    {
      return ((*_2780_v)((int)0));
    }
  }
  else
  {
    {
      int32_t _2787_n = (int)((_2780_v)->size(0));
      std::vector<int32_t> _10439_;
      for (int32_t _2789_i = (int)0, _10440_ = _2787_n; _2789_i < _10440_ ; ++_2789_i)
      {
        _10439_.push_back(((*_2780_v)(_2789_i))->numInst());
      }
      auto _10441_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10439_.size()),_10439_.begin(),_10439_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2788_numnz = _10441_;
      _checked_ptr_1<int32_t> _ptr__2788_numnz(_2788_numnz ? _2788_numnz->raw() : nullptr,_2788_numnz ? _2788_numnz->size(0) : 0);
      std::vector<std::shared_ptr< monty::ndarray< int32_t,1 > >> _10442_;
      for (int32_t _2791_i = (int)0, _10443_ = _2787_n; _2791_i < _10443_ ; ++_2791_i)
      {
        _10442_.push_back(((*_2780_v)(_2791_i))->getShape());
      }
      auto _10444_ = std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > >(new monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 >(monty::shape(_10442_.size()),_10442_.begin(),_10442_.end()));
      std::shared_ptr< monty::ndarray< std::shared_ptr< monty::ndarray< int32_t,1 > >,1 > > _2790_shapes = _10444_;
      std::vector<int32_t> _10445_;
      for (int32_t _2793_i = (int)0, _10446_ = _2787_n; _2793_i < _10446_ ; ++_2793_i)
      {
        _10445_.push_back((int)((((*_2790_shapes)(_2793_i)))->size(0)));
      }
      auto _10447_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10445_.size()),_10445_.begin(),_10445_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2792_nds = _10447_;
      _checked_ptr_1<int32_t> _ptr__2792_nds(_2792_nds ? _2792_nds->raw() : nullptr,_2792_nds ? _2792_nds->size(0) : 0);
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2794_shape0 = ((*_2790_shapes)((int)0));
      _checked_ptr_1<int32_t> _ptr__2794_shape0(_2794_shape0 ? _2794_shape0->raw() : nullptr,_2794_shape0 ? _2794_shape0->size(0) : 0);
      int32_t _2795_nd0 = _ptr__2792_nds[(int)0];
      int32_t _2796_rnnz = (int)0;
      int32_t _10448_ = (int)0;
      int32_t _10449_ = _2787_n;
      for (int32_t _2797_i = _10448_; _2797_i < _10449_; ++_2797_i)
      {
        {
          _2796_rnnz += _ptr__2788_numnz[_2797_i];
        }
      }
      int32_t _2798_rnd = safe_add( _2781_dim,(int)1 );
      int32_t _10450_ = (int)0;
      int32_t _10451_ = _2787_n;
      for (int32_t _2799_i = _10450_; _2799_i < _10451_; ++_2799_i)
      {
        {
          if ((_2798_rnd < _ptr__2792_nds[_2799_i]))
          {
            {
              _2798_rnd = _ptr__2792_nds[_2799_i];
            }
          }
          {}
        }
      }
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2800_rshape = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2798_rnd)));
      _checked_ptr_1<int32_t> _ptr__2800_rshape(_2800_rshape ? _2800_rshape->raw() : nullptr,_2800_rshape ? _2800_rshape->size(0) : 0);
      int32_t _10452_ = (int)0;
      int32_t _10453_ = _2795_nd0;
      for (int32_t _2801_i = _10452_; _2801_i < _10453_; ++_2801_i)
      {
        {
          _ptr__2800_rshape[_2801_i] = _ptr__2794_shape0[_2801_i];
        }
      }
      int32_t _10454_ = _2795_nd0;
      int32_t _10455_ = _2798_rnd;
      for (int32_t _2802_i = _10454_; _2802_i < _10455_; ++_2802_i)
      {
        {
          _ptr__2800_rshape[_2802_i] = (int)1;
        }
      }
      _ptr__2800_rshape[_2781_dim] = (int)0;
      int32_t _10456_ = (int)0;
      int32_t _10457_ = _2787_n;
      for (int32_t _2803_i = _10456_; _2803_i < _10457_; ++_2803_i)
      {
        {
          int32_t _10458_;
          bool _10459_ = ((int)((((*_2790_shapes)(_2803_i)))->size(0)) > _2781_dim);
          if (_10459_)
          {
            _10458_ = ((*((*_2790_shapes)(_2803_i)))(_2781_dim));
          }
          else
          {
            _10458_ = (int)1;
          }
          _ptr__2800_rshape[_2781_dim] += _10458_;
        }
      }
      int32_t _10460_ = (int)0;
      int32_t _10461_ = _2787_n;
      for (int32_t _2804_i = _10460_; _2804_i < _10461_; ++_2804_i)
      {
        {
          int32_t _10462_ = (int)0;
          int32_t _10463_ = (int)((((*_2790_shapes)(_2804_i)))->size(0));
          for (int32_t _2805_j = _10462_; _2805_j < _10463_; ++_2805_j)
          {
            {
              if (((_2805_j != _2781_dim) && (((*((*_2790_shapes)(_2804_i)))(_2805_j)) != _ptr__2800_rshape[_2805_j])))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Variable dimensions do not match"));
                }
              }
              {}
            }
          }
          int32_t _10464_ = (int)((((*_2790_shapes)(_2804_i)))->size(0));
          int32_t _10465_ = _2798_rnd;
          for (int32_t _2806_j = _10464_; _2806_j < _10465_; ++_2806_j)
          {
            {
              if (((_2806_j != _2781_dim) && ((int)1 != _ptr__2800_rshape[_2806_j])))
              {
                {
                  throw ::mosek::fusion::DimensionError(std::string ("Variable dimensions do not match"));
                }
              }
              {}
            }
          }
        }
      }
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2807_rnidxs = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2796_rnnz)));
      _checked_ptr_1<int64_t> _ptr__2807_rnidxs(_2807_rnidxs ? _2807_rnidxs->raw() : nullptr,_2807_rnidxs ? _2807_rnidxs->size(0) : 0);
      std::shared_ptr< monty::ndarray< int64_t,1 > > _2808_rsp = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2796_rnnz)));
      _checked_ptr_1<int64_t> _ptr__2808_rsp(_2808_rsp ? _2808_rsp->raw() : nullptr,_2808_rsp ? _2808_rsp->size(0) : 0);
      std::vector<int32_t> _10466_;
      for (int32_t _2810_i = (int)0, _10467_ = (int)((_2780_v)->size(0)); _2810_i < _10467_ ; ++_2810_i)
      {
        int32_t _10468_;
        bool _10469_ = (_2781_dim < (int)((((*_2790_shapes)(_2810_i)))->size(0)));
        if (_10469_)
        {
          _10468_ = ((*((*_2790_shapes)(_2810_i)))(_2781_dim));
        }
        else
        {
          _10468_ = (int)1;
        }
        _10466_.push_back(_10468_);
      }
      auto _10470_ = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_10466_.size()),_10466_.begin(),_10466_.end()));
      std::shared_ptr< monty::ndarray< int32_t,1 > > _2809_sdims = _10470_;
      _checked_ptr_1<int32_t> _ptr__2809_sdims(_2809_sdims ? _2809_sdims->raw() : nullptr,_2809_sdims ? _2809_sdims->size(0) : 0);
      int32_t _2811_d0 = (int)1;
      int32_t _10471_ = (int)0;
      int32_t _10472_ = _2781_dim;
      for (int32_t _2812_i = _10471_; _2812_i < _10472_; ++_2812_i)
      {
        {
          int32_t _10473_;
          bool _10474_ = (_2812_i < (int)((_2794_shape0)->size(0)));
          if (_10474_)
          {
            _10473_ = _ptr__2794_shape0[_2812_i];
          }
          else
          {
            _10473_ = (int)1;
          }
          _2811_d0 *= _10473_;
        }
      }
      int32_t _2813_d1 = (int)0;
      int32_t _10475_ = (int)0;
      int32_t _10476_ = _2787_n;
      for (int32_t _2814_i = _10475_; _2814_i < _10476_; ++_2814_i)
      {
        {
          _2813_d1 += _ptr__2809_sdims[_2814_i];
        }
      }
      int32_t _2815_d2 = (int)1;
      int32_t _10477_ = safe_add( _2781_dim,(int)1 );
      int32_t _10478_ = (int)((_2794_shape0)->size(0));
      for (int32_t _2816_i = _10477_; _2816_i < _10478_; ++_2816_i)
      {
        {
          _2815_d2 *= _ptr__2794_shape0[_2816_i];
        }
      }
      if ((_2811_d0 == (int)1))
      {
        {
          int32_t _2817_k = (int)0;
          int64_t _2818_kspofs = (int)0;
          int32_t _10479_ = (int)0;
          int32_t _10480_ = _2787_n;
          for (int32_t _2819_i = _10479_; _2819_i < _10480_; ++_2819_i)
          {
            {
              ((*_2780_v)(_2819_i))->inst(_2817_k,_2808_rsp,_2817_k,_2807_rnidxs);
              int32_t _10481_ = (int)0;
              int32_t _10482_ = _ptr__2788_numnz[_2819_i];
              for (int32_t _2820_j = _10481_; _2820_j < _10482_; ++_2820_j)
              {
                {
                  _ptr__2808_rsp[_2817_k] += _2818_kspofs;
                  ++ _2817_k;
                }
              }
              _2818_kspofs += ::mosek::fusion::p_Set::size(((*_2790_shapes)(_2819_i)));
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _10483_;
          bool _10484_ = (_2796_rnnz < safe_mul( safe_mul( _2811_d0,_2813_d1 ),_2815_d2 ));
          if (_10484_)
          {
            _10483_ = _2808_rsp;
          }
          else
          {
            _10483_ = nullptr;
          }
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2782_m0,_2800_rshape,_10483_,_2807_rnidxs)).as<::mosek::fusion::Variable>();
        }
      }
      else
      {
        {
          int32_t _2821_b = (int)0;
          int32_t _10485_ = (int)0;
          int32_t _10486_ = _2787_n;
          for (int32_t _2822_i = _10485_; _2822_i < _10486_; ++_2822_i)
          {
            {
              _2821_b += ((*_2780_v)(_2822_i))->inst(_2821_b,_2808_rsp,_2821_b,_2807_rnidxs);
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2823_curperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2796_rnnz)));
          _checked_ptr_1<int32_t> _ptr__2823_curperm(_2823_curperm ? _2823_curperm->raw() : nullptr,_2823_curperm ? _2823_curperm->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2824_tgtperm = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2796_rnnz)));
          _checked_ptr_1<int32_t> _ptr__2824_tgtperm(_2824_tgtperm ? _2824_tgtperm->raw() : nullptr,_2824_tgtperm ? _2824_tgtperm->size(0) : 0);
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2825_offset = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(_2787_n)));
          _checked_ptr_1<int32_t> _ptr__2825_offset(_2825_offset ? _2825_offset->raw() : nullptr,_2825_offset ? _2825_offset->size(0) : 0);
          int32_t _10487_ = (int)0;
          int32_t _10488_ = (_2787_n - (int)1);
          for (int32_t _2826_i = _10487_; _2826_i < _10488_; ++_2826_i)
          {
            {
              _ptr__2825_offset[safe_add( _2826_i,(int)1 )] = safe_add( _ptr__2825_offset[_2826_i],_ptr__2809_sdims[_2826_i] );
            }
          }
          int32_t _10489_ = (int)0;
          int32_t _10490_ = _2796_rnnz;
          for (int32_t _2827_i = _10489_; _2827_i < _10490_; ++_2827_i)
          {
            {
              _ptr__2823_curperm[_2827_i] = _2827_i;
            }
          }
          int32_t _2828_k = (int)0;
          int32_t _10491_ = (int)0;
          int32_t _10492_ = _2787_n;
          for (int32_t _2829_i = _10491_; _2829_i < _10492_; ++_2829_i)
          {
            {
              int32_t _10493_ = (int)0;
              int32_t _10494_ = _ptr__2788_numnz[_2829_i];
              for (int32_t _2830_j = _10493_; _2830_j < _10494_; ++_2830_j)
              {
                {
                  int64_t _2831_i0 = (_ptr__2808_rsp[_2828_k] / safe_mul( _ptr__2809_sdims[_2829_i],_2815_d2 ));
                  int64_t _2832_i1 = ((_ptr__2808_rsp[_2828_k] / _2815_d2) % _ptr__2809_sdims[_2829_i]);
                  int64_t _2833_i2 = (_ptr__2808_rsp[_2828_k] % _2815_d2);
                  _ptr__2808_rsp[_2828_k] = safe_add( safe_mul( safe_add( safe_add( safe_mul( _2831_i0,_2813_d1 ),_2832_i1 ),_ptr__2825_offset[_2829_i] ),_2815_d2 ),_2833_i2 );
                  ++ _2828_k;
                }
              }
            }
          }
          int32_t _2834_maxdim = (int)1;
          int32_t _10495_ = (int)0;
          int32_t _10496_ = _2798_rnd;
          for (int32_t _2835_i = _10495_; _2835_i < _10496_; ++_2835_i)
          {
            {
              int32_t _10497_;
              bool _10498_ = (_2834_maxdim < _ptr__2800_rshape[_2835_i]);
              if (_10498_)
              {
                _10497_ = _ptr__2800_rshape[_2835_i];
              }
              else
              {
                _10497_ = _2834_maxdim;
              }
              _2834_maxdim = _10497_;
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _2836_s = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_2798_rnd)));
          _checked_ptr_1<int64_t> _ptr__2836_s(_2836_s ? _2836_s->raw() : nullptr,_2836_s ? _2836_s->size(0) : 0);
          _ptr__2836_s[(_2798_rnd - (int)1)] = (int)1;
          int32_t _10499_ = (int)1;
          int32_t _10500_ = _2798_rnd;
          for (int32_t _2837_i = _10499_; _2837_i < _10500_; ++_2837_i)
          {
            {
              _ptr__2836_s[((_2798_rnd - _2837_i) - (int)1)] = safe_mul( _ptr__2836_s[(_2798_rnd - _2837_i)],_ptr__2800_rshape[(_2798_rnd - _2837_i)] );
            }
          }
          std::shared_ptr< monty::ndarray< int32_t,1 > > _2838_ptr = std::shared_ptr< monty::ndarray< int32_t,1 > >(new monty::ndarray< int32_t,1 >(monty::shape(safe_add( _2834_maxdim,(int)1 ))));
          _checked_ptr_1<int32_t> _ptr__2838_ptr(_2838_ptr ? _2838_ptr->raw() : nullptr,_2838_ptr ? _2838_ptr->size(0) : 0);
          int32_t _10501_ = (int)0;
          int32_t _10502_ = _2798_rnd;
          for (int32_t _2839_ii = _10501_; _2839_ii < _10502_; ++_2839_ii)
          {
            {
              int32_t _2840_i = ((_2798_rnd - _2839_ii) - (int)1);
              int32_t _10503_ = (int)0;
              int32_t _10504_ = safe_add( _ptr__2800_rshape[_2840_i],(int)1 );
              for (int32_t _2841_j = _10503_; _2841_j < _10504_; ++_2841_j)
              {
                {
                  _ptr__2838_ptr[_2841_j] = (int)0;
                }
              }
              int32_t _10505_ = (int)0;
              int32_t _10506_ = _2796_rnnz;
              for (int32_t _2842_j = _10505_; _2842_j < _10506_; ++_2842_j)
              {
                {
                  ++ _ptr__2838_ptr[safe_add( ((_ptr__2808_rsp[_2842_j] / _ptr__2836_s[_2840_i]) % _ptr__2800_rshape[_2840_i]),(int)1 )];
                }
              }
              int32_t _10507_ = (int)0;
              int32_t _10508_ = _ptr__2800_rshape[_2840_i];
              for (int32_t _2843_j = _10507_; _2843_j < _10508_; ++_2843_j)
              {
                {
                  _ptr__2838_ptr[safe_add( _2843_j,(int)1 )] += _ptr__2838_ptr[_2843_j];
                }
              }
              int32_t _10509_ = (int)0;
              int32_t _10510_ = _2796_rnnz;
              for (int32_t _2844_jj = _10509_; _2844_jj < _10510_; ++_2844_jj)
              {
                {
                  int32_t _2845_j = _ptr__2823_curperm[_2844_jj];
                  int64_t _2846_idx = ((_ptr__2808_rsp[_2845_j] / _ptr__2836_s[_2840_i]) % _ptr__2800_rshape[_2840_i]);
                  _ptr__2824_tgtperm[_ptr__2838_ptr[_2846_idx]] = _2845_j;
                  ++ _ptr__2838_ptr[_2846_idx];
                }
              }
              std::shared_ptr< monty::ndarray< int32_t,1 > > _2847_tmp = _2823_curperm;
              _checked_ptr_1<int32_t> _ptr__2847_tmp(_2847_tmp ? _2847_tmp->raw() : nullptr,_2847_tmp ? _2847_tmp->size(0) : 0);
              _2823_curperm = _2824_tgtperm;
              _ptr__2823_curperm.update(_2823_curperm ? _2823_curperm->raw() : nullptr, _2823_curperm ? _2823_curperm->size(0) : 0);
              _2824_tgtperm = _2847_tmp;
              _ptr__2824_tgtperm.update(_2824_tgtperm ? _2824_tgtperm->raw() : nullptr, _2824_tgtperm ? _2824_tgtperm->size(0) : 0);
            }
          }
          std::shared_ptr< monty::ndarray< int64_t,1 > > _10511_;
          bool _10512_ = (_2796_rnnz < safe_mul( safe_mul( _2811_d0,_2813_d1 ),_2815_d2 ));
          if (_10512_)
          {
            std::vector<int64_t> _10513_;
            for (int32_t _2848_i = (int)0, _10514_ = _2796_rnnz; _2848_i < _10514_ ; ++_2848_i)
            {
              _10513_.push_back(_ptr__2808_rsp[_ptr__2823_curperm[_2848_i]]);
            }
            auto _10515_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10513_.size()),_10513_.begin(),_10513_.end()));
            _10511_ = _10515_;
          }
          else
          {
            _10511_ = nullptr;
          }
          std::vector<int64_t> _10516_;
          for (int32_t _2849_i = (int)0, _10517_ = _2796_rnnz; _2849_i < _10517_ ; ++_2849_i)
          {
            _10516_.push_back(_ptr__2807_rnidxs[_ptr__2823_curperm[_2849_i]]);
          }
          auto _10518_ = std::shared_ptr< monty::ndarray< int64_t,1 > >(new monty::ndarray< int64_t,1 >(monty::shape(_10516_.size()),_10516_.begin(),_10516_.end()));
          return (::mosek::fusion::p_SliceVariable::_new_SliceVariable(_2782_m0,_2800_rshape,_10511_,_10518_)).as<::mosek::fusion::Variable>();
        }
      }
    }
  }
}
// End mosek.fusion.Var.dstack

void mosek::fusion::p_Var::destroy()
{
}
void mosek::fusion::Var::destroy() { mosek::fusion::p_Var::_get_impl(this)->destroy(); }
// } class Var
// class BoundInterfaceConstraint { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:26:4-69:5
mosek::fusion::BoundInterfaceConstraint::BoundInterfaceConstraint(mosek::fusion::p_BoundInterfaceConstraint *_impl) : mosek::fusion::SliceConstraint(_impl) { /*std::cout << "BoundInterfaceConstraint()" << this << std::endl; */ }
mosek::fusion::BoundInterfaceConstraint::~BoundInterfaceConstraint() {  /* std::cout << "~BoundInterfaceConstraint()" << this << std::endl << std::flush; */ }
mosek::fusion::p_BoundInterfaceConstraint::p_BoundInterfaceConstraint
  (::mosek::fusion::BoundInterfaceConstraint * _pubthis) :     mosek::fusion::p_SliceConstraint(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.BoundInterfaceConstraint.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:39:6-47:7
mosek::fusion::BoundInterfaceConstraint::t mosek::fusion::p_BoundInterfaceConstraint::_new_BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2850_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2851_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2852_nativeidxs,
    bool _2853_islower)
{ return new mosek::fusion::BoundInterfaceConstraint(_2850_m,_2851_shape,_2852_nativeidxs,_2853_islower); }
mosek::fusion::BoundInterfaceConstraint::BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2850_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2851_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2852_nativeidxs,
    bool _2853_islower) : 
   BoundInterfaceConstraint(new mosek::fusion::p_BoundInterfaceConstraint(this))
{ mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->_initialize(_2850_m,_2851_shape,_2852_nativeidxs,_2853_islower); }
void mosek::fusion::p_BoundInterfaceConstraint::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2850_m,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2851_shape,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2852_nativeidxs,
    bool _2853_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2851_shape(_2851_shape ? _2851_shape->raw() : nullptr, _2851_shape ? _2851_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2852_nativeidxs(_2852_nativeidxs ? _2852_nativeidxs->raw() : nullptr, _2852_nativeidxs ? _2852_nativeidxs->size(0) : 0);
    mosek::fusion::p_SliceConstraint::_initialize(_2850_m,_2851_shape,_2852_nativeidxs);
    mosek::fusion::p_BoundInterfaceConstraint::_get_impl(::mosek::fusion::BoundInterfaceConstraint::t(_pubthis))->islower = _2853_islower;
  }
}
// mosek.fusion.BoundInterfaceConstraint.ctor

// mosek.fusion.BoundInterfaceConstraint.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:31:6-37:7
mosek::fusion::BoundInterfaceConstraint::t mosek::fusion::p_BoundInterfaceConstraint::_new_BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::SliceConstraint > _2854_c,
    bool _2855_islower)
{ return new mosek::fusion::BoundInterfaceConstraint(_2854_c,_2855_islower); }
mosek::fusion::BoundInterfaceConstraint::BoundInterfaceConstraint
  (monty::rc_ptr< ::mosek::fusion::SliceConstraint > _2854_c,
    bool _2855_islower) : 
   BoundInterfaceConstraint(new mosek::fusion::p_BoundInterfaceConstraint(this))
{ mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->_initialize(_2854_c,_2855_islower); }
void mosek::fusion::p_BoundInterfaceConstraint::_initialize  (monty::rc_ptr< ::mosek::fusion::SliceConstraint > _2854_c,
    bool _2855_islower)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_SliceConstraint::_initialize(_2854_c);
    mosek::fusion::p_BoundInterfaceConstraint::_get_impl(::mosek::fusion::BoundInterfaceConstraint::t(_pubthis))->islower = _2855_islower;
  }
}
// mosek.fusion.BoundInterfaceConstraint.ctor

// Begin mosek.fusion.BoundInterfaceConstraint.dual
// Method mosek.fusion.BoundInterfaceConstraint.dual @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:64:6-68:7
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::BoundInterfaceConstraint :: dual() { return mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->dual(); }
std::shared_ptr< monty::ndarray< double,1 > > mosek::fusion::p_BoundInterfaceConstraint::dual()
{
  std::shared_ptr< monty::ndarray< double,1 > > _2856_target = std::shared_ptr< monty::ndarray< double,1 > >(new monty::ndarray< double,1 >(monty::shape(_pubthis->getSize())));
  _checked_ptr_1<double> _ptr__2856_target(_2856_target ? _2856_target->raw() : nullptr,_2856_target ? _2856_target->size(0) : 0);
  dual_lu((int)0,_2856_target,islower);
  return _2856_target;
}
// End mosek.fusion.BoundInterfaceConstraint.dual

// Begin mosek.fusion.BoundInterfaceConstraint.slice
// Method mosek.fusion.BoundInterfaceConstraint.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:62:6-124
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint :: __mosek_2fusion_2BoundInterfaceConstraint__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2857_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2858_lasta) { return mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->__mosek_2fusion_2BoundInterfaceConstraint__slice(_2857_firsta,_2858_lasta); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2857_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2858_lasta) { return __mosek_2fusion_2BoundInterfaceConstraint__slice(_2857_firsta,_2858_lasta); }
/* override: mosek.fusion.Constraint.slice*/
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::__mosek_2fusion_2Constraint__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2857_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2858_lasta) { return __mosek_2fusion_2BoundInterfaceConstraint__slice(_2857_firsta,_2858_lasta); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_BoundInterfaceConstraint::__mosek_2fusion_2BoundInterfaceConstraint__slice(std::shared_ptr< monty::ndarray< int32_t,1 > > _2857_firsta,std::shared_ptr< monty::ndarray< int32_t,1 > > _2858_lasta)
{
  _checked_ptr_1<int32_t> _ptr__2857_firsta(_2857_firsta ? _2857_firsta->raw() : nullptr, _2857_firsta ? _2857_firsta->size(0) : 0);
  _checked_ptr_1<int32_t> _ptr__2858_lasta(_2858_lasta ? _2858_lasta->raw() : nullptr, _2858_lasta ? _2858_lasta->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Constraint > _2859_c = ::mosek::fusion::p_Constraint::__mosek_2fusion_2Constraint__slice(_2857_firsta,_2858_lasta);
  return (__mosek_2fusion_2BoundInterfaceConstraint__from_(_2859_c)).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.BoundInterfaceConstraint.slice

// Begin mosek.fusion.BoundInterfaceConstraint.slice
// Method mosek.fusion.BoundInterfaceConstraint.slice @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:61:6-122
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint :: __mosek_2fusion_2BoundInterfaceConstraint__slice(int32_t _2860_first,int32_t _2861_last) { return mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->__mosek_2fusion_2BoundInterfaceConstraint__slice(_2860_first,_2861_last); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::slice(int32_t _2860_first,int32_t _2861_last) { return __mosek_2fusion_2BoundInterfaceConstraint__slice(_2860_first,_2861_last); }
/* override: mosek.fusion.Constraint.slice*/
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::__mosek_2fusion_2Constraint__slice(int32_t _2860_first,int32_t _2861_last) { return __mosek_2fusion_2BoundInterfaceConstraint__slice(_2860_first,_2861_last); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_BoundInterfaceConstraint::__mosek_2fusion_2BoundInterfaceConstraint__slice(int32_t _2860_first,int32_t _2861_last)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2862_c = ::mosek::fusion::p_Constraint::__mosek_2fusion_2Constraint__slice(_2860_first,_2861_last);
  return (__mosek_2fusion_2BoundInterfaceConstraint__from_(_2862_c)).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.BoundInterfaceConstraint.slice

// Begin mosek.fusion.BoundInterfaceConstraint.index
// Method mosek.fusion.BoundInterfaceConstraint.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:60:6-115
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint :: __mosek_2fusion_2BoundInterfaceConstraint__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2863_idxa) { return mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->__mosek_2fusion_2BoundInterfaceConstraint__index(_2863_idxa); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2863_idxa) { return __mosek_2fusion_2BoundInterfaceConstraint__index(_2863_idxa); }
/* override: mosek.fusion.Constraint.index*/
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::__mosek_2fusion_2Constraint__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2863_idxa) { return __mosek_2fusion_2BoundInterfaceConstraint__index(_2863_idxa); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_BoundInterfaceConstraint::__mosek_2fusion_2BoundInterfaceConstraint__index(std::shared_ptr< monty::ndarray< int32_t,1 > > _2863_idxa)
{
  _checked_ptr_1<int32_t> _ptr__2863_idxa(_2863_idxa ? _2863_idxa->raw() : nullptr, _2863_idxa ? _2863_idxa->size(0) : 0);
  monty::rc_ptr< ::mosek::fusion::Constraint > _2864_c = ::mosek::fusion::p_Constraint::__mosek_2fusion_2Constraint__index(_2863_idxa);
  return (__mosek_2fusion_2BoundInterfaceConstraint__from_(_2864_c)).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.BoundInterfaceConstraint.index

// Begin mosek.fusion.BoundInterfaceConstraint.index
// Method mosek.fusion.BoundInterfaceConstraint.index @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:59:6-114
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint :: __mosek_2fusion_2BoundInterfaceConstraint__index(int32_t _2865_idx) { return mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->__mosek_2fusion_2BoundInterfaceConstraint__index(_2865_idx); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::index(int32_t _2865_idx) { return __mosek_2fusion_2BoundInterfaceConstraint__index(_2865_idx); }
/* override: mosek.fusion.Constraint.index*/
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::BoundInterfaceConstraint::__mosek_2fusion_2Constraint__index(int32_t _2865_idx) { return __mosek_2fusion_2BoundInterfaceConstraint__index(_2865_idx); }
monty::rc_ptr< ::mosek::fusion::Constraint > mosek::fusion::p_BoundInterfaceConstraint::__mosek_2fusion_2BoundInterfaceConstraint__index(int32_t _2865_idx)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _2866_c = ::mosek::fusion::p_Constraint::__mosek_2fusion_2Constraint__index(_2865_idx);
  return (__mosek_2fusion_2BoundInterfaceConstraint__from_(_2866_c)).as<::mosek::fusion::Constraint>();
}
// End mosek.fusion.BoundInterfaceConstraint.index

// Begin mosek.fusion.BoundInterfaceConstraint.from_
// Method mosek.fusion.BoundInterfaceConstraint.from_ @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/BoundInterfaceConstraint.mbi:49:6-57:7
monty::rc_ptr< ::mosek::fusion::BoundInterfaceConstraint > mosek::fusion::p_BoundInterfaceConstraint::__mosek_2fusion_2BoundInterfaceConstraint__from_(monty::rc_ptr< ::mosek::fusion::Constraint > _2867_c)
{
  monty::rc_ptr< ::mosek::fusion::Constraint > _10519_ = _2867_c;
  if (false) {}
  else if (nullptr != dynamic_cast<::mosek::fusion::SliceConstraint*>(_10519_.get()))
  {
    ::mosek::fusion::SliceConstraint::t _2868_cc (dynamic_cast<::mosek::fusion::SliceConstraint*>(_10519_.get()));
    {
      return ::mosek::fusion::p_BoundInterfaceConstraint::_new_BoundInterfaceConstraint(_2868_cc,islower);
    }
  }
  else
  {
    mosek::fusion::Constraint::t _2869_cc = _10519_;
    {
      return nullptr;
    }
  }
}
// End mosek.fusion.BoundInterfaceConstraint.from_

void mosek::fusion::p_BoundInterfaceConstraint::destroy()
{
}
void mosek::fusion::BoundInterfaceConstraint::destroy() { mosek::fusion::p_BoundInterfaceConstraint::_get_impl(this)->destroy(); }
// } class BoundInterfaceConstraint
// class LinearPSDConstraint { // @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDConstraint.monty:17:4-183:5
mosek::fusion::LinearPSDConstraint::LinearPSDConstraint(mosek::fusion::p_LinearPSDConstraint *_impl) : mosek::fusion::ModelConstraint(_impl) { /*std::cout << "LinearPSDConstraint()" << this << std::endl; */ }
mosek::fusion::LinearPSDConstraint::~LinearPSDConstraint() {  /* std::cout << "~LinearPSDConstraint()" << this << std::endl << std::flush; */ }
mosek::fusion::p_LinearPSDConstraint::p_LinearPSDConstraint
  (::mosek::fusion::LinearPSDConstraint * _pubthis) :     mosek::fusion::p_ModelConstraint(_pubthis),
    _pubthis(_pubthis)
{}
// mosek.fusion.LinearPSDConstraint.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDConstraint.monty:48:6-58:7
mosek::fusion::LinearPSDConstraint::t mosek::fusion::p_LinearPSDConstraint::_new_LinearPSDConstraint
  (monty::rc_ptr< ::mosek::fusion::LinearPSDConstraint > _2870_c,
    monty::rc_ptr< ::mosek::fusion::Model > _2871_m)
{ return new mosek::fusion::LinearPSDConstraint(_2870_c,_2871_m); }
mosek::fusion::LinearPSDConstraint::LinearPSDConstraint
  (monty::rc_ptr< ::mosek::fusion::LinearPSDConstraint > _2870_c,
    monty::rc_ptr< ::mosek::fusion::Model > _2871_m) : 
   LinearPSDConstraint(new mosek::fusion::p_LinearPSDConstraint(this))
{ mosek::fusion::p_LinearPSDConstraint::_get_impl(this)->_initialize(_2870_c,_2871_m); }
void mosek::fusion::p_LinearPSDConstraint::_initialize  (monty::rc_ptr< ::mosek::fusion::LinearPSDConstraint > _2870_c,
    monty::rc_ptr< ::mosek::fusion::Model > _2871_m)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    mosek::fusion::p_ModelConstraint::_initialize(_2870_c,_2871_m);
    conid = mosek::fusion::p_LinearPSDConstraint::_get_impl(_2870_c)->conid;
    slackidxs = mosek::fusion::p_LinearPSDConstraint::_get_impl(_2870_c)->slackidxs;
    nativeidxs = mosek::fusion::p_LinearPSDConstraint::_get_impl(_2870_c)->nativeidxs;
    conedim = mosek::fusion::p_LinearPSDConstraint::_get_impl(_2870_c)->conedim;
    shape = mosek::fusion::p_LinearPSDConstraint::_get_impl(_2870_c)->shape;
  }
}
// mosek.fusion.LinearPSDConstraint.ctor

// mosek.fusion.LinearPSDConstraint.ctor @ bld/docker-ubuntu-aarch64-20/final/default/src/fusion/LinearPSDConstraint.monty:27:6-46:7
mosek::fusion::LinearPSDConstraint::t mosek::fusion::p_LinearPSDConstraint::_new_LinearPSDConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2872_model,
    const std::string &  _2873_name,
    int32_t _2874_conid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2875_shape,
    int32_t _2876_conedim,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2878_slackidxs)
{ return new mosek::fusion::LinearPSDConstraint(_2872_model,_2873_name,_2874_conid,_2875_shape,_2876_conedim,_2877_nativeidxs,_2878_slackidxs); }
mosek::fusion::LinearPSDConstraint::LinearPSDConstraint
  (monty::rc_ptr< ::mosek::fusion::Model > _2872_model,
    const std::string &  _2873_name,
    int32_t _2874_conid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2875_shape,
    int32_t _2876_conedim,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2878_slackidxs) : 
   LinearPSDConstraint(new mosek::fusion::p_LinearPSDConstraint(this))
{ mosek::fusion::p_LinearPSDConstraint::_get_impl(this)->_initialize(_2872_model,_2873_name,_2874_conid,_2875_shape,_2876_conedim,_2877_nativeidxs,_2878_slackidxs); }
void mosek::fusion::p_LinearPSDConstraint::_initialize  (monty::rc_ptr< ::mosek::fusion::Model > _2872_model,
    const std::string &  _2873_name,
    int32_t _2874_conid,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2875_shape,
    int32_t _2876_conedim,
    std::shared_ptr< monty::ndarray< int32_t,1 > > _2877_nativeidxs,
    std::shared_ptr< monty::ndarray< int64_t,1 > > _2878_slackidxs)
{
  monty::rc_reserve __reserve_this(_pubthis); 
  {
    _checked_ptr_1<int32_t> _ptr__2875_shape(_2875_shape ? _2875_shape->raw() : nullptr, _2875_shape ? _2875_shape->size(0) : 0);
    _checked_ptr_1<int32_t> _ptr__2877_nativeidxs(_2877_nativeidxs ? _2877_nativeidxs->raw() : nullptr, _2877_nativeidxs ? _2877_nativeidxs->size(0) : 0);
    _checked_ptr_1<int64_t> _ptr__2878_slackidxs(_2878_slackidxs ? _2878_slackidxs->raw() : nullptr, _2878_slackidxs ? _2878_slackidxs->size(0) : 0);
    mosek::fusion::p_ModelConstraint::_initialize(_2872_model,_2873_name,_2875_shape